; BSD 3-Clause License
; Copyright (c) 2023, Jerome Shidel

; Redistribution and use in source and binary forms, with or without
; modification, are permitted provided that the following conditions are met:

; 1. Redistributions of source code must retain the above copyright notice, this
;    list of conditions and the following disclaimer.

; 2. Redistributions in binary form must reproduce the above copyright notice,
;    this list of conditions and the following disclaimer in the documentation
;    and/or other materials provided with the distribution.

; 3. Neither the name of the copyright holder nor the names of its
;    contributors may be used to endorse or promote products derived from
;    this software without specific prior written permission.

; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
; AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
; DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
; FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
; CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
; OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
; OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

; NASM 2.15.05, or later

use16

cpu 8086

%ifdef DEVICE_DRIVER
	org 0x0000
%else
	org 0x0100
%endif

section .text

%include "macros.asm" 			; include some common macro definitions

%idefine mfStrategy	00000001b
%idefine mfLinkedState	00000010b

;------------------------------------------------------------------------------
; SYS Specific
;------------------------------------------------------------------------------

%ifdef DEVICE_DRIVER

Header:
	.ChainNext: 	dd -1			; pointer to next device driver
	.Attributes:	dw 0x8000		; character device
	.Strategy:	dw Driver.Strategy	; set tREQUEST block pointer
	.Entry:		dw Driver.Routine	; driver interrupt call
	.Name:		db 'UMBTESTx'		; 8 character driver name
	.Request:	dd -1

Driver.Strategy:				; set request block pointer
	mov		[cs:Header.Request], bx
	mov		[cs:Header.Request+2], es
	retf

Driver.Routine:					; driver function dispatcher
	pushf
	push		ax
	push		di
	push		es

	les		di, [cs:Header.Request]
	mov		al, [es:di+tREQUEST.Function]
	test		al, al			; test command 0, initialize
	jz		Initialize

Driver.Error:
	mov		ax, 0x8103		; set error & done bits, and
						; error code 3, unknown command
Driver.Return:
	mov		[es:di+tREQUEST.Status], ax
	pop		es
	pop		di
	pop		ax
	popf
	retf

Initialize:
	; set pointer to first byte to free ourself
	mov		[es:di+tREQUEST.Address+2], cs
	mov		[es:di+tREQUEST.Address], word Header
	push		bx
	push		cx
	push		dx
	push		si
	push		ds
	push		bp
	call		ShowIntro
	pop		bp
	pop		ds
	pop		si
	pop		dx
	pop		cx
	pop		bx
	jmp		Driver.Error


%else
;------------------------------------------------------------------------------
; COM Specific
;------------------------------------------------------------------------------
	; assumed on DOS program start
	; mov		ax, cs
	; mov		ds, ax
	; mov		es, ax

	; adjust stack, DOS will restore on termination.
	mov		ax, Stack.Top
	mov		sp, ax

	; shrink program memory block, free all normal memory above the program
	mov		cl, 4
	shr		ax, cl		; divide ax by 16
	inc		ax		; can waste up to 16 bytes, don't care
	mov		bx, ax		; new size in paragraphs
	mov		ah, 0x4a
	; es already equals cs
	int		0x21
	jnc		ShowIntro
	mov		dx, Failed_Shrink
	jmp		Die
%endif

;------------------------------------------------------------------------------
; Initialization stuff
;------------------------------------------------------------------------------

ShowIntro:
	PrintMessage	Intro
	WordAsHex	cs
	PrintMessage	CRLF

	; Get current alloc Strategy
	mov		ax, 0x5800
	int		0x21
	jnc		GotStrategy
	mov		dx, Failed_GetStrategy
	jmp		Die

GotStrategy:
	; print the current strategy
	mov		[Strategy], ax
	PrintMessage	Success_GetStrategy
	WordAsHex	[Strategy]
	PrintMessage	CRLF

	; get current umb link state
	mov		ax, 0x5802	; Get UMB Link State
	int		0x21
	jnc		.GotLinkState
	mov		dx, Failed_GetLinkState
	jmp		Die

.GotLinkState:
	mov		[LinkedState], ax
	PrintMessage	Success_GetLinkedState
	ByteAsHex	[LinkedState]
	PrintMessage	CRLF

;------------------------------------------------------------------------------
; Testing routine
;------------------------------------------------------------------------------
	mov		si, TestTable
	cld
Testing:
	lodsb
	test		al, al			; 0, end of test table
	jz		.EndTest

	; prepare data values
	xor		ah, ah
	mov		[Request.KBSize], ax	; save size in KB
	mov		cl, 6
	shl		ax, cl
	mov		[Request.Size], ax	; Paragraphs size for request
	xor		ah, ah
	lodsb
	mov		[Request.LinkedState], ax ; Linked state as word
	lodsb
	mov		[Request.Strategy], ax	  ; strategy as word

	; show what is being tested
	PrintMessage	Testing_1
	WordAsInt	[Request.Size]
	PrintMessage	Testing_2
	WordAsInt	[Request.KBSize]
	PrintMessage	Testing_3
	mov		dx, Testing_4
	cmp		[Request.LinkedState], word 0
	je		.ShowLinkedState
	mov		dx, Testing_5
.ShowLinkedState:
	PrintMessage	dx
	WordAsHex	[Request.Strategy]
	PrintMessage	Testing_6
	PrintMessage	CRLF

	; set linked state
	mov		ax, 0x5803
	mov		bx, [Request.LinkedState]
	int		0x21
	jc		.SetLinkFailed
	or		[Flags], byte mfLinkedState

	; verify linked state
	mov		ax, 0x5802
	int		0x21
	jc		.VerifyLinkFailed
	cmp		al, [Request.LinkedState]
	jne		.VerifyLinkFailed

.IgnoreLinkFail:
	; set strategy
	mov		ax, 0x5801
	mov		bx, [Request.Strategy]
	int		0x21
	jc		.SetStrategyFailed
	or		[Flags], byte mfStrategy

	; verify strategy
	mov		ax, 0x5800
	int		0x21
	jc		.VerifyStrategyFailed
	cmp		al, [Request.Strategy]
	jne		.VerifyStrategyFailed

.IgnoreStrategyFail:
	; DOS allocate memory block
	mov		ah, 0x48
	mov		bx, [Request.Size]
	int		0x21
	jc		.AllocFailed
	mov		[MemBlock], ax
	PrintMessage	Success_Alloc
	WordAsHex	[MemBlock]

	; DOS free memory block
	mov		es, [MemBlock]
	mov		ah, 0x49
	int		0x21
	jc		.FreeFailed
	PrintMessage	Success_Freed
	PrintMessage	CRLF
	jmp		Testing

.AllocFailed:
	push		ax
	PrintMessage	Failed_Alloc
	pop		ax
	WordAsHex	ax
	PrintMessage	Failed_Largest
	WordAsInt	bx
	PrintMessage	Failed_Paragraphs
	PrintMessage	CRLF
	jmp		Testing

.FreeFailed:
	push		ax
	PrintMessage	CRLF
	PrintMessage	Failed_Free
	pop		ax
	WordAsHex	ax
	PrintMessage	CRLF
	jmp		Die

.VerifyLinkFailed:
	push		ax
	PrintMessage	Failed_Verification
	pop		ax
	ByteAsHex	al
	PrintMessage 	COMMA

.SetLinkFailed:
	PrintMessage 	Failed_SetLinkState
	PrintMessage	CRLF
	%ifdef REQUIRE_VERIFY
		jmp		Critical
	%else
		jmp		.IgnoreLinkFail
	%endif

.VerifyStrategyFailed:
	push		ax
	PrintMessage	Failed_Verification
	pop		ax
	ByteAsHex	al
	PrintMessage 	COMMA

.SetStrategyFailed:
	PrintMessage 	Failed_SetStrategy
	PrintMessage	CRLF
	%ifdef REQUIRE_VERIFY
		jmp		Critical
	%else
		jmp		.IgnoreStrategyFail
	%endif

.EndTest:
	PrintMessage 	Success_EndOfTest
	PrintMessage	CRLF

;------------------------------------------------------------------------------
; Shutdown stuff
;------------------------------------------------------------------------------

Terminate:
	xor		al, al
	jmp		ExitToDOS

Die:
	PrintMessage	dx
	PrintMessage	CRLF

Critical:
	mov		al, 0x01 	; error code 0x01

ExitToDOS:
	push		ax
	test		[Flags], byte mfLinkedState
	jz		.FixStrategy
	xor		bh, bh
	mov		bl, [LinkedState]
	mov		ax, 0x5803
	int		0x21
	; nothing we can do about an error here
.FixStrategy:
	test		[Flags], byte mfStrategy
	jz		.ExitNow
	xor		bh, bh
	mov		bl, [LinkedState]
	mov		ax, 0x5801
	int		0x21
	; nothing we can do about an error here either
.ExitNow:
	pop		ax

%ifdef DEVICE_DRIVER
	ret
%else
	mov 		ah, 0x4c
	int		0x21
%endif

	Include_MacroCode		; include code referenced by macros

section .data

CRLF:			db 0x0d,0x0a,'$'
COMMA:			db ', $'
Intro:			db 'UMB allocation test program at segment 0x$'
Success_EndOfTest:	db 'Tests completed successfully.$'
Failed_Shrink:		db 'Failed to reduce program memory block.$'
Failed_GetStrategy:	db 'Get allocation strategy failed.$'
Failed_GetLinkState:	db 'Get UMB linked state failed.$'
Success_GetStrategy:	db 'Got initial allocation strategy: 0x$'
Success_GetLinkedState:	db 'Got initial UMB linked state: 0x$'
Success_Alloc:		db 'Allocation success, segment $'
Success_Freed:		db ', freed.$'

Failed_Verification:	db 'Verification failure, value is $'
Failed_SetLinkState:	db 'Failed to set UMB link state. '
	%ifdef REQUIRE_VERIFY
		db 'Terminate.$'
	%else
		db 'Ignore.$'
	%endif
Failed_SetStrategy:	db 'Failed to set allocation strategy. '
	%ifdef REQUIRE_VERIFY
		db 'Terminate.$'
	%else
		db 'Ignore.$'
	%endif
Failed_Alloc:		db 'Allocation failed, error $'
Failed_Largest:		db ', largest block is $'
Failed_Paragraphs:	db ' paragraphs.$'
Failed_Free:		db 'Failed to free block, error $'

Testing_1:		db 'Test $'
Testing_2:		db ' Paragraph ($'
Testing_3:		db 'KB), UMB $'
Testing_4:		db 'un'
Testing_5:		db 'linked, strategy 0x$'
Testing_6:		db '$'

Strategy:		dw 0
LinkedState:		dw 0

Request.KBSize:		dw 0
Request.Size:		dw 0
Request.LinkedState:	dw 0
Request.Strategy:	dw 0

MemBlock:		dw 0

Flags:			dw 0

TestTable:
%include "settings.inc"

%ifndef DEVICE_DRIVER
section .bss

Stack.Bottom:		resw 512 ; 1k, way more than we need
Stack.Top:
%endif