/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* lowlevel code for parallel port data acquisition module */
/* Update 4/2005: track_lost() function introduced, "faster 2ysec" */

#define TRY_CIO_CTR  1          /* classic ISA digi I/O and counter card */
#define CIO_CTR_BASE 0x200      /* port base, + 2 is digi input port */
    /* DPI eyetrackers usually use INVERTED buttons in 4 lower bits */
    /* which are 8 = CAL  4 = NEXT  2 = YES  1 = NO */
int cio_inverse = 0x0f;         /* assume inverse (xor bitmask) */


#define TRY_DAS_16 1            /* classical ISA 12bit 120 kHz AD7800 board */
#define DAS_16_BASE 0x340       /* port base. More details below. */


/* We connect afeed (inverted on sub-d) to all ~CS controls for  */
/* data from the module to the computer! Avoids port fighting... */

#include "tracker.h"    /* includes all the headers */

static int lpt_base = 0;	/* printer port if detected, -1 if disabled */
int tracking = 0;		/* 1 while tracking, set only by get_raw_xy */
int lpt_inverse = 0;		/* XOR mask for buttons at printer port */



/* returns true if mouse instead of real eyetracker is used */
int using_mouse(void)
{
  return (lpt_base < 0);
} /* using_mouse */


/* returns true if last call to get_raw_xy found track lost (or blink) */
int track_lost(void)
{
  return (!tracking);
} /* track_lost */


/* find printer port device */
void init_lpt(void)
{
  int j;
  int x, y; /* x is not used here but will normally be 0..4095 or -1 */

  j = 0; /* trial number */
  do {
      int i;

      i = 0; /* port number */
      do {
          lpt_base = peek (0x40, 8+i+i); /* check LPT(i-1) port */
          if (!lpt_base) {	/* no printer port installed here */
	      i++;
	      y = 0;
          } else {
	      (void) get_raw_xy (&x, &y);	/* test-read that port */
	      i++; /* do not get stuck! */
	      if ( (y<0) || (y>4095) ) {	/* no device found? */
	          y = 0; /* if bit stuck to 1, assume error */
	          out8 (lpt_base + 2, 0x0c);	/* normal mode again */
	      }
          } /* if y is nonzero here, we found the right lpt_base */
      } while ( (i<3) && (!y) );
      j++;
  } while ( (j<100) && (!y) && (!lpt_base) );	/* try 100 (was: 5) times */

#if TRY_DAS_16
  if ( (in8 (DAS_16_BASE + 8) & 0xa0) == 0x20 ) {       /* minimalistic... */
      printf ("There seems to be a DAS16 ADC device at port %x. Initializing it.\n",
	  DAS_16_BASE);
	  /* 3 msb of base+8 are: busy, singleended, mux16 */
      out8 (DAS_16_BASE + 11,
      in8 (DAS_16_BASE + 11) & 0xf0);
          /* clear 4 lsb to force -5 ... +5 Volts range */
      out8 (DAS_16_BASE + 2, 0x20);
          /* set MUX range to channels 0 .. 2 (X, Y, TLB) */
      lpt_base = DAS_16_BASE;
      y = 1;
      printf ("    Using DAS16 in -5..+5 Volts range, Ch0..2 = X/Y/track\n");
  } /* DAS16 check */
#endif

  if (!y) {
      printf ("No printer port ADC device with non-zero Y coordinate found.\n");
      printf ("Using mouse / keyboard driver as fallback.\n");
      lpt_base = -1;
  } else {
      int key;

      printf ("    %s ADC device found at port %x. Press ESC to ignore it\n",
          (lpt_base == DAS_16_BASE) ? "DAS16" : "Printer port", lpt_base);
      printf ("    and use the mouse. Otherwise press ENTER to continue.\n");
      
      do {
	    key = getxkey ();
      } while ( (key != 27) && (key != 13) );   /* wait for response */
      
      if (key == 27) {				/* user wants only mice */
          printf ("  You selected using the mouse driver - ignoring ADC.\n");
          lpt_base = -1;			/* force mouse usage */
      }
  } /* device found */

} /* init_lpt */



uint32 get_raw_mouse(int * x, int * y) /* return raw mouse x/y coordinates */
{
  static int mickey_x = -1; /* mickey X coordinate mouse */
  static int mickey_y = -1; /* mickey Y coordinate mouse */
  static int mouse_buttons = 0;
  int86regs r;

#ifdef DEBUG_MOUSE_TIMING
  uint32 T;
#endif

  if (mickey_x < 0) {
      if ( (!peek (0, 0x33*4)) || (!peek (0, (0x33*4) + 2)) ) {
	  printf("You need a mouse driver now, sorry.\n");
	  exit (1);
      }
      r.w.ax = 0;       /* Function AX 0 is "init / start mouse driver" */
      (void) intr (0x33, &r, &r);
      if (r.w.ax != 0xffff) {
	  printf("Mouse driver not cooperating, sorry.\n");
	  exit (1);
      } else {
	  printf("Mouse with %d buttons found and activated.\n",
	    (r.w.bx == 0xffff) ? 2 : (int)r.w.bx );
      }
      mickey_x = 2047;
      mickey_y = 2047;
  } /* mouse initialization */

#ifdef DEBUG_MOUSE_TIMING
  T = get_ytime ();
#endif
  r.w.ax = 0x0b;        /* Function AX B is "get relative movement" */
  (void) intr (0x33, &r, &r);
  /* calling function B not only reads the value but also resets it */

  { /* update position, using mickeys */
      int delta_x = (sint16)(r.w.cx);
      int delta_y = (sint16)(r.w.dx);
      mickey_x += delta_x; /* movement unit is mickeys - fine grained */
      mickey_y += delta_y; /* movement unit is mickeys - fine grained */
      if ((mickey_x < 1) || (mickey_x > 4094))
	  mickey_x = (mickey_x < 1) ? 1 : 4094;
      if ((mickey_y < 1) || (mickey_y > 4094))
	  mickey_y = (mickey_y < 1) ? 1 : 4094;
  } /* position update */

#ifdef DEBUG_MOUSE_TIMING
  printf("int 33.b took %u ysec\n", get_ytime () - T);
#endif

  if (r.w.cx | r.w.dx) {

      /* This can be very SLOW because the driver waits for a data packet */
      /* So we only do it if the mouse has moved according to int 33.000b */
#ifdef DEBUG_MOUSE_TIMING
      T = get_ytime ();
#endif
      r.w.ax = 3;               /* Function AX 3 is "get position / buttons */
      (void) intr (0x33, &r, &r);
      mouse_buttons = r.w.bx & 3; /* only check left / right button */
#ifdef DEBUG_MOUSE_TIMING
      printf("int 33.3 took %u ysec\n", get_ytime () - T);
#endif
      tracking = (mouse_buttons & 3) ? 0 : 1;	/* click to get lost :-) */
  } /* check for mouse buttons */

  *x = mickey_x;
  *y = mickey_y;
  return get_ytime ();

} /* get_raw_mouse */



/* generate timing for LTC1286: CLK must stay stable at least 2ysec  */
/* and total clock frequency must not exceed 200 kHz. Given, because */
/* (2 * wait_2ysec) + (2 * write clock) + (1 * read data) with I/O   */
/* speeds of 1 ysec per LPT port I/O certainly takes that long.      */
void wait_2ysec(void)
{
  /* two possible versions: trust LPT port access to take > 1ysec */
  /* or use get_ytime() to be far on the safe side (wait 2.?ysec) */

#ifdef SAFE_WAIT_2YSEC

  uint32 first_time;
  first_time = get_ytime ();
  do {
#if 0
!   __asm__ __volatile__ ("inb $0x71, %%al\n" : : "a");
!	/* we are bored and read the CMOS data port all the time */
#endif
  } while ((get_ytime () - first_time) < 2)
  /* this assumes that get_ytime takes far less than 1 ysec, which is */
  /* true only if your processor has a CPU TSC (time stamp counter).  */

#else
   __asm__ __volatile__ (       /* volatile -> do not optimize */
       "inb %%dx, %%al\n"
#ifndef FAST_2YSEC
       "inb %%dx, %%al\n"	/* as wait_2ysec() is between inb/outb  */
       /* calls anyway, only 1 extra inb is already enough to force the */
       /* timespan between both inb/outb calls to at least 2 ysec...    */
#endif
       :                        /* output: nothing */
       : "d" (lpt_base)         /* input: port     */
       : "ax"                   /* modified: EAX   */
       );

#endif

} /* wait_2ysec */



uint32 get_raw_xy(int * x, int * y)  /* return raw x/y coordinates */
{   /* values will be in 0..4095 range or -1 for 12bit A/D module. */
    /* Track loss / blink is only stored in "tracking". If sample  */
    /* failed, only x will be forced to -1. See also calibr8.c ... */
  int bits, lpt_control;
  uint32 sample_moment, time_check;
  int xtmp, ytmp, i;

  if (!lpt_base)                        /* A/D device not yet known? */
      init_lpt ();

  if (lpt_base < 0)                     /* A/D device simulation mode? */
      return get_raw_mouse (x, y);	/* use only mouse for simluation */


#ifdef TRY_DAS_16
  if (lpt_base == DAS_16_BASE) {        /* if selected device is DAS16 */
      int buf, channel, ad_x, ad_y, tlb;

      ad_x = -1;
      ad_y = -1;
      tlb = -1;
     
      time_check = get_ytime ();
      while (in8 (DAS_16_BASE + 8) & 0x80) {            /* ADC busy */
	  if ( (get_ytime () - time_check) > 250 )      /* 1/4 msec */
	      break;                            /* we lost patience */
      } /* wait until ADC free */
      out8 (DAS_16_BASE, 0);        /* start a NEW conversion */
      sample_moment = get_ytime (); /* not elegant to put this here */
      
      do {
	  time_check = get_ytime ();
	  while (in8 (DAS_16_BASE + 8) & 0x80) {        /* ADC busy */
	      if ( (get_ytime () - time_check) > 250 )  /* 1/4 msec */
		  break;                        /* we lost patience */
	  } /* wait until ADC free */
	  out8 (DAS_16_BASE, 0);        /* start a NEW conversion */
	      /* this cycles trough the previously defined MUX  */
	      /* range automatically with every new conversion. */         
	  buf = in8 (DAS_16_BASE + 1);  /* 8 MSB of PREVIOUS A/D result */
	  buf <<= 4;
	  channel = in8 (DAS_16_BASE);  /* only 4 LSB are the channel   */
	  buf |= channel >> 4;          /* 4 LSB of PREVIOUS A/D result */
	  channel &= 0x0f;              /* keep only real channel info  */
	  
	  switch (channel) {            /* what did Santa give us today? */
	      case 0: ad_x = buf; break;
	      case 1: ad_y = buf; break;
	      case 2: tlb = buf; break;
	  } /* switch */
	  if (channel > 2)              /* unwanted channel? */
	      break;                    /* something went really wrong. */
      } while ( (ad_x < 0) || (ad_y < 0) ||
		(tlb < 0) );            /* until all fetched */
      
      *x = ad_x;
      *y = ad_y;
      tracking = (tlb > 2047) ? 1 : 0;	/* TLB is 5 V while tracking */
					/* 0 V for track loss or blink */
      
      /* If get_ytime() now is much bigger than sample_moment: Problem! */
      time_check = get_ytime () - sample_moment;
      
      return sample_moment + (time_check / 2);
  
  } /* DAS 16 reading */
#endif


  lpt_control = in8 (lpt_base+2) | 0x20; /* always use direction: IN */
  lpt_control |= 1; /* set STB to true, to pull ~STB on port low!    */
      /* (the A/D chips want this on LOW during init: used as CLK!)  */

  time_check = get_ytime ();

  out8 (lpt_base+2, lpt_control & ~2);  /* DISABLE the chips */
      /* pull ~CS high for a while to initiate a new conversion cycle   */
      /* ~CS is connected to ~AFEED which is INVERSE of lpt_control & 2 */
      /* Only an disable -> enable transition starts a new conversion!  */
  wait_2ysec ();
  out8 (lpt_base+2, lpt_control | 2);   /* ENABLE all chips */
      /* let the fun begin! (important: printer port is in IN direction) */
  wait_2ysec ();
  bits = in8 (lpt_base+1);      /* get status port */
  tracking = (bits & TLBBIT);	/* check if tracking lost (or blink) */

  if ((get_ytime () - time_check) < 5) {
      printf ("Unexpected problem in acquire.c! LPT I/O takes < 1ysec.\n");
      printf ("Please recompile with alternative wait_2ysec version.\n");
      exit (1);
  }

  for (i = 0; i<2; i++) {
      (void) in8 (lpt_base);                    /* takes 1ysec itself */
	  /* for ISA or LPT ports, this takes 1.2ysec. Less for PCI.  */
      out8 (lpt_base+2, lpt_control & ~1);      /* let CLK rise */
      wait_2ysec ();
      out8 (lpt_base+2, lpt_control | 1);       /* let CLK fall */
      wait_2ysec ();
  } /* 2 clock cycles for sample and hold */
  
  sample_moment = get_ytime ();
  xtmp = 0;
  ytmp = 0;

  for (i = 4096; i; i>>=1) {
      bits = in8 (lpt_base);                    /* fetch actual data bits */
      if (bits & X_ADCBIT) xtmp |= i;
      if (bits & Y_ADCBIT) ytmp |= i;
      out8 (lpt_base+2, lpt_control & ~1);      /* let CLK rise */
      wait_2ysec ();
      out8 (lpt_base+2, lpt_control | 1);       /* let CLK fall */
      wait_2ysec ();
  } /* collect 12+1 bits: 0, then MSB ... LSB ... MSB, then zeroes */
  /* because of the MSB-LSB-MSB scheme, you can pick an endianness */

  for (i = 1; i<12; i++) {
      (void) in8 (lpt_base); /* could pick up bits 1 .. 11 again here */
      out8 (lpt_base+2, lpt_control & ~1);      /* let CLK rise */
      wait_2ysec ();
      out8 (lpt_base+2, lpt_control | 1);       /* let CLK fall */
      wait_2ysec ();
  } /* extra cycles just return 0 bits */
  /* we could pull ~CS high earlier and stop clocking earlier,  */
  /* but LTC1286 wants at least 80 ysec total cycle time anyway */

  /* keep CLK down - no need to pull up ~CS now. */
  *x = xtmp;
  *y = ytmp;

  if ((ytmp | xtmp) & 4096) {   /* zero bit missing - chip not there? */
      *x = -1;			/* signal reading error */
      *y = -1;
  } /* no chip found!? */

  out8 (lpt_base+2, 0x0c);	/* disable ADC device again: */
  				/* set port to normal output mode again. */
  return sample_moment;

} /* get_raw_xy */



/* If keyboard key, buttons will have key ASCII << 8 and full */
/* key << 16 ORed in. */
uint32 get_buttons(int * bits) /* return bitmask: pressed buttons */
{  /* low bits are combination of ...BUTTON values defined above. */
  int mybuttons, shift1, shift2;

  mybuttons = 0;

  if (!lpt_base)
      init_lpt ();


#ifdef TRY_CIO_CTR
  { /* variable scope CIO */
      int data_in;

      data_in = in8 (CIO_CTR_BASE + 2) & 0x0f;
      if ( (data_in != 0) && (data_in != 0x0f) ) {
	  data_in ^= cio_inverse;       /* invert if inverse buttons */
	  if (data_in & 8) mybuttons |= CALBUTTON;
	  if (data_in & 4) mybuttons |= NEXTBUTTON;
	  if (data_in & 2) mybuttons |= YESBUTTON;
	  if (data_in & 1) mybuttons |= NOBUTTON;
      } else  { /* actual CIO CTR buttonbox activity found */
	  cio_inverse = (data_in == 0x0f) ? 0x0f : 0;
	      /* auto-adjust to normal or inverse buttons :-) */
      } /* all 4 buttons released */
      
  } /* variable scope CIO */
#endif


  if (kbhit ()) {       /* quick check whether key is waiting */
      int key;

      key = getxkey ();         /* adds 0x100 or 0x200 if specials */
	  /* getch() would use DOS */
      mybuttons |= key << 16;                           /* keyboard, verbose */
      switch (key) {
	  case  27: mybuttons |= ESCBUTTON; break;      /* ESC */
	  case   4:                                     /* ^D */
	  case   3: mybuttons |= ABORTBUTTON; break;    /* ^C */
	  case 'Y':
	  case 'y': mybuttons |= YESBUTTON; break;
	  case 'N':
	  case 'n': mybuttons |= NOBUTTON; break;
	  case 'C':
	  case 'c': mybuttons |= CALBUTTON; break;
	  case  13: mybuttons |= NEXTBUTTON; break;     /* ENTER */
	  default:  if (key < 0x100)
	      mybuttons |= key << 8;                    /* keyboard */
      } /* switch / case */
  } /* key waiting */


  if ( (lpt_base < 0) || (peekb (0x40,0x17) & 0x10) ) {
      /* no printer device found - or Scroll Lock active! */

      shift1 = peekb (0x40, 0x17);      /* 1st shift bit mask */
      shift2 = peekb (0x40, 0x18);      /* 2nd shift bit mask */

      if (shift1 & 2) mybuttons |= YESBUTTON; /* left  shift */
      if (shift1 & 1) mybuttons |= NOBUTTON;  /* right shift */

#if 0   /* needs 102 key keyboard (Bochs does not simulate one). */
      if ( (shift1 & 8) && (!(shift2 & 2)) )
	  mybuttons |= CALBUTTON;       /* alt, but not left one */
      if ( (shift1 & 4) && (shift2 & 1) )
	  mybuttons |= NEXTBUTTON;      /* ctrl, left one */
#else
      if ( shift1 & 8 )
	  mybuttons |= CALBUTTON;       /* any alt */
      if ( shift1 & 4 )
	  mybuttons |= NEXTBUTTON;      /* any ctrl */
#endif

  }


  if ( (lpt_base > 0) && (lpt_base != DAS_16_BASE) ) {
      /* DAS 16 is an A/D device but the buttonbox access is    */
      /* of course NOT like for a printer port device. However, */
      /* lpt_base > 0 means "use other than mouse for A/D"...   */
      int keybtn;       /* save */

      keybtn = mybuttons;
      out8 (lpt_base+2, in8 (lpt_base+2) | 0x23);
	  /* 1 "stb" (inverted on sub-d 25 - we want it OFF there)   */
	  /* 0x20 "direction: IN"   2 "afeed" (inverted on sub-d 25) */
      mybuttons = in8 (lpt_base) ^ lpt_inverse;
      mybuttons &= YESBUTTON | NOBUTTON | CALBUTTON | NEXTBUTTON; /* mask */

      if (mybuttons == (YESBUTTON | NOBUTTON | CALBUTTON | NEXTBUTTON) ) {
          lpt_inverse ^= YESBUTTON | NOBUTTON | CALBUTTON | NEXTBUTTON;
          mybuttons = 0;	/* assume all released, not all pressed */
      } /* other button polarity found -> auto-adjust dynamically */

      mybuttons |= keybtn;      /* restore / OR */
  }

  *bits = mybuttons;
  return get_ytime();

} /* get_buttons */



uint32 debounce(int buttons, int msecs) /* wait until all  of the */
{  /* given buttons have stayed unpressed for at least msecs...   */
   uint32 first_time, second_time, debounce_time;
   int my_buttons;

   msecs = ((msecs > 10000) || (msecs < 1)) ? 10000 : msecs; /* sanitize */
   debounce_time = (uint32)msecs * 1000;        /* 1000ysec/msec */
   do {
       do { /* wait until all desired buttons released for the first time */
	   first_time = get_buttons (&my_buttons);
       } while (buttons & my_buttons);

       do { /* loop until success or button pressed again */
	   second_time = get_buttons (&my_buttons);
       } while ( ((second_time-first_time) < debounce_time) ||
	   (buttons & my_buttons) );
   } while (buttons & my_buttons); /* repeat if still bouncing */

  return get_ytime ();

} /* debounce */

