/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* calibration module: interactive calibration, parameter calculation */
/* and conversion of raw coordinates into calibrated coordinates.     */
/* we use a 3x3 grid for this, 1 row margin on top/bottom, 5 columns  */
/* margin on left/right side.                                         */

/* UPDATE 4/2005: New calparams_?[3], drift_update(), drift correct.  */
/* UPDATE 4/2005: Freshness check in get_pixel_xy, TLB display.       */

#include "tracker.h"	/* include all headers */
#include "math.h"	/* fabs, for drift_update */

double calparams_x[4] = { 0, 0, 0, 0};	/* x = [0]*rawx*rawx +[1]*rawx +[2] */
double calparams_y[4] = { 0, 0, 0, 0};	/* y = [0]*rawy*rawy +[1]*rawy +[2] */
double pixel_eyex, pixel_eyey;		/* for hi-res things */

char middlemark1[3] = { '*', 0, 0 };	/* either * or filled >< (11/2004) */
char middlemark2[3] = { '+', 0, 0 };	/* either + or >< */

uint32 prev_get_xy_time = 0;		/* know how fresh pixel_* coords are */
			/* also triggers smoothing if get_xy called too often */

#define INIT_MARK 0	/* prepare only operator screen */
#define GET_MARK  1	/* show mark and get measurement, then hide again */


/* updates drift correction values based on current gaze position compared */
/* to passed drift correction mark position in virtual (!) screen pixels   */
/* returns the amount of adjustment in the coordinate pair. Why not ;-).   */
/* returns nonzero if the drift correction attempt got accepted and used.  */ 
int drift_update(int * markx, int * marky)
{
  double dc_f_x, dc_f_y;			/* float gaze position */
  double markx_f, marky_f;			/* float mark location */
  double new_dx, new_dy;			/* float calparams*[3] */
  int dc_x, dc_y, i;				/* just to trigger sampling */

  markx_f = *markx;				/* as *markx will change */
  marky_f = *marky;

  dc_f_x = dc_f_y = 0.0;
  for (i = 0; i < 20; i++) {			/* avg. of several samples */
      double f_x, f_y;
      (void)get_xy (&dc_x, &dc_y);		/* fetch a fresh sample */
      (void)get_pixel_xy (&f_x, &f_y);		/* get exact values for it */
      dc_f_x += f_x;	  /* *** TODO: check if eye is actually stable *** */
      dc_f_y += f_y;
  }
  dc_f_x /= 20.0;				/* calculate the average */
  dc_f_y /= 20.0;

  /* now compare eye position to drift correction gaze mark position... */
  new_dx = calparams_x[3] + markx_f - dc_f_x;	/* planned new offset */
  new_dy = calparams_y[3] + marky_f - dc_f_y;	/* planned new offset */
  *markx = (int)(0.5 + new_dx);			/* report absolute */
  *marky = (int)(0.5 + new_dy);			/* report absolute */

  /* statistics from a SMI experiment: 75% of corrections were less than  */
  /* 1 degree, 19% were 1 to 2 degrees, 3% were 2 to 3 degrees, the whole */
  /* screen taking about 60 degrees (in 1024x768 pixels) viewing angle... */
  /* WE use abs., not delta, offset limits for the accept/reject check:   */
  /* Units are virtual screen pixels based on 8x16 font: Common screen    */
  /* sizes in that unit are 512..640 by 384..400. Just using SOME limit.  */
  if ( (fabs(new_dx) > (2.5* 8)) ||		/* e.g. 2.0 or 2.5 chars */
       (fabs(new_dy) > (1.5*16)) ) {		/* e.g. 1.0 or 1.5 chars */
    return 0;					/* correction rejected */
  } else {
    calparams_x[3] = new_dx;			/* modify shift offset */
    calparams_y[3] = new_dy;			/* modify shift offset */
  }
  return 1;					/* drift correction okay */
} /* drift_update */


/* returns double precision pixel coordinate version of last get_xy result */
/* NOTE: this values use virtual screen size based on 8x16 font assumption */
void get_pixel_xy(double * x, double * y)	/* invalid if char x < 0!  */
{
  if ((get_ytime () - prev_get_xy_time) > 1000UL) { /* sample too old? (4/2005) */
      int ignoredx, ignoredy;			/* data was > 1 msec old!  */
      (void)get_xy (&ignoredx, &ignoredy);
      /* force a new hardware A/D read-out, which can take 100s of ysecs */
  } /* safety check - should not be needed in current NEWTRACK */

  *x = pixel_eyex;
  *y = pixel_eyey;
} /* get_pixel_xy */


/* fetches raw coordinates and turns them into calibrated coordinates */
uint32 get_xy(int * x, int * y)
{
  uint32 fetch_time;
  int rawx, rawy, cols, rows, i;
  static int prevx, prevy;
  double fx, fy;

  fetch_time = get_raw_xy (&rawx, &rawy);

  rows = get_resolution_y (&cols, &i);
  /* if (!(cols & 1)) cols--; */		/* make odd (obsolete 11/2004) */
  if (!(rows & 1)) rows--;			/* make odd */
  *x = rawx;
  *y = rawy;
  if ((rawx < 0) || (rawy < 0))
      return fetch_time;	/* pass on error / track loss unconverted */
      				/* (pixel coordinates not touched) */
  /* 4/2005 NOTE: blinks trigger track_lost(), not "error" at the moment! */

  if ((calparams_x[0] == 0) && (calparams_x[1] == 0)) {
      *x = -1;			/* not calibrated yet */
      *y = rawy;
      				/* (pixel coordinates not touched) */
      return fetch_time;	/* return as blink if not yet calibrated */
  }

  if ((fetch_time - prev_get_xy_time) < 1000) {
    int a;

    a = prevx;
    prevx = rawx;
    rawx = (rawx + a) / 2;
    a = prevy;
    prevy = rawy;
    rawy = (rawy + a) / 2;
  } else { /* 11/2004: smooth out a bit if > 1 kHz sampling rate found */
    prevx = rawx;
    prevy = rawy;
  }
  prev_get_xy_time = fetch_time;


  /* convert to -1 .. +1 range first */
  fx = 1.0 * rawx;
  fx = (fx * fx * calparams_x[0]) + (fx * calparams_x[1]) + calparams_x[2];
  fy = 1.0 * rawy;
  fy = (fy * fy * calparams_y[0]) + (fy * calparams_y[1]) + calparams_y[2];
  /* now convert to screen size range */
  fx = (fx + 1.0) / 2;
  fy = (fy + 1.0) / 2;
  fx *= 0.999999 * (cols-(5+5+1));
  fy *= 0.999999 * (rows-(1+1+1));

  fx += (1.0/ 8.0) * calparams_x[3];	/* drift correct - new 4/2005 */
  fy += (1.0/16.0) * calparams_y[3];	/* drift correct - new 4/2005 */
  /* calparams_?[3] is in virt. pixel units, so we div. by virt. font size */

  *x = 5 + 0.5 + fx;		/* truncate cal. range to 5..cols-(5+1) */
  *y = 1 + 0.5 + fy;		/* truncate cal. range to 1..rows-(1+1) */
  /* anything in char -> ?.00 ... ?.99 as coordinate */
  /* calibration mark centers are at ?.50 coordinates, so...  */
  /* distance of calibration mark centers ==  char coord dist */
  /* left mark, column 5 (zero based count) -> center 5.5 * 8 */
  /* right mar, column (cols-1)-5 -> center cols-5.5 * 8 ...  */

  pixel_eyex = (5 + 0.5 + fx) * 8;	/* as above, with virtual 8x16 font */
  pixel_eyey = (1 + 0.5 + fy) * 16;	/* as above, with virtual 8x16 font */
  /* top left of char -> ?.00, bottom right of chara -> ?.99 */

  if ( (pixel_eyex < 0) || (pixel_eyex >= (8 * cols)) ||
       (*x < 0) || (*x >= cols) ||
       (pixel_eyey < 0) || (pixel_eyey >= (16 * rows)) ||
       (*y < 0) || (*y >= rows) ) {
      *x = -1;			/* treat off-scale as "track lost" */
      *y = -1;			/* treat off-scale as "track lost" */
      				/*   (do not clip pixel values)    */
  }

  return fetch_time;
} /* get_xy */


double smart_average(int a, int b, int c);
double smart_average(int a, int b, int c)
{
  /* "non-smart" version was: 1.0 * (a + b + c) / 3.0 */
  /* 11/2004 "smart" version: Average of two-of-three which agree most */

  if ( (abs(a-b) < abs(b-c)) && (abs(a-b) < abs(a-c)) )
    return (0.5 * (a + b));

  if ( (abs(b-c) < abs(a-b)) && (abs(b-c) < abs(a-c)) )
    return (0.5 * (b + c));

  /* if ( (abs(a-c) < abs(b-c)) && (abs(a-c) < abs(a-b)) ) */
  return (0.5 * (a + c));
} /* smart_average */


/* takes an array of 9 coordinate pairs from calibrate() and */
/* calculates the 3+3 quadratic function parameters for X+Y. */
/* goal: a, b, c for f(x) = a(x*x) + bx + c which is -1, 0   */
/* and 1 for xf[0], xf[1], xf[2] respectively - same for y.  */
void update_calparams(int * coords)
{
  double xf[3];		/* raw X for -1, 0, 1 (averages) */
  double yf[3];		/* raw Y for -1, 0, 1 (averages) */
  double a, b, c;

#define CAL_AVERAGE(n1,n2,n3) smart_average (coords[n1], coords[n2], coords[n3])
  xf[0] = CAL_AVERAGE(  0+0,  6+0, 12+0);
  xf[1] = CAL_AVERAGE(  0+2,  6+2, 12+2);
  xf[2] = CAL_AVERAGE(  0+4,  6+4, 12+4);
  
  yf[0] = CAL_AVERAGE(  0+1,  0+3,  0+5);
  yf[1] = CAL_AVERAGE(  6+1,  6+3,  6+5);
  yf[2] = CAL_AVERAGE( 12+1, 12+3, 12+5);
#undef CAL_AVERAGE

  b =  (xf[1]-xf[0]) * (xf[2]*xf[2] - xf[1]*xf[1]);
  b -= (xf[2]-xf[1]) * (xf[1]*xf[1] - xf[0]*xf[0]);
  if (b == 0) {		/* impossible calibration */
    impossible_calibration:
    calparams_x[0] = calparams_x[1] = calparams_x[2] = 0;
    return;
  }
  b = ( (xf[2]*xf[2] - xf[1]*xf[1]) - (xf[1]*xf[1] - xf[0]*xf[0]) ) / b;

  a = xf[2]*xf[2] - xf[1]*xf[1];	/* or use sqr(xf[1])-sqr(xf[0]) */
  if (a == 0)
      goto impossible_calibration;
  a =  1.0 / a;
  a *= 1.0 - (b * (xf[2] - xf[1]));	/* or use xf[1] - xf[0] */

  c = - (a*xf[1]*xf[1] + b*xf[1]);	/* or use any other of the three */

  calparams_x[0] = a;
  calparams_x[1] = b;
  calparams_x[2] = c;
  calparams_x[3] = 0;			/* drift correction - new 4/2005 */

  b =  (yf[1]-yf[0]) * (yf[2]*yf[2] - yf[1]*yf[1]);
  b -= (yf[2]-yf[1]) * (yf[1]*yf[1] - yf[0]*yf[0]);
  if (b == 0)		/* impossible calibration */
      goto impossible_calibration;
  b = ( (yf[2]*yf[2] - yf[1]*yf[1]) - (yf[1]*yf[1] - yf[0]*yf[0]) ) / b;

  a = yf[2]*yf[2] - yf[1]*yf[1];	/* or use sqr(yf[1])-sqr(yf[0]) */
  if (a == 0)
      goto impossible_calibration;
  a =  1.0 / a;
  a *= 1.0 - (b * (yf[2] - yf[1]));	/* or use yf[1] - yf[0] */

  c = - (a*yf[1]*yf[1] + b*yf[1]);	/* or use any other of the three */

  calparams_y[0] = a;
  calparams_y[1] = b;
  calparams_y[2] = c;
  calparams_y[3] = 0;			/* drift correction - new 4/2005 */

} /* update_calparams */



/* process one calibration mark */
void do_calmark(int num, int * coords, int type, int cols, int rows)
{
  int xstart, ystart;
  int xcoord, ycoord;
  int eyex, eyey;
  int buttons;
  uint32 ytime;
  char numbers[80];

  xstart = coords[0];
  ystart = coords[1];

  xcoord = 5 + num % 3;
  ycoord = 1 + num / 3;
  if (xcoord > 5)
      xcoord = (xcoord == 6) ? (cols / 2) : (cols - (5+1));
  if (ycoord > 1)
      ycoord = (ycoord == 2) ? (rows / 2) : (rows - (1+1));
  /* special treatment of middle column below added 11/2004 */

  (void) put_string (xcoord, ycoord, 1, 4, 4, -1, -1,
      (xcoord == (cols/2)) ? middlemark2 : "+");		/* RED */
  if (type != INIT_MARK) {
      (void) put_string (xcoord, ycoord, 0, 15, 15, -1, -1,
      (xcoord == (cols/2)) ? middlemark1 : "*");
  } else {
      (void) put_string (xcoord-5, ycoord+1, 1, 7, 7, -1, -1,
          "(----,----)");
      return; 		/* do nothing else yet */
  }

  do {
      char * symbols[4] = { "*", "+", "*", "+" };	/* animated phases */
      int colors[4] = { 9, 10, 10, 9 };		/* BLUE, GREEN, GREEN, BLUE */

      ytime = get_raw_xy (&xstart, &ystart);
      ytime %= 250000;	/* 4 blink cycles per second */
      ytime /= 62500;	/* 4 phases per blink cycle */
      (void) get_xy (&eyex, &eyey);	/* returns -1 during 1st calibration */
      (void) put_string (xcoord, ycoord, 0, colors[ytime], 7,
          eyex, eyey, (xcoord == (cols/2)) ?
              ((ytime & 1) ? middlemark2 : middlemark1) : symbols[ytime]);

      if ( (xstart >= 0) && (ystart >= 0) ) {
          sprintf (numbers, "(%4d,%4d)", xstart, ystart);
          (void) put_string (xcoord-5, ycoord+1, 1, 15, 7, eyex, eyey,
              numbers);
      } else {
          (void) put_string (xcoord-5, ycoord+1, 1,  7, 7, eyex, eyey,
              "(----,----)");
      }
      (void) get_buttons (&buttons);
  } while ( (!(buttons & NEXTBUTTON)) || (xstart < 0) || (ystart < 0) );
  (void) debounce (NEXTBUTTON, DEBOUNCE);

  (void) put_string (xcoord-5, ycoord+1, 1, 7, 7, eyex, eyey, numbers);
      /* remove highlighting: color GREY 7 instead of WHITE 15 */
  (void) put_string (xcoord, ycoord, 1, 2, 2, eyex, eyey,
      (xcoord == (cols/2)) ? middlemark1 : "*");	/* GREEN */
#if 0
  (void) put_string (xcoord, ycoord, 0, 7, 7, -1, -1,
      (xcoord == (cols/2)) ? middlemark2 : ".");	/* GREY */
#else
  (void) put_string (xcoord, ycoord, 0, 7, 7, -1, -1,
      (xcoord == (cols/2)) ? ". " : ".");		/* GREY */
#endif
  coords[0] = xstart;
  coords[1] = ystart;

} /* do_calmark */



void calibrate(void)	/* everything interactive, so no parameters (yet) */
{
  char one_line[170];
  char empty[170];
  int raw_coords[18];	/* measured x, y, x, y... of calibration marks: */
  			/* first 3 in top row, then middle, then bottom */
  int i, cols, rows, fake, j;
  uint32 zerotime;

  calibrate_again:			/* label: repeat whole calibration */

  rows = get_resolution_y (&cols, &fake);
  calparams_x[3] = 0;			/* reset drift correction - 4/2005 */
  calparams_y[3] = 0;			/* reset drift correction - 4/2005 */

  if (cols > 160) {			/* if pixel units used */
      cols /= 8;			/* 9 pixel wide font not yet */
      rows /= peek (0x40, 0x85);	/* char height in scanlines  */
  }

  if ( (cols > 160) || (cols < 37) || (rows < 9) ) {
      (void) put_string (2, 1, 1, 4, 12, -1, -1,
          "Calibration impossible, wrong screen mode.");
      (void) put_string (2, 2, 1, 4, 12, -1, -1,
          "size! Press any button to continue");	/* really any */
      do {
          (void) get_buttons (&i);
      } while (!i);
      (void) debounce (-1, DEBOUNCE);
      return;
  }

  for (i = 0; i < cols; i++)
      empty[i] = ' ';
  empty[cols] = 0;
  
  clear_screen (0, CAL_BG, CAL_BG, CAL_BG);	/* clear subject screen */
  clear_screen (1, -1, -1, -1);			/* clear operator screen */

  /* if (!(cols & 1)) cols--; */	/* make odd (obsolete 11/2004) */
  if (!(rows & 1)) rows--;		/* make odd */

  if (!(cols & 1)) {			/* even number of columns? (11/2004) */
    middlemark1[0] = 0x10;			/* "filled >" */
    middlemark1[1] = 0x11;			/* "filled <" */
    middlemark1[2] = 0;
    sprintf (middlemark2, "><");
  } else {				/* middlemarks are 1 or 2 chars long */
    sprintf (middlemark1, "*");
    sprintf (middlemark2, "+");
  }

  sprintf (one_line, "Please look exactly at the stars when they show up.");
  (void) put_string ((cols - strlen(one_line)) / 2, (rows/2)+1, 0,  7, 15,
      -1, -1, one_line);
  sprintf (one_line,
      "Press NEXT to start and after each mark, or ESC to abort now.");
  (void) put_string ((cols - strlen(one_line)) / 2, (rows/2)+1, 1, 10, 15,
      -1, -1, one_line);
  zerotime = get_ytime ();

  do { /* initial "check voltages" screen */
      int eyex, eyey, rawx, rawy;

      (void) get_raw_xy (&rawx, &rawy);	/* returns rawx -1 on error / track loss */
      (void) get_xy (&eyex, &eyey);	/* returns eyex -1 if not cal'ed */

      sprintf(one_line, track_lost() ? "TLB " : "run ");
      /* track_lost() display added 4/2005 */

      if (rawx < 0) {
          sprintf (one_line + strlen (one_line), "%s raw=(----,----)",
              middlemark1);
      } else {
          sprintf (one_line + strlen (one_line), "%s raw=(%4d,%4d)",
              middlemark1, rawx, rawy);
      }
      if (eyex < 0) {
          sprintf (one_line + strlen (one_line), " char=(--,--)");
      } else {
          sprintf (one_line + strlen (one_line), " char=(%2d,%2d)", 
              eyex, eyey);
      }

      if ( (zerotime) && ((get_ytime () - zerotime) > (10 * 1000000)) ) {
        zerotime = 0;
        (void) put_string (0, (rows/2)+1, 0,  7, 15, -1, -1, empty);
      } /* remove subject instruction line after 10 seconds */

      (void) put_string (cols/2, rows/2, 0, 7, 7, -1, -1, middlemark1);
      (void) put_string ((cols/2) - 4, rows/2, 1, 2, 2, eyex, eyey, one_line);
      (void) get_buttons (&i);
  } while (!(i & (NEXTBUTTON | ABORTBUTTON | ESCBUTTON)));

  do {
     if (i & (ABORTBUTTON | ESCBUTTON)) {
         if ((calparams_x[0] == 0) && (calparams_x[1] == 0)) {	/* invalid cal? */
             sprintf (one_line, "Calibration aborted without useable data!");
             (void) put_string (0, (rows/2)+1, 0,  7, 15, -1, -1, empty);
             (void) put_string ((cols - strlen (one_line)) / 2, (rows/2) + 1,
                 0, 10, 15, -1, -1, one_line);
             sprintf (one_line, "WARNING: "
                 "Calibration left but no previous valid calibration to revert to!");
             (void) put_string (0, (rows/2),   1, 10, 15, -1, -1, empty);
             (void) put_string ((cols - strlen (one_line)) / 2, (rows/2),
                 1, 10, 15, -1, -1, one_line);
             sprintf (one_line,
                 "No coordinates will be available to other program parts.");
             (void) put_string (0, (rows/2)+1, 1, 10, 15, -1, -1, empty);
             (void) put_string ((cols - strlen (one_line)) / 2, (rows/2)+1,
                 1, 10, 15, -1, -1, one_line);
             /* We allow this anyway, because e.g. the tracker might be simply off */
             /* in which case we cannot do anything except leaving the program...  */
             (void) debounce (-1, 4000);	/* delay for 4 seconds */
         }
         clear_screen (0, -1, -1, -1);	/* clear subject screen */
         clear_screen (1, -1, -1, -1);	/* clear operator screen */
         /* only (rows/2) and (rows/2)+1 really needed that now */
         return;
     }
     (void) get_buttons(&i);
  } while (i & NEXTBUTTON);	/* next and abort overlap, both use ctrl */
  (void) debounce (NEXTBUTTON, DEBOUNCE);

  (void) put_string (0, (rows/2),   0,  7, 15, -1, -1, empty);
  (void) put_string (0, (rows/2),   1, 10, 15, -1, -1, empty);
  (void) put_string (0, (rows/2)+1, 0,  7, 15, -1, -1, empty);
  (void) put_string (0, (rows/2)+1, 1, 10, 15, -1, -1, empty);

  for (i = 0; i < 9; i++) {		/* init only operator screen now */
      raw_coords[i+i+0] = -1;		/* mark as "todo" point */
      raw_coords[i+i+1] = -1;
      do_calmark (i, &raw_coords[i+i], INIT_MARK, cols, rows);
  }

  srand ((unsigned int) get_ytime ());	/* seed random number generator */
  for (i = 0; i < 9; i++) {		/* fetch measured data */
      do {
          j = rand () % 9;		/* pick a random "todo" point   */
      } while (raw_coords[j+j] != -1);	/* other point if already done  */
      if ( (calparams_x[0] == 0) && (calparams_x[1] == 0) )
          j = i;			/* do not randomize first time! */
      /* could add a loop here: repeat each until measurement plausible */
      do_calmark (j, &raw_coords[j+j], GET_MARK, cols, rows);
  }

  update_calparams (&raw_coords[0]);	/* calculate actual parameters */

  (void) get_buttons (&i);
  if (i & (ABORTBUTTON | ESCBUTTON)) {
      clear_screen (0, -1, -1, -1);	/* clear subject screen */
      clear_screen (1, -1, -1, -1);	/* clear operatorscreen */
      /* added clear_screen here 11/2004 */
      return;
  }

  if ((calparams_x[0] == 0) && (calparams_x[1] == 0))	/* invalid cal? */
      goto calibrate_again;		/* yes, it is a goto! */

  sprintf (one_line,
      "Press NEXT to accept or CAL to calibrate again.");
  (void) put_string ((cols - strlen(one_line)) / 2, rows/2 - 1, 1, 10, 15,
      -1, -1, one_line);

  /* We use %3.4f instead of %5.6g because only the former can specify */
  /* a MINIMUM number of digits after the decimal point, e.g. 4...     */
  /* Important: %..f is for float while %..e and %..g are for double!  */

  sprintf (one_line,
      "x = (% 4.4e)*x*x %+5.5f*x %+4.3f", calparams_x[0],
          (float)(calparams_x[1]), (float)(calparams_x[2]));
  (void) put_string ((cols - strlen(one_line)) / 2, rows/2 + 2, 1, 10, 15,
      -1, -1, one_line);
  sprintf (one_line,
      "y = (% 4.4e)*y*y %+5.5f*y %+4.3f", calparams_y[0],
          (float)(calparams_y[1]), (float)(calparams_y[2]));
  (void) put_string ((cols - strlen(one_line)) / 2, rows/2 + 3, 1, 10, 15,
      -1, -1, one_line);

  do {
      int eyeX, eyeY;

      (void) get_xy (&eyeX, &eyeY);	/* show eye cursor while waiting */
      (void) put_string (0, 0, 0, 7, 7, eyeX, eyeY, "");
      (void) put_string (0, 0, 1, 4, 4, eyeX, eyeY, "");
      (void) get_buttons (&i);
  } while (!(i & (NEXTBUTTON | CALBUTTON | ABORTBUTTON)));
  (void) debounce (NEXTBUTTON | CALBUTTON, DEBOUNCE);

  (void) put_string (0, rows/2 - 1, 1, 10, 15, -1, -1, empty);
  (void) put_string (0, rows/2 + 2, 1, 10, 15, -1, -1, empty);
  (void) put_string (0, rows/2 + 3, 1, 10, 15, -1, -1, empty);
  (void) put_string (0, 0,          0,  7,  7, -1, -1, "");
  /* leave marks on screen, only remove cursors and parameters */

  if (i & CALBUTTON)
      goto calibrate_again;		/* yes, it is a goto! */

} /* calibrate */

