/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* parsing trials in (SMI Eyelink Saarbruecken "reading") experiment script */
/* This file: line-wrap the text of one trial stimulus */
/* UPDATE 10/2004: support reserved lines at top or bottom */
/* UPDATE  4/2005: allow words to start with the \n escape */
/* UPDATE  4/2005: let layout() use alignment, TOPLEFT or CENTERED */

#include "tracker.h"		/* include all needed headers */

#include "script.h"		/* script processing specific things */

#define STIMULUS_LINES 30	/* > max. number of USED lines / screen */

char * stim_lines[STIMULUS_LINES] =	/* pointers to lines */
 { NULL, };			/* start with all null pointers */

int reslines = 0;	/* 10/2004: number of reserved lines */
	/* or, if SINGLE_LINES flag set, single target line number */

int global_text_alignment = CENTERED;	/* 4/2005: TOPLEFT or CENTERED */
	/* or add an alignment argument to get_line_ypos and remove */
	/* this "global" (only used in layout.c) variable... */


char * get_line(int line)	/* pick a line of text from layout result */
{
  if ((line < 0) || (line > STIMULUS_LINES))
      return NULL;
  if ((stim_lines[line] == NULL) || !stim_lines[line][0])
      return NULL;
  return stim_lines[line];
} /* get_line */



/* assuming that there are lines lines on screen, return an Y coordinate */
/* for line number line (0 based) that helps with a nice centered layout */
/* ... and write an useful X coordinate to xpos as well. Layout depends  */
/* on mode (TOPLEFT or CENTERED) selected by the last layout() call.     */
/* Has to return a bigger value for line 1 than for line 0 even if there */
/* is only 1 line on screen, because analyze needs to know line spacing. */
int get_line_ypos(int line, int lines, int * xpos)
{
  int a, cols, rows, spreadlines, yspacing;
  char * theline;

  rows = get_resolution_y (&cols, &a);	/* ignore fake flag */
  rows -= 2;				/* margin: keep top / bottom free */
  if (reslines < SINGLE_LINE)
      rows -= (reslines > 0) ? reslines : (-reslines);	/* reserved lines */
      /* else: just pretend to use full screen if SINGLE_LINE mode */

  yspacing = 0;				/* minimum is 0+1 yspacing */
  spreadlines = 0;			/* yspacing >= 1 must be reached */

  while (global_text_alignment == CENTERED) {
      if (yspacing == 3) break;		/* maximum desirable value */
      if (spreadlines <= rows) {	/* space for more empty lines? */
          yspacing++;			/* try more spacing */
          spreadlines = ((lines-1) * yspacing) + 1;
          /* last line is not followed by spacing... */
      } else {
          yspacing--;			/* back one step */
          break;			/* found maximum possible value */
      }
  }					/* space inflation loop */

  if (global_text_alignment == TOPLEFT) {	/* use READING style layout */
      yspacing = (rows < 20) ? 0 : 1;	/* double spacing unless LOW res */
      spreadlines = ((lines-1) * yspacing) + 1;
      /* last line is not followed by spacing... */
      if ((spreadlines > rows) && (yspacing > 0))	/* not right READING */
          yspacing--;			/* style, but screen got too full... */
  }

  theline = get_line (line);		/* we have to "parse" that line */
  a = 0;

  if (theline != NULL) {
      for (; *theline; theline++) {
          if (*theline != INVSPACE)
              a++;			/* length, not counting INVSPACES */
      } /* loop over string */
  } /* any line at all */

  if (global_text_alignment == CENTERED) {	/* new choice point 4/2005 */
      *xpos = (cols - a) / 2;		/* for centered display */
  } else {
      *xpos = TOPLEFT_MIN_XMARGIN;	/* cannot sub-case this value yet */
  }

  if (global_text_alignment == TOPLEFT) {	/* all this: new 4/2005 */
     if (reslines >= SINGLE_LINE) {	/* forced position, used for "image2" */
     					/* for single text line image trials  */
         return (reslines - SINGLE_LINE) + (2 * line);
     } else {				/* automatic spacing / position */
         int top_reslines;		/* only care for reserved on top */
         int useable_lines;		/* screen size w/o margins / reserved */

         top_reslines = 1;		/* one line margin if nothing reserved */
         if (reslines > 0)		/* if some lines reserved on top */
             top_reslines = reslines;
         useable_lines =  rows + 2;	/* use all rows for this calculation */
         if (reslines < 0)		/* -n means n reserved at bottom    */
             useable_lines += reslines;	/* exclude reserved area at bottom  */
         useable_lines -= top_reslines;	/* exclude margin / reserved on top */
         if (lines > ((useable_lines+1)/2)) {
             return (top_reslines + line); /* crowded screen, gotta stay small */
         } else {
             return (top_reslines + (2*line)); /* normal READING style layout! */
         }
     }					/* automatic spacing / position */
  }					/* TOPLEFT processing */

  /* the following is *only* reached for CENTERED layout, not for TOPLEFT */
  rows += 2;				/* calculate with "all" rows */

  if (reslines >= SINGLE_LINE) {	/* big override enabled? (10/2004) */
      if (lines <= 1)
          return (3 * line) + (reslines - SINGLE_LINE);
          /* pretend that we have bigger spacing, for analyze fuzz! the */
          /* "single line" itself will just have line equal 0 anyway... */

      /* SINGLE_LINE but actually more in use? Then move out of the way! */
      if ((reslines - SINGLE_LINE) <= ((rows+1)/2)) {	/* upper half? */
          return line + (reslines - SINGLE_LINE) - (lines - 1);
          /* whole thing has to "scroll up" to keep "bottom line" high */
          /* ** bad luck if we end up out of range here ** */
      } else {						/* lower half */
          return line + (reslines - SINGLE_LINE);
          /* extra lines can just follow below the main line */
          /* ** bad luck if we end up out of range here ** */
      }   /* return some useful and well-controlled row number... */
          /* Use actual minimal spacing if more than a single line exists */
  }

  a = (rows - spreadlines) / 2;		/* first used line for centered text */
  if (reslines > 0)			/* 10/2004: if reserved lines at top */
      a += reslines;

  return a + (line * yspacing);		/* return suitable row number */

} /* get_line_ypos */



void LogString(char * s)
{
  uint32 firsttime;

  firsttime = timestamp;	/* log lines use "timestamp - firsttime" */
  				/* (dummy timestamp) */
  LOG_STRING (s);		/* can return void */
} /* LogString */


/* do layout for one screen of text, with given X margin and X spacing...  */
/* this function does not process X alignment or Y margin/spacing/position */
int layout_sized(char * text, int margin, int spacing)
{				/* returns number of lines found */
  char oneword[80];		/* buffers one word */
  char oneline[500];		/* buffers one line, for screen and log */
  char * current;
  int wordlen, thislinelen;
  int i, y, rows, cols;
  int invspaces;		/* invspaces are in string but not on screen */
  char lastchar;

  current = text;
  oneword[0] = oneline[0] = '\0';

  rows = get_resolution_y (&cols, &i);	/* i becomes 1 if fake mode */
  rows -= (reslines != 0) ? 1 : 2;	/* 2 lines reserved (changed 10/2004) */
	/* only 1 line on 1 end reserved if reslines reserved manually! */
  if (reslines < SINGLE_LINE)
      rows -= (reslines > 0) ? reslines : (-reslines);	/* reserved lines */
      /* else: just pretend that full screen is useable if SINGLE_LINE */
  cols -= margin+margin;	/* margin is reserved, too */

  for (y = 0; y < STIMULUS_LINES; y++) {
      if (stim_lines[y] == NULL) {
          stim_lines[y] = malloc (500);
          if (stim_lines[y] == NULL) {
              LogString ("* OUT OF MEMORY DURING LAYOUT *");
              return 0;		/* not enough RAM to layout */
          }
      }
      stim_lines[y][0] = '\0';	/* zap previous layout / buffer */
  }

  y = 0;
  thislinelen = 0;
  invspaces = 0;
  do {
      wordlen = get_word (current, oneword, &current, 0);

      if (wordlen) {
          lastchar = oneword[wordlen-1];
          if (lastchar == '\n') {
              oneword[wordlen-1] = '\0';	/* remove \n from string */
              wordlen--;
          } /* chomp \n  */
          if (lastchar == INVSPACE)
              invspaces++;		/* in string but not on screen */
      } else {
          break;		/* end of stimulus text, we are done */
      }

      strcpy (oneline + thislinelen, oneword);
      thislinelen += wordlen;		/* can be after an invisible break */

      if ( (lastchar != INVSPACE) && (lastchar != '\n') ) {
          int spccount;

          for (spccount = 0; spccount < spacing; spccount++)
              oneline[spccount+thislinelen] = ' ';
          oneline[spacing+thislinelen] = '\0';
          thislinelen += spacing;
      } /* add spaces after word */

      if (lastchar == '\n') {			/* manual linebreak? */
          strcpy (stim_lines[y], oneline);	/* save full line in result */
          oneline[0] = '\0';			/* initialize new line */
          y++;					/* next line */
          invspaces = 0;			/* nothing yet */
          thislinelen = 0;			/* nothing yet */
      } /* manual linebreak */

      if ( (thislinelen-invspaces) > cols ) {
          char * lastspace;			/* ... linebreak needed? */
          char * xspace;

          if (lastchar != INVSPACE) {		/* chop ' ' after word first */
              /* (not reached if lastchar was \n ...) */
              thislinelen -= spacing;
              oneline[thislinelen] = '\0';
          } /* chop trailing spaces */

          if ( (thislinelen-invspaces) <= cols ) {	/* 10/2004: was "<" */
              /* fits now, so we only have to start a new line */
              lastspace = NULL;
          } else {
              lastspace = strrchr (oneline, ' ');	/* too long, so we wrap */
              if ( lastspace == NULL ) {	/* cannot wrap? */
                  LogString ("* stimulus too wide *");
                  /* LogString (oneline); */
                  oneline[invspaces + cols] = '\0';	/* clip */
                  lastspace = NULL;		/* too wide, so no wrapping */
              } /* line still too wide? */

              xspace = lastspace;
              lastspace++;			/* first thing on new line */
              while (*xspace == ' ') {
                  *xspace = '\0';
                  xspace--;
              } /* remove spaces backwards */
          } /* had to move last word to new line */

          strcpy (stim_lines[y], oneline);	/* save full line in result */
          y++;					/* start with next line! */

          if ( (y >= STIMULUS_LINES) ) {
              /* removed 10/2004: LogString ("* stimulus too long *"); */
              /* LogString (oneline); */
              /* not necessarily too long: maybe just used a too big margin! */
              return 0;				/* un-layoutable long     */
          } /* too many lines to store */

          invspaces = 0;
          thislinelen = 0;

          if (lastspace != NULL) {			/* word has to be moved */
              int spccount;

              strcpy (stim_lines[y], lastspace);	/* copy word to new line */
              thislinelen = 0;
              while (stim_lines[y][thislinelen]) {	/* find len and invspaces */
                  if (stim_lines[y][thislinelen] == INVSPACE)
                      invspaces++;
                  thislinelen++;
              } /* count word properties */
              if (lastchar != INVSPACE) { 	/* unless wrapped word ends with INVSPACE */
                  for (spccount = 0; spccount < spacing; spccount++)
                      stim_lines[y][spccount+thislinelen] = ' '; /* add spaces again! */
                  stim_lines[y][spacing+thislinelen] = '\0';
                  thislinelen += spacing;
              } /* add spaces again unless wrapped word ends with INVSPACE */
          } else {
              stim_lines[y][0] = '\0';			/* new line is empty */
          } /* start new line */

          strcpy (oneline, stim_lines[y]);	/* overlapping-free copy  */

      } /* wrapping a line */

  } while (wordlen || (lastchar=='\n'));	/* changed 4/2005: allow \n */
  	/* (words which consist ONLY of \n no longer trigger end-of-text) */

  if (oneline[0] == '\0')			/* nothing in there yet? */
      return y;

  strcpy (stim_lines[y], oneline);		/* update last line */
  y++;
  return y+1;					/* number of lines */

} /* layout_sized */



/* process text for one screen. Returns number of lines used */
/* 10/2004 addition: reserve N lines at top or (negative values) bottom */
/* Size of reserved area is stored in a global variable! */
int layout(char * text, int lines, int alignment) { /* find best text layout */
  int lines1, lines2, rows, cols, i;

  reslines = lines;			/* copy to global variable */
  global_text_alignment = alignment;	/* copy to global variable */

  rows = get_resolution_y (&cols, &i);	/* i becomes 1 if fake mode */
  if (cols | i) { cols++; i++; }	/* dummy use, avoid compiler warning */
  /* only rows is actually used in layout()... */

  rows -= (reslines != 0) ? 1 : 2;	/* 2 lines reserved (changed 10/2004) */
	/* only 1 line on 1 end reserved if reslines reserved manually! */
  if (reslines < SINGLE_LINE)
      rows -= (reslines > 0) ? reslines : (-reslines);	/* reserved lines */
      /* else: just pretend that full screen is useable if SINGLE_LINE */
  rows = (rows+1)/2;			/* max number of layoutable lines */

	/* get minimal layout for reference */
  lines1 = layout_sized (text, (alignment == TOPLEFT) ? TOPLEFT_MIN_XMARGIN :
      MIN_XMARGIN, MIN_XSPACING);
  if ((lines1 <= 1) ||	/* if already reached optimum, or if layout is */
      (lines1 > rows))	/* not even possible with minimal margins, return */
      return (lines1 > rows) ? 0 : lines1;

  	/* prepare more generous layout as default */
  lines2 = layout_sized (text, (alignment == TOPLEFT) ? TOPLEFT_PREF_XMARGIN :
      PREF_XMARGIN, PREF_XSPACING);

  	/* if the more spacy layout uses more lines, use smaller layout... */
  if (lines2 > lines1) {
      /* do not just return lines1 here, but revert all internal layout */
      /* in other words, un-do the side effect of layout_sized PREF_* ! */
      return layout_sized (text, (alignment == TOPLEFT) ? TOPLEFT_MIN_XMARGIN :
          MIN_XMARGIN, MIN_XSPACING);
  } else {
      return lines2;
  }

} /* layout */

