#!/usr/bin/perl

# Public Domain pipe filter by Eric Auer 3/2005: Read NEWTRACK log
# file contents, return a fixed and filtered version of them, to
# make it easier for Thorsten's READING log analyzer to use them.
#
# Modified 4/2005: Disable fixes which are made obsolete by an
# update to NEWTRACK.
my $bugfix = 0;		# 1 to enable fixes for buggy older NEWTRACK
#
# Suggested log filter calling style:
# dos2unix < some.log | perl filter-logs.pl > some-new.log
#
# WARNING: UTF-8 dos2unix versions need "-iso" or "-c iso"
# dos2unix option to get Umlauts converted correctly! When dos2unix
# is an alias for "recode ibmpc..lat1", Umlauts will be okay, too.

# TRIGGER MAIN   -> 6th number value is 400 if gaze trigger succeeded,
#   less otherwise. Would be nice if the log analyzer could use that.
# FIXATION       -> timestamp is end-timestamp, 3rd number is start
#   timestamp, 4th number is duration. Values are made more intuitive
#   by removing the end-ramp timespan and re-calculating the duration.
# Coordinates are scaled: tag/numbers
#   IMAGETRIAL/1st-2nd, INFO WORD/2nd-5th, TRIGGER MAIN/2nd-5th,
#   FIXATION/1st-2nd (other values are preserved unchanged) and new:
#   ENTER WORD/2nd-5th, LEAVE WORD/2nd-5th (log events now allowed) 4/2005
# TRIAL ABORTED  -> shows a warning and the TRIALID argument
# TRIAL REPEATED -> shows a warning and the TRIALID argument
#
# Only done if $bugfix is on:
# IMAGETRIAL or TRIALID -> zap timestamp
# INFO WORD -> subtract 8 from 5th number value (bug fix)
# ENTER / LEAVE WORD -> remove (were buggy, not analyzed anyway)
#   ENTER / LEAVE uses different algorithms than FIXATION, so be
#   know what you are doing when you use it even with bug-fixed
#   versions of NEWTRACK. Bug was caused by INFO WORD pmaxx error.
# TRIAL RESULT 0 -> fixed to "TRIAL_RESULT 0" - this log problem
#   happens for aborted trials only. Successful trials all have a
#   TRIAL_RESULT n line with a value of 1 or 2 for n.
#
# Line format: timestamp " " tag " " 0 or more " "-sep. value(s).

my $where = "notrial";		# current trial id
my $filler = 0;			# 1 while a filler is processed
my $lastfix = 0;		# start time of previous fixation

foreach (<STDIN>) {
  my $line = $_;
  chomp($line);
  ($timestamp, $tag, @values) = split(' ',$line);

  if ($tag eq "IMAGETRIAL") {	# zap time
    if ($bugfix > 0) {
      $timestamp =~ s/[0-9]/0/g;	# zap timestamp
    }
  }

  if ($tag eq "TRIALID") {	# zap timestamp, store trial ID
    if ($bugfix > 0) {
      $timestamp =~ s/[0-9]/0/g;	# zap timestamp
    }
    $where = $values[0];	# keep trial ID for later
    $lastfix = 0;		# initialize fixation time check
    # print STDERR "Trial: $where ";
    if ($where =~ /[^-]*-[^-]*-[^-]/) {
      # print STDERR "(*)";	# items have min. 2 "-" in the id.
      $filler = 0;
    } else {
      # print STDERR "(filler)";
      $filler = 1;
    }
    # print STDERR "\n";
  }

  if ($tag eq "TRIGGER") {
    my $tword;
    if ($filler > 0) {
      $tword = "Filler";
    } else {
      $tword = "Trial ";
    }
    if ($values[6] == 400) {
      print STDERR "" . $tword . " $where GAZE triggered.\n";
    } else {
      print STDERR "" . $tword . " $where MANUALLY triggered.\n";
    }
  }

  if ($tag eq "TRIAL") {
    if ($values[0] eq "RESULT") {	# not used if newer NEWTRACK
      $tag = "TRIAL_RESULT";	# fix log style bug for aborted...
      shift @values;		# remove "RESULT" array item
      print STDERR "Fixed log for aborted trial $where.\n";
      # this is only needed in ($bugfix > 0) case, as older NEWTRACK
      # logged TRIAL_RESULT 0 as TRIAL RESULT 0 for aborted trials.
    }
    if ($values[0] eq "ABORTED") {
      print STDERR "Found aborted trial $where.\n";
    }
    if ($values[0] eq "REPEATED") {
      print STDERR "Found repeated trial $where.\n";
    }
  }

  if ($tag eq "IMAGETRIAL") {	# scale coordinates
    $values[0] *= 2;	# min-image-object-y
    $values[1] *= 2;	# max-image-object-y
  }

  if ($tag eq "DISPLAY_COORDS") {	# scale coordinates
    $values[0] *= 2;	# min-X
    $values[1] *= 2;	# min-Y
    $values[2] *= 2;	# max-X
    $values[3] *= 2;	# max-Y
  }

  if ($tag eq "SYNCTIME") {
    $lastfix = $timestamp;	# don't let fixations cross SYNCTIME
  }

  if ($tag eq "FIXATION") {		# scale coordinates
    my $tslen = length($timestamp);	# to be restored later
    $values[0] *= 2;	# X
    $values[1] *= 2;	# Y ([2] and [3] are starttime and duration)
    #
    # fixation endtime is logged AFTER ramp towards "moving" (4msec)
    #   (because it takes 4msec to be sure that the fixation ended),
    # starttime is logged INCLUDING ramp towards "stable" (5msec),
    # duration is logged WITHOUT ramp towards "moving" (4msec).
    #
    # example: "325 FIXATION x y 320 2" means eye movement started at
    # 325-4 after stopping at 320. Possible because "moving" is more
    # sensitive than "stable", which duplicates PCEXPT settings.
    #
    $timestamp -= 4;	# make endtime timestamp more intuitive
    if ($values[2] < $lastfix) {
      if ($values[2] < ($lastfix-1)) {	# 1 msec overlap is no problem
        print STDERR "Fixing fixation overlap by " . ($lastfix - $values[2]);
        print STDERR "msec in trial $where at $timestamp\n";
      }
      $values[2] = $lastfix;	# clamp start time to avoid overlap
      $values[3] = 1 + ($timestamp - $values[2]);	# new duration
    }
    $values[3] -= 1;	# more logical, at least for back to back fix
    $lastfix = $timestamp;
    while (length($timestamp) < $tslen) {
      $timestamp = "0" . $timestamp;	# re-add leading 0s for layout
    }
  }

  if ( ( ($tag eq "INFO") && ($values[0] eq "WORD") ) ||
       ( ($tag eq "TRIGGER") && ($values[0] eq "MAIN") ) ||
       ( ( ($tag eq "ENTER") || ($tag eq "LEAVE") ) &&
         ($values[0] eq "WORD") ) ) {
    $values[2] *= 2;	# min X / trigger X / avg X
    $values[3] *= 2;	# min Y / trigger Y / avg Y
    if ($bugfix > 0) {  # ONLY modify this "pmaxx" for buggy NEWTRACKs
      if ($tag eq "INFO") {	# INFO WORD is logged wrong:
        $values[4] -= 8;	# fix buggy max X value
      }
    }
    $values[4] *= 2;	# max X / (trigger X) / current X
    $values[5] *= 2;	# max Y / (trigger Y) / current Y
  }

  my $do_log;
  if ($filler == 1) {
    $do_log = 0;	# default: skip log events for fillers
    if ( ($tag eq "TRIAL_RESULT") || ($tag eq "IMAGETRIAL") ||
      ($tag eq "TRIALID") || ($tag eq "SYNCTIME") || ($tag eq "TRIAL")) {
      $do_log = 1;	# for fillers only those 5 events are logged
    }
  } else {
    $do_log = 1;	# default: use log events for items
    if ( ($tag eq "ENTER") || ($tag eq "LEAVE") ) {
      if ($bugfix > 0) {
        $do_log = 0;	# ENTER / LEAVE are wrong for buggy NEWTRACKs
      }
    }
  }

  if ( $do_log == 1 ) {
    print STDOUT "$timestamp $tag ";
    print STDOUT "" . join(' ', @values) . "\n";
  } # all other lines have $do_log == 0 and are dropped from the log
}

# done :-)
