/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* New DPI Eyetracker software for dual PCI VGA / Pentium TSC timed */
/* Connect serial A/D and buttons to printer port. Software and     */
/* hardware design written and conceived by Eric Auer in Mar 2004.  */

/* Compile with DJGPP (gcc) for DOS for best realtime performance.  */
/* A Linux port (worse for realtime) should be possible, too.       */
/* Compile:   rerror error.log gcc -Wall -O3 -s -o newtrack.exe *.c */
/* Do not forget that you need CWSDPMI to run NEWTRACK in DOS. You  */
/* can get it from the DJGPP download site http://www.delorie.com/  */
/* UPDATE 4/2005: Add "script=" as nicer keyword for "stimuli=".    */
/* UPDATE 4/2005: Add "reading=" READING script compatibility mode. */


#include "tracker.h"    /* includes all other headers */

#include <signal.h>     /* for Ctrl-C / Ctrl-Break configuration */

#if GRAPHICS
#include "vesa.h"	/* graphics mode init: vesainit() */
#endif

#define EVIL_FAST_FLIP 1 /* enable evil hack for "flip" speed gain */

int main (int argc, char * argv[])
{
  uint32 start_ytime, ytime;
  int x, y, i;
#if GRAPHICS
  int wantgfx;
#endif
  int columns, rows;		/* only used to init get_resolution_y() */
  int use_reading_semantics;	/* 4/2005 for READING style scripts */

  size_t filesize;              /* size of stimulus file */
  unsigned char * stimuli;      /* contents of stimulus file */
  unsigned char * logdata;      /* buffer for log data */

  char * subject = NULL;        /* subject name, no whitespace allowed */
  char * logfile = NULL;        /* logfile name */
  char * stimfile = NULL;       /* stimulus file name */

  columns = 42;
  rows = 23;

  cpu_mhz = 0;                  /* tell timer to calibrate first */
                                /* (global variable) */

  if ( (argc > 1) && (!strcmp(argv[1], "flip")) ) {
#if EVIL_FAST_FLIP /* evil hack: make timer.c skip calibration */
      cpu_mhz = 100;
#endif
      printf ("Switching to other screen\n");
      (void) select_screen(1, 0);       /* flip screen if 1st arg... */
      printf ("Other screen should be active now.\n");
      return 0;                         /* ... and exit */
  } /* just flip - do as little as possible apart from that */

  printf ("NEWTRACK eye tracking software - (c) Eric Auer 2004/2005 - License: GPL 2\n");
  printf ("\n  Buttons:    parport, keyboard (enable with scroll-lock) or CIO.\n");
  printf   ("  ADC:        2 LTC1286 (parport), mouse (simulation) or DAS16.\n");
  printf   ("  Dual-head:  2 PCI/AGP, interlaced (simulation) or mono+color.\n");
#if GRAPHICS
  printf   ("  Graphics:   VESA 16bpp LFB mode, PCX file display.\n");
#endif
  printf   ("  ySec clock: Processor TSC or normal PC timer chip.\n\n");

#if GRAPHICS
  wantgfx = 0;
  lfbSel = 0;				/* mark VESA as not yet initialized */
#endif
  use_reading_semantics = 0;			/* 4/2005 */

  for (i = 1; i < argc; i++) {
      if (!strncmp (argv[i], "graphics",8)) {
#if GRAPHICS
	  wantgfx = 1;
	  XGFX = XGFX_LO;
	  YGFX = YGFX_LO;
	  if (argv[i][8] == '=') {
	    wantgfx = atoi(argv[i] + 9);	/* fetch user kHz rate */
	    if (wantgfx < 31) wantgfx = 1;	/* replace nonsense values */
	  } /* "graphics=number" instead of plain "graphics" detected */
#else
	  printf ("Sorry, this version is compiled without graphics support.\n");
#endif
      } else if (!strcmp (argv[i], "hires")) {
#if GRAPHICS
          wantgfx = 1;				/* no user kHz rate */
          XGFX = XGFX_HI;
          YGFX = YGFX_HI;
#else
	  printf ("Sorry, this version is compiled without graphics support.\n");
#endif
      } else if (!strncmp (argv[i], "hires=", 6)) {
#if GRAPHICS
          wantgfx = atoi(argv[i] + 6);		/* fetch user kHz rate */
          if (wantgfx < 31) wantgfx = 1;	/* replace nonsense values */
          XGFX = XGFX_HI;
          YGFX = YGFX_HI;
#else
	  printf ("Sorry, this version is compiled without graphics support.\n");
#endif
      } else if (!strncmp (argv[i], "subject=", 8)) {
	  subject = argv[i] + 8;
      } else if (!strncmp (argv[i], "stimuli=", 8)) {
	  stimfile = argv[i] + 8;
      } else if (!strncmp (argv[i], "reading=", 8)) {	/* 4/2005 */
	  stimfile = argv[i] + 8;
	  use_reading_semantics = 1;			/* compatibility mode */
      } else if (!strncmp (argv[i], "script=", 7)) {
	  stimfile = argv[i] + 7;	/* alias for "stimuli" added 4/2005 */
      } else if (!strncmp (argv[i], "logfile=", 8)) {
	  logfile = argv[i] + 8;
      } else {
	  if ( (strcmp (argv[i], "/?")) && (strncmp (argv[i], "-h", 2)) )
	      printf ("Invalid argument: %s\n", argv[i]);
	      /* Note: DJGPP-compiled programs use / as synonym for backslash! */
	  printf ("Allowed arguments are:\n"
	         "  graphics=khz     (enable graphics mode for khz kHz screen)\n"
	         "  hires=khz        (enable hi-res graphics mode...)\n"
		 "  subject=name     (define subject name, no spaces allowed)\n"
		 "  script=filename  (use specified script file)\n"
		 "  reading=filename (use specified READING style script)\n"
		 "  logfile=filename (defaults to SUBJECTNAME.log)\n");
	  printf ("Alternative syntax is using 'flip' as the only argument,\n"
		  "  to switch between the available PCI/AGP graphics cards\n"
		  "  (does not start the experiment but returns at once)\n");
	  exit (1);
      } /* argument if ... chain */
  } /* arguments */

  while ( (subject == NULL) ||
	  (strlen (subject) < 1) ) {    /* not yet known? */
      if (subject == NULL) {
	  subject = (char *)malloc (80);
	  subject[0] = '\0';
      }
      printf ("Please specify subject name (no spaces allowed):\n");
      scanf ("%70s", subject);
  }                             /* keep asking until valid subject name */
  while (strchr (subject, ' ') != NULL) {
     *strchr (subject, ' ') = '_';
  } /* remove spaces */
  while (strchr (subject, '\t') != NULL) {
     *strchr (subject, ' ') = '_';
  } /* remove tabulators */


  do {		/* now using the universal file read function */
      filesize = fileread ("stimulus", stimfile, &stimuli);
      /* buffer is allocated as needed. Interactive if stimfile is NULL. */

      if ((filesize == 0) && (stimfile != NULL)) {
          printf ("Cannot open command-line specified stimulus file: %s\n",
              stimfile);
          exit(1);
      } /* read error but not interactive */
  } while (filesize == 0);      /* keep asking until stimulus file found */


  i = 0;                        /* try given or default file name first */
  do {
      if ((logfile == NULL) || (i < 0)) {
	  if (logfile == NULL) {
	      logfile = (char *)malloc (200);
	      logfile[0] = '\0';
	      sprintf (logfile, "%.180s.log", subject); /* default name */
	  } else {
	      printf ("Please specify log file name:\n");
	      scanf ("%160s", logfile);
	  }
      }
      i = open (logfile, O_CREAT | O_EXCL | O_WRONLY,
	  S_IRUSR | S_IWUSR ); /* must be writeable and will be writeable */
	  /* if file has been there before, we refuse to open it. */
      if (i < 0) perror("Try again. Cannot create NEW logfile");
  } while (i < 0);                      /* keep asking until valid logfile name */
  close(i); /* Fixed 4/2004 - forgotten to close() */

  logdata = (char *)malloc (LOGDATASIZE);       /* enough for 1 trial */
			/* ... but preferrably for the whole experiment */
  if (logdata == NULL) {
    printf ("Cannot initialize - not enough RAM free.\n");
    exit (1);
  }
  logdata[0] = '\0';            /* start with empty string */


  printf("\n>> Subject: %s\n>> Log file: %s\n", subject, logfile);
  printf(">> Stimulus file: %d bytes, %s\n",
      (int)filesize, (stimfile==NULL) ? "interactively selected" : stimfile);
      /* stimfile is NULL if the file was selected interactively */
#if GRAPHICS
  if (wantgfx) {
      printf(">> Graphics screen size: %dx%d, 16 bits per pixel.", XGFX, YGFX);
      if (wantgfx > 1) printf(" [%d kHz]", wantgfx);
      printf("\n");
  }
#endif
  printf("\n");

  (void) get_ytime ();          /* first use - triggers initialization, too */
  (void) get_buttons (&i);      /* first use - triggers initialization, too */
  (void) get_raw_xy (&x, &y);   /* first use - triggers initialization, too */
  (void) select_screen(0, 0);   /* first use - triggers initialization, too */

  printf ("\nPlease release all buttons. To abort at any time, press Ctrl-D.\n");
  printf ("Buttons:  LeftShift = Y = YES,  RightShift = N = NO,\n");
  printf ("          RightAlt = C = CAL,   Ctrl = Enter = NEXT,\n");
  printf ("          Escape = skip trial,  Ctrl-D = abort program.\n\n");
  
  do {
      (void) get_buttons (&i);
      printf ("Buttons: %d\r", i);      /* yes, \r - just a "flash". */
  } while (i & (CALBUTTON | ABORTBUTTON));
  (void) debounce (YESBUTTON | NOBUTTON | CALBUTTON | NEXTBUTTON, DEBOUNCE);
  
  (void) signal (SIGINT, SIG_IGN);      /* block default ^C / ^Break action */
  /* from now on, only the button check can be used to detect ^C */

  printf ("Press CAL button when ready for calibration or ABORT to abort.\n");
  do {
      start_ytime = get_ytime ();
      ytime = get_raw_xy (&x, &y) - start_ytime;
      /* measure delay "call get_raw_xy called" ... sample moment */
      printf (" get_raw_xy --> (%4d,%4d) (%2u/%2u ysec) ",
          x, y, ytime,
	  get_ytime () - start_ytime);
	  /* show how long get_raw_xy took, too */
      start_ytime = get_ytime ();
      ytime = get_buttons (&i) - start_ytime;
      printf (" get_buttons %s> %2u (%2u ysec)   \r",
	  (i) ? "==" : "--", i, ytime);
  } while (!(i & (CALBUTTON | ABORTBUTTON)));
  (void) debounce (CALBUTTON, DEBOUNCE);

  printf("\n\n");       /* last line is only for printf when in fake mode */

  if (i & ABORTBUTTON) goto shutdown_tracker;


#if GRAPHICS
  if (wantgfx) {
      i = vesainit(XGFX, -YGFX, 16,	/* use -16 instead of 16 to debug */
          ((wantgfx > 1) ? wantgfx : 0));	/* enter graphics mode */
          /* -YGFX means: Y resolution at least YGFX, centering allowed */
          /* try to use wantgfx kHz refresh rate on VBE 3.0 capable VGAs */
          /* (negative bpp depth, e.g. -16, would trigger verbose output) */
      if (i) {
          printf ("Sorry, no suitable graphics mode available on this system!\n");
          printf ("Error code: %d\n", i);
          wantgfx = 0;
          goto shutdown_tracker;
      } /* VESA initialization failure */

      select_screen (1, 0);		/* re-read screen1 geometry! */
      select_screen (0, 0);		/* re-read screen0 geometry! */

  } /* graphics mode requested */
#endif


  rows = get_resolution_y (&columns, &i);       /* get max text mode x/y */
      /* (i is now nonzero if fake dual head mode) */


  calibrate ();         /* do the whole interactive calibration stuff */

  sprintf (logdata, "%7.7u PARAMS %s %s"	/* parameter log added 11/2004 */
#if GRAPHICS
      " %d %d %d"
#endif
      "\r\n", 0, subject, (stimfile == NULL) ? "[prompted]" : stimfile
      /* logfile name is "implicit" anyway */
#if GRAPHICS
      , XGFX, YGFX, wantgfx
#endif
      );

  do_script (stimuli, logdata + strlen(logdata),
    use_reading_semantics);	/* do the whole interactive experiment */


  shutdown_tracker:             /* label for common safe exit point */

#if GRAPHICS
  (void) select_screen (0, 0);  /* back to screen 0 again */
  if (wantgfx)
      vesainit(0, 0, 0, 0);	/* return to text mode */
  printf ("Text resolution was: %dx%d %s\n",
      columns, rows, (i) ? "[fake dual head]" : "[dual head]");
#endif

  if (strlen (logdata) > 0) {
      printf ("Saving %d bytes from log buffer\n", strlen (logdata));
      i = open (logfile, O_WRONLY | O_APPEND | O_BINARY);
      if (i < 0) {
	  perror ("Unexpected error opening logfile for writing");
      } else {
	  char * tododata = logdata;
	  size_t todolen = strlen (tododata);

	  while (todolen) {
	      int j; /* fixed 4/2004 - must not use i */
	      
	      j = write (i, tododata, todolen);
	      if (j < 0) break; /* real write error, break out of loop */
	      tododata += j;
	      todolen -= j;
	  } /* writing (write can return after each self-selected chunk) */
	  if (todolen) {
	      printf ("Could not write last %lu of %d bytes. ",
		  todolen, strlen (logdata));
	      perror ("Reason");
	  }
	  close (i);        /* CLOSING the file is important for stat() ! */
      } /* writing to logfile */
  } /* have to flush logdata buffer first */
 
  filesize = 0;
  { /* variable scope: fileinfo */
      struct stat fileinfo;

      if (stat (logfile, &fileinfo)) {
	  perror ("Unexpected error opening logfile");
      } else {
	  if ( (S_IWUSR & fileinfo.st_mode) &&
	      S_ISREG(fileinfo.st_mode) ) {
	      filesize = fileinfo.st_size;
	  } else {
	      printf ("Unexpected logfile access error\n");
	  } /* no writeableable regular file */
      } /* file exists */
  } /* variable scope: fileinfo */

  if (filesize == 0) {
      printf ("Log file is still empty. Deleting '%s' again.\n", logfile);
      if (unlink (logfile)) perror ("Deletion failed");
      /* Luckily delete of a still open file fails silently!? */
      /* Still open files have not-yet-update stat() info...  */
  } else {
      printf ("Logfile size is %lu bytes.\n", filesize);
  } /* logfile has nonzero size */

  free (stimuli);
  free (logdata);

  printf ("Leaving!\n");
  exit (0);

} /* main */
