/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* Test program and VESA LFB HiColor PCX display, Eric Auer 6/2004 */
/* With palette display and array maximum finder functions...      */
/*   gcc -DTESTPCX=1 -s -Wall -O3 pcxshow.c vesainit.c fileread.c  */
/* See DJGPP library help: http://www.delorie.com/djgpp/doc/libc/  */

#ifndef TESTPCX
#define TESTPCX 0	/* set to compile the  standalone  test program */
#endif
#define SHOWPCXSIZE 0	/* set to let pcxshow print size info on screen */

#define BLUR 1	/* set to make areas spill over background a bit */
		/* this helps with areas with thin borders etc.! */

#define ZAPBORDERS 3	/* force N pixels of border to background color */
		/* done in showpcx(), but only if histo[] is present... */

#define FILL_DEPTH 2000	/* maximum recursion level for doFill */
		/* check added 10/2004. doFill modified to use loops, */
		/* which reduces stack usage a lot for most images... */

#define CRT_KHZ 0	/* screen refresh rate in kHz, 0 for default */

#include "tracker.h"	/* includes arch.h etc., sets XGFX / YGFX */
#include "vesa.h"	/* includes some global variables etc.    */


/* ****************************************************************** */


/* xpp and ypp are the position: negative xpp is centered, negative ypp */
/* means ypp pixels distance from bottom. Returns: # of non-black slots */
int showpalette(int xpp, int ypp)
{
    int xpalpos, palcount, ypalpos, ypix, xpix, n;

    xpalpos = xpp;
    if (xpp < 0) xpalpos = (vesamode.width-768) / 2;	/* centered */
    ypalpos = ypp;
    if (ypp < 0) ypalpos = vesamode.height+ypp-20;	/* from bottom */
    palcount = 0;

    xpix = xpalpos;
    for (n = 0; n < 256; n++) {
        if (palette[n]) palcount++;		/* count non-black */
        for (ypix = ypalpos; ypix < ypalpos+20; ypix++) {        
            putpixel16(xpix, ypix, palette[n]);
            putpixel16(xpix+1, ypix, palette[n]);
            putpixel16(xpix+2, ypix, palette[n]);
        } /* y drawing loop */
        xpix+=3;
    } /* x drawing loop */

    return palcount;
} /* showpalette */


/* returns the index of the biggest value in field */
int findmax (uint32 * field, int fieldlen)
{
  int x;
  int col = -1;			/* peak value location */
  uint32 cnt = 0;		/* peak value */

  for (x = 0; x < fieldlen; x++) {
      if (field[x] > cnt) {
          col = x;
          cnt = field[x];
      }
  } 				/* find-maximum-in-array loop */

  return col;
} /* findmax */


/* ****************************************************************** */


/* Draws a PCX image on a specified location on VESA screen   */
/* input: size of file in bytes, pointer to buffer, xlocation */
/* (negative: center), ylocation (negative: from bottom),     */
/* buffer for VESA mode sized pixel byte array (or NULL, data */
/* will be written to SCREEN coordinates in buffer!),         */
/* an uint32[256] histogram pointer or NULL. Returns 0 if ok. */
/* The real VESA screen is only updated if show is nonzero... */
int pcxshow(uint32 pcxsize, unsigned char * pcxdata,
    int xpos, int ypos, uint8 * rawpixels, uint32 * histo, int show)
{
  int xres, yres;

  if (pcxsize < (128+768))	/* too short for required parts? */
      return -1;		/* error 1: wrong file format */

  if ((pcxdata[0] != 10) || (pcxdata[1] < 5) /* version */ ||
      (pcxdata[2] != 1) /* RLE */ || (pcxdata[3] != 8) /* bits */ ||
      (pcxdata[0x41] != 1) /* 1 plane */ ) {
    /* Ignored: [10] ... [3F] EGA palette, [40] byte 0 [44] word paltype */
    /* [12] word Xdpi [14] word Ydpi */
      return -1;		/* error 1: wrong file format */
  } /* header checks */

#define GETw(n) ( ((uint16)(pcxdata[n])) | ((uint16)(pcxdata[n+1])<<8) )

  xres = 1 + GETw(8) - GETw(4);
  yres = 1 + GETw(10) - GETw(6);

#if SHOWPCXSIZE
  printf(" X %u-%u, Y %u-%u (%dx%d)", GETw(4), GETw(8), GETw(6), GETw(10),
      xres, yres);
  if (GETw(0x42) != xres) printf(" %d bytes/line?", GETw(0x42));
#endif

  if (xpos < 0) xpos = (vesamode.width - xres) / 2;	/* centered */
  if (ypos < 0) ypos = vesamode.height - yres;		/* from bottom */

  if ( ((xpos+xres) > vesamode.width) || ((ypos+yres) > vesamode.height) ||
      (xpos < 0) || (ypos < 0) )
      return -2;					/* image too big */

  /* real data starts at 0x80 */
  /* 256 color palette starts at end-0x300 */
  if (pcxdata[pcxsize-769] == 12) {	/* palette marker found */
      uint32 n;
      uint32 p = pcxsize - 768;		/* buffer pointer */

      for (n = 0; n < 256; n++) {
          /* Convert RGB to HiColor (15-16bpp) / TrueColor (24-32bpp) pixel: */
          palette[n] = RGB2c(pcxdata[p], pcxdata[p+1], pcxdata[p+2]);
          /* (<-> Must use the HARDWARE palette if 256 color VESA mode) */
          p += 3;
      } /* palette reading loop */
  } else {				/* palette read / else create one */
      int n;

      for (n = 0; n < 256; n++)
          palette[n] = RGB2c(n, n, n);	/* black to white gradient */
  } 					/* palette initialization */

  if (histo != NULL) {
      int n;

      for (n = 0; n < 256; n++) histo[n] = 0;	/* zap histogram data */
  }

  { 					/* main data decompression loop */
      uint32 x, y, ptr;
      uint8 col;
      uint32 rawwidth = vesamode.width;

      x = xpos;
      y = ypos;
      ptr = 0x80;			/* main data starts at this offset */

      do {
          col = pcxdata[ptr];		/* fetch byte */
          ptr++;
          if (col >= 0xc0) {		/* repeated byte and/or byte > 0xbf */
              uint8 cnt;

              cnt = col & 0x3f;		/* count: up to 63 copies */
              col = pcxdata[ptr];	/* actual color (can be > 0xbf) */
              ptr++;
              if (histo != NULL) histo[col] += cnt;

              for ( ; cnt ; cnt--) {	/* must not wrap a line (given) */
                  if (rawpixels != NULL)
                      rawpixels[x + (y * rawwidth)] = col;
                  if (show) putpixel16(x, y, palette[col]);
                  x++;
              } /* run length loop */
          } else {			/* normal single pixel */
              if (histo != NULL) histo[col]++;
              if (rawpixels != NULL)
                  rawpixels[x + (y * rawwidth)] = col;
              if (show) putpixel16(x, y, palette[col]);
              x++;
          }				/* single pixel */
          if (x >= (xpos + xres)) {	/* wrapped around to next line? */
              x = xpos;
              y++;
          }
      } while ((ptr < pcxsize) && (y < (ypos + yres)));
  } /* data decompression and histogram buildup */

  if ((histo != NULL) && (ZAPBORDERS > 0)) {
      uint32 x, y, borderwidth, rawwidth;
      uint8 bgcol = findmax (histo, 256);

      rawwidth = vesamode.width;
      for (borderwidth = 0; borderwidth < ZAPBORDERS; borderwidth++) {
          for (x = xpos; x < (xres + xpos); x++) {
              if (show) {
                  putpixel16(x, ypos + borderwidth, palette[bgcol]);
                  putpixel16(x, ypos + (yres-1) - borderwidth, palette[bgcol]);
              }
              if (rawpixels != NULL) {
                  rawpixels[x + ((ypos + borderwidth) * rawwidth)] = bgcol;
                  rawpixels[x + ((ypos + (yres-1) - borderwidth) * rawwidth)] = bgcol;
              }
          } /* horizontal */
          for (y = ypos; y < (yres + ypos); y++) {
              if (show) {
                  putpixel16(xpos + borderwidth, y, palette[bgcol]);
                  putpixel16(xpos + (xres-1) - borderwidth, y, palette[bgcol]);
              }
              if (rawpixels != NULL) {
                  rawpixels[xpos + borderwidth + (y * rawwidth)] = bgcol;
                  rawpixels[xpos + (xres-1) - borderwidth + (y * rawwidth)] = bgcol;
              }
          } /* vertical */
      } /* loop until enough border width eliminated */
  } /* border suppression (needs histo) */

  return 0;				/* everything okay */

} /* pcxshow */


/* ****************************************************************** */

#define NORTH 1
#define EAST 2
#define SOUTH 3
#define WEST 4

void doFillInternal (const int look, int x, int y);
static int fill_depth;
static int fill_area;
static uint32 fill_size;

/* recursive fill algorithm for pixelspace  */
/* "where to look", x, y, "what to fill in" */
/* 0 is "outside", 255 is "inside", rest is */
/* "already filled by other area" ...       */
/* returns size of detected area in pixels. */
uint32 doFill (const int look, int x, int y, const int last, const int depth_left)
{ /*  uint8 look uint16 x uint16 y uint8 last uint16 depth_left */
    fill_depth = depth_left;	/* copy to global variable */
    fill_area = last;		/* copy to global variable */
    fill_size = 0;
    doFillInternal (look, x, y);	/* start recursion */
    return fill_size;		/* count of pixels in this object */
} /* doFill */

/* internally used by doFill: less stack usage with help of global vars */
void doFillInternal (const int look, int x, int y)
{
  int mark; /* using shorter int types does not optimize anyway... */
            /* same for argument types. Bad luck for our stack. */

  if ((x >= vesamode.width) || (y >= vesamode.height) || (x < 0) || (y < 0))
      return;			/* coordinates out of range */

  if ((fill_depth <= 0) ||	/* too complex, ran out of stack levels? */
      (spacePIX(x, y) <= fill_area))	/* already seen or "outside" */
      return;

  /* spacePIX(x, y) = fill_area; */	/* make it part of our area (moved below) */

  fill_depth--;			/* remember that calls are deeper now */
  switch (look) {		/* look around for more of that */
      case SOUTH:		/* check SOUTH, EAST, WEST */
          mark = y;
          do {
              spacePIX(x, y) = fill_area;	/* make it part of our area */
              fill_size++;
              y++;
          } while ((y < vesamode.height) && (spacePIX(x, y) > fill_area));
          /* loop while we can find unseen inside parts for our area */
          for (y--; y >= mark ; y--) {
              doFillInternal(EAST, x+1, y);
              doFillInternal(WEST, x-1, y);
          } /* loop back and search "sideways" along the way */
          break;
      case WEST:		/* check WEST, SOUTH, NORTH */
          mark = x;
          do {
              spacePIX(x, y) = fill_area;	/* make it part of our area */
              fill_size++;
              x--;
          } while ((x != -1) && (spacePIX(x, y) > fill_area));
          for (x++; x <= mark; x++) {
              doFillInternal(NORTH, x, y-1);
              doFillInternal(SOUTH, x, y+1);
          }
          break;
      case EAST:		/* check EAST, SOUTH, NORTH */
          mark = x;
          do {
              spacePIX(x, y) = fill_area;	/* make it part of our area */
              fill_size++;
              x++;
          } while ((x < vesamode.width)  && (spacePIX(x, y) > fill_area));
          for (x--; x >= mark; x--) {
              doFillInternal(SOUTH, x, y+1);
              doFillInternal(NORTH, x, y-1);
          }
          break;
      case NORTH:		/* check NORTH, EAST, WEST */
          mark = y;
          do {
              spacePIX(x, y) = fill_area;	/* make it part of our area */
              fill_size++;
              y--;
          } while ((y != -1) && (spacePIX(x, y) > fill_area));
          for (y++;  y <= mark; y++) {
              doFillInternal(WEST, x-1, y);
              doFillInternal(EAST, x+1, y);
          }
          break;
      default:			/* else start looking in ALL directions */
          spacePIX(x, y) = fill_area;	/* make it part of our area */
          fill_size++;
          doFillInternal(NORTH, x, y-1);
          doFillInternal(SOUTH, x, y+1);
          doFillInternal(WEST, x-1, y);
          doFillInternal(EAST, x+1, y);
  } /* switch */
  fill_depth++;	/* lower stack load again */

  return;
} /* doFillInternal */


/* Analyze the VESA screen sized pixelspace uint8 array and */
/* mark all non-background objects by separate colors       */
/* Returns the number of separate areas found, max. 255     */
/* pixelspace values bground count as background. Value 255 */
/* is reserved and means uninitialized (also background)    */
/* The result is stored as area number for each pixelspace  */
/* pixel, area number 0 being reserved for the background.  */
/* Set visualize to have some visual output of the result.  */
/* returned min_y and max_y are the "objects found" range.  */
int imagechunker (uint8 bground, int visualize, int * min_y, int * max_y)
{
  int p;		/* pointer into pixelspace */
  uint8 last;		/* area number */
  uint16 zipcode[64];	/* color coded areas :-) */

  if ((pixelspace == NULL) || (min_y == NULL) || (max_y == NULL))
      return 0;		/* failure */

  for (p = 0; p < (vesamode.height * vesamode.width); p++) {
      uint8 c = pixelspace[p];
      pixelspace[p] = ((c == bground) || (c == 255)) ? 0 : 255;
      /* transform to 0 background 255 foreground */
  } /* loop over pixels */

#if BLUR	/* blur feature added 10/2004: chunks get more "sticky" */
  if ((vesamode.width <= 2048) && (vesamode.height <= 2048)) {
      int xb, yb;
#if 0
      uint8 * blurbuf;
      blurbuf = malloc(2048);
#else
      static uint8 blurbuf[2048];
#endif

      if (blurbuf != NULL) for (yb = 0; yb < vesamode.height; yb++) {
          for (xb = 0; xb < vesamode.width; xb++)
              blurbuf[xb] = spacePIX(xb, yb);
          if (blurbuf[1] || blurbuf[2]) spacePIX(0, yb) = 255;
          if (blurbuf[2] || blurbuf[3]) spacePIX(1, yb) = 255;
          for (xb = 2; xb < vesamode.width-2; xb++)
              if (blurbuf[xb-1] || blurbuf[xb-2] ||
                  blurbuf[xb+1] || blurbuf[xb+2]) spacePIX(xb, yb) = 255;
          xb = vesamode.width - 1;
          if (blurbuf[xb-1] || blurbuf[xb-2]) spacePIX(xb, yb) = 255;
          if (blurbuf[xb-2] || blurbuf[xb-3]) spacePIX(xb-1, yb) = 255;
      } /* in each line, bleed colors 2 pixels wide into background */

      if (blurbuf != NULL) for (xb = 0; xb < vesamode.width; xb++) {
          for (yb = 0; yb < vesamode.height; yb++)
              blurbuf[yb] = spacePIX(xb, yb);
          if (blurbuf[1] || blurbuf[2]) spacePIX(xb, 0) = 255;
          if (blurbuf[2] || blurbuf[3]) spacePIX(xb, 1) = 255;
          for (yb = 2; yb < vesamode.height-2; yb++)
              if (blurbuf[yb-1] || blurbuf[yb-2] ||
                  blurbuf[yb+1] || blurbuf[yb+2]) spacePIX(xb, yb) = 255;
          yb = vesamode.height - 1;
          if (blurbuf[yb-1] || blurbuf[yb-2]) spacePIX(xb, yb) = 255;
          if (blurbuf[yb-2] || blurbuf[yb-3]) spacePIX(xb, yb-1) = 255;
      } /* in each column, bleed colors 2 pixels high into background */

#if 0 /* no malloc no free... */
      if (blurbuf != NULL) free(blurbuf);
#endif
      /* could be more elegant by doing X and Y at the same time, and */
      /* using a "diamant" matrix. But this method is FASTER, I hope. */

#if 0 /* visualize blur */
      if (visualize) {
          for (yb = 0; yb < vesamode.height; yb++)
              for (xb = 0; xb < vesamode.width; xb++)
                  if (spacePIX(xb, yb))
                      putpixel16 (xb, yb, RGB2c(0, 255, 0));
          getxkey();
      } /* visualize (boring, BLUR just does what you guessed it would do) */
#endif /* visualize blur */
  } /* BLUR */
#endif

  last = 0;			/* nothing assigned to an area yet */
  *min_y = 9999;
  *max_y = 0;
  {				/* find all areas */
      int y;
      int x = 0;		/* we scan from left to right */
      searchNextPCXArea:
      for (y = 0; y < vesamode.height; y++) {
          if (spacePIX(x, y)) {		/* any non-empty pixel found */
              if (y < *min_y) *min_y = y;	/* pushing the limit */
              if (y > *max_y) *max_y = y;	/* pushing the limit */
          }				/* any non-empty pixel found */
          if (spacePIX(x, y) > last) {	/* not-yet-used pixel found? */
              /* this inner fill call is hit at most Nareas times */

              last++;	/* use fresh area number */

              (void) doFill (EAST, x, y, last, FILL_DEPTH);	/* fill! */
              /* we already do know that we must look to the east */
              /* return value is area size (ignored for now) */

              /* (x unchanged) */	/* restart search from HERE */
              goto searchNextPCXArea;	/* search for next area */
          }				/* yet unassigned pixel found */
      } /* loop over pixels */
      x++;				/* scan on */
      if (x < vesamode.width) goto searchNextPCXArea;
  } /* loop to find all areas */
  /* last is now number of areas found (1..last, 0 is background) */

  for (p = 0; p < 64; p++) {
      int c85[4] = { 0, 3*85, 85, 2*85 };	/* intense colors first */

      zipcode[p] = RGB2c( c85[(p>>4)&3], c85[(p>>2)&3], c85[p&3]);
  } /* area code palette creation */

  if (visualize) {
      uint16 bg = 0; 

      for (p = (vesamode.height-1); p >= *min_y; p--) {	/* SHOW areas */
          int x;

          for (x = 0; x < vesamode.width; x++) {
              uint8 c = spacePIX(x, p);		/* read from 8 bit buffer */

              if ((c) && (c <= last)) {
                  putpixel16(x, p, zipcode[c & 63]);	/* write to screen */
                  if (!bg) bg = RGB2c( 127, 127, 127);	/* start w/ backgr. */
              } else if ((!c) && (bg)) {
                  putpixel16(x, p, bg);		/* show defined background */
              }
          } /* loop over columns */
      } /* loop over lines */

  } /* if visualize */

  /* if *max_y is 0 here and *min_y is 9999 here, image was empty */

  return last;
} /* area assignment stuff */


/* ****************************************************************** */


#if TESTPCX		/* for testing: main program */

#define PRINToverGFX 1	/* show some status messages in graphics mode */
#define ANALYZEhisto 1	/* do histogram and show most used color */
#define SHOWpalette  1	/* show a sample of every color */

/* Main program, obviously */
int main(int argc, char * argv[])
{
  int n, e;
  uint8 * pcxdata;
  uint8 bgcolor;
  uint32 pcxsize;
  uint32 * histogram;

  pixelspace = NULL;

  if (argc > 1) {
    pcxsize = fileread("PCX", argv[1], &pcxdata);
  } else {
    pcxsize = fileread("PCX", NULL, &pcxdata);
  }
  if (pcxsize < (128+768)) {
      printf("File too small, probably no 256 color PCX.\n");
      return 1;
  }

#ifdef HIRES
  XGFX = XGFX_HI;	/* could add a command line option to select hi-res */
  YGFX = YGFX_HI;	/* same here */
#else
  XGFX = XGFX_LO;	/* could add a command line option to select hi-res */
  YGFX = YGFX_LO;	/* same here */
#endif

#if PRINToverGFX
  if ((e=vesainit(XGFX, -YGFX, -16, CRT_KHZ)) != 0) { /* -YGFX: "at least YGFX" */
#else
  if ((e=vesainit(XGFX, -YGFX, 16, CRT_KHZ)) != 0) { /* -YGFX: "at least YGFX" */
#endif
    printf("Bad luck, no %dx%d HiColor LFB mode Error %d.\n", XGFX, YGFX, -e);
    return -e;
  }

  pixelspace = malloc (vesamode.width * vesamode.height);
  /* will handle NULL automatically as "do not use pixelspace" */
  if (pixelspace != 0)
      for (n = 0; n < (vesamode.width * vesamode.height); n++)
          pixelspace[n] = 255;		/* "nothing else on screen" */
  histogram = malloc (sizeof(uint32) * 256);
  /* will handle NULL automatically as "do not create histogram" */

  e = pcxshow (pcxsize, pcxdata,
      -1 /* x: centered */, 50 /* y: 50 from top */,
      pixelspace, histogram, 1);	/* *** main call *** */


#if SHOWpalette
  n = showpalette (-1, -10);
#endif

#if ANALYZEhisto
  {
      int x, y, best;

      best = findmax (histogram, 256);	/* find most used color index */

      for (y = vesamode.height-10; y < vesamode.height; y++)
          for (x = 32; x < (vesamode.width-32); x++)
              putpixel16(x, y, palette[best]);
				/* draw a bar sample of the "most" color */
#if PRINToverGFX
      printf (" File size %d, color %d most used (%d times). Key to continue.\n",
          pcxsize, best, histogram[best]);
#endif
    bgcolor = (uint8)(best & 0xff);
  }
#else
  bgcolor = 0;
#endif	/* ANALYZEhisto */

#if ANALYZEhisto
  getxkey();
  if (pixelspace != NULL) {		/* need pixelspace workspace */
      int areas, minY, maxY;

      areas = imagechunker (bgcolor, 1, &minY, &maxY);
      					/* analyze image and show! */
#if PRINToverGFX
      printf (" %d areas found in rows %d-%d.", areas, minY, maxY);
#endif
  }
#endif	/* ANALYZEhisto */

#if PRINToverGFX
  e = -e;				/* convert to positive */
  if (e) printf ("\n PCXShow returned error %d (%s).\n", e,
      ((e < 0) || (e > 2)) ? "unknown" : ( (e == 1) ?
      "PCX must be version 5+, RLE, 1*8 bit" :	/* error -1 */
      "PCX too big, does not fit on screen" /* (at this position) */ ) );

  printf (" %d colors used. Press any key to return to DOS.\n", n);
#endif

  getxkey();

  e = vesainit (0, 0, -1, 0);
  return e;

} /* main */

#endif /* TEXTPCX */
