/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* screen output function: fancy "put a string on screen"   */
/* includes highlighting of the word "under" the eye cursor */
/* and drawing of the eye cursor itself. All configurable.  */
/* UPDATE  4/2005: introduced FONT_ZOOM factor, was fixed to 2 before */
/* UPDATE  4/2005: optimized spaces in clear_screen, for faster VESA  */

#include "tracker.h"    /* includes all the headers */

#if GRAPHICS
#include "vesa.h"
#include <dpmi.h>		/* __dpmi_int, int86 is not enough */

uint8 * font_buf;		/* for 8x16 BIOS bitmap font copy  */
uint16 * fake_screen_bufs;	/* 2 text mode buffers, only 1     */
				/* used in VESA unless fake dual   */
int inversetext = 0;		/* set for screen 0 black on white mode */
uint16 last_cls_color = 0;	/* used by VESA clear_screen and put_char */
#endif

#define PROPER_CLEAR 0		/* set to 1 to get full but slower VESA */
	/* subject clear screen in interlaced fake dual head mode: else */
	/* half of the subject part of screen is not cleared, but the   */
	/* (usually called at the same time) operator "screen" clear    */
	/* will take care of that. Does not affect real dual head case. */

#define MONO_ATTRIBS_FOR_MONO 1	/* if operator screen is mono, fix "colors" */

typedef struct EyeCursor {
  uint16 x, y;		/* bytes would have been enough... */
  	/* y always in terms of real_y if fake dual head text mode */
  uint16 item;		/* low byte: char / high byte: colors */
} EYE_CURSOR;		/* item is zero if nothing to restore */

EYE_CURSOR eye_cursor_old[2] = { {100, 100, 0}, {100, 100, 0}};
	/* location and char "under cursor", for screen 0 and 1 */



#if GRAPHICS
/* put a char on either screen, simple if the screen is in text mode */
/* in fake dual head graphics mode, interlace 2*24 lines of (2*8)x16 */
/* and in real dual head graphics mode, use 24 lines of (2*8)x(2*16) */
/* (example is for 1024*768 XGFX*YGFX mode, 2 is from FONT_ZOOM) ... */
/* Coordinates are in chars, color is text mode style color pair     */
/* If fake dual head text mode is on, real_y coordinate is used!     */
/* Graphical fake dual: mx is -1 or +1 * size of virtual text buffer */
/* (-1 screen0, +1 screen1). Text: mx is real width. Graphics: mx=0. */
void put_char(int xc, int yc, int mx, uint8 color, uint8 ch);
void put_char(int xc, int yc, int mx, uint8 color, uint8 ch)
{
  uint16 hicolor, hiback;		/* colors in framebuffer format */
  int yzoom;				/* vertical font zoom for VESA */
  int x0, y0, y;
  /* mx values: 0 graphic, 1..132 text */
  /* <0 screen 0 fake dual graphic >132 screen 1 fake dual graphic */

  /* if ((xc < 0) || (yc < 0) || (xc >= 80) || (yc >= 25)) exit(39); */

  if ( (mx <= 0) || (mx > 132) ) {	/* graphic target? */
      int where, previous;

      where = ((mx > 0) ? mx : 0) + (yc * 132) + xc;	/* ONE meaning of mx */
      /* if ((xc >= 64) || (mx > 132*24)) exit(40); */
      previous = fake_screen_bufs[where];
      fake_screen_bufs[where] = ((uint16)color<<8) | ch;
				/* store to virtual screen buffer, is only */
      				/* read for eye cursor backstore right now */
      			/* mx < 0 means screen is 0, mx > 0 means screen 1 */
      if (previous == fake_screen_bufs[where])
          return;		/* no changes -> "lazy drawing". Important */
          			/* because text display speed is critical! */
  } else {				/* text target */
      /* if (mx > 80) exit(41); */
      poke (ScreenSeg, (yc * 2 * mx) + xc + xc,	/* OTHER meaning of mx */
          ((uint16)color<<8) | ch );
      return;			/* if this is text mode, we are *DONE* now */
  }

  /* next part is GRAPHICS mode only: mx 0 normal, < 0 / > 132 fake 1st/2nd */

  if (inversetext) {			/* <-- this is a global variable... */
      y = (color & 8) ? 128 : 0;	/* (sic!) full or softer contrast */
      hicolor = RGB2c( ((color & 4) ? y : 255),
          ((color & 2) ? y : 255), ((color & 1) ? y : 255) );
      y = (color & 8) ? 128 : 0;	/* (sic!) full or softer contrast */
      hiback = RGB2c( ((color & 0x40) ? y : 255),
          ((color & 0x20) ? y : 255), ((color & 0x10) ? y : 255) );
  } else {
      y = (color & 8) ? 255 : 192;	/* normal or bright color */
      hicolor = RGB2c( ((color & 4) ? y : 0),
          ((color & 2) ? y : 0), ((color & 1) ? y : 0) );
      y = (color & 8) ? 255 : 192;	/* normal or bright background (no blinking) */
      hiback = RGB2c( ((color & 0x40) ? y : 0),
          ((color & 0x20) ? y : 0), ((color & 0x10) ? y : 0) );
  }
  if ( ((color & 0xf0) == 0) && (last_cls_color) )
    hiback = last_cls_color;		/* substitute black */
  if ( ((color & 0x0f) == 0) && (last_cls_color) )
    hicolor = last_cls_color;		/* substitute black */

  x0 = (FONT_ZOOM*8) * xc;		/* char location */
  y0 = (FONT_ZOOM*16) * yc;		/* char location */

  yzoom = (mx==0) ? FONT_ZOOM : (FONT_ZOOM/2);	/* zoom for FONT */
  	/* reduce zoom if interlaced graphics fake dual mode */

  if (mx > 132) {			/* 2nd screen in interlaced mode */
      y0 += (FONT_ZOOM/2)*16;		/* "below" 1st screen, fake dual */
      if (FONT_ZOOM & 1) y0 += 16;	/* adjust for bigger 1st screen font */
      /* if FONT_ZOOM was only 1, delta Y for interlace is 0, not good... */
  } else {				/* 1st screen is subject screen */
      if ((mx < 0) && (FONT_ZOOM & 1)) yzoom++;
      /* if odd FONT_ZOOM, give subject screen the bigger zoom if interlaced */
  } /* interlaced mode handling */

  if (ch == 255)			/* a super-fast non-drawing whitespace */
      return;

  if (ch == ' ') {			/* faster drawing of spaces */
      int x;

      for (y = y0; y < (y0 + (16*yzoom)); y++)
          for (x = x0; x < (x0 + (8*FONT_ZOOM)); x++)
              putpixel16 (x, y, hiback);	/* background color rectangle */

      return;				/* we are *DONE* now with VESA spaces */
  } /* whitespace handling */

  for (y = 0; y < 16; y++) {		/* loop over font pixel rows */
      uint8 fontmask, fontbyte;
      int ybase, x;

      fontbyte = font_buf[(16*(uint16)ch) + y]; /* 1 line from font */
      x = x0;

      if (yzoom > 1) {
        if ((yzoom == 2) && (FONT_ZOOM == 2)) {	/* use 2x2 zoom? */
            ybase = y0 + (yzoom * y);		/* advance zoomed */
            for (fontmask = 0x80; fontmask; fontmask >>= 1) {
                uint16 c;

                c = (fontbyte & fontmask) ? hicolor : hiback;
                /* if ((ybase+1 >= YGFX) || (x+FONT_ZOOM >= XGFX)) exit(42); */
                putpixel16 (x, ybase, c);	/* line doubling */
                putpixel16 (x+1, ybase, c);	/* line doubling */
                putpixel16 (x, ybase+1, c);	/* line doubling */
                x++;				/* next pixel */
                putpixel16 (x, ybase+1, c);	/* line doubling */
                x++;				/* next pixel */

            } /* x loop with 2x2 zoom */
        } else {			/* non-optimized bigger zoom factors */
            ybase = y0 + (yzoom * y);		/* advance zoomed */
            for (fontmask = 0x80; fontmask; fontmask >>= 1) {
                uint16 c;
                int zoomx, zoomy;

                c = (fontbyte & fontmask) ? hicolor : hiback;
                /* if (ybase+yzoom >= YGFX) exit(142); */
                /* if (x+FONT_ZOOM >= XGFX) exit(42); */
                for (zoomx = 0; zoomx < FONT_ZOOM; zoomx++) {
                    for (zoomy = 0; zoomy < yzoom; zoomy++) {
                        putpixel16 (x, ybase+zoomy, c);	/* vertical scaling */
                    }
                    x++;			/* next pixel */
                } /* draw pixel as FONT_ZOOM * yzoom rectangle */

            } /* x loop with arbitrary other FONT_ZOOM * yzoom zoom */
        }
      } else {					/* else no Y zoom at all */
        ybase = y0 + y;				/* advance unzoomed */
        for (fontmask = 0x80; fontmask; fontmask >>= 1) {
          uint16 c;

          c = (fontbyte & fontmask) ? hicolor : hiback;
          /* if ((ybase+1 >= YGFX) || (x+FONT_ZOOM >= XGFX)) exit(43); */
          if (FONT_ZOOM == 2) {
              putpixel16 (x, ybase, c);		/* no line doubling */
              x++;				/* next pixel */
              putpixel16 (x, ybase, c);		/* no line doubling */
              x++;				/* next pixel */
          } else {
              int zoomx;

              for (zoomx = 0; zoomx < FONT_ZOOM; zoomx++) {
                  putpixel16 (x, ybase, c);	/* no line doubling */
                  x++;				/* next pixel */
              }					/* arbitrary X zoom */
          }
        } /* x loop without zoom */
      }						/* unzoomed Y */

  } /* y loop */

  return;
} /* put_char */
#endif	/* GRAPHICS */



/* masking support to be added later. Equal colors == no highlighting. */
/* string can be NULL. eyex... can be -1 to disable cursor & highlight */
/* (is "no cursor but highlighting" useful? Not supported yet...)      */
uint32 put_string(int x, int y, int screen, int normcolor, int focuscolor,
    int eyex, int eyey, char * string)  /* write string at coordinates */
{      /* on screen in color, highlighting focused, marking eye cursor */
  int maxx, maxy, fake, real_y;
#if GRAPHICS
  uint16 screen_buf_x, screen_buf_y;	/* screen buffer coordinates   */
  int magic;				/* for put_char mode selection */
#else
  uint16 screen_buf_p;                  /* offset into screen buffer   */
#endif
  
  if ( (screen < 0) || (screen > 1) )
      return get_ytime (); /* invalid screen */

  (void) select_screen (screen, 0);	/* uses fast lazy switching  */

  maxy = get_resolution_y (&maxx, &fake);
  /* maxy is min(screen0,screen1) limits, same for maxx -> caveat:   */
  /* if screen0 if graphical and screen1 is text mode, screen1 is 80x? */
  /* fake is true if actually two images have to share one screen,   */
  /* either a text (lfbSel == 0) or a graphical one. Access differs. */
  /* If two TEXT screens exist, they must have both the same width!  */

  if ( (x >= maxx) || (y >= maxy) )
      return get_ytime (); /* off scale */

#if GRAPHICS
  if ((lfbSel > 0) && (font_buf == NULL)) {
      __dpmi_regs r;				/* from dpmi.h */
      int n;
      uint16 fontseg, fontoff;

      fake_screen_bufs = malloc(132 * maxy * 2 * ((fake) ? 2 : 1));
      font_buf = malloc(256 * 16);	/* 256 char, 16 bytes each */

      if ( (fake_screen_bufs == NULL) || (maxx > 132) || (font_buf == NULL) )
          return get_ytime ();	/* initialization error */

      for (n = 0; n < (132 * maxy * ((fake) ? 2 : 1)); n++)
          fake_screen_bufs[n] = 0x2007;	/* init as grey on black space */

      r.x.ax = 0x1130;			/* get EGA+ font information */
      r.x.bx = 0x0600;			/* VGA+ 8x16 font */
      __dpmi_int(0x10, &r);		/* call video BIOS (see dpmi.h) */
      /* returns ES:BP pointer, CX bytes/char on screen, DL max row */
      fontseg = r.x.es;			/* the DOS segment of the pointer */
      /* int86x would try to translate segments :-(. We want raw values.  */
      fontoff = r.x.bp;			/* DOS offset of the font pointer */

      for (n = 0; n < (16*256); n++) {
          font_buf[n] = peekb (fontseg, fontoff + n);
      } /* copy 256 char entities, each 8x16 pixels (bits) big */

      for (n = 0; n < 16; n++) {
          font_buf[(0x10*16)+n] = font_buf[(((int)'*')*16)+n] >> 4;
          font_buf[(0x11*16)+n] = font_buf[(((int)'*')*16)+n] << 4;
      } /* create chars with left and right half of "*" for calibr8.c: */
      /* "\x10\x11" will look like "*" moved to the right by 0.5 chars */
      /* (this looks like a ">< of filled triangles" in text mode...)  */

  } /* in graphics mode */
#endif	/* GRAPHICS */

#if MONO_ATTRIBS_FOR_MONO
  if (ScreenSeg == 0xb000) {            /* mono -> adjust attributes */
      normcolor = (normcolor & 8) ? 1 : 7;
      focuscolor = (focuscolor & 8) ? 1 : 7;
      /* VERY simple mono cards can only check for: high nibble 8 means    */
      /* blink, all other means normal. Low nibble 1 means underline, 8    */
      /* means invisible, all other means normal. Not even a "bright" bit! */
  }
#endif

  real_y = y;				/* default: no fake dual */
#if GRAPHICS
  magic = maxx;				/* default: text mode */
	/* 0: normal GFX, -bufsize: upperdual GFX, +bufsize lowerdual GFX  */
	/* textwidth: normal TEXT. if txt+gfx, must use real width for txt */
  if ( (fake) && (lfbSel == 0) )
      real_y = y + y + screen;		/* fake dual-head TEXT mode! */
  if (lfbSel > 0) {
      if (fake) {			/* fake dual-head GRAPHICAL mode! */
          magic = (screen == 1) ? (132 * maxy) : (-132 * maxy);
      } else
          magic = (screen == 1) ? 80 : 0;	/* 80x? TEXT on screen 1, */
          				/* GRAPHICAL mode on screen 0 */
  }					/* any graphical mode */
#else
  if (fake) real_y = y + y + screen;    /* fake dual-head mode! */
#endif

  if ( (real_y == eye_cursor_old[screen].y) &&	/* only if in THIS line */
       eye_cursor_old[screen].item ) {		/* do we have to remove a cursor? */
#if GRAPHICS
      put_char(eye_cursor_old[screen].x, eye_cursor_old[screen].y, magic,
          eye_cursor_old[screen].item >> 8, eye_cursor_old[screen].item & 0xff);
#else
      poke (ScreenSeg, (eye_cursor_old[screen].x * 2) +
          (maxx * 2 * eye_cursor_old[screen].y),
          eye_cursor_old[screen].item);
#endif
      eye_cursor_old[screen].item = 0;	/* invalidate */
  }

#if GRAPHICS
  screen_buf_x = x;
  screen_buf_y = real_y;
#else
  screen_buf_p = (real_y * 2 * maxx) + x + x;
#endif

  if ( (string != NULL) && (eyey != y) ) {
      while ( (*string) && (x < maxx) ) {       /* fast writing loop */
	  if (*string != INVSPACE) {    /* invisible breaks are invisible */
#if GRAPHICS
	      put_char (screen_buf_x, screen_buf_y, magic,
	          ((*string == ' ') ? 7 : normcolor), *string );
	      screen_buf_x++;		/* advance cursor */
#else
	      poke (ScreenSeg, screen_buf_p,
	          (((*string == ' ') ? 7 : normcolor)<<8) | *string );
	      screen_buf_p += 2;	/* advance cursor */
#endif
	      x++;                      /* advance column */
	  };                            /* if invisible, do not write it. */
	  string++;                     /* go to next char in string...   */
      } /* fast writing loop */
      string = NULL;				/* skip slow writing loop */
  } /* fast writing loop */


  if (string != NULL) {				/* slow writing loop */
      uint16 hiword_p;				/* offset into screen buffer */
				      		/* only bit flag if GRAPHICS */
      char * hiword;				/* highlighted word */
      int hiword_x;				/* column on screen */

      hiword = string;                          /* which word to highlight */
#if GRAPHICS
      hiword_p = 1;				/* odd means "invalid" */
      						/* x in screen_buf_x here */
#else
      hiword_p = screen_buf_p | 1;		/* odd means "invalid" */
#endif
      hiword_x = x;

      while ( (*string) && (x < maxx) ) {

	  if ( (hiword_p & 1) &&		/* only if none found yet */
	       ( (*string == ' ') || (*string == INVSPACE) ) ) {
	      hiword = string;			/* possible word-in-focus */
#if GRAPHICS
	      hiword_p = 1;			/* not valid yet */
#else
	      hiword_p = screen_buf_p | 1;	/* not valid yet */
#endif
	      hiword_x = x;
	      /* pointers point to space BEFORE possible word in focus! */
	  }

	  if (x == eyex) {              /* we actually hit the eye cursor */
	      hiword_p &= ~1;           /* mark as valid */
	      /* also disables further search for other words in focus! */
	  } /* word in focus */

	  if (*string != INVSPACE) {    /* invisible breaks are invisible */
#if GRAPHICS
	      put_char (screen_buf_x, screen_buf_y, magic,
	          ((*string == ' ') ? 7 : normcolor), *string );
	      screen_buf_x++;		/* advance cursor */
#else
	      poke (ScreenSeg, screen_buf_p,
	          (((*string == ' ') ? 7 : normcolor)<<8) | *string );
	      screen_buf_p += 2;        /* advance cursor */
#endif
	      x++;                      /* advance column */
	  };                            /* if invisible, do not write it... */

	  string++;                     /* go to next char in string...   */

      } /* slow writing loop with highlighting preparations */

      if (!(hiword_p & 1)) {            /* is there a highlighted word? */
	  while ( *hiword && ( (*hiword == INVSPACE) || (*hiword == ' ') ) &&
	      (hiword_x < maxx) ) {
	      if (*hiword != INVSPACE) {        /* if a visible space */
#if GRAPHIC
		  /* nothing done - hiword_p is only a bit flag here, */
		  /* hiword_x should contain the required information */
#else
		  hiword_p += 2;
#endif
		  hiword_x++;
	      } /* invspaces do not advance the cursor / column */
	      hiword++;
	  } /* skip over extra spaces before the word */
	  if (hiword_x <= eyex) {       /* eyex in "visible part of word"? */
	      while ( *hiword && (*hiword != INVSPACE) && (*hiword != ' ') &&
		  (hiword_x < maxx) ) {
#if GRAPHICS
		  put_char (hiword_x, screen_buf_y, magic,
		      ((*hiword == ' ') ? 7 : focuscolor), *hiword );
		  /* (hiword_p is only a bit flag here) */
#else
		  poke (ScreenSeg, hiword_p,
		      (((*hiword == ' ') ? 7 : focuscolor)<<8) | *hiword );
		  hiword_p += 2;                /* advance cursor */
#endif
		  hiword_x++;                   /* advance column */
		  hiword++;                     /* advance in string */
	      } /* while not at right edge and no (inv)space or eof reached */
	  } /* if not only highlighting because of leading spaces */
      } /* highlighting is active */

  } /* any string with possible highlighting in the current line */


  if ( (real_y != eye_cursor_old[screen].y) &&	/* it could wait until now! */
       eye_cursor_old[screen].item ) {		/* do we have to remove a cursor? */
#if GRAPHICS
      put_char(eye_cursor_old[screen].x, eye_cursor_old[screen].y, magic,
          eye_cursor_old[screen].item >> 8, eye_cursor_old[screen].item & 0xff);
#else
      poke (ScreenSeg, (eye_cursor_old[screen].x * 2) +
          (maxx * 2 * eye_cursor_old[screen].y),
          eye_cursor_old[screen].item);
#endif
      eye_cursor_old[screen].item = 0;		/* invalidate */
  } /* late cursor removal because cursor is not on current line */
      /* (this is better because it results in less flicker) */
  

  /* draw eye cursor, after storing a backup of the char at that place */
  if ( (eyex >= 0) && (eyex < maxx) && (eyey >= 0) && (eyey < maxy) ) {
      real_y = eyey;
#if GRAPHICS
      if ( (fake) && (lfbSel == 0) )
          real_y = eyey + eyey + screen;	/* fake dual-head TEXT mode! */
#else
      if (fake) real_y = eyey + eyey + screen;	/* fake dual-head mode! */
#endif
      eye_cursor_old[screen].x = eyex;
      eye_cursor_old[screen].y = real_y;
#if GRAPHICS
      if ( (magic <= 0) || (magic > 132) ) {	/* have to read a virtual */
          int buf_base;				/* text mode buffer here? */

          buf_base = (screen > 0) ? (132 * maxy) : 0;
          eye_cursor_old[screen].item =
              fake_screen_bufs[buf_base + (real_y * 132) + eyex];
      } else {					/* real text, magic wide */
          eye_cursor_old[screen].item = peek (ScreenSeg,
              (real_y * 2 * magic) + eyex + eyex);
      }
#else
          eye_cursor_old[screen].item = peek (ScreenSeg,
              (real_y * 2 * maxx) + eyex + eyex);
#endif
      if (ScreenSeg == 0xb000) {                /* monochrome version? */
	  poke (ScreenSeg, (real_y * 2 * maxx) + eyex + eyex, (7<<8) | '*' );
      } else {
#if GRAPHICS
	  put_char (eyex, real_y, magic, focuscolor, '*');
#else
	  poke (ScreenSeg, (real_y * 2 * maxx) + eyex + eyex,
	      (focuscolor<<8) | '*' );
#endif
      } /* color version */
  } /* painting eye cursor */

  return get_ytime ();
} /* put_string */



/* clear the selected text screen to the selected background color */
/* -1 means "black or white depending on inversetext */
void clear_screen(int scr, int r, int g, int b)
{
  int tcolor, i, j, rows, cols, fake;
  char filled[170];
  char empty[170];
#if GRAPHICS
  uint16 cls_color = 0;
#endif

  tcolor = 7 + ((r >= 128) ? 0x40 : 0) + ((g >= 128) ? 0x20 : 0) +
      ((b >= 128) ? 0x10 : 0);	/* text is "grey on ..." */
      /* inversetext TEXT color is already handled by put_char */

#if GRAPHICS
  if ( (lfbSel > 0) && (scr == 0) ) {
      if (r < 0) r = (inversetext) ? 255 : 0;
      if (g < 0) g = (inversetext) ? 255 : 0;
      if (b < 0) b = (inversetext) ? 255 : 0;
      cls_color = RGB2c (r, g, b);
      last_cls_color = cls_color;	/* update put_char's notion of black */
      tcolor = 7;		/* use "custom black", no classic text colors */
  }
#endif

  rows = get_resolution_y (&cols, &fake); /* fake is nonzero if fake mode */
  if (cols > 160)
      return;

  for (i = 0; i < cols; i++) {
      filled[i] = 255;	/* filled is used because of lazy update put_string */
      			/* char 255 just is "screen changed", nondrawing... */
      empty[i] = ' ';
  }
  filled[cols] = 0;
  empty[cols] = 0;

  select_screen(scr, 0);

#if GRAPHICS
  if ( (lfbSel > 0) && (scr == 0) ) {	/* VESA subject screen? */
#if PROPER_CLEAR	/* see define at start of this file */
      if (fake) {	/* interlaced: have to clear whole VESA canvas */

          for (i = 0; i < vesamode.height; i++)
              for (j = 0; j < vesamode.width; j++)
                  putpixel16 (j, i, cls_color);

      } else {		/* non-interlaced: most of canvas will be cleared */
                        /* by the clear text screen loop later anyway...  */
#else
      {
#endif
          for (i = 0; i < vesamode.height; i++) {	/* right slack */
              j = (cols+1) * (8*FONT_ZOOM);
              while (j < vesamode.width) {
                  putpixel16(j, i, cls_color);
                  j++;
              }
          } /* Y clear loop: right slack */

          i = (rows * (16*FONT_ZOOM)) - (16*(FONT_ZOOM/2));
          while (i < vesamode.height) {			/* bottom slack */
              for (j = 0; j < vesamode.width; j++)
                  putpixel16 (j, i, cls_color);
              i++;
          } /* Y clear loop: bottom */

      }			/* fake / nonfake dual head VESA clear screen */
  } 			/* clear VESA subject screen */
#endif

  for (i = 0; i < rows; i++) {	/* clear text screen, flush text buffers */
      (void) put_string (0, i, scr,  tcolor, tcolor, -1, -1, filled);
      (void) put_string (0, i, scr,  tcolor, tcolor, -1, -1, empty);
  }

} /* clear_screen */

