/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* timer with ysec precision */

#include "tracker.h"	/* includes the other headers */
#include "time.h"	/* DJGPP uclock as fallback (1.1 MHz) */

#if 0 /* moved to header file */
static uint32 cpu_mhz = 0;	/* 0: needs init  1: use uclock  other: MHz */
#endif

int check_for_tsc (void) /* -1 if RDTSC possible */
{
  uint32 check;
  __asm__ (
      "pushfl\n"		/* save */
      "pushfl\n"
      "popl %%eax\n"		/* read flags */
      "movl %%eax, %%ebx\n"	/* save old version in ebx */
      "xor $0x200000, %%eax\n"
      "pushl %%eax\n"
      "popfl\n"			/* try to update flags */
      "pushfl\n"
      "popl %%eax\n"		/* read back flags */
      "xor %%ebx, %%eax\n"	/* compare to saved version */
      "popfl\n"			/* restore */
      : "=a" (check) :		/* nonzero if CPUID avail */ /* no inputs */
      : "bx"			/* changes (EAX and) EBX */
      );

  if (!(check & 0x200000)) {
     printf ("CPUID not supported on this CPU, using uclock.\n");
     return -1;
  }

  __asm__ (
      "xor %%eax, %%eax\n"	/* query for CPU level and string */
      "cpuid\n"			/* string is now in ECX:EDX:EBX, level EAX */
      "xor %%edx, %%edx\n"	/* for type 1 query this would have TSC flag */
      "xor %%ebx, %%ebx\n"	/* will hold family later */
      "cmp $1, %%eax\n"		/* at least type 1 flags avail? */
      "jb 0f\n"			/* skip forward if... */
      "mov $1, %%eax\n"		/* type 1 query now: flags */
      "cpuid\n"			/* EDX test 16 will be != 0 if TSC avail */
      "movl %%eax,%%ebx\n"	/* store family etc into EBX */
      "shr $8, %%ebx\n"		/* shift away model and revision */
      "and $15, %%ebx\n"	/* mask away newer sub-fields */
      "0:\n"
      "and $16, %%edx\n"
      "movl %%edx, %%eax\n"	/* store result in "EDX test 16" */
      "orl %%ebx, %%eax\n"	/* OR family into result */
      : "=a" (check) :		/* outputs EAX */ /* no inputs */
      : "bx", "cx", "dx" /* changes all general 32bit regs (returns EAX) */
      );

  printf (
      "Calibrating timer for i%d86 type CPU (with%s TSC for exact timing)...\n",
      (check & 15) > 9 ? 9 : (check & 15), (check & 16) ? "" : "out");
  if (!(check & 16))
      return -1; /* tell that no TSC found */

  /* if we reach this but still crash, RDTSC is blocked outside ring 0 */
  /* (only the ring 0 owner could give us permission then) */
  return 0;

} /* check_for_tsc */


uint32 get_ytime (void)
{
  uint32 ytime;

  if (!cpu_mhz) {	/* do we have to initialize uclock or TSC first? */
      uint32 tsc_hi, tsc_lo;
      uclock_t uclock_time;
      uint8 seconds;
      int tsc_flag;

      tsc_flag = check_for_tsc (); /* -1 if no RDTSC */

      out8 (0x70, 0);		/* select seconds access in CMOS */
      seconds = in8 (0x71);
      do {
          out8 (0x70, 0);	/* access seconds again */
      } while (in8 (0x71) == seconds); /* wait until seconds change */

      out8 (0x70, 10);		/* access status flags */
      while (in8 (0x71) & 0x80) {
          out8 (0x70, 10);	/* access status flag again */
      }			/* wait until time rollover has settled */

      if (tsc_flag < 0) {
          uclock_time = uclock ();
          tsc_lo = uclock_time & 0xffffffff;
          tsc_hi = uclock_time >> 32;
      } else
      __asm__ (			/* opcode is 0f 31 */
          "rdtsc"		/* read 64bit CPU cycle count EDX:EAX */
	  : "=a" (tsc_lo), "=d" (tsc_hi)	/* returns EAX and EDX */
	  :			/* no inputs */
	   			/* modifies nothing except ret. values */
	  );

      out8 (0x70, 0);		/* select seconds access in CMOS */
      seconds = in8 (0x71);
      do {
        out8 (0x70, 0);		/* access seconds again */
      } while (in8 (0x71) == seconds); /* wait until seconds change */

      if (tsc_flag < 0) {

          cpu_mhz = 1;
          printf("uclock frequency should be %lld Hz, found %lld Hz, okay.\n",
              (uclock_t)UCLOCKS_PER_SEC, uclock ());	/* always okay :-) */
          return (uint32) (
              (uclock () * (uclock_t)1000000) / UCLOCKS_PER_SEC );

      } else
      __asm__ (			/* opcode is 0f 31 */
          "movl %%edx, %%ecx\n"	/* old timestamp, hi */
          "movl %%eax, %%ebx\n"	/* old timestamp, lo */
	  "rdtsc\n"		/* read 64bit CPU cycle count EDX:EAX */
	  "subl %%ebx, %%eax\n"	/* find difference to previous sample */
	  "sbbl %%ecx, %%edx\n"	/* ... 64bit difference, that is      */
	  "movl $1000000, %%ebx\n" /* want to find MEGAcycle/s   */
	  "divl %%ebx"
	  : "=a" (cpu_mhz)	/* returns EAX */
	  : "d" (tsc_hi), "a" (tsc_lo)	/* input EDX:EAX old timestamp */
	  : "bx"		/* modifies (EAX, EDX) and EBX */
	  );

    if (cpu_mhz < 2) cpu_mhz = 2;	/* should never happen... */
    printf("Processor speed is %d Mcycle/s.\n", cpu_mhz);
      
  } /* initialization if needed */

  if (cpu_mhz < 2) {		/* uclock variant if no TSC available */
      uclock_t huge_time;	/* uclock_t is long long int, 64 bit  */

      huge_time = uclock () * 1000000;	/* up to 1.1M * 1M * seconds uptime */
				/* overflows after > 4000 hours */
      huge_time /= UCLOCKS_PER_SEC;	/* convert to microseconds */
      return (uint32) huge_time;	/* overflows every 72 minutes */

  } else
  __asm__ (
      "rdtsc\n"			/* get current TSC time */
      /* EBX already cpu_mhz */	/* divide by CPU clock in MHz */
      "pushl %%eax\n"
      "movl %%edx, %%eax\n"	/* high part only first */
      "movl $0, %%edx\n"
      "divl %%ebx\n"		/* EDX is now %-wrapped to 0..cpu_mhz-1 */
      "popl %%eax\n"
      "divl %%ebx\n"		/* EAX is now lower 32bit of ysec time */
      : "=a" (ytime)		/* returns EAX */
      :	"b" (cpu_mhz)		/* input EBX: cpu_mhz */
      : "dx"			/* modifies (EAX,) EDX (and EBX) */
      );

  return ytime;

} /* get_ytime */
