/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* common header file for tracker parts */
#ifndef TRACKER_H
#define TRACKER_H

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>	/* exit() */

#include <fcntl.h>	/* open() etc. */
#include <sys/stat.h>	/* file open modes */


#include "arch.h"	/* things to adjust to DJGPP / DOS, for example */

#define GRAPHICS 1	/* define to enable VESA graphics support */

#if GRAPHICS		/* used by: dualhead, screen, pcxshow, trials */
			/*  (in the latter only to clear the screen)  */
#define XGFX_HI 1280	/* 1024 or 1280 okay here */
#define YGFX_HI 800	/* do not use more than 32 * 25 here! */
			/* otherwise PCX canvas extends beyond TEXT canvas! */
#define XGFX_LO 1024	/* 1024 is enough for 64 columns of text */
#define YGFX_LO 736	/* 736 is 32 * 23 (odd number of lines) */

#define FONT_ZOOM 2	/* new 4/2005: zoom LOGGED coords and font in VESA  */
			/* mode - virtual font 8x16 differs from real font  */
			/* size used in dualhead.c and screen.c. Must be 2+ */
			/* for FAKE dual head VESA. Was hardcoded 2 before. */
			/* Calib needs zoom below 4, zoom 3 is unstable(?). */

#define XGFX vesamode.width
#define YGFX vesamode.height

#endif /* GRAPHICS */

/* Amount of bytes to store the data of all trials */
/* (later versions might save between trials if buffer is too full) */

#define LOGDATASIZE (4096*1024)	/* should be enough if >= 8 msec / event */


/* Definitions for general use and acquire.c */

#define YESBUTTON 1	/* yes or okay - subject (keyboard alias: y) */
#define NOBUTTON 2	/* no          - subject (keyboard alias: n) */
#define CALBUTTON 4	/* calibrate   - admin   (keyboard alias: c) */
#define NEXTBUTTON 8	/* next item   - admin   (keyboard alias: enter) */
#define ESCBUTTON 16	/* skip trial  - keyboard ESC */
#define ABORTBUTTON 32	/* leave to OS - keyboard ^C or ^D */

#define DEBOUNCE 200	/* default debounce time in milliseconds */

/* acquire.c */

#define TLBBIT 8	/* bit which is 1 while tracking (STATUS PORT) */
#define X_ADCBIT 0x10	/* bit which has X ADC data stream */
#define Y_ADCBIT 0x20	/* bit which has Y ADC data stream */


/* chunker.c and further stimulus processing: Definition of \_ char.  */
/* used in analyze.c, layout.c and screen.c as well: Invisible space. */

#define INVSPACE '\001'	/* char which marks an invisible area break   */


/* timer.c (uses CPU TimeStampCounter, or DJGPP uclock as fallback) */

uint32 cpu_mhz; /* auto-calibrates at first get_ytime call */

uint32 get_ytime (void); /* return low 32 bits of current ysecond uptime */


/* acquire.c (ret. values are all timestamps) */

int using_mouse(void);
    /* returns true if mouse instead of real eyetracker is used */

int track_lost(void);
    /* returns true if last call to get_raw_xy found track lost (or blink) */

uint32 get_raw_xy(int * x, int * y); /* return raw x/y coordinates */
    /* values will be in 0..4095 range for 12bit A/D module.       */
    /* x=-1 if A/D readout error happened - used to find the port. */

uint32 get_buttons(int * bits); /* return bitmask: pressed buttons */
    /* low bits are combination of ...BUTTON values defined above. */
    /* If keyboard key, buttons will have key ASCII << 8 and full  */
    /* key << 16 ORed in. */

uint32 debounce(int buttons, int msecs); /* wait until all  of the */
    /* given buttons have stayed unpressed for at least msecs...   */

int get_tracking (void);	/* returns nonzero while tracking  */
    /* tracking -> no blink and tracking not lost. */
    /* replaces "force x of get_raw_xy to -1 while track lost"...  */

/* dualhead.c */

uint32 select_screen(int screen, int enable); /* enable: -1, 0 or 1 */
    /* screen can be 0 or 1, 0 is "which is active at startup"...   */

int get_resolution_y(int * x, int * fake);	/* returns lines... */
    /* resolution can be in units of chars OR pixels. For screen 0. */
    /* fake is nonzero for fake dual head mode */


/* screen.c */

#if GRAPHICS			/* only used in graphics mode for now */
extern int inversetext;		/* affects only VESA mode on screen 0 */
#endif

extern uint16 ScreenSeg;	/* select_screen() can override this value. */
    /* (stored in dualhead) */	/* (only happens in legacy VGA + mono mode) */

uint32 put_string(int x, int y, int screen, int normcolor, int focuscolor,
    int eyex, int eyey, char * string); /* write string at coordinates */
    /* on screen in color, highlighting focused, updating eye cursor */

void clear_screen(int screen, int red, int green, int blue);
   /* use -1 for red, green and blue to use default: black or white, */
   /* depending on inversetext value. Clears the screen completely,   */
   /* including "lazy update" buffers and, if applicable, VESA canvas. */

/* calibr8.c */

#define CAL_BG 127	/* calibration background grey level, mainly for */
			/* graphics mode: 0 black, 255 white, -1 default */

void calibrate(void);	/* later versions could take parameters... */

/* updates drift correction values based on current gaze position compared */
/* to passed drift correction mark position in virtual (!) screen pixels   */
/* returns the amount of adjustment in the coordinate pair. Why not ;-).   */
/* returns nonzero if the drift correction attempt got accepted and used.  */
int drift_update(int * markx, int * marky);		/* new 4/2005 */

/* returns double precision pixel coordinate version of last get_xy result */
/* NOTE: this values use virtual screen size based on 8x16 font assumption */
void get_pixel_xy(double * x, double * y);      /* invalid if char x < 0!  */
    /* if get_xy returned a negative x, pixel coords will be invalid...    */
    /* this assumes a virtual 8x16 font, no matter what font is really on. */

uint32 get_xy(int * x, int * y); /* return text screen x/y coordinates */
    /* Needs calibration. If track loss or blink, x will be forced to -1. */


/* chunker.c */

int get_word(char * str, char * word, char ** afterword, int skipcomments);
    /* returns length and a new afterword pointer and word contents.   */
    /* copies first word from str, skipping initial space and comments */
    /* (not if skipcomments is 0). Word is empty at real end of line.  */
    /* Never copies more than 70 chars. Comment lines == 1 empty word. */


/* script.c: interface to main program - other functions in script.c   */
/* and some additional .c files are described in script.h, not here.   */

/* if reading is nonzero, assume READING script semantics */
void do_script(char * script, char * logbuf, int reading);
    /* the whole script thing: parse experiment script in 2 passes, */
    /* first the define lines, then the trial lines. See script.h!  */


/* fileread.c */

/* Read file filename - or if filename is NULL, prompt the user  */
/* to specify the name of some 'describe' file. Returns the file */
/* size and a pointer to the loaded file contents in RAM.        */
/* The buffer will be allocated and returned by the function.    */       
uint32 fileread(char * describe, char * filename, unsigned char ** buffer);


/* pcxshow.c */

#if GRAPHICS

/* xpp and ypp are the position: negative xpp is centered, negative ypp */
/* means ypp pixels distance from bottom. Returns: # of non-black slots */
int showpalette(int xpp, int ypp);

/* returns the index of the biggest value in field */
int findmax (uint32 * field, int fieldlen);

/* Draws a PCX image on a specified location on VESA screen   */
/* input: size of file in bytes, pointer to buffer, xlocation */
/* (negative: center), ylocation (negative: from bottom),     */
/* buffer for VESA mode sized pixel byte array (or NULL, data */
/* will be written to SCREEN coordinates in buffer!),         */
/* an uint32[256] histogram pointer or NULL. Returns 0 if ok. */
/* The real VESA screen is only updated if show is nonzero... */
int pcxshow(uint32 pcxsize, unsigned char * pcxdata,
    int xpos, int ypos, uint8 * rawpixels, uint32 * histo, int show);

/* recursive fill algorithm for pixelspace  */
/* "where to look", x, y, "what to fill in" */
/* 0 is "outside", 255 is "inside", rest is */
/* "already filled by other area" ...       */
/* "where to look" should be 0 initially!   */
/* depth_left is to limit nesting depth...  */
/* returned value is size of area in pixels */
uint32 doFill (int look, int x, int y, int last, int depth_left);

/* Analyze the VESA screen sized pixelspace uint8 array and */
/* mark all non-background objects by separate colors       */
/* Returns the number of separate areas found, max. 255     */
/* pixelspace values bground count as background. Value 255 */
/* is reserved and means uninitialized (also background)    */
/* The result is stored as area number for each pixelspace  */
/* pixel, area number 0 being reserved for the background.  */
/* Set visualize to have some visual output of the result.  */
/* returned min_y and max_y are the "objects found" range.  */
int imagechunker (uint8 bground, int visualize, int * min_y, int * max_y);

#endif /* GRAPHICS */

#endif
