/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* find an X*Y N bpp VESA graphics mode with LFB support, enable it */
/* (for the current screen - be careful! - and provide some generic */
/* access functions. And disable it again, of course...             */

#define DUMMYGFX 0

#include "vesa.h"	/* header for all VESA related stuff  */
			/* includes tracker.h and arch.h etc. */

#include <dpmi.h>	/* to call the BIOS -- or use dos.h */

/* in vesa.h: static struct VESA_MODE vesamode; */


/* Enter VESA graphics mode with selected (pixel or char) size... */
/* width 0 means leave VESA and return to BIOS 80x25 text mode 3. */
/* negative height means "min -height". bits is the color depth.  */
/* negative bits allows status messages to be printf()ed...       */
/* ADDED 11/2004 --> */
/* if khz is set, tries to configure to khz kHz hrefresh rate, if */
/* VBE 3.0 present (newer nVidia / 3dfx), but max 140 Hz vrefresh */
/* returns nonzero value on failure */
int vesainit(int width, int height, int bits, int khz)
{
  __dpmi_regs r;
  int segV, selV;

  if (!width) {
#if DUMMYGFX
#else
    r.x.ax = 0x0003;	/* AH=0 iss mode change, AL=3 is 80x25 text mode */
    __dpmi_int(0x10, &r);
#endif
    vesamode.width = 0;
    vesamode.height = 0;
    return 0;				/* worked (we assume)! */
  } /* 80x25 mode */

  segV = __dpmi_allocate_dos_memory(512>>4, &selV);
		  	/* size unit is 16 bytes (paragraphs) */
  if (segV == -1) {
    if (bits < 0) printf("Out of DOS memory in vesainit!\n");
    return -1;				/* out of DOS memory */
  }

  r.x.ax = 0x4f00;			/* VESA install check, get info */
  r.x.es = segV;			/* DOS buffer segment */
  r.x.di = 0;				/* buffer offset */
  pokeb(segV, 0, 'V');
  pokeb(segV, 1, 'B');
  pokeb(segV, 2, 'E');
  pokeb(segV, 3, '2');

  __dpmi_int(0x10, &r);	/* video BIOS */

  if (r.x.ax != 0x004f)	{	/* failed? */
    if (bits < 0) printf("No VESA drivers present!\n");
    __dpmi_free_dos_memory(selV);	/* release buffer */
    return -2;				/* no VESA */
  }

  {			/* call worked - copy interesting parts */
    VESA_CARD vesainfo;

    uint16 modelist[256];
    int i, modeseg, modeoff;

    _movedatab(selV, 0, _my_ds(), (uint32)&(vesainfo.magic[0]),
      sizeof(vesainfo));	/* or use (uint32)&vesainfo.magic ... */

    if ( (strncmp(vesainfo.magic, "VESA", 4)) ||
         (vesainfo.version < 0x0200) ) {
      if (bits < 0) printf("At least VESA VBE 2.0 required, only %x.%x found!\n",
        (vesainfo.version>>8), (vesainfo.version & 0xff));
      __dpmi_free_dos_memory(selV);	/* release buffer */
      return -3;			/* too old VESA version */
    }
    if ((vesainfo.version < 0x0300) && (khz)) {	/* new 11/2004 */
      if (bits < 0) printf("Not VESA VBE 3.0 - cannot set refresh rate.\n");
      khz = -khz;			/* just simulate */
    }

    modeseg = vesainfo.modelistPTR >> 16;
    modeoff = vesainfo.modelistPTR & 0xffff;

    for (i = 0; ( (i < 255) && (modeoff < 0xfffe) ); i++) {
      modelist[i] = peek(modeseg, modeoff);

      if (modelist[i] == 0xffff)
        break;				/* end of list */

      modeoff += 2;
    } /* copy mode list */

    if (bits < 0) printf("VESA %x.%x, %d modes: ",
        vesainfo.version>>8, vesainfo.version & 0xff, i);

    r.x.bx = 0;				/* mode not found yet */

    for (i = 0; (modelist[i] != 0xffff); i++) {
      r.x.ax = 0x4f01;			/* get VESA mode info */
      r.x.es = segV;			/* buffer DOS segment */
      r.x.di = 0;			/* buffer offset */
      r.x.cx = modelist[i];		/* mode */

      if (r.x.cx == 0xffff)
        break;				/* reached end of list */

      __dpmi_int(0x10, &r);		/* video BIOS */

      if (r.x.ax != 0x004f)
        continue;			/* failed? */

      _movedatab(selV, 0, _my_ds(), (uint32)&(vesamode.modeflags),
        sizeof(vesamode));

      if ( ((vesamode.modeflags & 128) == 0) ||
           (vesamode.width != width) ||
           (vesamode.bits_pixel != ((bits<0) ? (-bits) : bits)) ||
           ((height>0) && (vesamode.height != height)) ||
           ((height<0) && (vesamode.height < (-height))) )
        continue;	/* no LFB support or wrong size / depth */
	/* we do not care for color, BIOS support, graphics ... */
        /* Would be better if we checked modeflags & 0x19 for being 0x19: */
        /* 1 for "1" (supported by current hardware - for VBE/AF dual???) */
        /* 8 for "color", 0x10 for "graphics" ... */

      if ( (vesamode.memorymode != 6) && (vesamode.memorymode != 4) &&
           (vesamode.memorymode != 3) && (vesamode.memorymode != 0) )
        continue;	/* only allow text, EGA, MCGA and HiColor/TrueColor */
	/* for mode 4, MCGA, planes should be 1 */

      r.x.bx = modelist[i];		/* found! */
      break;
    } /* find suitable mode (loop) */

    __dpmi_free_dos_memory(selV);	/* release buffer */

    if (!r.x.bx) {
      if (bits < 0) printf("No %dx%d %d bpp LFB mode found%s!\n",
        width, ((height<0) ? (-height) : height), -bits,
        ((height<0) ? " (or higher)" : "") );
      return -4;			/* no suitable VESA mode */
    }

    if (bits < 0) printf("%dx%d (%d bpl, %d bpp) LFB (at 0x%x) mode 0x%x.\n",
        vesamode.width, vesamode.height,
        vesamode.bytes_line, vesamode.bits_pixel, vesamode.lfbPTR, r.x.bx);
  } /* end of variable scope for mode list processing */

  r.x.ax = 0x4f02;			/* select VESA mode */
  r.x.bx |= 0x4000;			/* select LFB variant */

#if DUMMYGFX
#else
  __dpmi_int(0x10, &r);			/* video BIOS */

  if (r.x.ax != 0x004f) {
    if (bits < 0) printf("VESA mode set failed!\n");
    (void) vesainit (0, 0, 0, 0);	/* return to text mode */
    return -5;				/* VESA mode set failed */
  }
  if (khz) {				/* go for VESA VBE 3.0 refresh set */
    if ((khz = vbe3refresh(khz, r.x.bx)) != 0) {
      if (bits < 0) {
        printf("Refresh rate setup failed, code %d", khz);
      }  /* can continue anyway - will just not have custom refresh rate */
      /* might want to re-init mode, though... only needed if BIOS buggy */
    } /* refresh rate set failed */
  } /* refresh rate set requested */	/* new 11/2004 */
#endif

  {					/* variable scope: LFB init */
    __dpmi_meminfo memory_mapping;

    memory_mapping.address = vesamode.lfbPTR;	/* physical linear address */
    memory_mapping.size = ( (vesamode.bytes_line * vesamode.height)
      + 65535) & (uint32)0xffff0000;	/* round up to multiple of 64k */
      /* or use total memory size (64k units) from int 10.4f00 */


    if (__dpmi_physical_address_mapping(&memory_mapping) != 0) {
      (void) vesainit (0, 0, 0, 0);	/* return to text mode */
      if (bits < 0)
        printf ("physical address mapping @ %lx, size %lx failed.\n",
          memory_mapping.address, memory_mapping.size);
      return -6;			/* LFB init failed */
    }	/* else memory_mapping.address updated: logical linear address now */
	/* (in first 1 MB, physical / logical are identity-mapped, though) */

    __dpmi_lock_linear_region(&memory_mapping);

    if (!lfbSel) {
      lfbSel = __dpmi_allocate_ldt_descriptors(1);	/* alloc 1 slot */
      if (lfbSel <= 0) {
        (void) vesainit (0, 0, 0, 0);	/* return to text mode */
        return -7;			/* could not get LDT slot for LFB */
      }
    }

    if (bits < 0) {
      if (vesamode.lfbPTR != memory_mapping.address) {
        printf("LFB seg 0x%x at: 0x%x->0x%lx size: %ldk RGB: %d%d%d\n", lfbSel,
          vesamode.lfbPTR,  memory_mapping.address, memory_mapping.size/1024,
          vesamode.redBits, vesamode.greenBits, vesamode.blueBits);
      } else {
        printf("LFB seg 0x%x at: 0x%x size: %ldk RGB: %d%d%d\n", lfbSel,
          vesamode.lfbPTR, memory_mapping.size/1024,
          vesamode.redBits, vesamode.greenBits, vesamode.blueBits);
      }
    } /* bits < 0 */

    if (height < 0) {	/* pick center slice of selected height */
      vesamode.lfbPTR += (vesamode.bytes_line *
        ( (vesamode.height - (-height)) / 2 ) );	/* cheat move */
      vesamode.height = -height;			/* cheat move */
      memory_mapping.address = vesamode.lfbPTR;		/* update */
      memory_mapping.size = vesamode.height * vesamode.bytes_line;
    } /* height centering (added 10/2004) */

    __dpmi_set_segment_base_address(lfbSel, memory_mapping.address);
    __dpmi_set_segment_limit(lfbSel, memory_mapping.size - 1);

  }					/* variable scope: LFB init */

  return 0;				/* everything worked :-) */
} /* vesainit */

