{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$I INFERNO.DEF}
unit FFBiltIn; { Standard Built in File Formats }

interface

uses Inferno;

var
    DRV_Handler : PFormatHandler;
    FNT_Handler : PFormatHandler;
    IGG_Handler : PFormatHandler;
    IGS_Handler : PFormatHandler;
    IGA_Handler : PFormatHandler;

procedure RegisterFormatBuiltIn;

implementation

function DRV_Check(FileName : String) : boolean; far;
var
    H : PDriver;
    F : boolean;
begin
    F := False;
    ClearError;
    H := FileRead(FileName, 0, 64);
    if Assigned(H) then begin
        F := (H^.Platform = fiInferno) and
             (H^.VersionCompat = DriverVersionCompat) and
             (Pos(',' + H^.Class + ',', ',' + dcAllTypes + ',') > 0);
        FreeMem(H, 64);
    end;
    DRV_Check := NoError and F;
    ClearError;
end;

function DRV_Process(var P : Pointer; var Size : word) : integer; far;
begin
    ClearError;
    if not ConfigureDriver(PDriver(P)) then begin
        FreeMem(P, Size);
        P := nil;
        Size := 0;
    end;
    DRV_Process := GetError;
end;

function DRV_Load(FileName : String; var P : Pointer; var Size : LongInt) : integer; far;
var
    WS : word;
begin
    Size := 0;
    ClearError;
    if FileLoad(FileName, P, WS) then begin
        if not ConfigureDriver(PDriver(P)) then begin
            FreeMem(P, WS);
            P := nil;
        end else
            Size := WS;
    end;
    DRV_Load := GetError;
end;

type
    PFontFile = ^TFontFile;
    TFontFile = record { 16 byte header }
        FileID      : array[0..5] of char; { 'BITFNT' }
        Format      : byte;                { currently only 0 or 1 }
        Width       : byte;                { Width of characters in pixels }
        Height      : byte;                { Height of characters in pixels }
        MonoWidth   : byte;                { Monospace display width in pixels }
        Vertical    : byte;                { Vertical Offset }
        Spacing     : byte;                { Space between non-monospace chars }
        Reserved    : DWord;
    end; { if format 1, then immediately followed a TFontOffsets array }

function FNT_Check(FileName : String) : boolean; far;
var
    H : PFontFile;
    F : boolean;
    S : String;
    Zzz : LongInt; { I fell asleep when I picked this variable name }
begin
    F := False;
    ClearError;
    H := FileRead(FileName, 0, Sizeof(TFontFile));
    if Assigned(H) then begin
        S[0] := Char(Length(fiBitFontFile));
        Move (H^.FileID, S[1], Length(S));
        if (S = fiBitFontFile) then begin
            F := (H^.Format < 2) and
                (H^.Width <= 32) and
                (H^.Height <= 32);
        end else begin
            Zzz := FileSizeOf(FileName);
            F := (Zzz > 0) and
                (Zzz mod 256 = 0) and
                (Zzz div 256 >= 8) and
                (Zzz div 256 <= 32);
        end;
        FreeMem(H,  Sizeof(TFontFile));
    end;
    FNT_Check := NoError and F;
    ClearError;
end;

function FNT_Save(FileName : String; P : Pointer; Size : LongInt) : integer; far;
 var
    Header : TFontFile;
begin
    ClearError;
    if (PFont(P)^.Width <> 8) or (PFont(P)^.Height > 25) then begin
        FillChar(Header, Sizeof(Header), 0);
        move(fiBitFontFile[1], Header.FileID, Length(fiBitFontFile));
        with PFont(P)^ do begin
            { for now, save as Format 0, without Offset Data. Maybe if
              ImgEdit ever supports manual adjustment of that Data, then
              start saving Format 1 so it is preserved. Otherwise, Format
              loading will just make it up. }
            Header.Format := 0;
            Header.Width := Width;
            Header.Height := Height;
            Header.MonoWidth := MonoWidth;
            Header.Vertical := Vertical;
            Header.Spacing := Spacing;
        end;
        if FileSave(FileName, @Header, Sizeof(Header)) then
            FileAppend(FileName, PFont(P)^.BitMap, FontBitmapSize(PFont(P)));
    end else begin
        FileSave(FileName, PFont(P)^.BitMap, FontBitmapSize(PFont(P)));
    end;
    FNT_Save := GetError;
end;

function FNT_Process(var P : Pointer; var Size : word) : integer; far;
var
    X : Pointer;
    H : PFontFile;
    Font : PFont;
    Sz, BSz : Word;
    S : String;
begin
    ClearError;
    Font := New(PFont);
    if Assigned(Font) then
        Font^.BitMap := nil;
    H := P;
    Sz := Size;
    if Assigned(Font) then with Font^ do begin
        S[0] := Char(Length(fiBitFontFile));
        Move (H^.FileID, S[1], Length(S));
        if (S = fiBitFontFile) then begin
            Width       := H^.Width;
            Height      := H^.Height;
            MonoWidth   := H^.MonoWidth;
            Vertical    := H^.Vertical;
            Spacing     := H^.Spacing;
            ByteWidth   := Width div 8;
            if Width mod 8 <> 0 then Inc(ByteWidth);
            X := H;
            IncPtr(X, Sizeof(TFontFile));
            BSz := Sz - Sizeof(TFontFile);
            if H^.Format = 1 then
                Dec(BSz, SizeOf(TFontOffsets));
            GetMem(BitMap, BSz);
            if Assigned(Bitmap) then begin
                if H^.Format = 0 then begin
                    Move(X^, BitMap^, BSz);
                    FontOffsetReset(Font);
                end else begin
                    Move(X^, Offsets, SizeOf(TFontOffsets));
                    IncPtr(X, SizeOf(TFontOffsets));
                    Move(X^, BitMap^, BSz);
                end;
            end;
        end else begin
            { plain 8x something bitmap font }
            Width := 8;
            ByteWidth :=1;
            Height := Sz div 256;
            Vertical := 0;
            Spacing := 1;
            MonoWidth := Width;
            BSz := Sz;
            GetMem(BitMap, BSz);
            if Assigned(Bitmap) then begin
                Move(H^, BitMap^, BSz);
                FontOffsetReset(Font);
            end;
        end;
    end;
    if Assigned(H) then
        FreeMem(H, Sz);
    if IsError then begin
        if Assigned(Font) and Assigned(Font^.BitMap) then
            FreeMem(Font^.BitMap, BSz);
        if Assigned(Font) then
            Dispose(Font);
        Font := nil;
        BSz := 0;
    end;
    P := Font;
    Size := BSz;
    FNT_Process := GetError;
end;

function FNT_Load(FileName : String; var P : Pointer; var Size : LongInt) : integer; far;
var
    FS : Word;
begin
    if FileLoad(FileName, P, FS) then begin
        if FNT_Process(P, FS) = 0 then
            Size := FS;
    end;
    if IsError then
        Size := 0;
    FNT_Load := GetError;
end;


const
    biTerminate     = 0;
    biHeader        = 1;
    biPalettes      = 2;
    biImage         = 3;
    biSpriteControl = 10;
    biSpriteImage   = 11;
    biSpriteBMask   = 12;
    biSpriteIMask   = 13;
    biUser          = 100;
    { 6-255 reserved for future use }

    IGGFileID       : Str6 = 'IMAGEX';
    IGSFileID       : Str6 = 'SPRITE';

type
    TIGGFileBlock = record
        Size : LongInt;
        ID   : byte;
    end;
    TIGGFileHeader = record
        Block    : TIGGFileBlock;
        FileID   : array[0..5] of byte; { IMAGEX / SPRITE }
        PalType  : byte;                { >0 first use palette table revision }
        BBP      : byte;                { Bits Per Pixel }
        Reserved : array[0..2] of byte;
    end;
    TIGGBitMapHeader = record
        Block       : TIGGFileBlock;
        Width       : word;             { width }
        Height      : word;             { height }
        OffsetX     : integer;          { Display Offset X }
        OffsetY     : integer;          { Display Offset Y }
        Encoding    : byte;             { Compression used }
        Reserved    : word;
    end;
    TIGGPalettes = record
        Block       : TIGGFileBlock;
        Colors      : word;
        Reserved    : byte;
        Palettes    : TRGBPalettes;
    end;
    TIGSSpriteControl = record
        Block       : TIGGFileBlock;
        Width       : word;             { width }
        Height      : word;             { height }
        HotSpot     : TPoint;
        Count       : integer;
        { Suggestions }
        Initial     : integer;          { Initial Sprite Index }
        FPS         : integer;          { >0, Sprite is Animated }
        Kind        : TSpriteKind;      { Sprite Kind }
        Level       : byte;             { Sprite Drawing Level }
    end;
    TIGSSpriteData = record
        Block : TIGGFileBlock;
        Index : word;
        Sequence : Word;
        Compression : byte;
    end;

function CalcPalettes(var BP : TIGGPalettes) : integer;
var
    PP : TRGBPalettes;
    I : integer;
begin
    FillChar(BP, Sizeof(BP), 0);
    CalcPalettes := 0;
    with BP do begin
        Colors      := 256;
        Reserved    := 0;
        Video^.GetPalettes(Palettes);
        Block.Size  := Sizeof(TRGBPalette) * BP.Colors +
             Sizeof(BP.Colors) + Sizeof(BP.Reserved) + Sizeof(BP.Block);
        Block.ID    := biPalettes;
    end;
    for I := 1 to Video^.ColorProfiles do begin
        Video^.GetColorProfile(I, PP);
        if CompareBytes(PP, BP.Palettes, Sizeof(PP)) = 0 then begin
            CalcPalettes := I;
            Break;
        end;
    end;
end;

function IGG_Check(FileName : String) : boolean; far;
var
    F : boolean;
    S : String;
    X : ^TIGGFileHeader;

begin
    F := False;
    ClearError;
    X := FileRead(FileName, 0, SizeOf(TIGGFileHeader));
    if Assigned(X) then begin
        S[0] := Chr(Length(IGGFileID));
        move(X^.FileID, S[1], Length(S));
        F := (S = IGGFileID) and (X^.Block.ID = biHeader) and (X^.BBP = 8);
        FreeMem(X, SizeOf(TIGGFileHeader));
    end;
    IGG_Check := NoError and F;
end;

function IGG_Save(FileName : String; P : Pointer; Size : LongInt) : integer; far;
var
    FH : TIGGFileHeader;
    BH : TIGGBitMapHeader;
    BP : TIGGPalettes;
    BT : TIGGFileBlock;
    BI : PDIBitMap;
    Image : PImage;
begin
    Image := P;
    FileName := FileBase(FileName) + 'IGG';
    ClearError;
    FillChar(FH, Sizeof(FH), 0);
    FillChar(BH, Sizeof(BH), 0);

    with FH do begin
        Block.Size  := Sizeof(FH);
        Block.ID    := biHeader;
        move(IGGFileID[1], FileID, SizeOf(FileID));
        PalType     := CalcPalettes(BP);
        BBP         := 8;
    end;
    BI := ImageToDIBitMap(Image);
    DIBitMapImplode(BI, icaDataRun);
    if NoError then FileSave(FileName, @FH, FH.Block.Size);
    if NoError and (FH.PalType = 0) then
        FileAppend(FileName, @BP, BP.Block.Size);
    with BH do begin
        Width       := Image^.Width;
        Height      := Image^.Height;
        OffsetX     := 0;
        OffsetY     := 0;
        Encoding    := BI^.Compression;
        Reserved    := 0;
        Block.ID    := biImage;
        Block.Size  := + Sizeof(BH) + DIBitMapSizeData(BI);
    end;

    if NoError then FileAppend(FileName, @BH, Sizeof(BH));
    if NoError then FileAppend(FileName, @BI^.ImageData, DIBitMapSizeData(BI));

    FreeDIBitMap(BI);

    BT.Size   := 0;
    BT.ID     := biTerminate;
    if NoError then FileAppend(FileName, @BT, Sizeof(BT));

    IGG_Save := GetError;
end;

function IGG_Process(var P : Pointer; var Size : word) : integer; far;
var

    FS : word;
    BI : PDIBitMap;
    Image : PImage;
    S : String;
    X : Pointer;
    Pro : word;
    Pal : TRGBPalettes;

begin
    Image := nil;
    ClearError;
    X := P;
    S[0] := Chr(Length(IGGFileID));
    move(TIGGFileHeader(X^).FileID, S[1], Length(S));
    if (S <> IGGFileID) or (TIGGFileBlock(X^).ID <> biHeader) then
        SetError(erInvalid_File_Format);
    Pro := 0;
    Video^.GetColorProfile(Pro, Pal);
    while NoError and (TIGGFileBlock(X^).ID <> biTerminate) and
    (TIGGFileBlock(X^).Size > 0) do begin
        if Ofs(X^) - Ofs(P^) >= Size then begin
            {$IFDEF LOGS}
                LOG('IGG not properly terminated. ');
            {$ENDIF}
            break;
        end;
        case TIGGFileBlock(X^).ID of
            biTerminate : Break;
            biHeader    : begin
                Pro := TIGGFileHeader(X^).PalType;
                Video^.GetColorProfile(Pro, Pal);
            end;
            biUser      : begin end; { ignore }
            biPalettes  : begin
                move(TIGGPalettes(X^).Palettes, Pal, TIGGPalettes(X^).Colors * Sizeof(TRGBPalette));
                if (FormatPaletteMode = ipmComplete) or (FormatPaletteMode = ipmOverride) then
                    Video^.SetPalettes(Pal);
            end;
            biImage     : begin
                Video^.FreeImage(Image);
                BI := NewDIBitmap(TIGGBitMapHeader(X^).Width, TIGGBitMapHeader(X^).Height);
                if Assigned(BI) then begin
                    BI^.Compression := TIGGBitMapHeader(X^).Encoding;
                    Move(Bytes(X^)[Sizeof(TIGGBitMapHeader)], BI^.ImageData,
                        DIBitMapSizeData(BI));
                    DIBitMapExplode(BI);
                    Image := DIBitMapToImage(BI);
                    FreeDIBitMap(BI);
                    if ImageCompress then
                        Video^.ImageImplode(Image);
                end;
            end;
        else
            { ignore }
        end;
        IncPtr(X, TIGGFileBlock(X^).Size);
    end;
    FreeMem(P, Size);
    Size := 0;
    P := nil;
    if IsError then begin
        Video^.FreeImage(Image)
    end else begin
        P := Image;
        Size := Video^.ImageSizeOf(Image);
    end;
    IGG_Process := GetError;
end;

function IGG_Load(FileName : String; var P : Pointer; var Size : LongInt) : integer; far;
var
    FS : Word;
begin
    if FileLoad(FileName, P, FS) then begin
        if IGG_Process(P, FS) = 0 then
            Size := FS;
    end;
    if IsError then
        Size := 0;
    IGG_Load := GetError;
end;

function IGS_Check(FileName : String) : boolean; far;
var
    F : boolean;
    S : String;
    X : ^TIGGFileHeader;
begin
    F := False;
    ClearError;
    X := FileRead(FileName, 0, SizeOf(TIGGFileHeader));
    if Assigned(X) then begin
        S[0] := Chr(Length(IGSFileID));
        move(X^.FileID, S[1], Length(S));
        F := (S = IGSFileID) and (X^.Block.ID = biHeader) and (X^.BBP = 8);
        FreeMem(X, SizeOf(TIGGFileHeader));
    end;
    IGS_Check := NoError and F;
end;

function IGS_Process(var P : Pointer; var Size : word) : integer; far;
var

    FS : word;
    BI : PDIBitMap;
    BB : PMask;
    Sprite : PSprite;
    S : String;
    X : Pointer;
    Pro, CP : word;
    Pal : TRGBPalettes;
begin
    Sprite := nil;
    ClearError;
    X := P;
    S[0] := Chr(Length(IGSFileID));
    move(TIGGFileHeader(X^).FileID, S[1], Length(S));
    if (S <> IGSFileID) or (TIGGFileBlock(X^).ID <> biHeader) then
        SetError(erInvalid_File_Format);
    Pro := 0;
    Video^.GetColorProfile(Pro, Pal);
    while NoError and (TIGGFileBlock(X^).ID <> biTerminate) and
    (TIGGFileBlock(X^).Size > 0) do begin
        if Ofs(X^) - Ofs(P^) >= Size then begin
            {$IFDEF LOGS}
                LOG('IGS not properly terminated. ');
            {$ENDIF}
            break;
        end;
        case TIGGFileBlock(X^).ID of
            biTerminate : Break;
            biHeader    : begin
                Pro := TIGGFileHeader(X^).PalType;
                Video^.GetColorProfile(Pro, Pal);
            end;
            biUser      : begin end; { ignore }
            biPalettes  : begin
                move(TIGGPalettes(X^).Palettes, Pal, TIGGPalettes(X^).Colors * Sizeof(TRGBPalette));
                if (FormatPaletteMode = ipmComplete) or (FormatPaletteMode = ipmOverride) then
                    Video^.SetPalettes(Pal);
            end;
            biSpriteControl : begin
                Video^.FreeSprite(Sprite);
                with TIGSSpriteControl(X^) do begin
                    {$IFDEF LOGS} Log(IntStr(Width) + ':' + IntStr(Height) + '*' + IntStr(Count)); {$ENDIF}
                    Sprite := Video^.NewSprite(Width, Height, Count, False);
                    if not Assigned(Sprite) then Break;
                    Sprite^.Kind := Kind;
                    Sprite^.Level := Level;
                    if FPS > 0 then
                        Sprite^.Animate := -1
                    else
                        Sprite^.Animate := 0;
                    Sprite^.HotSpot := HotSpot;
                    if Initial >= Count then begin
                        SetError(erRange_Check_Error);
                        Break;
                    end;
                    Sprite^.Index := Initial;
                end;
            end;
            biSpriteImage : begin
                if Not Assigned(Sprite) then begin
                    SetError(erInvalid_File_Format);
                    Break;
                end;
                with TIGSSpriteData(X^) do begin
                    {$IFDEF LOGS} Log('Index ' + IntStr(Index) + ':' + IntStr(Sequence) +
                        ',' + IntStr(Block.Size)); {$ENDIF}
                    if Index >= Sprite^.Count then begin
                        SetError(erRange_Check_Error);
                        Break;
                    end;
                    Sprite^.Sprites^[Index].Sequence := Sequence;
                    BI := NewDIBitmap(Sprite^.Area.Right - Sprite^.Area.Left + 1,
                        Sprite^.Area.Bottom - Sprite^.Area.Top + 1);
                    if Not Assigned(BI) then Break;
                    BI^.Compression := Compression;
                    if not Assigned(BI) then Break;
                    CP := Block.Size - Sizeof(TIGSSpriteData);
                    if CP > DIBitMapSizeData(BI) then begin
                        SetError(erRange_Check_Error);
                        exit;
                    end;
                    Move(Bytes(X^)[Sizeof(TIGSSpriteData)], BI^.ImageData, CP);
                    Video^.FreeImage(Sprite^.Sprites^[Index].Image);
                    DIBitMapExplode(BI);
                    Sprite^.Sprites^[Index].Image := DIBitMapToImage(BI);
                    FreeDIBitMap(BI);
                    if ImageCompress then
                        Video^.ImageImplode(Sprite^.Sprites^[Index].Image);
                    {$IFDEF LOGS} Log('Image ' + PtrHex(Sprite^.Sprites^[Index].Image) +
                        ' Index ' + IntStr(Index) + ':' + IntStr(Sequence) +
                        ',' + IntStr(Block.Size)); {$ENDIF}

                end;
            end;
            biSpriteIMask : begin
                if Not Assigned(Sprite) then begin
                    SetError(erInvalid_File_Format);
                    Break;
                end;
                with TIGSSpriteData(X^) do begin
                    if Index >= Sprite^.Count then begin
                        SetError(erRange_Check_Error);
                        Break;
                    end;
                    BI := NewDIBitmap(Sprite^.Area.Right - Sprite^.Area.Left + 1,
                        Sprite^.Area.Bottom - Sprite^.Area.Top + 1);
                    if not Assigned(BI) then Break;
                    BI^.Compression := TIGSSpriteData(X^).Compression;
                    CP := Block.Size - Sizeof(TIGSSpriteData);
                    if CP > DIBitMapSizeData(BI) then begin
                        SetError(erRange_Check_Error);
                        exit;
                    end;
                    Move(Bytes(X^)[Sizeof(TIGSSpriteData)], BI^.ImageData, CP);
                    Video^.FreeImage(Sprite^.Sprites^[Index].IMask);
                    DIBitMapExplode(BI);
                    Sprite^.Sprites^[Index].IMask := DIBitMapToImage(BI);
                    FreeDIBitMap(BI);
                    if ImageCompress then
                        Video^.ImageImplode(Sprite^.Sprites^[Index].IMask);
                end;
            end;
            biSpriteBMask : begin
                if Not Assigned(Sprite) then begin
                    SetError(erInvalid_File_Format);
                    Break;
                end;
                with TIGSSpriteData(X^) do begin
                    if Index >= Sprite^.Count then begin
                        SetError(erRange_Check_Error);
                        Break;
                    end;
                    {$IFDEF LOGS}
                    Log('Load Mask #' +IntStr(Index));
                    {$ENDIF}
                    Video^.FreeMask(Sprite^.Sprites^[Index].BMask);
                    BB := Video^.NewMask(
                        Sprite^.Area.Right - Sprite^.Area.Left + 1,
                        Sprite^.Area.Bottom - Sprite^.Area.Top + 1);
                    if not Assigned(BB) then Break;
                    BB^.Compression := Compression;
                    CP := Block.Size - Sizeof(TIGSSpriteData);
                    if CP > Video^.MaskSizeData(BB) then begin
                        SetError(erRange_Check_Error);
                        exit;
                    end;
                    Move(Bytes(X^)[Sizeof(TIGSSpriteData)], BB^.ImageData, CP);
                    Video^.MaskExplode(BB);
                    Sprite^.Sprites^[Index].BMask := Video^.CloneMask(BB);
                    Video^.FreeMask(BB);
                    {$IFDEF LOGS}
                    Log('end load mask');
                    {$ENDIF}
                    { if ImageCompress then
                        Video^.MaskImplode(Sprite^.Sprites^[Index].BMask); }
                end;
            end;

        else
            { ignore }
        end;
        IncPtr(X, TIGGFileBlock(X^).Size);
    end;
    FreeMem(P, Size);
    {$IFDEF LOGS} Log('Released old data'); {$ENDIF}
    Size := 0;
    P := nil;
    if IsError then begin
        Video^.FreeSprite(Sprite)
    end else begin
        P := Sprite;
        Size := SizeOf(TSprite);
    end;
    {$IFDEF LOGS} Log('Finished, ' + IntStr(Size)); {$ENDIF}
    IGS_Process := GetError;
end;

function IGS_Save(FileName : String; P : Pointer; Size : LongInt) : integer; far;
var
    FH : TIGGFileHeader;
    SC : TIGSSpriteControl;
    SD : TIGSSpriteData;
    BP : TIGGPalettes;
    BT : TIGGFileBlock;
    BI : PDIBitMap;
    BB : PMask;
    Sprite : PSprite;
    I : integer;
begin
    Sprite := P;
    FileName := FileBase(FileName) + 'IGS';
    ClearError;
    FillChar(FH, Sizeof(FH), 0);
    FillChar(BP, Sizeof(BP), 0);
    FillChar(SC, Sizeof(SC), 0);
    FillChar(SD, Sizeof(SD), 0);

    with FH do begin
        Block.Size  := Sizeof(FH);
        Block.ID    := biHeader;
        move(IGSFileID[1], FileID, SizeOf(FileID));
        PalType     := CalcPalettes(BP);
        BBP         := 8;

    end;
    if NoError then FileSave(FileName, @FH, FH.Block.Size);
    if NoError and (FH.PalType = 0) then FileAppend(FileName, @BP, BP.Block.Size);
    with SC do begin
        Kind        := Sprite^.Kind;
        Level       := Sprite^.Level;
        if Sprite^.Animate <> 0 then
            FPS := 18
        else
            FPS := 0;
        Width       := Sprite^.Area.Right - Sprite^.Area.Left + 1;
        Height      := Sprite^.Area.Bottom - Sprite^.Area.Top + 1;
        HotSpot     := Sprite^.HotSpot;
        Initial     := Sprite^.Index;
        Count       := Sprite^.Count;
        Block.ID    := biSpriteControl;
        Block.Size  := SizeOf(SC);
    end;

    if NoError then FileAppend(FileName, @SC, Sizeof(SC));

    if NoError then for I := 0 to Sprite^.Count - 1 do begin
        if not Assigned(Sprite^.Sprites^[I].Image) then Continue;
        BI := ImageToDIBitMap(Sprite^.Sprites^[I].Image);
        if not Assigned(BI) then Break;
        DIBitMapImplode(BI, icaDataRun);
        with SD do begin
            Block.ID := biSpriteImage;
            Block.Size := Sizeof(SD) + DIBitMapSizeData(BI);
            Index := I;
            Sequence := Sprite^.Sprites^[I].Sequence;
            Compression := BI^.Compression;
        end;
        if NoError then FileAppend(FileName, @SD, Sizeof(SD));
        if NoError then FileAppend(FileName, @BI^.ImageData, DIBitMapSizeData(BI));
        FreeDIBitMap(BI);
        if Assigned(Sprite^.Sprites^[I].IMask) then begin
            BI := ImageToDIBitMap(Sprite^.Sprites^[I].IMask);
            if not Assigned(BI) then Break;
            DIBitMapImplode(BI, icaDataRun);
            with SD do begin
                Block.ID := biSpriteIMask;
                Block.Size := Sizeof(SD) + DIBitMapSizeData(BI);
                Index := I;
                Sequence := Sprite^.Sprites^[I].Sequence;
                Compression := BI^.Compression;
            end;
            if NoError then FileAppend(FileName, @SD, Sizeof(SD));
            if NoError then FileAppend(FileName, @BI^.ImageData, DIBitMapSizeData(BI));
            FreeDIBitMap(BI);
        end;
        if Assigned(Sprite^.Sprites^[I].BMask) then begin
            {$IFDEF LOGS}
            Log('Begin Save Mask #' + IntStr(I));
            {$ENDIF}
            BB := Video^.CloneMask(Sprite^.Sprites^[I].BMask);
            if not Assigned(BB) then Break;
            { Video^.MaskImplode(BB); }
            with SD do begin
                Block.ID := biSpriteBMask;
                Block.Size := Sizeof(SD) + Video^.MaskSizeData(BB);
                Index := I;
                Sequence := Sprite^.Sprites^[I].Sequence;
                Compression := BB^.Compression;
            end;
            if NoError then FileAppend(FileName, @SD, Sizeof(SD));
            if NoError then
                FileAppend(FileName, @BB^.ImageData, SD.Block.Size - Sizeof(SD));
            Video^.FreeMask(BB);
            {$IFDEF LOGS}
            Log('End Save Mask');
            {$ENDIF}
        end;
    end;

    BT.Size   := 0;
    BT.ID     := biTerminate;
    if NoError then FileAppend(FileName, @BT, Sizeof(BT));

    IGS_Save := GetError;
end;

function IGS_Load(FileName : String; var P : Pointer; var Size : LongInt) : integer; far;
var
    FS : Word;
begin
    {$IFDEF LOGS} Log('IGS_Load started'); {$ENDIF}
    if FileLoad(FileName, P, FS) then begin
        if IGS_Process(P, FS) = 0 then
            Size := FS;
    end;
    if IsError then
        Size := 0;
    IGS_Load := GetError;
    {$IFDEF LOGS} Log('IGSLoad Finished'); {$ENDIF}
end;

function IGA_Check(FileName : String) : boolean; far;
var
    F : boolean;
begin
    F := False;
    ClearError;
    { Test File }
    IGA_Check := NoError and F;
    ClearError;
end;

function IGA_Process(var P : Pointer; var Size : word) : integer; far;
begin
    ClearError;
    SetError(erOperation_Not_Supported);
    { Perform data conversion, free old pointer memory, return new pointer
      and it's size. Caller will not free old pointer memory for you! }
    if Assigned(P) then begin
        FreeMem(P, Size);
        P := nil;
    end;
    IGA_Process := GetError;
end;

function IGA_Save(FileName : String; P : Pointer; Size : LongInt) : integer; far;
begin
    ClearError;
    SetError(erOperation_Not_Supported);
    IGA_Save := GetError;
end;

function IGA_Load(FileName : String; var P : Pointer; var Size : LongInt) : integer; far;
var
    FS : Word;
begin
    if FileLoad(FileName, P, FS) then begin
        if IGA_Process(P, FS) = 0 then
            Size := FS;
    end;
    if IsError then
        Size := 0;
    IGA_Load := GetError;
end;

procedure RegisterFormatBuiltIn;
begin

    DRV_Handler := New(PFormatHandler);
    with DRV_Handler^ do begin
        Kind   := ffDriver;
        UID    := 'BINDRV'; { unique ID }
        Compat := 0;
        Exts   := 'DRV;086;186;286;386;486;586;686'; { file extensions }
        Check  := DRV_Check;
        Save   := nil; { Cannot Save DRV Format }
        Process:= DRV_Process;
        Load   := DRV_Load; { DRV_Load; }
    end;
    RegisterFileFormat(DRV_Handler);

    FNT_Handler := New(PFormatHandler);
    with FNT_Handler^ do begin
        Kind   := ffBitmapFont;
        UID    := 'BITFNT'; { unique ID }
        Compat := 0;
        Exts   := 'FNT'; { file extensions }
        Check  := FNT_Check;
        Process:= FNT_Process;
        Save   := FNT_Save;
        Load   := FNT_Load;
    end;
    RegisterFileFormat(FNT_Handler);

    IGG_Handler := New(PFormatHandler);
    with IGG_Handler^ do begin
       Kind   := ffImage;
        UID    := 'IMGIGG';
        Compat := 0;
        Exts   := 'IGG';
        Check  := IGG_Check;
        Process:= IGG_Process;
        Save   := IGG_Save;
        Load   := IGG_Load;
    end;
    RegisterFileFormat(IGG_Handler);

    IGS_Handler := New(PFormatHandler);
    with IGS_Handler^ do begin
        Kind   := ffSprite;
        UID    := 'SPRIGS'; { unique ID }
        Compat := 0;
        Exts   := 'IGS;'; { file extensions }
        Check  := IGS_Check;
        Process:= IGS_Process;
        Save   := IGS_Save;
        Load   := IGS_Load;
    end;
    RegisterFileFormat(IGS_Handler);

    IGA_Handler := New(PFormatHandler);
    with IGA_Handler^ do begin
        Kind   := ffAudio;
        UID    := 'SNDIGA'; { unique ID }
        Compat := 0;
        Exts   := 'IGA;'; { file extensions }
        Check  := IGA_Check;
        Process:= IGA_Process;
        Save   := IGA_Save;
        Load   := IGA_Load;
    end;
    RegisterFileFormat(IGA_Handler);

end;

end.