{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$IFDEF SECTINT}

function KeyboardEvent(var Event : TEvent) : boolean;
function KeyboardPress(var Event : TEvent) : boolean;
function KeyboardRelease(var Event : TEvent) : boolean;
function KeyPress(var Event : TEvent; KeyCode : word) : boolean;
function KeyRelease(var Event : TEvent; KeyCode : word) : boolean;
function AnyControlKey(var Event : TEvent) : boolean;
function LeftControlKey(var Event : TEvent) : boolean;
function RightControlKey(var Event : TEvent) : boolean;
function AnyAltKey(var Event : TEvent) : boolean;
function LeftAltKey(var Event : TEvent) : boolean;
function RightAltKey(var Event : TEvent) : boolean;
function AnyShiftState(var Event : TEvent) : boolean;
function AnyShiftKey(var Event : TEvent) : boolean;
function LeftShiftKey(var Event : TEvent) : boolean;
function RightShiftKey(var Event : TEvent) : boolean;
function CapsLockEnabled(var Event : TEvent) : boolean;

{$ENDIF}

{$IFDEF SECTIMP}
var
    IntState     : word;
    ShiftState   : DWord;
    ScanCode     : word;
    HoldCode     : byte;
    ExtraCode    : word;

const
    KeyMap : array [boolean] of String = (
        #0 + #27 + '1234567890-=' + #8 + #9 + 'qwertyuiop[]' +
        #13 + #0 + 'asdfghjkl;' + #39 + '`' + #0 + '\zxcvbnm,./' +
        #0 + #0 + #0 + ' ',

        #0 + #27 + '!@#$%^&*()_+' + #8 + #9 + 'QWERTYUIOP{}' +
        #13 + #0 + 'ASDFGHJKL:"~' + #0 + '|ZXCVBNM<>?' +
        #0 + #0 + #0 + ' '
    );

procedure KeyboardHandler; interrupt;
var
    Event : TEvent;
    Caps : boolean;
begin
    asm   { Fetch Keystroke }
        mov         ah, HoldCode
        in          al, $60
        cmp         al, $e0
        je          @@Hold
        cmp         al, $e1
        je          @@Hold
        cmp         al, $f0
        je          @@Hold
        mov         Scancode, ax

        {
        mov         ah, MaintainKeystate
        cmp         ah, False
        je          @@NoChange
        }

        mov         bx, $0040
        push        bx
        pop         es
        mov         bh, ah
        mov         bl, $80
        mov         ah, al
        mov         di, $0017
        mov         si, $0096
        mov         cx, [es:di]
        mov         dx, [es:si]
        and         al, $7f
        cmp         al, $2a
        je          @@LeftShift
        cmp         al, $36
        je          @@RightShift
        cmp         al, $3a
        je          @@CapsLock
        cmp         al, $1d
        je          @@CntrlKey
        cmp         al, $38
        je          @@AltKey
        jmp         @@NoChange
    @@LeftShift:
        cmp         bh, $e0    { pc-keyboard pushes a dup Shift for some keys }
        je          @@NoChange
        and         cl, $fd
        test        ah, bl
        jnz         @@StateChange
        or          cl, $02
        jmp         @@StateChange
    @@RightShift:
        cmp         bh, $e0    { pc-keyboard pushes a dup Shift for some keys }
        je          @@NoChange
        and         cl, $fe
        test        ah, bl
        jnz         @@StateChange
        or          cl, $01
        jmp         @@StateChange
    @@CapsLock:
        test        ah, bl
        jnz         @@NoChange
        test        cl, $40
        jz          @@CapsLockon
        and         cl, $bf
        and         dh, $fb
        jmp         @@StateChange
    @@CapsLockOn:
        or          cl, $40
        or          dh, $04
        jmp         @@StateChange

    @@CntrlKey:
        test        bh, bh
        jnz         @@RightCntrl
        and         ch, $fe
        test        ah, bl
        jnz         @@AnyCtrl
        or          ch, $01
        jmp         @@AnyCtrl
    @@RightCntrl:
        and         dl, $fb
        test        ah, bl
        jnz         @@AnyCtrl
        or          dl, $04
        jmp         @@AnyCtrl
    @@AnyCtrl:
        and         cl, $fb
        test        ch, $01
        jnz         @@CtrlActive
        test        dl, $04
        jz          @@StateChange
    @@CtrlActive:
        or          cl, $04
        jmp         @@StateChange

    @@AltKey:
       test        bh, bh
        jnz         @@RightAlt
        and         ch, $fd
        test        ah, bl
        jnz         @@AnyAlt
        or          ch, $02
        jmp         @@AnyAlt
    @@RightAlt:
        and         dl, $f7
        test        ah, bl
        jnz         @@AnyAlt
        or          dl, $08
        jmp         @@AnyAlt
    @@AnyAlt:
        and         cl, $f7
        test        ch, $02
        jnz         @@AltActive
        test        dl, $08
        jz          @@StateChange
    @@AltActive:
        or          cl, $08
        jmp         @@StateChange

    @@StateChange:
        mov         [es:di], cx
        mov         [es:si], dx
    @@NoChange:
        xor         al, al
        mov         HoldCode, al
        jmp         @@Done
    @@Hold:
        mov         HoldCode, al
    @@Done:
    end;

    if ScanCode <> 0 then begin
        Event.ScanCode  := ScanCode and $ff7f;
        if Event.ScanCode = $e11d then begin
            ExtraCode := ScanCode;
        end else begin
            if ScanCode and $80 = $80 then
                Event.Kind := evKeyRelease
            else
                Event.Kind := evKeyPress;
            Event.ShiftCode := MemW[Seg0040:$0017];
            Event.StatusCode := MemW[Seg0040:$0096];
            ShiftState.L := Event.ShiftCode;
            ShiftState.H := Event.StatusCode;
            Caps := Event.ShiftCode and kssUppercase <> 0;
            if Lo(Event.ScanCode) >= Length(KeyMap[Caps]) then
                Event.KeyCode := 0
            else
                Event.KeyCode := ScanCode and $ff00 or Ord(KeyMap[Caps][Lo(Event.ScanCode) + 1]);
            { Logitech sends some stuff we just don't need }
            if (Event.ScanCode <> $e02a) then begin
                if ExtraCode <> 0 then begin
                    Event.ScanCode := (ExtraCode and $ff00) or (Event.ScanCode and $00ff);
                    if Event.Kind = evKeyRelease then
                        ExtraCode := 0;
                end;
                PutEvent(Event);
            end;
        end;
        ScanCode := 0;
    end;

    asm { Cleanup }
        in          al, $61
        mov         ah, al
        or          al, $80
        out         $61, al
        xchg        ah, al
        out         $61, al

        mov         al, $20
        out         $20, al
    end;

end;

{ Keyboard Status Testing }
function KeyboardEvent(var Event : TEvent) : boolean;
begin
    KeyboardEvent := (Event.Kind = evKeyPress) or (Event.Kind = evKeyRelease);
end;

function AnyControlKey(var Event : TEvent) : boolean;
begin
    AnyControlKey := KeyboardEvent(Event) and
        (Event.ShiftCode and kssAnyCtrl = kssAnyCtrl);
end;

function LeftControlKey(var Event : TEvent) : boolean;
begin
    LeftControlKey := KeyboardEvent(Event) and
        (Event.ShiftCode and kssLeftCtrl = kssLeftCtrl);
end;

function RightControlKey(var Event : TEvent) : boolean;
begin
    RightControlKey := KeyboardEvent(Event) and
        (Event.StatusCode and ksfRightCtrl = ksfRightCtrl);
end;

function AnyAltKey(var Event : TEvent) : boolean;
begin
    AnyAltKey := KeyboardEvent(Event) and
        (Event.ShiftCode and kssAnyAlt = kssAnyAlt);
end;

function LeftAltKey(var Event : TEvent) : boolean;
begin
    LeftAltKey := KeyboardEvent(Event) and
        (Event.ShiftCode and kssLeftAlt = kssLeftAlt);
end;

function RightAltKey(var Event : TEvent) : boolean;
begin
    RightAltKey := KeyboardEvent(Event) and
        (Event.StatusCode and ksfRightAlt = ksfRightAlt);
end;

function AnyShiftState(var Event : TEvent) : boolean;
begin
    AnyShiftState := KeyboardEvent(Event) and
        (Event.ShiftCode and kssUppercase <> 0);
end;

function AnyShiftKey(var Event : TEvent) : boolean;
begin
    AnyShiftKey := KeyboardEvent(Event) and
        (Event.ShiftCode and (kssLeftShift or kssRightShift) <> 0);
end;

function LeftShiftKey(var Event : TEvent) : boolean;
begin
    LeftShiftKey := KeyboardEvent(Event) and
        (Event.ShiftCode and kssLeftShift <> 0);
end;

function RightShiftKey(var Event : TEvent) : boolean;
begin
    RightShiftKey := KeyboardEvent(Event) and
        (Event.ShiftCode and kssRightShift <> 0);
end;

function CapsLockEnabled(var Event : TEvent) : boolean;
begin
    CapsLockEnabled := KeyboardEvent(Event) and
        (Event.ShiftCode and kssCapsLockState <> 0);
end;

function KeyboardPress(var Event : TEvent) : boolean;
begin
    KeyboardPress := Event.Kind = evKeyPress;
end;

function KeyboardRelease(var Event : TEvent) : boolean;
begin
    KeyboardRelease := Event.Kind = evKeyRelease;
end;

function KeyPress(var Event : TEvent; KeyCode : word) : boolean;
begin
    KeyPress := (Event.Kind = evKeyPress) and (Event.KeyCode = KeyCode);
end;

function KeyRelease(var Event : TEvent; KeyCode : word) : boolean;
begin
    KeyRelease := (Event.Kind = evKeyRelease) and (Event.KeyCode = KeyCode);
end;


{$ENDIF}