{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{ Danger Engine based program, see https://gitlab.com/DOSx86/sge }

{$I DANGER.DEF}
{ DEFINE NODELAY}

program ImgEditor;

uses Danger, SimpleUI, FFmtBMP, FFmtASM, FFmtPas;

const
    Version = '2021-11-10';

type
    TEditMode = (emFont, emImage, emSprite);
    TSpriteMode = (smImage, smMask);

const
    cmNewFont    = cmUser + 1;
    cmNewImage   = cmUser + 2;
    cmNewSprite  = cmUser + 3;

    cmEditCopy   = cmUser + 10;
    cmEditPaste  = cmUser + 11;

    cmImageClear = cmUser + 30;
    cmImageRotate= cmUser + 31;
    cmImageFlip  = cmUser + 32;
    cmImageInvert= cmUser + 33;
    cmImageShift = cmUser + 34;
    cmImageUp    = cmUser + 35;
    cmImageDown  = cmUser + 36;
    cmImageLeft  = cmUser + 37;
    cmImageRight = cmUser + 38;

    cmDrawColor  = cmUser + 50;
    cmEraseColor = cmUser + 51;

    cmDrawMode   = cmUser + 100;
    cmFillMode   = cmUser + 101;
    cmSampleMode = cmUser + 102;

    cmEditSeqAdd = cmUser + 201;
    cmEditSeqDel = cmUser + 202;
    cmEditFrameAdd = cmUser + 211;
    cmEditFrameDel = cmUser + 212;

    cmSeqNext   = cmUser + 230;
    cmSeqPrev   = cmUser + 231;
    cmFrameNext = cmUser + 232;
    cmFramePrev = cmUser + 233;

    cmSwitchImage   = cmUser + 240;
    cmSwitchMask    = cmUser + 241;
    cmImgToMask     = cmUser + 242;
    cmAnimate       = cmUser + 243;

    cmSaveAsSprite  = cmUser + 999;
    cmSaveAsFmt  = cmUser + 1000;
    cmSaveAsMax  = cmSaveAsFmt + 99;

const
    EditMode : TEditMode = emImage;
    DeskColor : TColor = 1;
    SpriteIdxColor : TColor = 8;
    StatusBarColor : TColor = 15;
    MaxEditX : word = 160;
    MaxEditY : word = 160;
    NewImgX : integer = 32;
    NewImgY : integer = 32;
    DrawFontColor : TColor = 15;
    DrawMaskColor : TColor = 15;
    DrawColor : TColor = 15;
    EraseColor : TColor = 0;
    FontHot : TColor = 15;
    FontCold : TColor = 8;

    DrawOfsX : integer = 8;
    DrawOfsY : integer = 24;
    DrawFntX : integer = 180;
    ScaleX : integer = 10;
    ScaleY : integer = 10;
    HoverColor : TColor = 4;
    POfsX : integer = 160;
    POfsY : integer = 24;
    MOfsX : integer = 156;
    MOfsY : integer = 20;
    ModifiedFlag : boolean = false;
    BiosMode : word = 0;
    StatusBarPadding : TPoint = (X:3; Y:1);

var
    FileName    : String;
    FChar       : byte;
    EFont       : PFont;
    FCopy       : PImage;
    EImage,
    FImage      : PImage;
    FSprite     : PSprite;
    Width, Height,
    AsciiWide, AsciiHigh, AsciiStart, AsciiLast : integer;
    ELastX, ELastY,
    EOfsX, EOfsY,
    EMaxX, EMaxY : integer;
    MenuImport,
    MenuSave,
    MenuSaveAs,
    MenuEdit,
    MenuRotate,
    MenuPaste,
    MenuSeqAdd, MenuSeqDel,
    MenuImgAdd, MenuImgDel : PMenuItem;
    BtnDrawColor,
    BtnEraseColor : PButtonItem;
    HoverBox : record
        Where : TPoint;
        Top, Bottom, Left, Right : PImage;
    end;
    Ticks : record
        COrg, CEnd,
        AOrg, AEnd  : PSprite;
    end;
    GSets : array[0..6] of string;
    BtnSet : string;
    BtnDraw, BtnFill, BtnEyeDrop,
    BtnImg, BtnMask, BtnImgToMask, BtnPlay,
    BtnImgNext, BtnImgPrev,
    BtnSeqNext, BtnSeqPrev
     : PButtonItem;
    DrawMode, LastMode : word;
    SpriteIdx, SpriteIdxPrev, SpriteIdxNext,
    SpriteSeq, SpriteSeqIdx,
    SpriteSeqs, SpriteImgs : word;
    SpriteMode : TSpriteMode;
    MenuFont, BtnFont : TFontSettings;
    SpriteBehind : TColor;
    LSI : String;
    Animating : boolean;
    SpriteModified : boolean;
    LastColor : record
        Draw, Erase : TColor;
    end;

procedure ShowColors;
begin
    if EditMode = emFont then exit;
    if (EditMode = emSprite) and (SpriteMode = smMask) then begin
        with BtnDrawColor^ do begin
            Video^.SpriteUndraw(Sprite);
            Video^.Region(Area.Left, Area.Top, Area.Right, Area.Bottom, DeskColor);
        end;
        with BtnEraseColor^ do begin
            Video^.SpriteUndraw(Sprite);
            Video^.Region(Area.Left, Area.Top, Area.Right, Area.Bottom, DeskColor);
        end;
    end else begin
        with BtnDrawColor^ do begin
            Video^.SpriteUndraw(Sprite);
            Video^.Region(Area.Left, Area.Top, Area.Right, Area.Bottom, DrawColor);
        end;
        with BtnEraseColor^ do begin
            Video^.SpriteUndraw(Sprite);
            Video^.Region(Area.Left, Area.Top, Area.Right, Area.Bottom, EraseColor);
        end;
    end;
end;

procedure PopUpColorChange(Btn : PButtonItem; var Color : TColor);
const
    PickerScaleX : integer = 10;
    PickerScaleY : integer = 10;
var
    E : TEvent;
    I : integer;
    XO, YO, X, Y, W, H : integer;
    LX, LY : integer;
    B : PImage;
    RH, RW : integer;
begin
    if Video^.Colors = 256 then begin
        RH := 16;
        RW := RH;
    end else begin
        RW := Video^.Colors;
        RH := 1;
    end;
    H := (RH + 1) * PickerScaleY - 1;
    W := (RW + 1) * PickerScaleX - 1;
    XO := Btn^.Area.Left;
    YO := Btn^.Area.Top - H - 1;
    Video^.Region(XO, YO, XO + W, YO + H, 7);
    I := 1;
    Video^.Region(XO, YO, XO + W, YO + I, 0);
    Video^.Region(XO, YO + H - I, XO + W, YO + H, 0);
    Video^.Region(XO, YO, XO + I, YO + H, 0);
    Video^.Region(XO + W - I, YO, XO + W, YO + H, 0);
    Inc(XO, PickerScaleX div 2);
    Inc(YO, PickerScaleY div 2);
    B := Video^.NewImage(PickerScaleX + 2, PickerScaleY + 2);
    for Y := 0 to (RH - 1) do
        for X := 0 to (RW - 1) do begin
            if Color = Y * RH + X then begin
                Video^.Region(XO + X * PickerScaleX-1, YO + Y * PickerScaleY -1,
                XO + X * PickerScaleX + PickerScaleX,
                YO + Y * PickerScaleY + PickerScaleY,
                Y * RH + X);
            end else begin
                Video^.Region(XO + X * PickerScaleX + 1, YO + Y * PickerScaleY + 1,
                XO + X * PickerScaleX + PickerScaleX - 2,
                YO + Y * PickerScaleY + PickerScaleY - 2,
                Y * RH + X);
            end;
        end;
    H := RH * PickerScaleY - 1;
    W := RW * PickerScaleX - 1;
    PurgeEvents;
    X := -1;
    Y := -1;
    LX := X;
    LY := Y;
    repeat
        if E.Kind and evMouse <> 0 then begin
            X := (E.Position.X - XO) div PickerScaleX;
            Y := (E.Position.Y - YO) div PickerScaleY;
            if (X > (RW - 1)) or (Y > (RH - 1)) or (E.Position.X < XO) or
            (E.Position.Y < YO) then begin
                X := -1;
                Y := -1;
            end;
            if (LX <> X) or (LY <> Y) then begin
                if Assigned(B) and (LX > -1) and (LY > -1) then
                    Video^.PutImage(B, XO + LX * PickerScaleX-1, YO + LY * PickerScaleY-1);
                LX := X;
                LY := Y;
                if Assigned(B) and (X > -1) and (Y > -1) then begin
                    Video^.GetImage(B, XO + X * PickerScaleX-1, YO + Y * PickerScaleY-1);
                    Video^.Region(XO + X * PickerScaleX-1, YO + Y * PickerScaleY -1,
                    XO + X * PickerScaleX + PickerScaleX,
                    YO + Y * PickerScaleY + PickerScaleY,
                    4);
                    Video^.Region(XO + X * PickerScaleX + 1, YO + Y * PickerScaleY + 1,
                    XO + X * PickerScaleX + PickerScaleX - 2,
                    YO + Y * PickerScaleY + PickerScaleY - 2,
                    Y * RH + X);
                end;

            end;
        end;
        Video^.Update;
        While not GetEvent(E) do Idle;
        Video^.Prepare;
        if MouseEvent(E) then MouseMove(E.Position);
    until (E.Kind = evKeyPress) or (E.Kind = evMouseClick);
    if (E.Kind = evMouseClick) and (LX <> -1) and (LY <> -1) then begin
        Color := LY * RH + LX;
    end;
    Video^.FreeImage(B);
    PurgeEvents;
end;

procedure ChangeDrawColor;
begin
    PopUpColorChange(BtnDrawColor, DrawColor);
end;

procedure ChangeEraseColor;
begin
    PopUpColorChange(BtnEraseColor, EraseColor);
end;

procedure Modified(State : Boolean); forward;
procedure ImageToEdit(X, Y : integer); forward;

procedure SpriteBtnBars;
begin
    if (EditMode = emSprite) then begin
        with BtnImgToMask^.Area do Video^.Region(Left - 2, Top, Left - 2, Bottom, 7);
        with BtnImgPrev^.Area do Video^.Region(Left - 2, Top, Left - 2, Bottom, 7);
        with BtnSeqPrev^.Area do Video^.Region(Left - 2, Top, Left - 2, Bottom, 7);
        with BtnSeqNext^.Area do Video^.Region(Right + 2, Top, Right + 2, Bottom, 7);
    end;
end;

procedure DisplaySpriteInfo;
var
    Y : integer;
    S : String;
    Hold : TFontSettings;
begin
    if Assigned(FSprite) then begin
        {
        S := IntStr(SpriteSeq) + '.' + IntStr(SpriteIdx) + ' ' +
            IntStr(SpriteSeqs) + ':' + IntStr(SpriteIdxPrev) + ',' +
            IntStr(SpriteIdxNext);
        Video^.Region(240, 0, GetMaxX - 1, Video^.TextHeight(S), 15);
        Video^.PutText(GetMaxX - 2 - Video^.TextWidth(S), 2, S, 8);
        }
        S := IntStr(SpriteSeq) + '.' + IntStr(SpriteSeqIdx);
        if (LSI <> S) then with BtnSeqNext^.Area do begin
            GetFontSettings(Hold);
            SetFontSettings(BtnFont);
            Y := Top + (Bottom - Top - Video^.TextHeight(S));
            if LSI <> '' then begin
                Video^.Region(Right + 5, Top,
                    Right + 5 + Video^.TextWidth(LSI), Bottom,
                    StatusBarColor);
            end;
            Video^.PutText(Right + 5, Y, S, SpriteIdxColor);
            LSI := S;
            SetFontSettings(Hold);
        end;
    end;
end;

procedure StandByBar;
var
    X, Y : integer;
    S : String;
begin
    {$IFDEF LOGS}
        Log('Standby...');
    {$ENDIF}
    Video^.SpriteUndrawAll;
    S := RawNLS('STANDBY');
    Y := GetMaxY - FontHeight -  StatusBarPadding.Y * 2 + 1;
    X := GetMaxX - Video^.TextWidth(S) - StatusBarPadding.X;
    Video^.Region(0, Y, GetMaxX, GetMaxY, StatusBarColor);
    Video^.PutText(X, Y + StatusBarPadding.Y, S, 4 );
    SpriteBtnBars;
    Video^.Update;
    Video^.SpriteUndrawAll;
    LSI := '';
end;

procedure DisplayFileInfo;
var
    X, Y : integer;
    W0, W1, W2 : integer;
    S1, S2 : String;
    C : TColor;
begin
    S1 := IntStr(Width) + 'x' + IntStr(Height);
    S2 := FileName;
    While Pos(PathDelim, S2) > 0 do Delete(S2, 1,  Pos(PathDelim, S2));
    W0 := Video^.TextWidth(' ');
    if W0 > 4 then W0 := 4;
    W1 := Video^.TextWidth(S1);
    W2 := Video^.TextWidth(S2);
    X := GetMaxX - W0 - W1 - W2 - StatusBarPadding.X + 1;
    Y := GetMaxY - FontHeight -  StatusBarPadding.Y  + 1;
    If ModifiedFlag then C := 4 else C := 8;
    Video^.PutText(X, Y, S2, C );
    Video^.PutText(X + W0 + W2, Y, S1, 7 );
    DisplaySpriteInfo;
end;

procedure StatusBar;
begin
    LSI := '';
    Video^.SpriteUndrawAll;
    Video^.Region(0, GetMaxY - FontHeight -  StatusBarPadding.Y * 2 + 1,
        GetMaxX, GetMaxY, StatusBarColor);
    SpriteBtnBars;
    DisplayFileInfo;
end;

function NewTickMark(Direction : boolean; C : TColor):PSprite;
var
    P : PSprite;
    X, Z : integer;
begin
    P := Video^.NewSprite(5, 5, 1, true);
    if Assigned(P) then begin
        Video^.ImageFill(P^.Sprites^[0].Image, 0);
        For Z := 0 to 4 do
            for X := Z to 4 - Z do
                if Direction then
                    Video^.ImagePutPixel(P^.Sprites^[0].Image, X,4-Z,C)
                else
                    Video^.ImagePutPixel(P^.Sprites^[0].Image, X,Z,C);
        Video^.FreeMask(P^.Sprites^[0].BMask);
        P^.Sprites^[0].BMask := Video^.ImageToMask(P^.Sprites^[0].Image, 0);
        Video^.ImageImplode(P^.Sprites^[0].Image);
    end;
    NewTickMark := P;
end;

procedure MakeTicks;
begin
    if not Assigned(Ticks.COrg) then begin
        Ticks.COrg := NewTickMark(False, 15);
        Ticks.CEnd := NewTickMark(False, 7);
        Ticks.AOrg := NewTickMark(True, 2);
        Ticks.AEnd := NewTickMark(True, 4);
    end;
    Video^.SpriteSort;
    if EditMode <> emFont then begin
        Video^.SpriteHide(Ticks.COrg);
        Video^.SpriteHide(Ticks.AOrg);
        Video^.SpriteHide(Ticks.CEnd);
        Video^.SpriteHide(Ticks.AEnd);
    end else begin
        Video^.SpriteShow(Ticks.COrg);
        Video^.SpriteShow(Ticks.AOrg);
        Video^.SpriteShow(Ticks.CEnd);
        Video^.SpriteShow(Ticks.AEnd);
    end;
end;

procedure AdjustTick(P : PSprite; Bit : integer; Bottom : boolean);
var
    X, Y : Integer;
begin
    if Not Assigned(P) then Exit;
    if Bit >= Width then Bit:= Width - 1;
    X := DrawOfsX + Bit * ScaleX + ScaleX div 2;
    if Bottom then
        Y := DrawOfsY + Height * ScaleY
    else
        Y := DrawOfsY + P^.Area.Top - P^.Area.Bottom  - 1;
     Video^.SpriteMove(P, X, Y);
end;

procedure UpdateTickMarks;
var
    I, O, M, S : integer;
begin
     if not (Assigned(Ticks.COrg) and (EditMode = emFont)) then exit;
     with EFont^.Offsets[FChar] do begin
         AdjustTick(Ticks.COrg, Offset, false);
         AdjustTick(Ticks.CEnd, Maximum, false);
         Video^.SpriteSetVisible(Ticks.COrg, Offset <= Maximum);
         Video^.SpriteSetVisible(Ticks.CEnd, Offset <= Maximum);
     end;
     for S := Low(GSets) to High(GSets) do
        if Pos(Chr(FChar), GSets[S]) > 0 then break;
     O := Width;
     M := 0;
    with EFont^ do
        for I := 1 to Length(GSets[S]) do  begin
            if O > Offsets[Ord(GSets[S][I])].Offset then O := Offsets[Ord(GSets[S][I])].Offset;
            if M < Offsets[Ord(GSets[S][I])].Maximum then M := Offsets[Ord(GSets[S][I])].Maximum;
        end;
     AdjustTick(Ticks.AOrg, O, true);
     AdjustTick(Ticks.AEnd, M, true);
     Video^.SpriteSetVisible(Ticks.AOrg, O <= M);
     Video^.SpriteSetVisible(Ticks.AEnd, O <= M);
end;

procedure SetFontData;
begin
    Width := EFont^.Width;
    Height := EFont^.Height;
    AsciiStart := 0;
    AsciiLast := -1;
    AsciiWide := (GetMaxX - 1 - DrawFntX - DrawOfsX) div (Width + 1);
    AsciiHigh := (GetMaxY - 1 - DrawOfsY * 2) div (Height + 1);
end;

function CloneFont(Source : PFont) : PFont;
var
    P : PFont;
begin
    CloneFont := nil;
    P := New(PFont);
    if Not Assigned(P) then Exit;
    FillChar(P^, Sizeof(TFont), 0);
    if Assigned(Source) and Assigned(Source^.Bitmap) then begin
        P^ := Source^;
        GetMem(P^.Bitmap, FontBitmapSize(P));
        if Assigned(P^.Bitmap) then begin
            Move(Source^.Bitmap^, P^.BitMap^,  FontBitmapSize(P));
        end
    end;
    if not Assigned(P^.Bitmap) then begin
        Dispose(P);
        P := nil;
    end;
    CloneFont := P;
end;

procedure GetFontChar(C : byte);
var
    B, O, I, J : word;
    LastStart : integer;
begin
    FChar := C;
    UpdateTickMarks;
    While AsciiStart > C do Dec(AsciiStart, AsciiWide);
    While AsciiStart + AsciiWide * AsciiHigh <= C do Inc(AsciiStart, AsciiWide);
    Video^.ImageFill(FImage, 0);
    B := EFont^.ByteWidth;
    O := B * Height * C;
    for J := 0 to Height - 1 do
        for I := 0 to Width - 1 do
            if (Bytes(EFont^.Bitmap^)[O + (J * B) + I div 8] shl (I mod 8)) and $80 = $80 then
                Video^.ImagePutPixel(FImage, I, J, DrawFontColor);
end;

procedure SetFontChar;
var
    B, O, I, J, L, H : word;
begin
    L := Width;
    H := 0;
    B := EFont^.ByteWidth;
    O := B * Height * FChar;
    FillChar(Bytes(EFont^.Bitmap^)[O], B * Height, 0);
    for J := 0 to Height - 1 do
        for I := 0 to Width - 1 do
            if Video^.ImageGetPixel(FImage, I, J) <> 0 then begin
                Bytes(EFont^.Bitmap^)[O + (J * B) + I div 8] :=
                    Bytes(EFont^.Bitmap^)[O + (J * B) + I div 8] or ($80 shr (I mod 8));
                if I < L then L := I;
                if I > H then H := I;
            end;
     EFont^.Offsets[FChar].Offset := L;
     EFont^.Offsets[FChar].Maximum := H;
end;

procedure FontResize(NewW, NewH : word);
var
    OldF, NewF : pointer;
    OldS, OldB, OldO, OldW, OldH,
    NewS, NewB, NewO   : Word;
    C, X, Y : integer;

    function OldBit(X, Y : integer) : boolean;
    begin
        OldBit := (Bytes(OldF^)[OldO + (Y * OldB) + X div 8] shl (X mod 8)) and $80 = $80;
    end;

begin
    NewB := NewW div 8;
    if NewW mod 8 <> 0 then Inc(NewB);
    NewF := nil;
    NewS := NewB * NewH * 256;
    GetMem(NewF, NewS);
    if not Assigned(NewF) then FatalError(erInsufficient_Memory, 'font resize');
    OldF := EFont^.Bitmap;
    OldW := EFont^.Width;
    OldH := EFont^.Height;
    OldB := EFont^.ByteWidth;

    EFont^.Width := NewW;
    EFont^.Height := NewH;
    EFont^.Bitmap := NewF;
    EFont^.ByteWidth := NewB;

    FillChar(NewF^, NewS, 0);
    for C := 0 to 255 do begin
        OldO := C * OldH * OldB;
        NewO := C * NewH * NewB;
        for Y := 0 to OldH - 1 do
            for X := 0 to OldW - 1 do
                if OldBit(X, Y) then
                    Bytes(NewF^)[NewO + (Y * NewB) + X div 8] :=
                     Bytes(NewF^)[NewO + (Y * NewB) + X div 8] or ($80 shr (X mod 8));
    end;
    FontOffsetReset(EFont);
    FreeMem(OldF, OldS);
end;

procedure NewFontFile;
var
    F : PFont;
    AW, AH : word;
    FW, FH : word;
    S : String;
begin
    Video^.FreeImage(FImage);
    Video^.FreeSprite(FSprite);
    Video^.FreeFont(EFont);
    EditMode := emFont;
    FW := NewImgX;
    FH := NewImgY;
    if FH < 8 then FH := 8;
    if FW > 32 then FW := 32;
    if FH > 32 then FH := 32;
    Width := FW;
    Height := FH;
    S := ZPad(Width, 2) + ZPad(Height, 2) + 'N-' + Language + '.FNT';
    { Try to clone an existing resource font }
    FontBestMatch(S, F);
    if Assigned(F) then begin
        EFont := CloneFont(F);
        FontResize(FW, FH);
    end else begin
        FatalError(erFile_Not_Found,'new font ' + S);
    end;
    FImage := Video^.NewImage(Width,Height);
    SetFontData;
    FileName := FileBase(FileName) + 'FNT';
    GetFontChar($41);
    Modified (True);
end;

procedure LoadFontFile;
var
    NewF, NewP : Pointer;
    NewS : Word;
    Sz : LongInt;
begin
    Video^.FreeImage(EImage);
    Video^.FreeImage(FImage);
    Video^.FreeSprite(FSprite);
    Video^.FreeFont(EFont);
    EditMode := emFont;
    if FormatLoad(FileName, NewF, Sz) then begin
        EFont := CloneFont(PFont(NewF));
        SetFontData;
        Modified(False);
        FImage := Video^.NewImage(Width,Height);
        GetFontChar($41);
    end else begin
        FatalError(erOperation_Not_Supported, 'load font');
        Width := 8;
        Height := 8;
        Filename := 'NONAME.FNT';
        NewFontFile;
    end;
end;

procedure NewImageFile;
begin
    Video^.FreeImage(FImage);
    EditMode := emImage;
    Width := NewImgX;
    Height := NewImgY;
    FImage := Video^.NewImage(Width,Height);
    Video^.ImageFill(FImage, EraseColor);
    if FormatType(FileName) <> ffImage then
        FileName := FileBase(FileName) + 'IGG';
    Modified(True);
end;

procedure LoadImageFile;
var
    Kind : integer;
    Size : LongInt;
begin
    Video^.FreeImage(EImage);
    Video^.FreeImage(FImage);
    Video^.FreeSprite(FSprite);
    Video^.FreeFont(EFont);
    EditMode := emImage;
    Kind := FormatType(FileName);
    if Kind <> ffImage then
        FatalError(erInvalid_File_Format, FileName);
    if not FormatLoad(Filename, Pointer(FImage), Size) then
        FatalError(GetError, FileName);
    Video^.ImageExplode(FImage);
    EditMode := emImage;
    NewImgX := FImage^.Width;
    NewImgY := FImage^.Height;
    Width := NewImgX;
    Height := NewImgY;
    Modified(False);
end;

procedure ImageFromSprite(Index : integer; Mask : boolean);
begin
    Video^.FreeImage(FImage);
    if Mask then begin
        FImage := Video^.MaskToImage(FSprite^.Sprites^[Index].BMask, DrawMaskColor)
    end else begin
        FImage := Video^.CloneImage(FSprite^.Sprites^[Index].Image);
        Video^.ImageExplode(FImage);
    end;
end;

procedure ImageToSprite(Index : integer; Mask : boolean);
begin
    if Mask then begin
        Video^.FreeMask(FSprite^.Sprites^[Index].BMask);
        FSprite^.Sprites^[Index].BMask := Video^.ImageToMask(FImage, DrawMaskColor);
    end else begin
        Video^.FreeImage(FSprite^.Sprites^[Index].Image);
        FSprite^.Sprites^[Index].Image := Video^.CloneImage(FImage);
        If ImageCompress then
            Video^.ImageImplode(FSprite^.Sprites^[Index].Image);
    end;
end;

procedure SpriteToImage(Index : integer; Mask : boolean);
begin
    Video^.FreeImage(FImage);
    if Mask then begin
        FImage := Video^.MaskToImage(FSprite^.Sprites^[Index].BMask, DrawMaskColor);
    end else begin
        FImage := Video^.CloneImage(FSprite^.Sprites^[Index].Image);
        Video^.ImageExplode(FImage);
    end;
end;

procedure NewSpriteFile;
begin
    Video^.FreeSprite(FSprite);
    Animating := False;
    EditMode := emImage;
    NewImageFile;
    LSI := '';
    EditMode := emSprite;
    FSprite := Video^.NewSprite(FImage^.Width, FImage^.Height, 1, False);
    if Assigned(FSprite) then begin
        SpriteIdx := 0;
        Video^.FreeImage(FSprite^.Sprites^[SpriteIdx].Image);
        FSprite^.Sprites^[SpriteIdx].Image := Video^.Cloneimage(FImage);
        FSprite^.Sprites^[SpriteIdx].BMask := Video^.ImageToMask(
            FSprite^.Sprites^[SpriteIdx].Image, EraseColor);
        FileName := FileBase(FileName) + 'IGS';
        Video^.SpriteSort;
        SpriteModified := False;
        Modified(True);
    end else
end;

procedure LoadSpriteFile;
var
    Kind : integer;
    Size : LongInt;
begin
    LSI := '';
    {$IFDEF LOGS}
    Animating := False;
    Log('Begin load Sprite');
    {$ENDIF}
    Video^.FreeImage(EImage);
    Video^.FreeImage(FImage);
    Video^.FreeSprite(FSprite);
    Video^.FreeFont(EFont);
    EditMode := emSprite;
    Kind := FormatType(FileName);
    if Kind <> ffSprite then
        FatalError(erInvalid_File_Format, FileName);
    if not FormatLoad(Filename, Pointer(FSprite), Size) then
        FatalError(GetError, FileName);
    if Assigned(FSprite) then begin
        SpriteIdx := FSprite^.Index;
        ImageFromSprite(SpriteIdx, False);
        NewImgX := FImage^.Width;
        NewImgY := FImage^.Height;
        Width := NewImgX;
        Height := NewImgY;
        Video^.SpriteSort;
        SpriteModified := False;
        Modified(False);
    {$IFDEF LOGS}
    Log('Sprite Loaded');
    {$ENDIF}
    end else
        NewSpriteFile;
end;

procedure SaveFile;
begin
    StandByBar;
    if EditMode = emFont then begin
        SetFontChar;
        GetFontChar(FChar);
        ImageToEdit(EOfsX, EOfsY);
        if not FormatSave(FileName, EFont, FontBitmapSize(EFont)) then
            FatalError(GetError, 'save ' + FileName + ' font');
    end else if EditMode = emSprite then begin
        ImageToSprite(SpriteIdx, SpriteMode = smMask);
        if Not FormatSave(FileName, FSprite, Video^.SpriteSizeOf(FSprite)) then
            FatalError(GetError, 'save file ' + FileName);
    end else begin
        if Not FormatSave(FileName, FImage, Video^.ImageSizeOf(FImage)) then
            FatalError(GetError, 'save file ' + FileName);
    end;
    Modified(False);
end;

procedure SaveFileAs(Ext : String);
begin
    {$IFDEF LOGS}
        Log('file save as ' + FileName + '(' + EXT + ') ');
    {$ENDIF}
    StandByBar;
    {$IFDEF LOGS}
        Log('file save as ' + FileName + '(' + EXT + ') ');
    {$ENDIF}
    if EditMode = emFont then begin
        SaveFile;
    end else if Ext = 'IGS' then begin
        if EditMode <> emSprite then begin
            FSPrite := Video^.NewSprite(FImage^.Width, FImage^.Height, 1, False);
            if Assigned(FSprite) then begin
                Video^.FreeImage(FSprite^.Sprites^[0].Image);
                FSprite^.Sprites^[0].Image := Video^.Cloneimage(FImage);
                FSprite^.Sprites^[0].BMask := Video^.ImageToMask(
                    FSprite^.Sprites^[0].Image, EraseColor);
                if Not FormatSave(FileBase(FileName) + Ext, FSprite, Video^.SpriteSizeOf(FSprite)) then
                   FatalError(GetError, 'save ' + Ext + ' file ' + FileName);
                Video^.FreeSprite(FSprite);
            end;
            StatusBar;
        end else begin
            ImageToSprite(SpriteIdx, SpriteMode = smMask);
            if Not FormatSave(FileBase(FileName) + Ext, FSprite, Video^.SpriteSizeOf(FSprite)) then
                FatalError(GetError, 'save ' + Ext + ' file ' + FileName);
            Modified(False);
            StatusBar;
        end;
    end else begin
        if Not FormatSave(FileBase(FileName) + Ext, FImage, Video^.ImageSizeOf(FImage)) then
            FatalError(GetError, 'save ' + Ext + ' file ' + FileName);
        if (UCase(FileExt(FileName)) <> Ext) and ModifiedFlag then begin
            Modified(true);
        end else
            Modified(False);
        StatusBar;
    end;
end;

procedure SaveFileAsFmt(Kind : word; FmtIndex : integer);
var
    FH : PFormatHandler;
    Ext : String;
begin
    if Kind = ffSprite then begin
        Ext := 'IGS';
    end else begin
        FH := FormatIndex(Kind, FmtIndex);
        if not Assigned(FH) then
            FatalError(erInvalid_Pointer_Operation, 'format lookup');
        Ext := FH^.Exts;
        if Pos(';', Ext) > 0 then
            Ext := Copy(Ext, 1, Pos(';', Ext) - 1 );
    end;
    SaveFileAs(Ext);
end;

procedure DrawEditBit (X, Y : integer; Color : word);
begin
    X := DrawOfsX + X * ScaleX;
    Y := DrawOfsY + Y * ScaleY;
    Video^.Region(X + 1, Y + 1, X + ScaleX - 1, Y + ScaleY - 1, Color);
end;

procedure DrawEditBorder (X, Y : integer; Color : word);
begin
    X := DrawOfsX + X * ScaleX;
    Y := DrawOfsY + Y * ScaleY;
    Video^.Region(X, Y, X + ScaleX, Y, Color);
    Video^.Region(X, Y + ScaleY, X + ScaleX, Y + ScaleY, Color);
    Video^.Region(X, Y + 1, X, Y + ScaleY - 1, Color);
    Video^.Region(X + ScaleX, Y + 1, X + ScaleX, Y + ScaleY - 1, Color);
end;

procedure DrawEditImage;
var
    X, Y : integer;
begin
    for Y := 0 to EImage^.Height - 1 do
        for X := 0 to EImage^.Width - 1  do begin
            DrawEditBorder(X, Y, 0);
            DrawEditBit(X, Y, Video^.ImageGetPixel(EImage, X, Y));
        end;
end;

procedure DrawAsciiBorder(X, Y : integer; Color : Word);
begin
    Video^.Region(X - 1, Y - 1, X + Width, Y - 1, Color);
    Video^.Region(X - 1, Y + Height, X + Width, Y + Height, Color);
    Video^.Region(X - 1, Y, X - 1, Y + Height, Color);
    Video^.Region(X + Width, Y, X + Width, Y + Height, Color);
end;

function AsciiAtXY( X, Y : integer) : integer;
begin
    AsciiAtXY := -1;
    X := ((X - DrawFntX) div (Width + 1));
    Y := ((Y - DrawOfsY) div (Height + 1));
    if (X < 0) or (Y < 0) or (X >= AsciiWide) or (Y >= AsciiHigh) then exit;
    AsciiAtXY := AsciiStart + Y * AsciiWide + X;
end;

procedure DrawAsciiTable;
var
    Hold : TFontSettings;
    FC : TColor;
    X, Y, TX, TY : integer;
begin
    GetFontSettings(Hold);
    Video^.SetFont(EFont);
    Video^.SetMonospace(True);
    Video^.SetFontDirection(dmRight);
    if AsciiLast <> AsciiStart then begin
        Video^.Region(DrawFntX - 1, DrawOfsY - 1,
            DrawFntX + AsciiWide * (Width + 1) + 1, DrawOfsY + AsciiHigh * (Height + 1) + 1,
            0);
        for Y := 0 to AsciiHigh - 1 do
            for X := 0 to AsciiWide - 1 do begin
                TX := DrawFntX + X * (Width + 1);
                TY := DrawOfsY + Y * (Height + 1);
                if AsciiStart + X + Y * AsciiWide < 256 then begin
                    if AsciiStart + X + Y * AsciiWide = FChar then begin
                        FC := FontHot;
                        POfsX := TX;
                        POfsY := TY;
                    end else
                        FC := FontCold;
                    Video^.PutChar(TX, TY, Char(AsciiStart + X + Y * AsciiWide), FC);
                end else if AsciiStart + Y * AsciiWide < 256 then
                    Video^.Region(TX, TY, TX + Width +2, TY + Height + 2, DeskColor)
                else
                    Video^.Region(TX - 1, TY, TX + Width +2, TY + Height + 2, DeskColor)
            end;
        DrawAsciiBorder(PofsX, PofsY, 7);
        AsciiLast := AsciiStart;
    end else begin
        Video^.Region(POfsX, POfsY, POfsX + Width -1, POfsY + Height -1, 0 );
        DrawAsciiBorder(PofsX, PofsY, 0);
        Video^.PutChar(PofsX, PofsY, Char(AsciiAtXY(POfsX, POfsY)), 8 );
        for Y := 0 to AsciiHigh - 1 do
            for X := 0 to AsciiWide - 1 do
                if AsciiStart + X + Y * AsciiWide = FChar then begin
                    TX := DrawFntX + X * (Width + 1);
                    TY := DrawOfsY + Y * (Height + 1);
                    POfsX := TX;
                    POfsY := TY;
                    Video^.Region(POfsX, POfsY, POfsX + Width -1, POfsY + Height -1, 0 );
                    Video^.PutChar(TX, TY, Char(AsciiStart + X + Y * AsciiWide), FontHot );
                end;
        DrawAsciiBorder(PofsX, PofsY, 7);
    end;
    SetFontSettings(Hold);
end;

procedure DrawPreview;
begin
    if EditMode = emFont then
        DrawAsciiTable
    else begin
        POfsX := (GetMaxX - MaxEditX div 2) - FImage^.Width div 2;
        POfsY := 88 - FImage^.Height div 2;
        if POfsX + FImage^.Width > GetMaxX - DrawOfsX then
            POfsX := GetMaxX - DrawOfsX - FImage^.Width + 1;
        if POfsX < MOfsX then POfsX := MOfsX;
        if POfsY < MOfsY then POfsY := MOfsY;
        Video^.FreeImage(HoverBox.Top);
        Video^.FreeImage(HoverBox.Bottom);
        Video^.FreeImage(HoverBox.Left);
        Video^.FreeImage(HoverBox.Right);
        if not Animating then
            Video^.PutImage(FImage, POfsX, POfsY);
    end;
end;

procedure ImageToEdit(X, Y : integer);
var
    I, J : integer;
begin
    EOfsX := X;
    EOfsY := Y;
    for J := 0 to EMaxY - 1 do
        for I := 0 to EMaxX - 1 do
            Video^.ImagePutPixel(EImage, I, J,
                Video^.ImageGetPixel(FImage, X + I, Y + J));
end;

procedure AdjustScale;
var
    V : integer;
begin
    if EImage^.Width > EImage^.Height then
        V := EImage^.Width
    else
        V := EImage^.Height;
    ScaleX := (GetMaxY - DrawOfsY * 2 - 8) div V;
    if ScaleX = 0 then ScaleX := 1;
    ScaleY := ScaleX;
end;

procedure MakeEditImage;
begin
    Video^.FreeImage(EImage);
    EOfsX := 0;
    EOfsY := 0;
    EMaxX := FImage^.Width;
    EMaxY := FImage^.Height;
    if EditMode <> emFont then begin
        if EMaxX > 16 then EMaxX := 16;
        if EMaxY > 16 then EMaxY := 16;
    end;
    EImage := Video^.NewImage(EMaxX, EMaxY);
    AdjustScale;
    ImageToEdit(EOfsX, EOfsY);
end;

procedure SetHoverBox(X, Y : integer);
begin
    if Assigned(HoverBox.Top) and ((HoverBox.Top^.Width <> EMaxX + 2) or
    (HoverBox.Left^.Height <> EMaxY)) then begin
        Video^.FreeImage(HoverBox.Top);
        Video^.FreeImage(HoverBox.Bottom);
        Video^.FreeImage(HoverBox.Left);
        Video^.FreeImage(HoverBox.Right);
    end;
    with HoverBox do begin
        if not Assigned(HoverBox.Top) then begin
            Where.X := -1;
            Where.Y := -1;
            Top    := Video^.NewImage(EMaxX + 2, 1);
            Bottom := Video^.NewImage(EMaxX + 2, 1);
            Left   := Video^.NewImage(1, EMaxY);
            Right  := Video^.NewImage(1, EMaxY);
            If not (Assigned(Top) and Assigned(Bottom) and Assigned(Left) and
            Assigned(Right)) then FatalError(erInsufficient_Memory, 'set hoverbox');
        end;
        if (Where.X <> -1) and (Where.Y <> -1) and (not Animating) then begin
            Video^.PutImage(HoverBox.Top, Where.X - 1, Where.Y - 1);
            Video^.PutImage(HoverBox.Bottom, Where.X - 1, Where.Y + EMaxY);
            Video^.PutImage(HoverBox.Left, Where.X - 1, Where.Y);
            Video^.PutImage(HoverBox.Right, Where.X + EMaxX, Where.Y);
        end;
        Where.X := X;
        Where.Y := Y;
        if (X <> -1) and (Y <> -1) and (not Animating) then begin
            Video^.GetImage(HoverBox.Top, Where.X - 1, Where.Y - 1);
            Video^.GetImage(HoverBox.Bottom, Where.X - 1, Where.Y + EMaxY);
            Video^.GetImage(HoverBox.Left, Where.X - 1, Where.Y);
            Video^.GetImage(HoverBox.Right, Where.X + EMaxX, Where.Y);
            Video^.Region(Where.X - 1, Where.Y - 1, Where.X + EMaxX, Y - 1, HoverColor);
            Video^.Region(Where.X - 1, Where.Y + EMaxY, Where.X + EMaxX, Y + EMaxY, HoverColor);
            Video^.Region(Where.X - 1, Where.Y, Where.X - 1, Where.Y + EMaxY - 1, HoverColor);
            Video^.Region(Where.X + EMaxX, Where.Y, Where.X + EMaxX, Where.Y + EMaxY - 1, HoverColor);
        end;
    end;
end;

procedure HoverMove(X, Y : integer);
begin
    if X < PofsX then X := POfsX;
    if Y < PofsY then Y := POfsY;
    if X + EMaxX >= POfsX + FImage^.Width then X := POfsX + FImage^.Width - EMaxX;
    if Y + EMaxY >= POfsY + FImage^.Height then Y := POfsY + FImage^.Height - EMaxY;
    if (HoverBox.Where.X = X) and (HoverBox.Where.Y = Y) then Exit;
    SetHoverBox(X, Y);
    EOfsX := X - POfsX;
    EOfsY := Y - POfsY;
    ImageToEdit(EOfsX, EOfsY);
    DrawEditImage;
end;

procedure HoverImage(var E : TEvent);
begin
    if (E.Buttons <> 0) then
        if (E.Position.X >= POfsX) and (E.Position.X < POfsX + FImage^.Width) and
        (E.Position.Y >= POfsY) and (E.Position.Y < POfsY + FImage^.Height) then
            HoverMove(E.Position.X - EMaxX div 2, E.Position.Y - EMaxY div 2);
end;

procedure HoverSet(X, Y : integer);
begin
    HoverMove(POfsX + X, POfsY + Y);
end;

procedure HoverEdit(X, Y : integer);
begin
    if X < DrawOfsX then X := -1 else X := (X - DrawOFSX) div ScaleX;
    if Y < DrawOfsY then Y := -1 else Y := (Y - DrawOFSY) div ScaleY;
    if (X < 0) or (Y < 0) or (X >= EImage^.Width) or (Y >= EImage^.Height) then begin
        X := -1;
        Y := -1;
    end;
    if (ELastX = X) and (ELastY = Y) then Exit;
    if ELastX >= 0 then
        DrawEditBorder(ELastX, ELastY, 0);
    if X >= 0 then
        DrawEditBorder(X, Y, 7);
    ELastX := X;
    ELastY := Y;
end;

procedure ContinueDrawing(var Event : TEvent);
const
    Extra = 1;
var
    TX, TY, TXO, TYO : integer;
begin
    TXO := EOfsX;
    TYO := EOfsY;
    TX := Event.Position.X;
    TY := Event.Position.Y;
    if TX < DrawOfsX then TX := -1 else TX := (TX - DrawOFSX) div ScaleX;
    if TY < DrawOfsY then TY := -1 else TY := (TY - DrawOFSY) div ScaleY;
    if (TX < 0) then begin
        if TXO > 0 then begin
            Dec(TXO);
            Event.Position.X := DrawOfsX + Extra;
        end;
    end else if (TX >= EImage^.Width) then begin
        if TXO < FImage^.Width -  EImage^.Width then begin
            Inc(TXO);
            Event.Position.X := DrawOfsX + EImage^.Width * ScaleX - Extra;
        end;
    end;
    if (TY < 0) then begin
        if TYO > 0 then begin
            Dec(TYO);
            Event.Position.Y := DrawOfsY + Extra;
        end;
    end else if (TY >= EImage^.Height) then begin
        if TYO < FImage^.Height -  EImage^.Height then begin
            Inc(TYO);
            Event.Position.Y := DrawOfsY + EImage^.Height * ScaleY - Extra;
        end;
    end;
    if (TXO <> EOfsX) or (TYO <> EOfsY) then begin
        HoverMove(POfsX + TXO, POfsY + TYO);
        SetMousePosition(Event.Position.X, Event.Position.Y);
    end;
end;

procedure SwitchChar(C : Integer);
begin
    if C < 0 then Exit;
    if C > 255 then Exit;
    if FChar = C then exit;
    SetFontChar;
    GetFontChar(C);
    ImageToEdit(EOfsX, EOfsY);
    DrawEditImage;
    DrawPreview;
end;

procedure HoverAscii(var E : TEvent);
var
    C : integer;
begin
    { if E.Kind and evMouseClick <> evMouseClick then Exit; }
    if not (MouseEvent(E) and (E.Buttons <> 0)) then exit;
    C := AsciiAtXY(E.Position.X, E.Position.Y);
    if (C < 0) then exit;
    E.Kind := evNull;
    if (C <> FChar) then
        SwitchChar(C);
end;

procedure Modified(State : Boolean);
begin
    if Animating then SpriteModified := True;
    if ModifiedFlag = State then Exit;
    ModifiedFlag  := State;
    MenuSetEnabled (MenuSave, State);
    StatusBar;
end;

function Alternate(var E : TEvent) : boolean;
begin
    Alternate := (E.Trigger = $02) or                  { Right Mouse, or }
        (E.Message.L and $208 = 8);                    { Right Alt Key }
end;

procedure UI_PostProcess;
var
    LX, LY : integer;
begin
    LX := ELastX;
    LY := ELastY;
    ELastX := -1;
    ELastY := -1;
    if EditMode <> emFont then begin
       SetHoverBox(-1, -1);
    end;
    ImageToEdit(EOfsX, EOfsY);
    if EditMode = emFont then begin
       SetFontChar;
    end;
    DrawEditImage;
    DrawPreview;
    if EditMode <> emFont then begin
       SetHoverBox(POfsX + EOfsX, POfsY + EOfsY);
    end;
    HoverEdit(LY, LY);
end;

procedure ImagePostProcess;
begin
    UI_PostProcess;
    Modified(True);
end;

procedure UI_Clear(var E : TEvent);
var
    C : Integer;
begin
    if Alternate(E) then
        C := EraseColor
    else
        C := DrawColor;
    Video^.ImageFill(FImage, C);
    ImagePostProcess;
end;

procedure UI_Invert(var E : TEvent);
var
    X, Y, C : Integer;
begin
    if EditMode = emFont then begin
        for Y := 0 to FImage^.Height - 1 do
            for X := 0 to FImage^.Width - 1 do
                if Video^.ImageGetPixel(FImage, X, Y) <> EraseColor then
                    Video^.ImagePutPixel(FImage, X, Y, EraseColor)
                else
                    Video^.ImagePutPixel(FImage, X, Y, DrawColor);
    end else begin
        for Y := 0 to FImage^.Height - 1 do
            for X := 0 to FImage^.Width - 1 do
                Video^.ImagePutPixel(FImage, X, Y,
                    Not Video^.ImageGetPixel(FImage, X, Y))
    end;
    ImagePostProcess;
end;

procedure UI_Rotate(var E : TEvent);
var
    NImage : PImage;
    NSize : word;
    X, Y : integer;
begin
    NSize := Video^.ImageSize(Height, Width);
    NImage := Video^.NewImage(Height, Width);
    if not Assigned(NImage) then Exit;
    if Alternate(E) then begin
        for Y := 0 to FImage^.Height - 1 do
            for X := 0 to FImage^.Width - 1 do
                Video^.ImagePutPixel(NImage, FImage^.Height - Y - 1, X, Video^.ImageGetPixel(FImage, X, Y));
    end else begin
        for Y := 0 to FImage^.Height - 1 do
            for X := 0 to FImage^.Width - 1 do
                Video^.ImagePutPixel(NImage, Y, FImage^.Width - X - 1, Video^.ImageGetPixel(FImage, X, Y));
    end;
    Video^.FreeImage(FImage);
    FImage := NImage;
    Width := FImage^.Width;
    Height := FImage^.Height;
    if EditMode = emFont then begin
        ImageToEdit(EOfsX, EOfsY);
        SetFontChar;
        DrawEditImage;
        DrawPreview;
    end else begin
        SetHoverBox(-1, -1);
        Video^.FreeImage(EImage);
        if Width = Height then begin
            MakeEditImage;
            DrawEditImage;
            DrawPreview;
            SetHoverBox(POfsX + EOfsX, POfsY + EOfsY);
        end;
    end;
    Modified(True);
end;

procedure UI_Flip(var E : TEvent);
begin
    if Alternate(E) then
        Video^.ImageMirror(FImage)
    else
        Video^.ImageFlip(FImage);
    ImagePostProcess;
end;

procedure UI_ShiftImage(var E : TEvent; Direction : byte);
begin
    if EditMode = emFont then begin { For Fonts invert shift alternate color }
        if Alternate(E) then
            Video^.ShiftImage(FImage, Direction, 1, DrawColor)
        else
            Video^.ShiftImage(FImage, Direction, 1, EraseColor);
    end else begin
        if Alternate(E) then
            Video^.ShiftImage(FImage, Direction, 1, EraseColor)
        else
            Video^.ShiftImage(FImage, Direction, 1, DrawColor);
    end;
    ImagePostProcess;
end;

procedure UI_Refresh;
var
    LX, LY : integer;
begin
    Video^.SpriteUndrawAll;
    Video^.Fill(DeskColor);
    MainMenuBar;
    ImageToEdit(EOfsX, EOfsY);
    DrawPreview;
    StatusBar;
    MenuSetEnabled(MenuSave, ModifiedFlag);
    UpdateTickMarks;
    ShowColors;
    DrawEditImage;
    if EditMode <> emFont then begin
        LX := HoverBox.Where.X;
        LY := HoverBox.Where.Y;
        HoverBox.Where.X := -1;
        HoverBox.Where.Y := -1;
        SetHoverBox(LX,LY);
    end else begin
        DrawAsciiTable;
    end;
    UIRefresh;
end;

procedure SetDrawMode(mode : word);
begin
    if (mode = cmSampleMode) and (DrawMode <> cmSampleMode) then
        LastMode := DrawMode;
    DrawMode := Mode;
    ButtonSetActive(BtnDraw, DrawMode = cmDrawMode);
    ButtonSetActive(BtnFill, DrawMode = cmFillMode);
    ButtonSetActive(BtnEyeDrop, DrawMode = cmSampleMode);
end;

procedure SetPixel(Color : word; Alternate : boolean);
var
    LX, LY : integer;
    CC : TColor;
begin
    if (ELastX < 0) or (ELastY < 0) then exit;
    case DrawMode of
        cmDrawMode : begin
            Video^.ImagePutPixel(FImage, EOfsX + ELastX, EOfsY + ELastY, Color);
            DrawEditBit(ELastX, ELastY, Color);
            { ShowEditImage; }
            if not Animating then
                Video^.PutPixel(POfsX + EOfsX + ELastX, POfsY + EOfsY + ELastY, Color)
            else
                SpriteModified := True;
            if ModifiedFlag <> True then
                Modified(True);
        end;
        cmFillMode : begin
            Video^.ImageFloodFill(FImage, EOfsX + ELastX, EOfsY + ELastY, Color);
            if ModifiedFlag <> True then
                Modified(True);
            ImagePostProcess;
        end;
        cmSampleMode : begin
            CC := Video^.ImageGetPixel(FImage, EOfsX + ELastX, EOfsY + ELastY);
            if Not Alternate then begin
                if DrawColor = CC then exit;
                DrawColor := CC
            end else begin
                if EraseColor = CC then exit;
                EraseColor := CC;
            end;
            UI_Refresh;
            SetDrawMode(LastMode);
        end;
    end;
end;

procedure PasteImage;
var
    X, Y : integer;
    C : TColor;
begin
    if not Assigned(FCopy) then Exit;
    for Y := 0 to FCopy^.Height - 1 do
        for X := 0 to FCopy^.Width - 1 do begin
            C := Video^.ImageGetPixel(FCopy, X, Y);
            if (EditMode = emFont) and (C <> 0) then C := DrawColor;
            Video^.ImagePutPixel(FImage, EOfsX + X, EOfsY + Y, C);
            DrawEditBit(X, Y, C);
            { ShowEditImage; }
            if not Animating then
                Video^.PutPixel(POfsX + EOfsX + X, POfsY + EOfsY + Y, C);
            if ModifiedFlag <> True then
                Modified(True);
        end;
    if EditMode = emFont then begin
        SetFontChar;
        GetFontChar(FChar);
    end;
    SpriteModified := True;
    ImageToEdit(EOfsX, EOfsY);
end;

procedure SpriteDataUpdate;
var
    I : Integer;
begin
    SpriteSeq := 0;
    SpriteSeqIdx := 0;
    SpriteImgs := 0;
    SpriteSeqs := 0;
    SpriteIdxPrev := SpriteIdx;
    SpriteIdxNext := SpriteIdx;
    if Assigned(FSprite) then begin
        SpriteSeq := FSprite^.Sprites^[SpriteIdx].Sequence;
        for I := 0 to FSprite^.Count - 1 do begin
            if FSprite^.Sprites^[I].Sequence = SpriteSeq then begin
                Inc(SpriteImgs);
                if (I < SpriteIdx) and
                ((I > SpriteIdxPrev) or (SpriteIdxPrev = SpriteIdx)) then begin
                    SpriteIdxPrev := I;
                    Inc(SpriteSeqIdx);
                end;
                if (I > SpriteIdxNext) and (SpriteIdxNext = SpriteIdx) then
                    SpriteIdxNext := I;
            end;
            if FSprite^.Sprites^[I].Sequence > SpriteSeqs then
                SpriteSeqs := FSprite^.Sprites^[I].Sequence;
        end;
        DisplaySpriteInfo;
    end;
end;

procedure SpriteConfig;
var
    EnoughMem : boolean;
begin
    SpriteDataUpdate;
    if Assigned(FImage) then begin
        EnoughMem := ((MaxAvail > (FImage^.DataSize + FImage^.DataSize shr 3 + 128))
         and (MemAvail > (FImage^.DataSize * 2 + FImage^.DataSize shr 3 + 128))) or
         (not Assigned(FSprite));
     end else
        EnoughMem := False;

    ButtonSetVisible(BtnImgToMask, EditMode = emSprite);
    ButtonSetVisible(BtnImgPrev, (EditMode = emSprite) and (SpriteIdx > SpriteIdxPrev));
    ButtonSetVisible(BtnImg, (EditMode = emSprite) and (SpriteMode <> smImage));
    ButtonSetVisible(BtnMask, (EditMode = emSprite) and (SpriteMode <> smMask));
    ButtonSetVisible(BtnImgNext, (EditMode = emSprite) and (SpriteIdx < SpriteIdxNext));
    ButtonSetVisible(BtnSeqPrev, (EditMode = emSprite) and (SpriteSeq > 0));
    ButtonSetVisible(BtnPlay, (EditMode = emSprite) and (SpriteIdxPrev <> SpriteIdxNext));
    ButtonSetVisible(BtnSeqNext, (EditMode = emSprite) and (SpriteSeq < SpriteSeqs));

    MenuSetEnabled(MenuImgAdd, (EditMode <> emFont) and (EnoughMem));

    MenuSetEnabled(MenuSeqAdd, (EditMode = emSprite) and (EnoughMem));
    MenuSetEnabled(MenuSeqDel, (EditMode = emSprite) and (SpriteSeqs > 0));
    MenuSetEnabled(MenuImgDel, (EditMode = emSprite) and (SpriteIdxPrev <> SpriteIdxNext));

    ButtonSetEnabled(BtnPlay, (EditMode = emSprite) and (SpriteIdxPrev <> SpriteIdxNext));
    ButtonSetEnabled(BtnImg, (EditMode = emSprite) and (SpriteMode <> smImage));
    ButtonSetEnabled(BtnMask, (EditMode = emSprite) and (SpriteMode <> smMask));

    Animating := Animating and (EditMode = emSprite) and (SpriteIdxPrev <> SpriteIdxNext);
    ButtonSetActive(BtnPlay, Animating);

{    ButtonSetEnabled(BtnImgPrev, (EditMode = emSprite) and (SpriteIdx > SpriteIdxPrev));
    ButtonSetEnabled(BtnImgNext, (EditMode = emSprite) and (SpriteIdx < SpriteIdxNext));
    ButtonSetEnabled(BtnSeqPrev, (EditMode = emSprite) and (SpriteSeq > 0));
    ButtonSetEnabled(BtnPlay, (EditMode = emSprite) and (SpriteIdxPrev <> SpriteIdxNext));
    ButtonSetEnabled(BtnSeqNext, (EditMode = emSprite) and (SpriteSeq < SpriteSeqs)); }
end;

procedure PromoteToSprite;
begin
    {$IFDEF LOGS}
    Log('PromoteToSprite');
    {$ENDIF}
    FSprite := Video^.NewSprite(FImage^.Width, FImage^.Height, 1, False);
    if Assigned(FSprite) then begin
        SpriteIdx := 0;
        Video^.FreeImage(FSprite^.Sprites^[SpriteIdx].Image);
        FSprite^.Sprites^[SpriteIdx].Image := Video^.Cloneimage(FImage);
        FSprite^.Sprites^[SpriteIdx].BMask := Video^.ImageToMask(
            FSprite^.Sprites^[SpriteIdx].Image, EraseColor);
        FileName := FileBase(FileName) + 'IGS';
        Video^.SpriteSort;
        SpriteModified := True;
        Modified(True);
    end else
        FatalError(erInsufficient_Memory, '');
    EditMode := emSprite;
    {$IFDEF LOGS}
    LogStop;
    {$ENDIF}
    SpriteConfig;
    UI_PostProcess;
end;

procedure ChangeSprite(Index : word; Mask : boolean);
begin
    ImageToSprite(SpriteIdx, SpriteMode = smMask);
    SpriteIdx := Index;
    if Mask then begin
        LastColor.Draw := DrawColor;
        LastColor.Erase := EraseColor;
        SpriteMode := smMask;
        DrawColor := DrawMaskColor;
        EraseColor := 0;
    end else begin
        if SpriteMode = smMask then begin
            DrawColor := LastColor.Draw;
            EraseColor := LastColor.Erase;
        end;
        SpriteMode := smImage;
    end;
    SpriteToImage(SpriteIdx, SpriteMode = smMask);
    ButtonSetVisible(BtnDrawColor, SpriteMode <> smMask);
    ButtonSetVisible(BtnEraseColor, SpriteMode <> smMask);
    ButtonSetEnabled(BtnDrawColor, SpriteMode <> smMask);
    ButtonSetEnabled(BtnEraseColor, SpriteMode <> smMask);
    ShowColors;
    UI_PostProcess;
    SpriteConfig;
end;

function SpriteNewImage : boolean;
var
    N : PMaskedImages;
    { I : word; }
begin
    {$IFDEF LOGS}
    Log('SpriteNewImage');
    {$ENDIF}
    ImageToSprite(SpriteIdx, SpriteMode = smMask);
    SpriteNewImage := False;
    GetMem(N, (FSprite^.Count + 1) * Sizeof(TMaskedImage));
    if not Assigned(N) then exit;
    Move(FSprite^.Sprites^, N^, FSprite^.Count * Sizeof(TMaskedImage));
    N^[FSprite^.Count].Image := Video^.CloneImage(N^[SpriteIdx].Image);;
    N^[FSprite^.Count].IMask := nil;
    N^[FSprite^.Count].BMask := Video^.CloneMask(N^[SpriteIdx].BMask);
    N^[FSprite^.Count].Sequence := SpriteSeq;
    if Assigned(N^[FSprite^.Count].Image) and
    Assigned(N^[FSprite^.Count].BMask) then begin
        FreeMem(FSprite^.Sprites, (FSprite^.Count) * Sizeof(TMaskedImage));
        FSprite^.Sprites := N;
        Inc(FSprite^.Count);
        Modified(True);
        SpriteDataUpdate;
    end else begin
        Video^.FreeImage(N^[FSprite^.Count].Image);
        Video^.FreeMask(N^[FSprite^.Count].BMask);
        FreeMem(N, (FSprite^.Count + 1) * Sizeof(TMaskedImage));
        Exit;
    end;
    SpriteNewImage := True;
end;

procedure AddSpriteFrame;
begin
    if EditMode <> emSprite then PromoteToSprite;
    SpriteModified := True;
    if SpriteNewImage then
        ChangeSprite(FSprite^.Count - 1, SpriteMode = smMask);
end;

procedure AddSpriteSequence;
begin
    if EditMode <> emSprite then PromoteToSprite;
    SpriteModified := True;
    if SpriteNewImage then begin
        Inc(SpriteSeqs);
        FSprite^.Sprites^[FSPrite^.Count - 1].Sequence := SpriteSeqs;
        ChangeSprite(FSprite^.Count - 1, SpriteMode = smMask);
    end;
end;

procedure SpriteMakeMask(var E : TEvent);
begin
    ImageToSprite(SpriteIdx, SpriteMode = smMask);
    with FSprite^.Sprites^[SpriteIdx] do begin
        if Alternate(E) then begin
            Video^.MaskInvert(BMask);
        end else begin
            SpriteMode := smImage;
            SpriteToImage(SpriteIdx, False);
            Video^.FreeMask(BMask);
            BMask := Video^.ImageToMask(FImage, EraseColor)
        end;
    end;
    Modified(True);
    SpriteMode := smMask;
    SpriteToImage(SpriteIdx, SpriteMode = smMask);
    SpriteConfig;
    UI_PostProcess;
end;

procedure ChangeSpriteSeq(Wanted : word);
var
    L, X, I : integer;
begin
    if (Wanted < 0) or (Wanted > SpriteSeqs) then exit;
    L := 0;
    X := 0;
    for I := 0 to FSprite^.Count - 1 do begin
        if FSprite^.Sprites^[I].Sequence = Wanted then begin
            L := I;
            Inc(X);
            If X > SpriteSeqIdx then break;
        end;
    end;
    ChangeSprite(L, SpriteMode = smMask);
    if Animating then Video^.SpriteChange(FSprite, L);
end;

procedure DeleteFrame;
var
    N : PMaskedImages;
    KillIdx, HIdx : word;
    I, TI : word;
begin
    {$IFDEF LOGS}
    Log('Sprite Del Frame');
    {$ENDIF}
    KillIdx := SpriteIdx;
    if SpriteIdx < SpriteIdxNext then begin
        ChangeSprite(SpriteIdxNext, SpriteMode = smMask);
        HIdx := SpriteIdxNext - 1;
    end else begin
        ChangeSprite(SpriteIdxPrev, SpriteMode = smMask);
        HIdx := SpriteIdxPrev;
    end;
    GetMem(N, (FSprite^.Count - 1) * Sizeof(TMaskedImage));
    if not Assigned(N) then exit;
    TI := 0;
    for I := 0 to FSprite^.Count - 1 do begin
        if I <> KillIdx then begin
            N^[TI] := FSprite^.Sprites^[I];
            Inc(TI);
        end else begin
            Video^.FreeImage(FSprite^.Sprites^[I].Image);
            Video^.FreeImage(FSprite^.Sprites^[I].IMask);
            Video^.FreeMask(FSprite^.Sprites^[I].BMask);
        end;
    end;
    FreeMem(FSprite^.Sprites, (FSprite^.Count) * Sizeof(TMaskedImage));
    FSprite^.Sprites := N;
    Dec(FSprite^.Count);
    Modified(True);
    SpriteIdx := HIdx;
    SpriteToImage(SpriteIdx, SpriteMode = smMask);
    SpriteConfig;
    UI_PostProcess;
end;

procedure DeleteSeq;
var
    N : PMaskedImages;
    KillSeq : word;
    I, TI : word;
begin
    {$IFDEF LOGS}
    Log('Sprite Del Sequence ' + IntStr(SpriteSeq));
    {$ENDIF}
    KillSeq := SpriteSeq;
    if SpriteSeq < SpriteSeqs then begin
        ChangeSpriteSeq(SpriteSeq + 1)
    end else
        ChangeSpriteSeq(SpriteSeq - 1);
    {$IFDEF LOGS}
    Log('KillSeq ' + IntStr(KillSeq));
    Log('Switched to ' + IntStr(SpriteSeq) + '.' + IntStr(SpriteIdx) +
        ', Sprite Count ' + IntStr(FSprite^.Count));
    {$ENDIF}
    GetMem(N, FSprite^.Count * Sizeof(TMaskedImage));
    if not Assigned(N) then exit;
    TI := 0;
    for I := 0 to FSprite^.Count - 1 do begin
        if FSprite^.Sprites^[I].Sequence <> KillSeq then begin
            N^[TI] := FSprite^.Sprites^[I];
            if N^[TI].Sequence > KillSeq then
                Dec(N^[TI].Sequence);
            {$IFDEF LOGS}
            Log('Remap Sprite ' + IntStr(FSprite^.Sprites^[I].Sequence) +
                '.' + IntStr(I) + ' to ' +
                IntStr(N^[TI].Sequence) + '.' + IntStr(TI));
            {$ENDIF}
            Inc(TI);
        end else begin
            If SpriteIdx > I then Dec(SpriteIdx);
            {$IFDEF LOGS}
            Log('Remove Sprite ' + IntStr(FSprite^.Sprites^[I].Sequence) +
                '.' + IntStr(I) + ', Sprite Index ' + IntStr(SpriteIdx));
            {$ENDIF}
            Video^.FreeImage(FSprite^.Sprites^[I].Image);
            Video^.FreeImage(FSprite^.Sprites^[I].IMask);
            Video^.FreeMask(FSprite^.Sprites^[I].BMask);
        end;
    end;
    FreeMem(FSprite^.Sprites, (FSprite^.Count) * Sizeof(TMaskedImage));
    GetMem(FSprite^.Sprites, (TI) * Sizeof(TMaskedImage));
    if Not Assigned(FSprite^.Sprites) then { no way this can happen !!
    Like checking for 1 + 1 = 3. Nope, no way. I mean, we just freed more memory
    than we are asking to allocate. }
        FatalError(erInsufficient_Memory, 'reallocate image list');
    Move(N^, FSprite^.Sprites^, (TI) * Sizeof(TMaskedImage));
    FreeMem(N, (FSprite^.Count) * Sizeof(TMaskedImage));
    FSprite^.Count := TI;
    Modified(True);
    if SpriteIdx >= FSprite^.Count then SpriteIdx := 0;
    SpriteToImage(SpriteIdx, SpriteMode = smMask);
    SpriteConfig;
    UI_PostProcess;
    {$IFDEF LOGS}
    Log('Switched to ' + IntStr(SpriteSeq) + '.' + IntStr(SpriteIdx) +
        ', Sprite Count ' + IntStr(FSprite^.Count));
    LogStop;
    {$ENDIF}
end;

procedure ToggleAnimate;
begin
    Animating := not Animating;
    SpriteConfig;
    if Animating then begin
        SpriteBehind := EraseColor;
        Video^.Region(POfsX - 1, POfsY - 1, POfsX + Width, POfsY + Height,
        SpriteBehind);
        Video^.SpriteMove(FSprite, POfsX, POfsY);
        Video^.SpriteChange(FSprite, SpriteIdx);
        { FSprite^.Animate := True; Don't want low level animation }
        Video^.SpriteShow(FSprite);
    end else begin
        Video^.SpriteHide(FSprite);
        { FSprite^.Animate := False; }
        DrawPreview;
        UI_Refresh;
    end;
end;

procedure SpriteAnimate;
var
    N, S : word;
begin
    if SpriteBehind <> EraseColor then begin
        Video^.SpriteUndraw(FSprite);
        SpriteBehind := EraseColor;
        Video^.Region(POfsX - 1, POfsY - 1, POfsX + Width, POfsY + Height,
        SpriteBehind);
    end;
    N := FSprite^.Index;
    S := FSprite^.Sprites^[N].Sequence;
    repeat
        Inc(N);
        if N >= FSprite^.Count then N := 0;
    until S = FSprite^.Sprites^[N].Sequence;
    if (N = SpriteIdx) and (SpriteModified) then begin
        ImageToSprite(SpriteIdx, SpriteMode = smMask);
        SpriteModified := False;
    end;
    Video^.SpriteChange(FSprite, N);
end;

procedure FileImport;
const
    FN : String = '';
var
    P : Pointer;
    Sz : LongInt;
    I : PImage;
    X, Y : integer;
    C : TColor;
begin
    if FN = '' then FN := FileBase(FileName);
    if DialogFile('IMPORT', FN, ffImage, true) then
        if FormatLoad(FN, P, Sz) then begin
            I := PImage(P);
            Video^.ImageExplode(I);
            for Y := 0 to I^.Height - 1 do if Y + EOfsY < FImage^.Height then
                for X := 0 to I^.Width - 1 do if X + EOfsX < FImage^.Width then begin
                    C := Video^.ImageGetPixel(I, X, Y);
                    Video^.ImagePutPixel(FImage, EOfsX + X, EOfsY + Y, C);
                end;
            ImageToEdit(EOfsX, EOfsY);
            SpriteModified := True;
            Modified(True);
            Video^.FreeImage(I);
        end;
    UI_Refresh;
end;

function FileOpen : boolean;
var
    FN : String;
begin
    FileOpen := False;
    FN := FileName;
    if DialogFile('OPEN', FN, ffBitmapFont or ffImage or ffSprite, true) then begin
        FileName := FN;
        UI_Refresh;
        StandbyBar;
        case FormatType(FileName) of
            ffBitmapFont : LoadFontFile;
            ffImage : LoadImageFile;
            ffSprite : LoadSpriteFile;
        else
            FatalError(erInvalid_File_Format, FileName);
        end;
        FileOpen := True;
    end else
        UI_Refresh;
end;

function MainLoop : boolean;
var
    NeedTerminate : boolean;
    E : TEvent;
    X, Y : integer;
    T : PImage;
    S : String;
    LTT : LongInt;
    F : boolean;
    SC : Word;
    WasDrawing : boolean;
begin
    Video^.Prepare;
    AsciiLast := -1;
    ELastX := -1;
    ELastY := -1;
    if EditMode = emFont then begin
        DrawColor := DrawFontColor;
        EraseColor := 0;
    end;
    ButtonSetVisible(BtnDrawColor, EditMode <> emFont);
    ButtonSetVisible(BtnEraseColor, EditMode <> emFont);
    ButtonSetEnabled(BtnDrawColor, EditMode <> emFont);
    ButtonSetEnabled(BtnEraseColor, EditMode <> emFont);
    SpriteMode := smImage;
    Video^.Fill(DeskColor);
    MakeEditImage;
    MakeTicks;
    DrawPreview;
    MainMenuBar;
    StatusBar;
    MenuSetEnabled(MenuImport, (EditMode <> emFont));
    MenuSetEnabled(MenuRotate, (EditMode <> emFont) or (Width = Height));
    MenuSetEnabled(MenuPaste,  Assigned(FCopy));
    MenuShow(UI.Menu);
    MouseShow;
    Video^.SpriteSort;
    NeedTerminate := False;
    if EditMode <> emFont then
        SetHoverBox(POfsX + EOfsX,POfsY +EOfsY);
    MenuSetEnabled(MenuSave, ModifiedFlag);
    MenuSetEnabled(MenuSaveAs, EditMode <> emFont);
    UpdateTickMarks;
    ShowColors;
    DrawEditImage;
    SpriteConfig;
    UIRefresh;
    PurgeEvents;
    LTT := -1;
    F := False;
    WasDrawing := False;
    Animating := False;
    LastColor.Draw := DrawColor;
    LastColor.Erase := EraseColor;
    repeat
        if LTT <> TimerTick then begin
            LTT := TimerTick;
            Video^.SpriteNextAll(False);
            if (EditMode = emSprite) and Animating and (LTT and 3 = 0) then
                SpriteAnimate;
        end;
        Video^.Update;
        While (not GetEvent(E)) and (LTT = TimerTick) do Idle;
        if E.Kind = evNull then Continue;
        Video^.Prepare;
        if MouseEvent(E) then begin
            if WasDrawing and (E.Buttons <> 0) then
                ContinueDrawing(E)
            else
                WasDrawing := False;
            MouseMove(E.Position);
        end;

        if (not UIEvents(E)) then begin
            { Left/Right Shift + Arrow Key to "Shift Image" }
            if (E.Kind = evKeyPress) and (Hi(E.Scancode) = $e0) and
            ((E.Shiftcode = 1) or (E.Shiftcode = 2)) then begin
                SC := E.ShiftCode;
                case E.ScanCode of
                    $e048 : begin { Shift+Up }
                        E.Kind := evCommand;
                        E.Command := cmImageUp;
                    end;
                    $e050 : begin { Shift+Down }
                        E.Kind := evCommand;
                        E.Command := cmImageDown;
                    end;
                    $e04b : begin { Shift+Left }
                        E.Kind := evCommand;
                        E.Command := cmImageLeft;
                    end;
                    $e04d : begin { Shift+Right }
                        E.Kind := evCommand;
                        E.Command := cmImageRight;
                    end;
                end;
                if E.Kind = evCommand then begin
                    E.Trigger := 0;
                    if SC = 2 then
                        E.Message.L := $208
                    else
                        E.Message.L := $8;
                    E.Message.H := 0;
                    PutEvent(E);
                    E.Kind := evNull;
                end;
            end;
            if E.Kind = evKeyPress then begin
                { Normal Arrow Key to Change Character or Move Edit HoverBox }
                if (E.ShiftCode and $030c = 0) then begin
                    if (EditMode = emFont) then begin
                        case E.ScanCode of
                            $e047 : SwitchChar(0); { Home }
                            $e04f : SwitchChar(255); { End }
                            $e049 : SwitchChar(FChar - AsciiWide * AsciiHigh); { PgUp }
                            $e051 : SwitchChar(FChar + AsciiWide * AsciiHigh); { PgDn }
                            $e048 : SwitchChar(FChar - AsciiWide); { Up }
                            $e050 : SwitchChar(FChar + AsciiWide); { Down }
                            $e04b : SwitchChar(FChar - 1); { left }
                            $e04d : SwitchChar(FChar + 1); { Right }
                        else
                            if E.KeyCode <> 0 then
                            SwitchChar(Lo(E.Keycode));
                        end;
                    end else begin
                        case E.ScanCode of
                            $e047 : begin
                                HoverSet(0, 0);
                            end; { Home }
                            $e04f : begin
                                HoverSet(FImage^.Width -1, FImage^.Height -1);
                            end; { End }
                            $e049 : begin
                                HoverSet(EOfsX, EOfsY - EMaxY - 1);
                            end; { PgUp }
                            $e051 : begin
                                HoverSet(EOfsX, EOfsY + EMaxY - 1);
                            end; { PgDn }
                            $e048 : begin
                                HoverSet(EOfsX, EOfsY - 1);
                            end; { Up }
                            $e050 : begin
                                HoverSet(EOfsX, EOfsY + 1);
                            end; { Down }
                            $e04b : begin
                                HoverSet(EOfsX - 1, EOfsY);
                            end; { left }
                            $e04d : begin
                                HoverSet(EOfsX + 1, EOfsY);
                            end; { Right }
                        end;
                    end;
                end;
            end else
            if MouseEvent(E) then begin
                if (EditMode = emFont) then
                    HoverAscii(E)
                else
                    HoverImage(E);
            end;
            if MouseEvent(E) then begin
                HoverEdit(E.Position.X, E.Position.Y);
                if E.Buttons and 1 = 1 then
                    SetPixel(DrawColor, false)
                else if E.Buttons and 2 = 2 then
                    SetPixel(EraseColor, true);
                WasDrawing := (ELastX >= 0) or (ELastY >= 0) and
                    (EditMode <> emFont);
            end;
        end else begin
            if (ELastX > 0) then
               HoverEdit(0, 0);
        end;

        if CommandEvent(E) then begin
            case E.Command of
                cmFileOpen : if FileOpen then Break;
                cmFileImport : FileImport;
                cmFileSave : SaveFile;
                cmFileSaveAs : begin end;
                cmSaveAsFmt..cmSaveAsMax : SaveFileAsFmt(ffImage, E.Command - cmSaveAsFmt);
                cmSaveAsSprite : SaveFileAsFmt(ffSprite, cmSaveAsSprite);
                cmExit : NeedTerminate := True;
                cmEditCopy : begin
                    if EditMode = emFont then begin
                        SetFontChar;
                        GetFontChar(FChar);
                    end;
                    ImageToEdit(EOfsX, EOfsY);
                    Video^.FreeImage(FCopy);
                    FCopy := Video^.CloneImage(EImage);
                    MenuSetEnabled(MenuPaste, Assigned(FCopy));
                end;
                cmEditPaste : PasteImage;
                cmDrawMode, cmFillMode, cmSampleMode : begin
                    SetDrawMode(E.Command);
                end;
                cmDrawColor : begin
                    ChangeDrawColor;
                    UI_Refresh;
                end;
                cmEraseColor : begin
                    ChangeEraseColor;
                    UI_Refresh;
                end;
                cmNewFont : begin
                    StandByBar;
                    NewFontFile;
                    Break;
                end;
                cmNewImage : begin
                    NewImageFile;
                    Break;
                end;
                cmNewSprite : begin
                    NewSpriteFile;
                    SpriteConfig;
                    Break;
                end;
                cmImageClear : UI_Clear(E);
                cmImageInvert : UI_Invert(E);
                cmImageRotate : begin
                    UI_Rotate(E);
                    if Width <> Height then Break;
                end;
                cmImageFlip     : UI_Flip(E);
                cmImageUp       : UI_ShiftImage(E, dmUp);
                cmImageDown     : UI_ShiftImage(E, dmDown);
                cmImageLeft     : UI_ShiftImage(E, dmLeft);
                cmImageRight    : UI_ShiftImage(E, dmRight);
                cmEditFrameAdd  : AddSpriteFrame;
                cmEditSeqAdd    : AddSpriteSequence;
                cmEditSeqDel    : DeleteSeq;
                cmEditFrameDel  : DeleteFrame;
                cmSeqNext       : ChangeSpriteSeq(SpriteSeq + 1);
                cmSeqPrev       : ChangeSpriteSeq(SpriteSeq - 1);
                cmFrameNext     : ChangeSprite(SpriteIdxNext, SpriteMode = smMask);
                cmFramePrev     : ChangeSprite(SpriteIdxPrev, SpriteMode = smMask);
                cmSwitchImage   : ChangeSprite(SpriteIdx, False);
                cmSwitchMask    : ChangeSprite(SpriteIdx, True);
                cmImgToMask     : SpriteMakeMask(E);
                cmAnimate       : ToggleAnimate;
                cmAbout : begin
{                    Video^.Fill(DeskColor); }
                    AboutBox;
                    if EditMode = emFont then
                        Break
                    else
                        UI_Refresh;
                end;
            end;
        end;

    until NeedTerminate;
    MainLoop := not NeedTerminate;
    MouseHide;
    MenuHide(UI.Menu);
    Video^.FreeImage(EImage);
end;

procedure Execute;
begin
    if FileExists(FileName) then begin
        if FileExt(FileName) = 'FNT' then
            LoadFontFile
        else if FileExt(FileName) = 'IGS' then
            LoadSpriteFile
        else
            LoadImageFile;
    end else begin
        if FileExt(FileName) = 'FNT' then
            NewFontFile
        else if FileExt(FileName) = 'IGS' then
            NewSpriteFile
        else
            NewImageFile
    end;
    if Assigned(FImage) then begin
        repeat
            { Just stay in the main loop }
        until not MainLoop;
    end;
end;

procedure MakeMenu;
var
    P, C, MainMenu : PMenuItem;
    Key  : word;
    X : PFormatHandler;
    I : integer;
begin
    GetFontSettings(MenuFont);
    MainMenu := NLSMenuItem('FILE', cmMenuFileu);
        MenuChild(MainMenu, NLSMenuItem('FILE.NEW.FONT', cmNewFont));
        MenuChild(MainMenu, NLSMenuItem('FILE.NEW.IMAGE', cmNewImage));
        MenuChild(MainMenu, NLSMenuItem('FILE.NEW.SPRITE', cmNewSprite));
        MenuChild(MainMenu, NewMenuSeparator);
        C := MenuChild(MainMenu, NLSMenuItem('FILE.OPEN', cmFileOpen));
        MenuChild(MainMenu, NewMenuSeparator);
        MenuImport := MenuChild(MainMenu, NLSMenuItem('FILE.IMPORT', cmFileImport));
        MenuChild(MainMenu, NewMenuSeparator);
        MenuSave := MenuChild(MainMenu, NLSMenuItem('FILE.SAVE', cmFileSave));
        MenuSaveAs := MenuChild(MainMenu, NLSMenuItem('FILE.SAVEAS', cmFileSaveAs));
        MenuSaveAs^.State := msDisabled;
        I := 0;
        X := FormatIndex(ffImage, 0);
        if Assigned(X) then begin
            MenuChild(MenuSaveAs, NLSMenuItem('FILE.SAVEAS.' + X^.UID, cmSaveAsFmt + I));
            P := MenuChild(MenuSaveAs, NLSMenuItem('FILE.SAVEAS.SPRIGS', cmSaveAsSprite));
            { P^.State := msDisabled; }
            Inc(I);
            X := FormatIndex(ffImage, I);
            if Assigned(X) then begin
                MenuChild(MenuSaveAs, NewMenuSeparator);
                while Assigned(X) do begin
                    if Assigned(X^.Save) then
                        MenuChild(MenuSaveAs, NLSMenuItem('FILE.SAVEAS.' + X^.UID, cmSaveAsFmt + I));
                    Inc(I);
                    X := FormatIndex(ffImage, I);
                end;
            end;
        end;
        MenuChild(MainMenu, NewMenuSeparator);
        MenuChild(MainMenu, NLSMenuItem('FILE.EXIT', cmExit));
    MenuEdit := MenuSibling(MainMenu, NLSMenuItem('EDIT', cmMenuEdit));
        MenuChild(MenuEdit, NLSMenuItem('EDIT.COPY', cmEditCopy));
        MenuPaste := MenuChild(MenuEdit, NLSMenuItem('EDIT.PASTE', cmEditPaste));
        MenuChild(MenuEdit, NewMenuSeparator);
        MenuImgAdd := MenuChild(MenuEdit, NLSMenuItem('EDIT.FRAME.ADD', cmEditFrameAdd));
        MenuImgDel := MenuChild(MenuEdit, NLSMenuItem('EDIT.FRAME.DEL', cmEditFrameDel));
        MenuChild(MenuEdit, NewMenuSeparator);
        MenuSeqAdd := MenuChild(MenuEdit, NLSMenuItem('EDIT.SEQ.ADD', cmEditSeqAdd));
        MenuSeqDel := MenuChild(MenuEdit, NLSMenuItem('EDIT.SEQ.DEL', cmEditSeqDel));

    P := MenuSibling(MainMenu, NLSMenuItem('TOOLS', cmMenuTools));
        MenuChild(P, NLSMenuItem('IMAGE.INVERT', cmImageInvert));
        MenuChild(P, NewMenuSeparator);
        MenuRotate := MenuChild(P, NLSMenuItem('IMAGE.ROTATE', cmImageRotate));
        MenuChild(P, NLSMenuItem('IMAGE.FLIP', cmImageFlip));
        C := MenuChild(P, NLSMenuItem('IMAGE.SHIFT', cmImageShift));
        MenuChild(C, NLSMenuItem('IMAGE.SHIFT.UP', cmImageUp));
        MenuChild(C, NLSMenuItem('IMAGE.SHIFT.DOWN', cmImageDown));
        MenuChild(C, NLSMenuItem('IMAGE.SHIFT.LEFT', cmImageLeft));
        MenuChild(C, NLSMenuItem('IMAGE.SHIFT.RIGHT', cmImageRight));
        MenuChild(P, NewMenuSeparator);
        MenuChild(P, NLSMenuItem('IMAGE.CLEAR', cmImageClear));
    P := MenuSibling(MainMenu, NLSMenuItem('HELP', cmMenuHelp));
    C := MenuChild(P, NLSMenuItem('HELP.ABOUT', cmAbout));
    UI.Menu := MainMenu;
    MenuRender(UI.Menu, False);
end;

procedure MakeButtons;
var
    Hold : TFontSettings;
    YP : integer;
    F : PFont;
    S : String;
begin
    GetFontSettings(Hold);
    { Color Buttons }
    S := AsPath(AppInfo.Subdir.Fonts) + '0808N-' + Language + '.FNT';
    FontBestMatch(S, F);
    if Assigned(F) then Video^.SetFont(F);
    Video^.SetMonospace(False);
    Video^.SetFontDirection(dmRight);
    GetFontSettings(BtnFont);
    BtnDrawColor := NewButtonItem(-1, -1, RawNLS('COLOR.DRAW'), cmDrawColor, kbNone);
    BtnDrawColor^.Invisible := True;
    BtnEraseColor := NewButtonItem(-1, -1, RawNLS('COLOR.ERASE'), cmEraseColor, kbNone);
    BtnEraseColor^.Invisible := True;
    { Editing Buttons }
    BtnDraw := ButtonFromAsset(BtnSet + 'DRAW.IGG', cmDrawMode, kbNone);
    BtnFill := ButtonFromAsset(BtnSet + 'FILL.IGG', cmFillMode, kbNone);
    BtnEyeDrop := ButtonFromAsset(BtnSet + 'EYEDROP.IGG', cmSampleMode, kbNone);
    BtnImg := ButtonFromAsset(BtnSet + 'IMAGE.IGG', cmSwitchImage, kbNone);
    BtnMask := ButtonFromAsset(BtnSet + 'MASK.IGG', cmSwitchMask, kbNone);
    BtnImgToMask := ButtonFromAsset(BtnSet + 'IMG2MASK.IGG', cmImgToMask, kbNone);
    BtnPlay := ButtonFromAsset(BtnSet + 'PLAY.IGG', cmAnimate, kbNone);
    BtnImgNext := ButtonFromAsset(BtnSet + 'IMGNXT.IGG', cmFrameNext, kbNone);
    BtnImgPrev := ButtonFromAsset(BtnSet + 'IMGPRV.IGG', cmFramePrev, kbNone);
    BtnSeqNext := ButtonFromAsset(BtnSet + 'SEQNXT.IGG', cmSeqNext, kbNone);
    BtnSeqPrev := ButtonFromAsset(BtnSet + 'SEQPRV.IGG', cmSeqPrev, kbNone);
    BtnDraw^.Sticky := True;
    BtnFill^.Sticky := True;
    BtnEyeDrop^.Sticky := True;
    BtnPlay^.Sticky := True;
    SetDrawMode(DrawMode);
    { Render and move into place }
    ButtonRenderAll;
    YP := GetMaxY - Hold.Font^.Height - (BtnDrawColor^.Area.Bottom - BtnDrawColor^.Area.Top) - 3;
    ButtonMove(BtnDrawColor, DrawOfsX, YP );
    ButtonMove(BtnEraseColor, BtnDrawColor^.Area.Right + 2, YP );

    ButtonMove(BtnDraw, 2, GetMaxY - 12);
    ButtonMove(BtnFill, BtnDraw^.Area.Right + 1, GetMaxY - 12);
    ButtonMove(BtnEyeDrop, BtnFill^.Area.Right + 1, GetMaxY - 12);

    ButtonMove(BtnImgToMask, BtnEyeDrop^.Area.Right + 4, GetMaxY - 12);

    ButtonMove(BtnImgPrev, BtnImgToMask^.Area.Right + 4, GetMaxY - 12);
    ButtonMove(BtnImg, BtnImgPrev^.Area.Right + 1, GetMaxY - 12);
    ButtonMove(BtnMask, BtnImgPrev^.Area.Right + 1, GetMaxY - 12);
    ButtonMove(BtnImgNext, BtnMask^.Area.Right + 1, GetMaxY - 12);

    ButtonMove(BtnSeqPrev, BtnImgNext^.Area.Right + 4, GetMaxY - 12);
    ButtonMove(BtnPlay, BtnSeqPrev^.Area.Right + 1, GetMaxY - 12);
    ButtonMove(BtnSeqNext, BtnPlay^.Area.Right + 1, GetMaxY - 12);

    SetFontSettings(Hold);
end;

procedure ProcessCommandLn;
var
    I, X, Y, P : integer;
    Opt : String;
begin
    I := 0;
    while I < ParamCount do begin
        Inc(I);
        Opt := UCase(ParamStr(I));
        if (Opt = '/H') or (Opt = '/?') then begin
            PrintHelp;
        end;
        if Opt = '/BM' then begin
            AppInfo.Driver.Video   := 'BIOSVID.DRV';
            Inc(I);
            if UCase(ParamStr(I)) <> 'LIST' then
                Val(ParamStr(I), AppInfo.VideoMode, P)
            else
                AppInfo.VideoMode:=$ffff;
            Inc(I);
        end else if Opt = '/VGA' then begin
            AppInfo.Driver.Video   := 'VGA386.DRV'; { at present, default }
        end else if Opt = '/LOG' then begin
            {$IFDEF LOGS} Logging := True; {$ENDIF}
        end else
        if Opt = '/SIZE' then begin
            Inc(I);
            Opt := UCase(ParamStr(I));
            P := Pos(',', Opt);
            X := 0;
            Y := 0;
            if P = 0 then P := Pos('X', Opt);
            if P = 0 then P := Pos(':', Opt);
            if P > 0 then begin
                X := StrInt(Copy(Opt, 1, P - 1));
                Y := StrInt(Copy(Opt, P + 1, 255));
            end else begin
                X := StrInt(Opt);
            end;
            if X < 1 then X := Y;
            if Y < 1 then Y := X;
{            if (X > 0) and (X <= MaxEditX) and (Y > 0) and (Y <= MaxEditY) then begin }
                NewImgX := X;
                NewImgY := Y;
{            end; }
            if (NewImgX < 1) or (NewImgX > MaxEditX) or (Y < 1) or (Y > MaxEditY) then begin
                WriteLn(FormatNLS('ERROR.TOOBIG', IntStr(MaxEditX) + NLSDelim + IntStr(MaxEditY)));
                FatalError(erRange_Check_Error, 'new image size');
            end;
        end else if Opt[1] = '/' then begin
            { ignore all invalid switches }
        end else begin
            FileName := Opt;
            if FileExt(FileName) = 'FNT' then begin
                NewImgX := 8;
                NewImgY := 14;
            end else begin
                NewImgX := 24;
                NewImgY := 24;
            end;

        end;
    end;
end;

procedure AddGSet(S : byte; F, L : Byte);
var
    I : integer;
begin
    For I := F to L do
        GSets[S] := GSets[S] + Chr(I);
end;

procedure Initialize;
begin
    if not MousePresent then begin
        WriteLn(RawNLS('ERROR.MOUSE'));
        Terminate(erOperation_Not_Supported);
    end;

    AppInfo.Title := 'ImgEdit';
    AppInfo.Version := Version;
    {$IFDEF LOGS} Log(AppInfo.Title + ' Init'); {$ENDIF}
    EditMode := emFont;
    EFont := nil;
    FCopy := nil;
    FImage := nil;
    FSprite := nil;
    EImage := nil;
    FileName := 'NONAME.IGG';
    BtnSet   := AsPath('BTNS1212');
    ProcessCommandLn;
    Ignite;
    If GetMaxX > 320 then
        DrawFntX := GetMaxX - 140;
    if Video^.Colors = 4 then begin
        FontHot := 3;
        FontCold := 2;
        HoverColor := 1;
    end;
    Video^.PutText(0, GetMaxY - FontHeight * 2, ' ' + FormatNLS('LOADING', AppInfo.Title), 15);
    Video^.Update;
    DrawMode := cmDrawMode;
    MakeMenu;
    MakeButtons;
{    if not AssetLoadAll('*',false) then begin
        Delay(5000);
        FatalError(GetError,RawNLS('ASSET.ERROR'));
    end; }
    FillChar(HoverBox, Sizeof(HoverBox), 0); { nil them pointers. :-) }
    EFont := CloneFont(nil);
    FillChar(GSets, Sizeof(GSets), 0);
    AddGSet(0, $30, $39);
    AddGSet(1, $41, $5A);
    AddGSet(2, $61, $7A);
    AddGSet(3, $20, $2f);
    AddGSet(3, $5b, $60);
    AddGSet(3, $7b, $7e);
    AddGSet(4, $80, $af);
    AddGSet(4, $e0, $ff);
    AddGSet(5, $00, $1f);
    AddGSet(5, $7f, $7f);
    AddGSet(6, $b0, $df);
end;

procedure Finalize;
begin
    Extinguish;
end;

begin
    Initialize;
    Execute;
    Finalize;
    { FatalError(42, 'life, the universe and everything'); }
end.