/*
**  GNU Pth - The GNU Portable Threads
**  Copyright (c) 1999-2006 Ralf S. Engelschall <rse@engelschall.com>
**  This file copyright (c) 2006 Gordon Schumacher <whiplash@pobox.com>
**
**  This file is part of GNU Pth, a non-preemptive thread scheduling
**  library which can be found at http://www.gnu.org/software/pth/.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
**  USA, or contact Ralf S. Engelschall <rse@engelschall.com>.
**
**  pth_sympipe.c: Simulate a pipe using a memory buffer for platforms
**                 that do not support them (currently DJGPP)
*/
                             /* ``The difficulty of communicating does
                                  not alleviate its necessity.''
                                            -- Albert Cahn            */

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "pth_p.h"

#define MAX_PIPE        16    // Maximum number of pseudo-pipes
#define DEF_PIPE_ALLOC  512   // Default size of pipe buffer, in bytes

#ifndef MIN
#define MIN(a,b)    (((a) <= (b)) ? (a) : (b))
#endif

#ifndef MAX
#define MAX(a,b)    (((a) >= (b)) ? (a) : (b))
#endif

typedef struct
{
   char*    buf;
   char*    start;
   char*    end;
   ssize_t  buflen;
} __pipedef;

static __pipedef  pipes[MAX_PIPE];

/* Emulate a pipe using a buffer in memory */
/* Note that you cannot treat it as halves */
intern int sympipe_open(int pipedes[2])
{
   int   i;
   for (i = 0; i < MAX_PIPE; i++)
   {
      if (pipes[i].buf == NULL)
      {
         pipedes[0] = pipedes[1] = i;
         pipes[i].start = pipes[i].end = pipes[i].buf = (char*) malloc(DEF_PIPE_ALLOC);
         if (pipes[i].buf == NULL)
         {
            errno = ENOMEM;
            return -1;
         }
         pipes[i].buflen = DEF_PIPE_ALLOC;
         return 0;
      }
   }
   errno = EMFILE;
   return -1;
}


intern int sympipe_close(int pd)
{
   if (pipes[pd].buf == NULL)
   {
      errno = EBADF;
      return -1;
   }
   free(pipes[pd].buf);
   pipes[pd].buf = NULL;
   return 0;
}


intern ssize_t sympipe_read(int pd, void *buffer, size_t length)
{
   ssize_t  actlength;

   if (pipes[pd].buf == NULL)
   {
      errno = EBADF;
      return -1;
   }
   if (pipes[pd].start == pipes[pd].end)
      return 0;
   if (pipes[pd].end > pipes[pd].start)
   {
      actlength = MIN(length, pipes[pd].end - pipes[pd].start);
      memcpy(buffer, pipes[pd].start, actlength);
      pipes[pd].start += actlength;
   }
   else
   {
      ssize_t  firstlength;
      ssize_t  secondlength = 0;

      firstlength = MIN(length, (pipes[pd].buf + pipes[pd].buflen) - pipes[pd].start);
      if (length > firstlength)
         secondlength = MIN(length - firstlength, pipes[pd].end - pipes[pd].buf);
      actlength = firstlength + secondlength;
      memcpy(buffer, pipes[pd].start, firstlength);
      if (secondlength > 0)
      {
         memcpy((char*) buffer + firstlength, pipes[pd].buf, secondlength);
         pipes[pd].start = pipes[pd].buf + secondlength;
      }
      else
         pipes[pd].start += firstlength;
   }
   if (pipes[pd].start == pipes[pd].end)
      pipes[pd].start = pipes[pd].end = pipes[pd].buf;
   return actlength;
}


intern int sympipe_write(int pd, const void *buffer, unsigned length)
{
   ssize_t  firstlength;
   ssize_t  secondlength;

   if (pipes[pd].buf == NULL)
   {
      errno = EBADF;
      return -1;
   }
   if (length == 0)
      return 0;

   do
   {
      secondlength = 0;
      // Get our transfer lengths
      if (pipes[pd].end >= pipes[pd].start)
      {
         firstlength = MIN(length, (pipes[pd].buf + pipes[pd].buflen) - pipes[pd].end);
         if (firstlength < length)
            secondlength = MIN(length - firstlength, pipes[pd].start - pipes[pd].buf);
      }
      else
         firstlength = MIN(length, pipes[pd].start - pipes[pd].end);

      // Reallocate if it won't fit
      if (firstlength + secondlength < length)
      {
         ssize_t  newbuflen = MAX(length * 2, pipes[pd].buflen * 2);
         ssize_t  datalen;
         char*    newbuf = (char*) malloc(newbuflen);

         if (pipes[pd].end >= pipes[pd].start)
         {
            datalen = pipes[pd].end - pipes[pd].start;
            memcpy(newbuf, pipes[pd].start, datalen);
         }
         else
         {
            datalen = (pipes[pd].buf + pipes[pd].buflen) - pipes[pd].start;
            memcpy(newbuf, pipes[pd].start, datalen);
            memcpy(newbuf + datalen, pipes[pd].buf, pipes[pd].end - pipes[pd].buf);
            datalen += pipes[pd].end - pipes[pd].buf;
         }
         free(pipes[pd].buf);
         pipes[pd].buf = newbuf;
         pipes[pd].start = pipes[pd].buf;
         pipes[pd].end = pipes[pd].start + datalen;
         pipes[pd].buflen = newbuflen;
      }
   } while (firstlength + secondlength < length);

   // Now copy it in
   memcpy(pipes[pd].end, buffer, firstlength);
   if (secondlength > 0)
   {
      memcpy(pipes[pd].buf, (char*) buffer + firstlength, secondlength);
      pipes[pd].end = pipes[pd].buf + secondlength;
   }
   else
      pipes[pd].end += firstlength;

   return (int) length;
}

