type
    TFileName = array [0..255] of Char;

    TSAF_Header = record
        ID    : String;
        Notes : PCharArray;
    end;

    TSAF_Chunk = record
        ID   : word;
        Size : LongInt;
    end;

    TSAF_Info = record
        Block     : TSAF_Chunk;
        PartIndex : word;
        { only valid for info block in first file }
        PartTotal  : word;
        FileTotal  : LongInt;
        Reserved1  : LongInt;
        DirTotal   : LongInt;
        Reserved2  : LongInt;
        ByteTotal  : LongInt;
        Reserved3  : LongInt;
        SliceSize  : LongInt;
        Counter    : LongInt;
        Reserved4  : LongInt;
        ReqVersion : Word;
    end;

    TSAF_Category = record
        Block : TSAF_Chunk;
        Name  : TFileName;
    end;

    TSAF_Directory = record
        Block  : TSAF_Chunk;
        Uniq   : LongInt;
        Attrib : word;
        Stamp  : LongInt;
        Name   : TFileName;
    end;

    TSAF_File = record
        Block     : TSAF_Chunk;
        Uniq      : LongInt;
        Attrib    : word;
        Stamp     : LongInt;
        Offset    : Longint;
        Reserved1 : LongInt;
        Size      : Longint;
        Reserved2 : LongInt;
        Dir       : LongInt;
        Name      : TFileName;
    end;

    TSAF_Verify = record
        Block     : TSAF_Chunk;
        Uniq      : LongInt;
        Signature : LongInt;
    end;

    TSAF_Message = record
        Block       : TSAF_Chunk;
        Uniq        : LongInt;
        Level       : word;
        Language    : TFileName;
    end;

    TSAF_AnyChunk = record { Largest Chunk Blob }
        Block       : TSAF_Chunk;
        Data        : array[0..Sizeof(TSAF_File) - Sizeof(TSAF_Chunk)] of byte;
    end;

const
    scNull      = 0;
    scPart      = 1;
    scInfo      = 2;
    scCategory  = 3;
    scDirectory = 4;
    scFile      = 5;
    scVerify    = 6;
    scMessage   = 7;

type
    PArchiveFile = ^TArchiveFile;
    TArchiveFile = object (TDiskFile)
        CanWait      : boolean;
        TestOnly,
        Report       : Boolean;
        Info,
        Summary      : TSAF_Info;
        InfoPos      : LongInt;
        Category     : String; { for updating }
        LastCategory : String; { for tracking }
        Categories   : String; { for extraction }
        ArchiveName  : String;
        Notes        : Pointer;
        NotesPos     : LongInt;
        NotesSize    : word;
        NotesName    : String;
        PsuedoSize,
        PsuedoPos : LongInt;
        NeedUpdate : boolean;
        Dirs        : PStringCache;
        NeedDir     : boolean;
        constructor Create(AOwner : PObject);
        destructor  Destroy; virtual;
        function  GetClassName : String; virtual;
        procedure Assign( AFileName : String ); virtual;
        procedure BlockRead(const ABuf; ACount: Word; var AResult: Word); virtual;
        procedure BlockWrite(const ABuf; ACount: Word; var AResult: Word); virtual;
        procedure Close; virtual;
        procedure Erase; virtual;
        function FileSize: Longint; virtual;
        function  FilePos: Longint; virtual;

        procedure Flush; virtual;
        procedure Rename( ANewName : String ); virtual;
        procedure Reset; virtual;
        procedure Rewrite; virtual;
        procedure Seek(AFilePos: Longint); virtual;
        procedure Truncate; virtual;
        function GetName : String; virtual;

        procedure WriteHeader; virtual;
        procedure ReadHeader(Display : boolean); virtual;
        procedure SetPartName; virtual;
        procedure UpdateInfo; virtual;

        procedure CreateArchive; virtual;
        procedure AppendArchive; virtual;
        procedure AppendScan; virtual;
        procedure ExtractArchive; virtual;
        procedure ReportArchive; virtual;
        procedure ArchiveSummary; virtual;
        procedure OpenArchive(Style : String; Display : boolean); virtual;
        function  Unique : LongInt; virtual;

        procedure NewSlice; virtual;
        function  SliceAvail : LongInt; virtual;
        procedure CheckSlice; virtual;
        procedure CheckCategory; virtual;
        procedure AddDir (ADirName : String; AAttr : word; ADate : LongInt); virtual;
        procedure AddFile (AFileName : String; AAttr : word; ADate : LongInt); virtual;

        procedure SpringCleaning; virtual;
        function  SliceSizeStr : String; virtual;
        procedure SetNotes(AFileName : String); virtual;
        procedure GetNotes; virtual;
        procedure AddText(AFileName : String); virtual;
        function  CompareCategories : boolean; virtual;
     end;

function Signature ( const Data; Size : word; Origin : longint) : longint; assembler;
asm
    PUSH DS
    MOV  CX, Size
	LES  BX, Origin
	PUSH ES
	POP  DX
    LDS  SI, Data
    CLD
 @@1:
    CMP     CX, 0
    JE      @@3
    LODSB
    CLC
    RCL   BX, 1
    RCL   DX, 1
    JNC   @@2
    OR    BX, 1
@@2:
    XOR   BL, AL
    DEC   CX
    JMP   @@1
@@3:
    MOV  AX, BX
    POP  DS
end;

procedure CreateDir( ADirName : String; AAtrib : word; AStamp : LongInt );
var
    F : File;
begin
    if Testing then Exit;
     if MakeDir(ADirName) <> 0 then
        ShowError('', IntStr(DosError) + FormatDelim + ADirName, DosError, True);
    if AStamp <> 0 then begin
        Assign(F, ADirName);
        SetFAttr(F, AAtrib);
        if DosError <> 0 then
            ShowError('', IntStr(DosError) + FormatDelim + ADirName, DosError, False);
        SetFTime(F, AStamp);
        if DosError <> 0 then
            ShowError('', IntStr(DosError) + FormatDelim + ADirName, DosError, False);
    end;
end;

constructor TArchiveFile.Create(AOwner : PObject);
begin
{$IFOPT D+} Debug('create', 'TArchiveFile'); {$ENDIF}
    inherited Create(AOwner);
    CheckMemory(Sizeof(TStringCache));
    Dirs := New(PStringCache, Create(DirCache));
    ArchiveName := '';
    FillChar(Info, Sizeof(Info), 0);
    with Info do begin
        Block.ID   := scInfo;
        Block.Size := Sizeof(TSAF_Info);
        ReqVersion := SupVersion;
    end;
    Category := DefaultCategory; { default }
    LastCategory := Category;
    Categories := Category;
    InfoPos := -1;
    PsuedoPos := 0;
    PsuedoSize := -1;
    Notes := nil;
    NotesSize := 0;
    NotesPos := -1;
    NotesName := '';
    NeedUpdate := False;
    NeedDir := False;
    Report := False;
    TestOnly := False;
    Summary := Info;
    CanWait := False;
end;

destructor TArchiveFile.Destroy;
begin
{$IFOPT D+} Debug('destroy', 'TArchiveFile'); {$ENDIF}
    inherited Destroy;
    if Assigned(Notes) then begin
        FreeMem(Notes, NotesSize);
        Notes := nil;
        NotesSize := 0;
    end;
    Dispose(Dirs, Destroy);
end;

function TArchiveFile.GetClassName : String;
begin
    GetClassName := 'TArchiveFile';
end;

procedure TArchiveFile.Assign( AFileName : String );
begin
{$IFOPT D+} Debug('assign', AFileName); {$ENDIF}
    ArchiveName := AFileName;
    SetPartName;
    if not TestOnly then begin
        inherited Assign(Name);
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        CheckError(@Self, True);
    end;

    with Info do begin

        if PartTotal <= PartIndex then
            PartTotal := PartIndex + 1;

        if Info.PartIndex = 0 then begin
            Block.ID   := scInfo;
            Block.Size := Sizeof(TSAF_Info);
        end else begin
            Block.ID   := scPart;
            Block.Size := Sizeof(Block) + Sizeof(PartIndex);
        end;
    end;
end;

procedure TArchiveFile.BlockRead(const ABuf; ACount: Word; var AResult: Word);
begin
{$IFOPT D+} Debug('blockread', Name + SPACE_char + IntStr(ACount)); {$ENDIF}
    if not TestOnly then begin
        inherited BlockRead(aBuf, ACount, AResult);
{$IFOPT D+} Debug('/', IntStr(Result) + SPACE_char + IntStr(AResult)); {$ENDIF}
        CheckError(@Self, True);
     end else begin
        FillChar(Pointer(@ABuf)^, ACount, 0);
        AResult := ACount;
        PsuedoPos := PsuedoPos + ACount;
     end;
end;

procedure TArchiveFile.BlockWrite(const ABuf; ACount: Word; var AResult: Word);
begin
{$IFOPT D+} Debug('blockwrite', Name + SPACE_char + IntStr(ACount)); {$ENDIF}
    if not TestOnly then begin
        inherited BlockWrite(aBuf, ACount, AResult);
{$IFOPT D+} Debug('/', IntStr(Result) + SPACE_char + IntStr(AResult)); {$ENDIF}
        CheckError(@Self, True);
     end else begin
        AResult := ACount;
        PsuedoPos := PsuedoPos + ACount;
        if PsuedoPos >= PsuedoSize then
            PsuedoSize := PsuedoPos + 1;
     end;
end;

procedure TArchiveFile.Close;
begin
{$IFOPT D+} Debug('close', Name); {$ENDIF}
    if not TestOnly then begin
        inherited Close;
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        CheckError(@Self, True);
    end else begin
        PsuedoPos := 0;
        PsuedoSize := -1;
    end;
    if NeedUpdate then
        UpdateInfo;
end;

procedure TArchiveFile.Erase;
begin
{$IFOPT D+} Debug('erase', Name); {$ENDIF}
    if not TestOnly then begin
        inherited Erase;
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        CheckError(@Self, True);
    end;
end;

function TArchiveFile.FileSize: Longint;
{$IFOPT D+}
var
    R : LongInt;
begin
    Debug('filesize', Name);
    if TestOnly then
        R := PsuedoSize
    else begin
        R := inherited FileSize;
        Debug('/', IntStr(Result) + SPACE_char + IntStr(R));
        CheckError(@Self, True);
    end;
    FileSize := R;
end;
{$ELSE}
begin
    if TestOnly then
        FileSize := PsuedoSize
    else begin
        FileSize := inherited FileSize;
        CheckError(@Self, True);
    end;
end;
{$ENDIF}

function TArchiveFile.FilePos: Longint;
{$IFOPT D+}
var
    R : LongInt;
begin
    Debug('filepos', Name);
    if TestOnly then
        R := PsuedoPos
    else begin
        R := inherited FilePos;
        Debug('/', IntStr(Result) + SPACE_char + IntStr(R));
        CheckError(@Self, True);
    end;
    FilePos := R;
end;
{$ELSE}
begin
    if TestOnly then
        FilePos := PsuedoPos
    else begin
        FilePos := inherited FilePos;
        CheckError(@Self, True);
    end;
end;
{$ENDIF}

procedure TArchiveFile.Flush;
begin
{$IFOPT D+} Debug('flush', Name); {$ENDIF}
    if not TestOnly then begin
        inherited Flush;
 {$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
       CheckError(@Self, True);
    end;
end;

procedure TArchiveFile.Rename( ANewName : String );
begin
{$IFOPT D+} Debug('rename', Name + ' as ' + ANewName); {$ENDIF}
    if not TestOnly then begin
        inherited Rename (ANewName);
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        CheckError(@Self, True);
    end else
        Assign(ANewName);
end;

procedure TArchiveFile.Reset;
begin
{$IFOPT D+} Debug('reset', Name); {$ENDIF}
    if not TestOnly then begin
        repeat
        inherited Reset;
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        if CanWait and (Result = 2) then ChangeMedia(@Self);
        until (Result <> 2) or (not CanWait);
        CheckError(@Self, True);
        CanWait := True;
    end else begin
        PsuedoPos := 0;
        PsuedoSize := 0;
    end;
end;

procedure TArchiveFile.Rewrite;
begin
{$IFOPT D+} Debug('rewrite', Name); {$ENDIF}
    if not TestOnly then begin
        inherited Rewrite;
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        CheckError(@Self, True);
    end else begin
        PsuedoPos := 0;
        PsuedoSize := 0;
    end;
end;

procedure TArchiveFile.Seek(AFilePos: Longint);
begin
{$IFOPT D+} Debug('seek', Name + SPACE_char + IntStr(AFilePos)); {$ENDIF}
    if not TestOnly then begin
        inherited Seek(AFilePos);
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        CheckError(@Self, True);
    end else
        PsuedoPos := AFilePos;
end;

procedure TArchiveFile.Truncate;
begin
{$IFOPT D+} Debug('truncate', Name); {$ENDIF}
    if not TestOnly then begin
        inherited Truncate;
{$IFOPT D+} Debug('/', IntStr(Result)); {$ENDIF}
        CheckError(@Self, True);
    end else
        PsuedoSize := PsuedoPos + 1;
end;

function TArchiveFile.GetName : String;
begin
{$IFOPT D+} Debug('getname', Name); {$ENDIF}
   if not TestOnly then
        GetName := inherited GetName
    else
        GetName := BaseNameOf(Name) + '.' + ExtensionOf(Name);
end;

procedure TArchiveFile.SetPartName;
var
    S : string;
begin
{$IFOPT D+} Debug('setpartname', Name); {$ENDIF}
    if Info.PartIndex = 0 then
        S := Extension
    else
    if Info.PartIndex < 1000 then
        S := ZPad(IntStr(Info.PartIndex), 3)
    else if Info.PartIndex < 3600 then
        S := Char(55 + Info.PartIndex div 100) + ZPad(IntStr(Info.PartIndex mod 100), 2)
    else
        S := 'ERROR-' + IntStr(Info.PartIndex);
    if (Length(S) > 3) then
        ShowError('BAD_MAX', IntStr(erAccess_Denied) + FormatDelim  + ArchiveName, erAccess_Denied, True);
   Name := PathOf(ArchiveName) + BaseNameOf(ArchiveName) + '.' + S;
end;

procedure TArchiveFile.WriteHeader;
var
    S : String;
    Yr, Mn, Dy, Dw : word;
    Hr, Mi, Sc, Hn : word;
    X : TSAF_Info;
begin
{$IFOPT D+} Debug('+writeheader', Name); {$ENDIF}
    if Info.PartIndex = 0 then begin
        if InfoPos = -1 then begin
            S := Title + ' Archive ' + Format;
            WriteLn(S);
            ShowTextLn(mcVerbose, S);
            GetDate(Yr, Mn, Dy, DW);
            GetTime(Hr, Mi, Sc, Hn);
            S := ParseKeyValue(Stamper.Stamp,
                DateStr(Yr, Mn, Dy) + FormatDelim +
                TimeStr(Hr, Mi, Sc)
            );
            WriteLn(S);
            ShowTextLn(mcVerbose, S);
            if Assigned(Notes) then begin
                MaybeTextLn(mcGeneral, ParseMessage('INC_HEAD',
                    ChrStr(FormatDelim, 2) + NotesName +
                    ChrStr(FormatDelim, 5) + + SizeStr(NotesSize) + FormatDelim +
                    SizeStr(NotesSize) + ChrStr(FormatDelim, 2)));
                S := '';
                NotesPos := FilePos;
                ShowTextLn(mcVerbose, S);
                WriteRecord(Notes^, NotesSize);
            end;
            WriteChar(EOF_char);
            InfoPos := FilePos;
            WriteRecord(Info, Info.Block.Size);
            NeedUpdate := True;
        end else begin
            Seek(InfoPos);
            WriteRecord(Info, Info.Block.Size);
            NeedUpdate := False;
        end;
    end else begin
        WriteRecord(Info, Info.Block.Size);
        NeedUpdate := True;
    end;
    SpringCleaning;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.UpdateInfo;
var
    HoldPart : word;
    X : boolean;
begin
    if not NeedUpdate then exit;
{$IFOPT D+} Debug('+updateinfo', Name); {$ENDIF}
    NeedUpdate := False;
    if InfoPos <= 0 then
        ShowError('', IntStr(erInternal_Error) + FormatDelim  + ArchiveName, erInternal_Error, True);
    HoldPart := Info.PartIndex;
    Info.PartIndex := 0;
    Info.ReqVersion := SupVersion;
{    if HoldPart <> 0 then begin }
        X := (Flags and flOpened = flOpened);
        if X then Close;
        SetMode(flRandom);
        Assign(ArchiveName);
        Reset;
        MaybeTextLn(mcVerbose, ParseMessage('SAF_IUP', Name));
        WriteHeader;
        Close;
{    end else
        WriteHeader; }
    Info.PartIndex := HoldPart;
    Assign(ArchiveName);
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.ReadHeader(Display : boolean);
var
    S, C : String;
    X : LongInt;
    R, I, LF : word;
    B : TSAF_Info;
begin
{$IFOPT D+} Debug('+readheader', Name); {$ENDIF}
    Category := DefaultCategory;
    if Info.PartIndex = 0 then begin
        S := Title + ' Archive ';
        FillChar(C, Sizeof(C), 0);
        C[0] := S[0];
        ReadRecord(C[1], Length(C));
        if C <> S then
            ShowError('', IntStr(erInvalid_Format) + FormatDelim  + Name, erInvalid_Format, True);
        ThemeColor(clHeadID);
        if Display then
            ShowText(mcVerbose, C);
        LF := 0;
        if Assigned(Notes) then
            FreeMem(Notes, NotesSize);
        Notes := nil;
        NotesPos := -1;
        NotesSize := 0;
        repeat
            X := FilePos;
            BlockRead(C[1], Sizeof(C) - 1, R);
            I := 0;
            while I < R do begin
                if (C[I + 1] <> EOF_char) then begin
                    if Display and ((LF < 1) or (not Report)) then begin
                        if LF < 2 then begin
                                ShowText(mcVerbose, C[I + 1])
                        end else begin
                                ShowText(mcGeneral, C[I + 1])
                        end;
                    end;
                    if LF < 2 then begin { first two lines are header & CRLF }
                        if C[I + 1] = LF_char then
                            Inc(LF);
                        case LF of
                            1 : ThemeColor(clHeadDate);
                            2 : begin
                                ThemeColor(clHeadNote);
                                NotesPos := X + I + 1;
                            end;
                        end;
                    end;
                end else begin
                   if NotesPos <> -1 then
                        NotesSize := X + I + 1 - NotesPos;
                   Break;
                end;
                Inc(I);
            end;
        until (I < R);
        Seek(X + I + 1);
        if Display then
            ShowTextLn(mcGeneral, '');
        InfoPos := FilePos;
        ThemeColor(clNormal);
    end;
    FillChar(B, Sizeof(B), 0);
 {$IFOPT D+} Debug('+readinfo', Name); {$ENDIF}
    ReadRecord(B, Info.Block.Size);
{$IFOPT D+}
    FilePos; { Just want to display them in debugging information }
    FileSize;
{$ENDIF}
    if (B.Block.ID <> Info.Block.ID) or (B.Block.Size <> Info.Block.Size) then
        ShowError('', IntStr(erInvalid_Format) + FormatDelim  + Name, erInvalid_Format, True);
    if B.PartIndex <> Info.PartIndex then
        MakeError(@Self, erInvalid_Data, True);
    if (B.ReqVersion > SupVersion) then
        ShowError('BAD_VER', IntStr(erInvalid_Format) + FormatDelim  + Name, erInvalid_Format, False);
    if Info.PartIndex = 0 then
        Info := B;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
    SpringCleaning;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.CreateArchive;
begin
{$IFOPT D+} Debug('+createarchive', Name); {$ENDIF}
    if (Flags and flOpened = flOpened) then
        Close;
    FillChar(Info, Sizeof(Info), 0);
    with Info do begin
        Block.ID   := scInfo;
        Block.Size := Sizeof(TSAF_Info);
        SliceSize  := Slicing;
    end;
    InfoPos := -1;
    SetMode(flRandom);
    Assign(ArchiveName);
    Rewrite;
    Close;
    Reset;
    MaybeTextLn(mcGeneral, ParseMessage('SAF_CREATE', Name));
    MaybeTextLn(mcVerbose, ParseMessage('SAF_SLICING', SliceSizeStr));
    WriteHeader;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.OpenArchive(Style : String; Display : boolean);
begin
{$IFOPT D+} Debug('+openarchive', Name); {$ENDIF}
    if (Flags and flOpened = flOpened) then
        Close;
    FillChar(Info, Sizeof(Info), 0);
    with Info do begin
        Block.ID   := scInfo;
        Block.Size := Sizeof(TSAF_Info);
    end;
    InfoPos := -1;
    SetMode(flRandom);
    Assign(ArchiveName);
    Reset;
    MaybeTextLn(mcGeneral, ParseMessage(Style, Name));
    ReadHeader(Display);
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.AppendScan;
var
    Item : PStringItem;
    Search : TSearchRec;
begin
    Item := PStringItem(IncludeList^.GetFirst);

    while Assigned(Item) and (not CheckAbort) do begin
        if NoTailDelim(Item^.GetString) = Item^.GetString then
            FindFirst(Item^.GetString, faDirectory or Attribs, Search)
        else
            FindFirst(Item^.GetString + WildCard, faDirectory or Attribs, Search);
        While DosError = 0 do begin
            if (Search.Name = '.') or (Search.Name = '..') then begin
                { Ignore current and parent directory Entries }
            end else
            if (Search.Name[1] = '.') and (Attribs and faHidden <> faHidden) then begin
                { Treat .Files as hidden for more Unix and Linux like behaviour }
            end else
            if Search.Attr and faDirectory = faDirectory then begin
                if Not (NoSubDir or IsExcluded(PathOf(Item^.GetString) + Search.Name)) then begin
                    InsertInList(IncludeList, PathOf(Item^.GetString) + TailDelim(Search.Name) + WildCard);
                    if not NoEmptyDir then
                        AddDir(PathOf(Item^.GetString) + Search.Name, Search.Attr, Search.Time);
                end;
            end else begin
                if not IsExcluded(PathOf(Item^.GetString) + Search.Name) then begin
                    AddFile(PathOf(Item^.GetString) + Search.Name, Search.Attr, Search.Time);
                end;
            end;
            FindNext(Search);
        end;
        Dispose(Item, Destroy);
        Item := PStringItem(IncludeList^.GetFirst);
    end;
    ArchiveSummary;
end;

procedure TArchiveFile.AppendArchive;
var
    P : Pointer;
    C : TSAF_AnyChunk;
    X : longInt;
begin
{$IFOPT D+} Debug('+appendarchive', Name); {$ENDIF}
    OpenArchive('SAF_APPEND', False);
    if Info.PartIndex + 1 < Info.PartTotal then begin
        Close;
        Info.PartIndex := Info.PartTotal - 1;
        Assign(ArchiveName);
        Reset;
        MaybeTextLn(mcVerbose, ParseMessage('SAF_APPEND', Name));
        ReadHeader(False);
    end;
{$IFOPT D+} Debug('+quickscan', Name); {$ENDIF}
   while not EOF do begin
        X := FilePos;
        ReadRecord(C.Block, Sizeof(C.Block) );
{$IFOPT D+} Debug('+chunk', IntStr(C.Block.ID) + SPACE_char + SizeStr(C.Block.Size)); {$ENDIF}
        case C.Block.ID of
            scPart : SpringCleaning;
            scDirectory : begin end; { Add to dircache }
            scCategory : begin
                if C.Block.Size > Sizeof(TSAF_AnyChunk) then
                    MakeError(@Self, erInvalid_Data, True);
                ReadRecord(C.Data, C.Block.Size - Sizeof(C.Block) );
                P := @C;
                LastCategory := PascalStr(TSAF_Category(P^).Name);
            end;
        end;
        Inc(X, C.Block.Size);
        Seek(X);
{$IFOPT D+} Debug('-', ''); {$ENDIF}
   end;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

{$I EXTRACT.INC}

procedure TArchiveFile.ReportArchive;
begin
{$IFOPT D+} Debug('+reportarchive', Name); {$ENDIF}
    Report := True;
    ExtractArchive;
    Report := False;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

function TArchiveFile.Unique : LongInt;
begin
{$IFOPT D+} Debug('unique', Name + SPACE_char + IntStr(Info.Counter + 1)); {$ENDIF}
    Inc(Info.Counter);
    Unique := Info.Counter;
end;

function TArchiveFile.SliceAvail : LongInt;
begin
    SliceAvail := Info.SliceSize - FileSize;
end;

procedure TArchiveFile.NewSlice;
begin
{$IFOPT D+} Debug('+newslice', Name); {$ENDIF}
    UpdateInfo;
    if (Flags and flOpened = flOpened) then
        Close;
    Inc(Info.PartIndex);
    Assign(ArchiveName);
    Rewrite;
    MaybeTextLn(mcGeneral, ParseMessage('SAF_SLICE', Name));
    WriteHeader;
    SpringCleaning;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.CheckSlice;
var
    FB : TSAF_Chunk;
    Zero : array [0..512] of byte;
begin
    if SliceAvail > 512 then exit;
    if (SliceAvail > Sizeof(FB)) then begin
        FB.ID := scNull;
        FB.Size := SliceAvail;
        FillChar(Zero, Sizeof(Zero), 0);
        WriteRecord(FB, Sizeof(FB));
        WriteRecord(Zero, SliceAvail);
    end;
    NewSlice;
end;

procedure TArchiveFile.CheckCategory;
var
    Cat : TSAF_Category;
begin
    if LastCategory = Category then exit;
{$IFOPT D+} Debug('+checkcategory', Name); {$ENDIF}
    CheckSlice;
    FillChar(Cat, Sizeof(Cat), 0);
    Cat.Block.ID := scCategory;
    Cat.Block.Size := SizeOf(Cat) - Sizeof(Cat.Name) + Length(Category) + 1; { Null terminate string }
    Move(Category[1], Cat.Name, Length(Category));
    WriteRecord(Cat, Cat.Block.Size);
    LastCategory := Category;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.AddDir (ADirName : String; AAttr : word; ADate : LongInt);
var
    Dir : TSAF_Directory;
    F : File;
    Discard : word;
    DT : TDateTime;
begin
{$IFOPT D+} Debug('+adddir', Name + SPACE_char + ADirName); {$ENDIF}
    CheckCategory;
    CheckSlice;
    CheckCategory;
    FillChar(Dir, Sizeof(Dir), 0 );
    System.Assign(F, ADirName);
    with Dir do begin
        Block.ID   := scDirectory;
        Uniq := Dirs^.Find(ADirName);
        if Uniq = -1 then begin
            Inc(Info.DirTotal);
            Inc(Summary.DirTotal);
            Uniq       := Unique;
            Dirs^.Put(Uniq, ADirName);
        end;
        if ADate = 0 then begin
            GetFAttr (F, Attrib);
            GetFTime (F, Stamp); { Is broken or just doesn't work in DOSBox }
            if Stamp = 0 then begin
                GetDate(DT.Year, DT.Month, DT.Day, Discard);
                GetTime(DT.Hour, DT.Min, DT.Sec, Discard);
                PackTime(DT, Stamp);
            end;
            DOSError := 0;
        end else begin
            Attrib     := AAttr;
            Stamp      := ADate;
        end;
        Block.Size := SizeOf(Dir) - Sizeof(Name) + Length(ADirName) + 1; { Null terminate string }
        Move(ADirName[1], Name, Length(ADirName));
    end;
    MaybeTextLn(mcGeneral, ParseMessage('INC_DIR',
        IntStr(Dir.Block.ID) + FormatDelim + SizeStr(Dir.Block.Size) + FormatDelim +
        ADirName + FormatDelim + IntStr(Dir.Uniq) + FormatDelim +
        AttribStr(Dir.Attrib) + FormatDelim + StampStr(Dir.Stamp) ));
    WriteRecord(Dir, Dir.Block.Size);
    NeedUpdate := True;
    NeedDir := False;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.AddFile (AFileName : String; AAttr : word; ADate : LongInt);
var
    AF : TSAF_File;
    SF : TSAF_Verify;
    bSize, cSize, wSize : LongInt;
    Buf : Pointer;
    LastPart : LongInt;
    UDir : LongInt;
    XDir, XFileName, MDir : String;
begin
{$IFOPT D+} Debug('+addfile', Name + SPACE_char + AFileName); {$ENDIF}
    CheckCategory;
    CheckSlice;

    XFileName := AFileName;
    UDir := -1;
    XDir := PathOf(AFileName);
    if (XDir <> '') and (XDir <> '\') then
        XDir := NoTailDelim(XDir);
    if (XDir <> '') then begin
        UDir := Dirs^.Find(XDir);
        if (UDir = -1) or NeedDir then begin
            AddDir(XDir, 0, 0);
            UDir := Dirs^.Find(XDir);
        end;
        if UDir <> -1 then
            XFileName := System.Copy(AFileName, Length(PathOf(AFileName)) + 1,
            Length(AFileName));
    end;
{$IFOPT D+} Debug('', XFileName + ' (' + IntStr(UDir) + ') ' + XDir); {$ENDIF}

    MDir := XDir;
    if MDir <> '' then
        MDir := TailDelim(MDir);
    CheckMemory(BufferSize);
    GetMem(Buf, BufferSize);
    Inc(Info.FileTotal);
    Inc(Summary.FileTotal);
    FillChar(AF, Sizeof(AF), 0 );
    FillChar(SF, Sizeof(SF), 0 );
    CheckMemory(Sizeof(TDiskFile));
    FileTemp := New(PDiskFile, Create(nil));
    FileTemp^.Assign(AFileName);
    with SF do begin
        Block.ID := scVerify;
        Block.Size := Sizeof(TSAF_Verify);
    end;
    with AF do begin
        Block.ID   := scFile;
        Uniq       := Unique;
        SF.Uniq    := Uniq;
        Attrib     := AAttr;
        Stamp      := ADate;
        Dir        := UDir;
{       Compression := 0; }
{       Offset      := 0; }
{       Size        := 0; }
{       Name        := null; }
        cSize      := SizeOf(AF) - Sizeof(Name) + Length(XFileName) + 1; { Null terminate string }
        Block.Size := cSize;
        GetFSize(FileTemp^.FRec, Size);
        if DOSError <> 0 then begin
            Size := 0;
            ShowError('', IntStr(DOSError) + FormatDelim  + AFileName, DOSError, False);
        end else begin
            Inc(Info.ByteTotal, Size);
            Inc(Summary.ByteTotal, Size);
        end;
        Move(XFileName[1], Name, Length(XFileName));
    end;
    if AF.Size = 0 then begin
        MaybeTextLn(mcGeneral, ParseMessage('INC_FILE',
            IntStr(AF.Block.ID) + FormatDelim + SizeStr(AF.Block.Size) + FormatDelim +
            XFileName + FormatDelim + IntStr(AF.Uniq) + FormatDelim +
            AttribStr(AF.Attrib) + FormatDelim + StampStr(AF.Stamp) +  FormatDelim +
            IntStr(0) + FormatDelim + SizeStr(0) + FormatDelim +
            SizeStr(0) + FormatDelim + MDir + FormatDelim +
            IntStr(UDir) ));
        WriteRecord(AF, AF.Block.Size);
    end else begin
        LastPart := Info.PartIndex;
        FileTemp^.SetMode(flRead);
        FileTemp^.Reset;
        if FileTemp^.Result <> 0 then
            ShowError('', IntStr(FileTemp^.Result) + FormatDelim  + FileTemp^.Name, FileTemp^.Result, True);
        while AF.Size - AF.Offset > 0 do begin
            CheckCategory;
            CheckSlice;
            CheckCategory;
            AF.Offset := FileTemp^.FilePos;
            wSize := AF.Size - AF.Offset;
            if wSize > SliceAvail - cSize then
                wSize := SliceAvail - cSize;
            AF.Block.Size := wSize + cSize;
            if LastPart <> Info.PartIndex then begin
                if AF.Dir <> -1 then begin
                    Dirs^.Put(UDir, XDir);
                    AddDir(XDir, 0, 0);
                end;
            end;
            MaybeTextLn(mcGeneral, ParseMessage('INC_' +
                WhichStr(LastPart <> Info.PartIndex, 'FILE', 'CONT'),
                IntStr(AF.Block.ID) + FormatDelim + SizeStr(AF.Block.Size) + FormatDelim +
                XFileName + FormatDelim + IntStr(AF.Uniq) + FormatDelim +
                AttribStr(AF.Attrib) + FormatDelim + StampStr(AF.Stamp) +  FormatDelim +
                IntStr(AF.Offset) + FormatDelim + SizeStr(wSize) + FormatDelim +
                SizeStr(AF.Size) + FormatDelim + MDir + FormatDelim +
                IntStr(UDir) ));
            LastPart := Info.PartIndex;
            WriteRecord(AF, cSize);
            repeat
                bSize := wSize;
                if bSize > BufferSize then
                    bSize := BufferSize;
                FileTemp^.ReadRecord(Buf^, bSize);
                SF.Signature := Signature(Buf^, bSize, SF.Signature);
                if FileTemp^.Result <> 0 then
                    ShowError('', IntStr(FileTemp^.Result) + FormatDelim  + FileTemp^.Name,
                    FileTemp^.Result, True);
                WriteRecord(Buf^, bSize);
                dec(wSize, bSize);
            until wSize = 0;
            AF.Offset := FileTemp^.FilePos;
        end;
        WriteRecord(SF, Sizeof(SF));
        FileTemp^.Close;
    end;
    Dispose(FileTemp, Destroy);
    FileTemp := nil;
    FreeMem(Buf, BufferSize);
    NeedUpdate := True;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.SpringCleaning;
begin
{$IFOPT D+} Debug('springcleaning', Name); {$ENDIF}
    { Dirs^.Clear; }
    NeedDir := True;
    LastCategory := DefaultCategory;
end;

procedure TArchiveFile.ArchiveSummary;
begin
    MaybeTextLn(mcGeneral, ParseMessage('SUMMARY', IntStr(Info.PartTotal) + FormatDelim +
        IntStr(Summary.DirTotal) + FormatDelim + IntStr(Summary.FileTotal) + FormatDelim +
        SizeStr(Summary.ByteTotal) ) );
end;

function TArchiveFile.SliceSizeStr : String;
var
    I : integer;
begin
    SliceSizeStr := SizeStr(Info.SliceSize);
    for I := 0 to High(Floppies) do
        if Floppies[I].Bytes = Info.SliceSize then begin
            SliceSizeStr :=  ParseMessage('FLOPPY', ParseKeyValue(
            WhichStr( Floppies[I].Bytes >= 1024 * 1024, KBYtes, MBytes),
            Floppies[I].Name));
            Break;
        end;
end;

procedure TArchiveFile.SetNotes(AFileName : String);
var
    NS : LongInt;
begin
{$IFOPT D+} Debug('+setnote', AFileName); {$ENDIF}
    if Assigned(Notes) then
        FreeMem(Notes, NotesSize);
    Notes := nil;
    NotesSize := 0;
    NotesName := AFilename;
    CheckMemory(Sizeof(TDiskFile));
    FileTemp := New(PDiskFile, Create(nil));
    FileTemp^.Assign(AFileName);
    FileTemp^.SetMode(flRead);
    FileTemp^.Reset;
    if FileTemp^.Result <> 0 then
        ShowError('', IntStr(FileTemp^.Result) + FormatDelim  + FileTemp^.Name, FileTemp^.Result, True);
    NS := FileTemp^.FileSize;
    if NS >= MaxNotesSize then begin
        NS := MaxNotesSize;
        ShowError('BAD_MSG', IntStr(erInvalid_Data) + FormatDelim  + FileTemp^.Name +
        FormatDelim + SizeStr(MaxNotesSize) , erInvalid_Data, False);
    end;
    NotesSize := NS;
{$IFOPT D+} Debug('', IntStr(NotesSize)); {$ENDIF}
    CheckMemory(NotesSize);
    GetMem(Notes, NotesSize);
    FileTemp^.ReadRecord(Notes^, NotesSize);
    if FileTemp^.Result <> 0 then begin
        ShowError('', IntStr(FileTemp^.Result) + FormatDelim  + FileTemp^.Name, FileTemp^.Result, True);
        FreeMem(Notes, NotesSize);
        Notes := nil;
        NotesSize := 0;
    end;
    Dispose(FileTemp, Destroy);
    FileTemp := nil;
{$IFOPT D+} Debug('', ''); {$ENDIF}
end;

procedure TArchiveFile.GetNotes;
begin
    { fetch note from header, maybe needed for re-slicing }
{$IFOPT D+} Debug('+getnote', ''); {$ENDIF}
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

procedure TArchiveFile.AddText(AFileName : String);
var
    TF : TSAF_Message;
    bSize, cSize, wSize : LongInt;
    Buf : Pointer;
    LastPart : LongInt;
begin
{$IFOPT D+} Debug('+addtext', Name + SPACE_char + AFileName); {$ENDIF}
    CheckCategory;
    CheckSlice;

    CheckMemory(BufferSize);
    GetMem(Buf, BufferSize);
    Inc(Info.FileTotal);
    FillChar(TF, Sizeof(TF), 0 );
    CheckMemory(Sizeof(TDiskFile));
    FileTemp := New(PDiskFile, Create(nil));
    FileTemp^.Assign(AFileName);
    TF.Block.ID   := scMessage;
    cSize := SizeOf(TF) - Sizeof(TF.Language) + Length(EmbedLang) + 1; { Null terminate string }
    TF.Uniq       := Unique;
    if EmbedConf then
        TF.Level := mcAlways
    else if (Verbose > 0) then
        TF.Level   := mcVerbose
    else if (Verbose < 0) then
        TF.Level := mcCritical
    else
        TF.Level := mcGeneral;
    GetFSize(FileTemp^.FRec, wSize);
    if DOSError <> 0 then begin
        wSize := 0;
        ShowError('', IntStr(DOSError) + FormatDelim  + AFileName, DOSError, True);
    end;
    TF.Block.Size := cSize + wSize;
    Move(EmbedLang[1],TF.Language, Length(EmbedLang));

    if wSize = 0 then begin
        MaybeTextLn(mcGeneral, ParseMessage('INC_TEXT',
            IntStr(TF.Block.ID) + FormatDelim + SizeStr(TF.Block.Size) + FormatDelim +
            AFileName + FormatDelim + IntStr(TF.Uniq) + ChrStr(FormatDelim, 3) +
            IntStr(0) + FormatDelim + SizeStr(0) + FormatDelim +
            SizeStr(0) + ChrStr(FormatDelim, 2) ));
        WriteRecord(TF, TF.Block.Size);
    end else begin
        LastPart := Info.PartIndex;
        FileTemp^.SetMode(flRead);
        FileTemp^.Reset;
        if FileTemp^.Result <> 0 then
            ShowError('', IntStr(FileTemp^.Result) + FormatDelim  + FileTemp^.Name, FileTemp^.Result, True);
        while FileTemp^.FileSize - FileTemp^.FilePos > 0 do begin
            CheckCategory;
            CheckSlice;
            CheckCategory;
            wSize := FileTemp^.FileSize - FileTemp^.FilePos;
            if wSize > SliceAvail - cSize then
                wSize := SliceAvail - cSize;
            TF.Block.Size := wSize + cSize;
            MaybeTextLn(mcGeneral, ParseMessage('INC_' +
                WhichStr(LastPart <> Info.PartIndex, 'MORE', 'TEXT'),
                IntStr(TF.Block.ID) + FormatDelim + SizeStr(TF.Block.Size) + FormatDelim +
                AFileName + FormatDelim + IntStr(TF.Uniq) + ChrStr(FormatDelim, 4) +
                SizeStr(wSize) + FormatDelim + SizeStr(FileTemp^.FileSize) +
                 ChrStr(FormatDelim, 2) ));
            LastPart := Info.PartIndex;
            WriteRecord(TF, cSize);
            repeat
                bSize := wSize;
                if bSize > BufferSize then
                    bSize := BufferSize;
                FileTemp^.ReadRecord(Buf^, bSize);
                if FileTemp^.Result <> 0 then
                    ShowError('', IntStr(FileTemp^.Result) + FormatDelim  + FileTemp^.Name,
                    FileTemp^.Result, True);
                WriteRecord(Buf^, bSize);
                dec(wSize, bSize);
            until wSize = 0;
        end;
        FileTemp^.Close;
    end;
    Dispose(FileTemp, Destroy);
    FileTemp := nil;
    FreeMem(Buf, BufferSize);
    NeedUpdate := True;
{$IFOPT D+} Debug('-', ''); {$ENDIF}
end;

function TArchiveFile.CompareCategories : boolean;
var
    TempStr, TempCat, TempCur : String;
begin
    CompareCategories := True;
    TempCat := Ucase(Categories);
    TempCur := CategoryDelim + Ucase(LastCategory) + CategoryDelim;
    if (Pos(CategoryDelim + '*' + CategoryDelim, TempCur) > 0) then
        Exit;
    while TempCat <> '' do begin
        TempStr := PullStr(CategoryDelim, TempCat);
        if (TempStr = '*') or
        ((TempStr <> '') and (Pos(CategoryDelim + TempStr + CategoryDelim, TempCur) > 0)) then begin
            Exit;
        end;
    end;
    CompareCategories := False;
end;