; ****************************************************************************
; `convert' - various binary to decimal and hexadecimal conversion routines
;   Written by Michal H. Tyc
;
; This file is part of `BOOTMGR - multi-BOOT ManaGeR'.
;
; Copyright (c) 1997-2008 BTTR Software.  All rights reserved.
;
; This program is free software; you can redistribute it and/or modify it
; under the terms of the `MODIFIED' BSD LICENSE.  Please see `legal.txt'
; for details.
; ****************************************************************************


; Unsigned byte to two ASCII hex digits.
; On entry:
;   al = number
;   es:di = buffer
; On exit:
;   es:di = byte past the buffer
;   cl = 4
;   al = low digit
;   cf = cy

byte2hex:
  push ax       ; save number
  mov cl, 4
  shr al, cl    ; al = al / 16
  call .nibble  ; convert to hex digit
  pop ax        ; restore number
  and al, 15    ; al = al mod 16
.nibble:
  cmp al, 10         ; set CF if AL < 10
; cmc                ; need CF set if AL >= 10
; adc al, 30h        ; 0..9 -> '0'..'9', 10..15 -> 3bh..40h, set AF for 40h
; daa                ; leave '0'..'9', 3bh..40h -> 'A'..'F'
;                    ; below `negated' version of the above (no CMC needed)
  sbb al, 99h - '0'  ; 0..9 -> 96h..9fh (and set AF), 10..15 -> 0a1h..0a6h
  das                ; 96h..9fh -> '0'..'9' (CF was 1), 0a1h..0a6h -> 'A'..'F'
  stosb              ; store in buffer
  ret


; Unsigned byte to ASCII (3 digits).
; On entry:
;   al = number
; (+ see pint2asc below)

byte2asc:
  xor ah, ah
  mov cx, 3
  jmp pint2asc


gmkchars
  Db 'GMK'   ; data for sectors2kmg

; Number of 512-byte sectors (unsigned dword) to ASCII
; as binary Kilo-/Mega-/Gigabytes, 6 characters, with leading spaces.
; On entry:
;   dx:ax = number of sectors
;   (ds, es):(di + 1) = text buffer
; On exit:
;   dx = 0
;   ax = nonnegative
;   [(ds, es):di] = 'K', 'M' or 'G'
;   (+ see dword2asc below)

sectors2kmg:
  shr dx, 1
  rcr ax, 1  ; dx:ax = Kbytes = sectors/2
  mov bx, 2  ; first assume Kilobytes
.tst:
  mov cx, 10   ; scale factor 2 ** 10
  test dx, dx
  jne .shft    ; dx:ax > 65536
  test ax, ax
  jns .prn     ; dx:ax <= 7fffh, further scaling not needed
.shft:
  shr dx, 1
  rcr ax, 1
  loop .shft  ; dx:ax <<= 10
  dec bx      ; next stage
  jne .tst    ; if not yet `G'
.prn:
  mov cl, [bx + gmkchars]
  mov [di + 6], cl         ; place after the digits
  mov cl, 5                ; 5 digits


; Positive integer to ASCII in decimal, with leading spaces.
; On entry:
;   ax = positive integer (ax & 8000h = 0, ax < 32768)
; (+ see below)

pint2asc:
  cwd


; Unsigned dword to ASCII in decimal, with leading spaces.
; On entry:
;   dx:ax = number
;   es:(di + 1) = buffer
;   cx = maximum length (higher digits will be rejected)
; On exit:
;   es:di = location past the buffer
;   bx = cx = 0
;   si = 10
;   cf = nc
; Remark:
;   For dword by word division we use the fact that if x = y << 16 + z, then
;   x % 10 = ((y % 10) << 16 + z) % 10,
;   x / 10 = (y / 10) << 16 + ((y % 10) << 16 + z) / 10

dword2asc:
  push dx
  push ax     ; save the number
  add di, cx  ; point at the end of the buffer
  push di     ; save for later
  std         ; store characters backward
  mov si, 10  ; divisor (base)
  mov bx, dx  ; bx:ax = x = y << 16 + z
.digit:
  xor dx, dx
  xchg ax, bx    ; dx:ax = y, bx = z
  div si         ; dx = y % 10, ax = y / 10
  xchg ax, bx    ; dx:ax = (y % 10) << 16 + z, bx = y / 10
  div si         ; dx = x % 10, ax = ((y % 10) << 16 + z) / 10
  xchg ax, dx    ; ax = x % 10, dx = (x / 10) & 0ffffh
  or al, '0'     ; make a decimal digit (ax <= 9)
  stosb          ; store in buffer
  mov ax, dx     ; bx:ax = xnew = x / 10
  or dx, bx      ; side effect: cf = nc
  loopne .digit  ; if (x = xnew) != 0
  mov al, ' '
  rep stosb      ; store leading spaces, if any
  cld            ; restore normal string direction
  pop di
  inc di         ; now di points past the buffer
  pop ax
  pop dx         ; restore the number
  ret


; Cylinder/Head/Sector to ASCII in decimal, with leading spaces.
; On entry:
;   dh = head
;   cx = cylinder/sector in BIOS format
;   (ds, es):(di + 1) = buffer
;   For chs2asc.p1:
;     cf = cy: increase c and h by one (convert max to total number)
;     cf = nc: use c and h as is
; On exit:
;   dx:ax = c * h * s
;   bx = sectors
; (+ see dword2asc above)

chs2asc:
  clc
.p1:
  lahf                ; save carry flag
  mov bx, cx
  and bx, byte 3fh    ; bx = sector(s)
  mov dl, dh
  mov dh, 0           ; dx = head(s)
  sahf                ; restore cf
  adc dx, byte 0      ; convert max to number if needed
  xchg cl, ch
  rol ch, 1
  rol ch, 1
  and ch, 3           ; unscramble cylider bits: cx = cylinder(s)
  sahf                ; restore cf
  adc cx, byte 0      ; convert max to number if needed
  push cx             ; save cylinder(s)
  push dx             ; save head(s)
  push bx             ; save sector(s)
  push dx             ; save head(s)
  xchg ax, cx         ; ax = cylinder(s)
  mov cx, 4           ; 4 digits
  call pint2asc       ; convert to ASCII
  mov byte [di], ':'  ; separator
  pop ax              ; head(s)
  mov cl, 3           ; 3 digits
  call pint2asc       ; convert to ASCII
  mov byte [di], ':'  ; separator
  pop ax              ; sector(s)
  mov cl, 2           ; 2 digits
  call pint2asc       ; convert to ASCII
  mov bx, ax          ; return sectors in bx
  pop dx              ; head(s)
  mul dx              ; ax = heads * sectors, dx = 0
  pop dx              ; cylinder(s)
  mul dx              ; dx:ax = cylinders * heads * sectors = total_sectors
  ret


; String in linebuff to a number.
; On entry:
;   cl = base (2..36)
; On exit:
;   cf = nc: ok
;     bx = number
;     [si - 1] = terminating \0
;     ax = 0
;   cf = cy: error
;     [si - 1] = invalid character
;     bx = number made up of all preceding valid characters
;     ax = undefined

asc2byte:
  mov si, linebuff
  xor bx, bx         ; initialize ID value
.sp:
  lodsb
  cmp al, ' '
  je .sp             ; skip leading spaces
.dd:
  sub al, '0'
  jb .bad            ; bad digit
  cmp al, 9
  jbe .mult          ; ok, `0' to `9'
  sub al, '@' - '9'
  jb .bad            ; below `A'
  and al, 0dfh       ; convert to upper case
  cmp al, cl
; jnb .bad
  cmc
  jc .bad            ; wrong digit for this base
.mult:
  xchg ax, bx  ; recent digit to bl, previous to al
  mul cl       ; multiply previous value by base
  jb .bad      ; overflow, value doesn't fit in byte
  add bl, al   ; add to recent digit
  jb .bad      ; overflow
  lodsb        ; get next
.tz:
  test al, al           ; final \0? (and clear cf)
  jne .dd               ; no, try to decode digit
.bad:
  ret


; Access mode to ASCII (`LBA' or `CHS').
; On entry:
;   cf = nc: LBA mode
;   cf = cy: CHS mode
;   es:di = buffer
; On exit:
;   es:di = byte past the buffer
;   dx = `LB' or `CH'
;   al = `A' or `S'

accmode:
  mov ax, 'LB'
  mov dl, 'A'
  jnb .prt
  mov ax, 'CH'
  mov dl, 'S'
.prt:
  stosw
  xchg ax, dx
  stosb
  ret


; (end of convert.inc)
