/*

   partutil.cpp - A limited partition editor for DOS
   Copyright (C) 2025 Michael B. Brutman (mbbrutman@gmail.com)
   Web page: http://www.brutman.com/PartUtil/PartUtil.html

   Description: A limited partition editing utility that makes it easier
   to dual boot between operating systems without needing a boot manager.

   The basic features are:
     * Unhide or hide a DOS or Windows partition.
     * Set a partition active.

   With those two features you can get basic boot manager functionality.
   Unhide the partition you want to boot to, mark it active, and hide
   the others.

   Additional features:
     * Save the current MBR to a backup file.
     * Load an MBR from a file.

   Slightly dangerous features you can enable in expert mode.
     * Delete any type of partition.  (FDISK will only delete DOS partitions.)
     * Completely erase the MBR.  (Useful for initial setup before FDISK.)
     * Change a partition type arbitrarily.  (Can be used to hide any
       partition, not just DOS or Window type partitions.)

   Things this program does not do:
     * Create partitions or allow you edit the start or end of a partition.
       Use a full featured partition editor for that.
     * Display extended partitons.  That can be added later, but it's not
       really needed for switching the active partition.

   This should work on nearly any machine that runs DOS.  Tested with
   DOS 2.1, DOS 3.3, DOS 5.0, DOS 6.3, and Windows 98 so far.  Tested
   machines include a PCjr with jrIDE, a 386-40 with IDE, a Pentium 133
   with a SCSI drive, an Athlon 2000, VirtualBox, and VMWare.


   --- Compiling and building ---

   Compile using OpenWatcom 1.9: wcl -0 -s -mt -os partutil.cpp

     -0 generates 8086 friendly code
     -s to turn off stack overflow checking.  (Stack usage is minimal)
     -mt to generate a COM file
     -os to tell the compiler to favor compact code over performance


   --- Licensing ---

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see https://www.gnu.org/licenses/.

*/


#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>


// The maximum number of characters of input we accept from the user.
#define MAX_LINE_LEN (70)


// Inline assembler code to work with the drive and query the BIOS.
//
// BIOS drive numbering: A=0x0, B=0x1, first HD=0x80, second HD=0x81


// diskReset
//
// On error ah has an error code. https://stanislavs.org/helppc/int_13-1.html
//
// Convert the error code to a 16 bit number; no error = 0,
// high byte = 1 if error, low byte = error code.
extern uint16_t diskReset( uint8_t drive );
#pragma aux diskReset = \
  "mov ah, 0"         \
  "int 13h"           \
  "jc  diskResetBad"  \
  "mov ax, 0"         \
  "jmp diskResetExit" \
  "diskResetBad:"     \
  "mov al, 1"         \
  "xchg ah, al"       \
  "diskResetExit:"    \
  parm [dl]           \
  value [ax];


// readSector0 and writeSector0
//
// This only works on sector 0 because we don't have any business touching
// anything else.  It also saves us from the bit shifting required for
// the CHS values.
//
// Convert the error code to a 16 bit number; no error = 0,
// high byte = 1 if error, low byte = error code.
extern uint16_t readSector0( uint8_t drive, uint8_t far * buffer );
#pragma aux readSector0 = \
  "mov ax, 0x0201"        \
  "mov cx, 0x0001"        \
  "mov dh, 0x0"           \
  "int 13h"               \
  "jnc readSector0Good"   \
  "mov al, 1"             \
  "xchg ah, al"           \
  "jmp readSector0Exit"   \
  "readSector0Good:"      \
  "mov ax, 0x0000"        \
  "readSector0Exit:"      \
  modify [cx dh]          \
  parm [dl] [es bx]       \
  value [ax];

extern uint16_t writeSector0( uint8_t drive, uint8_t far * buffer );
#pragma aux writeSector0 = \
  "mov ax, 0x0301"         \
  "mov cx, 0x0001"         \
  "mov dh, 0x0"            \
  "int 13h"                \
  "jnc writeSector0Good"   \
  "mov al, 1"              \
  "xchg ah, al"            \
  "jmp writeSector0Exit"   \
  "writeSector0Good:"      \
  "mov ax, 0x0000"         \
  "writeSector0Exit:"      \
  modify [cx dh]           \
  parm [dl] [es bx]        \
  value [ax];


// getDriveParameters
//
// Get the CHS parameters from the BIOS so that we can show the size of the drive.
//
// Convert the error code to a 16 bit number; no error = 0,
// high byte = 1 if error, low byte = error code.
extern uint16_t getDriveParameters( uint8_t drive, uint8_t far * buffer );
#pragma aux getDriveParameters = \
  "mov ah, 0x08"          \
  "int 13h"               \
  "jnc getDriveParmsGood" \
  "mov al, 1"             \
  "xchg ah, al"            \
  "jmp getDriveParmsExit" \
  "getDriveParmsGood:"    \
  "mov ax, 0"             \
  "mov es:[di], dh"       \
  "mov es:[di+1], cx"     \
  "getDriveParmsExit:"    \
  modify [al bl cx dx]    \
  parm [dl] [es di]       \
  value [ax];


// Reboot
//
// Try using the 8042 keyboard controller method first.  Use the BIOS as
// a backup.
//
// The inline assembler isn't happy to make a hard coded far jump so we make
// it look like a far return instead.
extern void rebootAsm( void );
#pragma aux rebootAsm =  \
  "mov dx, 0x64"    \
  "mov al, 0xfe"    \
  "out dx, al"      \
                    \
  "mov ax, 0x40"    \
  "mov ds, ax"      \
  "mov ax, 0x1234"  \
  "mov bx, 0x72"    \
  "mov [bx], ax" \
  "mov ax, 0xF000"  \
  "push ax" \
  "mov ax, 0xfff0"  \
  "push ax" \
  "retf";


// Text descriptions of the various partition types.
//
// There are a lot of partition types and corner cases.  I only explictly
// show the ones interesting to DOS and Windows users; anything else shows
// up as "(Other)".  Those might be interesting too, but I can't list every
// variation.
//
// For each partition type there is generally a visible version and a hidden
// version.  Hidden prevents DOS or Windows from being able to see the
// filesytem within the partition and assign it a drive letter, thus allowing
// you to keep stable drive letters in extended partitions.
//
// https://en.wikipedia.org/wiki/Partition_type

typedef struct {
  uint8_t type;
  char *name;
} PartitionType_t;

PartitionType_t partitionTypes[] = {
  {0x00, "(Unused)"},
  {0x01, "FAT12"},
  {0x02, "XENIX root"},
  {0x03, "XENIX usr"},
  {0x04, "FAT16"},
  {0x05, "Extended"},
  {0x06, "FAT16B"},
  {0x07, "HPFS/NTFS"},
  {0x0A, "Boot Mgr"},
  {0x0B, "FAT32 CHS"},
  {0x0C, "FAT32 LBA"},
  {0x0E, "FAT16B LBA"},
  {0x0F, "Extended LBA"},

  {0x11, "Hidden FAT12"},
  {0x12, "(service/hibernate)"},
  {0x14, "Hidden FAT16"},
  {0x15, "Hidden Extended"},
  {0x16, "Hidden FAT16B"},
  {0x17, "Hidden HPFS/NTFS"},
  {0x1B, "Hidden FAT32 CHS"},
  {0x1C, "Hidden FAT32 LBA"},
  {0x1E, "Hidden FAT16B LBA"},
  {0x1F, "Hidden Extended LBA"},

  {0x30, "DR Personal CPM/86"},
  {0x35, "OS/2 JFS"},

  {0x41, "Old Linux/Minux"},
  {0x42, "Old Linux swap"},
  {0x43, "Old Linux"},
  {0x44, "GoBack"},

  {0x50, "Ontrack read-only"},
  {0x51, "Ontrack read-write"},
  {0x52, "DR CPM/80"},
  {0x53, "Ontrack auxiliary 3"},
  {0x54, "Ontrack drive overlay"},
  {0x55, "EZ-Drive redirector"},
  {0x56, "EZ-Drive or ATT DOS 3"},

  {0x80, "Minux (old)"},
  {0x81, "Minux (newer)"},
  {0x82, "Linux swap"},
  {0x83, "Linux filesystem"},
  {0x85, "Linux extended"},
  {0x86, "Linux RAID superblock"},
  {0x88, "Linux partition table"},
  {0x8E, "Linux LVM"},

  {0x93, "Hidden Linux filesystem"},

  {0xE5, "Tandy large-sector FAT"},
  {0xEF, "EFI system"},

  {0xFE, "PS/2 recovery/Linux LVM"},
  {0xFF, "XENIX bad block table"},
};

char * getPartitionDesc( uint8_t type ) {
  for ( int i=0; i < sizeof(partitionTypes); i++ ) {
    if ( partitionTypes[i].type == type ) {
      return partitionTypes[i].name;
    }
  }
  return "(Other)";
}


// Partition table entries use this format:
//
// Byte 0: Head (8 bits)
// Byte 1: Bits 0-5 Sector, Bits 6-7 upper two bits of cylinders
// Byte 3: Cylinder (10 bits total, lower 8 bits here)
//
// The BIOS returns heads in DH, and cylinders/sectors in CX.
// Bits 0-5 of CL are the sectors, 6-7 of CL are the high two bits
// of the cylinders, and CH has the lower 8 bits of the cylinders.
// I return those bytes in storage organized the same way as the
// partition table stores them.

typedef uint8_t PackedCHS_t[3];


// A parsed version of CHS that is easier to work with.
typedef struct {
  uint16_t cyls;
  uint16_t heads;
  uint16_t sectors;
} ParsedCHS_t;

ParsedCHS_t chs_Parse(PackedCHS_t chs) {
  ParsedCHS_t rc;
  rc.cyls = ((chs[1] & 0xC0) << 2) | chs[2];
  rc.heads = chs[0];
  rc.sectors = chs[1] & 0x3F;
  return rc;
}


// A Partition table entry in the MBR/boot sector.
typedef struct {
  uint8_t  attributes;
  PackedCHS_t startCHS;
  uint8_t  type;
  PackedCHS_t endCHS;
  uint32_t lbaStart;
  uint32_t sectors;
} PartitionEntry_t;


// The boot sector we read from disk (sector0) and the pending boot
// sector (newSector0).
typedef struct {
  uint8_t bootCode[0x1b8];
  uint8_t diskSignature[4];
  uint8_t reserved[2];
  PartitionEntry_t partition[4];
  uint16_t bootsectorSignature;
} MBR_t;

MBR_t sector0, newSector0;

inline bool Partition_isActive( PartitionEntry_t *p ) {
  return (p->attributes & 0x80) == 0x80;
}

inline uint8_t Partition_getType( PartitionEntry_t *p ) {
  return (p->type);
}


const char *HelpStr = 
  "Mutating commands:\n"
  "  setActive <N>           Mark partition N as active\n"
  "  clearActive <N>         Clear the active flag from partition N\n"
  "  hide <N>                Make partition N look hidden\n"
  "  unhide <N>              Unhide partition N\n"
  "  setType <N> <T>         (Expert mode) Set partition N to type T\n"
  "  delete <N>              (Expert mode) Delete partition N\n"
  "  wipeMBR                 (Expert mode) Zero out the entire MBR sector\n"
  "\n"
  "Other commands:\n"
  "  show                    Show the Current MBR and the Pending MBR\n"
  "  revert                  Undo all pending changes\n"
  "  write                   Write pending changes to the MBR\n"
  "  saveCurrentMBR <file>   Save the Current MBR to a file\n"
  "  loadPendingMBR <file>   Load the Pending MBR from a file\n"
  "  savePendingMBR <file>   Save the Pending MBR to a file\n"
  "  reboot                  Reboot the machine\n"
  "  quit                    No surprises here ...\n\n"
  "\"Current MBR\" is the MBR on disk.  \"Pending MBR\" shows the pending changes.";


void readMBRFromDisk( void );
void getDriveGeometry( void );
void printPartitionTable( PartitionEntry_t *p );

void readline( char *buffer );
bool getConfirmation( void );
char *getNextToken( char *input, char *target, uint16_t bufLen );


// Functions that implement user commands
//
// Not every parameter is used, but we need to pass them when using this
// table driven approach.  (The alternative, a set of string comparisons,
// drives me crazy.)

void setActive( char *nextTokenPtr );
void clearActive( char *nextTokenPtr );
void hide( char *nextTokenPtr );
void unhide( char *nextTokenPtr );
void deletePartition( char *nextTokenPtr );
void writeToMBR( char *nextTokenPtr );
void wipeMBR( char *nextTokenPtr );
void saveMBR( char *nextTokenPtr );
void loadPendingMBR( char *nextTokenPtr );
void savePendingMBR( char *nextTokenPtr );
void setType( char *nextTokenPtr );
void revert( char *nextTokenPtr );
void show( char *nextTokenPtr );
void reboot( char *nextTokenPtr );
void help( char *nextTokenPtr );
void quit( char *nextTokenPtr );


typedef void (*cmdF_t)(char *);
typedef struct {
  char *name;
  cmdF_t code;
} dt_t;

dt_t dispatch_table[] = {
  {"setActive", setActive},
  {"clearActive", clearActive},
  {"hide", hide},
  {"unhide", unhide},
  {"delete", deletePartition},
  {"write", writeToMBR},
  {"wipeMBR", wipeMBR},
  {"saveCurrentMBR", saveMBR},
  {"loadPendingMBR", loadPendingMBR},
  {"savePendingMBR", savePendingMBR},
  {"setType", setType},
  {"revert", revert},
  {"show", show},
  {"reboot", reboot},
  {"help", help},
  {"quit", quit},
};


// Globals

ParsedCHS_t Drive_CHS;
uint32_t DriveSize, DriveSize_tenths;
bool ExpertMode = false;
bool ShowTablesBeforePrompt = true;


int main(int argc, char *argv[]) {

  puts( "\nLow Budget Partition Utility (" __DATE__ ")\n"
        "(C)opyright 2025 M Brutman (mbbrutman@gmail.com, www.brutman.com)" );

  if ( (argc == 2) && (stricmp(argv[1], "-expert")==0) ) {
    ExpertMode = true;
  }

  getDriveGeometry( );
  readMBRFromDisk( );

  // dumpBytes( stdout, (uint8_t far *)&sector0, 512 );


  bool firstPrompt = true;

  while ( 1 ) {

    if ( ShowTablesBeforePrompt ) {
      printPartitionTable( &sector0.partition[0] );
      printPartitionTable( &newSector0.partition[0] );
    }
    ShowTablesBeforePrompt = true;

    if ( firstPrompt ) {
      printf("\n(Use \"help\" to get help)\n> " );
      firstPrompt = false;
    }
    else {
      printf( "\n> " );
    }

    char line[MAX_LINE_LEN];
    readline( line );

    char command[20];
    char *nextTokenPtr = getNextToken(line, command, 20);

    int i;
    for (i=0; i < sizeof(dispatch_table); i++) {
      if ( stricmp(dispatch_table[i].name, command) == 0 ) {
        dispatch_table[i].code(nextTokenPtr);
        break;
      }
    }

    if ( i==sizeof(dispatch_table) ) {
      puts( "Unknown command - use \"help\" to see commands." );
      ShowTablesBeforePrompt = false;
    }

  }

  return 0;
}



void readMBRFromDisk( void ) {

  uint16_t rc = diskReset(0x80);
  if ( rc ) {
    printf( "\nError resetting drive: 0x%02X\n", (rc & 0xFF) );
    exit(1);
  }

  rc = readSector0(0x80, (uint8_t far *)(&sector0));
  if ( rc ) {
    printf( "\nError reading sector 0: 0x%02X\n", (rc & 0xFF) );
    exit(1);
  }

  // Changes will go in newSector0;
  newSector0 = sector0;
}

void getSize( uint32_t sectors, uint32_t &size, uint32_t &size_tenths) {
  size = sectors / 2048;
  size_tenths = ((sectors * 10) / 2048) % 10;
} 

void getDriveGeometry( void ) {

  PackedCHS_t chs;
  uint16_t rc = getDriveParameters(0x80, (uint8_t far *)chs);
  if ( rc ) {
    printf( "\nError reading drive parameters: 0x%02X\n", (rc & 0xFF) );
    exit(1);
  }

  Drive_CHS = chs_Parse(chs);
  
  Drive_CHS.cyls++; Drive_CHS.heads++;

  uint32_t sectors = (uint32_t(Drive_CHS.sectors) * uint32_t(Drive_CHS.heads)) * (uint32_t(Drive_CHS.cyls));
  getSize( sectors, DriveSize, DriveSize_tenths );

}


void printPartitionTable( PartitionEntry_t *p ) {

  bool origTable = (p == sector0.partition);

  bool notChanged = (origTable == false) && (memcmp(&sector0, &newSector0, 512) == 0);

  if ( origTable ) {
    printf("\nDrive geometry: Cyls/Heads/Sectors: %u/%u/%u (%lu.%lu MB)\n",
      Drive_CHS.cyls, Drive_CHS.heads, Drive_CHS.sectors, DriveSize, DriveSize_tenths);
  }

  printf( "\n%s partition table: %s\n",
    ( origTable ? "Current" : "Pending" ),
    ( notChanged ? "(No changes)" : "" ) );

  if ( notChanged ) return;

  puts("  #: Active     Size  Start C/H/S    End C/H/S  Type");

  for ( int i=0; i < 4; i++, p++ ) {

    char startCHSStr[12], endCHSStr[12];

    ParsedCHS_t pCHS = chs_Parse(p->startCHS);
    sprintf(startCHSStr, "%u/%u/%u", pCHS.cyls, pCHS.heads, pCHS.sectors );

    pCHS = chs_Parse(p->endCHS);
    sprintf(endCHSStr, "%u/%u/%u", pCHS.cyls, pCHS.heads, pCHS.sectors );

    uint32_t mb1, mb2;
    getSize(p->sectors, mb1, mb2);

    printf( "  %d:   %c    %6lu.%lu  %11s  %11s  0x%02X %s\n",
        i+1, (Partition_isActive( p ) ? '*' : ' '), mb1, mb2,
        startCHSStr, endCHSStr, p->type, getPartitionDesc(p->type) );

  }

}


int getSlotParameter( char *nextTokenPtr ) {

  char tmp[5];
  getNextToken(nextTokenPtr, tmp, 5);

  int parm = atoi(tmp);

  if ( (parm < 1) || (parm >4) ) {
    puts( "Error: Partition slot must be between 1 and 4." );
    return -1;
  }

  // Work with the real slot number which is zero based.  I'd rather
  // have a display error than touch the wrong table entry.
  return parm - 1;
}


void setActiveState( char *nextTokenPtr, bool makeActive ) {

  int target = getSlotParameter( nextTokenPtr );
  if ( target == -1 ) {
    return;
  }

  PartitionEntry_t *table = &newSector0.partition[0];

  if ( (makeActive == true) && (Partition_isActive( &table[target] )) ) {
    printf( "Partition %d is already active.\n", target+1 );
    return;
  }

  // Set all primary partitions to be inactive first.
  for ( int i = 0; i < 4; i++ ) {
    if ( Partition_isActive( &table[i] ) ) {
      table[i].attributes = table[i].attributes & 0x7F;
    }
  }

  // If making one active, do it here.
  if ( makeActive == true ) {
    table[target].attributes |= 0x80;
  }

  printf( "Active flag on partition %d changed.\n", target+1 );

  return;
}

void setActive( char *nextTokenPtr ) { setActiveState( nextTokenPtr, true ); }
void clearActive( char *nextTokenPtr ) { setActiveState( nextTokenPtr, false ); }


// A list of partition types that we can toggle hidden on and off for.
// If you need to manipulate something not in this list use expert mode
// and settype.

uint8_t hiddenCanBeToggled[] = {
  0x01, 0x04, 0x05, 0x06, 0x07, 0x0b, 0x0c, 0x0e, 0x0f,
  0x11, 0x14, 0x15, 0x16, 0x17, 0x1b, 0x1c, 0x1e, 0x1f
};

void setHiddenState( char *nextTokenPtr, bool makeHidden ) {

  int target = getSlotParameter( nextTokenPtr );
  if ( target == -1 ) {
    return;
  }

  PartitionEntry_t *table = &newSector0.partition[0];

  uint8_t type = table[target].type;
  uint8_t newType = 0;

  for ( int i=0; i < sizeof(hiddenCanBeToggled); i++ ) {
    if ( hiddenCanBeToggled[i] == type ) {
      if ( makeHidden ) {
        newType = type | 0x10;
      } else {
        newType = type & 0xEF;
      }
      break;
    }
  }

  if ( newType == 0 ) {
    puts( "Hiding/unhiding this partition type is not supported." );
  }

  table[target].type = newType;

  printf( "Partition %d %shidden.\n", target+1, (makeHidden ? "" : "un") );

  return;
}

void hide( char *nextTokenPtr ) { setHiddenState(nextTokenPtr, true); }
void unhide( char *nextTokenPtr ) { setHiddenState(nextTokenPtr, false); }


void saveMBRInternal( char *nextTokenPtr, MBR_t *target ) {

  char name[80];
  getNextToken( nextTokenPtr, name, 80 );

  if ( *name == 0 ) {
    puts( "You need to provide a filename to write the MBR to." );
    return;
  }

  FILE *targetFile = fopen( name, "wb" );
  if ( targetFile == NULL ) {
    printf( "Error opening \"%s\" - aborting\n", name );
    return;
  }

  int rc = fwrite( target, 512, 1, targetFile );
  if ( rc != 1 ) {
    printf( "Error writing to \"%s\"\n", name );
    fclose( targetFile );
    return;
  }

  fclose( targetFile );
  printf("MBR saved to \"%s\"\n", name );

}

void saveMBR( char *nextTokenPtr ) { saveMBRInternal( nextTokenPtr, &sector0 ); }
void savePendingMBR( char *nextTokenPtr ) { saveMBRInternal( nextTokenPtr, &newSector0 ); }


void loadPendingMBR( char *nextTokenPtr ) {

  char name[80];
  getNextToken( nextTokenPtr, name, 80 );

  if ( *name == 0 ) {
    puts( "You need to provide a filename to read from." );
    return;
  }

  FILE *targetFile = fopen( name, "rb" );
  if ( targetFile == NULL ) {
    printf( "Error opening \"%s\" - aborting\n", name );
    return;
  }

  int rc = fread( &newSector0, 512, 1, targetFile );
  if ( rc != 1 ) {
    printf( "Error reading from \"%s\"\n", name );
    fclose( targetFile );
    return;
  }

  fclose( targetFile );
  printf("Pending MBR loaded from \"%s\"\n", name );

  // The signature is byte swapped because we're little-endian.
  if ( newSector0.bootsectorSignature != 0xAA55 ) {
    puts( "\n*** Warning, this does not look like an MBR. ***" );
  }
}


void writeToMBR( char *nextTokenPtr ) {

  if ( memcmp(&sector0, &newSector0, 512) == 0 ) {
    puts( "No changes made - not writing" );
  } else {

    printPartitionTable( &sector0.partition[0] );
    printPartitionTable( &newSector0.partition[0] );
    puts("");

    if ( getConfirmation() ) {
      puts("Confirmed - updating MBR");
      int rc = writeSector0(0x80, (uint8_t far *)(&newSector0));
      if ( rc ) {
        printf( "Error writing sector 0: %02x\n", rc );
        exit(1);
      }
      puts( "MBR written to disk." );
      readMBRFromDisk( );

      bool activeFound = false;
      PartitionEntry_t *table = &sector0.partition[0];
      for ( int i=0; i<4; i++ ) {
        if (table[i].attributes & 0x80) {
          activeFound = true;
          break;
        }
      }

      if ( activeFound == false ) {
        puts( "\n*** Warning: No partition is marked active." );
      }

    }

  }
}


bool expertModeFail( void ) {
  if ( ExpertMode == false ) {
    puts( "\"Expert mode\" is required to do this; restart with the -expert option." );
    return true;
  }
  return false;
}


void deletePartition( char *nextTokenPtr ) {
  if ( expertModeFail() ) { return; }
  int target = getSlotParameter( nextTokenPtr );
  if ( target == -1 ) {
    return;
  }
  PartitionEntry_t *table = &newSector0.partition[0];
  if ( getConfirmation() ) {
    void *tmp = &table[target];
    memset( tmp, 0, sizeof(PartitionEntry_t) );
    printf( "Partition %d has been deleted.\n", target+1 );
  }
}


void wipeMBR( char *nextTokenPtr ) {
  if ( expertModeFail() ) { return; }
  if ( getConfirmation() ) {
    memset( &newSector0, 0, sizeof(newSector0) );
    puts( "Pending MBR has been zeroed." );
  }
}


int hexCharToInt( char h ) {
  if ( (h >= '0') && (h <= '9') ) {
    return h - '0';
  }
  if ( (h >= 'A') && (h <= 'F') ) {
    return h - 'A' + 10;
  }
  if ( (h >= 'a') && (h <= 'f') ) {
    return h - 'a' + 10;
  }
  return -1;
}


void setType( char *nextTokenPtr ) {

  if ( expertModeFail() ) { return; }

  int target = getSlotParameter( nextTokenPtr );
  if ( target == -1 ) {
    return;
  }

  char newTypeStr[6];
  // Skip ahead of the slot parameter, then read the new type.
  nextTokenPtr = getNextToken( nextTokenPtr, newTypeStr, 6 );
  getNextToken( nextTokenPtr, newTypeStr, 6 );

  bool goodFormat = false;
  uint8_t newType = 0;

  if ( (newTypeStr[0] == '0') &&
       ((newTypeStr[1] == 'x') || (newTypeStr[1] == 'X')) &&
       (newTypeStr[4] == 0) ) {

    int rc = hexCharToInt(newTypeStr[2]);
    if ( rc != -1 ) {
      newType = rc * 16;
      rc = hexCharToInt(newTypeStr[3]);
      if ( rc != -1 ) {
        newType = newType + rc;
        goodFormat = true;
      }
    }

  }

  if ( goodFormat == true ) {
    PartitionEntry_t *table = &newSector0.partition[0];
    table[target].type = newType;
    printf( "Partition %d set to type 0x%0X\n", target+1, newType );
  } else {
    puts( "Bad type specified: use hexadecimal format. (Example: 0x06 or 0x1F)" );
  }
}


void revert( char *nextTokenPtr ) {
  newSector0 = sector0;
  puts("Changes reverted.");
}

void show( char *nextTokenPtr ) {
  // Do nothing ...  we show the current and pending MBRs after each command
  // runs except when running "help."
}

void reboot( char *nextTokenPtr ) {
  // No return from here.
  rebootAsm( );
}

void help( char *nextTokenPtr ) {
  puts( HelpStr );
  ShowTablesBeforePrompt = false;
}

void quit( char *nextTokenPtr ) {
  exit(0);
}




bool getConfirmation( void ) {
  char response[10];
  printf( "Type YES to confirm: " );
  readline( response );
  if ( stricmp(response, "yes") == 0 ) {
    return true;
  }
  puts( "No confirmation received, nothing done." );
  return false;
}


// readline
// 
// Use the DOS mechanism for getting input from the keyboards.  DOS
// provides the echoing and editing, limits the size of the input,
// and handles Ctrl-C for us.  Using this instead of reading via
// scanf saves a lot of space and is more robust, but we have to
// mind the overall line limit set here.
//
// This works with redirection too ...

extern void dosReadLine( uint8_t far *buffer );
#pragma aux dosReadLine = \
  "push ds"           \
  "push es"           \
  "pop  ds"           \
  "mov  ah, 0xa"      \
  "int  21h"          \
  "pop  ds"           \
  parm [es dx];


void readline( char *buffer ) {

  fflush( NULL );

  // [0] is the max we want to read,
  // [1] is what was read
  // [2-51] is the actual data, including the carriage return.  So
  //        with this we get up to 49 chars + the trailing null.

  uint8_t lineStruct[MAX_LINE_LEN+2];
  lineStruct[0] = MAX_LINE_LEN;
  dosReadLine( lineStruct );

  int i=0;
  while ( i<lineStruct[1] ) {
    buffer[i] = lineStruct[i+2];
    i++;
  }
  buffer[i]=0;

  puts("\n");
}


char *getNextToken( char *input, char *target, uint16_t bufLen ) {

  if ( input == NULL ) {
    *target = 0;
    return NULL;
  }

  // Skip leading whitespace
  int l = strlen(input);
  int i=0;
  while ( (i<l) && (input[i]==' ') ) {
    i++;
  }

  if ( i == l ) {
    *target=0;
    return NULL;
  }

  // We are at the first non-space character

  int j=0;
  for ( ; i<l; i++ ) {
    if ( input[i] == ' ' ) { break; }
    if ( j < bufLen) target[j++] = input[i];
  }

  if ( j < bufLen ) {
    target[j] = 0;
  }
  else {
    target[bufLen-1] = 0;
  }

  if ( i == l ) {
    return NULL;
  }

  return &input[i];
}
