/*
** $Id: dual.c,v 0.1 06/Nov/2018 $
** Dual Library
** See Copyright Notice in agena.h
*/

/* If not indicated otherwise, the formulas below have been taken from:
   https://blog.demofox.org/2014/12/30/dual-numbers-automatic-differentiation/ */

#define dual_c
#define LUA_LIB

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "agnhlps.h"
#include "lstate.h"

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_DUALLIBNAME "dual"
LUALIB_API int (luaopen_dual) (lua_State *L);
#endif


#define isdual(L, idx) (lua_isreg(L, idx) && agn_isutype(L, idx, "dual") && agn_regsize(L, idx) == 2)

#define checkdual(L, idx, procname) { \
  if (!(isdual(L, idx))) \
    luaL_error(L, "Error in " LUA_QS ": expected a dual number.", procname); \
}

#define createdual(L, x, y) { \
  if (isnan((x)) || isnan((y))) { \
    lua_pushundefined(L); \
  } else if (isinf((x))) { \
    lua_pushnumber(L, (x)); \
  } else if (isinf((y))) { \
    lua_pushnumber(L, (y)); \
  } else { \
    agn_createreg(L, 2); \
    agn_regsetinumber(L, -1, 1, (x)); \
    agn_regsetinumber(L, -1, 2, (y)); \
    lua_setmetatabletoobject(L, -1, "dual", 1); \
  } \
}

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_DUALLIBNAME "dual"
LUALIB_API int (luaopen_dual) (lua_State *L);
#endif


static int mt_dual2string (lua_State *L) {  /* at the console, the array is formatted as follows: */
  lua_Number x, y;
  checkdual(L, 1, "dual.__tostring");
  x = agn_reggetinumber(L, 1, 1);
  y = agn_reggetinumber(L, 1, 2);
  luaL_checkstack(L, 3 + (y >= 0), "not enough stack space");  /* 2.31.7 */
  lua_pushfstring(L, "%f", x);
  if (y == -0) y = 0;  /* 2.14.12 fix */
  if (y >= 0) lua_pushstring(L, "+");
  lua_pushfstring(L, "%f", y);
  lua_pushstring(L, "e");
  lua_concat(L, 3 + (y >= 0));
  return 1;
}


static void arrayeq (lua_State *L, int (*fn)(lua_Number, lua_Number, lua_Number), const char *procname) {  /* based on the linalg package (function meq) */
  int flag;
  lua_Number eps;
  checkdual(L, 1, procname);
  checkdual(L, 2, procname);
  eps = agn_getepsilon(L);
  flag = fn(agn_reggetinumber(L, 1, 1), agn_reggetinumber(L, 2, 1), eps) && fn(agn_reggetinumber(L, 1, 2), agn_reggetinumber(L, 2, 2), eps);
  lua_pushboolean(L, flag);
}

static int equal (lua_Number x, lua_Number y, lua_Number eps) {  /* strict equality, eps does not matter */
  return x == y;
}

static int mt_aeq (lua_State *L) {
  arrayeq(L, tools_approx, "dual.__aeq");
  return 1;
}


static int mt_eeq (lua_State *L) {
  arrayeq(L, equal, "dual.__(e)eq");
  return 1;
}


/* Class: dualnumber, source file: dualnumber.h,
   adl.stanford.edu/hyperdual/dualnumber.h
   by: Jeffrey A. Fike, Stanford University, Department of Aeronautics and Astronautics */
static int mt_le (lua_State *L) {
  lua_Number a1, b1;
  checkdual(L, 1, "dual.__le");
  checkdual(L, 2, "dual.__le");
  a1 = agn_reggetinumber(L, 1, 1);
  b1 = agn_reggetinumber(L, 2, 1);
  lua_pushboolean(L, a1 <= b1);
  return 1;
}


/* Class: dualnumber, source file: dualnumber.h,
   adl.stanford.edu/hyperdual/dualnumber.h
   by: Jeffrey A. Fike, Stanford University, Department of Aeronautics and Astronautics */
static int mt_lt (lua_State *L) {
  lua_Number a1, b1;
  checkdual(L, 1, "dual.__lt");
  checkdual(L, 2, "dual.__lt");
  a1 = agn_reggetinumber(L, 1, 1);
  b1 = agn_reggetinumber(L, 2, 1);
  lua_pushboolean(L, a1 < b1);
  return 1;
}


static int mt_unm (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__unm");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, -a1, -a2);
  return 1;
}


/* https://ncatlab.org/nlab/show/dual+number */
static int mt_abs (lua_State *L) {
  checkdual(L, 1, "dual.__abs");
  lua_pushnumber(L, fabs(agn_reggetinumber(L, 1, 1)));  /* 2.14.12 change */
  /* https://gist.github.com/chris-taylor/2005955 states:
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, fabs(a1), a2*luai_numsign(a1)); */
  return 1;
}


/* use definition of Agena's sign operator for complex values */
static int mt_sign (lua_State *L) {
  lua_Number re, du;
  checkdual(L, 1, "dual.__sign");
  re = agn_reggetinumber(L, 1, 1);
  du = agn_reggetinumber(L, 1, 2);
  if (re > 0 || (re == 0 && du > 0))
    lua_pushnumber(L, 1);
  else if (re < 0 || (re == 0 && du < 0))
    lua_pushnumber(L, -1);
  else
    lua_pushnumber(L, 0);
  return 1;
}


static int mt_add (lua_State *L) {  /* extended 2.14.12 */
  lua_Number a1, a2, b1, b2;
  if (agn_isnumber(L, 1)) {
    checkdual(L, 2, "dual.__add");
    a1 = agn_tonumber(L, 1);
    a2 = 0;
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
  } else if (agn_isnumber(L, 2)) {
    checkdual(L, 1, "dual.__add");
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b1 = agn_tonumber(L, 2);
    b2 = 0;
  } else {
    checkdual(L, 1, "dual.__add");
    checkdual(L, 2, "dual.__add");
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
  }
  createdual(L, a1 + b1, a2 + b2);
  return 1;
}


static int mt_sub (lua_State *L) {  /* extended 2.14.12 */
  lua_Number a1, a2, b1, b2;
  if (agn_isnumber(L, 1)) {
    checkdual(L, 2, "dual.__sub");
    a1 = agn_tonumber(L, 1);
    a2 = 0;
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
  } else if (agn_isnumber(L, 2)) {
    checkdual(L, 1, "dual.__sub");
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b1 = agn_tonumber(L, 2);
    b2 = 0;
  } else {
    checkdual(L, 1, "dual.__sub");
    checkdual(L, 2, "dual.__sub");
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
  }
  createdual(L, a1 - b1, a2 - b2);
  return 1;
}


static int mt_mul (lua_State *L) {
  if (agn_isnumber(L, 1)) {
    lua_Number a, b1, b2;
    checkdual(L, 2, "dual.__mul");
    a = agn_tonumber(L, 1);
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
    createdual(L, a*b1, a*b2);
  } else if (isdual(L, 1) && isdual(L, 2)) {
    lua_Number a1, a2, b1, b2;
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
    createdual(L, a1*b1, a1*b2 + a2*b1);
  } else if (isdual(L, 1) && agn_isnumber(L, 2)) {  /* 2.14.12 extension */
    lua_Number a1, a2, b;
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b = agn_tonumber(L, 2);
    createdual(L, a1*b, a2*b);
  } else
    luaL_error(L, "Error in " LUA_QS ": expected number or dual number.", "dual.__mul");
  return 1;
}


static int mt_div (lua_State *L) {
  if (agn_isnumber(L, 1)) {  /* 2.14.12 extension */
    lua_Number a, b1, b2;
    checkdual(L, 2, "dual.__div");
    a = agn_tonumber(L, 1);
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
    if (b1 == 0)  /* 2.14.12 fix */
      lua_pushundefined(L);
    else {
      createdual(L, a/b1, (-a*b2)/(b1*b1));
    }
  } else if (agn_isnumber(L, 2)) {
    lua_Number b, a1, a2;
    checkdual(L, 1, "dual.__div");
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b = agn_tonumber(L, 2);
    if (b == 0)  /* 2.14.12 fix */
      lua_pushundefined(L);
    else {
      createdual(L, a1/b, a2/b);
    }
  } else if (isdual(L, 2)) {
    lua_Number a1, a2, b1, b2;
    checkdual(L, 1, "dual.__div");
    a1 = agn_reggetinumber(L, 1, 1);
    a2 = agn_reggetinumber(L, 1, 2);
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
    if (b1 == 0)  /* 2.14.12 fix */
      lua_pushundefined(L);
    else {
      createdual(L, a1/b1, (a2*b1 - a1*b2) / (b1*b1));
    }
  } else
    luaL_error(L, "Error in " LUA_QS ": expected number or dual number.", "dual.__div");
  return 1;
}


static int mt_pow (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__pow");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  if (agn_isnumber(L, 2)) {
    lua_Number k;
    k = agn_tonumber(L, 2);
    createdual(L, sun_pow(a1, k, 0), k*a2 * sun_pow(a1, k - 1, 0));  /* 2.14.13 change from pow to sun_pow */
  } else if (isdual(L, 2)) {
    /* http://new.math.uiuc.edu/math198/MA198-2014/rgandre2/seminar.pdf, sheet #51 */
    lua_Number b1, b2;
    b1 = agn_reggetinumber(L, 2, 1);
    b2 = agn_reggetinumber(L, 2, 2);
    if (a1 == 0)
      lua_pushundefined(L);
    else {
      lua_Number f = sun_pow(a1, b1, 0);  /* 2.14.13 change from pow to sun_pow */
      createdual(L, f, f*(b2*luai_numln(a1) + a2*b1/a1) );
    }
  } else
    luaL_error(L, "Error in " LUA_QS ": expected number or dual number.", "dual.__pow");
  return 1;
}


static int mt_ipow (lua_State *L) {
  lua_Number a1, a2, k;
  checkdual(L, 1, "dual.__ipow");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  k = agn_checknumber(L, 2);
  createdual(L, luai_numipow(a1, k), k*a2 * luai_numipow(a1, k - 1));
  return 1;
}


static int mt_square (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__square");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, a1*a1, 2*a1*a2);
  return 1;
}


static int mt_cube (lua_State *L) {
  lua_Number a1, a2, ad;
  checkdual(L, 1, "dual.__cube");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  ad = a1*a1;
  createdual(L, ad*a1, 3*a2*ad);
  return 1;
}


/* See: https://www.osti.gov/servlets/purl/1368722, sheet #23 &
   http://www.euclideanspace.com/maths/algebra/realNormedAlgebra/other/dualNumbers/functions/index.htm */
static int mt_recip (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__recicp");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  if (a1 == 0)  /* 2.14.12 fix */
    lua_pushundefined(L);
  else {
    createdual(L, 1/a1, -a2/(a1*a1));
  }
  return 1;
}


static int mt_sqrt (lua_State *L) {
  lua_Number a1, a2, x;
  checkdual(L, 1, "dual.__sqrt");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  x = luai_numsqrt(a1);
  createdual(L, x, 0.5*a2/x);
  return 1;
}


/* https://arxiv.org/pdf/1411.0583.pdf */
static int mt_exp (lua_State *L) {
  lua_Number a1, a2, x;
  checkdual(L, 1, "dual.__exp");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  x = sun_exp(a1);
  createdual(L, x, a2*x);
  return 1;
}


/* https://gist.github.com/chris-taylor/2005955 &
   https://www.arctbds.com/volume4/arctbds_submission_28.pdf, Chapter 3.4 The dual logarithmic function */
static int mt_ln (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__ln");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, luai_numln(a1), a2/a1);
  return 1;
}


static int mt_sin (lua_State *L) {
  lua_Number a1, a2, si, co;
  checkdual(L, 1, "dual.__sin");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  sun_sincos(a1, &si, &co);  /* 2.24.2 tweak by 33 % */
  createdual(L, si, a2*co);
  return 1;
}


static int mt_cos (lua_State *L) {
  lua_Number a1, a2, si, co;
  checkdual(L, 1, "dual.__cos");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  sun_sincos(a1, &si, &co);  /* 2.24.2 tweak by 33 % */
  createdual(L, co, -a2*si);
  return 1;
}


static int mt_tan (lua_State *L) {
  lua_Number a1, a2, x;
  checkdual(L, 1, "dual.__tan");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  x = sun_cos(a1);
  createdual(L, sun_tan(a1), a2/(x*x));
  return 1;
}


/* arcsin defined in "DNAD a Simple Tool for Automatic Differentiation of Fortran Code.pdf",
   see https://digitalcommons.usu.edu/cgi/viewcontent.cgi?article=1029&context=mae_facpub */
static int mt_arcsin (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__arcsin");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, sun_asin(a1), a2/sun_hypot3(a1));
  return 1;
}


static int mt_arccos (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__arccos");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, sun_acos(a1), -a2/sun_hypot3(a1));
  return 1;
}


static int mt_arctan (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.__arctan");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, sun_atan(a1), a2/(1 + a1*a1));
  return 1;
}


/* https://gist.github.com/chris-taylor/2005955 */
static int mt_sinh (lua_State *L) {
  lua_Number a1, a2, sih, coh;
  checkdual(L, 1, "dual.__sinh");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  sun_sinhcosh(a1, &sih, &coh);  /* 2.24.2 tweak */
  createdual(L, sih, a2*coh);
  return 1;
}


/* https://gist.github.com/chris-taylor/2005955 */
static int mt_cosh (lua_State *L) {
  lua_Number a1, a2, sih, coh;
  checkdual(L, 1, "dual.__cosh");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  sun_sinhcosh(a1, &sih, &coh);  /* 2.24.2 tweak */
  createdual(L, coh, a2*sih);
  return 1;
}


/* https://gist.github.com/chris-taylor/2005955 &
   https://www.arctbds.com/volume4/arctbds_submission_28.pdf, Chapter 3.3 The dual hyperbolic functions (45) */
static int mt_tanh (lua_State *L) {
  lua_Number a1, a2, x;
  checkdual(L, 1, "dual.__tanh");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  x = 1/sun_cosh(a1);  /* 2.14.12 fixed; sech(x) = 1/cosh(x) */
  createdual(L, sun_tanh(a1), a2*x*x);
  return 1;
}


static int dual_hypot (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.hypot");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  lua_pushnumber(L, sun_hypot(a1, a2));
  return 1;
}

/* taken from the dual package for R, https://cran.r-project.org/src/contrib/dual_0.0.3.tar.gz; 2.24.2
   function obj = asinh(a)  # inverse hyperbolic sine
      obj = Dual(asinh(a.x), (1 / sqrt((a.x)^2 + 1))*a.d);
   end */
static int dual_arcsinh (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.arcsinh");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, sun_asinh(a1), a2*(1/sqrt(a1*a1 + 1)));
  return 1;
}


/* taken from the dual package for R, https://cran.r-project.org/src/contrib/dual_0.0.3.tar.gz; 2.24.2
   function obj = acosh(a)  # inverse hyperbolic cosine
      obj = Dual(acosh(a.x), (1 / sqrt((a.x)^2 - 1))*a.d);
   end */
static int dual_arccosh (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.arccosh");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, acosh(a1), a2*(1/sqrt(a1*a1 - 1)));
  return 1;
}


/* taken from the dual package for R, https://cran.r-project.org/src/contrib/dual_0.0.3.tar.gz; 2.24.2
   function obj = atanh(a)  # inverse hyperbolic tangent
      obj = Dual(atanh(a.x), (1 / (1 - (a.x)^2))*a.d);
   end */
static int dual_arctanh (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.arctanh");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, sun_atanh(a1), a2*(1/(1 - a1*a1)));  /* 2.40.0 tweak */
  return 1;
}


/* taken from the dual package for R, https://cran.r-project.org/src/contrib/dual_0.0.3.tar.gz; 2.24.2
   ifu <- 1.0 / x@f / log(2)
   ans <- dual(f = log2(x@f), grad = x@grad * ifu) */
static int dual_log2 (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.log2");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, sun_log2(a1), a2/a1/LN2);
  return 1;
}


/* taken from the dual package for R, https://cran.r-project.org/src/contrib/dual_0.0.3.tar.gz; 2.24.2
   ifu <- 1.0 / x@f / log(10)
   ans <- dual(f = log10(x@f), grad = x@grad * ifu) */
static int dual_log10 (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.log10");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, sun_log10(a1), a2/a1/LN10);
  return 1;
}


/* taken from the dual package for R, https://cran.r-project.org/src/contrib/dual_0.0.3.tar.gz; 2.24.2
   ifu <- 1.0 / (1.0 + x@f)
   ans <- dual(f = log1p(x@f), grad = x@grad * ifu) */
static int dual_lnplusone (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.lnplusone");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, log1p(a1), a2/(1.0 + a1));
  return 1;
}


/* taken from the dual package for R, https://cran.r-project.org/src/contrib/dual_0.0.3.tar.gz; 2.24.2
   tmp <- exp(x@f)
   ans <- dual(f = expm1(x@f), grad = x@grad * tmp) */
static int dual_expminusone (lua_State *L) {
  lua_Number a1, a2;
  checkdual(L, 1, "dual.expminusone");
  a1 = agn_reggetinumber(L, 1, 1);
  a2 = agn_reggetinumber(L, 1, 2);
  createdual(L, expm1(a1), a2*sun_exp(a1));
  return 1;
}


static int dual_dual (lua_State *L) {
  int nargs = lua_gettop(L);
  if (nargs == 1) {  /* get dual part */
    checkdual(L, 1, "dual.dual");
    agn_reggeti(L, 1, 2);
  } else if (nargs == 2) {
    lua_Number x, y;
    x = agn_checknumber(L, 1);
    y = agn_checknumber(L, 2);
    if (isfinite(x) && isfinite(y)) {
      createdual(L, x, y);
    } else
      lua_pushnumber(L, isnan(x) ? x : (isnan(y) ? y : x));
  } else
    luaL_error(L, "Error in " LUA_QS ": expected one or two arguments.", "dual.dual");
  return 1;
}


static const struct luaL_Reg dual_metalib [] = {  /* metamethods for numeric userdata `n` */
  {"__unm", mt_unm},                     /* unary minus */
  {"__aeq", mt_aeq},                     /* approximate equality mt */
  {"__eq", mt_eeq},                      /* equality mt */
  {"__eeq", mt_eeq},                     /* strict equality mt */
  {"__lt", mt_lt},                       /* less-than mt */
  {"__le", mt_le},                       /* less-or-equal mt */
  {"__abs", mt_abs},
  {"__sign", mt_sign},
  {"__add", mt_add},
  {"__sub", mt_sub},
  {"__mul", mt_mul},
  {"__div", mt_div},
  {"__square", mt_square},
  {"__cube", mt_cube},
  {"__recip", mt_recip},
  {"__sqrt", mt_sqrt},
  {"__sin", mt_sin},
  {"__cos", mt_cos},
  {"__tan", mt_tan},
  {"__arcsin", mt_arcsin},
  {"__arccos", mt_arccos},
  {"__arctan", mt_arctan},
  {"__sinh", mt_sinh},
  {"__cosh", mt_cosh},
  {"__tanh", mt_tanh},
  {"__ln", mt_ln},
  {"__exp", mt_exp},
  {"__pow", mt_pow},
  {"__ipow", mt_ipow},
  {"__tostring", mt_dual2string},        /* for output at the console, e.g. print(n) */
  {NULL, NULL}
};


static const luaL_Reg dualnumlib[] = {
  {"dual",    dual_dual},
  {"hypot",   dual_hypot},
  {"arcsinh", dual_arcsinh},
  {"arccosh", dual_arccosh},
  {"arctanh", dual_arctanh},
  {"expminusone", dual_expminusone},
  {"lnplusone",   dual_lnplusone},
  {"log2",    dual_log2},
  {"log10",   dual_log10},
  {"tostring", mt_dual2string},
  {NULL, NULL}
};


/*
** Open dual library
*/
LUALIB_API int luaopen_dual (lua_State *L) {
  luaL_newmetatable(L, "dual");  /* metatable for dual numbers, adds it to the registry with key 'dual' */
  luaL_register(L, NULL, dual_metalib);  /* assign C metamethods to this metatable */
  luaL_register(L, AGENA_DUALLIBNAME, dualnumlib);
  return 1;
}

