/*
** $Id: lookup.c,v 0.1 21/01/2024 $
** Lookup Library
** See Copyright Notice in agena.h
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define lookup_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"
#include "lbaselib.h"

typedef struct {
  /* We use an environment table for the lookup data. Directly operating with a Table* object unfortunately does not work as it
     often gets destroyed in a session. */
  int size;         /* number of all the keys */
  int sizeentries;  /* sum of all the elements */
} Lookup;

#define checklookup(L, n)      (Lookup *)luaL_checkudata(L, n, "lookup")

/* The function creates an empty rbtree and returns it. */
static int lookup_new (lua_State *L) {
  Lookup *a;
  int la, lh;
  la = agnL_optnonnegint(L, 1, 0);
  lh = agnL_optnonnegint(L, 2, 0);
  a = (Lookup *)lua_newuserdata(L, sizeof(Lookup));
  if (!a)
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "lookup.new");
  lua_setmetatabletoobject(L, -1, "lookup", 0);
  agn_setutypestring(L, -1, "lookup");
  lua_createtable(L, la, lh);
  lua_setfenv(L, -2);
  a->size = 0;
  a->sizeentries = 0;
  return 1;
}


/* Inserts values into a table t of tables. If t[key] represents a table, value is added to the end of its array part. If t[key] is
   unassigned, then t[key] := [value]. */
static int aux_gettablebykey (lua_State *L, int idx) {
  lua_getfenv(L, idx);
  lua_pushvalue(L, idx + 1);
  lua_gettable(L, -2);
  return lua_type(L, -1);
}

static int lookup_include (lua_State *L) {
  int i, nargs, type;
  Lookup *a = checklookup(L, 1);
  luaL_checkany(L, 3);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  type = aux_gettablebykey(L, 1);
  switch (type) {
    case LUA_TNIL: {
      agn_poptop(L);
      lua_getfenv(L, 1);
      lua_pushvalue(L, 2);
      lua_createtable(L, nargs - 3 + 1, 0);
      for (i=3; i <= nargs; i++) {
        lua_pushvalue(L, i);  /* push value */
        lua_rawseti(L, -2, i - 2);  /* insert value into new subtable */
        a->sizeentries++;
      }
      /* new subtable is at the stack top, set it to lookup table */
      lua_settable(L, -3);
      a->size++;
      break;
    }
    case LUA_TTABLE: {
      for (i=3; i <= nargs; i++) {
        agn_rawinsertfrom(L, -1, i);
        a->sizeentries++;
      }
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": malformed lookup subtable, fetched %s instead.", "lookup.include", lua_typename(L, type));
  }
  agn_poptop(L);  /* pop subtable or nil */
  return 0;
}


static int lookup_purge (lua_State *L) {
  int type;
  Lookup *a = checklookup(L, 1);
  luaL_checkany(L, 2);
  type = aux_gettablebykey(L, 1);
  switch (type) {
    case LUA_TNIL: {  /* do nothing, return null */
      break;
    }
    case LUA_TTABLE: {
      int size = agn_size(L, -1);
      lua_getfenv(L, 1);
      lua_pushvalue(L, 2);
      lua_pushnil(L);
      lua_settable(L, -3);
      agn_poptop(L);
      if (a->size > 0) a->size--;
      if (a->sizeentries > 0) a->sizeentries -= size;
      /* return purged subtable */
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": malformed lookup subtable, fetched %s instead.", "lookup.purge", lua_typename(L, type));
  }
  return 1;
}


static int lookup_gettable (lua_State *L) {
  (void)checklookup(L, 1);
  if (lua_gettop(L) == 1) {
    lua_getfenv(L, 1);
  } else {
    luaL_checkany(L, 2);
    aux_gettablebykey(L, 1);
  }
  return 1;
}


static int lookup_indices (lua_State *L) {
  Lookup *a = checklookup(L, 1);
  int nargs = lua_gettop(L);
  lua_getfenv(L, 1);
  lua_replace(L, 1);
  if (nargs == 1) {
    int i = 0;
    luaL_checkstack(L, 3, "not enough stack space");
    lua_createtable(L, a->size, 0);
    lua_pushnil(L);
    while (lua_next(L, 1) != 0) {
      lua_rawsetikey(L, -3, ++i);
    }
  } else {
    int flag;
    agn_intindices(L, 1, &flag);
  }
  return 1;
}


static int lookup_getsizes (lua_State *L) {
  Lookup *a = checklookup(L, 1);
  lua_pushinteger(L, a->size);         /* number of keys */
  lua_pushinteger(L, a->sizeentries);  /* number of all the elements in the lookup table */
  return 2;
}


static int lookup_setsizes (lua_State *L) {
  Lookup *a = checklookup(L, 1);
  a->size = agn_checknonnegint(L, 2);
  a->sizeentries = agnL_optnonnegint(L, 3, a->sizeentries);
  return 0;
}


#define testsentinel(L, hassentinel) { \
  if ((hassentinel)) { \
    if (!lua_equal(L, -1, 1)) return 2; \
    else { \
      lua_pushnil(L); \
      return 1; \
    } \
  } \
}

/* based on luaB_nextone */
static int lookup_nextone (lua_State *L) {
  int hassentinel, offset;
  (void)checklookup(L, 1);
  lua_getfenv(L, 1);
  lua_replace(L, 1);
  hassentinel = 0;
  offset = 1;
  if ( ( hassentinel = (lua_gettop(L) == 3) ) ) {  /* save sentinel to position 1, shifting up all other arguments */
    lua_pushvalue(L, 3);
    lua_insert(L, 1);
    offset++;
  }
  switch (lua_type(L, offset)) {
    case LUA_TTABLE: {
      lua_settop(L, offset + 1);  /* create a 2nd/3rd argument if there isn't one */
      if (lua_next(L, offset)) {
        testsentinel(L, hassentinel);
        return 2;
      } else {
        lua_pushnil(L);
        return 1;
      }
    }
    default: {
      luaL_error(L, "Error in " LUA_QS ": lookup table expected, got %s.", "lookup.nextone", luaL_typename(L, offset));
    }
  }
  return 1;
}


/* Returns an iterator function that when called returns one element after another from lookup table a. If there are no more elements left, the iterator function returns `null`. Example usage:

a := lookup.new();
lookup.include(a, 'abc', 1, 2, 3, 4)
lookup.include(a, 'xyz', -1, -2, -3, -4)
f := lookup.iterate(a);
while x := [f()] do
   if empty x then break fi;
   print(x)
od;

> f():  # traversal complete, no more element left */
static int iterate (lua_State *L) {
  lua_pushvalue(L, lua_upvalueindex(2));
  if (lua_next(L, lua_upvalueindex(1))) {
    lua_pushvalue(L, -2);
    lua_replace(L, lua_upvalueindex(2));
    return 2;  /* return key and value */
  }
  lua_pushnil(L);
  return 1;
}

static int lookup_iterate (lua_State *L) {  /* 3.9.6 */
  int nargs = lua_gettop(L);
  (void)checklookup(L, 1);
  luaL_checkstack(L, 2, "not enough stack space");  /* 3.18.4 fix */
  lua_getfenv(L, 1);
  if (nargs == 1)
    lua_pushnil(L);
  else
    lua_pushvalue(L, 2);
  lua_pushcclosure(L, &iterate, 2);
  return 1;
}


/* Traverse result which is at the stack top, change size information if necessary, and replace old entries with the new ones. */
static int aux_traverse (lua_State *L, int idx, Lookup *a, const char *procname) {
  lua_pushnil(L);
  while (lua_next(L, -2)) {
    luaL_checkstack(L, 2, "not enough stack space");
    lua_pushvalue(L, idx);    /* push original lookup table */
    lua_pushvalue(L, -3);     /* push key once again  */
    lua_gettable(L, -2);      /* get entry, replacing the key, lookup table still in the stack at -2 */
    if (!lua_istable(L, -1))
      luaL_error(L, "Error in " LUA_QS ": malformed lookup table.", procname);
    a->sizeentries -= agn_size(L, -1);
    agn_poptop(L);            /* drop original subtable */
    lua_pushvalue(L, -3);     /* push key once again  */
    lua_pushvalue(L, -3);     /* push new subtable */
    a->sizeentries += agn_size(L, -1);  /* reset size to number of values in new subtable */
    /* new subtable is at the stack top, set it to lookup table */
    lua_settable(L, -3);
    agn_poptoptwo(L);
  }
  return 1;
}

static int lookup_map (lua_State *L) {  /* taken from agnB_map, in lbaselib.c */
  int nargs, newstruct, multipass, strict, reshuffle, multret, descend, cacheT, type;
  Lookup *a;
  nargs = lua_gettop(L);
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = checklookup(L, 2);
  agnB_aux_fcheckoptions(L, &nargs, &newstruct, &multipass, &strict, &reshuffle, &multret, &descend, "lookup.map");
  newstruct = 0; descend = 1;
  if (!multipass)
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " option is not supported.", "lookup.map", "multipass");
  if (!strict)
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " option is not supported.", "lookup.map", "strict");
  if (newstruct)
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " option is not supported.", "lookup.map", "inplace");
  if (!descend)
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " option is not supported.", "lookup.map", "descend");
  lua_getfenv(L, 2);
  lua_replace(L, 2);
  type = lua_type(L, 2);
  luaL_typecheck(L, type == LUA_TTABLE, 2, "lookup table expected", type);
  lua_newtable(L);
  cacheT = lua_gettop(L);
  agnB_aux_deepdive(L, 2, cacheT, nargs, 0, 0, "lookup.map");
  lua_remove(L, cacheT);
  aux_traverse(L, 2, a, "lookup.map");
  return 1;
}


static int lookup_subs (lua_State *L) {  /* taken from agnB_subs, in lbaselib.c */
  int i, nargs, newstruct, multipass, strict, reshuffle, multret, descend, cacheT, type;
  Lookup *a;
  nargs = lua_gettop(L);
  a = checklookup(L, nargs);
  agnB_aux_fcheckoptions(L, &nargs, &newstruct, &multipass, &strict, &reshuffle, &multret, &descend, "lookup.subs");
  newstruct = 0; descend = 1;
  if (multret)
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " option is not supported.", "lookup.subs", "multret");
  if (newstruct)
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " option is not supported.", "lookup.subs", "inplace");
  if (!descend)
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " option is not supported.", "lookup.subs", "descend");
  luaL_checkstack(L, nargs, "too many arguments");
  for (i=1; i < nargs; i++)
    luaL_checktype(L, i, LUA_TPAIR);
  lua_getfenv(L, nargs);
  lua_replace(L, nargs);
  type = lua_type(L, nargs);
  luaL_typecheck(L, type == LUA_TTABLE, nargs, "lookup table expected", type);
  lua_newtable(L);
  cacheT = lua_gettop(L);
  agnB_aux_deepdive(L, nargs, cacheT, nargs, 1, strict, "lookup.subs");
  lua_remove(L, cacheT);
  aux_traverse(L, nargs, a, "lookup.subs");
  return 1;
}


/* Metamethods *******************************************************************************/

static int mt_empty (lua_State *L) {
  Lookup *a = checklookup(L, 1);
  lua_pushboolean(L, a->size == 0);
  return 1;
}


static int mt_filled (lua_State *L) {
  Lookup *a = checklookup(L, 1);
  lua_pushboolean(L, a->size != 0);
  return 1;
}


static int mt_tostring (lua_State *L) {
  Lookup *a = checklookup(L, 1);
  if (agn_getutype(L, 1)) {
    lua_pushfstring(L, "(%u, %u, %f)", a->size, a->sizeentries, a->size == 0 ? AGN_NAN : a->sizeentries/a->size);
    lua_concat(L, 2);
  } else
    luaL_error(L, "Error in " LUA_QS ": invalid array.", "lookup.__tostring");
  return 1;
}


static int mt_in (lua_State *L) {  /* 3.9.6 */
  (void)checklookup(L, 2);
  lua_getfenv(L, 2);
  lua_pushnil(L);
  while (lua_next(L, -2)) {
    if (agn_in(L, 1, -1, 0)) {
      lua_pop(L, 3);
      lua_pushtrue(L);
      return 1;
    }
    agn_poptop(L);
  }
  agn_poptop(L);
  lua_pushfalse(L);
  return 1;
}


static int mt_notin (lua_State *L) {  /* 3.9.6 */
  (void)checklookup(L, 2);
  lua_getfenv(L, 2);
  lua_pushnil(L);
  while (lua_next(L, -2)) {
    if (agn_in(L, 1, -1, 0)) {
      lua_pop(L, 3);
      lua_pushfalse(L);
      return 1;
    }
    agn_poptop(L);
  }
  agn_poptop(L);
  lua_pushtrue(L);
  return 1;
}


static int mt_gc (lua_State *L) {  /* garbage collect deleted userdata */
  lua_lock(L);
  (void)checklookup(L, 1);
  lua_setmetatabletoobject(L, 1, NULL, 1);
  lua_unlock(L);
  return 0;
}


static const struct luaL_Reg lookup_lib [] = {  /* metamethods for bags */
  {"__index",  lookup_gettable},  /* read access, 3.9.6 */
  {"__size",   lookup_getsizes},  /* retrieve the number of unique elements */
  {"__in",     mt_in},            /* metamethod for `in` operator */
  {"__notin",  mt_notin},         /* metamethod for `notin` operator */
  {"__empty",  mt_empty},         /* metamethod for `empty` operator */
  {"__filled", mt_filled},        /* metamethod for `filled` operator */
  {"__tostring", mt_tostring},    /* metamethod for `filled` operator */
  {"__gc", mt_gc},                /* metamethod for garbage collection */
  {NULL, NULL}
};


static const luaL_Reg lookuplib[] = {
  {"getsizes", lookup_getsizes},
  {"gettable", lookup_gettable},
  {"include",  lookup_include},
  {"indices",  lookup_indices},
  {"iterate",  lookup_iterate},
  {"map",      lookup_map},
  {"new",      lookup_new},
  {"nextone",  lookup_nextone},
  {"purge",    lookup_purge},
  {"setsizes", lookup_setsizes},
  {"subs",     lookup_subs},
  {NULL, NULL}
};


/*
** Open lookup library
*/
LUALIB_API int luaopen_lookup (lua_State *L) {
  /* metamethods */
  luaL_newmetatable(L, "lookup");
  luaL_register(L, NULL, lookup_lib);
  /* register library */
  luaL_register(L, AGENA_LOOKUPLIBNAME, lookuplib);
  return 1;
}

