/* This library implements a character buffer. Initiated 22/03/2020, 2.17.8

   It provides a memory file userdata that stores a string of almost unlimited length, along with functions to work with it.
   It can be used if you have to iteratively concatenate a lot of strings, being 20 times faster than
   the `&` operator.

   The package also provides the following metamethods:

  '__index',      read operation, e.g. n[p] or n[p to q], with p, q indices, both counting from 1
  '__writeindex', write operation, e.g. n[p] := value, with p the index, counting from 1
  '__size',       number of stored bytes
  '__tostring',   output at the console
  '__in',         `in` operator
  '__gc',         garbage collection
  '__eq',         `=` operator
  '__empty',      `empty` operator
  '__filled',     `filled` operator */

#include <stdio.h>
#include <stdlib.h>

#define field_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agnconf.h"  /* for uchar */
#include "agenalib.h"
#include "ldebug.h"
#include "charbuf.h"
#include "agnhlps.h"

#define checkfield(L, n)      (Charbuf *)luaL_checkudata(L, n, "field")

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_BITFIELDLIBNAME "field"
LUALIB_API int (luaopen_field) (lua_State *L);
#endif


/* Creates a bitfield of at least n bits and optionally sets zeros or ones into this field. The return is a byte buffer with
   initially all positions set to zero. If you pass optional ones or zeros, or the Booleans `true` or `false`, they are set
   from the right end to the left end of the bitfield, e.g. if we have

   > b := memfile.bitfield(4, 1, 1, 0, 0)

   we store 0b0011 = 3 decimal into the field. The number of bits actually allocated is always a multiple of 8. */
static int field_new (lua_State *L) {  /* 2.31.4 */
  size_t nbits;
  Charbuf *buf;
  int value;
  luaL_checkstack(L, lua_gettop(L), "too many arguments");
  nbits = agn_checkposint(L, 1);
  value = agnL_optnonnegint(L, 2, 0);
  buf = (Charbuf *)lua_newuserdata(L, sizeof(Charbuf));
  if (buf)
    bitfield_init(L, buf, nbits, value);
  else
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "bitfield.new");
  lua_setmetatabletoobject(L, -1, "bitfield", 1);
  return 1;
}


static int field_bitget (lua_State *L) {  /* 2.31.4 */
  bitfield_get(L, checkfield(L, 1), agn_checkposint(L, 2));
  return 1;
}


static int field_bitsetto (lua_State *L) {
  bitfield_setbit(L, checkfield(L, 1), agn_checkposint(L, 2), agn_checkinteger(L, 3));
  return 0;
}


static int field_bitset (lua_State *L) {
  bitfield_set(L, checkfield(L, 1), agn_checkposint(L, 2));
  return 0;
}


static int field_bitclear (lua_State *L) {
  bitfield_clear(L, checkfield(L, 1), agn_checkposint(L, 2));
  return 0;
}

static int field_bitflip (lua_State *L) {
  lua_pushinteger(L, bitfield_flip(L, checkfield(L, 1), agn_checkposint(L, 2)));
  return 1;
}


/* --- Metamethods -----------------------------------------------------------------------------------------------*/

static int mt_getsize (lua_State *L) {
  lua_pushnumber(L, bitfield_size(L, checkfield(L, 1)));
  return 1;
}


static int mt_tostring (lua_State *L) {  /* at the console, the array is formatted as follows: */
  if (luaL_isudata(L, 1, "field")) {
    unsigned char byte;
    int c, i, j;
    Charbuf *buf = lua_touserdata(L, 1);
    lua_checkstack(L, CHAR_BIT + 2);
    lua_pushfstring(L, "field(");
    c = 1;
    for (i=0; i < buf->size; i++) {
      byte = uchar(buf->data[i]);
      lua_pushfstring(L, "0b"); c++;
      for (j=CHAR_BIT-1; j >= 0; j--) {
        lua_pushfstring(L, "%d", ((LUAI_UINT32)(byte) & (1 << ((LUAI_UINT32)j))) != 0);
        c++;
      }
      lua_concat(L, c); c = 1;
      if (i < buf->size - 1) {
        lua_pushfstring(L, ", ");
        lua_concat(L, 2);
      }
    }
    lua_pushfstring(L, ")");
    lua_concat(L, 2);
  } else {
    void *p = lua_touserdata(L, 1);
    lua_pushfstring(L, (p != NULL) ? "userdata(%p)" : "unknown(%p)", lua_topointer(L, 1));
  }
  return 1;
}


static int mt_gc (lua_State *L) {
  Charbuf *buf = checkfield(L, 1);
  charbuf_free(buf);
  return 0;
}

/* ---------------------------------------------------------------------------------------------------------------*/

static const struct luaL_Reg field_fieldlib [] = {  /* metamethods for memfile `n' */
  {"__gc",         mt_gc},              /* please do not forget garbage collection */
  {"__tostring",   mt_tostring},        /* for output at the console, e.g. print(n) */
  {"__size",       mt_getsize},
  {NULL, NULL}
};


static const luaL_Reg bitfieldlib[] = {
  {"clear",        field_bitclear},
  {"flip",         field_bitflip},
  {"get",          field_bitget},
  {"new",          field_new},
  {"set",          field_bitset},
  {"setbit",       field_bitsetto},
  {NULL, NULL}
};

/*
** Open bitfield library
*/
LUALIB_API int luaopen_bitfield (lua_State *L) {
  /* metamethods */
  luaL_newmetatable(L, "field");
  luaL_register(L, NULL, field_bitfieldlib);
  /* register library */
  luaL_register(L, AGENA_BITFIELDLIBNAME, bitfieldlib);
  return 1;
}

