/*
** $Id: mp.c, initiated September 22, 2019 $
** GNU Multiple Precision Arithmetic Library (GMP) binding
** See Copyright Notice in agena.h
**
** NOTE: MinGW cannot compile this file if its name would be `gmp.c` as this seems to collide with the gmp.h header file, or whatever.
**
** Supports signed and unsigned integers, only.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>

#include <gmp.h>

#define mplib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"


#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_MPLIBNAME "mp"
LUALIB_API int (luaopen_mp) (lua_State *L);
#endif


#define checkmp(L, n) (Gmp *)luaL_checkudata(L, n, "mp")

typedef unsigned long int ui;

typedef struct Gmp {
  mpz_t ui;
  int base;
} Gmp;

#ifndef mp_bitcnt_t
#define mp_bitcnt_t size_t
#endif

#define creatempui(x) { \
  x = (Gmp *)lua_newuserdata(L, sizeof(Gmp)); \
  lua_setmetatabletoobject(L, -1, "mp", 1); \
  mpz_init(x->ui); \
  x->base = 0; \
}


/* Creates an unsigned integer object (mpz_t GMP userdata object) from an unsigned integer n, or a string str representing an unsigned integer.
   If you pass a string you may indicate whether it is in decimal format by passing the optional second argument 10, and if its is in hexadecimal
   encoding, pass 16, which is the default. See also: `mp.setstring`. */
static int mp_uint (lua_State *L) {
  Gmp *a;
  if (agn_isnumber(L, 1)) {
    creatempui(a);
    mpz_set_ui(a->ui, (ui)agn_checknonnegint(L, 1));
    a->base = 10;
  } else if (agn_isstring(L, 1)) {
    int r, base;
    const char *str = agn_tostring(L, 1);
    base = agnL_optnonnegint(L, 2, 16);
    if (base == 1 || base > 62 || (base & 1) == 1)  /* first bit reserved for mp.setstring flag */
      luaL_error(L, "Error in " LUA_QS ": base is invalid.", "mp.uint");
    creatempui(a);  /* first scan args, then push userdata */
    r = mpz_set_str(a->ui, str, base);
    a->base = base;
    if (r == -1)
      luaL_error(L, "Error in " LUA_QS ": allocation failed.", "mp.uint");
  } else
    luaL_error(L, "Error in " LUA_QS ": expected a number or string, got %s.", "mp.uint", luaL_typename(L, 1));
  return 1;  /* leave userdata on the top of the stack */
}


static int mp_sint (lua_State *L) {  /* Like mp.uint, but creates a signed integer mpz_t GMP userdata object. */
  Gmp *a;
  if (agn_isnumber(L, 1)) {
    creatempui(a);
    mpz_set_si(a->ui, agn_checknumber(L, 1));
    a->base = 10;
  } else if (agn_isstring(L, 1)) {
    int r, base;
    const char *str = agn_tostring(L, 1);
    base = agnL_optnonnegint(L, 2, 16);
    if (base == 1 || base > 62 || (base & 1) == 1)
      luaL_error(L, "Error in " LUA_QS ": base is invalid.", "mp.sint");
    creatempui(a);  /* first scan args, then push userdata */
    r = mpz_set_str(a->ui, str, base);
    a->base = base;
    if (r == -1)
      luaL_error(L, "Error in " LUA_QS ": allocation failed.", "mp.sint");
  } else
    luaL_error(L, "Error in " LUA_QS ": expected a number or string, got %s.", "mp.sint", luaL_typename(L, 1));
  return 1;  /* leave userdata on the top of the stack */
}


/*****************************************************************************************************

	Macro templates

******************************************************************************************************/

/* take two mpint arguments a, b, apply procname to them and push the result onto the stack in new mpint r. */
#define template_2args_1ret(procname) { \
  Gmp *a, *b, *r; \
  a = checkmp(L, 1); \
  b = checkmp(L, 2); \
  creatempui(r); \
  procname(r->ui, a->ui, b->ui); \
  r->base = a->base; \
}

/* take three mpint arguments r, a, b, apply procname to them; the result is implictly stored to mpint r. */
#define template_3args_composite(procname) { \
  Gmp *a, *b, *r; \
  r = checkmp(L, 1); \
  a = checkmp(L, 2); \
  b = checkmp(L, 3); \
  lua_settop(L, 1); \
  procname(r->ui, a->ui, b->ui); \
}

/* take two mpint arguments a, b, apply procname to them and push the result - a Lua/Agena number - onto the stack  */
#define template_2args_1num(procname) { \
  Gmp *a, *b; \
  a = checkmp(L, 1); \
  b = checkmp(L, 2); \
  lua_pushnumber(L, procname(a->ui, b->ui)); \
}

/* take a non-negative Agena integer, apply procname to it and push the result onto the stack in new mpint r. */
#define template_1int_1ret(procname) { \
  ui n; \
  Gmp *r; \
  n = agn_checknonnegint(L, 1); \
  creatempui(r); \
  procname(r->ui, n); \
  r->base = 10; \
}


/*****************************************************************************************************

	Signed and Unsigned Integer Arithmetic

******************************************************************************************************/

/* mp.add(a, b): Adds two mpints a, b, and returns a new mpint. Used by __add metamethod, i.e.: mp.add(a, b) = a + b. */
static int mp_add (lua_State *L) {  /* a + b */
  template_2args_1ret(mpz_add);
  return 1;
}


/* mp.subtract(a, b): Subtracts two mpints a, b and returns a new mpint. Used by __sub metamethod, i.e.: mp.subtract(a, b) = a - b. */
static int mp_subtract (lua_State *L) {  /* a - b  */
  template_2args_1ret(mpz_sub);
  return 1;
}


/* mp.multiply(a, b): Multiplies two mpints a, b and returns a new mpint. Used by __mul metamethod, i.e.: mp.multiply(a, b) = a * b. */
static int mp_multiply (lua_State *L) {  /* a * b */
  template_2args_1ret(mpz_mul);
  return 1;
}


/* mp.divide(a, b): Divides two mpints a, b and returns a new mpint. Used by __div metamethod, i.e.: mp.divide(a, b) = a / b. */
static int mp_divide (lua_State *L) {
  Gmp *a, *b, *r;
  a = checkmp(L, 1);
  b = checkmp(L, 2);
  creatempui(r);
  mpz_div(r->ui, a->ui, b->ui);
  r->base = a->base;
  return 1;
}


/* mp.addmul(r, a, b): Multiplies two mpints a, b, adds the result to r and returns the updated value of r: i.e.: mp.addmul(r, a, b) <=> r +:= a * b. */
static int mp_addmul (lua_State *L) {  /* r +:= a*b */
  template_3args_composite(mpz_addmul);
  return 1;
}


/* mp.submul(r, a, b): Multiplies two mpints a, b, subtracts the result from r and returns the updated value of r: i.e.: mp.submul(r, a, b) <=> r -:= a * b. */
static int mp_submul (lua_State *L) {  /* r -:= a*b */
  template_3args_composite(mpz_submul);
  return 1;
}


/* mp.modulus(a, b): Computes the modulus of two mpints a, b and returns a new mpint. Used by __mod metamethod, i.e.: mp.subtract(a, b) = a % b. */
static int mp_modulus (lua_State *L) {  /* r := a % b */
  template_2args_1ret(mpz_mod);
  return 1;
}


/* mp.neg(a): Returns -a, with a a mpint, as a new mpint. Used by __unm metamethod, i.e. mp.neg(a) <=> -a. */
static int mp_neg (lua_State *L) {
  Gmp *a, *r;
  a = checkmp(L, 1);
  creatempui(r);
  mpz_neg(r->ui, a->ui);
  return 1;
}


/* mp.mul2exp(a, b): Computes a * 2^b, with a, b mpints, and returns the result in a new mpint. The operation is equivalent to a left shift by b bits. */
static int mp_mul2exp (lua_State *L) {
  Gmp *a, *r;
  mp_bitcnt_t b;
  a = checkmp(L, 1);
  b = luaL_checkuint32_t(L, 2);
  creatempui(r);
  mpz_mul_2exp(r->ui, a->ui, b);
  r->base = a->base;
  return 1;
}


/* mp.tdiv(a, b): Returns both quotient and remainder of a / b, with a, b units, both rounded towards zero. */
static int mp_tdiv (lua_State *L) {
  Gmp *a, *b, *q, *r;
  a = checkmp(L, 1);
  b = checkmp(L, 2);
  creatempui(q);
  creatempui(r);
  mpz_tdiv_qr(q->ui, r->ui, a->ui, b->ui);
  r->base = a->base;
  q->base = a->base;
  return 2;
}


/* mp.tdivq(a, b): Divides two mpints a, b and returns the resulting quotient as a new mpint, rounded towards zero. */
static int mp_tdivq (lua_State *L) {
  template_2args_1ret(mpz_tdiv_q);
  return 1;
}


/* mp.tdivr(a, b): Divides two mpints a, b and returns the resulting remainder as a new mpint, rounded towards zero. */
static int mp_tdivr (lua_State *L) {
  template_2args_1ret(mpz_tdiv_r);
  return 1;
}


/* mp.powm(a, b, c): With three units, computes (a ^ b) % c and returns the result as a new mpint. */
static int mp_powm (lua_State *L) {
  Gmp *a, *b, *c, *r;
  a = checkmp(L, 1);
  b = checkmp(L, 2);
  c = checkmp(L, 3);
  creatempui(r);
  mpz_powm(r->ui, a->ui, b->ui, c->ui);
  r->base = a->base;
  return 1;
}


/* mp.root(a, n): Computes the truncated integer part of the n-th root of a, with a an mpint, n a number, and returns the result as a new mpint. */
static int mp_root (lua_State *L) {
  Gmp *a, *r;
  ui n;
  a = checkmp(L, 1);
  n = agn_checkposint(L, 2);
  creatempui(r);
  mpz_root(r->ui, a->ui, n);
  r->base = a->base;
  return 1;
}


/* mp.log2(a): Like `ilog2`, but for mpint a. */
static int mp_log2 (lua_State *L) {
  Gmp *a;
  ui i = 0;
  a = checkmp(L, 1);
  if (mpz_get_d(a->ui) == 0)
    lua_pushundefined(L);
  else {  /* search for the most significant bit */
    while (mpz_scan1(a->ui, i) != ULONG_MAX) i++;
    lua_pushnumber(L, (mpz_tstbit(a->ui, --i) == 1) ? i : AGN_NAN);  /* better sure than sorry */
  }
  return 1;
}


/*****************************************************************************************************

	Number Theoretic Functions

******************************************************************************************************/


/* mp.testprime: Checks whether mpint a is a prime and returns:
   - 0 if a is no prime,
   - 1 if a is probably prime,
   - 2 if a is definitely prime.
   The accuracy of the result can be controlled by the optional second parameter `reps` which by default is 15,
   with reasonable values between 15 and 50. */
static int mp_testprime (lua_State *L) {
  int reps;
  Gmp *a = checkmp(L, 1);
  reps = agnL_optposint(L, 2, 15);
  lua_pushnumber(L, mpz_probab_prime_p(a->ui, reps));
  /* 0 = no prime, 1 = probably prime, 2 = definitely prime */
  return 1;
}


/* mp.nextprime(a): Returns the next prime to mpint a, as a new mpint. */
static int mp_nextprime (lua_State *L) {
  Gmp *a, *r;
  a = checkmp(L, 1);
  creatempui(r);
  mpz_nextprime(r->ui, a->ui);
  r->base = a->base;
  return 1;
}


/* mp.gcd(a, b): Returns the greatest common divisor of mpints a and b, as a new mpint. */
static int mp_gcd (lua_State *L) {
  template_2args_1ret(mpz_gcd);
  return 1;
}


/* mp.gcdext(s, t, a, b):
   Returns the greatest common divisor of mpint a and mpint b and returns the result. In addition the function sets mpints s and t to coefficients
   satisfying a*s + b*t = (the return). */
static int mp_gcdext (lua_State *L) {
  Gmp *r, *a, *b, *c, *d;
  a = checkmp(L, 1);
  b = checkmp(L, 2);
  c = checkmp(L, 3);
  d = checkmp(L, 4);
  creatempui(r);
  mpz_gcdext(r->ui, a->ui, b->ui, c->ui, d->ui);
  r->base = a->base;
  return 1;
}


/* mp.lcm(a, b): Returns the least common multiple of mpints a and b, as a new mpint. */
static int mp_lcm (lua_State *L) {
  template_2args_1ret(mpz_lcm);
  return 1;
}


/* mp.invert(a, b): Computes the inverse of mpint a modulo mpint b and returns a mpint. */
static int mp_invert (lua_State *L) {
  template_2args_1ret(mpz_invert);
  return 1;
}


/* mp.jacobi(a, b): Computes the Jacobi symbol (a/b) of the mpints a, b, which is defined only if b is odd. The return is a new mpint. */
static int mp_jacobi (lua_State *L) {
  template_2args_1num(mpz_jacobi);
  return 1;
}


/* mp.legendre(a, b): Computes the Legendre symbol (a/p) of the mpints a, b, which is defined only if p is an odd positive prime. The return is a new mpint. */
static int mp_legendre (lua_State *L) {
  template_2args_1num(mpz_legendre);
  return 1;
}


/* mp.kronecker(a, b): Computes the Jacobi symbol (a/b) for mpints a, b, with the Kronecker extension (a/2)=(2/a) with odd a, and (a/2)=0 with even a.
   The return is a new mpint. */
static int mp_kronecker (lua_State *L) {
  template_2args_1num(mpz_kronecker);
  return 1;
}


/* mp.remove(a, b): Removes all occurrences of the factor a from b returns the result as a new mpint. */
static int mp_remove (lua_State *L) {
  template_2args_1ret(mpz_remove);
  return 1;
}


/* mp.factorial(n): Returns the factorial of number n (_not_ mpint) as a new mpint. */
static int mp_factorial (lua_State *L) {
  template_1int_1ret(mpz_fac_ui);
  return 1;
}


/* mp.fib(n): Returns the n-th Fibonacci number, with n a number (_not_ mpint), and returns a new mpint. */
static int mp_fib (lua_State *L) {
  template_1int_1ret(mpz_fib_ui);
  return 1;
}


/* mp.lucas(n): Sets the n-th Lucas number, with n a number (_not_ mpint), and returns a new mpint. */
static int mp_lucas (lua_State *L) {
  template_1int_1ret(mpz_lucnum_ui);
  return 1;
}


/* mp.primorial(n): Returns the primorial of number n (_not_ mpint), i.e. the product of all positive prime numbers <= n. The return is a new mpint. */
#if (!(defined(__POWERPC__) || defined(__ppc__) || defined(__powerpc) || defined(__APPLE__) ))
static int mp_primorial (lua_State *L) {
  template_1int_1ret(mpz_primorial_ui);
  return 1;
}
#endif

/* mp.binomial(n, k): Computes the binomial coefficient n (a mpint) over k (a number) and returns the result as a new mpint. */
static int mp_binomial (lua_State *L) {
  ui n;
  Gmp *a, *r;
  a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  creatempui(r);
  mpz_bin_ui(r->ui, a->ui, n);
  r->base = a->base;
  return 1;
}


/*****************************************************************************************************

	Bitwise operations

******************************************************************************************************/


/* mp.andint: Conducts a bitwise-and, `a and b`, with a, b mpints, and returns the result as a new mpint. */
static int mp_andint (lua_State *L) {
  template_2args_1ret(mpz_and);
  return 1;
}


/* mp.orint: Conducts a bitwise-or, `a or b`, with a, b mpints, and returns the result as a new mpint. */
static int mp_orint (lua_State *L) {
  template_2args_1ret(mpz_ior);
  return 1;
}


/* mp.xorint: Conducts a bitwise-xor, `a xor b`, with a, b mpints, and returns the result as a new mpint. */
static int mp_xorint (lua_State *L) {
  template_2args_1ret(mpz_xor);
  return 1;
}


/* mp.com(a): Computes the ones complement of mpint a and returns the result as a new mpint. */
static int mp_com (lua_State *L) {
  Gmp *a, *r;
  a = checkmp(L, 1);
  creatempui(r);
  mpz_com(r->ui, a->ui);
  r->base = a->base;
  return 1;
}


/* Scan mpint a, starting from bit n (a number, not an mpint), towards more significant bits, until the first 0 or 1 bit
   (respectively) is found. The functions return the index of the found bit. Bit positions start from 0. */
static int mp_scan0 (lua_State *L) {
  ui n;
  Gmp *a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  lua_pushnumber(L, mpz_scan0(a->ui, n));
  return 1;
}


static int mp_scan1 (lua_State *L) {
  ui n;
  Gmp *a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  lua_pushnumber(L, mpz_scan1(a->ui, n));
  return 1;
}


/* mp.popcount: Computes the population count of a (number of 1 bits in the binary representation) and returns the result a number. */
static int mp_popcount (lua_State *L) {
  Gmp *a = checkmp(L, 1);
  lua_pushnumber(L, mpz_popcount(a->ui));
  return 1;
}


/* mp.mostsigbit(a): Returns the position of the most significant bit (msb) as a number, counting from bit position number 0. If all bits are cleared,
   i.e. zero, returns -1. */
static int mp_mostsigbit (lua_State *L) {
  Gmp *a;
  ui i = 0;
  a = checkmp(L, 1);
  if (mpz_get_d(a->ui) == 0)
    lua_pushnumber(L, -1);
  else {  /* search for the most significant bit */
    while (mpz_scan1(a->ui, i) != ULONG_MAX) i++;
    lua_pushnumber(L, (mpz_tstbit(a->ui, --i) == 1) ? i : AGN_NAN);  /* better sure than sorry */
  }
  return 1;
}


/* mp.leastsigbit(a): Returns the position of the least significant bit (lsb) as a number, counting from bit position number 0. If all bits are cleared,
   i.e. zero, returns -1. */
static int mp_leastsigbit (lua_State *L) {
  Gmp *a;
  ui i = 0;
  a = checkmp(L, 1);
  if (mpz_get_d(a->ui) == 0)
    lua_pushnumber(L, -1);
  else {  /* search for the most significant bit */
    while (mpz_scan1(a->ui, i) != ULONG_MAX && mpz_tstbit(a->ui, i) == 0) i++;
    lua_pushnumber(L, (mpz_tstbit(a->ui, i) == 1) ? i : AGN_NAN);  /* better sure than sorry */
  }
  return 1;
}


/* mp.hamdist(a, b): Computes the hamming distance between mpint a and mpint b, i.e. the number of positions a and b have different bit values.
   The count is returned as a number. */
static int mp_hamdist (lua_State *L) {
  template_2args_1num(mpz_hamdist);
  return 1;
}


/* Sets bit n in mpint r. The function returns nothing. n is an integer position counting from 0. */
static int mp_setbit (lua_State *L) {
  ui n;
  Gmp *a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  mpz_setbit(a->ui, n);
  return 0;
}


/* Returns bit n in mpint r. The function returns a number. n is an integer position counting from 0. */
static int mp_getbit (lua_State *L) {
  ui n;
  Gmp *a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  lua_pushnumber(L, mpz_tstbit(a->ui, n));
  return 1;
}


/* Clears bit n in mpint r. The function returns nothing. n is an integer position counting from 0. */
static int mp_clrbit (lua_State *L) {  /* clear bit */
  ui n;
  Gmp *a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  mpz_clrbit(a->ui, n);
  return 0;
}


/* Conducts a bitwise complement (`not` operation) on bit number n in uint r. n is an integer position counting from 0. The function returns nothing. */
static int mp_combit (lua_State *L) {
  ui n;
  Gmp *a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  mpz_combit(a->ui, n);
  return 0;
}


/*****************************************************************************************************

	Miscellaneous

******************************************************************************************************/

/* mp.tonumber(a): Returns the numeric value in mpint a as a number. */
static int mp_tonumber (lua_State *L) {
  Gmp *a = checkmp(L, 1);
  lua_pushnumber(L, mpz_get_d(a->ui));
  return 1;
}


/* mp.swap(a, b): Swaps the values in mpints a and b. The function returns nothing. */
static int mp_swap (lua_State *L) {
  Gmp *a, *b;
  a = checkmp(L, 1);
  b = checkmp(L, 2);
  mpz_swap(a->ui, b->ui);
  return 0;
}


/* mp.cmp(a, b): Compares a and b and returns a positive number if a > b, 0 if a = b, and a negative number if a < b. */
static int mp_cmp (lua_State *L) {
  template_2args_1num(mpz_cmp);
  return 1;
}


/* mp.cmpabs(a, b): Compares the absolute values of a and b and returns a positive number if a > b, 0 if a = b, and a negative number if a < b. */
static int mp_cmpabs (lua_State *L) {
  template_2args_1num(mpz_cmpabs);
  return 1;
}


/* mp.iseven(a): Checks whether mpint a represents an even integer and returns `true` or `false`. */
static int mp_iseven (lua_State *L) {
  Gmp *a = checkmp(L, 1);
  lua_pushboolean(L, mpz_even_p(a->ui) != 0);
  return 1;
}


/* mp.isodd(a): Checks whether mpint a represents an odd integer and returns `true` or `false`. */
static int mp_isodd (lua_State *L) {
  Gmp *a = checkmp(L, 1);
  lua_pushboolean(L, mpz_odd_p(a->ui) != 0);
  return 1;
}


/* mp.setstring(str): Receives a string and converts it to an mpint. See also: mp.uint/mp.sint. */
static int mp_setstring (lua_State *L) {
  Gmp *r;
  size_t l;
  const char *str = agn_checklstring(L, 1, &l);
  creatempui(r);
  r->base = 16 | 0x01;  /* assuming that the user does not define an mpint to an odd base */
  mpz_import(r->ui, l + 1, 1, 1, 0, 0, str);  /* 2nd arg: add 1 for \0 ! */
  return 1;
}


/* mp.getstring(a): Returns the string in mpint a that was previously stored to it by calling the `mp.setstring` function. See also: mp.tostring. */
static int mp_getstring (lua_State *L) {
  char *buffer = NULL;
  Gmp *a = checkmp(L, 1);
  if ((a->base & 0x01) == 0) {
    lua_pushnil(L);
    return 1;
  }
  buffer = (char *)mpz_export(buffer, NULL, 1, 1, 0, 0, a->ui);
  lua_pushstring(L, buffer);
  xfree(buffer);
  return 1;
}


/* mp.sizeinbase(a, base): Returns the size of `a` measured in number of digits in the given `base`, an integer. `base` can vary from 2 to 62. */
static int mp_sizeinbase (lua_State *L) {
  ui n;
  Gmp *a = checkmp(L, 1);
  n = agn_checknonnegint(L, 2);
  if (n < 2 || n > 62)
    luaL_error(L, "Error in " LUA_QS ": invalid base.", "mp.sizeinbase");
  lua_pushnumber(L, mpz_sizeinbase(a->ui, n));
  return 1;
}


static int mp_attrib (lua_State *L) {
  char *buffer = NULL;
  Gmp *a = checkmp(L, 1);
  lua_createtable(L, 0, 3);
  lua_rawsetstringnumber(L, -1, "sizeinbase10", mpz_sizeinbase(a->ui, 10));
  lua_rawsetstringnumber(L, -1, "limbs", mpz_size(a->ui));
  lua_rawsetstringnumber(L, -1, "base", a->base);
  buffer = mpz_get_str(NULL, (a->base == 0) ? 16 : a->base & 0x1E, a->ui);
  if (!buffer) {
    lua_rawsetstringstring(L, -1, "value", buffer);
  }
  xfree(buffer);
  if ((a->base & 0x01)) {
    char *buffer = NULL;
    buffer = (char *)mpz_export(buffer, NULL, 1, 1, 0, 0, a->ui);
    lua_rawsetstringstring(L, -1, "string", buffer);
    xfree(buffer);
  }
  return 1;
}


/*****************************************************************************************************

	Metamethods

******************************************************************************************************/

/* abs methamethod and mp.argument: determimes the absolute value */
static int mt_abs (lua_State *L) {
  Gmp *a, *r;
  a = checkmp(L, 1);
  creatempui(r);
  mpz_abs(r->ui, a->ui);
  return 1;
}

static int mt_sign (lua_State *L) {
  Gmp *a = checkmp(L, 1);
  lua_pushnumber(L, mpz_sgn(a->ui));
  return 1;
}

static int mt_size (lua_State *L) {
  Gmp *a = checkmp(L, 1);
  lua_pushnumber(L, mpz_sizeinbase(a->ui, 10));
  return 1;
}

#define template_cmp(cond) { \
  Gmp *a, *b; \
  a = checkmp(L, 1); \
  b = checkmp(L, 2); \
  lua_pushboolean(L, (cond)); \
  return 1; \
}

static int mt_isequal (lua_State *L) {
  template_cmp(0 == mpz_cmp(a->ui, b->ui));
}

static int mt_islessthan (lua_State *L) {
  template_cmp(0 > mpz_cmp(a->ui, b->ui));
}

static int mt_islessorequal (lua_State *L) {
  template_cmp(0 >= mpz_cmp(a->ui, b->ui));
}

static int mt_tostring (lua_State *L) {  /*  */
  Gmp *a;
  a = checkmp(L, 1);  /* push userdata on stack */
  lua_settop(L, 1);   /* don't let the stack be polluted by futile additional arguments */
  if (agn_getutype(L, 1)) {
    char *buffer = mpz_get_str(NULL, (a->base == 0) ? 16 : a->base & 0x1E, a->ui);
    if (!buffer)
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "mp.__tostring");
    lua_pushfstring(L, "(%s)", buffer);  /* always return hex */
    lua_concat(L, 2);
    xfree(buffer);
  } else
    luaL_error(L, "Error in " LUA_QS ": invalid mp object.", "mp.__tostring");
  return 1;
}

static int mt_mpgc (lua_State *L) {  /* __gc method */
  Gmp *a;
  (void)L;
  a = checkmp(L, 1);
  mpz_clear(a->ui);
  lua_setmetatabletoobject(L, 1, NULL, 0);  /* 2.21.9 */
  return 0;
}

static const struct luaL_Reg mt_mplib [] = {
  {"__tostring", mt_tostring},  /* for output at the console, e.g. print(n) */
  {"__unm",      mp_neg},
  {"__add",      mp_add},
  {"__sub",      mp_subtract},
  {"__mul",      mp_multiply},
  {"__div",      mp_divide},
  {"__mod",      mp_modulus},
  {"__abs",      mt_abs},
  {"__sign",     mt_sign},
  {"__size",     mt_size},
  {"__eq",       mt_isequal},
  {"__lt",       mt_islessthan},
  {"__le",       mt_islessorequal},
  {"__gc",       mt_mpgc},
  {NULL, NULL}
};


static const luaL_Reg mplib[] = {
  {"add", mp_add},                         /* added on September 22, 2019 */
  {"addmul", mp_addmul},                   /* added on September 22, 2019 */
  {"andint", mp_andint},                   /* added on September 22, 2019 */
  {"argument", mt_abs},                    /* added on September 22, 2019 */
  {"attrib", mp_attrib},                   /* added on September 22, 2019 */
  {"binomial", mp_binomial},               /* added on September 22, 2019 */
  {"clrbit", mp_clrbit},                   /* added on September 22, 2019 */
  {"cmp", mp_cmp},                         /* added on September 22, 2019 */
  {"cmpabs", mp_cmpabs},                   /* added on September 30, 2019 */
  {"combit", mp_combit},                   /* added on September 22, 2019 */
  {"com", mp_com},                         /* added on September 22, 2019 */
  {"divide", mp_divide},                   /* added on September 22, 2019 */
  {"factorial", mp_factorial},             /* added on September 22, 2019 */
  {"fib", mp_fib},                         /* added on September 22, 2019 */
  {"gcd", mp_gcd},                         /* added on September 22, 2019 */
  {"gcdext", mp_gcdext},                   /* added on September 22, 2019 */
  {"getbit", mp_getbit},                   /* added on September 22, 2019 */
  {"getstring", mp_getstring},             /* added on September 22, 2019 */
  {"hamdist", mp_hamdist},                 /* added on September 22, 2019 */
  {"invert", mp_invert},                   /* added on September 22, 2019 */
  {"invert", mp_invert},                   /* added on September 22, 2019 */
  {"leastsigbit", mp_leastsigbit},         /* added on September 26, 2019 */
  {"lucas", mp_lucas},                     /* added on September 30, 2019 */
  {"iseven", mp_iseven},                   /* added on September 22, 2019 */
  {"isodd", mp_isodd},                     /* added on September 22, 2019 */
  {"jacobi", mp_jacobi},                   /* added on September 22, 2019 */
  {"kronecker", mp_kronecker},             /* added on September 22, 2019 */
  {"lcm", mp_lcm},                         /* added on September 22, 2019 */
  {"legendre", mp_legendre},               /* added on September 22, 2019 */
  {"log2", mp_log2},                       /* added on September 25, 2019 */
  {"modulus", mp_modulus},                 /* added on September 22, 2019 */
  {"mostsigbit", mp_mostsigbit},           /* added on September 26, 2019 */
  {"mul2exp", mp_mul2exp},                 /* added on September 22, 2019 */
  {"multiply", mp_multiply},               /* added on September 22, 2019 */
  {"neg", mp_neg},                         /* added on September 25, 2019 */
  {"nextprime", mp_nextprime},             /* added on September 22, 2019 */
  {"orint", mp_orint},                     /* added on September 22, 2019 */
  {"popcount", mp_popcount},               /* added on September 22, 2019 */
  {"powm", mp_powm},                       /* added on September 22, 2019 */
#if (!(defined(__POWERPC__) || defined(__ppc__) || defined(__powerpc) || defined(__APPLE__)))
  {"primorial", mp_primorial},             /* added on September 22, 2019 */
#endif
  {"remove", mp_remove},                   /* added on September 22, 2019 */
  {"root", mp_root},                       /* added on September 25, 2019 */
  {"scan0", mp_scan0},                     /* added on September 25, 2019 */
  {"scan1", mp_scan1},                     /* added on September 25, 2019 */
  {"setbit", mp_setbit},                   /* added on September 22, 2019 */
  {"setstring", mp_setstring},             /* added on September 22, 2019 */
  {"sint", mp_sint},                       /* added on September 27, 2019 */
  {"sizeinbase", mp_sizeinbase},           /* added on September 22, 2019 */
  {"submul", mp_submul},                   /* added on September 22, 2019 */
  {"subtract", mp_subtract},               /* added on September 22, 2019 */
  {"swap", mp_swap},                       /* added on September 22, 2019 */
  {"tdiv", mp_tdiv},                       /* added on September 22, 2019 */
  {"tdivq", mp_tdivq},                     /* added on September 22, 2019 */
  {"tdivr", mp_tdivr},                     /* added on September 22, 2019 */
  {"testprime", mp_testprime},             /* added on September 22, 2019 */
  {"tonumber", mp_tonumber},               /* added on September 22, 2019 */
  {"tostring", mt_tostring},               /* added on September 22, 2019 */
  {"uint", mp_uint},                       /* added on September 22, 2019 */
  {"xorint", mp_xorint},                   /* added on September 22, 2019 */
  {NULL, NULL}
};


/*
** Open mp library
*/

LUALIB_API int luaopen_mp (lua_State *L) {
  luaL_newmetatable(L, "mp");
  luaL_register(L, NULL, mt_mplib);  /* associate __gc method */
  luaL_register(L, AGENA_MPLIBNAME, mplib);
  return 1;
}

