/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FPUTC.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int fputc(int ch, FILE *fp)
	int putc(int ch, FILE *fp)

	Put a character into a stream. 

	fputc() is guaranteed to be a no preprocessor macro.

	ch is internally interpreted as (char)ch, thus, ignoring all
	values exceeding the range SCHAR_MIN <= ch <= UCHAR_MAX.

	Input:
		fp != NULL; a valid FILE* pointer

	Return:
		EOF: write error
		0..UCHAR_MAX: written character

	Note:
		putc() does neither: flush the buffer nor translates characters
		in text. Therefore putc is a synonym of fputc in the current
		implementation.

	Conforms to:
		<none>

	See also:
		fwrite, fputs, puts, fgetc

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FPUTC.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc int
fputc(int ch, FILE *fp)
{
   register int v;
   Inline_call

	assert(fp != NULL);
	
   v = fp->mode;
   /* If last op was a read ... */
   if ((v & __MODE_READING) && fflush(fp))
      return EOF;

   /* Can't write or there's been an EOF or error then return EOF */
   if ((v & (__MODE_WRITE | __MODE_EOF | __MODE_ERR)) != __MODE_WRITE)
      return EOF;

   /* In MSDOS translation mode */
#if __MODE_IOTRAN
   if (ch == '\n' && (v & __MODE_IOTRAN) && fputc('\r', fp) == EOF)
      return EOF;
#endif

   /* Buffer is full */
   if (fp->bufpos >= fp->bufend && fflush(fp))
      return EOF;

   /* Right! Do it! */
   *(fp->bufpos++) = ch;
   fp->mode |= __MODE_WRITING;

   /* Unbuffered or Line buffered and end of line */
   if (((ch == '\n' && (v & _IOLBF)) || (v & _IONBF))
       && fflush(fp))
      return EOF;

   /* Can the macro handle this by itself ? */
   if (v & (__MODE_IOTRAN | _IOLBF | _IONBF))
      fp->bufwrite = fp->bufstart;	/* Nope */
   else
      fp->bufwrite = fp->bufend;	/* Yup */

   /* Correct return val, because a signed character would
   	interfere with EOF! */
   return (uchar_t) ch;
}
