/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: LREALLOC.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	void *realloc(void *poi, size_t size)

	Reallocates an object of size bytes in the local heap.
	If size == 0, the pointer is free()'ed.
	If poi == NULL, the pointer is malloc()'ed.
	If size == 0 && poi == NULL, malloc(0) is performed.

	If DEBUG_HEAP was defined at compiled time, allocated memory
	is filled with dbgHeapAllocCh; freed memory with dbgHeapFreeCh.

	Input:
		<none>

	Return:
		NULL: not enough free memory || size == 0
		else: pointer to the reallocated memory

	Note:
		Local heap only

	Conforms to:
		ANSI

	See also:
		free, malloc, calloc, _fmalloc

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.6  1998/02/11 07:40:34  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <_alloc.h>
#include <_errstr.h>
#include <dos.h>
#include <limits.h>
#include <process.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: LREALLOC.C 1.6 1998/02/11 07:40:34 ska Exp ska $";
#endif

#if _IN_ALLOC != 1			/* far heap uses huge functions */

/* We can savely place the variable here, because realloc() is the
	one and only allocator of the local heap, thus, any "out of memory"
	condition is triggered here.
	By default the handler is initialized to always return NULL.
*/
_CLibVar _nomem_handler_t *_nomem_handler = (_nomem_handler_t*)_retNULL;

#endif /* _IN_ALLOC != 1 */


#if __HEAP_ALIGN		/* _heapsplit() checks alignment condition */
#define _heapsplit _heapalgnsplt
#endif	/* __HEAP_ALIGN */

_CLibFunc void ___PTR *
realloc(void ___PTR *poi
       ,size_t size)
{

#if _IN_ALLOC == 1		/* far interface calls huge interface */

	assert(_heap != NULL);		/* security check if heap is initialized */

	if(size >= SIZE_T_MAX - sizeof(_heapNode)
	 - (sizeof(_heapNode) / _HEAP_PAGE + 1) * _HEAP_PAGE - 2)
		return NULL;		/* sorry! too large */

	return (void ___PTR*) _hrealloc(poi, size);

#else	/* _IN_ALLOC != 1 */


/* local & huge interface */

	/* pointer to this/previous/next node */
	_pHeapNode thisNode, prevNode, nextNode;
	_heapNode tNode, pNode, nNode;		/* copy of these nodes */
	_pHeapNode p, q;		/* temporary node pointers */

	_pHeapData h;
#if __HEAP_ALIGN
	size_t size2;
#endif

	assert(_heap != NULL);		/* security check if heap is initialized */

	/* This condition will also allow a save useage of padNode() */
#if _IN_ALLOC == 0			/* check for local heap */
	if(size >= SIZE_T_MAX - HEAP_MARGIN - 2 * sizeof(_heapNode))
		return NULL;		/* sorry! too large */
#endif	/* _IN_ALLOC == 0 */

	if(poi) {			/* reallocate existing entry */
		if(!size) {		/* free the node */
			free(poi);
			return NULL;
		}

		if((thisNode = _heapfind(poi, &prevNode)) == NULL
		 || thisNode->type < HEAP_USED)		/* no such _allocated_ node */
			return NULL;

		if(nodeLength(thisNode) == size)		/* same length -> no change */
			return poi;

		if(nodeLength(thisNode) > size) {
			/* shrink the block -> split the node */
			_heapsplit(thisNode, size);
			return poi;
		}

		/* block grows. If growing failed, we preserve the values of the
			nodes free() might change. There is always a next node, because
			an used node cannot terminate the heap (a HEAP_END does). */
		nextNode = nxtNode(thisNode);
		if(prevNode)
			memcpy(&pNode, prevNode, sizeof(pNode));
		memcpy(&tNode, thisNode, sizeof(tNode));
		memcpy(&nNode, nextNode, sizeof(nNode));

		/* free the node, temporarily if necessary */
		_heapfree(thisNode, prevNode);
	}

	/* Now try to find a block large enough to hold the new one */
	p = _heapsearch(size);
	switch(p->type) {
	case HEAP_UNUSED:		/* hole large enough to hold the new block */
		/* This node will be split into two nodes, the first will hold
			the allocated memory area, the second the remaining unused
			bytes */

		/* before creating the node & possibly overwriting some 
			bytes in the data portion, we copy the data portion.
			Because the block grows, we copy the current & smaller size */
		if(poi) {
			memmove(nodeData(p), poi, nodeLength(&tNode));
#ifdef DEBUG_HEAP		/* zap the appended allocated space */
			if(dbgHeapAllocCh)
				memset(nodeData(p) + nodeLength(&tNode), dbgHeapAllocCh
				 , size - nodeLength(&tNode));
#endif
		}
#ifdef DEBUG_HEAP		/* zap the newly allocated space */
		else if(dbgHeapAllocCh)
			memset(nodeData(p), dbgHeapAllocCh, size);
#endif
		_heapsplit(p, size);
		break;

	case HEAP_END:			/* try to expand heap */
#if __HEAP_ALIGN		/* we align the end of heap, though, we could do it
			when the next node is allocated behind the node. That would
			require are far more management code than really saves
			memory in the data segment. */
		size2 = size;
		size = padNode(size, nodeData(p));
#else
#define size2 size
#endif

#if _IN_ALLOC == 0					/* local heap */

		q = (void*)_heapMax;		/* the maximum level of the heap */

		/* check for collision of heap & stack */
		if(q > p) {
			if((_pHeapData)q - (_pHeapData)p < size + HEAP_MARGIN + 2 * sizeof(_heapNode))
#else	/* _IN_ALLOC != 0 */		/* huge realloc() */
		if((p = _frgextend(p, size, size2)) == NULL)

#endif /* _IN_ALLOC == 0 */
			{
				/* not enough space to create the new node -> call the no
					memory handler */
				/* we need to make sure that the old allocted block is
					still accessable -> just restore the old three nodes! */
				if(poi) {
					if(prevNode)
						memcpy(prevNode, &pNode, sizeof(pNode));
					memcpy(thisNode, &tNode, sizeof(tNode));
					memcpy(nextNode, &nNode, sizeof(nNode));
				}
				/* the heap does not have enough bytes left -> check the
					nomem handler */
				if((h = _nomem_handler(size2)) != NULL) {
					/* it has got some memory. It does not matter if
						this area is not located in the heap. */
#ifdef DEBUG_HEAP		/* zap the allocated space */
					if(dbgHeapAllocCh)
						memset(h, dbgHeapAllocCh, size2);
#endif
					if(poi) {	/* copy the old contents */
						memcpy(h, poi,
						 nodeLength(&tNode) < size2
						  ? nodeLength(&tNode)
						  : size2);
						free(poi);	/* don't call _heapfree(), because
							possibly _nomem_handler() has called
							some heap functions. The heap might look
							totally different now! */
					}
				}
				return h;
			}

			/* before creating the node & possibly overwriting some 
				bytes in the data portion, we copy the data portion */
			if(poi) {
				memmove(nodeData(p), poi, nodeLength(&tNode));
#ifdef DEBUG_HEAP		/* zap the appended allocated space */
				if(dbgHeapAllocCh)
					memset(nodeData(p) + nodeLength(&tNode), dbgHeapAllocCh
					 , size - nodeLength(&tNode));
#endif
			}
#ifdef DEBUG_HEAP		/* zap the newly allocated space */
			else if(dbgHeapAllocCh)
				memset(nodeData(p), dbgHeapAllocCh, size);
#endif

			/* create a node */
			setNodeLength(p, size);
			nxtNode(p)->type = HEAP_END;
			break;
#if _IN_ALLOC == 0		/* local heap */
		}

		/* fall through */
	default:			/* heap corrupted */
			_abort(ERR_heapstkcoll);
#else	/* _IN_ALLOC != 0 */		/* huge interface */

	default:
			_abort(ERR_heapcorrupt);
#endif	/* _IN_ALLOC == 0 */
	}

#if _IN_ALLOC == 2			/* huge realloc() */
	/* we have called _heapfree(), but this function does not shrink
		the fragment if possible, do it now! */
	if(nNode.type == HEAP_END) {		/* OK, it was the last used node */
		/* Due to the possibility that the newly allocated node has
			overwritten the saved nodes or has created a new fragment,
			we don't rely on the cached information but call the
			fragment garbage collector */
		_frgcompact(nextNode);
	}
#endif	/* _IN_ALLOC == 2 */

	p->type = HEAP_USED;
	return nodeData(p);

#endif	/* _IN_ALLOC == 1 */

}
