/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: OPEN.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int open(char *fnam, int mode [, int attr])

	Open/create file depending on the mode parameter:
	
	The mode flags are declared in <fcntl.h>.

	mode is any combination of:
		O_APPEND	seek to end of file
		O_AUTOCOMMIT	flush system's buffer after each write <DOS/Win95 only>
		O_AUTOFAIL	don't call Critical Error handler, just fail system call <DOS/Win95 only>
		O_BINARY	open in binary mode <currently ignored>
		O_CREAT		create file if not already exists
		O_EXCL		if O_CREAT is set, but file exists, open() fails
		O_NDELAY	don't delay, immediate return failure <ignored>
		O_NOCOMPRESS	don't compress if volume would (Win95 only)
		O_NOINHERIT	don't pass handle to child
		O_TEXT		open in text mode <currently ignored>
		O_TRUNC		overwrite file, truncate to zero length
		O_UNBUFFERED	don't buffer (Win95 only, read/write in multiples of native size of volume)

	mode requires exactly one of the following access flags:
		O_RDONLY	open for read only
		O_WRONLY	open for write only
		O_RDWR		open for read & write
		O_SNOOP		open for read only, but don't update last access time (Win95 only)
	
	mode may have none or one of the following share flags <DOS/Win95 only>:
		O_DENAYALL	deny read & write access
		O_DENYWRITE	deny write access only (default, if write permission)
		O_DENYREAD	deny read access only
		O_DENYNONE	deny neither read nor write access (default, if read-only)
		O_DENYCOMP	compatibly mode, O_DENYALL and cause Critical Error for other process

	the attr parameter is used only when creating a new file and defines
	the file's access flags (ATTRIB.EXE/ CHMOD), any combination of
	(defined in <sys/stat.h>):
		S_IREAD		read access granted <cannot removed in DOS/Win95>
		S_IWRITE	write access granted
		S_IEXEC		execute permission granted <ignored in DOS/Win95>
		S_IHIDDEN	file is hidden <DOS/Win95 only>
		S_ISYSTEM	file is system-related <DOS/Win95 only>

	Input:
		fnam != NULL

	Return:
		<0 if call failed
		>=0: file descriptor; file handle

	Note:
		Currently only DOS 4+ is supported
		Ignored: text/binary; ^Z

	Conforms to:
		Posix

	See also:
		close, remove, unlink, read, write

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.4  1998/01/29 07:09:58  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#define _IN_OPEN			/* using open() prototype rather than open(...) */
#include <io.h>				/* include early to prevent include without define */
#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>

#define OPEN_CREATE (1 << 4)
#define OPEN_TRUNC (1 << 1)
#define OPEN_OPEN (1 << 0)
#ifdef LFN_WIN95
#define OPEN_FLAGS (O_AUTOCOMMIT | O_AUTOFAIL | O_NOCOMPRESS | O_UNBUFFERED \
	| 0xf7)
#else	/* !LFN_WIN95 */
#define OPENFLAGS (O_AUTOCOMMIT | O_AUTOFAIL | O_NOINHERIT | OPEN_SHARE | 3)
#endif	/* LFN_WIN95 */
#define OPEN_SHARE (7<<4)
#define OPEN_PERMISSIONS (7)

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: OPEN.C 1.4 1998/01/29 07:09:58 ska Exp $";
#endif

/****************	API
--------D-21716C-----------------------------                                   
INT 21 - Windows95 - LONG FILENAME - CREATE OR OPEN FILE                        
        AX = 716Ch                                                              
        BX = access mode and sharing flags (see #1122,also AX=6C00h)            
        CX = attributes                                                         
        DX = action (see #1121)                                                 
        DS:SI -> ASCIZ filename                                                 
        DI = alias hint (number to append to short filename for disambiguation) 
Return: CF clear if successful                                                  
            AX = file handle                                                    
            CX = action taken                                                   
                0001h file opened                                               
                0002h file created                                              
                0003h file replaced                                             
        CF set on error                                                         
            AX = error code (see #1020)                                         
                7100h if function not supported                                 
SeeAlso: AX=6C00h,AX=7141h,AX=7156h,AX=71A9h                                    
                                                                                
Bitfields for Windows95 long-name open action:                                  
Bit(s)  Description     (Table 1121)                                            
 0      open file (fail if file does not exist)                                 
 1      truncate file if it already exists (fail if file does not exist)        
 4      create new file if file does not already exist (fail if exists)         
Note:   the only valid combinations of multiple flags are bits 4&0 and 4&1      
                                                                                
Bitfields for Windows95 file access/sharing modes:                              
Bit(s)  Description     (Table 1122)                                            
 2-0    file access mode                                                        
        000 read-only                                                           
        001 write-only                                                          
        010 read-write                                                          
        100 read-only, do not modify file's last-access time                    
 6-4    file sharing modes                                                      
 7      no-inherit flag                                                         
 8      do not buffer data (requires that all reads/writes be exact physical    
          sectors)                                                              
 9      do not compress file even if volume normally compresses files           
 10     use alias hint in DI as numeric tail for short-name alias               
 12-11  unused??? (0)                                                           
 13     return error code instead of generating INT 24h if critical error       
          while opening file                                                    
 14     commit file after every write operation                                 
SeeAlso: #0749                                                                  


--------D-216C00-----------------------------                                   
INT 21 - DOS 4.0+ - EXTENDED OPEN/CREATE                                        
        AX = 6C00h                                                              
        BL = open mode as in AL for normal open (see also AH=3Dh)               
            bit 7: inheritance                                                  
            bits 4-6: sharing mode                                              
            bit 3 reserved                                                      
            bits 0-2: access mode                                               
                100 read-only, do not modify file's last-access time (DOS 7.0)  
        BH = flags                                                              
            bit 6 = auto commit on every write (see also AH=68h)                
            bit 5 = return error rather than doing INT 24h                      
            bit 4 = (FAT32) extended size (>= 2GB)                              
        CX = create attribute (see #1111)                                       
        DL = action if file exists/does not exist (see #1112)                   
        DH = 00h (reserved)                                                     
        DS:SI -> ASCIZ file name                                                
Return: CF set on error                                                         
           AX = error code (see #1020 at AH=59h/BX=0000h)                       
        CF clear if successful                                                  
           AX = file handle                                                     
           CX = status (see #1110)                                              
Notes:  the PC LAN Program only supports existence actions (in DL) of 01h,      
          10h with sharing=compatibility, and 12h                               
        DR DOS reportedly does not support this function and does not return    
          an "invalid function call" error when this function is used.          
        the documented bits of BX are stored in the SFT when the file is opened 
          (see #0982,#0983)                                                     
BUG:    this function has bugs (at least in DOS 5.0 and 6.2) when used with     
          drives handled via the network redirector (INT 2F/AX=112Eh):          
            - CX (attribute) is not passed to the redirector if DL=11h,         
            - CX does not return the status, it is returned unchanged because   
              DOS does a PUSH CX/POP CX when calling the redirector.            
SeeAlso: AH=3Ch,AH=3Dh,AX=6C01h,AH=71h,INT 2F/AX=112Eh                          
                                                                                
(Table 1110)                                                                    
Values for extended open function status:                                       
 01h    file opened                                                             
 02h    file created                                                            
 03h    file replaced                                                           
                                                                                
Bitfields for file create attribute:                                            
Bit(s)  Description     (Table 1111)                                            
 6-15   reserved                                                                
 5      archive                                                                 
 4      reserved                                                                
 3      volume label                                                            
 2      system                                                                  
 1      hidden                                                                  
 0      readonly                                                                
                                                                                
Bitfields for action:                                                           
Bit(s)  Description     (Table 1112)                                            
 7-4    action if file does not exist                                           
        0000 fail                                                               
        0001 create                                                             
 3-0    action if file exists                                                   
        0000 fail                                                               
        0001 open                                                               
        0010 replace/open                                                       
**************/

_CLibFunc int
open(char const * const fnam
    ,const int omode
    ,const int flags)		/* This parameter might not been passed */
{	REG int errnr;
#ifdef LFN_WIN95
	struct REGPACK r, r1;
#endif

	/* decode the open mode */
	if(omode & O_CREAT)			/* creation behaviour */
		r.r_dx = (omode & O_EXCL)? OPEN_CREATE
		 : (omode & O_TRUNC)? OPEN_CREATE | OPEN_TRUNC
		 : OPEN_CREATE | OPEN_OPEN;
	else
		r.r_dx = (omode & O_TRUNC)? OPEN_TRUNC: OPEN_OPEN;

	r.r_bx = omode & OPEN_FLAGS;

	switch(r.r_bx & OPEN_SHARE) {
	case 0:	/* nothing specified -> use default */
		r.r_bx |= (omode & (O_WRONLY | O_RDWR))
			? O_DENYWRITE		/* write permission requested */
			: O_DENYNONE;		/* read-only permission requested */
		break;

	case O_DENYCOMP:	/* compatibly mode */
		r.r_bx &= ~OPEN_SHARE;
		break;
	}

	/* Set the values for LFN & DOS API */
	r.r_ds = FP_SEG(fnam);		/* pointer to file name */
	r.r_si = FP_OFF(fnam);
	r.r_cx = _attribu2d(flags);

#ifdef LFN_WIN95
	/* Frist try the Win95 API */
	r.r_di = 1;					/* alias hint */
	r.r_ax = 0x716c;			/* Win95 open/create */
	/* because the call the DOS 4 style function is nearly the same
		the structure is preserved for the next try */
	memcpy(&r1, &r, sizeof(r));

	if((errnr = _callDOS(&r)) != 0) {			/* Win95 call failed */
		if(errnr == 1 || errnr == 0x7100) {
			/* Win95 not present -> call DOS 4 API */

				/* Suppress Win95 specific flags */
			r1.r_bx &= O_AUTOCOMMIT | O_AUTOFAIL | O_NOINHERIT
			 | OPEN_SHARE | 3;
			r1.r_ax = 0x6c00;

			if((errnr = _callDOS(&r1)) == 0) {	/* DOS 4+ API didnt failed */
				r.r_ax = r1.r_ax;		/* re-join both APIs */
				goto success;
			}
		}
#else	/* !LFN_WIN95 */
	r.r_ax = 0x6c00;
	if((errnr = _callDOS(&r)) != 0) {
#endif	/* LFN_WIN95 */
		errno = errnr;	/* preserve this error */
		return -1;			/* error! */
	}

success:
	/* One of the API returned successfully -> do whatever is needed */

/* We currently ignore:
	1) DOS pre-4
	2) ^Z
	3) text/binary					 - 1997/09/03 ska*/

	if(omode & O_APPEND)		/* seek to EOF */
		lseek(r.r_ax, 0l, 1);

	return r.r_ax;				/* return the handle */
}
