{
    Copyright (C) 2021 Jerome Shidel
    BSD 3-Clause License
}
unit Strings;

interface

    const
        miTrue             = 'TRUE';
        miFalse            = 'FALSE';
        miLongSeparator    = 'SEPARATOR.LONG';
        miResultsSeparator = 'SEPARATOR.RESULTS';
        miLongColumnWidth  = 'LONG.COLUMN.WIDTH';
        miBIOSErrorPrefix  = 'BIOS.ERROR.';
        miDOSErrorPrefix   = 'DOS.ERROR.';

    const
        MoreInfo            : boolean = true;
        MoreWidth           : integer = -1;
        LongSeperator       : String = '';
        NlsDelim            : char = '|';
        NlsVarTag           : char = '%';

    type
        Str2  = String[2];
        Str4  = String[4];
        Str6  = String[6];
        Str8  = String[8];
        Str9  = String[9];
        Str10 = String[10];
        Str13 = String[13];
        Str20 = String[20];

    function ChrStr ( C : Char; N : byte ) : String;
    function RPad( Str : String; C : Char; N : Byte ) : String;

    function TrueFalse(State : boolean) : String;
    function GetLongNLS (ID : String) : String;

    function IntStr  ( L : LongInt ) : Str20;

    function ByteHEX ( B : byte ) : Str2;
    function WordHEX ( W : Word ) : Str4;

    function HEXByte    ( B : byte ) : Str4;
    function HEXWord    ( W : Word ) : Str6;
    function HEXLong    ( L : LongInt ) : Str10;
    function HEXPtr     ( P : Pointer ) : Str13;

    function NextPos ( SubStr : String; S : String; StartPos : byte ) : byte;
    function ReplaceStr ( Str : String; O, N : String ) : String;
    function PullStr(SubStr : String; var S : String) : String;

    function FormatStr(S, Data : String; Long : boolean) : String;
    function FormatNLS(ID, Data : String; Long : boolean) : String;

    function BIOSError(Code : word) : String;
    function DOSError(Code : word) : String;

    function TFDOSError(Value : Boolean; Code : Word) : String;
    function FlagsDOSError(ID : String; Value : word; Code : Word) : String;


implementation

uses NLS;

const
    HEXPrefix   = '0x';
    HEXPtrDelim = ':';

function ChrStr ( C : Char; N : byte ) : String;
var
    S : String;
begin
    S[0] := Char(N);
    if N > 0 then FillChar (S[1], N, C);
    ChrStr := S;
end;

function RPad( Str : String; C : Char; N : Byte ) : String;
begin
    if Length(Str) < N then
        RPad := Str + ChrStr( C,  N - Length(Str) )
    else
        RPad := Str;
end;

function TrueFalse(State : boolean) : String;
begin
    if State then
        TrueFalse := GetRawNLS(miTrue)
    else
        TrueFalse := GetRawNLS(miFalse)
end;

function GetLongNLS (ID : String) : String;
var
    S : String;
    E : integer;
begin
    if MoreWidth = -1 then begin
       S := GetRawNLS(miLongColumnWidth);
       Val(S, MoreWidth, E);
       if E <> 0 then MoreWidth := 0;
       LongSeperator := GetRawNLS(miLongSeparator);
    end;
    S := '';
    if MoreInfo then begin
        S := GetRawNLS(ID + '.LONG');
        if S <> '' then
            S := S + LongSeperator;
        S := RPad(S, ' ', MoreWidth);
    end;
   {     GetLongNLS := S + GetRawNLS(ID) + ResultSeparator }
    GetLongNLS := S + GetRawNLS(ID);
end;

function IntStr  ( L : LongInt ) : Str20;
var
    S : Str20;
begin
    Str ( L, S );
    IntStr := S;
end;

function ByteHEX ( B : byte ) : Str2; assembler;
asm
      LES   DI, @RESULT
      MOV   AL, $02
      MOV   ES:[DI], AL
      MOV   AL, B
      MOV   AH, B
      AND   AH, $0F
      AND   AL, $F0
      MOV   CL, $04
      SHR   AL, CL
      CMP   AL, $9
      JNA   @@1
      ADD   AL, $07
@@1:
      ADD   AL, $30
      CMP   AH, $9
      JNA   @@2
      ADD   AH, $07
@@2:
      ADD   AH, $30
      MOV   ES:[DI + 1], AX
end;

function WordHEX ( W : Word ) : Str4;
begin
    WordHEX := ByteHEX(Hi(W)) + ByteHEX(Lo(W));
end;

function LongHex ( L : LongInt ) : Str8;
type
      WordCast = array[0..1] of word;
begin
      LongHEX := WordHEX(WordCast(L)[1]) + WordHEX(WordCast(L)[0]);
end;

function HEXByte ( B : Byte ) : Str4;
begin
    HEXByte := HEXPrefix + ByteHEX(B);
end;

function HEXWord ( W : Word ) : Str6;
begin
    HEXWord := HEXPrefix + WordHEX(W);
end;

function HEXLong ( L : LongInt ) : Str10;
begin
    HEXLong := HEXPrefix + LongHEX(L);
end;

function HEXPtr ( P : Pointer ) : Str13;
begin
    HEXPtr := {HEXPrefix + }
        WordHEX(Seg(P^)) +
        { HEXPtrDelim + }
        HEXPrefix + WordHEX(Ofs(P^));
end;

function NextPos ( SubStr : String; S : String; StartPos : byte ) : byte;
var
    StrPtr : ^String;
    TPos   : Byte;
begin
    if (StartPos = 0) or (StartPos > Length(S)) then
        NextPos := 0
    else begin
        Dec(StartPos);
        S[StartPos] := Char(Length(S) - StartPos);
        StrPtr := @S[StartPos];
        TPos := Pos(SubStr, StrPtr^);
        if TPos > 0 then Inc(TPos, StartPos);
        NextPos := TPos;
    end;
end;

function ReplaceStr ( Str : String; O, N : String ) : String;
var
    P : integer;
begin
    P := Pos(O, Str);
    while P > 0 do begin
        Delete(Str, P, Length(O));
        Insert(N, Str, P);
        P :=NextPos(O, Str, P + Length(N));
    end;
    ReplaceStr := Str;
end;

function PullStr(SubStr : String; var S : String) : String;
var
	P : integer;
begin
	P := Pos(SubStr, S);
	if P = 0 then begin
		PullStr := S;
		S := '';
	end else begin
		PullStr := Copy(S, 1, P - 1);
		S := Copy(S, P + Length(SubStr), Length(S));
	end;
end;

function FormatStr(S, Data : String; Long : boolean) : String;
var
    T : String;
    I : integer;
begin
    I := 0;
    while Data <> '' do begin
        T := PullStr(NlsDelim,Data);
        S := ReplaceStr(S, NlsVarTag + IntStr(I), T);
        Inc(I);
    end;
    FormatStr := S;
end;

function FormatNLS(ID, Data : String; Long : boolean) : String;
var
    S, T : String;
    I : integer;
begin
    if Long then
        S := GetLongNLS(ID)
    else
        S := GetRawNLS(ID);
    I := 0;
    while Data <> '' do begin
        T := PullStr(NlsDelim,Data);
        S := ReplaceStr(S, NlsVarTag + IntStr(I), T);
        Inc(I);
    end;
    FormatNLS := S;
end;

function BIOSError(Code : word) : String;
var
    S : String;
begin
    S := GetRawNLS(miBIOSErrorPrefix + IntStr(Code));
    if S = '' then
        S := GetRawNLS(miBIOSErrorPrefix + '?');
    BIOSError := FormatNLS(miBIOSErrorPrefix + 'MSG', IntStr(Code) +
        NlsDelim + S, False);
end;

function DOSError(Code : word) : String;
var
    S : String;
begin
    S := GetRawNLS(miDOSErrorPrefix + IntStr(Code));
    if S = '' then
        S := GetRawNLS(miDOSErrorPrefix + '?');
    DOSError := FormatNLS(miDOSErrorPrefix + 'MSG', IntStr(Code) +
        NlsDelim + S, False);
end;



function TFDOSError(Value : Boolean; Code : Word) : String;
begin
    TFDOSError := GetRawNLS(miTrue);
    if Code <> 0 then
        TFDOSError := DOSError(Code)
    else
        TFDOSError := GetRawNLS(miFalse);
end;

function FlagsDOSError(ID : String; Value : word; Code : Word) : String;
var
    S, T, C, U, F : String;
    I, V : word;
    B : boolean;
begin
    if Code <> 0 then
        S := DOSError(Code)
    else begin
        S := '';
        C := GetRawNLS(miResultsSeparator);
        U := GetRawNLS(ID + '.FLAG.?');
        F := GetRawNLS(ID + '.FLAG');
        B := False;
        for I := 0 to 15 do begin
            V := 1 shl I;
            if Value and V = V then begin
                T := GetRawNLS(ID + '.FLAG.' + WordHEX(V));
                if T = '' then T := U;
                if S <> '' then
                    S := S + C;
                S := S + FormatStr(F, HexWord(V) + NLSDelim + T, False);
                B := True;
            end;
        end;
        if B = false then
            S := GetRawNLS(ID + '.FLAG.NONE');
    end;
    FlagsDOSError := S + NLSDelim + '' + NLSDelim + '';
end;



end.