/*
 * four-in-a-row game - logic
 * Copyright (c) 2017 Andreas K. Foerster <info@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include "row4.h"

byte board[7][6], filled[7];
short int chips;

static short int column;

#define outside(x,y)  (0 > (x) || (x) > 6 || 0 > (y) || (y) > 5)

static int won P ((int, int, int, int *));
static void analyze P ((int, byte *));
static void mark_threat P ((int, int, int));
static void seek_threat P ((int, int, int));

const char *const copyright =
  "Copyright (c) 2017 Andreas K. Foerster <info@akfoerster.de>";

#ifdef DEUTSCH

const char *const license = "\
http://akfoerster.de/p/row4/\n\
\n\
Dieses Programm ist freie Software. Sie koennen es unter den Bedingungen\n\
der GNU Affero General Public License, wie von der Free Software\n\
Foundation veroeffentlicht, weitergeben und/oder modifizieren, entweder\n\
gemaess Version 3 der Lizenz oder (nach Ihrer Option) jeder spaeteren\n\
Version.\n\
\n\
Die Veroeffentlichung dieses Programms erfolgt in der Hoffnung, dass es\n\
Ihnen von Nutzen sein wird, aber OHNE IRGENDEINE GARANTIE, sogar ohne\n\
die implizite Garantie der MARKTREIFE oder der VERWENDBARKEIT FUER EINEN\n\
BESTIMMTEN ZWECK. Details finden Sie in der GNU Affero General Public\n\
License.\n\
\n\
Sie sollten ein Exemplar der GNU Affero General Public License zusammen\n\
mit diesem Programm erhalten haben.\n\
Falls nicht, siehe <http://www.gnu.org/licenses/>.";

#else

const char *const license = "\
Copyright (c) 2017 Andreas K. Foerster <info@akfoerster.de>\n\
\n\
http://akfoerster.de/p/row4/\n\
\n\
This program is free software: you can redistribute it and/or modify\n\
it under the terms of the GNU Affero General Public License as\n\
published by the Free Software Foundation, either version 3 of the\n\
License, or (at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU Affero General Public License for more details.\n\
\n\
You should have received a copy of the GNU Affero General Public License\n\
along with this program.  If not, see <http://www.gnu.org/licenses/>.";

#endif


extern void
reset_board ()
{
  memset (board, NONE, sizeof (board));
  memset (filled, 0, sizeof (filled));

  chips = 42;
  column = ERROR;
}


static int
won (player, x, y, num)
     int player, x, y, *num;
{
  if (0 <= y && y <= 5 && (board[x][y] & 3) == player)
    *num += 1;
  else
    *num = 0;

  return (*num == 4);
}


extern int
wincheck ()
{
  int x, y, player, num;
  register int i, j;

  x = column;
  if (x < 0)
    return FALSE;

  y = filled[x] - 1;
  player = board[x][y] & 3;

  /* vertical */
  for (num = 0, i = 0; i <= 5; ++i)
    if (won (player, x, i, &num))
      goto yeah;

  /* horizontal */
  for (num = 0, i = 0; i <= 6; ++i)
    if (won (player, i, y, &num))
      goto yeah;

  /* ascending diagonal */
  for (num = 0, i = 0, j = y - x; i <= 6; ++i, ++j)
    if (won (player, i, j, &num))
      goto yeah;

  /* descending diagonal */
  for (num = 0, i = 0, j = y + x; i <= 6; ++i, --j)
    if (won (player, i, j, &num))
      goto yeah;

  return FALSE;

yeah:
  chips = 0;
  return TRUE;
}


/* drop chip into slot x */
/* returns ý or ERROR */
extern int
drop (player, x)
     int player, x;
{
  int y;

  if (0 > x || x > 6 || 1 > player || player > 2)
    return ERROR;

  /* one above */
  y = filled[x];
  if (y > 5)
    return ERROR;		/* full */

  /* a filled field is no longer a threat */
  board[x][y] = player;
  column = x;
  filled[x] += 1;
  chips -= 1;

  seek_threat (x, y, player);

  return y;
}


/*
 * computer logic
 */

/* gets line of chips, turns it into line of threats */
static void
analyze (player, line)
     int player;
     byte *line;
{
  int i, threat;
  char s[8], *c;

  /* turn line into string */
  for (i = 0; i <= 6; ++i)
    {
      /* pl ignores threats */
      int pl = line[i] & 3;
      if (pl == NONE)
	s[i] = '.';
      else if (pl == player)
	s[i] = 'P';
      else
	s[i] = 'C';
    }

  s[7] = '\0';

  memset (line, 0, 7);
  threat = player << 2;

  c = strstr (s, "P.PP.P");
  if (c)
    {
      int p = c - s;
      line[p + 1] = line[p + 4] = threat;
    }

  c = strstr (s, ".PPP.");
  if (c)
    {
      int p = c - s;
      line[p] = line[p + 4] = threat;
    }

  c = strstr (s, ".PPP");
  if (c)
    line[c - s] = threat;

  c = strstr (s, "PPP.");
  if (c)
    line[c - s + 3] = threat;

  c = strstr (s, "P.PP");
  if (c)
    line[c - s + 1] = threat;

  c = strstr (s, "PP.P");
  if (c)
    line[c - s + 2] = threat;
}


static void
mark_threat (x, y, threat)
     int x, y, threat;
{
  if (threat && 0 <= y && y <= 5 && (board[x][y] & 3) == NONE)
    board[x][y] |= threat;
}


static void
seek_threat (x, y, player)
     int x, y, player;
{
  byte line[7];
  register int i, j;

  /* look for horizontal threat */
  for (i = 0; i <= 6; ++i)
    line[i] = board[i][y];
  analyze (player, line);
  for (i = 0; i <= 6; ++i)
    mark_threat (i, y, line[i]);

  /* look for vertical threat */
  for (i = 0; i <= 5; ++i)
    line[i] = board[x][i];
  line[6] = NONE;
  analyze (player, line);
  for (i = 0; i <= 5; ++i)
    mark_threat (x, i, line[i]);

  /* look for ascending diagonal threat */
  for (i = 0, j = y - x; i <= 6; ++i, ++j)
    line[i] = outside (i, j) ? NONE : board[i][j];
  analyze (player, line);
  for (i = 0, j = y - x; i <= 6; ++i, ++j)
    mark_threat (i, j, line[i]);

  /* look for descending diagonal threat */
  for (i = 0, j = y + x; i <= 6; ++i, --j)
    line[i] = outside (i, j) ? NONE : board[i][j];
  analyze (player, line);
  for (i = 0, j = y + x; i <= 6; ++i, --j)
    mark_threat (i, j, line[i]);
}


extern int
compute (player)
     int player;
{
  int f, mythreat;
  register int i;

  if (1 > player || player > 2)
    return ERROR;

  /* own threat, but not both */
  mythreat = player << 2;

  /* no chip set yet - the center is the best start */
  if (chips == 42)
    return 3;

  /* one chip - go next to it */
  if (chips == 41)
    return ((column < 4) ? column + 1 : column - 1);

  /* can I win? */
  for (i = 0; i <= 6; ++i)
    {
      f = filled[i];		/* above */
      if (f < 6 && (board[i][f] & mythreat) != 0)
	return i;
    }

  /* close threats */
  for (i = 0; i <= 6; ++i)
    {
      f = filled[i];		/* above */
      if (f < 6 && board[i][f] > 3)
	return i;
    }

  /* seek field that is not indirectly threatend - avoid current column */
  for (i = 0; i <= 3; ++i)
    {
      int c;

      c = 3 - i;
      f = filled[c];
      if (c != column && (f == 5 || (f < 5 && board[c][f + 1] < 3)))
	return c;

      if (i == 0)
	continue;

      c = 3 + i;
      f = filled[c];
      if (c != column && (f == 5 || (f < 5 && board[c][f + 1] < 3)))
	return c;
    }

  /* check the same for current column */
  f = filled[column];
  if (f == 5 || (f < 5 && board[column][f + 1] < 3))
    return column;

  /* all are indirectly threatened => give up an own threat */
  for (i = 0; i <= 3; ++i)
    {
      int c;

      c = 3 - i;
      f = filled[c];
      if (f < 5 && board[c][f + 1] == mythreat)
	return c;

      if (i == 0)
	continue;

      c = 3 + i;
      f = filled[c];
      if (f < 5 && board[c][f + 1] == mythreat)
	return c;
    }

  /* last resort: search for any free column */
  for (i = 0; i <= 3; ++i)
    if (filled[3 - i] < 6)
      return (3 - i);
    else if (i != 0 && filled[3 + i] < 6)
      return (3 + i);

  return ERROR;
}
