/*
    board.c
    Functions to display the game board.
    copyright (C) 2020 Jim Hall <jhall@freedos.org>

    This is free software under the GNU GPL (only version 2).
    See the file COPYING for the full text of the GNU GPL. 
*/

#include <conio.h>
#include <graph.h>
#include "board.h"
#include "win.h"

/*
    the Senet board uses a backwards "S" layout: (counting from 0)
    0  1  2  3  4  5  6  7  8  9
    19 18 17 16 15 14 13 12 11 10
    20 21 22 23 24 25 26 27 28 29
*/

void
set_square(int sq)
{
    int row, col;

    /* squares are 7 cols wide, at every 8 cols, starting at 1:
	1, 9, 17, ... */

    if ((sq >= 10) && (sq <= 19)) {
	/* middle row counts backwards */
	col = (19 - sq) * 8 + 1;
    }
    else {
	/* top and bottom rows count forwards */
	col = (sq > 19 ? sq - 20 : sq) * 8 + 1;
    }

    /* squares are 4 rows high, at every 5 rows, starting at 3 */

    row = (sq / 10) * 5 + 3;

    /* squares are 4 rows tall, so 1+3=4 */
    /* squares are 7 cols wide, so 1+6=7 */

    _settextwindow(row,col, row+3,col+6);
}

void
draw_square(int sq, int player, int is_active)
{
    int row;

    /* set the text window for the square */

    set_square(sq);

    /* set background color */

    if (is_active) {
	_setbkcolor(3); /* cyan */
	_settextcolor(11); /* br cyan */
    }
    else {
	_setbkcolor(1); /* blue */
	_settextcolor(9); /* br blue */
    }

    _clearscreen(_GWINDOW);

    /* draw markers on special squares */

    switch (sq) {
    case 29: /* need "1" to move off */
	_settextposition(1,4); /* centered */
	_outtext("\340"); /* alpha */
	break;
    case 28: /* need "2" to move off */
	_settextposition(1,3);
	_outtext("\341 \341"); /* beta */
	break;
    case 27: /* need "3" to move off */
	_settextposition(1,2);
	_outtext("\342 \342 \342"); /* gamma */
	break;
    case 26: /* trap square */
	for (row = 1; row <= 2; row++) {
	    _settextposition(row,2);
	    _outtext("\367\367\367\367\367"); /* waves */
	}
	break;
    case 14: /* resurrection square (revive from trap square) */
	_settextposition(1,4);
	_outtext("\014"); /* ankh */
	break;
    case 25: /* "stop here" square */
	for (row = 1; row <= 5; row++) {
	    _settextposition(row,7);
	    putch('\263'); /* vert bar (border) */
	}
    }

    /* draw markers to indicate board movement */

    switch (sq) {
    case 0: /* "start here" square */
	_settextposition(4,1); /* bottom left */
	putch('\032'); /* right arrow */
	break;
    case 9:
	_settextposition(4,7); /* bottom right */
	putch('\277'); /* corner */
	break;
    case 10:
	_settextposition(1,7); /* top right */
	putch('\331'); /* corner */
	break;
    case 19:
	_settextposition(4,1); /* bottom left */
	putch('\332'); /* corner */
	break;
    case 20:
	_settextposition(1,1); /* upper left */
	putch('\300'); /* corner */
	break;
    case 29:
	_settextposition(4,7); /* lower right */
	putch('\032'); /* right arrow */
    }

    /* draw the player piece (1, 2, or set empty) */
    /* we need to use _outtext instead of putch, because putch does not use
	color from _settextcolor (unless _clearscreen is called?) [BUG?] */

    _settextposition(3,4); /* centered */
    _settextcolor(15); /* br white */

    _outtext(player_ch(player));
}

const char *
player_ch(int player)
{
    /* return a DOS extended char that represents the player's piece */
    /* player can be 1 or 2 (or 0) */

    switch (player) {
    case 1:
	return("\001"); /* smiley face */
    case 2:
	return("\002"); /* alt smiley face */
    default:
	return(" "); /* empty */
    }
}