unit hiscores;

interface

const
  c_MaxChars = 15;
{ decrypted score record }
type TScoreRecordDec = record
    Name : String[15];
    Score : Word;
  end; {record}


function LoadScores : Boolean;
function SaveScores : Boolean;

var
  HiScoresArray : Array[0..2] of TScoreRecordDec;

implementation

uses
  GameCode;

type
  TScoreRecordEnc = Array[0..18] of Byte;
  { encryted hiscore record. The data contains
  1  byte  - CRC check, simply added the other 18 bytes (to prevent easy score editing..)
  1  byte  - difficulty level, 0..2
  15 bytes - name, every character ASCII code - 50 - charposition (so people can not easily read/edit it..)
  2  bytes - score
  }

const
  c_ScoreFilename = 'hiscores.dat';
  c_HeaderLength = 18;
  c_HeaderText : Array[0..c_HeaderLength-1] of char = (
  'U', 'r', 't', 'h', 'W', 'u', 'r', 'm', ' ', 'h', 'i', 's', 'c', 'o', 'r',
  'e', 's', ' '); {incase someone looks into the hiscores file, he will know what it is :) }


procedure ClearScores;
{ fill hiscores with dummy scores }
var
  i : Byte;
begin
  for i := 0 to 2 do
  begin
    HiScoresArray[i].Name := HiddenString(']]]]]]]]]]]]'); {'BdR            '}
    HiScoresArray[i].Score := 0;
  end;
end;

function LoadScores : Boolean;
var
  ScoreFile : File;
  ScoreRecordEnc : TScoreRecordEnc;
  i, CheckSum, Difficulty : Byte;
  strName : String;
  wScore : Word;
  FileCorrupt : Boolean;
begin
  LoadScores := False;
  FileCorrupt := False;

  { clear score list }
  ClearScores;

  {$I-}
  Assign(ScoreFile, c_ScoreFilename);
  Reset(ScoreFile, 1);
  {$I+}

  If IOResult <> 0 Then Begin
    {writeln('error opening scorefile');}
    Exit;
  End;

  {skip header }
  Seek(ScoreFile, c_HeaderLength);

  { read the 'encrypted' score records }
  while not eof(ScoreFile) do
  begin
    BlockRead(ScoreFile, ScoreRecordEnc, sizeOf(TScoreRecordEnc));
    { determine what checksum *should* be for this encrypted record }
    CheckSum := 0;
    for i := 1 to 18 do
      CheckSum := CheckSum + ScoreRecordEnc[i];
    { get Difficulty from encrypted record }
    Difficulty := ScoreRecordEnc[1];
    { get name from encrypted record }
    strName := '';
    for i := 2 to 16 do
      strName := strName + Chr(ScoreRecordEnc[i]+50+i);
    { get score from encrypted record }
    wScore := ScoreRecordEnc[17];
    wScore := wScore * 256;
    wScore := wScore + ScoreRecordEnc[18];
    { check if checksum is okay, this is to prevent easy score editing using a hexeditor :) }
    if (CheckSum = ScoreRecordEnc[0])
    and (Difficulty <= 2) then
    begin
      HiScoresArray[Difficulty].Name := strName;
      HiScoresArray[Difficulty].Score := wScore;
    end
    else
      FileCorrupt := True;
  end;

  Close(ScoreFile);

  { loadscore went okay, only if file not corrupt }
  LoadScores := not FileCorrupt;
end;

function SaveScores : Boolean;
var
{  ScoresFound : Boolean;}
  ScoreFile : File;
  ScoreRecordEnc : TScoreRecordEnc;
  i, j, tmpByte : Byte;
begin
  SaveScores := False;

  { precheck if there are any score to save }
(*  ScoresFound := False;
  for i := 0 to 2 do
    if (HiScoresArray[i].Score <> 0) then
      ScoresFound := True;

  { if no scores to save exit this function }
  if not ScoresFound then Exit;
*) {precheck is not needed anymore, this function is only called when a hiscore was added..}

  { prepare file }
  {$I-}
  Assign(ScoreFile, c_ScoreFilename);
  Rewrite(ScoreFile);
  Reset(ScoreFile, 1);
  {$I+}

  If IOResult <> 0 Then Exit;

  { write header text to file }
  BlockWrite(ScoreFile, c_HeaderText, c_HeaderLength);

  for i := 0 to 2 do
  begin
    if (HiScoresArray[i].Score <> 0) then { only store relevant scores }
    begin
      { put difficulty level into encrypted record }
      ScoreRecordEnc[1] := i;
      { encrypt the name, so it can not easily be seen with hexeditor }
      tmpByte := Length(HiScoresArray[i].Name);
      for j := 2 to 1+tmpByte do
        ScoreRecordEnc[j] := Ord(HiScoresArray[i].Name[j-1])-50-j;
      { supplement with spaces }
      if tmpByte < c_MaxChars then
        for j := 2+tmpByte to 1+c_MaxChars do
          ScoreRecordEnc[j] := (32-50-j); { #32=space }
      { put score into encrypted record }
      tmpByte := Hi(HiScoresArray[i].Score);  { 8 most significant bits }
      ScoreRecordEnc[17] := tmpByte;
      tmpByte := Lo(HiScoresArray[i].Score);  { 8 least significant bits }
      ScoreRecordEnc[18] := tmpByte;
      { put checksum into encrypted record }
      tmpByte := 0;
      for j := 1 to 18 do
        tmpByte := tmpByte + ScoreRecordEnc[j];
      ScoreRecordEnc[0] := tmpByte;
      { write to file }
      BlockWrite(ScoreFile, ScoreRecordEnc, SizeOf(ScoreRecordEnc));
    end;
  end;

  Close(ScoreFile);

  SaveScores := True;
end;

end.
