//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2019 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFMPEG library parser

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

#include "mpxplay.h"

#define INFFMPEG_AUDIO_INITERROR_LIMIT 100

extern unsigned long mpxplay_config_video_audiovisualization_type;

static mpxp_bool_t inffmpgdec_is_audio_ctx_valid(AVCodecContext *audio_ctx)
{
	return mpxplay_ffmpstrm_is_avctx_valid(audio_ctx, MPXPLAY_STREAMTYPEINDEX_AUDIO);
}

static void inffmpgdec_decodeaudio_reset(struct ffmpg_demuxer_data_s *ffmpi, mpxplay_packetlist_t *pktlist_ctx, AVCodecContext *audio_ctx)
{
	if(ffmpi->audio_frame)
		av_frame_unref(ffmpi->audio_frame);
	if(inffmpgdec_is_audio_ctx_valid(audio_ctx))
		avcodec_flush_buffers(audio_ctx);
	ffmpi->avsync_timestamp = AV_NOPTS_VALUE;
#ifdef INFFMP_USE_VIDEO_SYNC
	ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] = AV_NOPTS_VALUE;
	funcbit_enable(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC);
#endif
	mpxplay_ffmpgdec_queuelistelem_clear(&ffmpi->audiodec_curr_pktlist_elem);
	ffmpi->audioctx_init_error_count = ffmpi->audiosync_live_init_error_count = 0;
	funcbit_disable(pktlist_ctx->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
	mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC,"decodeaudio_reset RESET");
}

static int inffmpgdec_decodeaudio_clearbuf(struct ffmpg_demuxer_data_s *ffmpi, mpxplay_packetlist_t *pktlist_ctx, AVCodecContext *audio_ctx)
{
	int retcode_audio = 0;

	if(pktlist_ctx->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)
	{
		inffmpgdec_decodeaudio_reset(ffmpi, pktlist_ctx, audio_ctx);
		ffmpi->audioframe_samples_per_channel = 0;
		ffmpi->audioframe_copy_sampleblocksize = 0;
		ffmpi->audioframe_copy_samplepos = 0;
		mpxplay_ffmpgdec_packetqueue_clear(&ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO], MPXPLAY_PACKETLISTFLAG_CLEARBUF);
		if(ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].last_pkt)
			funcbit_enable(ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].last_pkt->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);  // FIXME: not thread safe
		mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC, "decodeaudio_clearbuf CLEARBUF");
	}

	if(inffmpgdec_is_audio_ctx_valid(audio_ctx) && (audio_ctx->codec_id == AV_CODEC_ID_MP2) && (ffmpi->retcode_demux != 0) && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets < 1))
	{  // FIXME: hack (to not play last, possibly invalid mp2 frame)
		retcode_audio = AVERROR(ESPIPE);
		mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "decodeaudio_clearbuf MP2 discarded frame");
	}

	return retcode_audio;
}

#ifdef INFFMP_USE_VIDEO_SYNC
static int inffmpgdec_decodeaudio_silentblock_by_vsync(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_streampacket_info_s *spi, const mpxp_int64_t avsync_vidtimestamp)
{
	const unsigned int streamflags_audio = ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO];
	int retcode_audio = 0;

	if( funcbit_test(streamflags_audio, (INFFMPG_STREAMFLAG_AVSYNC | INFFMPG_STREAMFLAG_AVRESYNC))
		&& (funcbit_test(streamflags_audio, INFFMPG_STREAMFLAG_AVRESYNC) || in_ffmpgdec_content_has_programs(ffmpi))
		&& (ffmpi->selected_stream_number[MPXPLAY_STREAMTYPEINDEX_VIDEO] >= 0)
	){
		if( (avsync_vidtimestamp <= 0)
		 || ( funcbit_test(streamflags_audio, INFFMPG_STREAMFLAG_AVRESYNC)
			  && ((ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] + 20000) > avsync_vidtimestamp) )
		){
			spi->bs_leftbytes = PCM_OUTSAMPLES * spi->block_align;
			pds_memset(spi->bitstreambuf, 0, spi->bs_leftbytes);
			funcbit_enable(spi->flags,MPXPLAY_SPI_FLAG_SILENTBLOCK);
			retcode_audio = AVERROR(EAGAIN);
			mpxplay_debugf(MPXPLAY_DEBUG_SILENTB,"decodeaudio_silentblock_by_vsync SILENT rs:%d ats:%lld vts:%lld avv:%lld",
				(funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVRESYNC)? 1 : 0),
				ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO], avsync_vidtimestamp, ffmpi->avsync_videotimestamp);
		}
		else if(funcbit_test(streamflags_audio, INFFMPG_STREAMFLAG_AVRESYNC))
		{
			funcbit_disable(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVRESYNC);
			mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC,"silentblock_by_vsync --------- AVRESYNC END ----------");
		}
	}

	return retcode_audio;
}
#endif // INFFMP_USE_VIDEO_SYNC

static int inffmpgdec_decodeaudio_load_next_packet(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_infile_info_s *miis, mpxplay_packetlist_t *pktlist_ctx)
{
	int retcode_audio = 0;

	if(!ffmpi->audiodec_curr_pktlist_elem)
		mpxplay_ffmpgdec_packetqueue_get((mpxp_ptrsize_t)ffmpi, &ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO], &ffmpi->audiodec_curr_pktlist_elem, 0);

	if(ffmpi->audiodec_curr_pktlist_elem)
	{
		AVCodecContext *packet_avctx = (AVCodecContext *)ffmpi->audiodec_curr_pktlist_elem->codec_ctx;
		AVCodecContext *audio_ctx = (AVCodecContext *)pktlist_ctx->codec_ctx;
		if( ((packet_avctx == audio_ctx) || (ffmpi->audiodec_curr_pktlist_elem->codecctx_counter > pktlist_ctx->codecctx_counter))
		 && inffmpgdec_is_audio_ctx_valid(packet_avctx))    // equal or newer valid ctx
		{
			AVPacket *pkt_audio = (AVPacket *)(ffmpi->audiodec_curr_pktlist_elem->frame_pkt);
			if(ffmpi->audiodec_curr_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)
			{
				avcodec_flush_buffers(packet_avctx);
				//in_fmpeg_audio_codecid_to_waveid(ffmpi, miis, NULL); // TODO: stream type change handling
				ffmpi->avsync_timestamp = INFFMPG_INVALID_PTS;
				mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC, "load_next_packet CLEARBUF ctx:%8.8X", (mpxp_ptrsize_t)audio_ctx);
			}

			if(pkt_audio && (pkt_audio->size > 0))
			{
				retcode_audio = avcodec_send_packet(packet_avctx, pkt_audio);
#ifdef MPXPLAY_USE_DEBUGF
				if(retcode_audio != 0) // this can happen normally at new stream or at seeking (continuity error)
				{
					mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "audio avcodec_send_packet %8.8X r:%d s:%d ts:%lld cx:%8.8X", (mpxp_ptrsize_t)ffmpi,
						 retcode_audio, pkt_audio->size, ffmpi->audiodec_curr_pktlist_elem->timestamp_us, (mpxp_ptrsize_t)audio_ctx);
				}
				else
				{
					mpxp_int64_t av_diff = ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] - ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_VIDEO];
					if( (ffmpi->selected_stream_number[MPXPLAY_STREAMTYPEINDEX_VIDEO] >= 0)
					 && ((av_diff < -INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE) || (av_diff > INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE))
					){
						AVStream *sta = ffmpi->selected_avstream[MPXPLAY_STREAMTYPEINDEX_AUDIO];
						AVStream *stv = ffmpi->selected_avstream[MPXPLAY_STREAMTYPEINDEX_VIDEO];
						mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC, "AUDIO PKT dts:%lld pts:%lld tb:%d/%d atb:%d/%d vtb:%d/%d as:%lld vs:%lld vd:%d",
							pkt_audio->dts, pkt_audio->pts, pkt_audio->time_base.den, pkt_audio->time_base.num,
							sta->time_base.den, sta->time_base.num, stv->time_base.den, stv->time_base.num,
							sta->start_time, stv->start_time, stv->codecpar->video_delay);
					}
				}
#endif
			}
			else
			{
				mpxplay_debugf(MPXPLAY_DEBUG_ERROR, "audio avcodec_send_packet SIZE:0 %8.8X", (mpxp_ptrsize_t)ffmpi);
			}
			mpxplay_ffmpgdec_queuelistelem_clear(&ffmpi->audiodec_curr_pktlist_elem);
			ffmpi->audioctx_init_error_count = 0;
		}
		else if(packet_avctx && (ffmpi->audiodec_curr_pktlist_elem->codecctx_counter >= pktlist_ctx->codecctx_counter) && (++ffmpi->audioctx_init_error_count < 4))
		{
			retcode_audio = AVERROR(ESPIPE);
			mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "mpxplay_ffmpgdec_packetqueue_get PACKET CTX UNINIT cx:%8.8X px:%8.8X",
					(mpxp_ptrsize_t)audio_ctx, (mpxp_ptrsize_t)packet_avctx);
		}
		else // else we drop the packet (containing the incorrect ctx) after a time (4 loops = 80ms)
		{
			mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "mpxplay_ffmpgdec_packetqueue_get PACKET DROP ec:%d cv:%d cp:%d nba:%d nbv:%d",
					ffmpi->audioctx_init_error_count, pktlist_ctx->codecctx_counter, ffmpi->audiodec_curr_pktlist_elem->codecctx_counter,
					ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets);
			mpxplay_ffmpgdec_queuelistelem_clear(&ffmpi->audiodec_curr_pktlist_elem);
			ffmpi->audioctx_init_error_count = 0;
		}
	}
	else
	{
		retcode_audio = AVERROR(ESPIPE);
		mpxplay_debugf(MPXPLAY_DEBUG_ADECODE, "mpxplay_ffmpgdec_packetqueue_get failed");
	}

	return retcode_audio;
}

static void inffmpgdec_decodeaudio_calculate_sampleoutblocksize(struct ffmpg_demuxer_data_s *ffmpi, int sample_rate, int bytes_per_sample)
{
	mpxp_int64_t pkt_duration;

	if((sample_rate > 0) && (bytes_per_sample > 0))
	{
		ffmpi->audioframe_samples_per_channel = ffmpi->audio_frame->linesize[0] / bytes_per_sample;
		if((ffmpi->audioframe_samples_per_channel <= 0) || ((ffmpi->audio_frame->nb_samples > 0) && (ffmpi->audioframe_samples_per_channel > ffmpi->audio_frame->nb_samples)))
			ffmpi->audioframe_samples_per_channel = ffmpi->audio_frame->nb_samples;
		if(ffmpi->audioframe_samples_per_channel > INFFMPG_MAX_OUTPUT_SAMPLENUM){
			mpxplay_debugf(MPXPLAY_DEBUG_ERROR,"in_ffmpgdec_decode_audio BUFOVERFLOW sp:%d > ms:%d", ffmpi->audioframe_samples_per_channel, INFFMPG_MAX_OUTPUT_SAMPLENUM);
			ffmpi->audioframe_samples_per_channel = INFFMPG_MAX_OUTPUT_SAMPLENUM;
		}

		pkt_duration = (mpxp_int64_t)AV_TIME_BASE * (mpxp_int64_t)ffmpi->audioframe_samples_per_channel / (mpxp_int64_t)sample_rate;
		if(pkt_duration < (AV_TIME_BASE / 20)) // more than 20 audio fps
			ffmpi->audioframe_copy_sampleblocksize = ffmpi->audioframe_samples_per_channel;
		else
			ffmpi->audioframe_copy_sampleblocksize = (AV_TIME_BASE / 25) * (mpxp_int64_t)ffmpi->audioframe_samples_per_channel / pkt_duration;
	}
	else
	{
		ffmpi->audioframe_copy_sampleblocksize = 0;
	}
	ffmpi->audioframe_copy_samplepos = 0;
}

static int inffmpgdec_decodeaudio_receive_next_frame(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_infile_info_s *miis, AVCodecContext *audio_ctx)
{
	int retcode_audio = 0;
	if(!ffmpi->audio_frame->data[0] || !ffmpi->audio_frame->linesize[0] || !ffmpi->audioframe_copy_sampleblocksize || (ffmpi->audioframe_copy_samplepos >= ffmpi->audioframe_samples_per_channel))
	{
		if(inffmpgdec_is_audio_ctx_valid(audio_ctx))
		{
			retcode_audio = avcodec_receive_frame(audio_ctx, ffmpi->audio_frame);
			if(retcode_audio == 0)
			{
				retcode_audio = in_fmpeg_audio_codecid_to_waveid(ffmpi, miis, ffmpi->audio_frame);
				if(retcode_audio >= 0)
				{
					int64_t new_timestamp = in_ffmpgdec_get_avframe_timestamp(ffmpi, MPXPLAY_STREAMTYPEINDEX_AUDIO, ffmpi->audio_frame);
					if(new_timestamp > 0)
						ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] = new_timestamp;
					inffmpgdec_decodeaudio_calculate_sampleoutblocksize(ffmpi, miis->audio_decoder_infos->freq, ffmpi->outsample_size);
#if MPXPLAY_USE_FFMPEG_V7X

#else
					mpxplay_debugf(MPXPLAY_DEBUG_ADECODE, "avcodec_receive_frame ra:%d ch:%d chl:%16.16llX ptsa:%lld ptsl:%lld", retcode_audio, ffmpi->audio_frame->ch_layout.nb_channels,
					  ffmpi->audio_frame->channel_layout, ffmpi->audio_frame->pts, ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO]);
#endif
				}
			}
		}
		else
		{
			retcode_audio = AVERROR(ESPIPE);
		}
	}
	return retcode_audio;
}

static int inffmpgdec_decodeaudio_copy_audio(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_infile_info_s *miis, const mpxp_int64_t avsync_vidtimestamp, int sample_blocksize, int channels, int bytes_per_sample)
{
	struct mpxplay_streampacket_info_s *spi = miis->audio_stream;
	int retcode_audio = 0;
#ifdef INFFMP_USE_VIDEO_SYNC
	mpxp_int64_t last_pts_audio = ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO];
	mpxp_int64_t diff = avsync_vidtimestamp - last_pts_audio;
	if(ffmpi->audio_frame->data[0] &&
	   (!funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC)
		|| (ffmpi->selected_stream_number[MPXPLAY_STREAMTYPEINDEX_VIDEO] < 0)
		|| !in_ffmpgdec_content_has_programs(ffmpi)
		|| (  (avsync_vidtimestamp > 0)
		   && (   ( funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && ((diff > INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE) || (diff < -INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE)) && (++ffmpi->audiosync_live_init_error_count > INFFMPEG_AUDIO_INITERROR_LIMIT) ) // FIXME: workaround if DVB A/V timestamps are incorrect (don't match)
		       || ( (avsync_vidtimestamp <= (last_pts_audio + 20000)) && ((avsync_vidtimestamp + (AV_TIME_BASE * 5)) > last_pts_audio) )
			  )
		   )
	   )
	){
#endif
		if(ffmpi->flags & INFFMPG_FLAG_PLANARAUDIO){
			int ch, sn;
			for(ch = 0; (ch < channels) && ffmpi->audio_frame->data[ch]; ch++){
				uint8_t *framedata = ffmpi->audio_frame->data[ch] + ffmpi->audioframe_copy_samplepos * bytes_per_sample;
				for(sn = 0; sn < sample_blocksize; sn++, framedata += bytes_per_sample)
					pds_memcpy(spi->bitstreambuf + ((sn * channels) + ch) * bytes_per_sample, framedata, bytes_per_sample);
			}
			spi->bs_leftbytes = sample_blocksize * channels * bytes_per_sample;
			mpxplay_debugf(MPXPLAY_DEBUG_ADECODE,"INFFMPG_FLAG_PLANARAUDIO ns:%d ls:%d lb:%d bb:%d", ffmpi->audio_frame->nb_samples, (ffmpi->audio_frame->linesize[0] / bytes_per_sample), spi->bs_leftbytes, spi->bs_bufsize);
		} else {
			int datapos_in = ffmpi->audioframe_copy_samplepos * channels * bytes_per_sample;
			int datasize = sample_blocksize * channels * bytes_per_sample;
			if((datapos_in + datasize) > ffmpi->audio_frame->linesize[0]){
				mpxplay_debugf(MPXPLAY_DEBUG_ERROR,"decodeaudio_copy_audio BUFOVERFLOW-IN ds:%d > bs:%d", (datapos_in + datasize), ffmpi->audio_frame->linesize[0]);
				datasize = ffmpi->audio_frame->linesize[0] - datapos_in;
			}
			if(datasize > spi->bs_bufsize){
				mpxplay_debugf(MPXPLAY_DEBUG_ERROR,"decodeaudio_copy_audio BUFOVERFLOW-OUT ds:%d > bs:%d", datasize, spi->bs_bufsize);
				datasize = spi->bs_bufsize;
			}
			if(datasize > 0)
			{
				spi->bs_leftbytes = datasize;
				pds_memcpy(spi->bitstreambuf, ffmpi->audio_frame->data[0] + datapos_in, datasize);
			}
		}
		if(funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC))
		{
			funcbit_disable(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC);
#ifdef MPXPLAY_USE_DEBUGF
#if MPXPLAY_USE_FFMPEG_V7X

#else
			mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC,"AVSYNC START a:%lld v:%lld adur:%lld diff:%lld",
					ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO], avsync_vidtimestamp, ffmpi->audio_frame->pkt_duration, diff);
#endif
			if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && ((diff > INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE) || (diff < -INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE)) && (ffmpi->audiosync_live_init_error_count > INFFMPEG_AUDIO_INITERROR_LIMIT))
			{
				mpxplay_debugf(MPXPLAY_DEBUG_ERROR,"OUT OF SYNC start");
			}
#endif
		}
		funcbit_disable(spi->flags, MPXPLAY_SPI_FLAG_SILENTBLOCK);
#ifdef INFFMP_USE_VIDEO_SYNC
	}
	else
	{
		spi->bs_leftbytes = sample_blocksize * channels;
		pds_memset(spi->bitstreambuf, 0, spi->bs_leftbytes);
		funcbit_enable(spi->flags, MPXPLAY_SPI_FLAG_SILENTBLOCK);
		mpxplay_debugf(MPXPLAY_DEBUG_SILENTB, "decodeaudio_copy_audio SILENT avv:%lld alpts:%lld diff:%lld",
				avsync_vidtimestamp, ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO], diff);
	}
#endif
	return retcode_audio;
}

static void inffmpgdec_decodeaudio_correct_currtimestamp(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_streampacket_info_s *spi, int sample_rate)
{
#ifdef MPXPLAY_GUI_QT
	if(mpxplay_config_video_audiovisualization_type
	 || (!funcbit_test(ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_AUDIO].control_flags, INFFMPG_FLAG_IS_LIVESTREAM)
	     && (ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_VIDEO] || ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_SUBTITLE]))
	){
		mpxp_int64_t timestamp_new = mpxplay_infile_callback_check_audio_timesync(ffmpi->fbds, ffmpi->avsync_clocktime, &ffmpi->avsync_timestamp, ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO], spi->bs_leftbytes, sample_rate);
		if((ffmpi->avsync_timestamp <= 0) && !funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC))
		{
			in_ffmpgdec_avsync_assign(ffmpi, timestamp_new);
			mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC, "decodeaudio RESYNC s:%d sy:%llu nba:%d nbv:%d", ffmpi->audioframe_samples_per_channel,
				ffmpi->avsync_timestamp, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets);
		}
	}
#endif
}

static void inffmpgdec_decodeaudio_audiovisualization_prepare(struct ffmpg_demuxer_data_s *ffmpi, mpxplay_packetlist_t *pktlist_ctx, int sample_rate, int sample_blocksize, int channels, int bytes_per_sample)
{
	if( mpxplay_config_video_audiovisualization_type && !ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_VIDEO] && funcbit_test(ffmpi->flags, INFFMPG_FLAG_CALLBACKVIDEOCFG)
	 && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets < INFFMPG_QUEUE_PACKETNUM_FILL_MAX) && ffmpi->audio_frame->data[0]
	){
		int retVal;

		AVFrame *visa_frame = av_frame_alloc();
		visa_frame->format = ffmpi->audio_frame->format;
		visa_frame->sample_rate = sample_rate;
		visa_frame->nb_samples = sample_blocksize;
#if MPXPLAY_USE_FFMPEG_V7X
		visa_frame->ch_layout = ffmpi->audio_frame->ch_layout;
#else
		visa_frame->channels = channels;
		visa_frame->channel_layout = ffmpi->audio_frame->channel_layout;
		if(visa_frame->channel_layout == 0)
			visa_frame->channel_layout = av_get_default_channel_layout(channels);
#endif
		visa_frame->pts = ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO];

		if((retVal = av_frame_get_buffer(visa_frame, bytes_per_sample)) == 0)
		{
			int ch;
			sample_blocksize *= bytes_per_sample; // in bytes from here
			if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_PLANARAUDIO))
			{
				for(ch = 0; (ch < channels) && ffmpi->audio_frame->data[ch]; ch++)
				{
					pds_memcpy(visa_frame->data[ch], ffmpi->audio_frame->data[ch] + ffmpi->audioframe_copy_samplepos * bytes_per_sample, sample_blocksize);
					visa_frame->linesize[ch] = sample_blocksize;
				}
			}
			else
			{
				sample_blocksize *= channels;
				pds_memcpy(visa_frame->data[0], ffmpi->audio_frame->data[0] + ffmpi->audioframe_copy_samplepos * channels * bytes_per_sample, sample_blocksize);
				visa_frame->linesize[0] = sample_blocksize;
			}
			mpxplay_ffmpgdec_packetqueue_put(&ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO], MPXPLAY_PACKETTYPE_AVFRAME_AUDIO, 0, pktlist_ctx->codec_ctx, pktlist_ctx->codecctx_counter, pktlist_ctx->avstream, visa_frame); // FIXME: dirty hack
		}
		mpxplay_debugf(MPXPLAY_DEBUG_ADECODE,"inffmpgdec_decodeaudio_audiovisualization_prepare PUT np:%d is:%d os:%d pts:%lld diff:%lld ret:%d if:%d of:%d", ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets, ffmpi->audio_frame->nb_samples,
				visa_frame->nb_samples, visa_frame->pts, (ffmpi->avsync_timestamp + pds_gettimeu() - ffmpi->avsync_clocktime - visa_frame->pts),
				retVal, ffmpi->audio_frame->format, visa_frame->format);
	}
}

int in_ffmpgdec_decode_audio(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_infile_info_s *miis)
{
	struct mpxplay_streampacket_info_s *spi = miis->audio_stream;
	int retcode_audio = 0, retry, streamtype_index = MPXPLAY_STREAMTYPEINDEX_AUDIO;
	AVCodecContext *audio_ctx;

	if(spi->bs_leftbytes){
		mpxplay_debugf(MPXPLAY_DEBUG_ADECODE,"inffmp_decode_audio LEFT: %d sy: %llu", spi->bs_leftbytes, ffmpi->avsync_timestamp);
		return retcode_audio;
	}

	retry = INFFMPG_QUEUE_PACKETNUM_FILL_STEP * 2;
	do{
		mpxplay_packetlist_t *pktlist_ctx = NULL;
		int retry_audiodec = 3;
		audio_ctx = NULL;

		if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD) && !funcbit_test(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE))
			in_ffmpdmux_read_packets(ffmpi);

#ifdef INFFMP_USE_DEMUXER_AUINIT
		if(mpxplay_ffmpgdec_packetqueue_get((mpxp_ptrsize_t)ffmpi, &ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO], &pktlist_ctx, 0) != MPXPLAY_ERROR_INFILE_OK)
#else
		if(in_ffmpgdec_infile_callback((mpxp_ptrsize_t)ffmpi, MPXPLAY_INFILE_CBKCTRL_GET_CODECCTXINIT_FFMPEG, (mpxp_ptrsize_t)&pktlist_ctx, streamtype_index, INFFMPG_MUTEXTIME_NOLOCK) != MPXPLAY_ERROR_INFILE_OK)
#endif
		{
			mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "in_ffmpgdec_decode_audio 1. pktlist_ctx:%8.8X ctx:%8.8X", (mpxp_ptrsize_t)pktlist_ctx, (pktlist_ctx)? (mpxp_ptrsize_t)pktlist_ctx->codec_ctx : 0);
			retcode_audio = AVERROR(ESPIPE);
			break;
		}

		if(!pktlist_ctx || !inffmpgdec_is_audio_ctx_valid((AVCodecContext *)pktlist_ctx->codec_ctx)){
			mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "in_ffmpgdec_decode_audio 2. pktlist_ctx:%8.8X ctx:%8.8X", (mpxp_ptrsize_t)pktlist_ctx, (pktlist_ctx)? (mpxp_ptrsize_t)pktlist_ctx->codec_ctx : 0);
			retcode_audio = AVERROR(ESPIPE);
			break;
		}

		audio_ctx = pktlist_ctx->codec_ctx;

		if((retcode_audio = inffmpgdec_decodeaudio_clearbuf(ffmpi, pktlist_ctx, audio_ctx)) != 0)
			break;

		do{
			const mpxp_int64_t avsync_vidtimestamp = ffmpi->avsync_videotimestamp;
#ifdef INFFMP_USE_VIDEO_SYNC
			if((retcode_audio = inffmpgdec_decodeaudio_silentblock_by_vsync(ffmpi, spi, avsync_vidtimestamp)) != 0)
			{
				retry = 1;
				break;
			}
#endif
			retcode_audio = inffmpgdec_decodeaudio_receive_next_frame(ffmpi, miis, audio_ctx);
			if(retcode_audio == 0)
			{
				int left_samples = ffmpi->audioframe_samples_per_channel - ffmpi->audioframe_copy_samplepos;
				int sample_blocksize = ((left_samples > 0) && (left_samples <= ffmpi->audioframe_copy_sampleblocksize))? left_samples : ffmpi->audioframe_copy_sampleblocksize;
				int sample_rate = (ffmpi->audio_frame->sample_rate > 0)? ffmpi->audio_frame->sample_rate : audio_ctx->sample_rate;
#if MPXPLAY_USE_FFMPEG_V7X
				int channels = (ffmpi->audio_frame->ch_layout.nb_channels > 0)? ffmpi->audio_frame->ch_layout.nb_channels : audio_ctx->ch_layout.nb_channels;
#else
				int channels = (ffmpi->audio_frame->ch_layout.nb_channels > 0)? ffmpi->audio_frame->ch_layout.nb_channels : ((ffmpi->audio_frame->channels > 0)? ffmpi->audio_frame->channels :  audio_ctx->channels);
#endif
				int bytes_per_sample = ffmpi->outsample_size;
				if((sample_blocksize > 0) && (sample_rate > 0) && (channels > 0) && (ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] >= 0))
				{
					inffmpgdec_decodeaudio_copy_audio(ffmpi, miis, avsync_vidtimestamp, sample_blocksize, channels, bytes_per_sample);
					inffmpgdec_decodeaudio_correct_currtimestamp(ffmpi, spi, sample_rate);
					inffmpgdec_decodeaudio_audiovisualization_prepare(ffmpi, pktlist_ctx, sample_rate, sample_blocksize, channels, bytes_per_sample);
					ffmpi->audioframe_copy_samplepos += sample_blocksize;
					ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] += (int64_t)AV_TIME_BASE * (int64_t)sample_blocksize / (int64_t)sample_rate;
				}
				else
				{
					av_frame_unref(ffmpi->audio_frame);
					ffmpi->audioframe_copy_sampleblocksize = 0;
				}
				if(spi->bs_leftbytes && !funcbit_test(spi->flags,MPXPLAY_SPI_FLAG_SILENTBLOCK))
				{
					retry = 1;
					break;
				}
			} else if(retcode_audio == AVERROR(EAGAIN)){
				ffmpi->audioframe_copy_sampleblocksize = 0;
				if((retcode_audio = inffmpgdec_decodeaudio_load_next_packet(ffmpi, miis, pktlist_ctx)) != 0)
				{
					retry = 1;
					break;
				}
			} else {
				if(inffmpgdec_is_audio_ctx_valid(audio_ctx))
					avcodec_flush_buffers(audio_ctx);
				mpxplay_debugf(MPXPLAY_DEBUG_ERROR,"inffmp_decode_audio ERR:%d ffmpi:%8.8X ctx:%8.8X avframe:%8.8X",
						retcode_audio, (mpxp_ptrsize_t)ffmpi, (mpxp_ptrsize_t)audio_ctx, (mpxp_ptrsize_t)ffmpi->audio_frame);
				retcode_audio = AVERROR(ESPIPE);
				retry = 1;
				break;
			}
		}while(--retry_audiodec);
	}while(--retry);

	if(audio_ctx)
	{
		in_ffmpgdec_infile_callback((mpxp_ptrsize_t)ffmpi, MPXPLAY_INFILE_CBKCTRL_SET_CODECCTX_CONSOLIDATE, (mpxp_ptrsize_t)audio_ctx, streamtype_index, INFFMPG_MUTEXTIME_NOLOCK);
	}

	mpxplay_debugf(MPXPLAY_DEBUG_ADECODE, "inffmp_decode_audio END %8.8X frd:%d ra:%d lb:%d r:%d nba:%d nbv:%d ptsl:%lld", (mpxp_ptrsize_t)ffmpi,
		ffmpi->retcode_demux, retcode_audio, spi->bs_leftbytes, retry, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets,
		ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets, ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO]);
	return retcode_audio;
}

#endif //MPXPLAY_LINK_INFILE_FF_MPEG
