//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: secondary/parallel native MPEG-TS demuxing to collect DVB EPG informations

//#define MPXPLAY_USE_DEBUGF 1
#define MPXPLAY_DEBUGOUT_HEAD NULL // stderr
#define MPXPLAY_DEBUGOUT_SECT NULL // stdout
#define MPXPLAY_DEBUGOUT_SDT NULL // stdout
#define MPXPLAY_DEBUGOUT_NIT NULL // stdout
#define MPXPLAY_DEBUGOUT_EIT NULL // stdout
#define MPXPLAY_DEBUGOUT_EITCURR NULL // stdout
#define MPXPLAY_DEBUGOUT_EITEXT NULL // stdout
#define MPXPLAY_DEBUGOUT_POS NULL // stdout
#define MPXPLAY_DEBUGOUT_APP NULL // stdout
#define MPXPLAY_DEBUGOUT_WARNING NULL // stderr
#define MPXPLAY_DEBUGOUT_ERROR NULL // stderr

#include "ffmpgdec.h"
#include "libavutil/crc.h"
#include "diskdriv/diskdriv.h"
#include "diskdriv/dtv_drv.h"

#if defined(MPXPLAY_LINK_INFILE_FF_MPEG)

#define INFFMPEPG_TS_SECTION_MAX_SIZE  4096 // by TS standard (EIT)

#define INFFMPEPG_TS_STREAMBUF_SIZE    262144
#define INFFMPEPG_TS_PROBE_SIZE        16384
#define INFFMPEPG_TS_SECTIONBUF_SIZE   (INFFMPEPG_TS_SECTION_MAX_SIZE * 2)

#define INFFMPEPG_TS_BASE_FRAME_SIZE  188
#define INFFMPEPG_TS_DVHS_FRAME_SIZE  192
#define INFFMPEPG_TS_FEC_FRAME_SIZE   204
#define INFFMPEPG_TS_MAX_FRAME_SIZE   INFFMPEPG_TS_FEC_FRAME_SIZE

#define INFFMPEPG_MPEGTS_PID_NIT   0x0010  // network information table id
#define INFFMPEPG_MPEGTS_PID_SDT   0x0011  // service description table id
#define INFFMPEPG_MPEGTS_PID_EIT   0x0012  // event information table id
#define INFFMPEPG_MPEGTS_PID_TDT   0x0014  // time and date table id (TDT, TOT)

#define INFFMPEPG_MPEGTS_TID_NIT   0x40    // network_information_section - actual_network
#define INFFMPEPG_MPEGTS_TID_SDT   0x42    // service_description_section - actual_transport_stream

typedef struct ffmp_epg_section_header_s {
	uint8_t  table_id;
	uint16_t section_length;
	uint16_t transport_stream_id;
	uint8_t  version_number;
	uint8_t  section_number;
	uint8_t  last_section_number;
	const uint8_t *section_data_beginpos;   // in section_data->section_data_buffer (assuming that the buffer doesn't change while using this point)
	uint32_t section_crc_value;
} ffmp_epg_section_header_s;

extern AVInputFormat ff_mpegts_demuxer;

// playlist/textconv.c function
extern int mpxplay_playlist_textconv_by_cpsrcname(char *cp_src_name,char *src_string,int src_len,char *dest_string,unsigned int dest_buflen);

// local static functions
static void inffmp_epg_mpegts_store_data(struct ffmpg_epg_info_s *epg_infos, uint8_t *dataptr, int datalen);
static void inffmp_epg_mpegts_handle_frame(struct ffmpg_epg_info_s *epg_infos);
static void inffmp_epg_process_section_sdt(struct ffmpg_epg_info_s *epg_infos);
static void inffmp_epg_process_section_nit(struct ffmpg_epg_info_s *epg_infos);
static void inffmp_epg_process_section_eit(struct ffmpg_epg_info_s *epg_infos);
static void inffmp_epg_process_section_tdt(struct ffmpg_epg_info_s *epg_infos);
static void inffmp_epg_update_application(struct ffmpg_epg_info_s *epg_infos, int program_id, struct mpxplay_dvb_epg_event_data_s *event_data);

//-----------------------------------------------------------------------------------------------------------------------------------------
// called from ffmpfile

void in_ffmp_epg_mpegts_demux_extfile(struct ffmpeg_external_files_info_s *extfilehandler, uint8_t *dataptr, int datalen)
{
	if(!extfilehandler)
		return;
	if(funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_LOADHEADONLY))
		return;
	if(extfilehandler->external_file_avctx && (extfilehandler->external_file_avctx->iformat != &ff_mpegts_demuxer))
		return;

	in_ffmp_epg_mpegts_demux_epginfo(&extfilehandler->epg_infos, dataptr, datalen);
}

void in_ffmp_epg_mpegts_clearbuf_extfile(struct ffmpeg_external_files_info_s *extfilehandler)
{
	if(!extfilehandler)
		return;
	if(extfilehandler->external_file_avctx && (extfilehandler->external_file_avctx->iformat != &ff_mpegts_demuxer))
		return;
	in_ffmp_epg_mpegts_clearbuf_epginfo(&extfilehandler->epg_infos);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// called from drv_dtv

void in_ffmp_epg_mpegts_demux_epginfo(struct ffmpg_epg_info_s *epg_infos, uint8_t *dataptr, int datalen)
{
	if(!dataptr || (datalen <= 0))
		return;

	inffmp_epg_mpegts_store_data(epg_infos, dataptr, datalen);
	while(mpxplay_bitstream_leftbytes(epg_infos->mpegts_data.stream_buffer) > INFFMPEPG_TS_PROBE_SIZE)
	{
		inffmp_epg_mpegts_handle_frame(epg_infos);
	}
}

void in_ffmp_epg_mpegts_clearbuf_epginfo(struct ffmpg_epg_info_s *epg_infos)
{
	if(!epg_infos)
		return;
	epg_infos->carrier_localized_datetime = 0;
	epg_infos->mpegts_data.mpegts_frame_size = 0;
	mpxplay_bitstream_reset(epg_infos->mpegts_data.stream_buffer);
	mpxplay_bitstream_reset(epg_infos->section_nit_data.section_data_buffer);
	mpxplay_bitstream_reset(epg_infos->section_sdt_data.section_data_buffer);
	mpxplay_bitstream_reset(epg_infos->section_eit_data.section_data_buffer);
	mpxplay_bitstream_reset(epg_infos->section_tdt_data.section_data_buffer);
	epg_infos->section_nit_data.last_continuity_counter = -2;
	epg_infos->section_sdt_data.last_continuity_counter = -2;
	epg_infos->section_eit_data.last_continuity_counter = -2;
	epg_infos->section_tdt_data.last_continuity_counter = -2;
	epg_infos->section_nit_data.section_started = FALSE;
	epg_infos->section_sdt_data.section_started = FALSE;
	epg_infos->section_eit_data.section_started = FALSE;
	epg_infos->section_tdt_data.section_started = FALSE;
	mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"CLEAR in_ffmp_epg_mpegts_clearbuf_epginfo");
}

// dealloc all epg datas related to the epg_infos
void in_ffmp_epg_mpegts_close(struct ffmpg_epg_info_s *epg_infos)
{
	struct mpxplay_dvb_epg_program_data_s *prog_data;

	if(!epg_infos)
		return;

	mpxplay_bitstream_free(epg_infos->mpegts_data.stream_buffer);
	mpxplay_bitstream_free(epg_infos->section_nit_data.section_data_buffer);
	mpxplay_bitstream_free(epg_infos->section_sdt_data.section_data_buffer);
	mpxplay_bitstream_free(epg_infos->section_eit_data.section_data_buffer);
	mpxplay_bitstream_free(epg_infos->section_tdt_data.section_data_buffer);

	prog_data = epg_infos->epg_program_data_chain;

	pds_memset(epg_infos, 0, sizeof(*epg_infos));

	mpxplay_dtvdrv_database_clear_programs(prog_data);
}

//---------------------------------------------------------------------------------------------------------------------------------------------
static void inffmp_epg_mpegts_store_data(struct ffmpg_epg_info_s *epg_infos, uint8_t *dataptr, int datalen)
{
	struct ffmpg_epg_mpegts_data_s *mpegts_infos = &epg_infos->mpegts_data;
	if(!mpegts_infos->stream_buffer)
	{
		mpegts_infos->stream_buffer = mpxplay_bitstream_alloc(INFFMPEPG_TS_STREAMBUF_SIZE);
		if(!mpegts_infos->stream_buffer)
			return;
		epg_infos->section_nit_data.last_continuity_counter = -2;
		epg_infos->section_sdt_data.last_continuity_counter = -2;
		epg_infos->section_eit_data.last_continuity_counter = -2;
		epg_infos->section_tdt_data.last_continuity_counter = -2;
	}
	if(mpxplay_bitstream_putbytes(mpegts_infos->stream_buffer, dataptr, datalen) != datalen)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_ERROR, "BUFFER bitstream_putbytes failed %d", datalen);
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------------
static int inffmp_epg_mpegts_frame_size_analyze(unsigned char *buf, const unsigned int bufsize, const unsigned int packet_analyze_size)
{
	unsigned int i, best_score = 0, stat[INFFMPEPG_TS_MAX_FRAME_SIZE];

	pds_memset(stat, 0, sizeof(stat));

	for (i = 0; i < bufsize; i++, buf++)
	{
		if (buf[0] == 0x47)
		{
			unsigned int x = i % packet_analyze_size;
			stat[x]++;
			if (best_score < stat[x])
				best_score = stat[x];
		}
	}

	return best_score;
}

static mpxp_bool_t inffmp_epg_mpegts_frame_size_detect(struct ffmpg_epg_info_s *epg_infos)
{
	const unsigned char frame_probe_sizes[3] = {INFFMPEPG_TS_BASE_FRAME_SIZE, INFFMPEPG_TS_DVHS_FRAME_SIZE, INFFMPEPG_TS_FEC_FRAME_SIZE};
	const struct ffmpg_epg_mpegts_data_s *mpegts_infos = &epg_infos->mpegts_data;
	struct mpxplay_bitstreambuf_s *bs = mpegts_infos->stream_buffer;
	unsigned char *buffer_headpos = mpxplay_bitstream_getbufpos(bs);
	const long buffer_leftbytes = mpxplay_bitstream_leftbytes(bs);
	int i, best_frame_size = 0, best_score = 0;

	if(buffer_leftbytes < INFFMPEPG_TS_PROBE_SIZE)
		return FALSE;

	for(i = 0; i < sizeof(frame_probe_sizes); i++)
	{
		const unsigned int packet_size = frame_probe_sizes[i];
		const unsigned int score = inffmp_epg_mpegts_frame_size_analyze(buffer_headpos, buffer_leftbytes, packet_size);
		if(score > best_score)
		{
			best_score = score;
			best_frame_size = packet_size;
		}

	}

	if(best_score >= (buffer_leftbytes / INFFMPEPG_TS_MAX_FRAME_SIZE / 2))
	{
		epg_infos->mpegts_data.mpegts_frame_size = best_frame_size;
		return TRUE;
	}

	return FALSE;
}

static mpxp_bool_t inffmp_epg_mpegts_sync_head(struct ffmpg_epg_info_s *epg_infos)
{
	struct ffmpg_epg_mpegts_data_s *mpegts_infos = &epg_infos->mpegts_data;
	struct mpxplay_bitstreambuf_s *bs = mpegts_infos->stream_buffer;
	unsigned int skipped_bytes = 0;
	mpxp_bool_t sync_result = TRUE;

	while(mpxplay_bitstream_leftbytes(bs) > INFFMPEPG_TS_MAX_FRAME_SIZE)
	{
		if(mpxplay_bitstream_get_byte(bs) == 0x47)
		{
			mpxplay_bitstream_skipbytes(bs, -1);
			break;
		}
		sync_result = FALSE;
		skipped_bytes++;
	}

	if(!sync_result || !epg_infos->mpegts_data.mpegts_frame_size)
	{
		sync_result = inffmp_epg_mpegts_frame_size_detect(epg_infos);
		mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"RESYNC res:%d skip:%d fs:%d bs:%d", (int)sync_result, skipped_bytes, epg_infos->mpegts_data.mpegts_frame_size, mpxplay_bitstream_leftbytes(bs));
	}
	return sync_result;
}

//---------------------------------------------------------------------------------------------------------------------------------------------
#define INFFMPG_EPG_AFC_PAYLOAD    (1 << 0)
#define INFFMPG_EPG_AFC_ADAPTFIELD (1 << 1)

static void inffmp_epg_mpegts_handle_frame(struct ffmpg_epg_info_s *epg_infos)
{
	struct ffmpg_epg_mpegts_data_s *mpegts_infos = &epg_infos->mpegts_data;
	struct mpxplay_bitstreambuf_s *bs = mpegts_infos->stream_buffer;
	struct ffmpg_epg_section_data_s *section_data;
	int val, pid_num, pid_start_indicator, transport_scrambling_control, adaption_field_control, continuity_counter, adaptation_field_length = 0, payload_size = 0;
	int start_bytespos;

	if(!inffmp_epg_mpegts_sync_head(epg_infos))
		return;
	if(mpxplay_bitstream_leftbytes(bs) < INFFMPEPG_TS_MAX_FRAME_SIZE)
		return;
	start_bytespos = mpxplay_bitstream_leftbytes(bs);

	val = mpxplay_bitstream_get_byte(bs);
	if(val != 0x47) // sync head byte of TS frame
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"HEAD sync byte error");
		return;
	}
	val = mpxplay_bitstream_get_be16(bs);

	if(val & 0x8000) // Transport error indicator (TEI)
	{
		mpxplay_bitstream_skipbytes(bs, epg_infos->mpegts_data.mpegts_frame_size - 3);
		mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"HEAD transport error skip: %d", epg_infos->mpegts_data.mpegts_frame_size - 3);
		return;
	}

	pid_num = val & 0x1FFF;
	switch(pid_num)
	{
		case INFFMPEPG_MPEGTS_PID_NIT: section_data = &epg_infos->section_nit_data; break;
		case INFFMPEPG_MPEGTS_PID_SDT: section_data = &epg_infos->section_sdt_data; break;
		case INFFMPEPG_MPEGTS_PID_EIT: section_data = &epg_infos->section_eit_data; break;
		case INFFMPEPG_MPEGTS_PID_TDT: section_data = &epg_infos->section_tdt_data; break;
		default: mpxplay_bitstream_skipbytes(bs, epg_infos->mpegts_data.mpegts_frame_size - 3); return; // we don't handle this section type, drop TS frame
	}

	pid_start_indicator = val & 0x4000; // section data begins (previous has finished)

	val = mpxplay_bitstream_get_byte(bs);
	transport_scrambling_control = val >> 6;
	adaption_field_control = (val & 0x30) >> 4;
	continuity_counter = val & 0x0F;

	payload_size = epg_infos->mpegts_data.mpegts_frame_size - 4;

	// check existence of payload in this TS frame
	if(!funcbit_test(adaption_field_control, INFFMPG_EPG_AFC_PAYLOAD))
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"HEAD no payload");
		goto err_out_handle_packet;
	}

	// check continuity counter of section
	if((section_data->last_continuity_counter >= 0) && (((section_data->last_continuity_counter + 1) & 0x0F) != continuity_counter)
	){
		mpxplay_bitstream_reset(section_data->section_data_buffer);
		section_data->section_started = FALSE;
		if(pid_num == INFFMPEPG_MPEGTS_PID_SDT)
		{
			mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"HEAD continuity error PID:%2.2X curr:%d last:%d", pid_num, continuity_counter, section_data->last_continuity_counter);
		}
		section_data->last_continuity_counter = continuity_counter;
		goto err_out_handle_packet;
	}
	section_data->last_continuity_counter = continuity_counter;

	// skip adaption field
	if(funcbit_test(adaption_field_control, INFFMPG_EPG_AFC_ADAPTFIELD))
	{
		adaptation_field_length = mpxplay_bitstream_get_byte(bs);
		payload_size--;
		if(adaptation_field_length >= payload_size)
		{
			mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"HEAD adaptation_field_length >= payload_size");
			goto err_out_handle_packet;
		}

		mpxplay_bitstream_skipbytes(bs, adaptation_field_length);
		payload_size -= adaptation_field_length;
	}

	// process section data (if it's complete: pid_started)
	if(pid_start_indicator)
	{
		val = mpxplay_bitstream_get_byte(bs);  // pointer_field present at pid_start_indicator
		payload_size--;
		if(val > payload_size)
		{
			mpxplay_debugf(MPXPLAY_DEBUGOUT_HEAD,"HEAD pointer_field:%d > payload_size:%d", val, payload_size);
			goto err_out_handle_packet;
		}

		if(section_data->section_started)
		{
			mpxplay_bitstream_putbytes(section_data->section_data_buffer, mpxplay_bitstream_getbufpos(bs), val);
			switch(pid_num)
			{
				case INFFMPEPG_MPEGTS_PID_NIT: inffmp_epg_process_section_nit(epg_infos); break;
				case INFFMPEPG_MPEGTS_PID_SDT: inffmp_epg_process_section_sdt(epg_infos); break;
				case INFFMPEPG_MPEGTS_PID_EIT: inffmp_epg_process_section_eit(epg_infos); break;
				case INFFMPEPG_MPEGTS_PID_TDT: inffmp_epg_process_section_tdt(epg_infos); break;
			}
		}
		else
		{
			mpxplay_bitstream_reset(section_data->section_data_buffer); // clear section buffer (this is not really needed here, buffer is probably empty)
		}
		mpxplay_bitstream_skipbytes(bs, val);
		payload_size -= val;
		section_data->section_started = TRUE;
	}

	// Store payload data, if pid has a start indicator (data shall be collected from the pid_start_indicator border only)
	if(section_data->section_started)
	{
		if(!section_data->section_data_buffer)
			section_data->section_data_buffer = mpxplay_bitstream_alloc(INFFMPEPG_TS_SECTIONBUF_SIZE);
		mpxplay_bitstream_putbytes(section_data->section_data_buffer, mpxplay_bitstream_getbufpos(bs), payload_size);
	}

err_out_handle_packet:
	mpxplay_bitstream_skipbytes(bs, payload_size);
#ifdef MPXPLAY_USE_DEBUGF
	if((start_bytespos - mpxplay_bitstream_leftbytes(bs)) != epg_infos->mpegts_data.mpegts_frame_size)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_ERROR,"HEAD END ps:%d ub:%d", epg_infos->mpegts_data.mpegts_frame_size, (start_bytespos - mpxplay_bitstream_leftbytes(bs)));
	}
#endif
}

//---------------------------------------------------------------------------------------------------------------------------------------------
/* create new program entry in the EPG list by service_id and service_name (optional); if entry already exists the function gives back pointer to the existent program */
static struct mpxplay_dvb_epg_program_data_s *inffmp_epg_program_new(struct ffmpg_epg_info_s *epg_infos, mpxp_int32_t service_id, char *service_name, mpxp_uint32_t ctrl_flags, mpxp_bool_t search_only)
{
	struct mpxplay_dvb_epg_program_data_s *prog_data = epg_infos->epg_program_data_chain;

	if(!prog_data && !search_only) // no program is allocated yet
	{
		prog_data = pds_calloc(1, sizeof(*prog_data));
		if(prog_data)
		{
			prog_data->program_id = service_id;
			epg_infos->epg_program_data_chain = prog_data;
			epg_infos->nb_program_datas = 1;
		}
	}
	else // search for program_id
	{
		struct mpxplay_dvb_epg_program_data_s *prog_prev = NULL;
		while(prog_data)
		{
			struct mpxplay_dvb_epg_program_data_s *prog_next;

			if(prog_data->program_id == service_id)
			{
				break; // found existent program
			}

			if((prog_data->program_id > service_id) && !search_only) // insert program into the chain (sort the chain by service_id)
			{
				struct mpxplay_dvb_epg_program_data_s *prog_new = pds_calloc(1, sizeof(*prog_new));
				if(prog_prev)
				{
					prog_prev->next_program = prog_new;
				}
				else
				{
					epg_infos->epg_program_data_chain = prog_new;
				}
				prog_new->program_id = service_id;
				prog_new->next_program = prog_data;
				prog_data = prog_new;
				epg_infos->nb_program_datas++;
				break;
			}

			prog_next = prog_data->next_program;
			if(!prog_next && !search_only) // add new program to the end of the chain
			{
				struct mpxplay_dvb_epg_program_data_s *prog_new = pds_calloc(1, sizeof(*prog_data));
				if(prog_new)
				{
					prog_new->program_id = service_id;
					prog_data->next_program = prog_new;
					prog_data = prog_new;
					epg_infos->nb_program_datas++;
				}
				break;
			}

			prog_prev = prog_data;
			prog_data = prog_next;
		}
	}

	if(service_name && prog_data && !prog_data->program_name) // save program (service) name
	{
		int name_len = pds_strlen(service_name);
		if(name_len > 0)
		{
			prog_data->program_name = pds_malloc(name_len + 1);
			if(prog_data->program_name)
				pds_memcpy(prog_data->program_name, service_name, name_len + 1);
			prog_data->ctrl_flags = ctrl_flags;
		}
	}

	// send program information to the higher level application (and to the DTV driver)
	if(epg_infos->control_cb && prog_data && prog_data->program_name && !funcbit_test(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_PROGNAME_SENT))
	{
		if(epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGRAMNAME, &prog_data->program_id, prog_data->program_name) >= 0) // we send this only once
		{
			if(prog_data->channel_id)
				if(epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGRAMCHANID, &prog_data->program_id, &prog_data->channel_id) >= 0)
					funcbit_enable(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_CHANID_SENT);
			epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGRAMCTRLFLGS, &prog_data->program_id, &prog_data->ctrl_flags);
			funcbit_enable(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_PROGNAME_SENT);
		}
	}

	return prog_data;
}
/* create new event entry for the program by date, begin_time and duration; if entry already exists the function gives back pointer to the existent event */
static struct mpxplay_dvb_epg_event_data_s *inffmp_epg_event_new(struct ffmpg_epg_info_s *epg_infos, mpxp_int32_t service_id, mpxp_uint32_t event_id, mpxp_uint64_t event_start_local_datetime, mpxp_uint32_t event_duration)
{
	struct mpxplay_dvb_epg_program_data_s *prog_data = inffmp_epg_program_new(epg_infos, service_id, NULL, 0, FALSE);
	return mpxplay_dtvdrv_database_event_add(prog_data, event_id, event_start_local_datetime, event_duration, TRUE);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
#define INFFMP_EPG_SECTION_MIN_SIZE   16 // valid for SDT, EIT only

// process section header
static int inffmp_epg_read_section_header(struct ffmp_epg_section_header_s *h, struct mpxplay_bitstreambuf_s *bs)
{
	if(mpxplay_bitstream_leftbytes(bs) < INFFMP_EPG_SECTION_MIN_SIZE)
		return -1;

	h->section_data_beginpos = mpxplay_bitstream_getbufpos(bs);
	h->table_id = mpxplay_bitstream_get_byte(bs);
	h->section_length = mpxplay_bitstream_get_be16(bs) & 0x0fff;      // section_syntax_indicator (1), reserved_future_use (1), reserved (2), section_length (12)
	if((h->section_length < (INFFMP_EPG_SECTION_MIN_SIZE - 3)) || (h->section_length > mpxplay_bitstream_leftbytes(bs)))
		return -1;
	h->section_length += 3;
	h->section_crc_value = PDS_GETB_BE32(h->section_data_beginpos + h->section_length - 4);
	h->transport_stream_id = mpxplay_bitstream_get_be16(bs);          // SDT: transport_stream_id, EIT: service_id, NIT: network id
	h->version_number = (mpxplay_bitstream_get_byte(bs) >> 1) & 0x1f; // reserved (2), version_number (5), current_next_indicator (1)
	h->section_number = mpxplay_bitstream_get_byte(bs);
	h->last_section_number = mpxplay_bitstream_get_byte(bs);

	return 0;
}

// skip section if it's previously processed
static mpxp_bool_t inffmp_epg_section_skip_processed(const struct ffmp_epg_section_header_s *h, struct ffmpg_epg_section_data_s *section_data)
{
	if((h->version_number == section_data->last_version_number) && (h->section_crc_value != 0) && (h->section_crc_value == section_data->last_crc_value))
		return TRUE;

	section_data->last_version_number = h->version_number;
	section_data->last_crc_value = h->section_crc_value;

	return FALSE;
}

// check section crc checksum
static int inffmp_epg_section_check_crc(struct ffmp_epg_section_header_s *h)
{
	const AVCRC *crc_table = av_crc_get_table(AV_CRC_32_IEEE);
	const uint32_t checksum = av_crc(crc_table, -1, h->section_data_beginpos, h->section_length);

	if(checksum)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"SECTION CRC mismatch len:%4d %8.8X %8.8X", h->section_length, h->section_crc_value, checksum);
		return -1;
	}

	return 0;
}

// read (common) header datas of a section (NIT, SDT, EIT, etc.)
static mpxp_bool_t inffmp_epg_section_common_header_read(struct ffmpg_epg_section_data_s *section_data, mpxp_uint8_t table_id)
{
	struct mpxplay_bitstreambuf_s *bs = section_data->section_data_buffer;
	struct ffmp_epg_section_header_s sect_head_data;

	mpxplay_debugf(MPXPLAY_DEBUGOUT_SECT,"HEAD SECTION %2.2X START leftbytes:%d", table_id, mpxplay_bitstream_leftbytes(bs));

	if(inffmp_epg_read_section_header(&sect_head_data, bs) < 0)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"HEAD SECTION %2.2X read_section_header failed", table_id);
		goto err_out_section;
	}
	if(sect_head_data.table_id != table_id)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_SECT,"HEAD SECTION mismatch reqTID:%d currTID:", table_id, sect_head_data.table_id);
		goto err_out_section;
	}
	if(inffmp_epg_section_skip_processed(&sect_head_data, section_data))
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_SECT,"HEAD SECTION %2.2X PROCESSED", table_id);
		goto err_out_section;
	}
	if(inffmp_epg_section_check_crc(&sect_head_data) < 0)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"HEAD SECTION %2.2X CRC failed", table_id);
		section_data->last_crc_value = 0;
		goto err_out_section;
	}

	return TRUE;

err_out_section:
	mpxplay_bitstream_reset(bs); // wipe out unused data from the section buffer
	return FALSE;
}

// read and allocate a DVB string preceded by its length
static char *inffmp_epg_read_string_descriptor(struct mpxplay_bitstreambuf_s *bs, mpxp_bool_t allow_short_desc)
{
	const char *cp_encoding_table[32] = { // TODO: some CPs are not supported by mpxplay_playlist_textconv (eg. ISO6937, KSC_5601, GB2312)
		"ISO6937", "ISO-8859-5", "ISO-8859-6", "ISO-8859-7", "ISO-8859-8", "ISO-8859-9", "ISO-8859-10", "ISO-8859-11",
		"", "ISO-8859-13", "ISO-8859-14", "ISO-8859-15", "", "", "", "","", "UCS-2BE", "KSC_5601", "GB2312", "UCS-2BE",
		"UTF-8", "", "", "", "", "", "", "", "", "", "" };
	unsigned int desclen, indatalen, outbuflen, outdatalen;
	char *str_in, *stroutbuf;

	desclen = indatalen = mpxplay_bitstream_get_byte(bs);
	if(!indatalen || (!allow_short_desc && (indatalen < 2))) // assumed incorrect length
		return NULL;
	if(indatalen > mpxplay_bitstream_leftbytes(bs))
		return NULL;
	str_in = mpxplay_bitstream_getbufpos(bs);

	outbuflen = indatalen * 3 + 1; // enough space for UTF-8 output
	stroutbuf = pds_malloc(outbuflen);
	if(!stroutbuf)
		return NULL;

	if((indatalen >= 3) && (str_in[0] == 0x10) && (!str_in[1]) && str_in[2] && (str_in[2] <= 15) && (str_in[2] != 12))
	{
		char iso8859_strname[12];
		snprintf(iso8859_strname, sizeof(iso8859_strname), "ISO-8859-%d", str_in[2]);
		str_in += 3;
		indatalen -= 3;
		outdatalen = mpxplay_playlist_textconv_by_cpsrcname(iso8859_strname, str_in, indatalen, stroutbuf, outbuflen);
	}
	else if(str_in[0] < 32)
	{
		indatalen --;
		outdatalen = mpxplay_playlist_textconv_by_cpsrcname((char *)cp_encoding_table[str_in[0]], &str_in[1], indatalen, stroutbuf, outbuflen);
		str_in++;
	}
	else
	{
		outdatalen = mpxplay_playlist_textconv_by_cpsrcname((char *)cp_encoding_table[0], str_in, indatalen, stroutbuf, outbuflen);
	}

	if(outdatalen < 2) // text conversion has failed, use raw data
	{
		if(!allow_short_desc && (indatalen < 2)) // invalid data
		{
			pds_free(stroutbuf);
			stroutbuf = NULL;
		}
		else
		{
			pds_memcpy(stroutbuf, str_in, indatalen);
			stroutbuf[indatalen] = 0;
		}
	}

	mpxplay_bitstream_skipbytes(bs, desclen);

	return stroutbuf;
}

// convert 16 bit MJD date format to 0xYYYYMMDD byte aligned packed integer format
static mpxp_uint32_t inffmp_epg_mjdate_to_hexa(int mjdate)
{
	const int yp = (int)(((double)mjdate - 15078.2)/365.25);
	const int mp = (int)(((double)mjdate - 14956.1 - (int)(yp * 365.25)) / 30.6001);
	const int c = ((mp == 14) || (mp == 15)) ? 1 : 0;
	return ((1900 + yp + c*1) << 16) | ((mp - 1 - c*12) << 8) | (mjdate - 14956 - (int)(yp*365.25) - (int)(mp*30.6001));
}

// convert 0x00HHMMSS BCD to to 0x00HHMMSS byte aligned packed integer format
static mpxp_uint32_t inffmp_epg_bcd_to_hexa(mpxp_uint32_t bcd)
{
	mpxp_uint32_t hours   = ((bcd >> 20) & 0xF) * 10 + ((bcd >> 16) & 0xF);
	mpxp_uint32_t minutes = ((bcd >> 12) & 0xF) * 10 + ((bcd >>  8) & 0xF);
	mpxp_uint32_t seconds = ((bcd >>  4) & 0xF) * 10 + (bcd & 0xF);
	return ((hours << 16) | (minutes << 8) | seconds);
}

// This 40-bit field contains the current time and date in UTC and MJD (see annex C).
// This field is coded as 16 bits giving the 16 LSBs of MJD followed by 24 bits coded as 6 digits in 4-bit BCD.
// Store it in 0xYYYYMMDD00HHMMSS byte aligned packed integer format (64 bits)
static mpxp_uint64_t inffmp_epg_read_datetime(struct mpxplay_bitstreambuf_s *bs)
{
	mpxp_uint32_t date_val, time_val;
	mpxp_uint64_t datetime_val;

	date_val = inffmp_epg_mjdate_to_hexa(mpxplay_bitstream_get_be16(bs));
	time_val = inffmp_epg_bcd_to_hexa((mpxplay_bitstream_get_byte(bs) << 16) | (mpxplay_bitstream_get_byte(bs) << 8) | mpxplay_bitstream_get_byte(bs));
	datetime_val = ((mpxp_uint64_t)date_val << 32) | (mpxp_uint64_t)time_val;

	return datetime_val;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// process collected/buffered section datas

// Network Information Table (NIT)
static void inffmp_epg_process_section_nit(struct ffmpg_epg_info_s *epg_infos)
{
	struct ffmpg_epg_section_data_s *section_data = &epg_infos->section_nit_data;
	struct mpxplay_bitstreambuf_s *bs = section_data->section_data_buffer;
	int desc_list_len;

	if(!inffmp_epg_section_common_header_read(section_data, INFFMPEPG_MPEGTS_TID_NIT))
		return;

	desc_list_len = mpxplay_bitstream_get_be16(bs) & 0x0fff; // reserved_future_use (4), network_descriptors_length (12)

	mpxplay_debugf(MPXPLAY_DEBUGOUT_NIT,"NIT SECTION first len:%d left:%d", desc_list_len, mpxplay_bitstream_leftbytes(bs));

	mpxplay_bitstream_skipbytes(bs, desc_list_len); // we don't process this list

	mpxplay_bitstream_skipbytes(bs, 2);  // reserved_future_use (4), transport_stream_loop_length (12)

	while(mpxplay_bitstream_leftbytes(bs) > 6)
	{
		int desc_list_leftbytes, desc_list_bytecount;

		mpxplay_bitstream_skipbytes(bs, 2); // transport_stream_id
		mpxplay_bitstream_skipbytes(bs, 2); // original_network_id
		desc_list_len = mpxplay_bitstream_get_be16(bs) & 0x0fff; // reserved_future_use (4), transport_descriptors_length (12)

		desc_list_leftbytes = mpxplay_bitstream_leftbytes(bs);
		if (!desc_list_len || (desc_list_len > desc_list_leftbytes))
		{
			mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"NIT desc_list_len small dl:%d lb:%d", desc_list_len, desc_list_leftbytes);
			break;
		}

		mpxplay_debugf(MPXPLAY_DEBUGOUT_NIT,"NIT LIST len:%d left:%d", desc_list_len, mpxplay_bitstream_leftbytes(bs));

		desc_list_bytecount = desc_list_len;
		while((desc_list_bytecount > 0) && (mpxplay_bitstream_leftbytes(bs) > 3))
		{
			int desc_tag = mpxplay_bitstream_get_byte(bs);
			int desc_len = mpxplay_bitstream_get_byte(bs);
			int desc_tag_leftbytes = mpxplay_bitstream_leftbytes(bs);

			mpxplay_debugf(MPXPLAY_DEBUGOUT_NIT,"NIT TAG %2.2X len:%d left:%d bc:%d", desc_tag, desc_len, desc_tag_leftbytes, desc_list_bytecount);

			if(desc_len > desc_tag_leftbytes)
			{
				mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"NIT desc_tag_len invalid dt:%2.2X dl:%d lb:%d", desc_tag, desc_len, mpxplay_bitstream_leftbytes(bs));
				goto err_out_section;
			}

			if((desc_len >= 4)  && (desc_tag == 0x83))
			{
				int i = desc_len >> 2;
				do
				{
					int service_id, logical_channel_id;
					if(mpxplay_bitstream_leftbytes(bs) < 4) // the following descriptor is 4 bytes
					{
						mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"NIT desc_tag_len invalid 2. dt:%2.2X dl:%d lb:%d", desc_tag, desc_len, mpxplay_bitstream_leftbytes(bs));
						goto err_out_section;
					}
					service_id = mpxplay_bitstream_get_be16(bs);
					logical_channel_id = mpxplay_bitstream_get_be16(bs) & 0x03ff;
					if(service_id && logical_channel_id)
					{
						struct ffmpg_demuxer_data_s *ffmpi = (struct ffmpg_demuxer_data_s *)epg_infos->ffmpi;  // the first multiplex may contain the channel ids of all multiplexes at DVB live streams
						mpxp_bool_t search_only = (ffmpi && !funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))? TRUE : FALSE;  // at local file we read only channel ids of this multiplex
						struct mpxplay_dvb_epg_program_data_s *prog_data = inffmp_epg_program_new(epg_infos, service_id, NULL, 0, search_only); // at live streams we read and save all of them, adding other program infos later
						if(prog_data && !funcbit_test(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_CHANID_SENT))
						{
							prog_data->channel_id = logical_channel_id;
							if(epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGRAMCHANID, &prog_data->program_id, &prog_data->channel_id) >= 0)
								funcbit_enable(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_CHANID_SENT);
						}
					}
					mpxplay_debugf(MPXPLAY_DEBUGOUT_NIT,"NIT 0x83 sid:%d chid:%d", service_id, logical_channel_id);
				}while(--i);
			}
			mpxplay_bitstream_skipbytes(bs, (desc_len - (desc_tag_leftbytes - mpxplay_bitstream_leftbytes(bs))));
			desc_list_bytecount -= (desc_len + 2);
		}
		mpxplay_bitstream_skipbytes(bs, (desc_list_len - (desc_list_leftbytes - mpxplay_bitstream_leftbytes(bs))));
	}

err_out_section:
	mpxplay_bitstream_reset(bs); // wipe out unused data from the section buffer
}

// Service Description Table (SDT)
static void inffmp_epg_process_section_sdt(struct ffmpg_epg_info_s *epg_infos)
{
	struct ffmpg_epg_section_data_s *section_data = &epg_infos->section_sdt_data;
	struct mpxplay_bitstreambuf_s *bs = section_data->section_data_buffer;
	int service_id, desc_list_len, desc_list_leftbytes, desc_list_bytecount;
	int desc_tag, desc_len, desc_tag_leftbytes;
	char *provider_name, *program_name;

	if(!inffmp_epg_section_common_header_read(section_data, INFFMPEPG_MPEGTS_TID_SDT))
		return;

	mpxplay_bitstream_skipbytes(bs, 2); // original_network_id
	mpxplay_bitstream_skipbytes(bs, 1); // reserved_future_use

	while(mpxplay_bitstream_leftbytes(bs) >= 8) // minimum 8 bytes : service_id (2), reserved (1), desc_list_len (2), desc_tag (1), desc_len (1), + 1
	{
		mpxp_uint32_t program_ctrlflags = 0, running_status, free_CA_mode;
		service_id = mpxplay_bitstream_get_be16(bs);
		mpxplay_bitstream_skipbytes(bs, 1); // reserved_future_use (6), EIT_schedule_flag (1), EIT_present_following_flag (1)
		desc_list_len = mpxplay_bitstream_get_be16(bs); // running_status (3), free_CA_mode (1), descriptors_loop_length (12)
		running_status = (desc_list_len >> 13) & 0x7;
		free_CA_mode = (desc_list_len >> 12) & 0x1;
		desc_list_len &= 0x0fff;
		desc_list_leftbytes = mpxplay_bitstream_leftbytes(bs);
		if (!desc_list_len || (desc_list_len > desc_list_leftbytes))
		{
			mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"SDT desc_list_len small dl:%d lb:%d", desc_list_len, desc_list_leftbytes);
			break;
		}
		if(!service_id)
			goto skip_desc_list;
		if((running_status < 2) || (running_status > 4)) // running status shall be "start_soon" (2), "paused" (3) or "running" (4), else program is not valid for us
			funcbit_enable(program_ctrlflags, INFFMPG_EPGPRGCTRLFLAG_NOTRUNNING_PRG);
		if(free_CA_mode)
			funcbit_enable(program_ctrlflags, INFFMPG_EPGPRGCTRLFLAG_ENCRYPTED_PRG);
		//mpxplay_debugf(MPXPLAY_DEBUGOUT_SDT,"SDT sec_len:%d list_len:%d sid:%d", h->section_len, desc_list_len, service_id);

		desc_list_bytecount = desc_list_len;
		while((desc_list_bytecount > 0) && (mpxplay_bitstream_leftbytes(bs) > 3))
		{
			desc_tag = mpxplay_bitstream_get_byte(bs);
			desc_len = mpxplay_bitstream_get_byte(bs);
			desc_tag_leftbytes = mpxplay_bitstream_leftbytes(bs);
			if(desc_len > desc_tag_leftbytes)
			{
				mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"SDT desc_tag_len small dt:%2.2X dl:%d lb:%d", desc_tag, desc_len, desc_tag_leftbytes);
				goto err_out_section;
			}

			if(desc_len)
			{
				switch(desc_tag)
				{
					case 0x48:
						if(mpxplay_bitstream_leftbytes(bs) < 3)
							goto err_out_section;
						mpxplay_bitstream_skipbytes(bs, 1); // service_type
						provider_name = inffmp_epg_read_string_descriptor(bs, FALSE);
						program_name = inffmp_epg_read_string_descriptor(bs, TRUE);
						if(program_name)
						{
							inffmp_epg_program_new(epg_infos, service_id, program_name, program_ctrlflags, FALSE);
							mpxplay_debugf(MPXPLAY_DEBUGOUT_SDT,"SDT 0x48 len:%2d sid:%d run:%2.2X CA:%d prg_name:%s", desc_len, service_id, running_status, free_CA_mode, program_name);
							pds_free(program_name);
						}
						pds_free(provider_name);
						break;
					default:
						//mpxplay_debugf(MPXPLAY_DEBUGOUT_SDT,"SDT 0x%2.2X sid:%d len:%d", desc_tag, service_id, desc_len);
						break;
				}
				mpxplay_bitstream_skipbytes(bs, (desc_len - (desc_tag_leftbytes - mpxplay_bitstream_leftbytes(bs))));
			}
			desc_list_bytecount -= (desc_len + 2);
		}
skip_desc_list:
		mpxplay_bitstream_skipbytes(bs, (desc_list_len - (desc_list_leftbytes - mpxplay_bitstream_leftbytes(bs))));
	}

err_out_section:
//#ifdef MPXPLAY_USE_DEBUGF
//	if(mpxplay_bitstream_leftbytes(bs) > 0)
//	{
//		mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"SDT END tid:%d skip bytes:%d", h->table_id, mpxplay_bitstream_leftbytes(bs));
//	}
//#endif
	mpxplay_bitstream_reset(bs); // wipe out unused data from the section buffer
}

// Time and Date table (TDT) and Time Offset Table (TOT)
static void inffmp_epg_process_section_tdt(struct ffmpg_epg_info_s *epg_infos)
{
	struct ffmpg_epg_section_data_s *section_data = &epg_infos->section_tdt_data;
	struct mpxplay_bitstreambuf_s *bs = section_data->section_data_buffer;
	struct ffmp_epg_section_header_s sect_head_data, *h = &sect_head_data;

	h->section_data_beginpos = mpxplay_bitstream_getbufpos(bs);
	h->table_id = mpxplay_bitstream_get_byte(bs);

	if((h->table_id != 0x70) && (h->table_id != 0x73)) // TDT and TOT table handling only
		goto err_out_section;

	h->section_length = mpxplay_bitstream_get_be16(bs) & 0x0fff;  // section_syntax_indicator (1), reserved_future_use (1), reserved (2), section_length (12)
	if((h->section_length < 5) || (h->section_length > mpxplay_bitstream_leftbytes(bs))) // UTC_time is 5 bytes (40 bits)
		goto err_out_section;

	if(h->table_id == 0x73) // TOT has CRC too
	{
		h->section_crc_value = PDS_GETB_BE32(h->section_data_beginpos + h->section_length + 3 - 4); // including section header (3), excluding CRC value (4)
		if(inffmp_epg_section_check_crc(h) < 0)
			goto err_out_section;
	}

	epg_infos->carrier_localized_datetime = pds_utctime_to_localtime(inffmp_epg_read_datetime(bs));
	mpxplay_debugf(MPXPLAY_DEBUGOUT_APP,"TDT END %d", h->table_id);

err_out_section:
	mpxplay_bitstream_reset(bs); // wipe out unused data from the section buffer
}

// Event Information Table (EIT)
static void inffmp_epg_process_section_eit(struct ffmpg_epg_info_s *epg_infos)
{
	struct ffmpg_demuxer_data_s *ffmpi = (struct ffmpg_demuxer_data_s *)epg_infos->ffmpi;
	struct ffmpg_epg_section_data_s *section_data = &epg_infos->section_eit_data;
	struct mpxplay_bitstreambuf_s *bs = section_data->section_data_buffer;
	struct ffmp_epg_section_header_s sect_head_data, *h = &sect_head_data;
	int val, desc_list_len, desc_list_leftbytes, desc_list_bytecount;
	int desc_tag, desc_len, desc_tag_leftbytes;
	mpxp_uint32_t event_duration, event_id, parental_rating_value = 0;
	mpxp_uint64_t event_start_local_datetime;
	char *event_name;

	if(inffmp_epg_read_section_header(h, bs) < 0)
		goto err_out_section;
	if((h->table_id < 0x4E) || (h->table_id == 0x4F) || (h->table_id > 0x5F)) /* 0x4E : actual_transport_stream, present/following;  0x50 - 0x5F - actual_transport_stream, schedule */
		goto err_out_section;
//    if(inffmp_epg_section_skip_processed(h, section_data)) // this is not useful for EIT (we should filter by the table_id and service_id too -> too much saved infos)
//    {
//    	mpxplay_debugf(MPXPLAY_DEBUGOUT_EIT,"EIT SECTION PROCESSED tid:%d", h->table_id);
//    	goto err_out_section;
//    }
	if(inffmp_epg_section_check_crc(h) < 0)
		goto err_out_section;
//    mpxplay_debugf(MPXPLAY_DEBUGOUT_EIT,"EIT SECTION BEGIN tid:%d --------------------------------------", h->table_id);

	mpxplay_bitstream_skipbytes(bs, 2); // transport_stream_id
	mpxplay_bitstream_skipbytes(bs, 2); // original_network_id
	mpxplay_bitstream_skipbytes(bs, 1); // segment_last_section_number
	mpxplay_bitstream_skipbytes(bs, 1); // last_table_id

	while(mpxplay_bitstream_leftbytes(bs) > 12)
	{
		struct mpxplay_dvb_epg_event_data_s *program_event_data = NULL;

		event_id = mpxplay_bitstream_get_be16(bs); // event_id (16 bits)
		event_start_local_datetime = pds_utctime_to_localtime(inffmp_epg_read_datetime(bs)); // date and time (40 bits), converted to local date/time
		event_duration = inffmp_epg_bcd_to_hexa((mpxplay_bitstream_get_byte(bs) << 16) | (mpxplay_bitstream_get_byte(bs) << 8) | mpxplay_bitstream_get_byte(bs)); // duration time (24 bits)

		desc_list_len = mpxplay_bitstream_get_be16(bs);
		desc_list_len &= 0xfff; /* higher 4 bits are: running_status (3) and free_CA_mode (1) */
		desc_list_leftbytes = mpxplay_bitstream_leftbytes(bs);
		if (!desc_list_len || (desc_list_len > desc_list_leftbytes)) // incorrect descriptor
		{
			//mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"EIT desc_list_len small dl:%d lb:%d", desc_list_len, desc_list_leftbytes);
			break;
		}
		if(!event_duration) // probably invalid event
		{
			goto skip_desc_list;
		}

		desc_list_bytecount = desc_list_len;
		while((desc_list_bytecount > 0) && (mpxplay_bitstream_leftbytes(bs) >= 4))
		{
			desc_tag = mpxplay_bitstream_get_byte(bs);
			desc_len = mpxplay_bitstream_get_byte(bs);
			desc_tag_leftbytes = mpxplay_bitstream_leftbytes(bs);
			if(desc_len > desc_tag_leftbytes)
			{
				mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"EIT desc_tag_len small dt:%2.2X dl:%d lb:%d", desc_tag, desc_len, desc_tag_leftbytes);
				break;
			}

			if(desc_len)
			{
				switch(desc_tag)
				{
					case 0x4d: // short event descriptor
						if(desc_len <= 5) // invalid length or no real content in this descriptor
							break;
						mpxplay_bitstream_skipbytes(bs, 3); // ISO_639_language_code str (unused)
						event_name = inffmp_epg_read_string_descriptor(bs, FALSE);
						if(!event_name)
							break;
						if(!program_event_data)
							program_event_data = inffmp_epg_event_new(epg_infos, h->transport_stream_id, event_id, event_start_local_datetime, event_duration);
						if(program_event_data)
						{
							char *event_description = inffmp_epg_read_string_descriptor(bs, FALSE);
							mpxplay_dtvdrv_database_event_add_short_description(program_event_data, event_name, event_description);
							if(parental_rating_value)
								mpxplay_dtvdrv_database_event_add_parental_rating(program_event_data, parental_rating_value);
							mpxplay_debugf(MPXPLAY_DEBUGOUT_EIT,"EIT 0x4d sid:%d date: %d.%2.2d.%2.2d time:%2d:%2.2d:%2.2d dur:%2d:%2.2d:%2.2d name:%s desc:%s",
								h->transport_stream_id,
								(mpxp_uint32_t)(event_start_local_datetime >> 48), (mpxp_uint32_t)((event_start_local_datetime >> 40) & 0xff), (mpxp_uint32_t)((event_start_local_datetime >> 32) & 0xff),
								(mpxp_uint32_t)((event_start_local_datetime >> 16) & 0xFF), (mpxp_uint32_t)((event_start_local_datetime >> 8) & 0xff), (mpxp_uint32_t)(event_start_local_datetime & 0xff),
								(mpxp_uint32_t)(event_duration >> 16), (mpxp_uint32_t)((event_duration >> 8) & 0xff), (mpxp_uint32_t)(event_duration & 0xff),
								event_name, event_description);
							pds_free(event_description);
						}
						pds_free(event_name);
						if(h->table_id == 0x4E) // actual_transport_stream, present/following
						{
							mpxplay_debugf(MPXPLAY_DEBUGOUT_EITCURR,"EIT 0x4E sid:%d date: %d.%2.2d.%2.2d time:%2d:%2.2d:%2.2d dur:%2d:%2.2d:%2.2d name:%s",
								h->transport_stream_id,
								(mpxp_uint32_t)(event_start_local_datetime >> 48), (mpxp_uint32_t)((event_start_local_datetime >> 40) & 0xff), (mpxp_uint32_t)((event_start_local_datetime >> 32) & 0xff),
								(mpxp_uint32_t)((event_start_local_datetime >> 16) & 0xFF), (mpxp_uint32_t)((event_start_local_datetime >> 8) & 0xff), (mpxp_uint32_t)(event_start_local_datetime & 0xff),
								(mpxp_uint32_t)(event_duration >> 16), (mpxp_uint32_t)((event_duration >> 8) & 0xff), (mpxp_uint32_t)(event_duration & 0xff),
								(program_event_data)? program_event_data->event_shortdesc_name : "");
							inffmp_epg_update_application(epg_infos, h->transport_stream_id, program_event_data);
						}
						break;
					case 0x4e: // extended event descriptor
						if(ffmpi && funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM)) // this section is handled from the driver level only at the case of live DVB stream
							break;
						if(desc_len <= 5) // invalid length or no real content in this descriptor
							break;
						if(!program_event_data)
						{
							program_event_data = inffmp_epg_event_new(epg_infos, h->transport_stream_id, event_id, event_start_local_datetime, event_duration);
							if(!program_event_data)
								break;
						}
						{
							mpxp_uint8_t curr_desc_num, last_desc_num;
							val = mpxplay_bitstream_get_byte(bs); // descriptor_number (4), last_descriptor_number (4)
							last_desc_num = (val & 0x0F);
							curr_desc_num = val >> 4;
							if(curr_desc_num > last_desc_num) // invalid descriptor_number
								break;
							if(curr_desc_num != program_event_data->extdesc_counter) // we want to read and collect extended event descriptor datas sequentially (and only once)
								break;
						}
						mpxplay_bitstream_skipbytes(bs, 3);   // ISO_639_language_code str (unused)
						val = mpxplay_bitstream_get_byte(bs); // length of items
						mpxplay_bitstream_skipbytes(bs, val); // skip items
						event_name = inffmp_epg_read_string_descriptor(bs, FALSE);
						if(mpxplay_dtvdrv_database_event_add_extended_description(program_event_data, event_name, TRUE))
							program_event_data->extdesc_counter++;
						//mpxplay_debugf(MPXPLAY_DEBUGOUT_EITEXT,"%d %s", event_id, event_name);
						pds_free(event_name);
						break;
					case 0x55: // parental rating descriptor
						if(ffmpi && funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM)) // this section is handled in the driver level only at the case of live DVB dtream
							break;
						if(desc_len < 4) // invalid length for this descriptor
							break;
						mpxplay_bitstream_skipbytes(bs, 3); // ISO_639_language_code str (unused)
						val = mpxplay_bitstream_get_byte(bs);
						if((val >= 0x01) && (val <= 0x0f)) {
							val += 3;
							if(program_event_data)
								mpxplay_dtvdrv_database_event_add_parental_rating(program_event_data, val);
							else
								parental_rating_value = val; // we don't create event entry here
						}
						//mpxplay_debugf(MPXPLAY_DEBUGOUT_EIT,"EIT 0x55 sid:%d tid:%d parental_rating:%d", h->transport_stream_id, h->table_id, val);
						/* currently we read the first parental rating descriptor only */
						break;
				}
				mpxplay_bitstream_skipbytes(bs, (desc_len - (desc_tag_leftbytes - mpxplay_bitstream_leftbytes(bs))));
			}
			desc_list_bytecount -= (desc_len + 2);
		}
skip_desc_list:
		mpxplay_bitstream_skipbytes(bs, (desc_list_len - (desc_list_leftbytes - mpxplay_bitstream_leftbytes(bs))));
		if(program_event_data && epg_infos->control_cb)
		{
			mpxp_uint32_t program_id = h->transport_stream_id;
			epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_DISKFILE_CFGFUNCNUM_ADD_PROGEVENTENTRY, &program_id, program_event_data);
		}
	}

err_out_section:
//#ifdef MPXPLAY_USE_DEBUGF
//	if(mpxplay_bitstream_leftbytes(bs) > 0)
//	{
//		if((h->table_id < 0x4E) || (h->table_id == 0x4F) || (h->table_id > 0x5F)){
//		}
//		else
//		{
//			mpxplay_debugf(MPXPLAY_DEBUGOUT_WARNING,"EIT END tid:%2.2X skip bytes:%d", h->tid, mpxplay_bitstream_leftbytes(bs));
//		}
//	}
//#endif
	mpxplay_bitstream_reset(bs); // wipe out unused data from the section buffer
}

//---------------------------------------------------------------------------------------------------------------------------

// search program in EPG database by program_id
struct mpxplay_dvb_epg_program_data_s *mpxplay_inffmp_epg_search_programdata_by_programid(struct ffmpg_epg_info_s *epg_infos, int program_id)
{
	struct mpxplay_dvb_epg_program_data_s *epg_program_data = epg_infos->epg_program_data_chain;

	while(epg_program_data)
	{
		if(epg_program_data->program_id == program_id)
			break;
		epg_program_data = epg_program_data->next_program;
	}

	return epg_program_data;
}

// get an EPG list in "HH:MM Description_name" format
struct mpxplay_dvb_epgeventlist_t *mpxplay_inffmp_epg_prepare_epginfolist(struct ffmpg_demuxer_data_s *ffmpi, int streamtype_index)
{
	struct mpxplay_dvb_epgeventlist_t *epg_list = NULL;
	struct mpxplay_dvb_epg_program_data_s *epg_program_data;
	struct mpxplay_dvb_epg_event_data_s *epg_program_events;
	struct ffmpeg_external_files_info_s *extfilehandler = &ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO];
	struct ffmpg_epg_info_s *epg_infos = &extfilehandler->epg_infos;
	mpxp_uint64_t local_datetime_val;
	int program_id, event_count;
	char **epg_event_list;

	if(!epg_infos->nb_program_datas || !epg_infos->epg_program_data_chain)
		return epg_list;

	program_id = in_ffmpstrm_all_programs_videostreams_program_id_get(ffmpi, streamtype_index);
	if(program_id <= 0)
		return epg_list;

	epg_program_data = mpxplay_inffmp_epg_search_programdata_by_programid(epg_infos, program_id);
	if(!epg_program_data)
		return epg_list;

	epg_list = pds_calloc(1, sizeof(*epg_list));
	if(!epg_list)
		return epg_list;

	epg_list->program_name = pds_malloc(pds_strlen(epg_program_data->program_name) + 1);
	if(!epg_list->program_name)
		goto err_out_prepare;
	pds_strcpy(epg_list->program_name, epg_program_data->program_name);

	epg_program_events = epg_program_data->epg_program_event_chain;
	if(!epg_program_data->nb_program_events || !epg_program_events) // no EPG event list, just program name -> valid
		return epg_list;

	epg_list->nb_program_events = epg_program_data->nb_program_events;

	local_datetime_val = ((ffmpi->flags & INFFMPG_FLAG_IS_LIVESTREAM)? pds_getdatetime() : epg_infos->carrier_localized_datetime);

	// skip events, which already have finished (begin of next event is less than local time)
	event_count = 0;
	do{
		if(epg_program_events->next_event && (epg_program_events->next_event->event_begin_date_time >= local_datetime_val))
			break;
		epg_list->nb_program_events--;
		epg_program_events = epg_program_events->next_event;
	}while(epg_program_events && (++event_count < epg_program_data->nb_program_events));

	if(!epg_list->nb_program_events || (event_count >= epg_program_data->nb_program_events) || !epg_program_events) // no EPG event list, just program name -> valid
		return epg_list;

	// allocate event list
	epg_list->program_event_list = epg_event_list = (char **)pds_calloc(epg_list->nb_program_events, sizeof(epg_list->program_event_list[0]));
	if(!epg_event_list)
		return epg_list; // we keep the program name (only) at the case of this error

	// fill epg list from program_event data
	event_count = 0;
	do{
		const unsigned int hour = (epg_program_events->event_begin_date_time >> 16) & 0xFF;
		const unsigned int minute = (epg_program_events->event_begin_date_time >> 8) & 0xFF;
		const int mlen = pds_strlen(epg_program_events->event_shortdesc_name) + 8; // +8 is "HH:MM "
		epg_event_list[event_count] = pds_malloc(mlen);
		if(!epg_event_list[event_count])
			break;
		snprintf(epg_event_list[event_count], mlen, "%2d:%2.2d\t%s", hour, minute, epg_program_events->event_shortdesc_name);
		epg_program_events = epg_program_events->next_event;
	}while(epg_program_events && (++event_count < epg_list->nb_program_events));

	return epg_list;

err_out_prepare:
	mpxplay_inffmp_epg_clear_epginfolist(epg_list);
	return NULL;
}

// clear dvb_epgeventlist
void mpxplay_inffmp_epg_clear_epginfolist(struct mpxplay_dvb_epgeventlist_t *epg_list)
{
	if(!epg_list)
		return;

	if(epg_list->program_name)
		pds_free(epg_list->program_name);

	if(epg_list->program_event_list)
	{
		unsigned int i;
		for(i = 0; i < epg_list->nb_program_events; i++)
			pds_free(epg_list->program_event_list[i]);
		pds_free(epg_list->program_event_list);
	}

	pds_free(epg_list);
}

//---------------------------------------------------------------------------------------------------------------------------
// save name and duration of current event (program title) of a specified program, send up to app
static void inffmp_epg_update_current_event(struct ffmpg_epg_info_s *epg_infos, struct mpxplay_dvb_epg_program_data_s *prog_data, char *event_name, mpxp_uint32_t event_duration)
{
	if(prog_data && ((pds_strcmp(prog_data->current_event_name, event_name) != 0) || (prog_data->current_event_duration != event_duration)))
	{
		unsigned int len = pds_strlen(event_name);
		if(len)
		{
			if(prog_data->current_event_name)
				pds_free(prog_data->current_event_name);
			prog_data->current_event_name = pds_malloc(len + 1);
			if(prog_data->current_event_name)
				pds_memcpy(prog_data->current_event_name, event_name, len + 1);
		}
		prog_data->current_event_duration = event_duration;
		prog_data->current_event_changed = TRUE;
	}
}

#ifdef MPXPLAY_GUI_QT

extern mpxp_bool_t mpxplay_dispqt_epgdialog_send_localfile_epginfos(void *protocol_data, char *filename);

static void inffmp_epg_send_events_to_localfile_epgdialog(struct ffmpg_epg_info_s *epg_infos, char *filename)
{
	struct ffmpg_demuxer_data_s *ffmpi = (struct ffmpg_demuxer_data_s *)epg_infos->ffmpi;
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = NULL;

	if(!mpxplay_dispqt_epgdialog_send_localfile_epginfos((void *)prot_data, filename)) // dialog is closed
		return;

	prot_data = mpxplay_dtvdrv_database_dup_programs_into_protocol(epg_infos->epg_program_data_chain);

	if(prot_data)
	{
		funcbit_enable(prot_data->prot_flags, (MPXPLAY_DVBEPG_PROTOCOL_FLAG_FULLDATA | MPXPLAY_DVBEPG_PROTOCOL_FLAG_LOCALFILE));
		prot_data->stream_datetime_val = epg_infos->carrier_localized_datetime;
		prot_data->protocol_id = MPXPLAY_DRVDTV_BDADEVTYPE_LOCALFILE;
		prot_data->frequency_data_chain->frequency = MPXPLAY_DRVDTV_FREQUENCY_LOCALFILE;
		prot_data->curr_program_id = ffmpi->program_number_current;
		if(!mpxplay_dispqt_epgdialog_send_localfile_epginfos((void *)prot_data, filename))
			mpxplay_dtvdrv_database_clear_protocols(prot_data); // dialog is closed (should not happen here)
	}
}

#endif

// send informations (title, duration, pos) of current program event to the higher level application
static void inffmp_epg_update_application(struct ffmpg_epg_info_s *epg_infos, int program_id, struct mpxplay_dvb_epg_event_data_s *event_data)
{
	struct ffmpg_demuxer_data_s *ffmpi = (struct ffmpg_demuxer_data_s *)epg_infos->ffmpi;
	mpxp_uint64_t current_datetime = ((!ffmpi || (ffmpi->flags & INFFMPG_FLAG_IS_LIVESTREAM))? pds_getdatetime() : epg_infos->carrier_localized_datetime); // live stream vs. local file
	mpxp_int64_t current_dtsecs = pds_datetimeval_to_seconds(current_datetime);
	mpxp_int64_t event_start_dtsecs = 0, event_duration_secs = 1;
	struct mpxplay_dvb_epg_program_data_s *program_data = NULL;
	mpxp_int32_t time_ms, prg_id;

#ifdef MPXPLAY_GUI_QT
	// send local file EPG infos, if SDT is not read yet
	if(!current_datetime && ffmpi && !funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
	{
		if(program_id == ffmpi->program_number_current)
		{
			struct ffmpeg_external_filelist_elem_s *files_chain = ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO].external_files_chain;
			inffmp_epg_send_events_to_localfile_epgdialog(epg_infos, (files_chain)? files_chain->external_filename : NULL);
			mpxplay_debugf(MPXPLAY_DEBUGOUT_APP,"APP epgdialog sent");
		}
		return;
	}
#endif

	// database handler may filter out EIT current event (because it's duplicated / incorrect), then we read a correct one from the database
	if(!event_data)
	{
		program_data = mpxplay_inffmp_epg_search_programdata_by_programid(epg_infos, program_id);
		event_data = mpxplay_dtvdrv_database_get_current_event(program_data);
	}

	if(event_data)
	{
		event_start_dtsecs = pds_datetimeval_to_seconds(event_data->event_begin_date_time);
		event_duration_secs = pds_timeval_to_seconds(event_data->event_duration_time);

		// check that the current time is in the range of this (got) program event
		if(current_datetime < event_data->event_begin_date_time)
			return;
		if(current_dtsecs >= (event_start_dtsecs + event_duration_secs))
			return;

		if(!program_data)
			program_data = mpxplay_inffmp_epg_search_programdata_by_programid(epg_infos, program_id);

		inffmp_epg_update_current_event(epg_infos, program_data, event_data->event_shortdesc_name, event_data->event_duration_time);
	}

	if(!epg_infos->control_cb)
		return;

	// called out of FFmpeg context
	if(!ffmpi && program_data && program_data->current_event_changed)
	{
		// update event in playlist editor at DVB live stream through drv_dtv.c
		prg_id = program_id;
		if(epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGEVENTCHANGE, &prg_id, NULL) == MPXPLAY_DISKDRIV_CFGERROR_SET_OK)
			program_data->current_event_changed = FALSE;
	}

	if(!ffmpi || (program_id != ffmpi->program_number_current))
		return;

	// send duration and position of current event at DTV/DVB live streams (local files: we shall able to seek a local file completely, not only in event range -> event range and pos are not send)
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
	{
		if(event_data)
		{
			//if((program_id != epg_infos->last_program_id) || (program_data && program_data->current_event_changed)) // FIXME: doesn't work at play start
			{
				time_ms = event_duration_secs * 1000;
				epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_DURATION_SET, &time_ms, NULL);
				mpxplay_debugf(MPXPLAY_DEBUGOUT_POS,"POS DUR %d", time_ms / 1000);
			}
			time_ms = (current_dtsecs - event_start_dtsecs) * 1000; // elapsed time in msecs (aka position in the program event)
			epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_POSITION_SET, &time_ms, NULL);
			mpxplay_debugf(MPXPLAY_DEBUGOUT_POS,"POS POS %d", time_ms / 1000);
		}
	}
#ifdef MPXPLAY_GUI_QT
	else // send EPG list to DVB/EPG dialog window
	{
		struct ffmpeg_external_filelist_elem_s *files_chain = ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO].external_files_chain;
		inffmp_epg_send_events_to_localfile_epgdialog(epg_infos, (files_chain)? files_chain->external_filename : NULL);
		mpxplay_debugf(MPXPLAY_DEBUGOUT_APP,"APP epgdialog sent");
	}
#endif

	// send program name and current event title if they are changed (local file)
	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_NOMETADATASEND))
	{
		int len;
		if(program_data && program_data->program_name)
		{
			len = -1;
			epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_PACKFUNCTTI3I(MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_UPDATE,MPXPLAY_TEXTCONV_TYPE_UTF8,I3I_ARTIST), program_data->program_name, &len);
		}
		if(event_data && event_data->event_shortdesc_name)
		{
			len = -1;
			epg_infos->control_cb(epg_infos->ccb_data, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_PACKFUNCTTI3I(MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_UPDATE,MPXPLAY_TEXTCONV_TYPE_UTF8,I3I_TITLE), event_data->event_shortdesc_name, &len);
		}
	}

	if(program_data)
		program_data->current_event_changed = FALSE;
}

#endif // defined(MPXPLAY_LINK_INFILE_FF_MPEG)
