//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2019 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFMPEG parsers for video streams

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

extern AVInputFormat ff_mpegps_demuxer;

int in_ffmppars_parse_stream_header(struct ffmpg_demuxer_data_s *ffmpi, unsigned int stream_count_min)
{
	AVFormatContext *s = ffmpi->fctx;
	AVPacket *pkt = ffmpi->pkt;
	int j, av_stream_count=-1;
	int prev_stnum=0;
	int retry_frame = (ffmpi->flags & INFFMPG_FLAG_AUTODETECT)? 200 : 1000;
	int retry_empty = 2, retry_bad = (s->flags & INFFMPG_FLAG_AUTODETECT)? 1 : 2;
	int retry_stream = 150;
	unsigned int stream_counters[INFFMPG_MAX_STREAMS];

	memset(stream_counters, 0, sizeof(stream_counters));

	avio_seek(s->pb, 0, SEEK_SET);

	do{
		int retcode;
		av_packet_unref(pkt);
		retcode = av_read_frame(ffmpi->fctx, pkt); // this requires to parse streams (instead of iformat->read_packet)
		if(retcode==AVERROR(EIO)) // !!! means empty data here
		{
			mpxplay_debugf(MPXPLAY_DEBUG_HEADER,"AVERROR(EIO)");
			if(avio_feof(s->pb))
				break;
			if(!(--retry_empty))
				break;
		}
		else if(retcode < 0)
		{
			mpxplay_debugf(MPXPLAY_DEBUG_HEADER,"retry_bad: %d retcode:%d", retry_bad, retcode);
			if(!(--retry_bad))
				break;
			prev_stnum = 0;
			retry_stream = 150;
		}
		else if((pkt->size > 0) && (pkt->stream_index < s->nb_streams))
		{
			mpxplay_debugf(MPXPLAY_DEBUG_HEADER,"r:%5d nbs:%d si:%d mt:%d cid:%X rs:%d",retry_frame,
					s->nb_streams, pkt->stream_index, s->streams[pkt->stream_index]->codecpar->codec_type,
					s->streams[pkt->stream_index]->codecpar->codec_id, retry_stream);
			stream_counters[pkt->stream_index]++;
			av_stream_count = -1;
			for(j = 0; (j < s->nb_streams) && (j < INFFMPG_MAX_STREAMS); j++){
				AVStream *st = s->streams[j];
				if((st->codecpar->codec_type == AVMEDIA_TYPE_AUDIO) || (st->codecpar->codec_type == AVMEDIA_TYPE_VIDEO)){
					av_stream_count++;
					//mpxplay_debugf(MPXPLAY_DEBUG_HEADER,"n:%d cid:%X sid:%d rs:%d",j,st->codec->codec_id,st->id,retry_stream);
				}
			}
			if(av_stream_count >= 0)
			{
//				if(ffmpi->flags & INFFMPG_FLAG_LOADHEADONLY) // FIXME: doesn't work on program streams
//					break;
				for(j = 0; (j < s->nb_streams) && (j < INFFMPG_MAX_STREAMS); j++)
					if(stream_counters[j] < stream_count_min) // ??? we have to read/find all streams 5 times to be sure that there are no more streams
						break;
				if((j == s->nb_streams) || (j == INFFMPG_MAX_STREAMS))
					break;
				if(!(--retry_stream))
					break;
			}
			retry_empty = 2;
			retry_bad = 2;
		}
	}while(--retry_frame);
	avio_seek(s->pb,0,SEEK_SET);
	av_packet_unref(pkt);
	mpxplay_debugf(MPXPLAY_DEBUG_HEADER,"asc:%d r:%d",av_stream_count, retry_frame);
	return av_stream_count;
}

static long inffmpg_get_pts_from_filepos(struct ffmpeg_external_files_info_s *extfilehandler, int primary_ff_stream_idx, mpxp_int64_t filepos, mpxp_int64_t *pts_vals, unsigned int index, unsigned int lastindex)
{
	AVFormatContext *fctx = extfilehandler->external_file_avctx;
	AVPacket *pkt = av_packet_alloc();
	long res,retry,retryrp;
#ifdef MPXPLAY_USE_DEBUGF
	long counter=0;
#endif

	avio_seek(fctx->pb, filepos, SEEK_SET);
	if(index==lastindex)
		retry=10000;
	else
		retry=1000;
	retryrp=2;
	do{
		pkt->stream_index=100;
		pkt->size=0;
		av_packet_unref(pkt);
#if (LIBAVFORMAT_VERSION_MAJOR > 60)
		if((res = av_read_frame(fctx,pkt)) < 0) // FIXME: this can be wrong, maybe we need a native packet reading
#else
		if((res = fctx->iformat->read_packet(fctx,pkt)) < 0)
#endif
			if((res == AVERROR(EIO)) || !(--retryrp))
				return res;
		mpxplay_debugf(MPXPLAY_DEBUG_STDUR,"%4d read_packet si:%d ssi:%d fp:%llu", retry, pkt->stream_index, primary_ff_stream_idx, (mpxp_int64_t)filepos);

		if((pkt->pts>=0) && ((fctx->nb_programs <= 1) || (pkt->stream_index==primary_ff_stream_idx)) && (pkt->size > 0)){ // not exact, but faster if we use the pts of other stream (and some TS files fail without this)
			mpxp_int64_t timestamp = in_ffmpgdec_get_avpacket_timestamp_fctx(fctx, primary_ff_stream_idx, (mpxp_int64_t)pkt->pts) - INFFMPG_TIMESTAMP_SHIFT;

			mpxplay_debugf(MPXPLAY_DEBUG_STDUR,"%5.5d i:%d si:%d fsi:%d pts:%llu dts:%llu fp:%llu",
					counter++,index,pkt->stream_index,primary_ff_stream_idx,pkt->pts,pkt->dts,(mpxp_int64_t)filepos);

			if(index==lastindex){
				if((timestamp>=0) && (timestamp>pts_vals[index]))
					pts_vals[index]=timestamp;
			}else{
				if(timestamp>=0){
					pts_vals[index]=timestamp;
					break;
				}
			}
		}
	}while(--retry);
	av_packet_free(&pkt);
	if(!retry)
		return -1;
	return 0;
}

//it reads timestamp from all streams, not only from a specified stream! (this is faster, but not exact)
mpxp_int64_t in_ffmppars_get_stream_duration_pts(struct ffmpeg_external_files_info_s *extfilehandler, int primary_ff_stream_idx)
{
	struct mpxplay_filehand_buffered_func_s *fbfs = extfilehandler->fbfs;
	AVFormatContext *fctx = extfilehandler->external_file_avctx;
	mpxp_filesize_t oldfilepos =-1, filepos, lastblocksize = 0, filesize = fbfs->filelength(extfilehandler->fbds);
	int i, checknum = (fctx->iformat == &ff_mpegps_demuxer)? 3 : 2;
	float diff1,diff2;
	mpxp_int64_t duration=0,pts_vals[INFFMPG_PTS_CHECK_NUM_MAX];

	oldfilepos=avio_tell(fctx->pb);

	do{
		pds_memset(&pts_vals[0],0,sizeof(pts_vals));
		filepos=0;
		for(i = 0; i < checknum; i++){
			if(inffmpg_get_pts_from_filepos(extfilehandler, primary_ff_stream_idx, filepos, &pts_vals[0], i, (checknum-1)) < 0)
				break;
			if(!i){ // calculate last block checksize from the first blocksize
				lastblocksize=max(INFFMPG_INITIAL_PACKETBUF_SIZE,fbfs->ftell(extfilehandler->fbds)*2);
				if(lastblocksize>(filesize/2))
					lastblocksize=filesize/2;
			}
			if(i==(checknum-2)) // last block follows
				filepos=filesize-lastblocksize;
			else
				filepos+=filesize/(checknum-1);
			mpxplay_debugf(MPXPLAY_DEBUG_STDUR,"c:%d i:%d pts:%llu fp:%d",checknum,i,pts_vals[i],(long)filepos);
		}

		if(checknum<3){ // TS files
			if(pts_vals[1]>pts_vals[0]){ // usually have to be
				duration=pts_vals[1]-pts_vals[0];
				break;
			} // invalid pts vals, search like MPG
			checknum=3;
		}else{ // MPG/VOB files (search for two sequential block with similar length)
			diff1=0;
			for(i=0;i<(checknum-1);i++){
				if(pts_vals[i+1]<pts_vals[i])
					diff1=0;
				else if(diff1==0)
					diff1=(float)pts_vals[i+1]-(float)pts_vals[i];
				else{
					diff2=pts_vals[i+1]-pts_vals[i];
					if(((diff1*1.4)>diff2) && ((diff1*0.6)<diff2)){ // max 40% difference between the length of block1 and block2
						if(checknum==3){
							duration=pts_vals[2]-pts_vals[0];
						}else{
							if(diff2>diff1)
								diff1=diff2;
							duration=(diff1*(float)(checknum-1));
						}
						goto err_out_durpts;
					}else
						diff1=diff2;
				}
			}
		}

		if(checknum>(INFFMPG_PTS_CHECK_NUM_MAX/2)){ // last pass, block-search failed
			if(pts_vals[1]>pts_vals[0]){
				for(i=1;i<(checknum-1);i++){
					if(pts_vals[i+1]<pts_vals[i]){ // pts[i] is the last valid pts, we calculate the duration from this value
						duration=(((float)pts_vals[i]-(float)pts_vals[0])*(float)(checknum-1)/(float)i);
						goto err_out_durpts;
					}
				}
				if(i==(checknum-1))
					duration=pts_vals[checknum-1]-pts_vals[0];
			}
		}
		checknum<<=1; // 3,6,12
	}while(checknum<=INFFMPG_PTS_CHECK_NUM_MAX);

err_out_durpts:

	avio_seek(fctx->pb, oldfilepos, SEEK_SET);

	//mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT,"sp0:%"PRIi64" sp1:%"PRIi64" ep:%"PRIi64" d:%"PRIi64"",pts_vals[0],pts_vals[1],pts_vals[checknum-1],duration);
	return duration;
}

#endif //MPXPLAY_LINK_INFILE_FF_MPEG
