//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: A/V playing with FFMpeg library

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

#include "mpxplay.h"
#include "mpxinbuf.h"
#include "tagging.h"
#include <diskdriv/dtv_drv.h>
#ifdef MPXPLAY_GUI_QT
#include <libavfilter/avfilter.h>
#include <disp_qt/disp_qt.h>
#endif

#ifdef MPXPLAY_GUI_QT
#define INFFMPG_ADITEXTSIZE_LONGNAME 128
#else
#define INFFMPG_ADITEXTSIZE_LONGNAME 8
#endif

extern AVInputFormat ff_aac_demuxer;
extern AVInputFormat ff_ape_demuxer;
extern AVInputFormat ff_avi_demuxer;
extern AVInputFormat ff_ac3_demuxer;
extern AVInputFormat ff_dts_demuxer;
extern AVInputFormat ff_flac_demuxer;
extern AVInputFormat ff_flv_demuxer;
extern AVInputFormat ff_matroska_demuxer;
extern AVInputFormat ff_mov_demuxer;
extern AVInputFormat ff_mp3_demuxer;
extern AVInputFormat ff_ogg_demuxer;
extern AVInputFormat ff_mpegts_demuxer;

extern int mpxplay_control_startup_programid_select;
extern unsigned long mpxplay_config_dvbepg_control_flags;
extern unsigned int mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM];

#ifdef MPXPLAY_GUI_CONSOLE
#include <display/display.h>
extern unsigned int refdisp;
#endif

static unsigned int inffmpeg_channel_layout_order[MPXPLAY_PCMOUTCHAN_MAX] =
{
 MPXPLAY_PCMOUTCHAN_FRONT_LEFT,
 MPXPLAY_PCMOUTCHAN_FRONT_RIGHT,
 MPXPLAY_PCMOUTCHAN_FRONT_CENTER,
 MPXPLAY_PCMOUTCHAN_LFE,
 MPXPLAY_PCMOUTCHAN_REAR_LEFT,
 MPXPLAY_PCMOUTCHAN_REAR_RIGHT,
 MPXPLAY_PCMOUTCHAN_FCENTER_LEFT,
 MPXPLAY_PCMOUTCHAN_FCENTER_RIGHT,
 MPXPLAY_PCMOUTCHAN_REAR_CENTER,
 MPXPLAY_PCMOUTCHAN_SIDE_LEFT,
 MPXPLAY_PCMOUTCHAN_SIDE_RIGHT
};

static void in_ffmpeg_avinfo_strings_assemble(struct ffmpg_demuxer_data_s *ffmpi,struct mpxplay_infile_info_s *miis, char *filename);
static void infmpeg_preanalyze_and_init_video(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds);
static mpxp_int32_t inffmpg_miis_control_function(void *cf_data, mpxp_uint32_t funcnum, mpxp_ptrsize_t arg1, mpxp_ptrsize_t arg2);

int in_fmpeg_audio_codecid_to_waveid(struct ffmpg_demuxer_data_s *ffmpi,struct mpxplay_infile_info_s *miis, AVFrame *audio_frame)
{
	struct mpxplay_streampacket_info_s *spi=miis->audio_stream;
	struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
	unsigned int new_audio_waveid, new_audio_bits, new_audio_freq, new_audio_filechannels;
	AVCodecParameters *codecpar;
	AVCodecContext *codec;
#if MPXPLAY_USE_FFMPEG_V7X
	AVChannelLayout channel_layout;
#else
	uint64_t channel_layout;
#endif
#ifdef MPXPLAY_GUI_CONSOLE
	enum AVCodecID av_codec_id;
#endif
	int sample_fmt, i, ch;
	AVStream *st;

	// set default values (if no valid audio stream / info)
	spi->streamtype = MPXPLAY_SPI_STREAMTYPE_AUDIO;
	new_audio_waveid = MPXPLAY_WAVEID_PCM_FLOAT;
	new_audio_bits = 1;
	new_audio_freq = 48000;
	new_audio_filechannels = 2;
	funcbit_disable(adi->infobits, ADI_FLAG_FLOATOUT);
	funcbit_disable(ffmpi->flags, INFFMPG_FLAG_PLANARAUDIO);

	// load values from codec or codecpar // TODO: check
	codec = ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_AUDIO];
	st = ffmpi->selected_avstream[MPXPLAY_STREAMTYPEINDEX_AUDIO];
	codecpar = (st)? st->codecpar : NULL;
#ifdef MPXPLAY_GUI_CONSOLE
	av_codec_id = (codec && (codec->codec_id > AV_CODEC_ID_NONE))? codec->codec_id : ((codecpar)? codecpar->codec_id : AV_CODEC_ID_NONE);
#endif
	sample_fmt = (audio_frame && (audio_frame->format >= 0))? audio_frame->format : ((codec && (codec->sample_fmt >= 0))? codec->sample_fmt : ((codecpar && (codecpar->format >= 0))? codecpar->format : AV_SAMPLE_FMT_FLT));

	switch(sample_fmt){
		case AV_SAMPLE_FMT_U8: new_audio_waveid=MPXPLAY_WAVEID_PCM_ULE; new_audio_bits=8; break;
		case AV_SAMPLE_FMT_S16:new_audio_waveid=MPXPLAY_WAVEID_PCM_SLE; new_audio_bits=16; break;
		case AV_SAMPLE_FMT_S32:new_audio_waveid=MPXPLAY_WAVEID_PCM_SLE; new_audio_bits=32; break;
		case AV_SAMPLE_FMT_FLT:new_audio_waveid=MPXPLAY_WAVEID_PCM_FLOAT; new_audio_bits=1; break;
		case AV_SAMPLE_FMT_DBL:new_audio_waveid=MPXPLAY_WAVEID_PCM_F64LE; new_audio_bits=1; break;
		case AV_SAMPLE_FMT_U8P: new_audio_waveid=MPXPLAY_WAVEID_PCM_ULE; new_audio_bits=8; funcbit_enable(ffmpi->flags, INFFMPG_FLAG_PLANARAUDIO); break;
		case AV_SAMPLE_FMT_S16P:new_audio_waveid=MPXPLAY_WAVEID_PCM_SLE; new_audio_bits=16; funcbit_enable(ffmpi->flags, INFFMPG_FLAG_PLANARAUDIO); break;
		case AV_SAMPLE_FMT_S32P:new_audio_waveid=MPXPLAY_WAVEID_PCM_SLE; new_audio_bits=32; funcbit_enable(ffmpi->flags, INFFMPG_FLAG_PLANARAUDIO); break;
		case AV_SAMPLE_FMT_FLTP:new_audio_waveid=MPXPLAY_WAVEID_PCM_FLOAT; new_audio_bits=1; funcbit_enable(ffmpi->flags, INFFMPG_FLAG_PLANARAUDIO); break;
		case AV_SAMPLE_FMT_DBLP:new_audio_waveid=MPXPLAY_WAVEID_PCM_F64LE; new_audio_bits=1; funcbit_enable(ffmpi->flags, INFFMPG_FLAG_PLANARAUDIO); break;
		default:mpxplay_debugf(MPXPLAY_DEBUG_WARNING,"Invalid codec->sample_fmt: %d", codec->sample_fmt);return -1;
	}
	if(audio_frame && audio_frame->sample_rate > 0)
		new_audio_freq = audio_frame->sample_rate;
	else if(codec && (codec->sample_rate > 0))
		new_audio_freq = codec->sample_rate;
	else if(codecpar && (codecpar->sample_rate > 0))
		new_audio_freq = codecpar->sample_rate;

#if MPXPLAY_USE_FFMPEG_V7X
	if(audio_frame && audio_frame->ch_layout.nb_channels > 0) {
		channel_layout = audio_frame->ch_layout;
	} else if(codec && (codec->ch_layout.nb_channels > 0)) {
		channel_layout = codec->ch_layout;
	} else if(codecpar && (codecpar->ch_layout.nb_channels > 0)) {
		channel_layout = codecpar->ch_layout;
	} else {
		av_channel_layout_default(&channel_layout, new_audio_filechannels);
	}
	new_audio_filechannels = channel_layout.nb_channels;
#else
	if(audio_frame && audio_frame->ch_layout.nb_channels > 0)
		new_audio_filechannels = audio_frame->ch_layout.nb_channels;
	else if(codecpar && (codecpar->ch_layout.nb_channels > 0))
		new_audio_filechannels = codecpar->ch_layout.nb_channels;
	else if(audio_frame && audio_frame->channels > 0)
		new_audio_filechannels = audio_frame->channels;
	else if(codec && (codec->channels > 0))
		new_audio_filechannels = codec->channels;
#endif

	if(new_audio_filechannels > MPXPLAY_PCMOUTCHAN_MAX)
		return -1;
	if((ffmpi->flags & INFFMPG_FLAG_PLANARAUDIO) && (new_audio_filechannels > AV_NUM_DATA_POINTERS))
		return -1;

#if MPXPLAY_USE_FFMPEG_V7X
	if(pds_memcmp(&channel_layout, &ffmpi->audioframe_channel_layout, sizeof(channel_layout)) != 0)
#else
	if(audio_frame && audio_frame->channel_layout)
		channel_layout = audio_frame->channel_layout;
	else if(codec && codec->channel_layout)
		channel_layout = codec->channel_layout;
	else if(codecpar)
		channel_layout = codecpar->channel_layout;
	else
		channel_layout = 0ULL;
	if(!channel_layout)
		channel_layout = av_get_default_channel_layout(new_audio_filechannels);

	if(channel_layout != ffmpi->audioframe_channel_layout)
#endif
	{
		if(adi->chanmatrix)
		{
			pds_free(adi->chanmatrix);
			adi->chanmatrix = NULL;
		}
		if(new_audio_filechannels >= 2) // TODO: add missing chanmatrix to output drivers for mono and stereo
		{
			adi->chanmatrix = pds_calloc(new_audio_filechannels, sizeof(*adi->chanmatrix));
			for(i = 0, ch = 0; (i < MPXPLAY_PCMOUTCHAN_MAX) && (ch < new_audio_filechannels); i++)
			{
#if MPXPLAY_USE_FFMPEG_V7X
				if(channel_layout.u.mask & (1 << i)) // TODO: only AV_CHANNEL_ORDER_NATIVE is handled
					adi->chanmatrix[ch++] = inffmpeg_channel_layout_order[i];
#else
				if(channel_layout & (1 << i))
					adi->chanmatrix[ch++] = inffmpeg_channel_layout_order[i];
#endif
			}
		}
	}

	if(codec && (codec->codec_id >= AV_CODEC_ID_AMR_NB) && (ffmpi->fctx->nb_streams <= 1))
	{
		mpxp_int64_t file_bitrate = miis->filesize * 1000 / (ffmpi->fctx->duration / 8000);
		if(codec->bit_rate <= 0)
			codec->bit_rate = file_bitrate;
		if(codecpar && (codecpar->bit_rate <= 0))
			codecpar->bit_rate = file_bitrate;
	}
	if(codec && (codec->bit_rate > 0) && (codec->bit_rate < 9999000) && (codec->codec_id >= AV_CODEC_ID_AMR_NB))
		adi->bitrate = codec->bit_rate/1000;
	else if(codecpar && (codecpar->bit_rate > 0) && (codecpar->bit_rate < 9999000) && (codecpar->codec_id >= AV_CODEC_ID_AMR_NB))
		adi->bitrate = codecpar->bit_rate / 1000;

	if((miis->timemsec <= MPXPLAY_TIME_BASE) && adi->bitrate && (ffmpi->fctx->nb_streams <= 1))
		miis->timemsec = miis->filesize * 8 / adi->bitrate;

	ffmpi->outsample_size = (new_audio_waveid==MPXPLAY_WAVEID_PCM_FLOAT)? sizeof(float) : ((new_audio_waveid==MPXPLAY_WAVEID_PCM_F64LE)? sizeof(double) : (new_audio_bits / 8));
	spi->block_align = ffmpi->outsample_size * new_audio_filechannels;
	spi->bs_framesize = INFFMPG_MAX_OUTPUT_SAMPLENUM * spi->block_align;
	funcbit_enable(spi->flags,(MPXPLAY_SPI_FLAG_NEED_DECODER|MPXPLAY_SPI_FLAG_CONTAINER));

	if( (spi->wave_id != new_audio_waveid) || (adi->bits != new_audio_bits) || (adi->freq != new_audio_freq) || (adi->filechannels != new_audio_filechannels)
#if MPXPLAY_USE_FFMPEG_V7X
	 || (pds_memcmp(&ffmpi->audioframe_channel_layout, &channel_layout, sizeof(channel_layout)) != 0)
#else
	 || (ffmpi->audioframe_channel_layout != channel_layout)
#endif
#ifdef MPXPLAY_GUI_CONSOLE
	 || (ffmpi->audioframe_codec_id != av_codec_id)
#endif
	){
		spi->wave_id = new_audio_waveid;
		adi->bits = new_audio_bits;
		adi->freq = new_audio_freq;
		adi->filechannels = new_audio_filechannels;
		adi->outchannels = new_audio_filechannels;
		ffmpi->audioframe_channel_layout = channel_layout;
#ifdef MPXPLAY_GUI_CONSOLE
		ffmpi->audioframe_codec_id = av_codec_id;
		in_ffmpeg_avinfo_strings_assemble(ffmpi, miis, NULL);
		funcbit_enable(refdisp, RDT_HEADER);
#endif
		return 1;
	}

	return 0;
}

static void in_ffmpeg_avinfo_strings_assemble(struct ffmpg_demuxer_data_s *ffmpi,struct mpxplay_infile_info_s *miis, char *filename)
{
	struct mpxplay_streampacket_info_s *spi = miis->audio_stream;
	struct mpxplay_audio_decoder_info_s *adi = miis->audio_decoder_infos;

	if(miis->longname)
		pds_free(miis->longname);
	if(adi->bitratetext)
		pds_free(adi->bitratetext);

	miis->longname = pds_malloc(INFFMPG_ADITEXTSIZE_LONGNAME + 8);
	adi->bitratetext = pds_malloc(INFFMPG_ADITEXTSIZE_LONGNAME + 8);
	if(miis->longname && adi->bitratetext)
	{
		char conatiner_name[32]="", *s;
		char codec_name[MPXPLAY_STREAMTYPEINDEX_PLAYNUM][32];
		char bit_rate_text[MPXPLAY_STREAMTYPEINDEX_PLAYNUM][80];
		int i;
		pds_strncpy(conatiner_name, (ffmpi->fctx->iformat->long_name)? (char *)ffmpi->fctx->iformat->long_name :
				((ffmpi->fctx->iformat->name)? (char *)ffmpi->fctx->iformat->name :
				pds_filename_get_extension_from_shortname(filename)), sizeof(conatiner_name));
		s = pds_strchr(conatiner_name, ',');
		if(s)
			*s = 0;
		pds_str_uppercase(conatiner_name);
#ifdef MPXPLAY_GUI_QT
		for(i = 0; i <= MPXPLAY_STREAMTYPEINDEX_VIDEO; i++)
#else
		for(i = 0; i <= MPXPLAY_STREAMTYPEINDEX_AUDIO; i++)
#endif
		{
			int ff_stream_index = ffmpi->selected_ff_stream_index[i];
			codec_name[i][0] = 0;
			bit_rate_text[i][0] = 0;
			if((ff_stream_index >= 0) && (ff_stream_index < ffmpi->fctx->nb_streams))
			{
				AVStream *st = ffmpi->fctx->streams[ff_stream_index];
				char resolution_text[64] = "", fps_text[64] = "", bitrate_text[64] = "";
				int bit_rate;
				if(!st || !st->codecpar)
					continue;
				pds_strncpy(codec_name[i],(char *)avcodec_get_name(st->codecpar->codec_id), sizeof(codec_name[i]));
				pds_str_uppercase(codec_name[i]);
#ifdef MPXPLAY_GUI_QT
				if(i == MPXPLAY_STREAMTYPEINDEX_VIDEO)
				{
					snprintf(resolution_text, sizeof(resolution_text), "%dx%d", st->codecpar->width, st->codecpar->height);
					resolution_text[sizeof(resolution_text) - 1] = 0;
					if(st->avg_frame_rate.num && st->avg_frame_rate.den)
					{
						snprintf(fps_text, sizeof(fps_text), ", %2.2f fps", (float)((float)st->avg_frame_rate.num / (float)st->avg_frame_rate.den));
						fps_text[sizeof(fps_text) - 1] = 0;
					}
				}
#endif
				bit_rate = st->codecpar->bit_rate / 1000;
				if(bit_rate)
				{
					if(bit_rate > 999)
					{
						snprintf(bitrate_text, sizeof(bitrate_text), "%.1f mb/s", (float)bit_rate / 1000.0);
					}
					else
					{
						snprintf(bitrate_text, sizeof(bitrate_text), "%d kb/s", bit_rate);
					}
					bitrate_text[sizeof(bitrate_text) - 1] = 0;
				}
				else if(i == MPXPLAY_STREAMTYPEINDEX_AUDIO)
				{
					snprintf(bitrate_text, sizeof(bitrate_text), " %d bits", ((adi->bits >= 8)? adi->bits : 32));
				}
				if(resolution_text[0] || bitrate_text[0])
				{
					snprintf(bit_rate_text[i], sizeof(bit_rate_text[i]), "%s%s%s%s", resolution_text, fps_text,
						((resolution_text[0] && bitrate_text[0])? ", " : ""), bitrate_text);
					bit_rate_text[i][sizeof(bit_rate_text[i]) - 1] = 0;
				}
			}
		}
#ifdef MPXPLAY_GUI_QT
		snprintf(miis->longname, INFFMPG_ADITEXTSIZE_LONGNAME, "%s -> %s%s%s", conatiner_name,
				(codec_name[MPXPLAY_STREAMTYPEINDEX_VIDEO][0])? codec_name[MPXPLAY_STREAMTYPEINDEX_VIDEO] : "",
				(codec_name[MPXPLAY_STREAMTYPEINDEX_AUDIO][0] && codec_name[MPXPLAY_STREAMTYPEINDEX_VIDEO][0])? " / " : "",
				(codec_name[MPXPLAY_STREAMTYPEINDEX_AUDIO][0])? codec_name[MPXPLAY_STREAMTYPEINDEX_AUDIO] : "");
		snprintf(adi->bitratetext, INFFMPG_ADITEXTSIZE_LONGNAME, "%s%s%s%s",
				(bit_rate_text[MPXPLAY_STREAMTYPEINDEX_AUDIO][0] && bit_rate_text[MPXPLAY_STREAMTYPEINDEX_VIDEO][0])? "A: " : "",
				(bit_rate_text[MPXPLAY_STREAMTYPEINDEX_AUDIO][0])? bit_rate_text[MPXPLAY_STREAMTYPEINDEX_AUDIO] : "",
				(bit_rate_text[MPXPLAY_STREAMTYPEINDEX_AUDIO][0] && bit_rate_text[MPXPLAY_STREAMTYPEINDEX_VIDEO][0])? "   V: " : (bit_rate_text[MPXPLAY_STREAMTYPEINDEX_VIDEO][0]? "V: " : ""),
				(bit_rate_text[MPXPLAY_STREAMTYPEINDEX_VIDEO][0])? bit_rate_text[MPXPLAY_STREAMTYPEINDEX_VIDEO] : "");
		miis->longname[INFFMPG_ADITEXTSIZE_LONGNAME] = adi->bitratetext[INFFMPG_ADITEXTSIZE_LONGNAME] = 0;
#else // MPXPLAY_GUI_CONSOLE
		snprintf(miis->longname, (MPXPLAY_ADITEXTSIZE_LONGNAME + 1), "%.4s/%-5s", conatiner_name, (codec_name[MPXPLAY_STREAMTYPEINDEX_AUDIO][0])? codec_name[MPXPLAY_STREAMTYPEINDEX_AUDIO] : "");
		snprintf(adi->bitratetext, (MPXPLAY_ADITEXTSIZE_BITRATE + 1), "%8s", (bit_rate_text[MPXPLAY_STREAMTYPEINDEX_AUDIO][0])? bit_rate_text[MPXPLAY_STREAMTYPEINDEX_AUDIO] : "");
		miis->longname[MPXPLAY_ADITEXTSIZE_LONGNAME] = adi->bitratetext[MPXPLAY_ADITEXTSIZE_BITRATE] = 0;
#endif
	}
}

static int inffmpg_assign_values_audiovideo(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_infile_info_s *miis, char *filename, mpxp_uint32_t openmode)
{
	struct mpxplay_audio_decoder_info_s *adi = miis->audio_decoder_infos;
	AVFormatContext *ic = ffmpi->fctx;
	int result = 0;

	miis->timemsec = ic->duration / (AV_TIME_BASE/MPXPLAY_TIME_BASE);
	if(!miis->timemsec)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_HEADER,"inffmpg_assign_values_audiovideo dur:%lld", ic->duration);
		return result;
	}

	if(openmode&MPXPLAY_INFILE_OPENMODE_INFO_DECODER)
	{
		if(in_fmpeg_audio_codecid_to_waveid(ffmpi, miis, NULL) >= 0)
		{
			struct mpxplay_streampacket_info_s *spi_audio = miis->audio_stream;
			mpxplay_debugf(MPXPLAY_DEBUG_HEADER,"ns:%d freq:%d chans:%d bits:%d duration:%d br:%d wid:%8.8X fmt:%d",
					ic->nb_streams,adi->freq,adi->outchannels,adi->bits,(long)miis->timemsec,adi->bitrate,spi_audio->wave_id,
					(ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_AUDIO])? ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_AUDIO]->sample_fmt : 0);
			in_ffmpeg_avinfo_strings_assemble(ffmpi, miis, filename);
			result = 1;
		}
		if(ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_VIDEO])
		{
			struct mpxplay_streampacket_info_s *spi_video = miis->video_stream;
			spi_video->streamtype = MPXPLAY_SPI_STREAMTYPE_VIDEO;
			result = 1;
		}
	}
	else
	{
		result = 1;
	}

	return result;
}

//------------------------------------------------------------------------------------------------------------------
static int inffmpg_infile_check(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,AVInputFormat *demuxer,mpxp_uint32_t openmode)
{
	struct ffmpg_demuxer_data_s *ffmpi;
	int i;

	mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "-----------------------------------------------------------------------------------------------");
	mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "inffmpg_infile_check BEGIN play:%d (om:%8.8X) d:%8.8X %s", (openmode & MPXPLAY_INFILE_OPENMODE_LOAD_PLAY)? 1 : 0, openmode, (mpxp_ptrsize_t)fbds, filename);

	funcbit_disable(openmode,(O_WRONLY|O_RDWR)); // ???

	ffmpi = pds_calloc(1, sizeof(struct ffmpg_demuxer_data_s));
	if(!ffmpi)
		goto err_out_chk;

	miis->private_data = ffmpi;
	miis->control_func = inffmpg_miis_control_function;
	ffmpi->struct_id = MPXPLAY_FFMPEGDEMUX_STRUCT_ID;
	ffmpi->fbfs = fbfs;
	ffmpi->fbds = fbds;
	ffmpi->miis = miis;

	if(!demuxer)
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_AUTODETECT);
	if(!funcbit_test(openmode, MPXPLAY_INFILE_OPENMODE_LOAD_PLAY)){
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_LOADHEADONLY);
		ffmpi->streamtype_limit = MPXPLAY_STREAMTYPEINDEX_VIDEO;
	}else
#ifdef MPXPLAY_GUI_QT
	if(mpxplay_dispqt_ffmpegvideo_configcbk_check(MPXPLAY_INFILE_CBKCFG_SRVFFMV_OPEN_CALLBACK) == 0)
		ffmpi->streamtype_limit = MPXPLAY_STREAMTYPEINDEX_SUBTITLE;
	else
#endif
		ffmpi->streamtype_limit = MPXPLAY_STREAMTYPEINDEX_AUDIO;

	ffmpi->primary_file_openmode = openmode;
	ffmpi->primary_file_demuxer = demuxer;
	ffmpi->primary_video_stream_index = MPXPLAY_STREAMTYPEINDEX_VIDEO;

	if(in_ffmpfile_filelist_file_add(ffmpi, filename, MPXPLAY_STREAMTYPEINDEX_VIDEO) < 0)
		goto err_out_chk;
	if(in_ffmpfile_file_open_by_streamindex(ffmpi, MPXPLAY_STREAMTYPEINDEX_VIDEO, -1, NULL, NULL) <= 0)
		goto err_out_chk;

	ffmpi->fctx = ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO].external_file_avctx; // just for easier primary file handling
	ffmpi->flags = ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO].control_flags;
	miis->filesize = fbfs->filelength(fbds);

	mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "inffmpg_infile_check avformat_open_input OK format:%s dur:%lld seekable:%d demuxer:%s", ffmpi->fctx->iformat->name, ffmpi->fctx->duration, (ffmpi->flags & INFFMPG_FLAG_IS_SEEKABLE), (demuxer? demuxer->name : ""));

	if((ffmpi->primary_file_openmode&MPXPLAY_INFILE_OPENMODE_INFO_DECODER) || (ffmpi->fctx->duration <= 0) || (ffmpi->flags & INFFMPG_FLAG_STREAM_PARSING)) // FIXME: hack for ff_ogg_demuxer to get ID3 infos
	{
		funcbit_disable(ffmpi->fctx->flags, (AVFMT_FLAG_NOPARSE | AVFMT_FLAG_NOFILLIN));
		if(!(ffmpi->primary_file_openmode & MPXPLAY_INFILE_OPENMODE_INFO_LENGTH))
			ffmpi->fctx->duration = (int64_t)miis->timemsec * (AV_TIME_BASE / MPXPLAY_TIME_BASE);
		ffmpi->pkt = av_packet_alloc();
		if(!ffmpi->pkt)
			goto err_out_chk;
		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && infmpeg_is_fileformat_video_stream(ffmpi->fctx))
			fbfs->fseek(fbds, 0, SEEK_SET);
//		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM)){ // TODO: live stream parsing
//			inffmp_parse_stream_header(ffmpi, 5);
//		}
//		else
		{
			if((ffmpi->primary_file_openmode&MPXPLAY_INFILE_OPENMODE_INFO_DECODER) || !(ffmpi->flags & INFFMPG_FLAG_STREAM_PARSING)){
				mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "inffmpg_infile_check avformat_find_stream_info BEGIN format:%s dur:%lld ns:%d",
									ffmpi->fctx->iformat->name, ffmpi->fctx->duration, ffmpi->fctx->nb_streams);
				//if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM)) // TODO: currently for testing only, but maybe we should replace avformat_find_stream_info (streams can be parsed at playing too)
					if(avformat_find_stream_info(ffmpi->fctx, NULL) < 0)
						goto err_out_chk;
				mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "inffmpg_infile_check avformat_find_stream_info END format:%s dur:%lld ns:%d",
					ffmpi->fctx->iformat->name, ffmpi->fctx->duration, ffmpi->fctx->nb_streams);
			}
			else if(in_ffmppars_parse_stream_header(ffmpi, 5) < 0)
				goto err_out_chk;
		}
		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && infmpeg_is_fileformat_video_stream(ffmpi->fctx))
			fbfs->fseek(fbds, 0, SEEK_SET);
		if(ffmpi->fctx->duration < AV_TIME_BASE)
			ffmpi->fctx->duration = AV_TIME_BASE;
	}

	if((ffmpi->primary_file_openmode&MPXPLAY_INFILE_OPENMODE_INFO_DECODER) || (ffmpi->flags & INFFMPG_FLAG_STREAM_PARSING))
	{
		int prog_num_curr_save;

		in_ffmpgdec_packet_queues_init(&ffmpi->codecctx_queues[0]);
		ffmpi->program_number_current = INFFMPG_INVALID_STREAM_INDEX;
		ffmpi->select_new_program_number = INFFMPG_INVALID_STREAM_INDEX;

		if(funcbit_test(ffmpi->primary_file_openmode, MPXPLAY_INFILE_OPENMODE_LOAD_PLAY))
		{
			if(in_ffmpgdec_content_has_programs(ffmpi))
			{
				mpxp_int32_t prog_id = ffmpi->miis->control_cb(ffmpi->miis->ccb_data, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_PROGID_GET, (void *)filename, NULL);
				if(prog_id > 0)
				{
					ffmpi->program_number_current = prog_id;
					funcbit_enable(ffmpi->flags, INFFMPG_FLAG_NOMETADATASEND);
				}
				else
				{
					ffmpi->program_number_current = mpxplay_control_startup_programid_select;
				}
				mpxplay_debugf(MPXPLAY_DEBUG_TEMP, "program_number_current %d", ffmpi->program_number_current);
			}
			if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && (ffmpi->streamtype_limit > MPXPLAY_STREAMTYPEINDEX_AUDIO))
			{
				if((mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_AUDIO] > MPXPLAY_EXTERNALSTREAM_LOADTYPE_SEARCH))// && ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_VIDEO]) // FIXME: ext audio streams are loaded for audio files too (codec_ctx cannot be tested here)
					in_ffmpfile_filelist_load_list_by_ext(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_AUDIO], filename, MPXPLAY_STREAMTYPEINDEX_AUDIO);
				if(mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_SUBTITLE] >= MPXPLAY_EXTERNALSTREAM_LOADTYPE_SEARCH)
					in_ffmpfile_filelist_load_list_by_ext(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_SUBTITLE], filename, MPXPLAY_STREAMTYPEINDEX_SUBTITLE);
			}
		}

		prog_num_curr_save = ffmpi->program_number_current;

		in_ffmpstrm_primary_streams_select(ffmpi);
		in_ffmpstrm_primary_streams_initialize(ffmpi, TRUE);

		if(funcbit_test(ffmpi->primary_file_openmode, MPXPLAY_INFILE_OPENMODE_LOAD_PLAY) && funcbit_test(ffmpi->flags, INFFMPG_FLAG_NOMETADATASEND) && (ffmpi->program_number_current != prog_num_curr_save))
		{
			funcbit_enable(ffmpi->flags, INFFMPG_FLAG_PROGIDCHGSEND); // the selected program cannot be played, send the new value to the upper layer
		}
	}

	if((ffmpi->flags & INFFMPG_FLAG_STREAM_PARSING) && (ffmpi->primary_file_openmode & MPXPLAY_INFILE_OPENMODE_INFO_LENGTH))
	{
		int selected_ff_stream_idx = (ffmpi->selected_stream_number[MPXPLAY_STREAMTYPEINDEX_AUDIO] >= 0)? ffmpi->selected_ff_stream_index[MPXPLAY_STREAMTYPEINDEX_AUDIO] : ffmpi->selected_ff_stream_index[MPXPLAY_STREAMTYPEINDEX_VIDEO];
		ffmpi->fctx->duration = in_ffmppars_get_stream_duration_pts(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO], selected_ff_stream_idx);
//		if(ffmpi->fctx->duration < AV_TIME_BASE)
//			if(avformat_find_stream_info(ffmpi->fctx, NULL) < 0) // FIXME: doesn't work here
//				goto err_out_chk;
	}

	if(!inffmpg_assign_values_audiovideo(ffmpi,miis,filename,ffmpi->primary_file_openmode))
		goto err_out_chk;

	if(ffmpi->primary_file_openmode & MPXPLAY_INFILE_OPENMODE_LOAD_PLAY)
	{
		ffmpi->audio_frame = av_frame_alloc();
		if(!ffmpi->audio_frame)
			goto err_out_chk;
		if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && (ffmpi->streamtype_limit > MPXPLAY_STREAMTYPEINDEX_AUDIO))
		{
			if((mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_AUDIO] == MPXPLAY_EXTERNALSTREAM_LOADTYPE_SEARCH) && ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_VIDEO])
				in_ffmpfile_filelist_load_list_by_ext(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_AUDIO], filename, MPXPLAY_STREAMTYPEINDEX_AUDIO);
		}
#ifdef INFFMP_USE_DEMUXER_MUTEX
		pds_threads_mutex_new(&ffmpi->mutexhnd_demuxer);
#endif
		pds_threads_mutex_new(&ffmpi->mutexhnd_ffmpeg);
		in_ffmpgdec_packet_queues_init(&ffmpi->packet_queues[0]);
#ifdef INFFMP_USE_DEMUXER_SEEK
		ffmpi->seek_request_pos = INFFMPG_SEEKREQUEST_INVALID_POS;
#endif
		ffmpi->avsync_timestamp = INFFMPG_INVALID_PTS;
		for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_NUM; i++)
			ffmpi->stream_last_pts[i] = INFFMPG_INVALID_PTS;
		if(funcbit_test(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_RESTORE) && funcbit_test(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_WAS_ENABLED))
		{
			ffmpi->videowall_mode_enabled = in_ffmpstrm_all_programs_videostreams_initialize(ffmpi);
		}
		if(!ffmpi->videowall_mode_enabled)
		{
			funcbit_disable(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_WAS_ENABLED);
		}
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK);
		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
		{
			infmpeg_preanalyze_and_init_video(ffmpi, fbfs, fbds);
		}
		else
		{
			avio_flush(ffmpi->fctx->pb);
			avformat_flush(ffmpi->fctx);
		}
		funcbit_enable(((struct mpxpframe_s *)fbds)->flags, MPXPLAY_MPXPFRAME_FLAG_OWN_SRVFFMV_CALLBACK); // TODO: config API
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLOSE, "ffmpg infile PLAY %8.8X fbds:%8.8X", (mpxp_ptrsize_t)ffmpi, (mpxp_ptrsize_t)fbds);
	}

	mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "inffmpg_infile_check END OK %8.8X", (mpxp_ptrsize_t)ffmpi);

	return MPXPLAY_ERROR_INFILE_OK;

err_out_chk:
	mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "inffmpg_infile_check FAILED");
	return MPXPLAY_ERROR_INFILE_CANTOPEN;
}

static void FFMPG_preinit(void)
{
	av_log_set_level(AV_LOG_QUIET);
	//av_log_set_level(AV_LOG_PANIC);
	//av_log_set_level(AV_LOG_ERROR);
	//av_log_set_level(AV_LOG_WARNING);
	//av_log_set_level(AV_LOG_INFO);
	//av_log_set_level(AV_LOG_VERBOSE);
	//av_log_set_level(AV_LOG_TRACE);
	//av_log_set_level(AV_LOG_DEBUG);
}

static void FFMPG_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis)
{
	struct mpxplay_audio_decoder_info_s *adi = miis->audio_decoder_infos;
	struct ffmpg_demuxer_data_s *ffmpi = miis->private_data;
	int i;

	if(!adi || !ffmpi || (ffmpi->struct_id != MPXPLAY_FFMPEGDEMUX_STRUCT_ID))
		return;

	funcbit_enable(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE);
#ifdef INFFMP_USE_DEMUXER_SEEK
	ffmpi->seek_request_pos = INFFMPG_SEEKREQUEST_INVALID_POS;
#endif
	if(ffmpi->mutexhnd_ffmpeg){
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLOSE, "ffmpg infile CLOSE BEGIN %8.8X nba:%d nbv:%d pn:%d", (mpxp_ptrsize_t)ffmpi,
			ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets,
			mpxplay_control_startup_programid_select);
	}
#ifdef MPXPLAY_GUI_QT
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_CALLBACKVIDEOCFG))
		mpxplay_dispqt_ffmpegvideo_config_callback(MPXPLAY_INFILE_CBKCFG_SRVFFMV_CLOSE_CALLBACK, (mpxp_ptrsize_t)in_ffmpgdec_infile_callback, (mpxp_ptrsize_t)ffmpi);
#endif
	ffmpi->struct_id = 0;
	if(ffmpi->mutexhnd_ffmpeg){
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "ffmpg infile CLOSE pds_threads_mutex_lock BEGIN");
		PDS_THREADS_MUTEX_LOCK(&ffmpi->mutexhnd_ffmpeg, INFFMPG_MUTEX_TIMEOUT);
		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD)){
			mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "ffmpg infile CLOSE mpxplay_timer_deletefunc BEGIN");
			mpxplay_timer_deletefunc(&in_ffmpdmux_read_packets, ffmpi);
		}
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "ffmpg infile CLOSE in_ffmpgdec_packet_queues_close BEGIN");
		mpxplay_ffmpgdec_queuelistelem_clear(&ffmpi->audiodec_curr_pktlist_elem);
		in_ffmpgdec_packet_queues_close(ffmpi, &ffmpi->packet_queues[0]);
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "ffmpg infile CLOSE in_ffmpgdec_packet_queues_close END");
	}
	if(ffmpi->pkt)
		av_packet_free(&ffmpi->pkt);
	if(ffmpi->audio_frame)
		av_frame_free(&ffmpi->audio_frame);
	in_ffmpgdec_packet_queues_close(ffmpi, &ffmpi->codecctx_queues[0]);

	for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_NUM; i++)
		in_ffmpfile_filelist_close(&ffmpi->external_file_infos[i]);
	in_ffmpfile_file_output_close(ffmpi);

	if(ffmpi->mutexhnd_ffmpeg){
		void *mutexhnd = ffmpi->mutexhnd_ffmpeg;
		ffmpi->mutexhnd_ffmpeg = NULL;
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "ffmpg infile CLOSE pds_threads_mutex_unlock BEGIN");
		PDS_THREADS_MUTEX_UNLOCK(&mutexhnd);
		pds_threads_mutex_del(&mutexhnd);
#ifdef INFFMP_USE_DEMUXER_MUTEX
		pds_threads_mutex_del(&ffmpi->mutexhnd_demuxer);
#endif
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLOSE, "ffmpg infile CLOSE END %8.8X", (mpxp_ptrsize_t)ffmpi);
	}
	if(miis->longname){
		pds_free(miis->longname);
		miis->longname = NULL;
	}
	if(adi->bitratetext){
		pds_free(adi->bitratetext);
		adi->bitratetext = NULL;
	}
	if(adi->chanmatrix)
	{
		pds_free(adi->chanmatrix);
		adi->chanmatrix = NULL;
	}
	pds_memset(ffmpi, 0, sizeof(*ffmpi));
	pds_free(ffmpi);
}

static int FFMPG_infile_demux(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis)
{
	struct ffmpg_demuxer_data_s *ffmpi = miis->private_data;
	struct mpxplay_streampacket_info_s *spi = miis->audio_stream;
	int retcode_demux = MPXPLAY_ERROR_INFILE_SYNC_IN, i, retcode_audio = 0;

	if(!ffmpi || (ffmpi->struct_id != MPXPLAY_FFMPEGDEMUX_STRUCT_ID) || funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
	{
		return MPXPLAY_ERROR_INFILE_EOF;
	}
	if(!ffmpi->fctx)
	{
		return retcode_demux;
	}

	if(PDS_THREADS_MUTEX_LOCK(&ffmpi->mutexhnd_ffmpeg, MPXPLAY_MUTEXTIME_SHORT) != MPXPLAY_ERROR_OK)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_ERROR,"FFMPG_infile_demux LOCK FAILED %8.8X", (mpxp_ptrsize_t)ffmpi);
		goto err_out_demux;
	}

	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
	{
		PDS_THREADS_MUTEX_UNLOCK(&ffmpi->mutexhnd_ffmpeg);
		retcode_demux = MPXPLAY_ERROR_INFILE_EOF;
		goto err_out_demux;
	}

	//mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"FFMPG_infile_demux START rd:%d", ffmpi->retcode_demux);

	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD) && !funcbit_test(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE))
	{
		ffmpi->fbfs = fbfs;
		ffmpi->fbds = fbds;
		ffmpi->miis = miis;
		in_ffmpdmux_read_packets(ffmpi);
	}

	retcode_demux = ffmpi->retcode_demux;

	retcode_audio = in_ffmpgdec_decode_audio(ffmpi, miis);
	if((retcode_audio != 0) && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets || funcbit_test(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE) || (ffmpi->select_new_stream_number[MPXPLAY_STREAMTYPEINDEX_AUDIO] == INFFMPG_STREAM_INDEX_LOST)))
	{
		spi->bs_leftbytes = PCM_OUTSAMPLES * spi->block_align;
		pds_memset(spi->bitstreambuf, 0, spi->bs_leftbytes);
		funcbit_enable(spi->flags,MPXPLAY_SPI_FLAG_SILENTBLOCK);
		mpxplay_debugf(MPXPLAY_DEBUG_SILENTB, "FFMPG_infile_demux FILL empty audio ff:%8.8X sp:%8.8X", (mpxp_ptrsize_t)ffmpi,spi->flags);
	}
	else
	{
		funcbit_disable(spi->flags,MPXPLAY_SPI_FLAG_SILENTBLOCK);
	}

	if(retcode_demux < 0)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC,"FFMPG_infile_demux %8.8X rd:%d ra:%d lb:%d nba:%d nbv:%d fl:%8.8X", (mpxp_ptrsize_t)ffmpi, retcode_demux, retcode_audio, spi->bs_leftbytes,
				ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets, ffmpi->flags);
		if(retcode_demux == AVERROR(ENOFILE))
			retcode_demux = MPXPLAY_ERROR_INFILE_EOF;
		else if(spi->bs_leftbytes || ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets || ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets)
			retcode_demux = MPXPLAY_ERROR_INFILE_OK;
		else if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) || (retcode_demux==AVERROR(ESPIPE)) || (retcode_demux==AVERROR(EAGAIN))
			|| ((retcode_demux==AVERROR(EPERM)) && funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_ASYNCREAD)))
			retcode_demux = MPXPLAY_ERROR_INFILE_SYNC_IN;
		else
			retcode_demux = MPXPLAY_ERROR_INFILE_EOF;
	}
	else if(!spi->bs_leftbytes && !ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets && !ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets)
	{
		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
			retcode_demux = MPXPLAY_ERROR_INFILE_SYNC_IN;
		else
			retcode_demux = MPXPLAY_ERROR_INFILE_NODATA;
//		mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"retcode_demux: NODATA %8.8X", (int)ffmpi);
	}
	else
	{
		retcode_demux = MPXPLAY_ERROR_INFILE_OK;
	}

//	if((retcode_demux != MPXPLAY_ERROR_INFILE_OK) || !spi->bs_leftbytes){
//		mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"FFMPG_infile_demux END %8.8X frd:%d rd:%d ra:%d lb:%d ea:%d an:%d vn:%d fpos:%d%% fl:%8.8X", (int)ffmpi, ffmpi->retcode_demux,
//				retcode_demux, retcode_audio,
//				spi->bs_leftbytes, AVERROR(EAGAIN), ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets,
//				ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets, (int)(100 * fbfs->ftell(fbds) / fbfs->filelength(fbds)), ffmpi->flags);
//	}

	if(retcode_demux != MPXPLAY_ERROR_INFILE_EOF)
	{
		ffmpi->miis = miis;
		ffmpi->fbfs = fbfs;
		ffmpi->fbds = fbds;
	}
	//mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"FFMPG_infile_demux END fbds:%8.8X ra:%d rd:%d", (unsigned int)ffmpi->fbds, retcode_audio, retcode_demux);

	PDS_THREADS_MUTEX_UNLOCK(&ffmpi->mutexhnd_ffmpeg);

err_out_demux:
	return retcode_demux;
}

static void INFFMPG_clearbuffs_all(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis,unsigned int seektype)
{
}

static void infmpeg_preanalyze_and_init_video(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_filehand_buffered_func_s *fbfs, void *fbds)
{
	int i;

	mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "infmpeg_preanalyze_and_init_video BEGIN %8.8X", (mpxp_ptrsize_t)ffmpi);

	funcbit_enable(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD);
	i = ffmpi->fctx->nb_streams * 2 + 100;
	do{
		ffmpi->fbfs = fbfs;
		ffmpi->fbds = fbds;
		in_ffmpdmux_read_packets(ffmpi);
	}while((--i) && (ffmpi->retcode_demux == 0) && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets < INFFMPG_QUEUE_PACKETNUM_FILL_MAX)
												&& (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets < INFFMPG_QUEUE_PACKETNUM_FILL_MAX));
	funcbit_disable(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD);

#ifdef MPXPLAY_GUI_QT
	if(!ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_AUDIO] && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets > 0) && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets <= INFFMPG_STILL_PICTURE_FRAME_MAX)) // FIXME: assumed still picture hack
	{
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE);
	}

	if(mpxplay_dispqt_ffmpegvideo_config_callback(MPXPLAY_INFILE_CBKCFG_SRVFFMV_OPEN_CALLBACK, (mpxp_ptrsize_t)in_ffmpgdec_infile_callback, (mpxp_ptrsize_t)ffmpi) == 0)
	{
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_CALLBACKVIDEOCFG);
	}
#endif
	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE))
	{
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD); // !!! we use it for audio too if video out is enabled
#ifdef INFFMP_USE_DEMUXER_THREAD
#ifdef INFFMP_USE_DEMUXER_BUFILL
		funcbit_enable(((struct mpxpframe_s *)fbds)->buffertype, (PREBUFTYPE_BUFFFILL_NOCHK | PREBUFTYPE_BUFFFILL_NOSKHLP)); // FIXME: API hack
#endif
		mpxplay_timer_addfunc(&in_ffmpdmux_read_packets, ffmpi, MPXPLAY_TIMERTYPE_THREAD | MPXPLAY_TIMERTYPE_REPEAT, 1);
#else
		mpxplay_timer_addfunc(&in_ffmpdmux_read_packets, ffmpi, MPXPLAY_TIMERTYPE_REPEAT, 1);
#endif
		mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "infmpeg_preanalyze_and_init_video mpxplay_timer_addfunc in_ffmpdmux_read_packets %8.8X", (mpxp_ptrsize_t)ffmpi);
	}

	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_CALLBACKVIDEOCFG))
	{
		funcbit_disable(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE);
		ffmpi->streamtype_limit = MPXPLAY_STREAMTYPEINDEX_AUDIO;
	}

	funcbit_disable(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK);

	mpxplay_debugf(MPXPLAY_DEBUG_OPENCLOSE,"infmpeg_preanalyze_and_init_video END %8.8X nba:%d nbv:%d sp:%d i:%d rd:%d", (mpxp_ptrsize_t)ffmpi,
	  ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets,
	  ((ffmpi->flags & INFFMPG_FLAG_STILLPICTURE)? 1 : 0), i, ffmpi->retcode_demux);
}

static long FFMPG_infile_fseek_av(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis,long newmpxframenum)
{
	struct ffmpg_demuxer_data_s *ffmpi = miis->private_data;
	mpxp_filesize_t newfilepos;
	mpxp_int64_t newtimepos;
	int i, seek_flags = 0, retcode;
#ifdef INFFMP_USE_DEMUXER_MUTEX
	int lock_result = -1;
#endif

	if(!ffmpi || (ffmpi->struct_id != MPXPLAY_FFMPEGDEMUX_STRUCT_ID) || funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
		return MPXPLAY_ERROR_INFILE_EOF;
	if(!ffmpi->fctx)
		return newmpxframenum;

#ifdef INFFMP_USE_DEMUXER_SEEK
	ffmpi->seek_request_pos = INFFMPG_SEEKREQUEST_INVALID_POS;
#endif

	mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "FFMPG_infile_fseek_av BEGIN frp:%8.8X ffmpi:%8.8X nf:%d", (mpxp_ptrsize_t)fbds, (mpxp_ptrsize_t)ffmpi, newmpxframenum);

	if((newmpxframenum > miis->allframes) || ((ffmpi->flags & INFFMPG_FLAG_STILLPICTURE) && (miis->seektype & MPX_SEEKTYPE_RELATIVE)))
	{
		return MPXPLAY_ERROR_INFILE_EOF;
	}
#ifdef INFFMP_USE_DEMUXER_SEEK
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_SEEK_CLEAR) && newmpxframenum && !(miis->seektype & MPX_SEEKTYPE_BOF))
	{
		return newmpxframenum;
	}
#endif
	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_SEEKABLE))
	{
		if(infmpeg_is_fileformat_video_stream(ffmpi->fctx)) // hack for DVB for faster start
			if(!newmpxframenum || (miis->seektype & MPX_SEEKTYPE_BOF))
				fbfs->fseek(fbds, newmpxframenum, SEEK_SET);  // we rewind only in the buffer
		return newmpxframenum;
	}

#ifdef INFFMP_USE_DEMUXER_MUTEX
	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK))
	{
		mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"FFMPG_infile_fseek_av LOCK begin");
		lock_result = PDS_THREADS_MUTEX_LOCK(&ffmpi->mutexhnd_demuxer, INFFMPG_MUTEX_TIMEOUT);
		mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"FFMPG_infile_fseek_av LOCK end res:%d", lock_result);
	}
#endif
	ffmpi->retcode_demux = MPXPLAY_ERROR_OK;

	ffmpi->fbfs = fbfs;
	ffmpi->fbds = fbds;

	if(miis->seektype&MPX_SEEKTYPE_BACKWARD)
		funcbit_enable(seek_flags,AVSEEK_FLAG_BACKWARD);
	if((miis->seektype&MPX_SEEKTYPE_PRECISE) || !ffmpi->nb_streams[MPXPLAY_STREAMTYPEINDEX_VIDEO])
		funcbit_enable(seek_flags,AVSEEK_FLAG_ANY);
	if(funcbit_test(miis->seektype, MPX_SEEKTYPE_PAUSE))
		funcbit_enable(seek_flags, AVSEEK_FLAG_PAUSE);

	newtimepos = (miis->allframes)? (mpxp_int64_t)((float)miis->timemsec * (float)newmpxframenum / (float)miis->allframes) * (AV_TIME_BASE/MPXPLAY_TIME_BASE) : 0;

	ffmpi->seek_request_flags = seek_flags;
	ffmpi->seek_request_pos = newtimepos;
#ifndef INFFMP_USE_DEMUXER_SEEK
	in_ffmpdmux_seek_process_all(ffmpi);
	if(ffmpi->seek_request_retcode != MPXPLAY_ERROR_OK)
	{
		newmpxframenum = ffmpi->seek_request_retcode;
		goto err_out_seek;
	}
#endif

	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK))
	{
#ifdef INFFMP_USE_DEMUXER_SEEK
		ffmpi->fbds = fbds;
		in_ffmpdmux_seek_process_all(ffmpi);
#endif
		infmpeg_preanalyze_and_init_video(ffmpi, fbfs, fbds);
	}
	else if(funcbit_test(miis->seektype, MPX_SEEKTYPE_PAUSE))
	{
		ffmpi->fbfs = fbfs;
		ffmpi->fbds = fbds;
#ifndef INFFMP_USE_DEMUXER_THREAD
		in_ffmpdmux_read_packets(ffmpi);
#endif
	}

err_out_seek:
#ifdef INFFMP_USE_DEMUXER_MUTEX
	if(lock_result == MPXPLAY_ERROR_OK)
	{
		PDS_THREADS_MUTEX_UNLOCK(&ffmpi->mutexhnd_demuxer);
	}
#endif

	mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"FFMPG_infile_fseek END np:%lld", ffmpi->seek_request_pos);

	return newmpxframenum;
}

//-------------------------------------------------------------------
static void inffmpg_tag_get_str(struct mpxplay_infile_info_s *miis, void *fbds, AVDictionaryEntry *t, unsigned int cmd, int i3i_index)
{
	int len;
	if(t && miis){
		char *str = t->value;
		if(str && str[0]){
			len = -1;
			miis->control_cb(fbds,MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_PACKFUNCTTI3I(cmd,MPXPLAY_TEXTCONV_TYPE_UTF8,i3i_index),str,&len);
		}
	}
}

void inffmpg_tags_update(struct mpxplay_infile_info_s *miis, void *fbds, AVDictionary *metadata, unsigned int cmd)
{
	AVDictionaryEntry *t;

	t = av_dict_get(metadata, "title", NULL, 0);
	inffmpg_tag_get_str(miis, fbds, t, cmd, I3I_TITLE);
	t = av_dict_get(metadata, "artist", NULL, 0);
	inffmpg_tag_get_str(miis, fbds, t, cmd, I3I_ARTIST);
	t = av_dict_get(metadata, "comment", NULL, 0);
	inffmpg_tag_get_str(miis, fbds, t, cmd, I3I_COMMENT);
	t = av_dict_get(metadata, "album", NULL, 0);
	inffmpg_tag_get_str(miis, fbds, t, cmd, I3I_ALBUM);
	t = av_dict_get(metadata, "genre", NULL, 0);
	inffmpg_tag_get_str(miis, fbds, t, cmd, I3I_GENRE);
	t = av_dict_get(metadata, "date", NULL, 0);
	inffmpg_tag_get_str(miis, fbds, t, cmd, I3I_YEAR);
	t = av_dict_get(metadata, "track", NULL, 0);
	inffmpg_tag_get_str(miis, fbds, t, cmd, I3I_TRACKNUM);
}

static int INFFMPG_tag_get(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis)
{
	struct ffmpg_demuxer_data_s *ffmpi = (struct ffmpg_demuxer_data_s *)miis->private_data;

	if(!ffmpi || (ffmpi->struct_id != MPXPLAY_FFMPEGDEMUX_STRUCT_ID) || !ffmpi->fctx || funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
		return MPXPLAY_ERROR_CFGFUNC_INVALIDDATA;

	if(ffmpi->fctx->metadata)
		inffmpg_tags_update(miis, fbds, ffmpi->fctx->metadata, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_PUT);

	return MPXPLAY_ERROR_INFILE_OK;
}

// called from application to control demuxer inside functionality without re-opening
static mpxp_int32_t inffmpg_miis_control_function(void *private_data, mpxp_uint32_t funcnum, mpxp_ptrsize_t arg1, mpxp_ptrsize_t arg2)
{
	struct ffmpg_demuxer_data_s *ffmpi = (struct ffmpg_demuxer_data_s *)private_data;
	mpxp_int32_t retval = MPXPLAY_ERROR_CFGFUNC_INVALIDDATA;

	if(!ffmpi || (ffmpi->struct_id != MPXPLAY_FFMPEGDEMUX_STRUCT_ID) || funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
		return retval;

	switch(funcnum)
	{
		case MPXPLAY_CFGFUNCNUM_INFILE_CF_NEWFILENAME_CHK:
		case MPXPLAY_CFGFUNCNUM_INFILE_CF_NEWFILENAME_SET:
		{
			if(in_ffmpgdec_content_has_programs(ffmpi))
			{
				mpxp_int32_t prog_id, freq = 0;
				int protocol_id = 0;
				prog_id = mpxplay_dtvdrv_parse_filename((char *)arg1, &protocol_id, &freq);
				if((prog_id > 0) && (protocol_id == MPXPLAY_DRVDTV_BDADEVTYPE_LOCALFILE) && (freq == MPXPLAY_DRVDTV_FREQUENCY_LOCALFILE))
				{
					if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
						break;
				}
				else
				{
					prog_id = ffmpi->miis->control_cb(ffmpi->miis->ccb_data, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_PROGID_GET, (void *)arg1, NULL); // parse prg_id from filename (arg1) in drv_dtv
				}
				if(prog_id > 0)
				{
					if(arg2)
					{
						struct mpxplay_dvb_epg_program_data_s *program_data = mpxplay_inffmp_epg_search_programdata_by_programid(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO].epg_infos, prog_id);
						if(program_data)
							*((mpxp_uint64_t *)arg2) = pds_timeval_to_seconds(program_data->current_event_duration);
					}
					if(funcnum == MPXPLAY_CFGFUNCNUM_INFILE_CF_NEWFILENAME_SET)
					{
						ffmpi->select_new_program_number = prog_id;
						if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
							funcbit_enable(ffmpi->flags, (INFFMPG_FLAG_NOMETADATASEND|INFFMPG_FLAG_NOPRGIDCHGSEND));
					}
					retval = MPXPLAY_ERROR_OK;
				}
			}
		}
		break;
		default: retval = MPXPLAY_ERROR_CFGFUNC_UNSUPPFUNC; break;
	}

	return retval;
}

static int INFFMPEG_autodetect_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,NULL,openmode);
}

static int INAAC_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_aac_demuxer,openmode);
}

static int INAPE_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_ape_demuxer,openmode);
}

static int INASF_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,NULL,openmode); // NULL because multiply parsers in FFMPEG
}

static int INAVI_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_avi_demuxer,openmode);
}

static int INDTS_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_dts_demuxer,openmode);
}

static int INAC3_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_ac3_demuxer,openmode);
}

static int INFLAC_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_flac_demuxer,openmode);
}

static int INFLV_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_flv_demuxer,openmode);
}

static int INMKV_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_matroska_demuxer,openmode);
}

static int INMOV_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_mov_demuxer,openmode);
}

static int INMP3_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_mp3_demuxer,openmode);
}

static int INMPG_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,NULL,openmode); // NULL because multiply parsers in FFMPEG
}

static int INOGG_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_ogg_demuxer,openmode);
}

static int INTS_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis,mpxp_uint32_t openmode)
{
 return inffmpg_infile_check(fbfs,fbds,filename,miis,&ff_mpegts_demuxer,openmode);
}

// scripts: AVS (avisynth), concat
// subtitles: AQT, ASS, (DC)STR, jacosub, LRC, microDVD, MCC, MPL2, SUB
// "SMI","SAMI",               // samidec.c (subtitle)

struct mpxplay_infile_func_s IN_FFMPEG_AUTODETECT={
 MPXPLAY_INFILEFUNC_FLAG_VIDEO,
 &FFMPG_preinit, NULL, &INFFMPEG_autodetect_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get,
 NULL, NULL,
 {"FFM",					// this is used for a general FFM (FFMPEG) name (FFM->CODEC)
 "4XM",						// 4xm.c
 "A64",						// a64.c
 "AAX",						// aaxdec.c
 "ACE",						// acedec.c
 "ACM",						// acm.c
 "ACT",						// act.c
 "ADP","DTK"				// adp.c, derf.c
 "ADS","SS2",				// ads.c
 "ADX",						// adxdec.c
 "AEA",						// aea.c
 "AFC",						// afc/c
 "AIF","AIFF","AFC","AIFC",	// aiffdec.c
 "AIX",						// aixdec.c
 "ALP",						// alp.c
 "AMR",						// amr.c
 "ANM",						// anm.c
 "APC",						// apc.c, apac.c
 "APM",						// apm.c
 "APNG",					// apngdec.c
 "APTX","APTXHD",			// aptxdec.c
 "CVG",						// argo_cvg.c
 "AST",						// astdec.c
 "AU",						// au.c
 "AVR",						// avr.c
#ifdef MPXPLAY_GUI_QT
 "OBU",						// av1dec.c
#endif
 "AVS",						// avs.c
#ifdef MPXPLAY_GUI_QT
 "AVS2",					// avs2dec.c
 "AVS3",					// avs3dec.c
#endif
 "VID",						// bethsoftvid.c
 "BFI",						// bfi.c
 "BIK",						// bink.c
 "IDF","ADF",				// bintext.c
 "BINKA",					// binka.c
 "BIT",						// bit.c
 "BMV",						// bmv.c
 "BOA",						// boadec.c
 "BONK",					// bonk.c
 "BRSTM","BFSTM","BCSTM",	// brstm.c
 "C93",						// c93.c
 "CAF",						// cafdec.c
 "CDG",						// cdg.c
 "CDXL","XL",				// cdxl.c
 "CINE",					// cinedec.c
 "C2",						// codec2.c
 "302","DAUD",				// dauddec.c
 "DFA",						// dfa.c
 "DAV",						// dhav.c
 "DFPWM",					// dfpwmdec.c
 "DSF",						// dsfdec.c
 "DSI",						// dsicin.c
 "DSS",						// dss.c
 "DTSHD",					// dtshddec.c
 "DV","DIF",				// dv.c
 "DXA",						// dxa.c
 "CDATA",					// eacdata.c
 "WVE","UV2",				// electronicarts.c
 "PAF","FAP",				// epafdec.c
 "FLM",						// filmstripdec.c
 "FITS",					// fitsdec.c
 "FLIC","FLI","FLC","FLX",	// flic.c
 "FWSE",					// fwse.c
 "KUX",						// flvdec.c
 "FRM",						// frmdec.c
 "FSB",						// fsb.c
 "G722","722",				// g722.c
 "TCO","RCO","G723_1",		// g723_1.c
 "G726",					// g726.c
 "G729",					// g729dec.c
 "GDV",						// gdv.c
 "GENH",					// genh.c
#ifdef MPXPLAY_GUI_QT
 "GIF",						// gifdec.c
#endif
 "GSM",						// gsmdec.c
 "GXF",						// gxf.c
#ifdef MPXPLAY_GUI_QT
 "H261",					// h261dec.c
 "H263",					// h263dec.c
 "H26L","H264","264","AVC",	// h264dec.c
 "HEVC","H265","265",		// hevc.c
#endif
 "HCA",						// hca.c
 "HCOM",					// hcom.c
 "HLS",						// hls.c
 "HNM",						// hnm.c
 "ICO",						// icodec.c  // FIXME: it seems it doesn't work (no video packet is demuxed)
 "CIN",						// idcin.c
 "ROQ",						// idroqdec.c
 "IFF",						// iff.c
 "IFV",						// ifv.c
 "LBC",						// ilbc.c
#ifdef MPXPLAY_GUI_QT
 "PIX",						// img2_alias_pix.c
 "BMP","CRI","DDS","DPX","EXR","GEM","JPG","JPEG","JPEGXL","PAM","PBM","PCX","PFM","PGM","PGX","PHM","PNG","PPM","PSD","QOI","SGI","SVG","TIF","TIFF","VBN","WEBP","XBM","XPM","XWD", // img2dec.c
#endif
 "IMF",						// imf.c
 "IMX",						// imx.c
#ifdef MPXPLAY_GUI_QT
 "CGI",						// ingenientdec.c (can be wrong)
#endif
 "MVE",						// ipmovie.c
#ifdef MPXPLAY_GUI_QT
 "IPU",						// ipudec.c
#endif
 "SF","IRCAM",				// ircamdec.c
 "ISS",						// iss.c
 "IV8",						// iv8.c
 "IVF",						// ivfdec.c
 "JV",						// jvdec.c
 "VAG",						// kvag.c
 "LAF",						// lafdec.c
 "LOAS",					// loasdec.c
 "DAT",						// luodatdec.c
 "LVF",						// lvfdec.c
 "LXF",						// lxfdec.c
 "M4V",						// m4vdec.c
 "MCA",						// mca.c
 "MGSTS",					// mgsts.c
 "J2K",                     // mj2kdec.c
 "MLP","THD",				// mlpdec.c
 "MLV",						// mlvdec.c
 "MM",						// mm.c
 "MMF",						// mmf.c
 "MODS",					// mods.c
 "MOFLEX",					// moflex.c
 "MPC",						// mpc, mpc8.c
 "MJPG",					// mpjpeg.c, mpjpegdec.c, rawdec.c, smjpegdec.c
 "MSF",						// msf.c
 "MSP",						// mspdec.c
 "MTAF",					// mtaf.c
 "MTV",						// mtv.c
 "MUSX",					// musx.c
 "MV",						// mvdec.c
 "MVI",						// mvi.c
 "MXF",						// mxfdec.c
 "MXG",						// mxg.c
 "V",						// ncdec.c
 "NIST","SPH",				// nistspheredec.c
 "NSP",						// nspdec.c
 "NSV",						// nsvdec.c
 "NUT",						// nutdec.c
 "NUV",						// nuv.c
 "OMA","OMG","AA3",			// omadec.c
 "PAF",						// paf.c
 "UW","UB","AL","UL","SLN",	// pcmdec.c
 "PJS",						// pjsdec.c
 "PMP",						// pmpdec.c
 "PVA",						// pva.c
 "PVF",						// pvfdec.c
 "QCP",						// qcp.c
 "R3D",						// r3d.c
 "MJPEG","MPO",				// rawdec.c
 "YUV","CIF","QCIF","RGB",	// rawvideodec.c
 "RSD",						// redspark.c
 "RKA",						// rka.c
 "RL2",						// rl2.c
 "RA","RM","RDT","IVR",		// rmdec.c
 "RPL",						// rpl.c
 "RSD",						// rsd.c
 "RSO",						// rsodec.c
 "SBC","MSBC",				// sbcdec.c
 "SBG",						// sbgdec.c
 "SDNS",					// sdns.c
 "SDR2",					// sdr2.c
 "SDS",                     // sdsdec.c
 "SDX",                     // sdxdec.c
 "CPK",						// segafilm.c
 "SER",						// serdec.c
#ifdef MPXPLAY_GUI_QT
 "SGA",						// sga.c
#endif
 "SHN",						// shortendec.c
 "VMD",						// sierravmd.c
 "VB","SON",				// siff.c
 "SMK",						// smacker.c
 "SAN",						// smush.c
 "SCD",						// scd.c
 "SOL",						// sol.c
 "SOX",						// soxdec.c
 "SUP",						// supdec.c
 "SVAG",					// svag.c
 "SVS",						// svs.c
 "SWF",						// swfdec.c
 "TAK",						// takdec.c
 "THP",						// thp.c
 "SEQ",						// tiertexseq.c
 "TMV",						// tmv.c
 "TTA",						// tta.c
 "TY","TY+",				// ty.c
 "V210","YUV10",			// rawvideodec.c
 "VAG",						// vag.c
 "VC1",						// vc1dec.c
 "VIV",						// vividas.c vivo.c
 "VOC",						// vocdec.c
 "VPK",						// vpk.c
 "VQF","VQL","VQE",			// vqf.c
 "WAV","W64", "CDW",		// wavdec.c
 "WC3",						// wc3movie.c
 "AUD",						// westwood_aud.c
 "VQA",						// westwood_vqa.c
 "WAY",						// wady.c
 "WA",						// wavarc.c
 "WSD",						// wsddec.c
 "WTV",						// wtvdec.c
 "WV",						// wvdec.c
 "WVE",						// wvedec.c
 "XA",						// xa.c
 "XMD",						// xmd.c
 "XMV",						// xmv.c
 "XVAG",					// xvag.c
 "XWMA",					// xwma.c
 "YOP",						// yop.c
 "Y4M",						// yuv4mpegdec.c
 NULL
 }
};

struct mpxplay_infile_func_s IN_FFMPEG_AAC_funcs={
 ( MPXPLAY_INFILEFUNC_FLAG_NOADETECT
 | MPXPLAY_TAGTYPE_PUT_SUPPORT(MPXPLAY_TAGTYPE_ID3V1|MPXPLAY_TAGTYPE_ID3V2|MPXPLAY_TAGTYPE_APETAG)
 | MPXPLAY_TAGTYPE_PUT_PRIMARY(MPXPLAY_TAGTYPE_APETAG)),
 NULL, NULL, &INAAC_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all, NULL, NULL, NULL,
 {"AAC",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_APE_funcs={
 ( MPXPLAY_INFILEFUNC_FLAG_NOADETECT
 | MPXPLAY_TAGTYPE_PUT_SUPPORT(MPXPLAY_TAGTYPE_ID3V1|MPXPLAY_TAGTYPE_ID3V2|MPXPLAY_TAGTYPE_APETAG)
 | MPXPLAY_TAGTYPE_PUT_PRIMARY(MPXPLAY_TAGTYPE_APETAG)),
 NULL, NULL, &INAPE_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all, NULL, NULL, NULL,
 {"APE","APL","MAC",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_ASF_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT | MPXPLAY_INFILEFUNC_FLAG_VIDEO | MPXPLAY_INFILEFUNC_FLAG_ONLYVIDEO,
 NULL, NULL, &INASF_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get,
 &mpxplay_ASF_infile_tag_put,
 NULL,
 {"ASF","WMV","WMA",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_AVI_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT | MPXPLAY_INFILEFUNC_FLAG_VIDEO | MPXPLAY_INFILEFUNC_FLAG_ONLYVIDEO,
 NULL, NULL, &INAVI_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get, NULL, NULL,
 {"AVI",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_AC3_funcs={
 ( MPXPLAY_INFILEFUNC_FLAG_NOADETECT
 | MPXPLAY_TAGTYPE_PUT_SUPPORT(MPXPLAY_TAGTYPE_ID3V1|MPXPLAY_TAGTYPE_APETAG)
 | MPXPLAY_TAGTYPE_PUT_PRIMARY(MPXPLAY_TAGTYPE_APETAG)),
 NULL, NULL, &INAC3_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all, NULL, NULL, NULL,
 {"AC3",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_DTS_funcs={
  ( MPXPLAY_INFILEFUNC_FLAG_NOADETECT
  | MPXPLAY_TAGTYPE_PUT_SUPPORT(MPXPLAY_TAGTYPE_ID3V1|MPXPLAY_TAGTYPE_APETAG)
  | MPXPLAY_TAGTYPE_PUT_PRIMARY(MPXPLAY_TAGTYPE_APETAG)),
 NULL, NULL, &INDTS_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all, NULL, NULL, NULL,
 {"DTS",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_FLAC_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT,
 NULL, NULL, &INFLAC_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get, &mpxplay_INFLAC_tag_put, NULL,
 {"FLAC","FLA",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_FLV_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT|MPXPLAY_INFILEFUNC_FLAG_VIDEO|MPXPLAY_INFILEFUNC_FLAG_ONLYVIDEO,
 NULL, NULL, &INFLV_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get, NULL, NULL,
 {"FLV",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_MKV_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT|MPXPLAY_INFILEFUNC_FLAG_VIDEO|MPXPLAY_INFILEFUNC_FLAG_ONLYVIDEO,
 NULL, NULL, &INMKV_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get, NULL, NULL,
 {"MKV","MK3D","MKA","MKS","WEBM",
#ifdef __DOS__
 "WEB",
#endif
 NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_MOV_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT|MPXPLAY_INFILEFUNC_FLAG_VIDEO,
 NULL, NULL, &INMOV_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get, &mpxplay_INMP4_tag_put, NULL,
 {"MOV","MP4","M4A","3GP","3G2","MJ2","PSP","ISM","ISMV","ISMA","F4V","AVIF",NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_MP3_funcs={
( MPXPLAY_INFILEFUNC_FLAG_NOADETECT
| MPXPLAY_TAGTYPE_PUT_SUPPORT(MPXPLAY_TAGTYPE_ID3V1|MPXPLAY_TAGTYPE_APETAG)
| MPXPLAY_TAGTYPE_PUT_PRIMARY(MPXPLAY_TAGTYPE_APETAG)),
 NULL, NULL, &INMP3_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all, NULL, NULL, NULL,
 {"MP3","MP2","MPA","M2A", NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_MPG_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT|MPXPLAY_INFILEFUNC_FLAG_VIDEO|MPXPLAY_INFILEFUNC_FLAG_ONLYVIDEO, // MPXPLAY_INFILEFUNC_FLAG_NOCHECK
 NULL, NULL, &INMPG_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all, NULL, NULL, NULL,
 {"MPG","MPEG","VOB","M2V",
#ifdef __DOS__
 "MPE",
#endif
 NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_OGG_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT|MPXPLAY_INFILEFUNC_FLAG_NOCHECK|MPXPLAY_INFILEFUNC_FLAG_VIDEO,
 NULL, NULL, &INOGG_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all,
 &INFFMPG_tag_get, &mpxplay_INOGG_tag_put, NULL,
 {"OGG","OGA","SPX","OPUS","OGM","OGV", // FIXME: SPX doesn't work in FFMPEG?
#ifdef __DOS__
 "OPU",
#endif
 NULL}
};

struct mpxplay_infile_func_s IN_FFMPEG_TS_funcs={
 MPXPLAY_INFILEFUNC_FLAG_NOADETECT|MPXPLAY_INFILEFUNC_FLAG_VIDEO|MPXPLAY_INFILEFUNC_FLAG_ONLYVIDEO, // MPXPLAY_INFILEFUNC_FLAG_NOCHECK
 NULL, NULL, &INTS_infile_open, &FFMPG_infile_close, &FFMPG_infile_demux, &FFMPG_infile_fseek_av, &INFFMPG_clearbuffs_all, NULL, NULL, NULL,
 {"TS","MTS","M2TS",
#ifdef __DOS__
 "M2T",
#endif
 NULL}
};

#endif //MPXPLAY_LINK_INFILE_FF_MPEG
