//**************************************************************************
//*                     This file is part of the                           *
//*                    Mpxplay-MMC - video player.                         *
//*                The source code of Mpxplay-MMC is                       *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: digital TV (DVB) program database handling

//#define MPXPLAY_USE_DEBUGF
#define MPXPLAY_DEBUG_OUTPUT     stdout

#include "mpxplay.h"

#if defined(MPXPLAY_LINK_INFILE_FF_MPEG) && defined(MPXPLAY_WIN32)

#define DTVDRIVE_LOCAL_DATABASE_NAME "DRVDTVDB.XML"

#include "dtv_drv.h"
#include "control/cntfuncs.h"

extern unsigned long mpxplay_config_dvbepg_pastevents_hours;

static void drvdtv_database_event_clear(struct mpxplay_dvb_epg_event_data_s *epg_event);
static void dtvdrv_database_program_sort_by_event_start(struct mpxplay_dvb_epg_program_data_s *prog_data);

static mpxp_bool_t drvdtv_database_is_loaded = FALSE;

//--------------------------------------------------------------------------------------------------------------------

struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_protocol_add_or_search(mpxp_uint32_t protocol_id, mpxp_bool_t search_only)
{
	struct dtvdrive_shared_data_s *shared_datas = &mpxplay_drvdtv_shared_drive_datas;
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = shared_datas->protocol_data_chain;

	if(protocol_id >= BDADEVTYPE_NUM)
		return NULL;

	if(!prot_data && !search_only) // no protocol is allocated yet
	{
		prot_data = pds_calloc(1, sizeof(*prot_data));
		if(prot_data)
		{
			prot_data->protocol_id = protocol_id;
			shared_datas->protocol_data_chain = prot_data;
			shared_datas->nb_protocols = 1;
		}
	}
	else // search for protocol_id
	{
		struct mpxplay_dvb_epg_protocol_data_s *prot_prev = NULL;
		while(prot_data)
		{
			struct mpxplay_dvb_epg_protocol_data_s *prot_next;

			if(prot_data->protocol_id == protocol_id)
			{
				break; // found existent protocol
			}

			if((prot_data->protocol_id > protocol_id) && !search_only) // insert protocol into the chain (sort the chain by protocol_id)
			{
				struct mpxplay_dvb_epg_protocol_data_s *prot_new = pds_calloc(1, sizeof(*prot_new));
				if(prot_prev)
				{
					prot_prev->next_protocol = prot_new;
				}
				else
				{
					shared_datas->protocol_data_chain = prot_new;
				}
				prot_new->protocol_id = protocol_id;
				prot_new->next_protocol = prot_data;
				prot_data = prot_new;
				shared_datas->nb_protocols++;
				break;
			}

			prot_next = prot_data->next_protocol;
			if(!prot_next && !search_only) // add new protocol to the end of the chain
			{
				struct mpxplay_dvb_epg_protocol_data_s *prot_new = pds_calloc(1, sizeof(*prot_new));
				if(prot_new)
				{
					prot_new->protocol_id = protocol_id;
					prot_data->next_protocol = prot_new;
					prot_data = prot_new;
					shared_datas->nb_protocols++;
				}
				break;
			}

			prot_prev = prot_data;
			prot_data = prot_next;
		}
	}

	return prot_data;
}

static struct mpxplay_dvb_epg_frequency_data_s *dtvdrv_database_frequency_add_or_search(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_bool_t search_only)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = mpxplay_dtvdrv_database_protocol_add_or_search(protocol_id, search_only);
	struct mpxplay_dvb_epg_frequency_data_s *freq_data = NULL;

	if(!prot_data)
		return freq_data;
	if(frequency < MPXPLAY_DRVDTV_FREQUENCY_MIN)
		return freq_data;

	freq_data = prot_data->frequency_data_chain;

	if(!freq_data && !search_only) // no freq is allocated yet
	{
		freq_data = pds_calloc(1, sizeof(*freq_data));
		if(freq_data)
		{
			freq_data->frequency = frequency;
			prot_data->frequency_data_chain = freq_data;
			prot_data->nb_frequencies = 1;
		}
	}
	else // search for frequency
	{
		while(freq_data)
		{
			struct mpxplay_dvb_epg_frequency_data_s *freq_next;

			if(freq_data->frequency == frequency)
			{
				break; // found existent frequency
			}

			freq_next = freq_data->next_frequency;
			if(!freq_next && !search_only) // add new frequency to the end of the chain
			{
				struct mpxplay_dvb_epg_frequency_data_s *freq_new = pds_calloc(1, sizeof(*freq_new));
				if(freq_new)
				{
					freq_new->frequency = frequency;
					freq_data->next_frequency = freq_new;
					freq_data = freq_new;
					prot_data->nb_frequencies++;
				}
				break;
			}

			freq_data = freq_next;
		}
	}


	return freq_data;
}

static struct mpxplay_dvb_epg_program_data_s *dtvdrv_database_program_add_or_search(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_uint32_t service_id, char *service_name, mpxp_bool_t search_only)
{
	struct mpxplay_dvb_epg_frequency_data_s *freq_data = dtvdrv_database_frequency_add_or_search(protocol_id, frequency, search_only);
	struct mpxplay_dvb_epg_program_data_s *prog_data = NULL;

	if(!freq_data)
		return prog_data;

	prog_data = freq_data->program_data_chain;

	if(!prog_data && !search_only) // no program is allocated yet
	{
		prog_data = pds_calloc(1, sizeof(*prog_data));
		if(prog_data)
		{
			prog_data->program_id = service_id;
			prog_data->current_event_duration = MPXPLAY_DVBEPG_EVENT_DURATION_MIN;
			freq_data->program_data_chain = prog_data;
			freq_data->nb_programs = 1;
		}
	}
	else // search for program_id
	{
		struct mpxplay_dvb_epg_program_data_s *prog_prev = NULL;
		while(prog_data)
		{
			struct mpxplay_dvb_epg_program_data_s *prog_next;

			if(prog_data->program_id == service_id)
			{
				break; // found existent program
			}

			if((prog_data->program_id > service_id) && !search_only) // insert program into the chain (sort the chain by service_id)
			{
				struct mpxplay_dvb_epg_program_data_s *prog_new = pds_calloc(1, sizeof(*prog_new));
				if(prog_prev)
				{
					prog_prev->next_program = prog_new;
				}
				else
				{
					freq_data->program_data_chain = prog_new;
				}
				prog_new->program_id = service_id;
				prog_new->next_program = prog_data;
				prog_new->current_event_duration = MPXPLAY_DVBEPG_EVENT_DURATION_MIN;
				prog_data = prog_new;
				freq_data->nb_programs++;
				break;
			}

			prog_next = prog_data->next_program;
			if(!prog_next && !search_only) // add new program to the end of the chain
			{
				struct mpxplay_dvb_epg_program_data_s *prog_new = pds_calloc(1, sizeof(*prog_new));
				if(prog_new)
				{
					prog_new->program_id = service_id;
					prog_new->current_event_duration = MPXPLAY_DVBEPG_EVENT_DURATION_MIN;
					prog_data->next_program = prog_new;
					prog_data = prog_new;
					freq_data->nb_programs++;
				}
				break;
			}

			prog_prev = prog_data;
			prog_data = prog_next;
		}
	}

	if(service_name && service_name[0] && prog_data && (pds_stricmp(prog_data->program_name, service_name) != 0)) // save program (service) name
	{
		int name_len = pds_strlen(service_name) + 1;
		prog_data->program_name = (char *)pds_malloc(name_len);
		if(prog_data->program_name)
			pds_memcpy(prog_data->program_name, service_name, name_len);
	}

	return prog_data;
}

struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_protocol_new(mpxp_uint32_t protocol_id)
{
	return mpxplay_dtvdrv_database_protocol_add_or_search(protocol_id, FALSE);
}

struct mpxplay_dvb_epg_frequency_data_s *mpxplay_dtvdrv_database_frequency_search(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency)
{
	return dtvdrv_database_frequency_add_or_search(protocol_id, frequency, TRUE);
}

struct mpxplay_dvb_epg_program_data_s *mpxplay_dtvdrv_database_program_search(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_uint32_t service_id)
{
	return dtvdrv_database_program_add_or_search(protocol_id, frequency, service_id, NULL, TRUE);
}

struct mpxplay_dvb_epg_program_data_s *mpxplay_dtvdrv_database_program_new(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_uint32_t service_id, char *service_name)
{
	return dtvdrv_database_program_add_or_search(protocol_id, frequency, service_id, service_name, FALSE);
}

//-----------------------------------------------------------------------------------------------------------------------------------------

// check all events, remove events which begins before the end of previous event (FIXME: maybe this is not required (can be removed), if runtime functions (bellow) are correct)
static void dtvdrv_database_event_check_all_starttime_validity(struct mpxplay_dvb_epg_program_data_s *prog_data)
{
	struct mpxplay_dvb_epg_event_data_s *event_data, *event_prev = NULL;
	mpxp_int64_t prev_event_end_secs = 0;

	if(!prog_data)
		return;

	event_data = prog_data->epg_program_event_chain;
	while(event_data)
	{
		const mpxp_int64_t event_start_secs = pds_datetimeval_to_seconds(event_data->event_begin_date_time);
		struct mpxplay_dvb_epg_event_data_s *event_next = event_data->next_event;
		if(event_start_secs < prev_event_end_secs) // event begins before the end of previous event (means invalid event) -> remove it
		{
			if(event_prev)
			{
				event_prev->next_event = event_next;
			}
			else
			{
				prog_data->epg_program_event_chain = event_next;
			}
			drvdtv_database_event_clear(event_data);
		}
		else
		{
			event_prev = event_data;
			prev_event_end_secs = event_start_secs + pds_timeval_to_seconds(event_data->event_duration_time);
		}
		event_data = event_next;
	}
}

static struct mpxplay_dvb_epg_event_data_s *dtvdrv_database_event_search_by_eventid(struct mpxplay_dvb_epg_program_data_s *prog_data, mpxp_uint32_t event_id)
{
	struct mpxplay_dvb_epg_event_data_s *event_data = prog_data->epg_program_event_chain;
	while(event_data)
	{
		if(event_data->event_id == event_id)
			break;
		event_data = event_data->next_event;
	}

	return event_data;
}

// returns false, if new event begins before the end of previous event (means invalid new event)
static mpxp_bool_t dtvdrv_database_event_check_new_starttime_validity(struct mpxplay_dvb_epg_event_data_s *prev_event_data, mpxp_uint64_t new_event_start_local_datetime, mpxp_uint32_t event_duration)
{
	if(prev_event_data)
	{
		const mpxp_int64_t prev_event_start_secs = pds_datetimeval_to_seconds(prev_event_data->event_begin_date_time);
		const mpxp_int64_t prev_event_end_secs = prev_event_start_secs + pds_timeval_to_seconds(prev_event_data->event_duration_time);
		const mpxp_int64_t new_event_start_abs_secs = pds_datetimeval_to_seconds(new_event_start_local_datetime);
		const mpxp_int64_t new_event_end_abs_secs = new_event_start_abs_secs + pds_timeval_to_seconds(event_duration);

		if((new_event_start_abs_secs >= prev_event_start_secs) && (new_event_start_abs_secs < prev_event_end_secs) && (new_event_end_abs_secs <= prev_event_end_secs))
			return FALSE;
	}
	return TRUE;
}

// scan events and remove event, if the time range is in the range of new event (at the case of start time or duration of events have changed)
// returns FALSE if the new event seems to be incorrect (part of an existent event -> we don't display these)
static mpxp_bool_t dtvdrv_database_event_check_and_remove_obsolete_events(struct mpxplay_dvb_epg_program_data_s *prog_data, struct mpxplay_dvb_epg_event_data_s *keep_event_data,
		mpxp_uint64_t event_start_local_datetime, mpxp_uint32_t event_duration)
{
	struct mpxplay_dvb_epg_event_data_s *event_data = prog_data->epg_program_event_chain;
	struct mpxplay_dvb_epg_event_data_s *event_prev = NULL;
	const mpxp_int64_t new_event_start_abs_secs = pds_datetimeval_to_seconds(event_start_local_datetime);
	const mpxp_int64_t new_event_end_abs_secs = new_event_start_abs_secs + pds_timeval_to_seconds(event_duration);
	mpxp_bool_t newevent_is_valid = TRUE;

	while(event_data)
	{
		struct mpxplay_dvb_epg_event_data_s *event_next = event_data->next_event;
		const mpxp_int64_t event_start_secs = pds_datetimeval_to_seconds(event_data->event_begin_date_time);
		const mpxp_int64_t event_end_secs = event_start_secs + pds_timeval_to_seconds(event_data->event_duration_time);

		if(event_data != keep_event_data) // if this event is not the new event
		{
			mpxp_bool_t wrong_beginpos = ((event_start_secs >= new_event_start_abs_secs) && (event_start_secs <= new_event_end_abs_secs)); // if the begin pos of this event is in the range of new event
			mpxp_bool_t wrong_endpos = ((event_end_secs > new_event_start_abs_secs) && (event_end_secs < new_event_end_abs_secs));         // or if the end pos of this event is in the range of new event
			mpxp_bool_t new_is_partof = ((event_start_secs <= new_event_start_abs_secs) && (event_end_secs >= new_event_end_abs_secs));    // new event is part of old event
			if(!new_is_partof && (wrong_beginpos || wrong_endpos)) // remove this event
			{
				if(event_prev)
				{
					event_prev->next_event = event_next;
				}
				else
				{
					prog_data->epg_program_event_chain = event_next;
				}
				drvdtv_database_event_clear(event_data);
			}
			else
			{
				if(new_is_partof)
				{
					newevent_is_valid = FALSE;
				}
				event_prev = event_data;
			}
		}
		else
		{
			event_prev = event_data;
		}
		event_data = event_next;
	}

	return newevent_is_valid;
}

/* create new event entry for the program by date, begin_time and duration; if entry already exists the function gives back pointer to the existent event */
struct mpxplay_dvb_epg_event_data_s *mpxplay_dtvdrv_database_event_add(struct mpxplay_dvb_epg_program_data_s *prog_data, mpxp_uint32_t event_id, mpxp_uint64_t event_start_local_datetime, mpxp_uint32_t event_duration, mpxp_bool_t live_stream_source)
{
	struct mpxplay_dvb_epg_event_data_s *event_data;

	if(!prog_data)
		return NULL;

	event_data = prog_data->epg_program_event_chain;
	if(!event_data) // no any event is allocated yet
	{
		event_data = pds_calloc(1, sizeof(*event_data));
		if(event_data)
		{
			event_data->event_id = event_id;
			event_data->event_begin_date_time = event_start_local_datetime;
			event_data->event_duration_time = event_duration;
			prog_data->epg_program_event_chain = event_data;
			prog_data->nb_program_events = 1;
		}
	}
	else if(live_stream_source && event_id && ((event_data = dtvdrv_database_event_search_by_eventid(prog_data, event_id)) != NULL)) // search for event by event_id
	{
		if(!event_start_local_datetime || !event_duration || ((event_data->event_begin_date_time == event_start_local_datetime) && (event_data->event_duration_time == event_duration)))
		{
			live_stream_source = FALSE; // we don't post-process this event
		}
		else
		{
			event_data->event_begin_date_time == event_start_local_datetime;
			event_data->event_duration_time = event_duration;
			dtvdrv_database_program_sort_by_event_start(prog_data); // TODO: optimize for one elem sort
		}
	}
	else // search for event by date & time
	{
		struct mpxplay_dvb_epg_event_data_s *event_prev = NULL;
		event_data = prog_data->epg_program_event_chain;

		while(event_data)
		{
			struct mpxplay_dvb_epg_event_data_s *event_next;

			if(event_data->event_begin_date_time == event_start_local_datetime)
			{
				if(event_data->event_duration_time == event_duration)
				{
					live_stream_source = FALSE; // we don't post-process this event
				}
				else if(dtvdrv_database_event_check_new_starttime_validity(event_prev, event_start_local_datetime, event_duration))
				{
					event_data->event_id = event_id;
					event_data->event_begin_date_time == event_start_local_datetime;
					event_data->event_duration_time = event_duration;
					dtvdrv_database_program_sort_by_event_start(prog_data); // TODO: optimize for one elem sort
				}
				else
				{
					event_data = NULL;
				}
				break; // found existent event in the chain
			}

			if(event_data->event_begin_date_time > event_start_local_datetime)  // insert event into the chain
			{
				if(dtvdrv_database_event_check_new_starttime_validity(event_prev, event_start_local_datetime, event_duration))
				{
					struct mpxplay_dvb_epg_event_data_s *event_new = pds_calloc(1, sizeof(*event_new));
					if(event_new)
					{
						event_new->event_id = event_id;
						event_new->event_begin_date_time = event_start_local_datetime;
						event_new->event_duration_time = event_duration;
						event_new->next_event = event_data;
						if(event_prev)
						{
							event_prev->next_event = event_new;
						}
						else
						{
							prog_data->epg_program_event_chain = event_new;
						}
						prog_data->nb_program_events++;
					}
					event_data = event_new;
				}
				else
				{
					event_data = NULL;
				}
				break;
			}

			event_next = event_data->next_event;
			if(!event_next)  // not found in the chain, add new event to the end of the chain
			{
				struct mpxplay_dvb_epg_event_data_s *event_new = pds_calloc(1, sizeof(*event_new));
				if(event_new)
				{
					event_new->event_id = event_id;
					event_new->event_begin_date_time = event_start_local_datetime;
					event_new->event_duration_time = event_duration;
					event_data->next_event = event_new;
					prog_data->nb_program_events++;
				}
				event_data = event_new;
				break;
			}

			event_prev = event_data;
			event_data = event_next;
		}
	}

	if(live_stream_source && event_data)
		if(!dtvdrv_database_event_check_and_remove_obsolete_events(prog_data, event_data, event_start_local_datetime, event_duration))
			event_data = NULL;

	return event_data;
}

// add short event descriptor (desc_tag = 0x4d) data to the event (it doesn't update the existent datas)
void mpxplay_dtvdrv_database_event_add_short_description(struct mpxplay_dvb_epg_event_data_s *event_data, char *event_name, char *event_description)
{
	if(!event_data)
		return;

	if(!event_data->event_shortdesc_name || (pds_strcmp(event_data->event_shortdesc_name, event_name) != 0))
	{
		unsigned int len = pds_strlen(event_name);
		if(len)
		{
			char *str_ptr = event_data->event_shortdesc_name;
			if(str_ptr) // clear all text descriptors at event_name change
			{
				event_data->event_shortdesc_name = NULL;
				pds_free(str_ptr);

				str_ptr = event_data->event_shortdesc_details;
				if(str_ptr)
				{
					event_data->event_shortdesc_details = NULL;
					pds_free(str_ptr);
				}

				str_ptr = event_data->event_extended_desc;
				if(str_ptr)
				{
					event_data->event_extended_desc = NULL;
					pds_free(str_ptr);
				}
				event_data->extdesc_counter = 0;
			}
			str_ptr = pds_malloc(len + 1);
			if(str_ptr)
			{
				pds_memcpy(str_ptr, event_name, len);
				str_ptr[len] = 0;
				event_data->event_shortdesc_name = str_ptr;
			}
		}
	}

	if(!event_data->event_shortdesc_details)
	{
		unsigned int len = pds_strlen(event_description);
		if(len)
		{
			char *str_ptr = pds_malloc(len + 1);
			if(str_ptr)
			{
				pds_memcpy(str_ptr, event_description, len);
				str_ptr[len] = 0;
				event_data->event_shortdesc_details = str_ptr;
			}
		}
	}
}

// add extended event description (0x4e) data to the event (append new string(s) sequentially to the existent one(s), controlled by extdesc_counter in inffmp_epg_process_section_eit)
// if 'append' argument is FALSE, the old string is simply replaced with the new one
mpxp_bool_t mpxplay_dtvdrv_database_event_add_extended_description(struct mpxplay_dvb_epg_event_data_s *event_data, char *event_ext_description, mpxp_bool_t append)
{
	unsigned int desclen, addlen;
	char *olddesc, *newdesc;

	if(!event_data)
		return FALSE;

	// calculate length of new string
	addlen = pds_strlen(event_ext_description);
	if(!addlen)
		return FALSE;

	// calculate the length of old (existent) data (considering 'append' argument)
	desclen = (append)? pds_strlen(event_data->event_extended_desc) : 0;

	// alloc new field for merged data
	newdesc = (char *)pds_malloc(desclen + addlen + 1);
	if(!newdesc)
		return FALSE;

	// merge old (if exists) and new datas
	if(desclen)
		pds_memcpy(newdesc, event_data->event_extended_desc, desclen);
	pds_memcpy(&newdesc[desclen], event_ext_description, addlen);
	newdesc[desclen + addlen] = 0;

	// delete old data, set new
	olddesc = event_data->event_extended_desc;
	event_data->event_extended_desc = newdesc;
	pds_free(olddesc);

	return TRUE;
}

/* add parental_rating (desc_tag = 0x55) value to the event  */
void mpxplay_dtvdrv_database_event_add_parental_rating(struct mpxplay_dvb_epg_event_data_s *event_data, mpxp_uint8_t parental_rate)
{
	if(!event_data)
		return;
	event_data->parental_rating = parental_rate;
}

//---------------------------------------------------------------------------------------------------------------
static void dtvdrv_database_protocols_set_default(void)
{
	unsigned int protocol_id;
	for(protocol_id = 0; protocol_id < BDADEVTYPE_NUM; protocol_id++)
	{
		struct mpxplay_dvb_epg_protocol_data_s *prot_data = mpxplay_dtvdrv_database_protocol_new(protocol_id);
		if(prot_data)
		{
			switch(protocol_id)
			{
				case BDADEVTYPE_DVB_T:
				case BDADEVTYPE_DVB_T2:
				default:
					prot_data->scan_freq_begin = MPXPLAY_DRVDTV_FREQUENCY_SCAN_BEGIN; // TODO: individual scan range for each protocols
					prot_data->scan_freq_end = MPXPLAY_DRVDTV_FREQUENCY_SCAN_END;
			}
		}
	}
}

// sort frequencies by program id (of first program elem of frequencies)
void mpxplay_dtvdrv_database_frequencies_sort_by_type(mpxp_uint32_t protocol_id, unsigned int sort_type)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = mpxplay_dtvdrv_database_protocol_add_or_search(protocol_id, TRUE);
	unsigned int i;

	if(!prot_data || !prot_data->frequency_data_chain || (prot_data->nb_frequencies <= 1))
		return;

	for(i = 0; i < (prot_data->nb_frequencies - 1); i++)
	{
		struct mpxplay_dvb_epg_frequency_data_s *freq_prev = NULL, *freq_curr = prot_data->frequency_data_chain;
		while(freq_curr)
		{
			struct mpxplay_dvb_epg_frequency_data_s *freq_next = freq_curr->next_frequency;
			mpxp_bool_t change_order = FALSE;

			if(!freq_next)
				break;

			switch(sort_type)
			{
				case MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORT_BY_PROGID:
				{
					struct mpxplay_dvb_epg_program_data_s *prog_curr = freq_curr->program_data_chain;
					struct mpxplay_dvb_epg_program_data_s *prog_next = freq_next->program_data_chain;
					if(prog_curr && prog_next && (prog_curr->program_id > prog_next->program_id))
						change_order = TRUE;
					break;
				}
				case MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORT_BY_FREQ:
					if(freq_curr->frequency > freq_next->frequency)
						change_order = TRUE;
					break;
			}

			if(change_order)
			{
				if(freq_prev)
					freq_prev->next_frequency = freq_next;
				else
					prot_data->frequency_data_chain = freq_next;
				freq_curr->next_frequency = freq_next->next_frequency;
				freq_next->next_frequency = freq_curr;
				freq_prev = freq_next;
			}
			else
			{
				freq_prev = freq_curr;
				freq_curr = freq_next;
			}
		}
	}
}

// sort frequencies by program id (of first program elem of frequencies)
static void dtvdrv_database_program_sort_by_event_start(struct mpxplay_dvb_epg_program_data_s *prog_data)
{
	unsigned int i;

	if(!prog_data || !prog_data->epg_program_event_chain || (prog_data->nb_program_events <= 1))
		return;

	for(i = 0; i < (prog_data->nb_program_events - 1); i++)
	{
		struct mpxplay_dvb_epg_event_data_s *event_data = prog_data->epg_program_event_chain;
		struct mpxplay_dvb_epg_event_data_s *event_prev = NULL;
		while(event_data)
		{
			struct mpxplay_dvb_epg_event_data_s *event_next = event_data->next_event;
			if(!event_next)
				break;

			if(event_data->event_begin_date_time > event_next->event_begin_date_time)
			{
				if(event_prev)
					event_prev->next_event = event_next;
				else
					prog_data->epg_program_event_chain = event_next;
				event_data->next_event = event_next->next_event;
				event_next->next_event = event_data;
				event_prev = event_next;
			}
			else
			{
				event_prev = event_data;
				event_data = event_next;
			}
		}
	}
}

//---------------------------------------------------------------------------------------------------------------
// duplicate event informations into a new epg_event_data field
static void dtvdrv_database_events_copy(struct mpxplay_dvb_epg_program_data_s *prog_dest, struct mpxplay_dvb_epg_event_data_s *events_src)
{
	struct mpxplay_dvb_epg_event_data_s *events_dest = NULL;
	while(events_src)
	{
		struct mpxplay_dvb_epg_event_data_s *event_new = pds_calloc(1, sizeof(*event_new));
		unsigned int len;

		if(!event_new)
			break;
		if(!events_dest)
			prog_dest->epg_program_event_chain = event_new;
		else
			events_dest->next_event = event_new;

		event_new->event_id = events_src->event_id;
		event_new->event_begin_date_time = events_src->event_begin_date_time;
		event_new->event_duration_time = events_src->event_duration_time;
		event_new->parental_rating = events_src->parental_rating;
		event_new->event_flags = events_src->event_flags;

		len = pds_strlen(events_src->event_shortdesc_name);
		if(len)
		{
			len++;
			event_new->event_shortdesc_name = (char *)pds_malloc(len);
			if(event_new->event_shortdesc_name)
				pds_memcpy(event_new->event_shortdesc_name, events_src->event_shortdesc_name, len);
		}
		len = pds_strlen(events_src->event_shortdesc_details);
		if(len)
		{
			len++;
			event_new->event_shortdesc_details = (char *)pds_malloc(len);
			if(event_new->event_shortdesc_details)
				pds_memcpy(event_new->event_shortdesc_details, events_src->event_shortdesc_details, len);
		}
		len = pds_strlen(events_src->event_extended_desc);
		if(len)
		{
			len++;
			event_new->event_extended_desc = (char *)pds_malloc(len);
			if(event_new->event_extended_desc)
				pds_memcpy(event_new->event_extended_desc, events_src->event_extended_desc, len);
		}
		prog_dest->nb_program_events++;
		events_dest = event_new;
		events_src = events_src->next_event;
	}
}

// duplicate program informations into a new epg_program_data field (except current_event_name and current_event_duration)
static void dtvdrv_database_programs_copy(struct mpxplay_dvb_epg_frequency_data_s *freq_dest, struct mpxplay_dvb_epg_frequency_data_s *freq_src, unsigned long control_flags)
{
	struct mpxplay_dvb_epg_program_data_s *prog_src = freq_src->program_data_chain;
	struct mpxplay_dvb_epg_program_data_s *prog_dest = NULL;
	while(prog_src)
	{
		struct mpxplay_dvb_epg_program_data_s *prog_new;

		if(!mpxplay_dtvdrv_database_check_program_validity(freq_src, prog_src, control_flags))
		{
			prog_src = prog_src->next_program;
			continue;
		}

		prog_new = pds_calloc(1, sizeof(*prog_new));
		unsigned int len;
		if(!prog_new)
			break;
		if(!prog_dest)
			freq_dest->program_data_chain = prog_new;
		else
			prog_dest->next_program = prog_new;
		prog_new->program_id = prog_src->program_id;
		prog_new->channel_id = prog_src->channel_id;
		prog_new->ctrl_flags = prog_src->ctrl_flags;
		len = pds_strlen(prog_src->program_name);
		if(len)
		{
			len++;
			prog_new->program_name = (char *)pds_malloc(len);
			if(prog_new->program_name)
				pds_memcpy(prog_new->program_name, prog_src->program_name, len);
		}

		dtvdrv_database_events_copy(prog_new, prog_src->epg_program_event_chain);

		freq_dest->nb_programs++;
		prog_dest = prog_new;
		prog_src = prog_src->next_program;
	}
}

// duplicate frequency informations into a new dvb_epg_frequency field
static void dtvdrv_database_frequencies_copy(struct mpxplay_dvb_epg_protocol_data_s *prot_dest, struct mpxplay_dvb_epg_frequency_data_s *freq_src, unsigned long control_flags)
{
	struct mpxplay_dvb_epg_frequency_data_s *freq_dest_chain = NULL;
	while(freq_src)
	{
		if(funcbit_test(freq_src->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT)) // TODO: check
		{
			struct mpxplay_dvb_epg_frequency_data_s *freq_new = pds_calloc(1, sizeof(*freq_new));
			if(!freq_new)
				break;
			if(!freq_dest_chain)
				prot_dest->frequency_data_chain = freq_new;
			else
				freq_dest_chain->next_frequency = freq_new;
			freq_new->frequency = freq_src->frequency;
			freq_new->freq_flags = freq_src->freq_flags;
			freq_new->freq_signal_quality = freq_src->freq_signal_quality;

			dtvdrv_database_programs_copy(freq_new, freq_src, control_flags);

			prot_dest->nb_frequencies++;
			freq_dest_chain = freq_new;
		}
		freq_src = freq_src->next_frequency;
	}
}

// copy all protocol informations from database into a new protocol structure, used by epg dialog (note: we create/send empty protocol info too)
struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_protocol_all_info(unsigned long control_flags)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_src = mpxplay_drvdtv_shared_drive_datas.protocol_data_chain;
	struct mpxplay_dvb_epg_protocol_data_s *prot_dest_begin = NULL, *prot_dest = NULL;

	do
	{
		struct mpxplay_dvb_epg_protocol_data_s *prot_new = pds_calloc(1, sizeof(*prot_dest)); // empty protocol infos is also sent
		if(!prot_new)
			break;

		if(!prot_dest_begin)
			prot_dest_begin = prot_new;
		else
			prot_dest->next_protocol = prot_new;
		prot_dest = prot_new;

		if(!prot_src) // empty protocol infos is also sent
			break;

		prot_dest->protocol_id = prot_src->protocol_id;

		dtvdrv_database_frequencies_copy(prot_dest, prot_src->frequency_data_chain, control_flags);

		prot_src = prot_src->next_protocol;
	}while(prot_src);

	return prot_dest_begin;
}

// copy one protocol informations from database into a new protocol structure, used by scan finish
struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_dup_protocol_info(mpxp_uint32_t protocol_id, unsigned long control_flags)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_src, *prot_dest;

	prot_src = mpxplay_dtvdrv_database_protocol_add_or_search(protocol_id, TRUE);
	prot_dest = pds_calloc(1, sizeof(*prot_dest)); // empty protocol infos is also sent

	if(!prot_src || !prot_dest)
		return prot_dest;

	prot_dest->protocol_id = prot_src->protocol_id;

	dtvdrv_database_frequencies_copy(prot_dest, prot_src->frequency_data_chain, control_flags);

	return prot_dest;
}

#ifdef MPXPLAY_GUI_QT
// copy program events of a program into a new protocol structure, used by inffmp_epg_send_events_to_epgdialog (local file EPG)
struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_dup_programs_into_protocol(struct mpxplay_dvb_epg_program_data_s *prog_src)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = NULL;
	struct mpxplay_dvb_epg_program_data_s *prog_dest;
	struct mpxplay_dvb_epg_frequency_data_s freq_src;

	if(!prog_src)
		return prot_data;

	prot_data = pds_calloc(1, sizeof(struct mpxplay_dvb_epg_protocol_data_s));
	if(!prot_data)
		return prot_data;

	prot_data->frequency_data_chain = pds_calloc(1, sizeof(struct mpxplay_dvb_epg_frequency_data_s));
	if(!prot_data->frequency_data_chain)
		goto err_out_copy_events;

	pds_memset(&freq_src, 0 ,sizeof(freq_src));
	freq_src.program_data_chain = prog_src;

	dtvdrv_database_programs_copy(prot_data->frequency_data_chain, &freq_src, 0);

	return prot_data;

err_out_copy_events:
	mpxplay_dtvdrv_database_clear_protocols(prot_data);
	return NULL;
}
#endif // MPXPLAY_GUI_QT

// copy selected frequency informations from database into a new protocol structure, used by scan process (note: we create/send empty freq info too)
struct mpxplay_dvb_epg_protocol_data_s *mpxplay_drvdtv_database_protocol_frequency_info(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, unsigned long control_flags)
{
	struct mpxplay_dvb_epg_protocol_data_s *dest_prot = pds_calloc(1, sizeof(*dest_prot));
	struct mpxplay_dvb_epg_frequency_data_s *dest_freq, *src_freq;

	if(!dest_prot)
		return dest_prot;

	dest_prot->protocol_id = protocol_id;
	dest_freq = pds_calloc(1, sizeof(*dest_prot->frequency_data_chain));
	if(!dest_freq)
		return dest_prot;
	dest_prot->frequency_data_chain = dest_freq;
	dest_freq->frequency = frequency;
	dest_prot->nb_frequencies = 1;

	src_freq = dtvdrv_database_frequency_add_or_search(protocol_id, frequency, TRUE);
	if(!src_freq)
		return dest_prot;

	dest_freq->freq_signal_quality = src_freq->freq_signal_quality;

	dtvdrv_database_programs_copy(dest_freq, src_freq, control_flags);

	return dest_prot;
}

// filter program entry by control flags (encrypted or non-media)
mpxp_bool_t mpxplay_dtvdrv_database_check_program_validity(struct mpxplay_dvb_epg_frequency_data_s *freq_data, struct mpxplay_dvb_epg_program_data_s *prog_data, unsigned long control_flags)
{
	if(!freq_data || !prog_data)
		return FALSE;

	if( funcbit_test(control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_FILTER_ENCRYPTED)
	 && funcbit_test(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_ENCRYPTED_PRG)
	){
		return FALSE;
	}

	if( funcbit_test(control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_FILTER_NONMEDIA)
	 && (  funcbit_test(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_NOTRUNNING_PRG)
	   || (funcbit_test(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT) && !prog_data->program_name)
	   || (funcbit_test(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_NIT) && !prog_data->channel_id) )
	){
		return FALSE;
	}

	return TRUE;
}

// check completeness/validity of the loading of the programs of a frequency
mpxp_bool_t mpxplay_drvdtv_database_frequency_programs_check(mpxplay_dvb_epg_frequency_data_s *freq_data)
{
	if(!freq_data || !freq_data->nb_programs)
		return FALSE;
	if(!funcbit_test(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT) || !funcbit_test(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_NIT))
		return FALSE;
	return TRUE;
}

// check completeness/continuity the loaded events of all programs of a frequency, after the current time
mpxp_bool_t mpxplay_drvdtv_database_frequency_programs_events_check(mpxplay_dvb_epg_frequency_data_s *freq_data)
{
	struct mpxplay_dvb_epg_program_data_s *prog_data;
	mpxp_uint64_t curr_datetime_val;

	if(!freq_data || !funcbit_test(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT))
		return FALSE;

	curr_datetime_val = pds_getdatetime();
	prog_data = freq_data->program_data_chain;
	while(prog_data)
	{
		if(prog_data->epg_program_event_chain)
		{
			struct mpxplay_dvb_epg_event_data_s *event_data = prog_data->epg_program_event_chain;
			mpxp_int64_t prev_end_secs = 0LL;
			while(event_data)
			{
				if(event_data->event_begin_date_time >= curr_datetime_val) // we don't care with the obsolete/past events (because they probably will not get new events to insert)
				{
					const mpxp_int64_t event_start_secs = pds_datetimeval_to_seconds(event_data->event_begin_date_time);
					if(prev_end_secs)
					{
						if(event_start_secs > (prev_end_secs + 30)) // this and previous events have gap (event is missing between them)
						{
#ifdef MPXPLAY_USE_DEBUGF
							int years = event_data->event_begin_date_time >> 48;
							int month = (event_data->event_begin_date_time >> 40) & 0xff;
							int day = (event_data->event_begin_date_time >> 32) & 0xff;
							int hour = (event_data->event_begin_date_time >> 16) & 0xff;
							int minute = (event_data->event_begin_date_time >> 8) & 0xff;
							mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "CHECK EVENT_START %s %s %4d.%2.2d.%2.2d %2d:%2.2d",
									prog_data->program_name, event_data->event_shortdesc_name,
									years, month, day, hour, minute);
#endif
							return FALSE;
						}
					}
					prev_end_secs = event_start_secs + pds_timeval_to_seconds(event_data->event_duration_time);
				}
				event_data = event_data->next_event;
			}
			if((prev_end_secs - pds_datetimeval_to_seconds(curr_datetime_val)) < (MPXPLAY_DRVDTV_EVENTS_SCAN_MIN_HOURS * 60 * 60)) // last found event is too early (wait for more)
			{
				mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "CHECK EARLY %s", prog_data->program_name);
				return FALSE;
			}
		}
		else if(prog_data->channel_id) // this seems to be a valid program, it shall contain events (but it doesn't contain yet)
		{
			mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "CHECK CHANID %s", prog_data->program_name);
			return FALSE;
		}
		prog_data = prog_data->next_program;
	}

	return TRUE;
}

//--------------------------------------------------------------------------------------------------------------------

// clear one event data
static void drvdtv_database_event_clear(struct mpxplay_dvb_epg_event_data_s *epg_event)
{
	char *short_name = epg_event->event_shortdesc_name;
	char *short_details = epg_event->event_shortdesc_details;
	char *ext_desc = epg_event->event_extended_desc;
	pds_memset(epg_event, 0, sizeof(*epg_event));
	if(short_name)
		pds_free(short_name);
	if(short_details)
		pds_free(short_details);
	if(ext_desc)
		pds_free(ext_desc);
	pds_free(epg_event);
}

// dealloc the event chain and the related descriptor datas
void mpxplay_dtvdrv_database_clear_events(struct mpxplay_dvb_epg_event_data_s *epg_event)
{
	while(epg_event)
	{
		struct mpxplay_dvb_epg_event_data_s *next_event = epg_event->next_event;
		drvdtv_database_event_clear(epg_event);
		epg_event = next_event;
	}
}

// dealloc the program chain and the related datas
void mpxplay_dtvdrv_database_clear_programs(struct mpxplay_dvb_epg_program_data_s *prog_data)
{
	while(prog_data)
	{
		struct mpxplay_dvb_epg_program_data_s *next_program = prog_data->next_program;
		struct mpxplay_dvb_epg_event_data_s *event_data = prog_data->epg_program_event_chain;
		char *prg_name = prog_data->program_name, *event_name = prog_data->current_event_name;

		pds_memset(prog_data, 0, sizeof(*prog_data));

		mpxplay_dtvdrv_database_clear_events(event_data);

		if(prg_name)
			pds_free(prg_name);
		if(event_name)
			pds_free(event_name);
		pds_free(prog_data);
		prog_data = next_program;
	}
}

// dealloc the frequency chain and the related datas
static void dtvdrv_database_clear_frequencies(struct mpxplay_dvb_epg_frequency_data_s *freq_data)
{
	while(freq_data)
	{
		struct mpxplay_dvb_epg_frequency_data_s *next_frequency = freq_data->next_frequency;
		struct mpxplay_dvb_epg_program_data_s *prog_data = freq_data->program_data_chain;
		pds_memset(freq_data, 0, sizeof(*freq_data));
		mpxplay_dtvdrv_database_clear_programs(prog_data);
		pds_free(freq_data);
		freq_data = next_frequency;
	}
}

static void dtvdrv_clear_freqencies_of_protocol(mpxp_uint32_t protocol_id)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = mpxplay_dtvdrv_database_protocol_add_or_search(protocol_id, TRUE);
	if(prot_data)
	{
		struct mpxplay_dvb_epg_frequency_data_s *freq_data = prot_data->frequency_data_chain;
		prot_data->frequency_data_chain = NULL;
		prot_data->nb_frequencies = 0;
		dtvdrv_database_clear_frequencies(freq_data);
	}
}

// clear frequencies of one protocol in database, used by scan-clear
void mpxplay_dtvdrv_database_clear_freqencies_of_protocol(mpxp_uint32_t protocol_id)
{
	mpxp_uint32_t prot2_id;

	dtvdrv_clear_freqencies_of_protocol(protocol_id);

	switch(protocol_id)
	{   // select siblings to clear
		case BDADEVTYPE_DVB_C: protocol_id = BDADEVTYPE_DVB_C2; prot2_id = BDADEVTYPE_ISDB_C; break;
		case BDADEVTYPE_DVB_S: protocol_id = BDADEVTYPE_DVB_S2; prot2_id = BDADEVTYPE_ISDB_S; break;
		case BDADEVTYPE_DVB_T: protocol_id = BDADEVTYPE_DVB_T2; prot2_id = BDADEVTYPE_ISDB_T; break;
		default: return;
	}

	dtvdrv_clear_freqencies_of_protocol(protocol_id);
	dtvdrv_clear_freqencies_of_protocol(prot2_id);
}

/* dealloc the protocol chain and the related datas */
void mpxplay_dtvdrv_database_clear_protocols(struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	while(prot_data)
	{
		struct mpxplay_dvb_epg_protocol_data_s *next_protocol = prot_data->next_protocol;
		struct mpxplay_dvb_epg_frequency_data_s *freq_data = prot_data->frequency_data_chain;
		pds_memset(prot_data, 0, sizeof(*prot_data));
		dtvdrv_database_clear_frequencies(freq_data);
		pds_free(prot_data);
		prot_data = next_protocol;
	}
}

//--------------------------------------------------------------------------------------------------------------------
// clear obsolete events by current date and time (keep them from the current event)
static void dtvdrv_database_consolidate_events(struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	const mpxp_uint64_t datetime_secs_past_limit = pds_datetimeval_to_seconds(pds_getdatetime()) - (mpxplay_config_dvbepg_pastevents_hours * 60 * 60);
	while(prot_data)
	{
		struct mpxplay_dvb_epg_frequency_data_s *freq_data = prot_data->frequency_data_chain;
		while(freq_data)
		{
			struct mpxplay_dvb_epg_program_data_s *prog_data = freq_data->program_data_chain;
			while(prog_data)
			{
				struct mpxplay_dvb_epg_event_data_s *epg_event = prog_data->epg_program_event_chain; // note: events are immediately sorted by date in mpxplay_dtvdrv_database_event_add
				while(epg_event)
				{
					struct mpxplay_dvb_epg_event_data_s *event_next = epg_event->next_event;
					if(event_next)
					{
						mpxp_uint64_t next_event_start_secs = pds_datetimeval_to_seconds(event_next->event_begin_date_time);
						if(next_event_start_secs >= datetime_secs_past_limit) // begin of next event (end of current epg_event) is in the 'keep' range, keep the events from this one
							break;
					}
					if(epg_event->event_shortdesc_name)
						pds_free(epg_event->event_shortdesc_name);
					if(epg_event->event_shortdesc_details)
						pds_free(epg_event->event_shortdesc_details);
					if(epg_event->event_extended_desc)
						pds_free(epg_event->event_extended_desc);
					pds_free(epg_event);
					epg_event = event_next;
					prog_data->epg_program_event_chain = epg_event; // set the new first elem (can be null)
				}
				prog_data = prog_data->next_program;
			}
			freq_data = freq_data->next_frequency;
		}
		prot_data = prot_data->next_protocol;
	}
}

struct mpxplay_dvb_epg_event_data_s *mpxplay_dtvdrv_database_get_current_event(struct mpxplay_dvb_epg_program_data_s *prog_data)
{
	struct mpxplay_dvb_epg_event_data_s *epg_event;
	mpxp_uint64_t datetime_val;

	if(!prog_data)
		return NULL;
	datetime_val = pds_getdatetime();

	epg_event = prog_data->epg_program_event_chain; // note: events are immediately sorted by date in mpxplay_dtvdrv_database_event_add
	while(epg_event)
	{
		struct mpxplay_dvb_epg_event_data_s *event_next = epg_event->next_event;
		if((datetime_val >= epg_event->event_begin_date_time) && (!event_next || (datetime_val < event_next->event_begin_date_time))) // current time is between this and next event
			break;
		epg_event = event_next;
	}

	return epg_event;
}

//--------------------------------------------------------------------------------------------------------------------

/* save the program chain and the related datas */
static void dtvdrv_database_save_program_events(void *filehand, struct mpxplay_dvb_epg_event_data_s *event_data)
{
	char stmp[1024];
	while(event_data)
	{
		if(event_data->event_shortdesc_name && event_data->event_begin_date_time && event_data->event_duration_time)
		{
			mpxplay_diskdrive_textfile_writeline(filehand, "   <event>");
			snprintf(stmp,sizeof(stmp), "    <evname>%s</evname>", event_data->event_shortdesc_name);
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			if(event_data->event_shortdesc_details)
			{
				snprintf(stmp,sizeof(stmp), "    <evdtl>%s</evdtl>", event_data->event_shortdesc_details);
				mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			}
			if(event_data->event_extended_desc)
			{
				snprintf(stmp,sizeof(stmp), "    <evext>%.1000s</evext>", event_data->event_extended_desc);
				mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			}
			snprintf(stmp,sizeof(stmp), "    <evid>%d</evid>", event_data->event_id);
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			snprintf(stmp,sizeof(stmp), "    <evbeg>%8.8X %8.8X</evbeg>", (mpxp_uint32_t)(event_data->event_begin_date_time >> 32), (mpxp_uint32_t)(event_data->event_begin_date_time & 0xFFFFFFFF));
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			snprintf(stmp,sizeof(stmp), "    <evlen>%8.8X</evlen>", event_data->event_duration_time);
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			if(event_data->parental_rating)
			{
				snprintf(stmp,sizeof(stmp), "    <evrat>%d</evrat>", event_data->parental_rating);
				mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			}
			if(event_data->event_flags & MPXPLAY_DVBEPG_EVENTFLAGS_SAVEMASK)
			{
				snprintf(stmp,sizeof(stmp), "    <evflag>%2.2X</evflag>", (event_data->event_flags & MPXPLAY_DVBEPG_EVENTFLAGS_SAVEMASK));
				mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			}
			mpxplay_diskdrive_textfile_writeline(filehand, "   </event>");
		}
		event_data = event_data->next_event;
	}
}

/* save the program chain and the related datas */
static void dtvdrv_database_save_programs(void *filehand, struct mpxplay_dvb_epg_program_data_s *prog_data)
{
	char stmp[256];
	while(prog_data)
	{
		mpxplay_diskdrive_textfile_writeline(filehand, "  <programdatas>");
		snprintf(stmp,sizeof(stmp), "   <programid>%d</programid>", prog_data->program_id);
		mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		snprintf(stmp,sizeof(stmp), "   <programchid>%d</programchid>", prog_data->channel_id);
		mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		snprintf(stmp,sizeof(stmp), "   <programflags>%2.2X</programflags>", prog_data->ctrl_flags);
		mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		if(prog_data->program_name)
		{
			snprintf(stmp,sizeof(stmp), "   <programname>%s</programname>", prog_data->program_name);
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		}
		dtvdrv_database_save_program_events(filehand, prog_data->epg_program_event_chain);
		mpxplay_diskdrive_textfile_writeline(filehand, "  </programdatas>");
		prog_data = prog_data->next_program;
	}
}

/* save the frequency chain and the related datas */
static void dtvdrv_database_save_frequencies(void *filehand, struct mpxplay_dvb_epg_frequency_data_s *freq_data)
{
	char stmp[256];
	while(freq_data)
	{
		if(funcbit_test(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT)) // TODO: check
		{
			mpxplay_diskdrive_textfile_writeline(filehand, " <frequencydatas>");
			snprintf(stmp,sizeof(stmp), "  <frequencyhz>%d</frequencyhz>", freq_data->frequency);
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			snprintf(stmp,sizeof(stmp), "  <frequencyflags>%8.8X</frequencyflags>", freq_data->freq_flags);
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			snprintf(stmp,sizeof(stmp), "  <frequencysigq>%8.8X</frequencysigq>", freq_data->freq_signal_quality);
			mpxplay_diskdrive_textfile_writeline(filehand, stmp);
			dtvdrv_database_save_programs(filehand, freq_data->program_data_chain);
			mpxplay_diskdrive_textfile_writeline(filehand, " </frequencydatas>");
		}
		freq_data = freq_data->next_frequency;
	}
}

/* save the frequency chain and the related datas */
static void dtvdrv_database_save_protocols(void *filehand, struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	char stmp[256];
	while(prot_data)
	{
		mpxplay_diskdrive_textfile_writeline(filehand, "<protocoldatas>");
		snprintf(stmp,sizeof(stmp), " <protocolname>%s</protocolname>", mpxplay_drvdtv_get_protocol_name(prot_data->protocol_id));
		mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		snprintf(stmp,sizeof(stmp), " <freqbandwidth>%d</freqbandwidth>", prot_data->freq_bandwidth);
		mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		snprintf(stmp,sizeof(stmp), " <scanfreqbegin>%d</scanfreqbegin>", prot_data->scan_freq_begin);
		mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		snprintf(stmp,sizeof(stmp), " <scanfreqend>%d</scanfreqend>", prot_data->scan_freq_end);
		mpxplay_diskdrive_textfile_writeline(filehand, stmp);
		dtvdrv_database_save_frequencies(filehand, prot_data->frequency_data_chain);
		mpxplay_diskdrive_textfile_writeline(filehand, "</protocoldatas>");
		prot_data = prot_data->next_protocol;
	}
}

static void drvdtv_database_parse_dbfile(void *filehand, struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	struct mpxplay_dvb_epg_program_data_s *prog_data = NULL;
	int protocol_id = -1, freq_bandwidth = 0, scan_freq_begin = 0, scan_freq_end = 0;
	int frequency_hz = 0, frequency_flags = 0, frequency_signal = 0, program_id = 0, program_channel_id = 0;
	mpxp_uint32_t program_ctrl_flags = 0;
	mpxp_uint32_t prog_event_duration = 0, prog_event_id = 0, prog_event_parental_rating = 0, prog_event_flags = 0;
	mpxp_uint64_t prog_event_begintime = 0;
	char program_name[128] = "", prog_event_name[128] = "", prog_event_details[512] = "", prog_event_extdesc[1024] = "", stmp[1024];
	do{
		char *databegin, *dataend, *beginp;
		int bytes;

		stmp[0] = 0;
		bytes = mpxplay_diskdrive_textfile_readline(filehand, stmp, sizeof(stmp) - 1);
		if(bytes <= 0)
			break;

		beginp = &stmp[0];
		while(beginp[0] == ' ') // skip spaces
			beginp++;
		if(beginp[0] != '<')    // after spaces a '<' shall be
			continue;
		beginp++;

		databegin = pds_strchr(beginp, '>'); // begin of config value
		if(!databegin)
			continue;
		*databegin++ = 0;

		dataend = pds_strrchr(databegin, '<'); // end of config value
		if(dataend)
			*dataend = 0;

		if(dataend > databegin)
		{
			if(pds_strcmp(beginp, "protocolname") == 0)
			{
				protocol_id = mpxplay_drvdtv_get_protocol_id(databegin);
			}
			else if(pds_strcmp(beginp, "freqbandwidth") == 0)
			{
				freq_bandwidth = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "scanfreqbegin") == 0)
			{
				scan_freq_begin = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "scanfreqend") == 0)
			{
				scan_freq_end = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "frequencyhz") == 0)
			{
				frequency_hz = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "frequencyflags") == 0)
			{
				frequency_flags = pds_atol16(databegin);
			}
			else if(pds_strcmp(beginp, "frequencysigq") == 0)
			{
				frequency_signal = pds_atol16(databegin);
			}
			else if(pds_strcmp(beginp, "programid") == 0)
			{
				program_id = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "programchid") == 0)
			{
				program_channel_id = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "programflags") == 0)
			{
				program_ctrl_flags = pds_atol16(databegin);
			}
			else if(pds_strcmp(beginp, "programname") == 0)
			{
				pds_strcpy(program_name, databegin);
			}
			else if(pds_strcmp(beginp, "evname") == 0)
			{
				pds_strncpy(prog_event_name, databegin, sizeof(prog_event_name));
				prog_event_name[sizeof(prog_event_name) - 1] = 0;
			}
			else if(pds_strcmp(beginp, "evdtl") == 0)
			{
				pds_strncpy(prog_event_details, databegin, sizeof(prog_event_details));
				prog_event_details[sizeof(prog_event_details) - 1] = 0;
			}
			else if(pds_strcmp(beginp, "evext") == 0)
			{
				pds_strncpy(prog_event_extdesc, databegin, sizeof(prog_event_extdesc));
				prog_event_extdesc[sizeof(prog_event_extdesc) - 1] = 0;
			}
			else if(pds_strcmp(beginp, "evid") == 0)
			{
				prog_event_id = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "evbeg") == 0)
			{
				prog_event_begintime = (((mpxp_uint64_t )pds_atol16(databegin)) << 32) | (mpxp_uint64_t)pds_atol16(databegin + 9); // YYYYMMDD 00HHMMSS (byte coded)
			}
			else if(pds_strcmp(beginp, "evlen") == 0)
			{
				prog_event_duration = pds_atol16(databegin); // 00HHMMSS (byte coded)
			}
			else if(pds_strcmp(beginp, "evrat") == 0)
			{
				prog_event_parental_rating = pds_atol(databegin);
			}
			else if(pds_strcmp(beginp, "evflag") == 0)
			{
				prog_event_flags = pds_atol16(databegin);
			}
			if((protocol_id >= 0) && (frequency_hz > 0) && (program_id > 0))
			{
				prog_data = mpxplay_dtvdrv_database_program_new(protocol_id, frequency_hz, program_id, program_name);
			}
		}
		else if(pds_strcmp(beginp, "/event") == 0)
		{
			if(prog_event_duration && prog_event_begintime && prog_event_name[0])
			{
				struct mpxplay_dvb_epg_event_data_s *event_data_new = mpxplay_dtvdrv_database_event_add(prog_data, prog_event_id, prog_event_begintime, prog_event_duration, FALSE);
				if(event_data_new)
				{
					mpxplay_dtvdrv_database_event_add_short_description(event_data_new, prog_event_name, prog_event_details);
					mpxplay_dtvdrv_database_event_add_extended_description(event_data_new, prog_event_extdesc, FALSE);
					event_data_new->parental_rating = prog_event_parental_rating;
					event_data_new->event_flags = prog_event_flags;
				}
			}
			prog_event_name[0] = 0;
			prog_event_details[0] = 0;
			prog_event_extdesc[0] = 0;
			prog_event_id = 0;
			prog_event_begintime = 0;
			prog_event_duration = 0;
			prog_event_parental_rating = 0;
			prog_event_flags = 0;
		}
		else if(pds_strcmp(beginp, "/programdatas") == 0)
		{
			if(prog_data)
			{
				if(program_channel_id)
					prog_data->channel_id = program_channel_id;
				if(prog_data)
					prog_data->ctrl_flags = program_ctrl_flags;
				dtvdrv_database_program_sort_by_event_start(prog_data);
				dtvdrv_database_event_check_all_starttime_validity(prog_data);
				prog_data = NULL;
			}
			program_id = 0;
			program_channel_id = 0;
			program_ctrl_flags = 0;
			program_name[0] = 0;
		}
		else if(pds_strcmp(beginp, "/frequencydatas") == 0)
		{
			if(frequency_flags || frequency_signal)
			{
				struct mpxplay_dvb_epg_frequency_data_s *freq_data = mpxplay_dtvdrv_database_frequency_search(protocol_id, frequency_hz);
				if(freq_data)
				{
					freq_data->freq_flags = frequency_flags;
					freq_data->freq_signal_quality = frequency_signal;
				}
				frequency_flags = 0;
				frequency_signal = 0;
			}
			frequency_hz = 0;
		}
		else if(pds_strcmp(beginp, "/protocoldatas") == 0)
		{
			struct mpxplay_dvb_epg_protocol_data_s *prot_data = mpxplay_dtvdrv_database_protocol_new(protocol_id);
			if(prot_data)
			{
				if(freq_bandwidth > 0)
					prot_data->freq_bandwidth = freq_bandwidth;
				if(scan_freq_begin > 0)
					prot_data->scan_freq_begin = scan_freq_begin;
				if(scan_freq_end > 0)
					prot_data->scan_freq_end = scan_freq_end;
			}
			protocol_id = -1;
			freq_bandwidth = 0;
			scan_freq_begin = 0;
			scan_freq_end = 0;
		}
	}while(1);
}

//--------------------------------------------------------------------------------------------------------------------

static void dtvdrv_database_get_localxmlfilename(char *strbuf)
{
	if(mpxplay_control_get_ini_path(strbuf) != MPXPLAY_ERROR_OK)
		return;
	pds_filename_assemble_fullname(strbuf, strbuf, DTVDRIVE_LOCAL_DATABASE_NAME);
}

int mpxplay_drvdtv_database_load_from_file(void)
{
	int retval = MPXPLAY_ERROR_OK;
	void *database_file_handler;
	char filename[MAX_PATHNAMELEN];

	if(drvdtv_database_is_loaded)
		return retval;

	filename[0] = 0;

	drvdtv_database_is_loaded = TRUE;

	dtvdrv_database_protocols_set_default();

	dtvdrv_database_get_localxmlfilename(&filename[0]);

	database_file_handler = mpxplay_diskdrive_textfile_open(NULL, filename, O_TEXT | O_RDONLY);
	if(database_file_handler)
	{
		drvdtv_database_parse_dbfile(database_file_handler, mpxplay_drvdtv_shared_drive_datas.protocol_data_chain);
		mpxplay_diskdrive_textfile_close(database_file_handler);
		dtvdrv_database_consolidate_events(mpxplay_drvdtv_shared_drive_datas.protocol_data_chain);
	}
	else
	{
		retval = MPXPLAY_ERROR_FILEHAND_CANTOPEN;
	}

	return retval;
}

static void drvdtv_database_save_to_file(void)
{
	void *database_file_handler;
	char stmp[MAX_PATHNAMELEN] = "";

	if(!mpxplay_drvdtv_shared_drive_datas.protocol_data_chain)
		return;

	dtvdrv_database_get_localxmlfilename(&stmp[0]);

	database_file_handler = mpxplay_diskdrive_textfile_open(NULL, stmp, O_TEXT | O_WRONLY | O_CREAT);
	if(database_file_handler)
	{
		unsigned long cfg = MPXPLAY_TEXTCONV_TYPE_UTF8;
		mpxplay_diskdrive_textfile_config(database_file_handler, MPXPLAY_DISKTEXTFILE_CFGFUNCNUM_SET_TEXTCODETYPE_DEST, &cfg, NULL);

		mpxplay_diskdrive_textfile_writeline(database_file_handler, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>");

		dtvdrv_database_consolidate_events(mpxplay_drvdtv_shared_drive_datas.protocol_data_chain);

		dtvdrv_database_save_protocols(database_file_handler, mpxplay_drvdtv_shared_drive_datas.protocol_data_chain);
		mpxplay_diskdrive_textfile_close(database_file_handler);
	}
}

void mpxplay_drvdtv_database_reset_all(void)
{
	mpxplay_dtvdrv_database_clear_protocols(mpxplay_drvdtv_shared_drive_datas.protocol_data_chain);
	dtvdrv_database_protocols_set_default();
}

void mpxplay_drvdtv_database_close(void)
{
	drvdtv_database_save_to_file();
	mpxplay_dtvdrv_database_clear_protocols(mpxplay_drvdtv_shared_drive_datas.protocol_data_chain);
}

#endif // defined(MPXPLAY_LINK_INFILE_FF_MPEG) && defined(MPXPLAY_WIN32)
