//**************************************************************************
//*                     This file is part of the                           *
//*                    Mpxplay-MMC - video player.                         *
//*                The source code of Mpxplay-MMC is                       *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: digital TV (DVB) handling by BDA driver

#ifndef DTV_DRV_H
#define DTV_DRV_H

#include "diskdriv.h"
#include "deparser/ffmpgdec.h"

#ifdef __cplusplus
extern "C" {
#endif

//#define MPXPLAY_DRVTV_ENABLE_DVBT2 1 // for testing only (it's better to use the normal DVBT)

#define DTVBDA_FEC(a,b)   (((a) << 16u) | (b))
#define DTVDBA_GUARD(a,b) (((a) << 16u) | (b))

enum {
	BDADEVTYPE_ATSC = 0, BDADEVTYPE_CQAM,
	BDADEVTYPE_DVB_C, BDADEVTYPE_DVB_C2,
	BDADEVTYPE_DVB_S, BDADEVTYPE_DVB_S2,
	BDADEVTYPE_DVB_T, BDADEVTYPE_DVB_T2,
	BDADEVTYPE_ISDB_C, BDADEVTYPE_ISDB_S, BDADEVTYPE_ISDB_T,
	BDADEVTYPE_NUM
};

#define MPXPLAY_DRVDTV_BDADEVTYPE_LOCALFILE        15 // fake BDADEVTYPE value for local-file dvb_epg dialog
#define MPXPLAY_DRVDTV_FREQUENCY_LOCALFILE 2000000000

#define MPXPLAY_DRVDTV_FREQUENCY_MIN         10000000
#define MPXPLAY_DRVDTV_FREQUENCY_BANDWIDTH    8000000 // default for scan only
#define MPXPLAY_DRVDTV_FREQUENCY_SCAN_BEGIN 474000000 // TODO: valid for DVB-T only?
#define MPXPLAY_DRVDTV_FREQUENCY_SCAN_END   858000000 //

#define MPXPLAY_DRVDTV_EVENTS_SCAN_MIN_HOURS   (2 * 24) // wait for this event at least before finishing EPG load

#define MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_OPENSHOWSIGSTR_100  5000 // show signal strength at opening a new freq/device till 5 secs (even if the signal strength is high)
#define MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_OPENSHOWSIGSTR_60  15000 // show signal strength if it's less or equal than 60 (usually bellow 60, the playing will not correct)
#define MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_SIGNAL 15000
#define MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_READ   6000
#define MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_MUTEX  6000

#define MPXPLAY_DRVDTV_INITIAL_CONTINUITY_LIMIT 3        // initial error detection in frames
#define MPXPLAY_DRVDTV_INITIAL_READERROR_LIMIT  3        // count of read error
#define MPXPLAY_DRVDTV_INITIAL_READ_DATASIZE    131072   // if filepos is less, then it's initial reading

#define MPXPLAY_DRVDTV_FLAG_OPENMODE_PLAY  (1 << 0) // dtv/bda open for play
#define MPXPLAY_DRVDTV_FLAG_OPENMODE_SCAN  (1 << 1) // dtv/bda open for freq/chan scan
#define MPXPLAY_DRVDTV_FLAG_TERMINATE      (1 << 2) // terminate the reading to this "file"
#define MPXPLAY_DRVDTV_FLAG_READWAIT       (1 << 3) // wait for incoming data (else return immediately)
#define MPXPLAY_DRVDTV_FLAG_PLAYTERMBYSCAN (1 << 4) // play has been terminated by scan (so reopen play after scan finish)

enum {DTVBDA_DATABUF_BUFTYPE_MULTIPLEX = 0, DTVBDA_DATABUF_BUFTYPE_PROGRAM}; // TODO: unimplemented driver level demuxer/splitter

#define DTVBDA_DATABUF_INITSIZE_PROG 1048576
#define DTVBDA_DATABUF_INITSIZE_MUX (1048576 * 10)

// dtvbda_data_buffer_s buffer_flags
#define DTVBDA_DATABUF_FLAGCTR_OPENMODE_PLAY      (1 << 0) // buffer is used for play or scan (not for copy)
#define DTVBDA_DATABUF_FLAGCTR_READLOCK           (1 << 1) // lock for read (delayed read, detect possible initial stream errors)
#define DTVBDA_DATABUF_FLAGINF_ERROR_CONTINUITY   (1 << 8) // lost blocks (by device)
#define DTVBDA_DATABUF_FLAGINF_ERROR_BUFFER_FULL  (1 << 9) // buffer became full
#define DTVBDA_DATABUF_FLAGINF_ERROR_MASK         (DTVBDA_DATABUF_FLAGINF_ERROR_CONTINUITY | DTVBDA_DATABUF_FLAGINF_ERROR_BUFFER_FULL)

typedef struct dtvresource_chainelem_s dtvresource_chainelem_t;
typedef struct dvb_device_s dvb_device_t;

typedef struct dtvbda_data_buffer_s
{
	mpxp_uint8_t *buffer_beginptr;
	unsigned int buffer_id;
	int buffer_size;
	int buffer_putpos;
	int buffer_getpos;
	int buffer_rewindbytes;        // buffered and read bytes, we should use it at MUX buffer only
	int buffer_forwardbytes;       // buffered but not read bytes
	unsigned int buffer_type;      // DTVBDA_DATABUF_BUFTYPE_MULTIPLEX or program number if > 0
	unsigned int buffer_flags;     // DTVBDA_DATABUF_FLAGINF_
	struct dtvbda_data_buffer_s *next_databuffer;
	dvb_device_t *dev_ptr;
}dtvbda_data_buffer_s;

enum {DTVBDA_SCAN_STATUS_TUNING = 0, DTVBDA_SCAN_STATUS_WAITINGSIGNAL, DTVBDA_SCAN_STATUS_ANALYZE, DTVBDA_SCAN_STATUS_FINISH};

typedef struct dtv_scan_info_s
{
	mpxp_uint32_t flags;
	mpxp_uint32_t status;
	mpxp_int32_t  scan_freq_curr_khz;   // frequency in kHz
	mpxp_uint32_t found_channels;
	struct mpxplay_dvb_epg_frequency_data_s *scan_epgload_freq_chain; // epg load has initial channels (don't do freq scanning)
	mpxp_int64_t  status_starttime_ms; // start time of current status in msecs (tune, waitsignal, analyze)
}dtv_scan_info_s;

typedef struct dvb_device_s
{
	mpxp_uint32_t flags;
	mpxp_uint32_t protocol_number_id;
	dtvresource_chainelem_t *dtv_resource;
	void *bdagraph_module;
	dtvbda_data_buffer_s *data_buffer;

	mpxp_int32_t adapter_num;
	mpxp_int32_t frequency;
	mpxp_int32_t program_number_id;    // DTVBDA_DATABUF_BUFTYPE_MULTIPLEX or program number if > 0
	mpxp_int32_t scan_freq_begin;      // frequency in kHz
	mpxp_int32_t scan_freq_end;        // frequency in kHz
	mpxp_int32_t scan_freq_bandwidth;  // frequency in kHz
	mpxp_int32_t scan_do_epgload;      // load epg
	mpxp_uint32_t signal_quality;

	/* DVB-S */
	mpxp_int32_t srate;
	mpxp_int32_t fec;
	mpxp_int32_t inversion;
	char lnb_polarization;
	mpxp_int32_t lowf, highf, switchf;
	mpxp_int32_t azimuth, elevation, longitude, network_id;
	char *mod, *range, *network_name;

	/* DVB-T */
	mpxp_int32_t fec_hp, fec_lp, bandwidth;
	mpxp_int32_t guard;
	mpxp_int32_t transmission;
	mpxp_int32_t hierarchy;
#ifdef MPXPLAY_DRVTV_ENABLE_DVBT2
	/* DVB-T2 */
	mpxp_int32_t plp_id;
#endif

	/* ATSC, CQAM */
	mpxp_int32_t physical_channel, major_channel, minor_channel;

	mpxp_int64_t timeout_at_read;
	mpxp_int64_t timeout_open_showsignalstrength;

	struct dtv_scan_info_s scan_infos;
	struct ffmpg_epg_info_s epg_infos;
}dvb_device_t;

typedef struct dtvresource_chainelem_s
{
	struct dtvresource_chainelem_s *next_res;
	unsigned int resource_id;
	dvb_device_t dev_infos;
}dtvresource_chainelem_t;

extern dvb_device_t *dtv_bda_device_alloc(void);
extern void dtv_bda_device_free(dvb_device_t *d);
extern int dtv_bda_device_start(dvb_device_t *d);
extern void dtv_bda_device_stop(dvb_device_t *d);
extern mpxp_bool_t dtv_bda_device_is_signal(dvb_device_t *d);
extern int  dtv_bda_resource_assign(dvb_device_t *dtv_device);
extern int  dtv_bda_resource_remove(dvb_device_t *dtv_device);
extern void mpxplay_dtv_bda_resources_stop(void);
extern void mpxplay_dtv_bda_resources_close(void);
extern int  dtv_bda_device_read(dvb_device_t *d, char *buf, int requested_bytes);
extern int  dtv_bda_device_get_signal_strength(dvb_device_t *d);

//------------------------------------------------------------------------------------------------------------
// drv_dtv.c
extern int mpxplay_drvdtv_get_protocol_id(char *protocol_name);
extern const char *mpxplay_drvdtv_get_protocol_name(unsigned int protocol_id);
extern void mpxplay_dtvdrv_assemble_filename(char *destbuf, unsigned int buflen, mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_uint32_t program_id);
extern mpxp_int32_t mpxplay_dtvdrv_parse_filename(char *filename, int *protocol_id, mpxp_int32_t *freqency_hz);

//------------------------------------------------------------------------------------------------------------
// dtvscan.c
extern int mpxplay_drvdtv_scan_start(dvb_device_t *d);
extern void mpxplay_drvdtv_scan_stop(dvb_device_t *d);

//------------------------------------------------------------------------------------------------------------
// dtvdbase.c

// common datas for all dtv programs
typedef struct dtvdrive_shared_data_s
{
	unsigned int nb_protocols;
	struct mpxplay_dvb_epg_protocol_data_s *protocol_data_chain;
}dtvdrive_shared_data_s;

extern struct dtvdrive_shared_data_s mpxplay_drvdtv_shared_drive_datas;

extern struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_protocol_add_or_search(mpxp_uint32_t protocol_id, mpxp_bool_t search_only);
extern struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_protocol_new(mpxp_uint32_t protocol_id);
extern struct mpxplay_dvb_epg_frequency_data_s *mpxplay_dtvdrv_database_frequency_search(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency);
extern struct mpxplay_dvb_epg_program_data_s *mpxplay_dtvdrv_database_program_search(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_uint32_t service_id);
extern struct mpxplay_dvb_epg_program_data_s *mpxplay_dtvdrv_database_program_new(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_uint32_t service_id, char *service_name);

extern struct mpxplay_dvb_epg_event_data_s *mpxplay_dtvdrv_database_event_add(struct mpxplay_dvb_epg_program_data_s *prog_data, mpxp_uint32_t event_id, mpxp_uint64_t event_start_local_datetime, mpxp_uint32_t event_duration, mpxp_bool_t remove_invalid_events);
extern void mpxplay_dtvdrv_database_event_add_short_description(struct mpxplay_dvb_epg_event_data_s *event_data, char *event_name, char *event_description);
extern mpxp_bool_t mpxplay_dtvdrv_database_event_add_extended_description(struct mpxplay_dvb_epg_event_data_s *event_data, char *event_ext_description, mpxp_bool_t append);
extern void mpxplay_dtvdrv_database_event_add_parental_rating(struct mpxplay_dvb_epg_event_data_s *event_data, mpxp_uint8_t parental_rate);

extern void mpxplay_dtvdrv_database_frequencies_sort_by_type(mpxp_uint32_t protocol_id, unsigned int sort_type);
extern struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_protocol_all_info(unsigned long control_flags);
extern struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_dup_protocol_info(mpxp_uint32_t protocol_id, unsigned long control_flags);
extern struct mpxplay_dvb_epg_protocol_data_s *mpxplay_dtvdrv_database_dup_programs_into_protocol(struct mpxplay_dvb_epg_program_data_s *prog_src);
extern struct mpxplay_dvb_epg_protocol_data_s *mpxplay_drvdtv_database_protocol_frequency_info(mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, unsigned long control_flags);
extern mpxp_bool_t mpxplay_dtvdrv_database_check_program_validity(struct mpxplay_dvb_epg_frequency_data_s *freq_data, struct mpxplay_dvb_epg_program_data_s *prog_data, unsigned long control_flags);
extern mpxp_bool_t mpxplay_drvdtv_database_frequency_programs_check(mpxplay_dvb_epg_frequency_data_s *freq_data);
extern mpxp_bool_t mpxplay_drvdtv_database_frequency_programs_events_check(mpxplay_dvb_epg_frequency_data_s *freq_data);

extern void mpxplay_dtvdrv_database_clear_events(struct mpxplay_dvb_epg_event_data_s *epg_event);
extern void mpxplay_dtvdrv_database_clear_programs(struct mpxplay_dvb_epg_program_data_s *prog_data);
extern void mpxplay_dtvdrv_database_clear_freqencies_of_protocol(mpxp_uint32_t protocol_id);
extern void mpxplay_dtvdrv_database_clear_protocols(struct mpxplay_dvb_epg_protocol_data_s *prot_data);

extern struct mpxplay_dvb_epg_event_data_s *mpxplay_dtvdrv_database_get_current_event(struct mpxplay_dvb_epg_program_data_s *prog_data);
extern int mpxplay_drvdtv_database_load_from_file(void);
extern void mpxplay_drvdtv_database_reset_all(void);
extern void mpxplay_drvdtv_database_close(void);

# ifdef __cplusplus
}
# endif

#endif // DTV_DRV_H
