//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2023 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
////function: Common GUI elems handling (for layout changes)

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT NULL
#define DISPQT_DEBUGOUT_EPG NULL // stdout
#define DISPQT_DEBUGOUT_EVENT NULL //stdout
#define DISPQT_DEBUGOUT_MENUBAR NULL
#define DISPQT_DEBUGOUT_QE NULL
#define DISPQT_DEBUGOUT_COMP stdout

#include <QtGui>
#include <QtWidgets>
#include <QtWinExtras/QtWinExtras>
#include <control/cntfuncs.h>
#include "disp_qt.h"
#include "moc_dialog_elems.h"
#include "moc_mainwindow.h"

extern "C" {
 extern unsigned int playcontrol;
}

#ifdef Q_OS_WIN

#define DISPQT_WCA_ACCENT_POLICY         19
#define DISPQT_ACCENT_DISABLE_BLURBEHIND  0
#define DISPQT_ACCENT_ENABLE_BLURBEHIND   3
#define DISPQT_ACCENT_FLAG_COLORIZE       2
#define DISPQT_ACCENT_FLAG_BORDER         0 // 0x20 | 0x40 | 0x80 | 0x100  // FIXME: border settings?

struct WINCOMPATTRDATA
{
	DWORD attribute;
	PVOID pData;
	ULONG dataSize;
};

typedef BOOL (WINAPI *tSetWindowCompositionAttribute)(HWND hwnd, WINCOMPATTRDATA *pAttrData);
static tSetWindowCompositionAttribute SetWindowCompositionAttribute = NULL;

// returns true if background is set to colorized blur (returns false, if blur cannot be set or set to simple blur)
bool mpxplay_dispqt_dialog_elems_setwindowcomposition_win10(HWND m_hWnd, unsigned long color_hexa, unsigned int winVer)
{
	BOOL success = FALSE, colorizedBlurBackground = FALSE;

	if(!SetWindowCompositionAttribute)
		SetWindowCompositionAttribute = (tSetWindowCompositionAttribute)GetProcAddress(GetModuleHandle("user32.dll"), "SetWindowCompositionAttribute");

	if(SetWindowCompositionAttribute && m_hWnd)
	{
		static int data[4] = {DISPQT_ACCENT_ENABLE_BLURBEHIND, 0, 0, 0};
		if( (winVer < MPXPLAY_MSWIN_VERSIONID_WIN11)
		 && ((color_hexa >> DISPQT_DIALOG_HEXACOLOR_ALPHA_SHIFT) != DISPQT_DIALOG_HEXACOLOR_ALPHA_FULL)
		){  // Win 8-10 can colorize blur background, if alpha is not zero (color is not full transparent)
			data[1] = (DISPQT_ACCENT_FLAG_COLORIZE | DISPQT_ACCENT_FLAG_BORDER);
			data[2] = color_hexa; // including alpha
			colorizedBlurBackground = TRUE;
			mpxplay_debugf(DISPQT_DEBUGOUT_COMP, "COMP %8.8X", color_hexa);
		}
		WINCOMPATTRDATA attrData = {DISPQT_WCA_ACCENT_POLICY, (PVOID)&data, (ULONG)sizeof(data)};
		success = SetWindowCompositionAttribute(m_hWnd, &attrData);
	}

	return (success && colorizedBlurBackground);
}

void mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10(HWND m_hWnd)
{
	if(SetWindowCompositionAttribute && m_hWnd)
	{
		static int data[4] = {DISPQT_ACCENT_DISABLE_BLURBEHIND, 0, 0, 0};
		WINCOMPATTRDATA attrData = {DISPQT_WCA_ACCENT_POLICY, (PVOID)&data, (ULONG)sizeof(data)};
		SetWindowCompositionAttribute(m_hWnd, &attrData);
	}
}
#endif // Q_OS_WIN

//-------------------------------------------------------------------------------------------

DispQtDialogElemWindow::DispQtDialogElemWindow(MainWindow *mainwindow, QWidget *parent, Qt::WindowFlags winFLags, enum dispqt_dialog_window_types dialogType) : QDialog(parent, winFLags)
{
	this->main_window = mainwindow;
	this->window_flags = winFLags;
	this->dialog_type = dialogType;
	this->color_background_hexa = DISPQT_DIALOG_HEXACOLOR_BLACK;
	connect(this, SIGNAL(dialogwin_signal_config_style_apply(bool)), this, SLOT(dialogwin_config_style_apply(bool)));
	this->dialogwin_config_style_apply(true);
}

bool DispQtDialogElemWindow::event(QEvent *event)
{
	bool cursor_hide = false;
	switch(event->type()){
		case QEvent::Enter: break;
		case QEvent::Leave: cursor_hide = true; break;
		default: goto err_out_event;
	}
	if(this->main_window->mainwin_is_on_video())
		emit this->main_window->signal_video_mouse_cursor_show(cursor_hide);

err_out_event:
	return QDialog::event(event);
}

void DispQtDialogElemWindow::keyPressEvent(QKeyEvent *event)
{
	if(event->key() != Qt::Key_Escape)
		this->main_window->handle_keypressevent(event);
	else
		this->close();
}

void DispQtDialogElemWindow::dialogwin_config_get_colors(void)
{
	this->main_window->mainwin_guilayout_get_base_colors(this->dialog_type, NULL, NULL, NULL, &this->color_background_hexa);
}

void DispQtDialogElemWindow::dialogwin_config_style_color(void)
{
	this->color_palette = this->palette();
	if(this->main_window->mainwin_guilayout_get_base_colors(this->dialog_type, &this->color_palette, &this->color_dialog_background, &this->color_dialog_text, &this->color_background_hexa))
	{
		if( this->main_window->mainwin_guibkg_is_glass()
		 && (  ((this->dialog_type != DISPQT_DIALOG_WINDOWTYPE_DIALOG) && (this->dialog_type != DISPQT_DIALOG_WINDOWTYPE_CONFIG))
			 || this->main_window->mainwin_guilayout_is_translucent_enabled() )
		){
			const QRect fullScreenGeometry = QApplication::desktop()->screenGeometry(this);
			this->bkgnd_image = QPixmap(":images/glass7.png");
			this->bkgnd_image = this->bkgnd_image.scaled(QSize(fullScreenGeometry.width(), fullScreenGeometry.height()), Qt::IgnoreAspectRatio);
			this->color_palette.setBrush(QPalette::Window, this->bkgnd_image);
		}
		this->setPalette(this->color_palette);
	}
}

void DispQtDialogElemWindow::dialogwin_config_style_apply(bool initial)
{
	const unsigned int winVer = this->main_window->ms_windows_version;

	if(this->dialog_type == DISPQT_DIALOG_WINDOWTYPE_VIDEOCTRLBAR)
		return;

	if(this->main_window->mainwin_guibkg_is_transparent())
	{
#ifdef Q_OS_WIN
		if(winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80)
		{
			if(this->main_window->mainwin_guilayout_is_translucent_enabled())
			{
				this->dialogwin_config_get_colors();
				mpxplay_dispqt_dialog_elems_setwindowcomposition_win10((HWND)this->window()->winId(), this->color_background_hexa, winVer);
			}
			else if(!initial)
			{
				mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10((HWND)this->window()->winId());
			}
		}
#endif
		if( this->main_window->mainwin_guibkg_is_transparent()
		 && (  (this->dialog_type == DISPQT_DIALOG_WINDOWTYPE_DVBEPG) // remove this to get transparent windows for all (else non EPG windows are opaque)
		    || (this->window_flags & Qt::FramelessWindowHint) )
		){
			QtWin::enableBlurBehindWindow(this); // to set full transparent without blur effect too OR to get extendFrameIntoClientArea OR FramelessWindowHint under Win7
		}
		if( (this->dialog_type == DISPQT_DIALOG_WINDOWTYPE_DVBEPG)
		 || (  this->main_window->mainwin_guilayout_is_translucent_enabled()
		    && !(this->window_flags & Qt::FramelessWindowHint)
		    && (winVer >= MPXPLAY_MSWIN_VERSIONID_VISTA) && (winVer < MPXPLAY_MSWIN_VERSIONID_WIN80))
		){
			QtWin::extendFrameIntoClientArea(this, -1, -1, -1, -1);  // we use transparent header for DVB EPG window only or under Win7 for all
		}
		setAttribute(Qt::WA_TranslucentBackground, true);
		setAttribute(Qt::WA_NoSystemBackground, false);
		if(!initial && (this->dialog_type == DISPQT_DIALOG_WINDOWTYPE_DVBEPG))
		{
			this->setWindowOpacity(1.0);
		}
	}
	else if(this->main_window->mainwin_guibkg_setting_is_transparent())
	{
		if(this->window_flags & Qt::FramelessWindowHint)
		{
			setAttribute(Qt::WA_TranslucentBackground, true);
			setAttribute(Qt::WA_NoSystemBackground, false);
		}
		else if(this->dialog_type == DISPQT_DIALOG_WINDOWTYPE_DVBEPG)
		{
			this->setWindowOpacity((qreal)DISPQT_CONFIG_MAINWIN_OPACITY_DIALOG_DEFAULT / (qreal)DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MAX);
		}
	}
	else if(!initial)
	{
#ifdef Q_OS_WIN
		if(winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80)
		{
			mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10((HWND)this->window()->winId());
		}
#endif
		QtWin::disableBlurBehindWindow(this);
		QtWin::resetExtendedFrame(this);
		setAttribute(Qt::WA_TranslucentBackground, false);
		if(this->dialog_type == DISPQT_DIALOG_WINDOWTYPE_DVBEPG)
		{
			this->setWindowOpacity(1.0);
		}
	}

	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
			setStyleSheet("QDialog { border-style: solid; border-radius: 6px; }"
				"QListView { background: rgba(0,45,75,100); color: white;}"
				"QListView::item:selected { background: rgb(0,85,160);}"
				"QComboBox { "
					"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,82,144), stop: 0.5 rgba(0,72,120), stop: 0.6 rgba(0,60,100), stop:1 rgba(0,90,152));"
					"color: #FFFFFF; border-radius: 3px; }"
				"QComboBox::drop-down { subcontrol-origin: margin; }"
				"QComboBox QAbstractItemView { background: rgb(0,40,70); color: #FFFFFF; border: 1px solid #406080; border-radius: 3px; selection-background-color: rgb(0,85,160); }");
			break;
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			setStyleSheet("QDialog { border-style: solid; border-radius: 6px; }"
				"QLabel { color: white; } QGroupBox::title { color : white; }"
				"QCheckBox::unchecked { background: transparent; color: white;} QCheckBox::checked { background: transparent; color: white;}"
				"QListView { background: rgba(0,45,75,100); color: white;}"
				"QListView::item:selected { background: rgb(0,85,160); }"
				"QComboBox { background: rgba(255,255,255,50); color: #FFFFFF; border-radius: 3px; }"
				"QComboBox::drop-down { subcontrol-origin: margin; }"
				"QComboBox QAbstractItemView { background: rgb(0,40,70); color: #FFFFFF; border: 1px solid #406080; border-radius: 3px; selection-background-color: rgb(0,85,160); }");
			break;
	}

	this->dialogwin_config_style_color(); // putting last to overrride stylesheet settings
}

//-------------------------------------------------------------------------------------------

DispQtDialogElemTabWidget::DispQtDialogElemTabWidget(MainWindow *mainwindow, QWidget *parent) : QTabWidget(parent)
{
	this->main_window = mainwindow;
	this->setFocusPolicy(Qt::NoFocus);
	this->tabwidget_config_style_apply(true);
	setDocumentMode(true);
	mpxplay_debugf(DISPQT_DEBUGOUT_EPG, "w:%d h:%d", this->width(), this->height());
}

void DispQtDialogElemTabWidget::keyPressEvent(QKeyEvent *event)
{
	if(event->key() != Qt::Key_Escape)
		this->main_window->handle_keypressevent(event);
	else
		QTabWidget::keyPressEvent(event);
}

void DispQtDialogElemTabWidget::tabwidget_config_style_apply(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			setStyleSheet("QTabWidget { background: transparent } "
				"QTabBar::tab  { background: transparent; color: white; border: 1px solid rgb(64,96,128); padding: 4px; min-width: 8ex; border-top-left-radius: 5px; border-top-right-radius: 5px; }"
				"QTabBar::tab:selected { "
					"background: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,110,192,110), stop: 0.5 rgba(0,96,160,110), stop: 0.6 rgba(0,90,150,110), stop:1 rgba(0,120,202,110));"
					"color: white;}"
				"QTabBar::tab:!selected { background: rgba(0,45,75,50); }"
				"QTabBar::tab:hover:selected { "
					"background: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,110,192,200), stop: 0.5 rgba(0,96,160,200), stop: 0.6 rgba(0,90,150,200), stop:1 rgba(0,120,202,200));"
					"color: white;}"
				"QTabBar::tab:hover:!selected { background: rgba(0,60,100,128); }");
			break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemPushButton::DispQtDialogElemPushButton(MainWindow *mainwindow, QWidget *parrent, QString buttonTitle) : QPushButton(buttonTitle, parrent)
{
	this->main_window = mainwindow;
	this->setFixedSize(70, 26);
	this->pushbutton_set_style(true);
}

void DispQtDialogElemPushButton::pushbutton_set_style(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			this->setStyleSheet(
				"QPushButton::!checked {"
					"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,82,144,110), stop: 0.5 rgba(0,72,120,110), stop: 0.6 rgba(0,60,100,110), stop:1 rgba(0,90,152,110));"
					"color: white; border: 1px solid rgb(64,96,128); border-radius: 3px; }"
				"QPushButton::checked, QPushButton:default {"
					"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,110,192,200), stop: 0.5 rgba(0,96,160,200), stop: 0.6 rgba(0,80,134,200), stop:1 rgba(0,120,202,200));"
					"color: white; border: 1px solid rgb(64,96,128); border-radius: 3px; border-style: outset; }"
				"QPushButton:hover:!checked { background: rgba(0,100,190,130); color: white; border: 0px;}"
				"QPushButton:hover:checked, QPushButton:hover:default { "
					"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,137,244,200), stop: 0.5 rgba(0,120,200,200), stop: 0.6 rgba(0,100,168,200), stop:1 rgba(0,150,255,200));"
					"color: white; border: 1px solid rgb(64,96,128); border-radius: 3px; border-style: outset; }"
				);
			break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemQmenu::DispQtDialogElemQmenu(MainWindow *mainwindow, QWidget *parent, enum dispqt_dialog_window_types dialogType) : QMenu(parent)
{
	this->main_window = mainwindow;
	this->dialog_type = dialogType;
	this->translucent_enabled_last = false;
	this->gui_layout_last = DISPQT_GUILAYOUT_OSDEF_CLASSIC;
	this->color_background_hexa_last = DISPQT_DIALOG_HEXACOLOR_INVALID;
	qmenu_config_style_apply(true);
	connect(this, SIGNAL(aboutToShow()), this, SLOT(qmenu_config_style_update()));
}

DispQtDialogElemQmenu::DispQtDialogElemQmenu(const QString &title, MainWindow *mainwindow, QWidget *parent, enum dispqt_dialog_window_types dialogType) : QMenu(title, parent)
{
	this->main_window = mainwindow;
	this->dialog_type = dialogType;
	this->translucent_enabled_last = false;
	this->gui_layout_last = DISPQT_GUILAYOUT_OSDEF_CLASSIC;
	this->color_background_hexa_last = DISPQT_DIALOG_HEXACOLOR_INVALID;
	setTitle(title);
	qmenu_config_style_apply(true);
	connect(this, SIGNAL(aboutToShow()), this, SLOT(qmenu_config_style_update()));
}

void DispQtDialogElemQmenu::qmenu_config_style_update()
{
	qmenu_config_style_apply(false);
}

void DispQtDialogElemQmenu::qmenu_config_style_apply(bool initial)
{
	const unsigned long gui_layout = this->main_window->mainwin_guilayout_get_type();
	const bool is_translucent_enabled = this->main_window->mainwin_guilayout_is_translucent_enabled();
	const unsigned int winVer = this->main_window->ms_windows_version;
	unsigned long color_background_hexa = DISPQT_DIALOG_HEXACOLOR_VALID;
	bool colorizedBlurBackground = false;

	this->main_window->mainwin_guilayout_get_base_colors(this->dialog_type, NULL, NULL, NULL, &color_background_hexa);

	if((is_translucent_enabled == this->translucent_enabled_last) && (color_background_hexa == this->color_background_hexa_last) && (gui_layout == this->gui_layout_last))
	{
		return;
	}

	if(this->main_window->mainwin_guibkg_is_transparent())
	{
#ifdef Q_OS_WIN
		if(winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80)
		{
			if(this->main_window->mainwin_guilayout_is_translucent_enabled())
			{
				colorizedBlurBackground = mpxplay_dispqt_dialog_elems_setwindowcomposition_win10((HWND)this->window()->winId(), color_background_hexa, winVer);
			}
			else if(!initial)
			{
				mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10((HWND)this->window()->winId());
			}
		}
		else
#endif
		{
			setWindowFlags(windowFlags() | Qt::FramelessWindowHint);
		}
		QtWin::enableBlurBehindWindow(this);
		setAttribute(Qt::WA_TranslucentBackground, true);
		setAttribute(Qt::WA_NoSystemBackground, false);
	}
	else if(this->main_window->mainwin_guibkg_setting_is_transparent())
	{
		if(winVer < MPXPLAY_MSWIN_VERSIONID_WIN80)
			setWindowFlags(windowFlags() | Qt::FramelessWindowHint);
		setAttribute(Qt::WA_TranslucentBackground, true);
		setAttribute(Qt::WA_NoSystemBackground, false);
	}
	else if(!initial)
	{
#ifdef Q_OS_WIN
		if(winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80)
		{
			mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10((HWND)this->window()->winId());
		}
#endif
		QtWin::disableBlurBehindWindow(this);
		setAttribute(Qt::WA_TranslucentBackground, false);
	}

	switch(gui_layout)
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			if( is_translucent_enabled // blur background
			 && (  !this->main_window->mainwin_guibkg_is_colorized()  // no colorization at all
				|| colorizedBlurBackground ) // or background is colorized by OS blur
			){
				setStyleSheet("QMenu { background: transparent; color: white; font-size:13px; border: 1px solid transparent;}"
					"QMenu::item:selected{ background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,110,192,200), stop: 0.5 rgba(0,96,160,200), stop: 0.6 rgba(0,90,150,180), stop:1 rgba(0,120,202,200)); }");
			}
			else if(this->dialog_type == DISPQT_DIALOG_WINDOWTYPE_QMENU_VIDEO) // video context menu
			{
				if(is_translucent_enabled) // blur background - reduced colorization
				{
					setStyleSheet("QMenu { background: rgba(0,45,75,50); color: white; font-size:13px; border: 1px solid transparent;}"
						"QMenu::item:selected{ background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,110,192,200), stop: 0.5 rgba(0,96,160,200), stop: 0.6 rgba(0,90,150,180), stop:1 rgba(0,120,202,200)); }");
				}
				else // clean transparency = increased colorization
				{
					setStyleSheet("QMenu { background: rgba(0,45,75,120); color: white; font-size:13px; border: 1px solid transparent;}"
						"QMenu::item:selected{ background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,110,192,200), stop: 0.5 rgba(0,96,160,200), stop: 0.6 rgba(0,90,150,180), stop:1 rgba(0,120,202,200)); }");
				}
			}
			else // other contextmenu - highest colorization (less transparency)
			{
				setStyleSheet("QMenu { background: rgba(0,45,75,180); color: white; font-size:13px; border: 1px solid transparent;}"
					"QMenu::item:selected{ background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,110,192,200), stop: 0.5 rgba(0,96,160,200), stop: 0.6 rgba(0,90,150,180), stop:1 rgba(0,120,202,200)); }");
			}
			break;
	}

	this->translucent_enabled_last = is_translucent_enabled;
	this->color_background_hexa_last = color_background_hexa;
	this->gui_layout_last = gui_layout;
}

DispQtDialogElemQmenu *DispQtDialogElemQmenu::addMenu(const QString &title)
{
	DispQtDialogElemQmenu *menu = new DispQtDialogElemQmenu(title, this->main_window, this, this->dialog_type);
	addAction(menu->menuAction());
	return menu;
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemMenubar::DispQtDialogElemMenubar(MainWindow *main_window, QWidget *parent) : QMenuBar(parent)
{
	this->main_window = main_window;
	this->menubar_set_style(true);
}

bool DispQtDialogElemMenubar::event(QEvent *event)
{
	mpxplay_debugf(DISPQT_DEBUGOUT_MENUBAR, "menu event: %d", event->type());
	switch(event->type()){
		case QEvent::Enter:
			this->main_window->action_play_pausenext->setChecked(funcbit_test(playcontrol, PLAYC_PAUSENEXT)? true : false);
			this->main_window->action_list_id3search->setChecked((mpxplay_control_keyboard_id3search_initialized())? true : false);
			// @suppress("No break at end of case")
		case QEvent::MouseMove:
		case QEvent::StatusTip: emit this->main_window->signal_video_mainwin_wakeup(false, false); break;
	}
	return QMenuBar::event(event);
}

void DispQtDialogElemMenubar::menubar_set_style(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			setStyleSheet("DispQtDialogElemMenubar { background: transparent; border-color: navy}"
				"QMenuBar::item { spacing: 0px; padding: 1px 10px; background: transparent; color : white; border-radius: 8px;}"
				//"QMenuBar::item { spacing: 0px; padding: 1px 10px; background: rgb(225,225,225,10%); color : white; border-radius: 8px;}"
			"QMenuBar::item:selected { background: qlineargradient(x1:0, y1:0, x2:0, y2:1,"
				"stop:0 rgb(255,255,255,50%), stop: 0.25 rgb(200,200,200,40%), "
				"stop:0.5 rgb(200,200,200,30%), stop:0.51 rgb(150,150,150,20%), stop:1 rgb(150,150,150,20%));"
				"border-style: solid; border-width: 1px; border-color: qlineargradient(x1:0, y1:0, x2:0, y2:1,"
				"stop:0 rgb(250,250,250,90%), stop: 1 rgb(200,200,200,90%)); "
				"}"
			"QMenuBar::item:pressed {background: rgb(100,100,100,30%); border-color: rgb(150,150,150,60%); border-style:inset;}"
			);
			break;
	}
}

DispQtDialogElemQmenu *DispQtDialogElemMenubar::addMenu(const QString &title)
{
	DispQtDialogElemQmenu *menu = new DispQtDialogElemQmenu(title, this->main_window, this, DISPQT_DIALOG_WINDOWTYPE_QMENU_MAIN);
	addAction(menu->menuAction());
	return menu;
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemComboBox::DispQtDialogElemComboBox(MainWindow *main_window, QWidget *parent, bool colorize) : QComboBox(parent)
{
	this->main_window = main_window;
	if(colorize)
		this->combobox_set_style(true);
}

void DispQtDialogElemComboBox::combobox_set_style(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
			setStyleSheet("QComboBox { "
					"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,82,144), stop: 0.5 rgba(0,72,120), stop: 0.6 rgba(0,60,100), stop:1 rgba(0,90,152));"
					"color: #FFFFFF; border-radius: 3px; }"
				"QComboBox::drop-down { subcontrol-origin: margin; }"
				"QComboBox QAbstractItemView { background: rgb(0,40,70); color: #FFFFFF; border: 1px solid #406080; border-radius: 3px; selection-background-color: rgb(0,85,160); }");
			break;
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			setStyleSheet("QComboBox { background: rgba(255,255,255,50); color: #FFFFFF; border-radius: 3px; }"
				"QComboBox::drop-down { subcontrol-origin: margin; }"
				"QComboBox QAbstractItemView { background: rgb(0,40,70); color: #FFFFFF; border: 1px solid #406080; border-radius: 3px; selection-background-color: rgb(0,85,160); }");
	    		//"QComboBox::down-arrow {border: 0px }"
			break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemProgressBar::DispQtDialogElemProgressBar(MainWindow *mainwindow, QWidget *parent) : QProgressBar(parent)
{
	this->main_window = mainwindow;
	this->progressbar_set_style(true);
}

void DispQtDialogElemProgressBar::progressbar_set_style(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			this->setStyleSheet("QProgressBar { text-align: center; }");
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			this->setStyleSheet(
				"QProgressBar { background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,82,144,110), stop: 0.5 rgba(0,72,120,110), stop: 0.6 rgba(0,60,100,110), stop:1 rgba(0,90,152,110));"
								"color: white; border: 1px solid rgb(64,96,128); border-radius: 4px; text-align: center; }"
				"QProgressBar::chunk { background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,137,244,200), stop: 0.5 rgba(0,120,200,200), stop: 0.6 rgba(0,100,168,200), stop:1 rgba(0,150,255,200)); }"
				);
			break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemHeaderView::DispQtDialogElemHeaderView(MainWindow *mainwindow, Qt::Orientation orientation, QWidget *parent) : QHeaderView(orientation, parent)
{
	this->main_window = mainwindow;
	this->headerview_set_style(mainwindow->mainwin_guibkg_is_transparent());
}

void DispQtDialogElemHeaderView::headerview_set_style(bool transparent_back)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			this->setStyleSheet("QHeaderView:section { padding-left: 0px;padding-right: 0px;border-width: 0px; margin-top: 3px; margin-bottom: 3px; margin-left: 0px; margin-right: 0px;}");
			break;
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			if(transparent_back)
			{
				this->setStyleSheet("QHeaderView {background: transparent}"
					"QHeaderView:section { padding-left: 0px;padding-right: 0px;border-width: 0px; margin-top: 3px; margin-bottom: 3px; margin-left: 0px; margin-right: 0px;"
						"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,82,144,110), stop: 0.5 rgba(0,72,120,110), stop: 0.6 rgba(0,60,100,110), stop:1 rgba(0,90,152,110));"
						"color: white; border: 1px solid rgb(64,96,128); border-radius: 3px;"
					"}"
					//"QHeaderView::section:checked { background-color: red;}" // FIXME: doesn't work
					"QHeaderView::down-arrow { color: white }" // TODO: arrow picts
					"QHeaderView::up-arrow { color: white }"
					);
				break;
			}
			// @suppress("No break at end of case")
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
			this->setStyleSheet("QHeaderView {background: transparent}"
				"QHeaderView:section { padding-left: 0px;padding-right: 0px;border-width: 0px; margin-top: 3px; margin-bottom: 3px; margin-left: 0px; margin-right: 0px;"
					"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgb(0,82,144), stop: 0.5 rgb(0,72,120), stop: 0.6 rgb(0,60,100), stop:1 rgb(0,90,152));"
					"color: white; border: 1px solid rgb(64,96,128); border-radius: 3px;"
				"}"
				//"QHeaderView::section:checked { background-color: red;}" // FIXME: doesn't work
				"QHeaderView::down-arrow { color: white }" // TODO: arrow picts
				"QHeaderView::up-arrow { color: white }"
				);
			break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemScrollBar::DispQtDialogElemScrollBar(MainWindow *mainwindow, Qt::Orientation orientation, QWidget *parent) : QScrollBar(orientation, parent)
{
	this->main_window = mainwindow;
	this->scrollbar_set_style(true);
}

void DispQtDialogElemScrollBar::scrollbar_set_style(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			if(this->orientation() == Qt::Vertical)
			{
				this->setStyleSheet(
					"QScrollBar:vertical { background: transparent; width: 17px; margin: 17px 0px 17px 0px; }"
					//"QScrollBar:vertical { background: rgba(0,90,152,50); width: 17px; margin: 17px 0px 17px 0px; }"
					"QScrollBar::handle:vertical {"
						"background-color: qlineargradient(x1:0, y1:0, x2:1, y2:0, stop:0 rgba(0,82,144,110), stop: 0.5 rgba(0,72,120,110), stop: 0.6 rgba(0,60,100,110), stop:1 rgba(0,90,152,110));"
						//"background: qlineargradient(x1:0, y1:0, x2:1, y2:0, stop:0 rgba(0,110,192,110), stop: 0.5 rgba(0,96,160,110), stop: 0.6 rgba(0,80,134,110), stop:1 rgba(0,120,202,110));"
						"color: white; border: 1px solid rgb(64,96,128); border-radius: 3px;"
						"min-height: 17px; width: 17px;}"
					"QScrollBar::add-page:vertical { background: transparent; width: 17px; }"
					"QScrollBar::sub-page:vertical { background: transparent; width: 17px; }"
					"QScrollBar::add-line:vertical { height: 17px; subcontrol-position: bottom; subcontrol-origin: margin;}"
					"QScrollBar::sub-line:vertical { height: 17px; subcontrol-position: top; subcontrol-origin: margin;}"
					//"QScrollBar::up-arrow:vertical, QScrollBar::down-arrow:vertical { width: 17px; height: 17px; background: rgba(0,90,152,110);}"
					//"QScrollBar::up-arrow:vertical, QScrollBar::down-arrow:vertical { border: 0px}" // TODO: arrow picts
				);
			}
			else // horizontal scrollbar
			{
				this->setStyleSheet(
					"QScrollBar:horizontal { background: transparent; height: 17px; margin: 0px 17px 0px 17px;}"
					"QScrollBar::handle:horizontal {"
						"background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,82,144,110), stop: 0.5 rgba(0,72,120,110), stop: 0.6 rgba(0,60,100,110), stop:1 rgba(0,90,152,110));"
						"color: white; border: 1px solid rgb(64,96,128); border-radius: 3px;"
						"min-width: 17px; height: 17px;}"
					"QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal { background: transparent; height: 17px;}"
					"QScrollBar::add-line:horizontal { width: 17px; subcontrol-position: right; subcontrol-origin: margin;}"
					"QScrollBar::sub-line:horizontal { width: 17px; subcontrol-position: left; subcontrol-origin: margin;}"
				);
			}
			break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------------

DispQtDialogElemTextEdit::DispQtDialogElemTextEdit(MainWindow *mainwindow, QWidget *parent) : QTextEdit(parent)
{
	this->main_window = mainwindow;
	this->vscrollbar = new DispQtDialogElemScrollBar(mainwindow, Qt::Vertical, this);
	this->textedit_set_style(true);
	this->setVerticalScrollBar(this->vscrollbar);
}

void DispQtDialogElemTextEdit::textedit_set_style(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
			this->setStyleSheet("QTextEdit {background:rgb(100,150,200); color: black; border: 1px solid rgb(64,96,128); border-radius: 5px; font-size:12px;}");
			break;
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			this->setStyleSheet("QTextEdit {background:rgba(0,90,145,80); color: white; border: 1px solid rgb(64,96,128); border-radius: 5px; font-size:12px;}");
			break;
	}
	if(!initial)
	{
		this->vscrollbar->scrollbar_set_style(initial);
	}
}

//===============================================================================================================================================

DispQtConfigpageCheckboxGroup::DispQtConfigpageCheckboxGroup(MainWindow *main_window, QWidget *parent, const struct configpage_chkbox_s *p_chkbox_config, int nb_config) : QVBoxLayout(parent)
{
	if(!p_chkbox_config || (nb_config <= 0))
		return;
	this->chkbox_config = p_chkbox_config;
	this->nb_of_chkboxconfig = nb_config;

	this->chkbox_widgets = new QCheckBox *[nb_config];
	if(!this->chkbox_widgets)
		return;

	this->setContentsMargins(3,0,3,3);
	this->setSpacing(5);

	for(int i = 0; i < nb_config; i++, p_chkbox_config++) {
		unsigned long *varptr = p_chkbox_config->chkvar;
		if(varptr) {
			this->chkbox_widgets[i] = new QCheckBox;
			this->chkbox_widgets[i]->setText(QString::fromUtf8(p_chkbox_config->name));
			if(p_chkbox_config->chkbit) {
				this->chkbox_widgets[i]->setCheckable(true);
				bool setc = (*(p_chkbox_config->chkvar) & p_chkbox_config->chkbit)? true:false;
				this->chkbox_widgets[i]->setChecked(setc);
			} else
				this->chkbox_widgets[i]->setDisabled(true);
			this->chkbox_widgets[i]->setToolTip(QString::fromUtf8(p_chkbox_config->tooltip));
			this->chkbox_widgets[i]->setToolTipDuration(DISPQT_CONFIG_TOOLTIP_DURATION);
			this->addWidget(this->chkbox_widgets[i]);
		} else {
			QLabel *label = new QLabel(QString::fromUtf8(p_chkbox_config->name));
			this->addWidget(label);
			this->chkbox_widgets[i] = NULL;
		}
	}
}

DispQtConfigpageCheckboxGroup::~DispQtConfigpageCheckboxGroup()
{
	if(this->chkbox_widgets)
		delete this->chkbox_widgets;
}

void DispQtConfigpageCheckboxGroup::checkboxgroup_applyall(void)
{
	for(int i = 0; i < this->nb_of_chkboxconfig; i++) {
		unsigned long *varptr = this->chkbox_config[i].chkvar;
		if(!varptr)
			continue;
		if(!this->chkbox_widgets[i]->isCheckable())
			continue;
		unsigned long chkbit = this->chkbox_config[i].chkbit;
		if(this->chkbox_widgets[i]->isChecked())
			funcbit_enable(*varptr, chkbit);
		else
			funcbit_disable(*varptr, chkbit);
	}
}

DispQtConfigpageCheckboxGroupbox::DispQtConfigpageCheckboxGroupbox(MainWindow *main_window, QWidget *parent, QString box_title, const struct configpage_chkbox_s *p_chkbox_config, int nb_config) : QGroupBox(box_title, parent)
{
	this->checkbox_group = new DispQtConfigpageCheckboxGroup(main_window, parent, p_chkbox_config, nb_config);
	this->checkbox_group->setMargin(3);
	this->setLayout(this->checkbox_group);
}

DispQtConfigpageCheckboxGroupbox::~DispQtConfigpageCheckboxGroupbox()
{
	if(this->checkbox_group)
		delete this->checkbox_group;
	this->close();
}

void DispQtConfigpageCheckboxGroupbox::checkboxgroup_applyall(void)
{
	if(this->checkbox_group)
		this->checkbox_group->checkboxgroup_applyall();
}

//--------------------------------------------------------------------------------------
DispQtConfigpageSpinboxGroup::DispQtConfigpageSpinboxGroup(MainWindow *main_window, QWidget *parent, const struct configpage_spinbox_s *p_spinbox_config, int nb_config) : QGridLayout(parent)
{
	if(!p_spinbox_config || (nb_config <= 0))
		return;
	this->spinbox_config = p_spinbox_config;
	this->nb_of_spinboxconfig = nb_config;

	this->spinbox_widgets = new QSpinBox *[nb_config];
	if(!this->spinbox_widgets)
		return;

	this->setMargin(3);
	this->setSpacing(0);
	this->setColumnStretch(0, 10);
	this->setColumnStretch(1, 1);

	for(int i = 0; i < nb_config; i++) {
		QLabel *spin_text = new QLabel(QString::fromUtf8(p_spinbox_config[i].name));
		this->addWidget(spin_text, i, 0);
		unsigned long *varptr = p_spinbox_config[i].var_ptr;
		if(varptr) {
			this->spinbox_widgets[i] = new QSpinBox;
			this->spinbox_widgets[i]->setAlignment(Qt::AlignRight);
			this->spinbox_widgets[i]->setAccelerated(true);
			this->spinbox_widgets[i]->setRange(p_spinbox_config[i].val_min, p_spinbox_config[i].val_max);
			this->spinbox_widgets[i]->setValue(*varptr);
			this->addWidget(this->spinbox_widgets[i], i , 1);
		} else {
			this->spinbox_widgets[i] = NULL;
		}
	}
}

DispQtConfigpageSpinboxGroup::~DispQtConfigpageSpinboxGroup()
{
	if(this->spinbox_widgets)
		delete this->spinbox_widgets;
}

void DispQtConfigpageSpinboxGroup::spinboxgroup_applyall(void)
{
	for(int i = 0; i < this->nb_of_spinboxconfig; i++) {
		unsigned long *varptr = this->spinbox_config[i].var_ptr;
		if(!varptr || !this->spinbox_widgets[i])
			continue;
		*varptr = this->spinbox_widgets[i]->value();
	}
}

DispQtConfigpageSpinboxGroupbox::DispQtConfigpageSpinboxGroupbox(MainWindow *main_window, QWidget *parent, QString box_title, const struct configpage_spinbox_s *p_spinbox_config, int nb_config) : QGroupBox(box_title, parent)
{
	this->spinbox_group = new DispQtConfigpageSpinboxGroup(main_window, parent, p_spinbox_config, nb_config);
	this->setLayout(this->spinbox_group);
}

DispQtConfigpageSpinboxGroupbox::~DispQtConfigpageSpinboxGroupbox()
{
	if(this->spinbox_group)
		delete this->spinbox_group;
	this->close();
}

void DispQtConfigpageSpinboxGroupbox::spinboxgroup_applyall(void)
{
	if(this->spinbox_group)
		this->spinbox_group->spinboxgroup_applyall();
}

//--------------------------------------------------------------------------------------
DispQtConfigpageEditlineGroup::DispQtConfigpageEditlineGroup(MainWindow *main_window, QWidget *parent, const struct configpage_editline_s *p_editline_config, int nb_config) : QGridLayout(parent)
{
	QString qs;

	if(!p_editline_config || (nb_config <= 0))
		return;
	this->editline_config = p_editline_config;
	this->nb_of_editlineconfig = nb_config;

	mpxplay_debugf(DISPQT_DEBUGOUT_QE,"qe ecfg:%8.8X", (mpxp_ptrsize_t)p_editline_config);

	this->lineedits = new QLineEdit *[nb_config];
	if(!this->lineedits)
		return;

	this->setMargin(3);
	this->setSpacing(0);
	this->setColumnStretch(0, (nb_config == 1)? 1 : 10);  // FIXME: hack
	this->setColumnStretch(1, 1);

	for(int i = 0; i < nb_config; i++) {
		char *name = (char *)p_editline_config[i].name;
		QLabel *label = new QLabel();
		if(!label)
			return;
		if(p_editline_config[i].name)
			label->setText(QString::fromUtf8(name));
		else
			label->clear();
		label->setToolTip(QString::fromUtf8(p_editline_config[i].tooltip));
		label->setToolTipDuration(DISPQT_CONFIG_TOOLTIP_DURATION);
		this->addWidget(label, i, 0);
		QLineEdit *qe = new QLineEdit();
		this->lineedits[i] = qe;
		if(!qe)
			return;
		qe->setText(QString::fromUtf8(p_editline_config[i].var_ptr));
		qe->setMaxLength(p_editline_config->var_size - 1);
		qe->setToolTip(QString::fromUtf8(p_editline_config[i].tooltip));
		qe->setToolTipDuration(DISPQT_CONFIG_TOOLTIP_DURATION);
		this->addWidget(qe, i, 1);
		mpxplay_debugf(DISPQT_DEBUGOUT_QE,"qe: %8.8X l:%d \"%s\" %d %8.8X \"%s\"", (mpxp_ptrsize_t)qe, qe->text().length(),
				qe->text().toUtf8().data(), pds_strlen(p_editline_config[i].var_ptr), (mpxp_ptrsize_t)p_editline_config[i].var_ptr,
				p_editline_config[i].var_ptr);
	}
}

DispQtConfigpageEditlineGroup::~DispQtConfigpageEditlineGroup()
{
	if(this->lineedits)
		delete this->lineedits;
}

void DispQtConfigpageEditlineGroup::editlinegroup_applyall(void)
{
	mpxplay_debugf(DISPQT_DEBUGOUT_QE,"qea ecfg:%8.8X n:%d", (mpxp_ptrsize_t)this->editline_config, this->nb_of_editlineconfig);
	for(int i = 0; i < this->nb_of_editlineconfig; i++)
	{
		char *varptr = this->editline_config[i].var_ptr;
		if(!varptr)
			continue;
		*varptr = 0;
		QLineEdit *qe = this->lineedits[i];
		if(qe && !qe->text().isNull() && !qe->text().isEmpty())
		{
			int maxlen = min(qe->maxLength(), (MPXINI_MAX_CHARDATA_LEN - 1));
			pds_strncpy(varptr, qe->text().toUtf8().data(), maxlen);
			varptr[maxlen] = 0;
			pds_strcutspc(varptr);
			mpxplay_debugf(DISPQT_DEBUGOUT_QE,"qea: %8.8X l:%d \"%s\" -> %8.8X \"%s\" ml:%d",
					(mpxp_ptrsize_t)qe, qe->text().length(), qe->text().toUtf8().data(), (mpxp_ptrsize_t)varptr, varptr, maxlen);
		}
	}
}

DispQtConfigpageEditlineGroupbox::DispQtConfigpageEditlineGroupbox(MainWindow *main_window, QWidget *parent, QString box_title, const struct configpage_editline_s *editline_config, int nb_config) : QGroupBox(box_title, parent)
{
	this->editline_group = new DispQtConfigpageEditlineGroup(main_window, parent, editline_config, nb_config);
	this->setLayout(this->editline_group);
}

DispQtConfigpageEditlineGroupbox::~DispQtConfigpageEditlineGroupbox()
{
	if(this->editline_group)
		delete this->editline_group;
	this->close();
}

void DispQtConfigpageEditlineGroupbox::editlinegroupbox_applyall(void)
{
	if(this->editline_group)
		this->editline_group->editlinegroup_applyall();
}

//--------------------------------------------------------------------------------------
DispQtConfigpageComboboxGroup::DispQtConfigpageComboboxGroup(MainWindow *main_window, QWidget *parent, const struct configpage_comboboxline_s *p_comboboxline_configs, unsigned int nb_configs) : QGridLayout(parent)
{
	this->comboboxgroup_config = p_comboboxline_configs;
	this->nb_of_comboboxconfig = nb_configs;

	this->gui_elems = new void *[nb_configs];
	if(!this->gui_elems)
		return;

	this->setContentsMargins(3,0,3,3);
	this->setSpacing(2);
	this->setColumnStretch(0, 1);
	this->setColumnStretch(1, 1);

	for(int c = 0; c < nb_configs; c++, p_comboboxline_configs++) {
		unsigned int selected_item = 0;

		//if((DISPQT_CONFIG_COMBOTYPE_GET(p_comboboxline_configs->datatype) != DISPQT_CONFIG_COMBOTYPE_CHKBOX) || !p_comboboxline_configs->data_var)
		{
			QLabel *label = new QLabel(QString::fromUtf8(p_comboboxline_configs->title));
			if(label) {
				label->setToolTip(QString::fromUtf8(p_comboboxline_configs->tooltip));
				label->setToolTipDuration(DISPQT_CONFIG_TOOLTIP_DURATION);
				this->addWidget(label, c, 0);
			}
		}

		switch(DISPQT_CONFIG_COMBOTYPE_GET(p_comboboxline_configs->datatype))
		{
			case DISPQT_CONFIG_COMBOTYPE_CHKBOX:
			{
				unsigned long *varptr = (unsigned long *)p_comboboxline_configs->data_var;
				if(varptr) {
					QCheckBox *chkbox = new QCheckBox;
					if(!chkbox)
						return;
					//chkbox->setText(QString::fromUtf8(p_comboboxline_configs->title));
					if(p_comboboxline_configs->nb_values) {
						chkbox->setCheckable(true);
						bool setc = (*varptr & p_comboboxline_configs->nb_values)? true:false;
						chkbox->setChecked(setc);
					} else
						chkbox->setDisabled(true);
					chkbox->setToolTip(QString::fromUtf8(p_comboboxline_configs->tooltip));
					chkbox->setToolTipDuration(DISPQT_CONFIG_TOOLTIP_DURATION);
					this->addWidget(chkbox, c, 1, Qt::AlignCenter);
					this->gui_elems[c] = (void *)chkbox;
				} else {
					this->gui_elems[c] = NULL;
				}
				break;
			}
			case DISPQT_CONFIG_COMBOTYPE_SPINBOX:
			{
				unsigned long *varptr = (unsigned long *)p_comboboxline_configs->data_var;
				if(varptr) {
					QSpinBox *spinbox = new QSpinBox;
					if(!spinbox)
						return;
					spinbox->setAlignment(Qt::AlignRight);
					spinbox->setAccelerated(true);
					spinbox->setRange(DISPQT_CONFIG_SPINBOX_VALUNPACK_MIN(p_comboboxline_configs->nb_values), DISPQT_CONFIG_SPINBOX_VALUNPACK_MAX(p_comboboxline_configs->nb_values));
					spinbox->setValue(*varptr);
					this->addWidget(spinbox, c , 1);
					this->gui_elems[c] = (void *)spinbox;
				} else {
					this->gui_elems[c] = NULL;
				}
				break;
			}
			case DISPQT_CONFIG_COMBOTYPE_EDITLINE:
			{
				QLineEdit *qe = new QLineEdit();
				if(!qe)
					return;
				qe->setText(QString::fromUtf8((char *)p_comboboxline_configs->data_var));
				qe->setMaxLength(p_comboboxline_configs->nb_values - 1);
				this->addWidget(qe, c, 1);
				qe->setToolTip(QString::fromUtf8(p_comboboxline_configs->tooltip));
				qe->setToolTipDuration(DISPQT_CONFIG_TOOLTIP_DURATION);
				this->gui_elems[c] = (void *)qe;
				break;
			}
			case DISPQT_CONFIG_COMBOTYPE_COMBOBOX:
			{
				DispQtDialogElemComboBox *combo_box = new DispQtDialogElemComboBox(main_window);
				if(!combo_box)
					return;

				for(int i = 0; i < p_comboboxline_configs->nb_values; i++) {
					combo_box->addItem(QString::fromUtf8(p_comboboxline_configs->data_description[i]));
					if(p_comboboxline_configs->datatype & ARG_NUM) {
						if(p_comboboxline_configs->data_var && (*((unsigned long *)p_comboboxline_configs->data_var) == pds_atol((char *)p_comboboxline_configs->data_values[i])))
							selected_item = i;
					} else if(p_comboboxline_configs->datatype & ARG_CHAR) {
						char strtmp[MPXINI_MAX_CHARDATA_LEN];
						if(p_comboboxline_configs->datatype & ARG_POINTER)
							pds_strncpy(strtmp, *((char **)p_comboboxline_configs->data_var), MPXINI_MAX_CHARDATA_LEN);
						else
							pds_strncpy(strtmp, (char *)p_comboboxline_configs->data_var, MPXINI_MAX_CHARDATA_LEN);
						strtmp[MPXINI_MAX_CHARDATA_LEN - 1] = 0;
						pds_str_clean(strtmp);
						mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "char \"%s\" \"%s\"", strtmp, (char *)p_comboboxline_configs->data_values[i]);
						if(pds_stricmp(strtmp, (char *)p_comboboxline_configs->data_values[i]) == 0)
							selected_item = i;
					}
				}
				combo_box->setCurrentIndex(selected_item);
				this->addWidget(combo_box, c, 1);
				this->gui_elems[c] = (void *)combo_box;
				break;
			}
		}
	}
}

DispQtConfigpageComboboxGroup::~DispQtConfigpageComboboxGroup()
{
	if(this->gui_elems)
	{
		const struct configpage_comboboxline_s *comboboxline_config = this->comboboxgroup_config;
		for(int c = 0; c < this->nb_of_comboboxconfig; c++, comboboxline_config++)
		{
			if(!this->gui_elems[c])
				continue;
			switch(DISPQT_CONFIG_COMBOTYPE_GET(comboboxline_config->datatype))
			{
				case DISPQT_CONFIG_COMBOTYPE_CHKBOX:
					delete ((QCheckBox *)this->gui_elems[c]);
					break;
				case DISPQT_CONFIG_COMBOTYPE_SPINBOX:
					delete ((QSpinBox *)this->gui_elems[c]);
					break;
				case DISPQT_CONFIG_COMBOTYPE_EDITLINE:
					delete ((QLineEdit *)this->gui_elems[c]);
					break;
				case DISPQT_CONFIG_COMBOTYPE_COMBOBOX:
					delete ((DispQtDialogElemComboBox *)this->gui_elems[c]);
					break;
			}
		}
		delete this->gui_elems;
	}
}

void DispQtConfigpageComboboxGroup::comboboxgroup_applyall(void)
{
	const struct configpage_comboboxline_s *comboboxline_config = (struct configpage_comboboxline_s *)this->comboboxgroup_config;
	for(int c = 0; c < this->nb_of_comboboxconfig; c++, comboboxline_config++)
	{
		if(!this->gui_elems[c])
			continue;
		switch(DISPQT_CONFIG_COMBOTYPE_GET(comboboxline_config->datatype))
		{
			case DISPQT_CONFIG_COMBOTYPE_CHKBOX:
			{
				QCheckBox *chkbox = (QCheckBox *)this->gui_elems[c];
				unsigned long *varptr = (unsigned long *)comboboxline_config->data_var;
				if(!varptr)
					continue;
				if(!chkbox->isCheckable())
					continue;
				unsigned long chkbit = comboboxline_config->nb_values;
				if(chkbox->isChecked())
					funcbit_enable(*varptr, chkbit);
				else
					funcbit_disable(*varptr, chkbit);
				break;
			}
			case DISPQT_CONFIG_COMBOTYPE_SPINBOX:
			{
				unsigned long *varptr = (unsigned long *)comboboxline_config->data_var;
				if(!varptr || !this->gui_elems[c])
					break;;
				*varptr = ((QSpinBox *)this->gui_elems[c])->value();
				break;
			}
			case DISPQT_CONFIG_COMBOTYPE_EDITLINE:
			{
				char *varptr = (char *)comboboxline_config->data_var;
				if(!varptr)
					break;
				*varptr = 0;
				QLineEdit *qe = (QLineEdit *)this->gui_elems[c];
				if(qe && !qe->text().isNull() && !qe->text().isEmpty())
				{
					int maxlen = min(qe->maxLength(), (MPXINI_MAX_CHARDATA_LEN - 1));
					pds_strncpy(varptr, qe->text().toUtf8().data(), maxlen);
					varptr[maxlen] = 0;
					pds_strcutspc(varptr);
				}
				break;
			}
			case DISPQT_CONFIG_COMBOTYPE_COMBOBOX:
			{
				int selected_item = ((DispQtDialogElemComboBox *)this->gui_elems[c])->currentIndex();
				if(selected_item >= comboboxline_config->nb_values || !comboboxline_config->data_var || !comboboxline_config->data_values[selected_item])
					return;
				if(comboboxline_config->datatype & ARG_NUM)
					*((unsigned long *)comboboxline_config->data_var) = pds_atol((char *)comboboxline_config->data_values[selected_item]);
				else if(comboboxline_config->datatype & ARG_CHAR)
				{
					if(comboboxline_config->datatype & ARG_POINTER)
						*(char **)comboboxline_config->data_var = (char *)comboboxline_config->data_values[selected_item];
					else {
						pds_strncpy((char *)comboboxline_config->data_var, (char *)comboboxline_config->data_values[selected_item], MPXINI_MAX_CHARDATA_LEN);
						((char *)comboboxline_config->data_var)[MPXINI_MAX_CHARDATA_LEN - 1] = 0;
					}
				}
				break;
			}
		}
	}
}

DispQtConfigpageComboboxGroupbox::DispQtConfigpageComboboxGroupbox(MainWindow *main_window, QWidget *parent, QString title, const struct configpage_comboboxline_s *p_comboboxline_configs, unsigned int nb_configs) : QGroupBox(title, parent)
{
	this->combobox_group = new DispQtConfigpageComboboxGroup(main_window, parent, p_comboboxline_configs, nb_configs);
	this->setLayout(this->combobox_group);
}

DispQtConfigpageComboboxGroupbox::~DispQtConfigpageComboboxGroupbox()
{
	if(this->combobox_group)
		delete this->combobox_group;
	this->close();
}

void DispQtConfigpageComboboxGroupbox::comboboxgroupbox_applyall(void)
{
	if(this->combobox_group)
		this->combobox_group->comboboxgroup_applyall();
}
