//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
////function: DVB scan dialog tab

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUGOUT_EPG stdout
#define DISPQT_DEBUGOUT_SCAN stdout

#include <QtGui>
#include <QtWidgets>
#include <QtWinExtras/QtWinExtras>
#include "disp_qt.h"
#include "moc_dvb_epg.h"
#include "moc_mainwindow.h"
#include "display/display.h"

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

extern "C" {
 extern unsigned long mpxplay_config_dvbepg_control_flags, mpxplay_config_dvbepg_pastevents_hours;
 extern const char *mpxplay_drvdtv_protocol_family_list[5];
}

//=========================================================================================================================

static char *dvb_epg_selected_scan_device = (char *)mpxplay_drvdtv_protocol_family_list[4]; // default is DVB-T
static char dvb_scan_freq_start[16] = "474000", dvb_scan_freq_end[16] = "858000", dvb_scan_freq_bandwidth[16] = "8000"; // TODO: read this from dtv database

static struct configpage_comboboxline_s dvb_epg_scan_settings_group[] = {
 {"Device type    ", (DISPQT_CONFIG_COMBOTYPE_COMBOBOX | ARG_CHAR | ARG_POINTER), &dvb_epg_selected_scan_device, 5, &mpxplay_drvdtv_protocol_family_list[0], &mpxplay_drvdtv_protocol_family_list[0], "Select DVB device type for scanning and listing\nnote: C/S/T devices scan C2/S2/T2 and ISDB channels too"},
 {"Frequency start",  DISPQT_CONFIG_COMBOTYPE_EDITLINE, &dvb_scan_freq_start[0], 16, NULL, NULL, "DVB scan start frequency in kHz" },
 {"Frequency end  ",  DISPQT_CONFIG_COMBOTYPE_EDITLINE, &dvb_scan_freq_end[0], 16, NULL, NULL, "DVB scan end frequency in kHz" },
 {"Bandwidth      ",  DISPQT_CONFIG_COMBOTYPE_EDITLINE, &dvb_scan_freq_bandwidth[0], 16, NULL, NULL, "DVB scan bandwidth in kHz" },
};

#define DISPQT_DVBEPG_SCANSETTINGS_COMBOX_MAX (sizeof(dvb_epg_scan_settings_group) / sizeof(struct configpage_comboboxline_s))

static unsigned int dvbepg_sort_control;
static const char *dvbepg_sort_values[] = {"0", "1"};
static const char *dvbepg_sort_description[] = {"Program ID   ", "Frequency    "}; // FIXME: extra spaces

static struct configpage_comboboxline_s dvb_epg_scan_result_control_group[] = {
 {"Sort channels by", (DISPQT_CONFIG_COMBOTYPE_COMBOBOX | ARG_NUM), &dvbepg_sort_control, (sizeof(dvbepg_sort_values) / sizeof(char *)), &dvbepg_sort_values[0], &dvbepg_sort_description[0], "Select program list sorting mode"},
 {"Hide encrypted",    DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_FILTER_ENCRYPTED, NULL, NULL, "Hide (don't list) encrypted program entries\nNote: this is a general DVB setting, not only scan related" },
 {"Hide non-media",    DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_FILTER_NONMEDIA, NULL, NULL, "Hide (don't list) invalid (non-media) program entries\nwhich has no name or valid channel id\nNote: this is a general DVB setting, not only scan related" },
 {"ToolTip items",     DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_EPGWINDOW_ITEMS_TOOLTIP, NULL, NULL, "Show EPG item titles in ToolTips over EPG window" },
 {"Past events hours", DISPQT_CONFIG_COMBOTYPE_SPINBOX, &mpxplay_config_dvbepg_pastevents_hours, DISPQT_CONFIG_SPINBOX_VALPACK_MINMAX(DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MIN, DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MAX), NULL, NULL, "Show past events in the EPG dialog\ntill the configured hours"},
};

#define DISPQT_DVBEPG_SCANRESULTCONTROL_COMBOX_MAX (sizeof(dvb_epg_scan_result_control_group) / sizeof(struct configpage_comboboxline_s))

DispQtDVBScanGWidget::DispQtDVBScanGWidget(MainWindow *mainwindow, QWidget *parent) : QWidget(parent)
{
	this->main_window = mainwindow;
	this->scan_handler = NULL;

	dvbepg_sort_control = mpxplay_config_dvbepg_control_flags & MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORTTYPE_MASK;

	this->button_startscan = new DispQtDialogElemPushButton(mainwindow, this, tr("Scan Chans"));
	this->button_startscan->setToolTip("Start scanning for additional channels\nfor the selected device type\n(without clearing the stored channels)");
	this->button_startscan->setFocusPolicy(Qt::NoFocus);
	this->button_startscan->setFixedHeight(DISPQT_EPG_SCAN_BUTTON_HEIGHT);
	this->button_loadepg = new DispQtDialogElemPushButton(mainwindow, this, tr("Scan EPG"));
	this->button_loadepg->setToolTip("Load all EPG infos for scanned channels\nand scan frequencies if it's not done yet");
	this->button_loadepg->setFocusPolicy(Qt::NoFocus);
	this->button_loadepg->setFixedHeight(DISPQT_EPG_SCAN_BUTTON_HEIGHT);
	this->button_stopscan = new DispQtDialogElemPushButton(mainwindow, this, tr("Stop Scan"));
	this->button_stopscan->setToolTip("Stop scanning of channels");
	this->button_stopscan->setFocusPolicy(Qt::NoFocus);
	this->button_stopscan->setFixedHeight(DISPQT_EPG_SCAN_BUTTON_HEIGHT);
	this->button_loadchannels = new DispQtDialogElemPushButton(mainwindow, this, tr("Load Chans"));
	this->button_loadchannels->setToolTip("Reload channels from the database\nfor the selected device type\n(with applying the control changes)");
	this->button_loadchannels->setFocusPolicy(Qt::NoFocus);
	this->button_loadchannels->setFixedHeight(DISPQT_EPG_SCAN_BUTTON_HEIGHT);
	this->button_clearchannels = new DispQtDialogElemPushButton(mainwindow, this, tr("Clear Chans"));
	this->button_clearchannels->setToolTip("Remove all channels and\nfrequencies from the database\nfor the selected device type");
	this->button_clearchannels->setFocusPolicy(Qt::NoFocus);
	this->button_clearchannels->setFixedHeight(DISPQT_EPG_SCAN_BUTTON_HEIGHT);

	QVBoxLayout *buttonsLayout = new QVBoxLayout;
	QWidget *buttonsWidget = new QWidget;
	buttonsLayout->setMargin(0);
	buttonsLayout->setSpacing(2);
	buttonsLayout->addWidget(this->button_startscan);
	buttonsLayout->addWidget(this->button_loadepg);
	buttonsLayout->addWidget(this->button_stopscan);
	buttonsLayout->addWidget(this->button_loadchannels);
	buttonsLayout->addWidget(this->button_clearchannels);
	buttonsLayout->setAlignment(Qt::AlignRight);
	buttonsWidget->setLayout(buttonsLayout);

	this->combobox_dvb_scan_settings = new DispQtConfigpageComboboxGroupbox(main_window, this, "Scan settings", &dvb_epg_scan_settings_group[0], DISPQT_DVBEPG_SCANSETTINGS_COMBOX_MAX);
	this->combobox_dvb_result_control = new DispQtConfigpageComboboxGroupbox(main_window, this, "Result control", &dvb_epg_scan_result_control_group[0], DISPQT_DVBEPG_SCANRESULTCONTROL_COMBOX_MAX);

	QHBoxLayout *settingsAndButtonsLayout = new QHBoxLayout;
	QWidget *settingsAndButtonsWidget = new QWidget(this);
	settingsAndButtonsLayout->setMargin(0);
	settingsAndButtonsLayout->addWidget(this->combobox_dvb_scan_settings);
	settingsAndButtonsLayout->addWidget(this->combobox_dvb_result_control);
	settingsAndButtonsLayout->addWidget(buttonsWidget);
	settingsAndButtonsLayout->addStretch(1);
	settingsAndButtonsWidget->setLayout(settingsAndButtonsLayout);

	this->label_progress = new QLabel("Scan progress (scan is stopped):");

	this->progressbar_scan = new DispQtDialogElemProgressBar(mainwindow, this);
	this->progressbar_scan->setRange(0, 100);
	this->progressbar_scan->setValue(0);

	QLabel *label_progtable = new QLabel("List of found channels and programs:");

	this->programtable_scan = new DispQtDVBScanProgramTable(mainwindow, this);

	this->dvbscan_config_style_apply(true);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(settingsAndButtonsWidget);
	mainLayout->addWidget(this->label_progress);
	mainLayout->addWidget(this->progressbar_scan);
	mainLayout->addWidget(label_progtable);
	mainLayout->addWidget(this->programtable_scan);
	setLayout(mainLayout);

	connect(this->button_startscan, SIGNAL(pressed()), this, SLOT(dvb_startscan_button_handler()));
	connect(this->button_loadepg, SIGNAL(pressed()), this, SLOT(dvb_loadepg_button_handler()));
	connect(this->button_stopscan, SIGNAL(pressed()), this, SLOT(dvb_stopscan_button_handler()));
	connect(this->button_loadchannels, SIGNAL(pressed()), this, SLOT(dvb_loadchannels_button_handler()));
	connect(this->button_clearchannels, SIGNAL(pressed()), this, SLOT(dvb_clearchannels_button_handler()));
}

void DispQtDVBScanGWidget::dvbscan_config_style_apply(bool initial)
{
	if(!initial)
	{
		this->button_startscan->pushbutton_set_style(initial);
		this->button_loadepg->pushbutton_set_style(initial);
		this->button_stopscan->pushbutton_set_style(initial);
		this->button_loadchannels->pushbutton_set_style(initial);
		this->button_clearchannels->pushbutton_set_style(initial);
	}
	this->progressbar_scan->progressbar_set_style(initial);
	this->programtable_scan->progtable_style_apply(initial);
}

void DispQtDVBScanGWidget::keyPressEvent(QKeyEvent *event)
{
	if(event->key() != Qt::Key_Escape)
		this->main_window->handle_keypressevent(event);
	else
		QWidget::keyPressEvent(event);
}

void DispQtDVBScanGWidget::dvbscanwidget_send_epginfos(struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	if(funcbit_test(prot_data->prot_flags, MPXPLAY_DVBEPG_PROTOCOL_FLAG_SCANFINISH))
	{
		this->label_progress->setText("Scan is finished!");
		if(this->progressbar_scan->value() != 0) // not stopped by Stop button
		{
			this->progressbar_scan->setRange(0, 100);
			this->progressbar_scan->setValue(100);
		}
	}
	if(funcbit_test(prot_data->prot_flags, (MPXPLAY_DVBEPG_PROTOCOL_FLAG_SCANFINISH | MPXPLAY_DVBEPG_PROTOCOL_FLAG_FULLDATA))) // FLAG_FULLDATA is initiated by other buttons (load, clear)
	{
		this->programtable_scan->progtable_clear();
		this->programtable_scan->progtable_send_epginfos(prot_data);
		this->dvb_stopscan_init();
	}
	else if(prot_data->frequency_data_chain)
	{
		this->progressbar_scan->setValue(prot_data->frequency_data_chain->frequency / 1000);
		this->programtable_scan->progtable_send_epginfos(prot_data);
	}
}

//-------------------------------------------------------------------------------------------
static void dvbepg_scan_scanning_start(void *dvbscaninfo)
{
	struct dispqt_dvbepg_startstop_info_s *dvbinfo = (struct dispqt_dvbepg_startstop_info_s *)dvbscaninfo;
	int err_code = 0;
	void *mdfd;
	char sout[256];

	if(!dvbinfo)
		return;
	mdfd = mpxplay_diskdrive_file_open(NULL, &dvbinfo->dvb_filename[0], O_RDONLY, &err_code);
	if(!mdfd)
	{
		snprintf(sout, sizeof(sout), "DVB scanning start failed for\n%s\n(missing device or wrong settings)", dvbinfo->dvb_filename);
		display_textwin_openwindow_errormsg_ok(sout);
		goto err_out_start;
	}
	*dvbinfo->filehandler = mdfd;

err_out_start:
	pds_free(dvbinfo);
}

void DispQtDVBScanGWidget::dvb_scan_apply_settings(void)
{
	this->combobox_dvb_scan_settings->comboboxgroupbox_applyall();
	this->combobox_dvb_result_control->comboboxgroupbox_applyall();
	funcbit_copy(mpxplay_config_dvbepg_control_flags, dvbepg_sort_control, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORTTYPE_MASK);
}

void DispQtDVBScanGWidget::dvb_start_scan_handler(bool load_epg)
{
	struct dispqt_dvbepg_startstop_info_s *dvbinfo;
	int freq_start, freq_end, freq_bandwidth;
	char str_loadepg[32];

	if(this->scan_handler)
		return;

	dvbinfo = (struct dispqt_dvbepg_startstop_info_s *)pds_calloc(1, sizeof(struct dispqt_dvbepg_startstop_info_s));
	if(!dvbinfo)
		return;

	this->programtable_scan->progtable_clear();
	this->label_progress->setText("Scan is running ...");
	this->dvb_scan_apply_settings();

	freq_start = pds_atol(dvb_scan_freq_start);
	freq_end = pds_atol(dvb_scan_freq_end);
	freq_bandwidth = pds_atol(dvb_scan_freq_bandwidth);
	if(freq_start < 10000)
		freq_start = 474000;
	if(freq_bandwidth <= 1000)
		freq_bandwidth = 8000;
	if(freq_end <= freq_start)
		freq_end = freq_start + freq_bandwidth - 1000;

	this->progressbar_scan->setRange(freq_start, freq_end);
	this->progressbar_scan->setValue(freq_start);

	if(load_epg)
		snprintf(str_loadepg, sizeof(str_loadepg), ":dvb-scan-epgload=1");
	else
		str_loadepg[0] = 0;
	snprintf(dvbinfo->dvb_filename, sizeof(dvbinfo->dvb_filename), "%s//dvb-scan-begin=%d:dvb-scan-end=%d:dvb-scan-bandwidth=%d%s",
			dvb_epg_selected_scan_device, freq_start, freq_end, freq_bandwidth, str_loadepg);
	dvbinfo->filehandler = &this->scan_handler;
	mpxplay_dispqt_mainthread_callback_init((void *)dvbepg_scan_scanning_start, (void *)dvbinfo, 0);
}

void DispQtDVBScanGWidget::dvb_startscan_button_handler(void)
{
	dvb_start_scan_handler(false);
}

void DispQtDVBScanGWidget::dvb_loadepg_button_handler(void)
{
	dvb_start_scan_handler(true);
}

static void dvbepg_scan_scanning_stop(void *dvbscaninfo)
{
	struct dispqt_dvbepg_startstop_info_s *dvbinfo = (struct dispqt_dvbepg_startstop_info_s *)dvbscaninfo;
	if(!dvbinfo)
		return;
	if(dvbinfo->filehandler && *dvbinfo->filehandler)
	{
		void *mdfd = *dvbinfo->filehandler;
		*dvbinfo->filehandler = NULL;
		mpxplay_diskdrive_file_close(mdfd);
	}
	pds_free(dvbinfo);
}

void DispQtDVBScanGWidget::dvb_stopscan_init(void)
{
	if(!this->scan_handler)
		return;
	struct dispqt_dvbepg_startstop_info_s *dvbinfo = (struct dispqt_dvbepg_startstop_info_s *)pds_calloc(1, sizeof(struct dispqt_dvbepg_startstop_info_s));
	if(!dvbinfo)
		return;
	dvbinfo->filehandler = &this->scan_handler;
	mpxplay_dispqt_mainthread_callback_init((void *)dvbepg_scan_scanning_stop, (void *)dvbinfo, 0);
}

void DispQtDVBScanGWidget::dvb_stopscan_button_handler(void)
{
	this->label_progress->setText("Scan is stopped !");
	this->progressbar_scan->setRange(0, 100);
	this->progressbar_scan->setValue(0);
	this->dvb_stopscan_init();
}

static void dvbepg_scan_load_channels(void)
{
	int protocol_id = mpxplay_drvdtv_get_protocol_id(dvb_epg_selected_scan_device);
	if(protocol_id >= 0)
	{
		struct mpxplay_dvb_epg_protocol_data_s *prot_data;
		mpxplay_dtvdrv_database_frequencies_sort_by_type(protocol_id, (mpxplay_config_dvbepg_control_flags & MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORTTYPE_MASK));
		prot_data = mpxplay_dtvdrv_database_dup_protocol_info(protocol_id, mpxplay_config_dvbepg_control_flags);
		if(prot_data)
		{
			funcbit_enable(prot_data->prot_flags, MPXPLAY_DVBEPG_PROTOCOL_FLAG_FULLDATA);
			if(!mpxplay_dispqt_epgdialog_send_scaninfos((void *)prot_data))
				mpxplay_dtvdrv_database_clear_protocols(prot_data); // dialog is closed (should not happen here)
		}
	}
}

void DispQtDVBScanGWidget::dvb_loadchannels_button_handler(void)
{
	this->dvb_scan_apply_settings();
	mpxplay_dispqt_mainthread_callback_init((void *)&dvbepg_scan_load_channels, NULL, 0);
}

static void dvbepg_scan_clearchannels_confirm_apply(void)
{
	int protocol_id = mpxplay_drvdtv_get_protocol_id(dvb_epg_selected_scan_device);
	if(protocol_id >= 0)
	{
		struct mpxplay_dvb_epg_protocol_data_s *prot_data;
		mpxplay_dtvdrv_database_clear_freqencies_of_protocol(protocol_id);
		// send an empty list to clear scan program table
		prot_data = mpxplay_dtvdrv_database_dup_protocol_info(protocol_id, mpxplay_config_dvbepg_control_flags);
		if(prot_data)
		{
			funcbit_enable(prot_data->prot_flags, MPXPLAY_DVBEPG_PROTOCOL_FLAG_FULLDATA);
			if(!mpxplay_dispqt_epgdialog_send_scaninfos((void *)prot_data))
				mpxplay_dtvdrv_database_clear_protocols(prot_data); // dialog is closed (should not happen here)
		}
	}
}

static void dvbepg_scan_clearchannels_confirm_init(void)
{
	display_textwin_openwindow_confirm(0, (char *)"Clear channels", (char *)"Clear/remove all channels of the selected device.\n\nAre you sure?\n", (void *)&dvbepg_scan_clearchannels_confirm_apply, NULL);
}

void DispQtDVBScanGWidget::dvb_clearchannels_button_handler(void)
{
	mpxplay_dispqt_mainthread_callback_init((void *)&dvbepg_scan_clearchannels_confirm_init, NULL, 0);
}

//-------------------------------------------------------------------------------------------
DispQtDVBScanProgramTable::DispQtDVBScanProgramTable(MainWindow *mainwindow, QWidget *parent) :  QTableView(parent)
{
	this->main_window = mainwindow;
//	this->font_name_curr[0] = 0;
	this->row_height = 16;
	this->model_proglist = this->model_create();

	this->vscrollbar = new DispQtDialogElemScrollBar(mainwindow, Qt::Vertical, this);
	this->setVerticalScrollBar(this->vscrollbar);
	this->hscrollbar = new DispQtDialogElemScrollBar(mainwindow, Qt::Horizontal, this);
	this->setHorizontalScrollBar(this->hscrollbar);

	//this->table_config_font();
	this->progtable_style_apply(true);

	this->header_view = new DispQtDialogElemHeaderView(main_window, Qt::Horizontal, parent);
	this->setHorizontalHeader(this->header_view);

	this->setModel(this->model_proglist);
	this->table_setsizes();

	this->setFocusPolicy(Qt::NoFocus);
	this->verticalHeader()->hide();
	this->verticalHeader()->setDefaultSectionSize(this->row_height); // TODO: font config
	this->setTabKeyNavigation(false);
	this->setShowGrid(false);
	this->setSelectionBehavior(QAbstractItemView::SelectRows);
	this->setSelectionMode(QAbstractItemView::SingleSelection);
	this->setEditTriggers(QTableWidget::NoEditTriggers);
	this->setSortingEnabled(false);
}

void DispQtDVBScanProgramTable::progtable_style_apply(bool initial)
{
	if(this->main_window->mainwin_guilayout_is_dark_background())
	{
		this->setStyleSheet("QTableView {background:rgba(0,60,100,110); color: white}");
	}
	else if(!initial)
	{
		this->setStyleSheet(QString());
	}
	if(!initial)
	{
		this->header_view->headerview_set_style(this->main_window->mainwin_guibkg_is_transparent());
		this->vscrollbar->scrollbar_set_style(initial);
		this->hscrollbar->scrollbar_set_style(initial);
	}
}

void DispQtDVBScanProgramTable::progtable_clear(void)
{
	this->model_proglist->removeRows(0, this->model_proglist->rowCount());
	this->selectRow(0);
}

void DispQtDVBScanProgramTable::progtable_send_epginfos(struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	QStandardItemModel *model = this->model_proglist;
	struct mpxplay_dvb_epg_frequency_data_s *freq_data = prot_data->frequency_data_chain;
	unsigned int rowcount = model->rowCount();
	char sout[32];

	while(freq_data)
	{
		struct mpxplay_dvb_epg_program_data_s *prog_data = freq_data->program_data_chain;
		while(prog_data)
		{
			if(!model->insertRows(rowcount, 1))
				break;

			snprintf(sout, sizeof(sout), "%3d", prog_data->program_id);
			model->setData(model->index(rowcount, 0), QString::fromLatin1(sout));
			snprintf(sout, sizeof(sout), "%3d", prog_data->channel_id);
			model->setData(model->index(rowcount, 1), QString::fromLatin1(sout));
			snprintf(sout, sizeof(sout), "%6d", freq_data->frequency / 1000);
			model->setData(model->index(rowcount, 2), QString::fromLatin1(sout));
			model->setData(model->index(rowcount, 3), QString::fromUtf8(prog_data->program_name));
			model->setData(model->index(rowcount, 4), QString::fromLatin1((funcbit_test(prog_data->ctrl_flags, INFFMPG_EPGPRGCTRLFLAG_ENCRYPTED_PRG))? "Yes" : ""));
			model->setData(model->index(rowcount, 5), QString::fromLatin1((mpxplay_dtvdrv_database_check_program_validity(freq_data, prog_data, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_FILTER_NONMEDIA))? "" : "No"));
			snprintf(sout, sizeof(sout), "%3d%%", freq_data->freq_signal_quality);
			model->setData(model->index(rowcount, 6), QString::fromLatin1(sout));
			rowcount++;

			prog_data = prog_data->next_program;
		}
		freq_data = freq_data->next_frequency;
	}
}

QStandardItemModel *DispQtDVBScanProgramTable::model_create(void)
{
	QStandardItemModel *model;

	model = new QStandardItemModel(0, 7, this);
	model->setHeaderData(0, Qt::Horizontal, QObject::tr(" Prog ID "));
	model->setHeaderData(1, Qt::Horizontal, QObject::tr(" Chan ID "));
	model->setHeaderData(2, Qt::Horizontal, QObject::tr(" Freqency "));
	model->setHeaderData(3, Qt::Horizontal, QObject::tr(" Program name "));
	model->setHeaderData(4, Qt::Horizontal, QObject::tr(" Encrypted "));
	model->setHeaderData(5, Qt::Horizontal, QObject::tr(" Media "));
	model->setHeaderData(6, Qt::Horizontal, QObject::tr(" Signal "));

	return model;
}

void DispQtDVBScanProgramTable::table_setsizes(void)
{
	this->horizontalHeader()->setSectionResizeMode(0, QHeaderView::ResizeToContents);
	this->horizontalHeader()->setSectionResizeMode(1, QHeaderView::ResizeToContents);
	this->horizontalHeader()->setSectionResizeMode(2, QHeaderView::ResizeToContents);
	this->horizontalHeader()->setSectionResizeMode(3, QHeaderView::Stretch);
	this->horizontalHeader()->setSectionResizeMode(4, QHeaderView::ResizeToContents);
	this->horizontalHeader()->setSectionResizeMode(5, QHeaderView::ResizeToContents);
	this->horizontalHeader()->setSectionResizeMode(6, QHeaderView::ResizeToContents);
}

void DispQtDVBScanProgramTable::mouseDoubleClickEvent(QMouseEvent *event)
{
	if(!event)
		return;
	if(!(event->button() & Qt::LeftButton))
		return;
	int row_number = rowAt(event->y());
	QStandardItemModel *model = this->model_proglist;
	struct dispqt_dvbepg_startstop_info_s *dvbinfo = (struct dispqt_dvbepg_startstop_info_s *)pds_calloc(1, sizeof(struct dispqt_dvbepg_startstop_info_s));
	if(!dvbinfo)
		return;
	QApplication::setOverrideCursor(Qt::BusyCursor);
	mpxplay_dtvdrv_assemble_filename(dvbinfo->dvb_filename, sizeof(dvbinfo->dvb_filename), mpxplay_drvdtv_get_protocol_id(dvb_epg_selected_scan_device),
			model->data(model->index(row_number, 2)).toInt() * 1000, model->data(model->index(row_number, 0)).toInt());
	mpxplay_dispqt_mainthread_callback_init((void *)mpxplay_dispqt_dvbepg_program_start, (void *)dvbinfo, 0);
	mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "table doubleclick: %d ", row_number);
	event->accept();
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
