//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2021 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: video render base functions (subtitle rendering)

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUGOUT_ERROR    stderr
#define DISPQT_DEBUGOUT_SUBTITLE stdout

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

#include "moc_video_qt.h"
#include "moc_mainwindow.h"

#define DISPQT_SUBTITLE_SHADOW_SIZE 2
#define DISPQT_SUBTITLE_BORDER_SIZE 8
#define DISPQT_SUBTITLE_SCALE_BASE 100

MMCDispQtVideoRendererBase::MMCDispQtVideoRendererBase(MainWindow *mainwindow, QWidget *parent)
{
	this->main_window = mainwindow;
	this->parent_widget = parent;
	this->subtitle_line_flags = 0;
	this->subtitle_screen_scale = DISPQT_SUBTITLE_SCALE_BASE;
	this->subtitle_bitmap_scale = DISPQT_SUBTITLE_SCALE_BASE;
	this->subtitle_font_char_pixelsize = this->subtitle_font_pointsize = 32;
	this->last_font_text_name[0] = 0;
	this->renderbase_subtitle_reset();
};

void MMCDispQtVideoRendererBase::renderbase_subtitle_reset(void)
{
	this->subtitle_pic_globalpos_x = this->subtitle_pic_globalpos_y = 0;
	this->subtitle_pic_width = this->subtitle_pic_heigth = 0;
	this->subtitle_pic_renderpos_y = 0;
	this->was_subtitle_rendered = false;
	this->last_subtitle_nb_lines = 0;
	this->last_bitmap_width = this->last_bitmap_height = 0;
	pds_memset(&this->last_line_texts[0][0], 0, sizeof(this->last_line_texts));
}

// initialize subtitle font for rendering, scale the font size relative to the video window size
bool MMCDispQtVideoRendererBase::renderbase_subtitle_font_init(struct dispqt_video_surface_info_s *video_surface_infos, unsigned int control_flags)
{
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;
	bool font_change = false;
	int scale;

	if(video_surface_infos->ffmpeg_output_frame && funcbit_test(control_flags, (DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_SCALETOVIDEO|DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO)))
		scale = 90 * video_surface_infos->ffmpeg_output_frame->width / video_surface_infos->fullscreen_size_x;
	else
		scale = DISPQT_SUBTITLE_SCALE_BASE * video_surface_infos->window_size_y / video_surface_infos->fullscreen_size_y; // TODO: maybe we should always use width for scaling
	if(scale < 5)
		scale = 5;

	if(gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE] && (pds_strcmp(this->last_font_text_name, gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE]) != 0))
	{
		this->font_subtitle.fromString(QString::fromUtf8(gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE]));
		pds_strncpy(this->last_font_text_name, gcfg->font_text_name[DISPQT_FONTTEXTID_SUBTITLE], sizeof(this->last_font_text_name));
		this->last_font_text_name[sizeof(this->last_font_text_name) - 1] = 0;
		this->subtitle_font_pointsize = this->font_subtitle.pointSize();
		font_change = true;
	}
	if(this->subtitle_screen_scale != scale)
	{
		this->subtitle_screen_scale = scale;
		font_change = true;
	}
	if(font_change)
	{
		this->subtitle_font_char_pixelsize = this->subtitle_font_pointsize * this->subtitle_screen_scale / DISPQT_SUBTITLE_SCALE_BASE;
		this->font_subtitle.setPointSize(this->subtitle_font_char_pixelsize);
		this->font_subtitle.setItalic(false);
		funcbit_disable(this->subtitle_line_flags, INFFMPEG_ASS_DECODED_FLAG_ITALIC);
	}

	if(video_surface_infos->ffmpeg_output_frame)
	{
		int new_bitmap_scale = this->subtitle_screen_scale;
		if(video_surface_infos->fullscreen_size_x > video_surface_infos->ffmpeg_output_frame->width)
			new_bitmap_scale = new_bitmap_scale * video_surface_infos->fullscreen_size_x / video_surface_infos->ffmpeg_output_frame->width;
		new_bitmap_scale = new_bitmap_scale * this->subtitle_font_pointsize / 40;
		if(this->subtitle_bitmap_scale != new_bitmap_scale)
		{
			this->subtitle_bitmap_scale = new_bitmap_scale;
			font_change = true;
		}
	}

	return font_change;
}

// calculate the size and position of the subtitle subpicture/frame (from the number of subtitle text lines and lengths)
// returns DISPQT_RENDERBASE_SUBPAINT_RETVAL_
int MMCDispQtVideoRendererBase::renderbase_subtitle_subpicture_calc(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int control_flags)
{
	int new_pos_x, new_pos_y, win_size_x, win_size_y, retval = DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP;

	if(!subtitle_infos)
		return retval;

	pds_memset(&this->subtitle_line_pixellens[0], 0, sizeof(this->subtitle_line_pixellens));

	if(funcbit_test(control_flags, (DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_SCALETOVIDEO |DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO)))
	{
		if(!video_surface_infos->ffmpeg_output_frame)
			return retval;
	}

	if(funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO))
	{
		win_size_x = video_surface_infos->ffmpeg_output_frame->width;
		win_size_y = video_surface_infos->ffmpeg_output_frame->height;
	}
	else
	{
		win_size_x = video_surface_infos->window_size_x;
		win_size_y = video_surface_infos->window_size_y;
	}

	if(subtitle_infos->subtitle_nb_lines)
	{
		int new_size_x = 0, new_size_y = 0;

		if(!funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_DRAWEVERY))
		{
			if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_TEXT)
			{
				// check difference between the new and previous subtitle texts (to not prepare new sub-picture, if not required)
				if(this->last_subtitle_nb_lines == subtitle_infos->subtitle_nb_lines)
				{
					bool texts_differ = false;
					for(int i = 0; i < subtitle_infos->subtitle_nb_lines; i++)
					{
						if(pds_strncmp(subtitle_infos->subtitle_datas[i].subtitle_line_data, &this->last_line_texts[i][0], sizeof(this->last_line_texts[i]) - 1) != 0)
						{
							pds_strncpy(&this->last_line_texts[i][0], subtitle_infos->subtitle_datas[i].subtitle_line_data, sizeof(this->last_line_texts[i]) - 1);
							this->last_line_texts[i][sizeof(this->last_line_texts[i]) - 1] = 0;
							texts_differ = true;
							break;
						}
					}
					if(!texts_differ)
						funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER);
				}
				if(!funcbit_test(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER))
				{
					// save new subtitle texts for a further reference/compare
					for(int i = 0; i < INFFMPEG_ASS_DECODED_MAX_LINES; i++)
					{
						if(subtitle_infos->subtitle_datas[i].subtitle_line_data && *(subtitle_infos->subtitle_datas[i].subtitle_line_data))
						{
							pds_strncpy(&this->last_line_texts[i][0], subtitle_infos->subtitle_datas[i].subtitle_line_data, sizeof(this->last_line_texts[i]) - 1);
							this->last_line_texts[i][sizeof(this->last_line_texts[i]) - 1] = 0;
						}
						else
						{
							pds_memset(&this->last_line_texts[i][0], 0, sizeof(this->last_line_texts[i]));
						}
					}
				}
			}
			else if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_BITMAP)
			{
				struct ffmpegvideo_subtitle_data_s *subt_data = &subtitle_infos->subtitle_datas[0];
				int image_size = subt_data->subtitle_bitmap_h * subt_data->subtitle_line_info;
				if(image_size)
				{
					int check_point = (image_size < (sizeof(this->last_line_texts[0]) * 2))? 0 : (image_size / 3); // check the bitmap at the 1/3 part of the sub-picture
					int check_len = min(image_size, sizeof(this->last_line_texts[0]));
					// check difference between the new and previous subtitle bitmaps (to not prepare new sub-picture, if not required)
					if((this->last_bitmap_width != subt_data->subtitle_bitmap_w) || (this->last_bitmap_height != subt_data->subtitle_bitmap_h))
					{
						this->last_bitmap_width = subt_data->subtitle_bitmap_w;
						this->last_bitmap_height = subt_data->subtitle_bitmap_h;
					}
					else if(pds_memcmp(this->last_line_texts[0], subt_data->subtitle_line_data + check_point, check_len) == 0)
					{
						funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER);
					}
					// save new subtitle bitmap part (middle of bitmap) for a further reference/compare
					if(!funcbit_test(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER))
					{
						pds_memcpy(this->last_line_texts[0], subt_data->subtitle_line_data + check_point, check_len);
					}
				}
			}
		}

		// initialize / resize / re-scale font
		if(this->renderbase_subtitle_font_init(video_surface_infos, control_flags))
		{  // in this case the sub-picture is always re-calculated and rendered again
			this->subtitle_pic_width = 0;
			this->subtitle_pic_heigth = 0;
			this->subtitle_pic_renderpos_y = 0;
			funcbit_disable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER);
		}

		if(!funcbit_test(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER)) // FIXME: conflict with CTRLFLAG_FULLSCREEN, but currently it's not a use-case
		{
			if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_TEXT)
			{
				// use FontMetrics to calculate sub-picture metrics
				QFontMetrics fm(this->font_subtitle);

				// calculate minimum required frame size for current subtitle text(s)
				for(int i = 0; i < subtitle_infos->subtitle_nb_lines; i++)
				{
					int textpixellen = fm.width(QString::fromUtf8(subtitle_infos->subtitle_datas[i].subtitle_line_data));
					if(textpixellen <= 0)
						continue;
					this->subtitle_line_pixellens[i] = textpixellen;
					textpixellen += DISPQT_SUBTITLE_BORDER_SIZE;
					new_size_x = max(new_size_x, textpixellen);
					new_size_y += this->subtitle_font_char_pixelsize * 32 / 20;
				}
			}
			else if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_BITMAP)
			{
				// calculate minimum required frame size for current subtitle bitmap(s)
				for(int i = 0; i < subtitle_infos->subtitle_nb_lines; i++)
				{
					struct ffmpegvideo_subtitle_data_s *subt_data = &subtitle_infos->subtitle_datas[i];
					if(subt_data->subtitle_line_data && subt_data->subtitle_line_info)
					{
						int bitmap_dest_w = ((subt_data->subtitle_bitmap_w * this->subtitle_bitmap_scale / DISPQT_SUBTITLE_SCALE_BASE) + 1) & ~1;
						int bitmap_dest_h = ((subt_data->subtitle_bitmap_h * this->subtitle_bitmap_scale / DISPQT_SUBTITLE_SCALE_BASE) + 1) & ~1;
						this->subtitle_line_pixellens[i] = bitmap_dest_w;
						bitmap_dest_w += DISPQT_SUBTITLE_BORDER_SIZE;
						new_size_x = max(new_size_x, bitmap_dest_w);
						new_size_y += bitmap_dest_h;
					}
				}
			}
			else
			{
				return retval;
			}
		}

		// update subpicture size
		if(new_size_x && new_size_y)
		{
			retval = DISPQT_RENDERBASE_SUBPAINT_RETVAL_NEWRENDER;
			if(funcbit_test(control_flags, (DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_FULLSCREEN))) // QWidget, GDI
			{
				if(this->subtitle_pic_width != win_size_x)
				{
					this->subtitle_pic_width = win_size_x;
					funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_RECTCHANGE);
				}
				if(this->subtitle_pic_heigth != win_size_y)
				{
					this->subtitle_pic_heigth = win_size_y;
					funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_RECTCHANGE);
				}
			}
			else // subpicture rendering (with the minimal required frame size) (D3D, Vulkan)
			{
				new_size_x = (new_size_x + 1) & ~1;
				new_size_y = (new_size_y + 2 * DISPQT_SUBTITLE_BORDER_SIZE) & ~1;
				if(this->subtitle_pic_width < new_size_x) // only expand size (never shrink)
				{
					this->subtitle_pic_width = new_size_x;
					funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_RECTCHANGE);
				}
				if(this->subtitle_pic_heigth < new_size_y) // only expand size
				{
					this->subtitle_pic_heigth = new_size_y;
					funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_RECTCHANGE);
				}
			}
			this->subtitle_pic_renderpos_y = this->subtitle_pic_heigth - new_size_y;
			if(funcbit_test(control_flags, (DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_FULLSCREEN)))
			{
				this->subtitle_pic_renderpos_y -= this->subtitle_font_char_pixelsize * 2;
			}
			else
			{
				this->subtitle_pic_renderpos_y /= 2; // to center one line vertically
				this->subtitle_pic_renderpos_y += DISPQT_SUBTITLE_BORDER_SIZE / 2;
				if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_TEXT)
				{
					this->subtitle_pic_renderpos_y += 11 * this->subtitle_font_char_pixelsize / 8; // FIXME: ??? (required manual shift)
				}
			}
			if(this->subtitle_pic_renderpos_y < 0)
				this->subtitle_pic_renderpos_y = 0;
		}
	}

	// calculate sub-picture screen (global) position
	if((retval > DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP) || (this->subtitle_pic_width && this->subtitle_pic_heigth))
	{
		new_pos_x = win_size_x / 2 - this->subtitle_pic_width / 2;
		if(new_pos_x < 0)
			new_pos_x = 0;
		else
			new_pos_x &= ~1;
		if(this->subtitle_pic_globalpos_x != new_pos_x)
		{
			this->subtitle_pic_globalpos_x = new_pos_x;
			funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_POSCHANGE);
		}
		new_pos_y = win_size_y - this->subtitle_pic_heigth;
		if(!funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO))
			new_pos_y -= this->subtitle_font_char_pixelsize * 2;
		if(new_pos_y < 0)
			new_pos_y = 0;
		else
			new_pos_y &= ~1;
		if(this->subtitle_pic_globalpos_y != new_pos_y)
		{
			this->subtitle_pic_globalpos_y = new_pos_y;
			funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_POSCHANGE);
		}
	}

	this->last_subtitle_nb_lines = subtitle_infos->subtitle_nb_lines;

	return retval;
}

// subtitle text rendering through QImage (into memory field by QPainter)
int MMCDispQtVideoRendererBase::renderbase_subtitle_qpaint(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int control_flags)
{
	bool do_clear_only = (!subtitle_infos || (funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO) && !video_surface_infos->ffmpeg_output_frame))? true : false;
	int retval = DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP;

	if(!subtitle_infos || (funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO) && !video_surface_infos->ffmpeg_output_frame))
	{
		if(!this->was_subtitle_rendered)
			return retval;
		// if no input data then we clear the previous subtitle frame, if there was any
		pds_memset(&this->last_line_texts[0][0], 0, sizeof(this->last_line_texts));
		funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_CLEAR);
	}
	else
	{
		retval = this->renderbase_subtitle_subpicture_calc(video_surface_infos, subtitle_infos, control_flags | DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_QIMAGERENDER);
		if(retval == DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP)
		{
			if(!this->was_subtitle_rendered)
				return retval;
			pds_memset(&this->last_line_texts[0][0], 0, sizeof(this->last_line_texts));
			funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_CLEAR);
		}
		else if(!funcbit_test(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_NEWRENDER) && !funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_DRAWEVERY))
		{
			return retval;
		}

		if(funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO))
		{
			this->subtitle_memory_render_image = QImage(video_surface_infos->ffmpeg_output_frame->data[0], video_surface_infos->ffmpeg_output_frame->width, video_surface_infos->ffmpeg_output_frame->height, video_surface_infos->ffmpeg_output_frame->linesize[0], DISPQT_VIDEO_QIMAGE_OUTPUT_QFORMAT);
		}
		else if((this->subtitle_memory_render_image.width() != this->subtitle_pic_width) || (this->subtitle_memory_render_image.height() != this->subtitle_pic_heigth))
		{
			this->subtitle_memory_render_image = QImage(this->subtitle_pic_width, this->subtitle_pic_heigth, DISPQT_VIDEO_SUBTITLE_QIMAGE_QFORMAT);
		}
	}
	if(!this->subtitle_memory_render_image.width() || !this->subtitle_memory_render_image.height())
	{
		return DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP;
	}
	if(!funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO))
	{
		this->subtitle_memory_render_image.fill(Qt::transparent);   // FIXME: clears the whole image (bellow methods don't work)
	}

	QPainter painter(&this->subtitle_memory_render_image);

	//painter.setCompositionMode(QPainter::CompositionMode_SourceOver);
	//painter.setBackground(QBrush(QColor(0,0,0,0)));
	//painter.eraseRect(video_surface_infos->sws_dst_pos_x, video_surface_infos->sws_dst_pos_y, video_surface_infos->sws_dst_width, video_surface_infos->sws_dst_height);
	// video surface background (transparent) (doesn't work on this way)
	//painter.fillRect(video_surface_infos->sws_dst_pos_x, video_surface_infos->sws_dst_pos_y, video_surface_infos->sws_dst_width, video_surface_infos->sws_dst_height, Qt::transparent);

#if MPXPLAY_DISPQT_ENABLE_RENDER_GDI
	if(funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_BLACKBORDERS))
	{
		// clear black window areas (out of video surface)    // TODO: optimize to not always do these
		painter.fillRect(0, 0, video_surface_infos->sws_dst_pos_x, video_surface_infos->window_size_y, Qt::black);        // left side
		int cx = video_surface_infos->sws_dst_pos_x + video_surface_infos->sws_dst_width;
		painter.fillRect(cx, 0, video_surface_infos->window_size_x - cx, video_surface_infos->window_size_y, Qt::black);  // right side
		painter.fillRect(0, 0, video_surface_infos->window_size_x, video_surface_infos->sws_dst_pos_y, Qt::black);        // top side
		int cy = video_surface_infos->sws_dst_pos_y + video_surface_infos->sws_dst_height;
		painter.fillRect(0, cy, video_surface_infos->window_size_x, video_surface_infos->window_size_y - cy, Qt::black);  // bottom side
	}
#endif

	retval |= this->renderbase_subtitle_qpaint(video_surface_infos, subtitle_infos, &painter, control_flags | DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_QIMAGERENDER);

	return retval;
}

// subtitle text rendering by QPainter
int MMCDispQtVideoRendererBase::renderbase_subtitle_qpaint(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos, QPainter *painter, unsigned int control_flags)
{
	int i, retval = DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP;

	if(!painter)
		return DISPQT_RENDERBASE_SUBPAINT_RETVAL_NOOP;

	if(funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_CLEARBORDERS) && this->was_subtitle_rendered)
	{
		// clear black subtitle areas
		int cx, cy;
		if(this->subtitle_pic_globalpos_x < video_surface_infos->sws_dst_pos_x)
			painter->fillRect(0, 0, video_surface_infos->sws_dst_pos_x, video_surface_infos->window_size_y, Qt::black); // left side
		cx = video_surface_infos->sws_dst_pos_x + video_surface_infos->sws_dst_width;
		if((video_surface_infos->window_size_x > cx) && ((this->subtitle_pic_globalpos_x + this->subtitle_pic_width) > cx))
			painter->fillRect(cx, 0, video_surface_infos->window_size_x - cx, video_surface_infos->window_size_y, Qt::black);  // right side
		if(this->subtitle_pic_globalpos_y < video_surface_infos->sws_dst_pos_y)
			painter->fillRect(0, 0, video_surface_infos->window_size_x, video_surface_infos->sws_dst_pos_y, Qt::black);   // top side
		cy = video_surface_infos->sws_dst_pos_y + video_surface_infos->sws_dst_height;
		if((video_surface_infos->window_size_y > cy) && ((this->subtitle_pic_globalpos_y + this->subtitle_pic_heigth) > cy))
			painter->fillRect(0, cy, video_surface_infos->window_size_x, video_surface_infos->window_size_y - cy, Qt::black);  // bottom side
		funcbit_enable(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_CLEAR);
	}
	this->was_subtitle_rendered = false;

	if(!subtitle_infos || !subtitle_infos->subtitle_nb_lines)
	{
		pds_memset(&this->last_line_texts[0][0], 0, sizeof(this->last_line_texts));
		return retval;
	}

	if(!funcbit_test(control_flags, DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_QIMAGERENDER))
	{
		retval |= this->renderbase_subtitle_subpicture_calc(video_surface_infos, subtitle_infos, control_flags);
		if(!funcbit_test(retval, DISPQT_RENDERBASE_SUBPAINT_RETVAL_NEWRENDER))
			return retval;
	}

	if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_TEXT)
	{
		painter->setFont(this->font_subtitle);
	}

	for(i = 0; i < subtitle_infos->subtitle_nb_lines; i++)
	{
		int x = ((this->subtitle_pic_width - this->subtitle_line_pixellens[i] + DISPQT_SUBTITLE_BORDER_SIZE) / 2) & ~1;
		int y = (this->subtitle_pic_renderpos_y + i * this->subtitle_font_char_pixelsize * 3 / 2) & ~1;
		if(funcbit_test(control_flags, (DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_FULLSCREEN|DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO)))
		{
			x += this->subtitle_pic_globalpos_x;
			y += this->subtitle_pic_globalpos_y;
		}
		if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_TEXT)
		{
			if(this->subtitle_line_flags != subtitle_infos->subtitle_datas[i].subtitle_line_info)
			{
				this->subtitle_line_flags = subtitle_infos->subtitle_datas[i].subtitle_line_info;
				this->font_subtitle.setItalic((this->subtitle_line_flags & INFFMPEG_ASS_DECODED_FLAG_ITALIC)? true : false);
				painter->setFont(this->font_subtitle);
			}
			painter->setPen(Qt::black);
			painter->drawText(x + DISPQT_SUBTITLE_SHADOW_SIZE, y + DISPQT_SUBTITLE_SHADOW_SIZE, QString::fromUtf8(subtitle_infos->subtitle_datas[i].subtitle_line_data)); // black shadow

			painter->setPen(Qt::white);
			painter->drawText(x, y, QString::fromUtf8(subtitle_infos->subtitle_datas[i].subtitle_line_data)); // white text

			mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "SUBTITLE drawText: l:%d w:%d cs:%d x:%2d y:%2d %s", i, video_surface_infos->window_size_x,
					this->subtitle_font_char_pixelsize, x, y, subtitle_infos->subtitle_datas[i].subtitle_line_data);
		}
		else if(subtitle_infos->subtitle_type == DISPQT_SUBTITLE_TYPE_BITMAP)
		{
			struct ffmpegvideo_subtitle_data_s *subt_data = &subtitle_infos->subtitle_datas[i];
			const int bitmap_dest_h = ((subt_data->subtitle_bitmap_h * this->subtitle_bitmap_scale / DISPQT_SUBTITLE_SCALE_BASE) + 1) & ~1;

			painter->setRenderHint(QPainter::SmoothPixmapTransform, true);

			QImage image((unsigned char *)subt_data->subtitle_line_data, subt_data->subtitle_bitmap_w, subt_data->subtitle_bitmap_h, subt_data->subtitle_line_info,
					(video_surface_infos->videoout_surface_pix_fmt == AV_PIX_FMT_RGB32)? QImage::Format_ARGB32 : QImage::Format_RGB32);   // FIXME: hack

			painter->drawImage(QRectF(x, y, this->subtitle_line_pixellens[i], bitmap_dest_h), image,
					QRectF(0, 0, subt_data->subtitle_bitmap_w, subt_data->subtitle_bitmap_h), Qt::NoFormatConversion);

			mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "SUBTITLE drawpic: wx:%d x:%d y:%d w:%d h:%d len:%d", video_surface_infos->window_size_x, x, y,
					this->subtitle_line_pixellens[i], bitmap_dest_h, subt_data->subtitle_line_info);
		}
	}

	this->was_subtitle_rendered = true;

	return retval;
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
