//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2022 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: Direct3D9 render output

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_ERROR stdout // stderr
#define DISPQT_DEBUG_WARNING stdout // stderr
#define DISPQT_DEBUGOUT_MIXER NULL // stdout
#define DISPQT_DEBUGOUT_LOAD NULL // stdout
#define DISPQT_DEBUGOUT_RENDER NULL // stdout
#define DISPQT_DEBUGOUT_SWAPCHAIN stdout
#define DISPQT_DEBUGOUT_TEXTURE stdout
#define DISPQT_DEBUGOUT_LUMINANCE NULL // stdout
#define DISPQT_DEBUGOUT_POOL NULL // stdout
#define DISPQT_DEBUGOUT_AUTOCROP NULL // stdout
#define DISPQT_DEBUGOUT_SUBTITLE NULL // stdout
#define DISPQT_DEBUGOUT_PIXSHADER NULL // stdout

#include "moc_config.h"

#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9

#include "moc_video_qt.h"
#include "moc_mainwindow.h"
#include "video_render_dx9.h"

static void dispqt_d3d9_shader_pixel_videoinfos_configure(struct dispqt_render_d3d9_data_s *evr_data);
static void dispqt_d3d9_videoframe_download_deinit(struct dispqt_render_d3d9_data_s *evr_data);

static struct dispqt_renderer_d3d9_static_info_s d3d9_static_datas;

mpxp_bool_t mpxplay_dispqt_videorender_d3d9_static_init(void)
{
	struct dispqt_renderer_d3d9_static_info_s *d3d9_datas = &d3d9_static_datas;
	struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;

	if(d3d9_datas->reopen_devicemanager)
	{
		d3d9_datas->reopen_devicemanager = FALSE;
		SAFE_RELEASE(d3d9_datas->d3d_device_manager);
	}

	if(!d3d9_datas->createDeviceManager)
	{
		if(!d3d9_datas->dxva_dll_hnd)
			d3d9_datas->dxva_dll_hnd = LoadLibrary("DXVA2.DLL");
		if(d3d9_datas->dxva_dll_hnd)
		{
			d3d9_datas->createDeviceManager = (pCreateDeviceManager9 *)GetProcAddress(d3d9_datas->dxva_dll_hnd, "DXVA2CreateDirect3DDeviceManager9");
			if(!d3d9_datas->createDeviceManager)
			{
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "GetProcAddress DXVA2CreateDirect3DDeviceManager9 FAILED");
			}
		}
		else
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "LoadLibrary DXVA2.DLL FAILED");
		}
	}

	if(!d3d9_datas->d3d_device_manager && d3d9_datas->createDeviceManager && d3d9_datas->d3d_device_ctx)
	{
		unsigned resetToken = 0;
		HRESULT hr;

		hr = d3d9_datas->createDeviceManager(&resetToken, &d3d9_datas->d3d_device_manager);
		if(SUCCEEDED(hr))
		{
			hr = d3d9_datas->d3d_device_manager->ResetDevice(d3d9_datas->d3d_device_ctx, resetToken);
			if(FAILED(hr))
			{
				SAFE_RELEASE(d3d9_datas->d3d_device_manager);
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "ResetDeviceManager FAILED hr:%8.8X", hr);
			}
		}
		else
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "createDeviceManager FAILED hr:%8.8X", hr);
		}
	}

	vri_datas->hwdevice_avpixfmt = AV_PIX_FMT_DXVA2_VLD;
	vri_datas->d3d_video_device = d3d9_datas->d3d_device_manager;

	return (d3d9_datas->d3d_device_manager)? TRUE : FALSE;
}

void mpxplay_dispqt_videorender_d3d9_static_close(void)
{
	struct dispqt_renderer_d3d9_static_info_s *d3d9_datas = &d3d9_static_datas;

	SAFE_RELEASE(d3d9_datas->d3d_device_manager);
	SAFE_RELEASE(d3d9_datas->d3d_device_ctx);
	SAFE_RELEASE(d3d9_datas->d3d_device_ptr);

	if(d3d9_datas->dxva_dll_hnd)
	{
		HMODULE d3d_dll = d3d9_datas->dxva_dll_hnd;
		d3d9_datas->dxva_dll_hnd = 0;
		FreeLibrary(d3d_dll);
	}
	if(d3d9_datas->d3d9_dll_hnd)
	{
		HMODULE d3d_dll = d3d9_datas->d3d9_dll_hnd;
		d3d9_datas->d3d9_dll_hnd = 0;
		FreeLibrary(d3d_dll);
	}
}

// ====================================================================================================================
// common functions to create texture or surface

static bool dispqt_d3d9_texture_create(struct dispqt_render_d3d9_data_s *evr_data, int width, int heigth,
		D3DFORMAT format, DWORD texture_usage, IDirect3DTexture9 **text_hand_ptr)
{
	D3DPOOL pool_type = (texture_usage & D3DUSAGE_DYNAMIC)? D3DPOOL_MANAGED : D3DPOOL_DEFAULT;
	HRESULT hr;

	funcbit_disable(texture_usage, D3DUSAGE_DYNAMIC);

    hr = evr_data->d3d_device_context->CreateTexture(width, heigth, 0, texture_usage, format, pool_type, text_hand_ptr, NULL);
    if (FAILED(hr)) {
    	mpxplay_debugf(DISPQT_DEBUG_ERROR, "dispqt_d3d9_texture_create FAILED tex:%8.8X u:%8.8X hr:%8.8X", (int)format, (int)texture_usage, hr);
        return false;
    }

    return true;
}

static bool dispqt_d3d9_surface_create(IDirect3DDevice9 *d3d_device_ctx,
		int width, int heigth, D3DFORMAT format, IDirect3DSurface9 **surface_hand_ptr)
{
	HRESULT hr;

	if(!d3d_device_ctx)
		return false;

    hr = d3d_device_ctx->CreateOffscreenPlainSurface(width, heigth, format, D3DPOOL_DEFAULT, surface_hand_ptr, NULL);
    if (FAILED(hr)) {
    	mpxplay_debugf(DISPQT_DEBUG_ERROR, "dispqt_d3d9_surface_create FAILED form:%8.8X hr:%8.8X", (int)format, hr);
        return false;
    }

    return true;
}

// ====================================================================================================================
// wrappers/callbacks for pool buffering (video_render_common.cpp)

static void *dispqt_d3d9_poolbuf_bufptr_get(void)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)mpxplay_video_render_infos.renderer_private_data;
	if(!evr_data)
		return NULL;
	return (void *)&evr_data->decoder_pool_datas[0];
}

static void *dispqt_d3d9_poolbuf_pixform_get(void *format_datas_buf, void *src_avframe)
{
	AVFrame *src_frame = (AVFrame *)src_avframe;
	void *form_datas = mpxplay_dispqt_render_common_input_select_native_color_format(&dispqt_render_d3d9_supported_pixel_formats[0], (dispqt_evr_format_desc_s *)format_datas_buf, src_frame->format, TRUE);
	if(!form_datas)
		form_datas = mpxplay_dispqt_render_common_input_select_native_color_format(&dispqt_render_d3d9_supported_pixel_formats[0], (dispqt_evr_format_desc_s *)format_datas_buf, AV_PIX_FMT_YUV420P, TRUE);
	return form_datas;
}

static mpxp_bool_t dispqt_d3d9_poolbuf_texture_alloc(void *format_datas, int tex_width, int tex_height, void **poolbufelem_surface_2d, void **poolbufelem_shader_resource_unused, void *mapped_resource)
{
	struct dispqt_evr_format_desc_s *form_datas = (struct dispqt_evr_format_desc_s *)format_datas;
	if(!format_datas)
		return FALSE;
	if(dispqt_d3d9_surface_create(d3d9_static_datas.d3d_device_ctx, tex_width, tex_height, (D3DFORMAT)form_datas->d3d_texture_format, (IDirect3DSurface9 **)poolbufelem_surface_2d))
	{
		struct dispqt_render_mapped_subresource_s *map_res = (struct dispqt_render_mapped_subresource_s *)mapped_resource;
		map_res->vidmem_beginptr = NULL;
		map_res->vidmem_linesize = 0;
		map_res->vidmem_linenum = tex_height;
		return TRUE;
	}
	return FALSE;
}

static void dispqt_d3d9_poolbuf_texture_free(void **poolbufelem_surface_2d, void **poolbufelem_shader_resource_unused)
{
	if(poolbufelem_surface_2d)
	{
		IDirect3DSurface9 *surface = *((IDirect3DSurface9 **)poolbufelem_surface_2d);
		SAFE_RELEASE(surface);
		*poolbufelem_surface_2d = NULL;
	}
}

static mpxp_bool_t dispqt_d3d9_poolbuf_texture_map(void *poolbufelem_surface_2d, void *mapped_resource)
{
	IDirect3DSurface9 *surface = (IDirect3DSurface9 *)poolbufelem_surface_2d;
	struct dispqt_render_mapped_subresource_s *map_res = (struct dispqt_render_mapped_subresource_s *)mapped_resource;
	D3DLOCKED_RECT d3drect;
	mpxp_bool_t success = FALSE;
	if(!surface || !map_res)
		return success;
	pds_memset(&d3drect, 0, sizeof(d3drect));
	if(SUCCEEDED(surface->LockRect(&d3drect, NULL, 0)))
	{
		map_res->vidmem_beginptr = d3drect.pBits;
		map_res->vidmem_linesize = d3drect.Pitch;
		success = TRUE;
	}
	return success;
}

static void dispqt_d3d9_poolbuf_texture_unmap(void *poolbufelem_surface_2d, void *mapped_resource)
{
	struct dispqt_render_mapped_subresource_s *map_res = (struct dispqt_render_mapped_subresource_s *)mapped_resource;
	IDirect3DSurface9 *surface = (IDirect3DSurface9 *)poolbufelem_surface_2d;
	if(map_res)
	{
		map_res->vidmem_beginptr = NULL;
		map_res->vidmem_linesize = 0;
	}
	if(surface)
		surface->UnlockRect();
}

static void dispqt_d3d9_poolbuf_callbacks_assign(void)
{
	struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;
	vri_datas->render_function_poolbuf_bufptr_get = dispqt_d3d9_poolbuf_bufptr_get;
	vri_datas->render_function_poolbuf_pixform_get = dispqt_d3d9_poolbuf_pixform_get;
	vri_datas->render_function_poolbuf_texture_alloc = dispqt_d3d9_poolbuf_texture_alloc;
	vri_datas->render_function_poolbuf_texture_free = dispqt_d3d9_poolbuf_texture_free;
	vri_datas->render_function_poolbuf_texture_map = dispqt_d3d9_poolbuf_texture_map;
	vri_datas->render_function_poolbuf_texture_unmap = dispqt_d3d9_poolbuf_texture_unmap;
}

// ====================================================================================================================

static int dispqt_d3d9_poolbufelem_check_and_select(struct dispqt_render_d3d9_data_s *evr_data,
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas, AVFrame *av_frame)
{
	struct dispqt_render_poolbufelem_s *bufelem;

	if(!mpxplay_video_render_infos.render_function_poolbufelem_validity_check((void *)av_frame))
		return 0;

	bufelem = (struct dispqt_render_poolbufelem_s *)av_frame->buf[0]->data;
	if(!bufelem->poolbufelem_texture_2d[0])
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "POOL buffer elem no texture");
		return -1;
	}
	dispqt_d3d9_poolbuf_texture_unmap(bufelem->poolbufelem_texture_2d[0], &bufelem->poolbufelem_texture_mappedResource[0]);
	if(bufelem->bufpoolelem_status != DISPQT_RENDER_BUFPOOLELEM_STATUS_DONE)
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "POOL buffer elem invalid status");
		return -1;
	}

	texture_datas->input_width = bufelem->allocated_width;
	texture_datas->input_height = bufelem->allocated_height;
	evr_data->d3d_hwdec_surface_ptr = (IDirect3DSurface9 *)bufelem->poolbufelem_texture_2d[0];

	mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL buffer elem selected %d", bufelem->elem_id);

	return 1;
}

// ====================================================================================================================
static LPD3DXBUFFER dispqt_render_d3d9_shader_compile(struct dispqt_render_d3d9_data_s *evr_data,
		char *shader_code_str, const char *shader_mainfunc_name, const char *shader_type_str)
{
	const char *shader_version_str = "2_0";
	LPD3DXBUFFER error_msgs = NULL;
	LPD3DXBUFFER compiled_shader = NULL;
	DWORD shader_flags = 0;
	char target_profile[64];
	HRESULT hr;

	if(!evr_data->d3d_api_funcs.d3d_shader_compiler_func)
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "d3d_shader_compiler_func is MISSING");
		return compiled_shader;
	}

	snprintf(target_profile, sizeof(target_profile), "%s_%s", shader_type_str, shader_version_str);

	hr = evr_data->d3d_api_funcs.d3d_shader_compiler_func(shader_code_str, pds_strlen(shader_code_str), NULL, NULL,
			shader_mainfunc_name, target_profile, shader_flags, &compiled_shader, &error_msgs, NULL);
	if (FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "d3d9_shader_compile FAILED tp:%s hr:%d", target_profile, (hr & 0xffff));
		if (error_msgs)
	    {
	    	mpxplay_debugf(DISPQT_DEBUG_ERROR, "HLSL Compilation Error: %s", (char*)error_msgs->GetBufferPointer());
	    }
		compiled_shader = NULL;
	}

	if(error_msgs)
		ID3DXBuffer_Release(error_msgs);

	return compiled_shader;
}

static void dispqt_d3d9_device_ctx_close(struct dispqt_render_d3d9_data_s *evr_data)
{
	SAFE_RELEASE(evr_data->d3d_screen_target_surface);
	SAFE_RELEASE(d3d9_static_datas.d3d_device_ctx);
	evr_data->d3d_device_context = NULL;
}

static bool dispqt_d3d9_device_ctx_init(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	D3DPRESENT_PARAMETERS *d3dpp = &evr_data->d3d_present_params;
	D3DDISPLAYMODE d3d_display_mode;
	DWORD init_flags;
	HRESULT hr;

	dispqt_d3d9_device_ctx_close(evr_data);

	hr = evr_data->d3d_device->GetAdapterDisplayMode(D3DADAPTER_DEFAULT, &d3d_display_mode);
	if (FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "GetAdapterDisplayMode FAILED");
		return false;
	}

	pds_memset(d3dpp, 0, sizeof(*d3dpp));

	if(evr_data->is_transparent_gui_used && SUCCEEDED(evr_data->d3d_device->CheckDeviceType(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, d3d_display_mode.Format, D3DFMT_A8R8G8B8, true)))
	{
		d3dpp->BackBufferFormat = D3DFMT_A8R8G8B8; // else the video surface is also transparent
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "Using desktop color format: D3DFMT_A8R8G8B8");
	}
	else
	{
		d3dpp->BackBufferFormat = d3d_display_mode.Format;
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "Using desktop color format: %8.8X", d3d_display_mode.Format);
	}

	d3dpp->Windowed               = TRUE;
	d3dpp->MultiSampleType        = D3DMULTISAMPLE_NONE;
	d3dpp->hDeviceWindow          = evr_data->parent_window_handler;
	d3dpp->BackBufferWidth        = video_surface_infos->fullscreen_size_x;
	d3dpp->BackBufferHeight       = video_surface_infos->fullscreen_size_y;
	d3dpp->PresentationInterval   = D3DPRESENT_INTERVAL_DEFAULT;
	d3dpp->SwapEffect             = funcbit_test(evr_data->d3d_device_caps.Caps3, D3DCAPS3_ALPHA_FULLSCREEN_FLIP_OR_DISCARD)? D3DSWAPEFFECT_DISCARD : D3DSWAPEFFECT_COPY;
	d3dpp->BackBufferCount        = DISPQT_D3D9_SWAPCHAN_BUFFERS - 1;

//  alternate settings for testing
//	d3dpp->Flags                = D3DPRESENTFLAG_VIDEO;
//	d3dpp->PresentationInterval = D3DPRESENT_INTERVAL_IMMEDIATE;
//	d3dpp->SwapEffect           = D3DSWAPEFFECT_COPY;
//	d3dpp->BackBufferWidth      = max((unsigned int)GetSystemMetrics(SM_CXVIRTUALSCREEN), video_surface_infos->fullscreen_size_x);
//	d3dpp->BackBufferHeight     = max((unsigned int)GetSystemMetrics(SM_CYVIRTUALSCREEN), video_surface_infos->fullscreen_size_y);
//	d3dpp->SwapEffect           = funcbit_test(evr_data->d3d_device_caps.Caps3, D3DCAPS3_ALPHA_FULLSCREEN_FLIP_OR_DISCARD)? D3DSWAPEFFECT_FLIP : D3DSWAPEFFECT_COPY;

	init_flags = (evr_data->d3d_device_caps.DevCaps & D3DDEVCAPS_HWTRANSFORMANDLIGHT)? D3DCREATE_HARDWARE_VERTEXPROCESSING : D3DCREATE_MIXED_VERTEXPROCESSING;
	if (evr_data->d3d_device_caps.DevCaps & D3DDEVCAPS_PUREDEVICE)
		funcbit_enable(init_flags, D3DCREATE_PUREDEVICE);

	DWORD required_vertex_hw_support[2] = { (init_flags | D3DCREATE_MULTITHREADED), init_flags};

	for (int v = 0; v < (sizeof(required_vertex_hw_support) / sizeof(DWORD)); v++)
	{
		hr = evr_data->d3d_device->CreateDevice(D3DADAPTER_DEFAULT,
				D3DDEVTYPE_HAL, evr_data->parent_window_handler,
				required_vertex_hw_support[v], d3dpp, &evr_data->d3d_device_context);
		if (SUCCEEDED(hr))
			break;
	}

	if (FAILED(hr) || !evr_data->d3d_device_context)
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "CreateDevice FAILED");
		return false;
	}

	hr = evr_data->d3d_device_context->GetRenderTarget(0, &evr_data->d3d_screen_target_surface);
	if (FAILED(hr))
	{
		SAFE_RELEASE(evr_data->d3d_device_context);
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "GetRenderTarget FAILED");
		return false;
	}

	d3d9_static_datas.d3d_device_ctx = evr_data->d3d_device_context;
	evr_data->update_vmixer_data = true;

	mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN INIT w:%d h:%d", d3dpp->BackBufferWidth, d3dpp->BackBufferHeight);

	return true;
}

static bool dispqt_render_d3d9_init(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_render_d3d9_api_s *d3d_api = &evr_data->d3d_api_funcs;
	HRESULT hr;

	if(!d3d9_static_datas.d3d_device_ptr)
	{
		PFN_DIRECT3D_CREATE9FUNC Direct3DCreate9Func;

		if(!d3d9_static_datas.d3d9_dll_hnd)
		{
			d3d9_static_datas.d3d9_dll_hnd = LoadLibrary("D3D9.DLL");
			if(!d3d9_static_datas.d3d9_dll_hnd)
			{
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "LoadLibrary D3D9.DLL FAILED");
				return false;
			}
		}

		Direct3DCreate9Func = (PFN_DIRECT3D_CREATE9FUNC)GetProcAddress(d3d9_static_datas.d3d9_dll_hnd, "Direct3DCreate9");
		if (!Direct3DCreate9Func)
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "GetProcAddress Direct3DCreate9 FAILED");
			return false;
		}

		d3d9_static_datas.d3d_device_ptr = Direct3DCreate9Func(D3D_SDK_VERSION);
		if(!d3d9_static_datas.d3d_device_ptr)
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "Direct3DCreate9Func FAILED");
			return false;
		}
	}
	evr_data->d3d_device = d3d9_static_datas.d3d_device_ptr;

	hr = evr_data->d3d_device->GetDeviceCaps(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, &evr_data->d3d_device_caps);
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "GetDeviceCaps FAILED hr:0x%0lx", hr);
	    return false;
	}

	if (!(evr_data->d3d_device_caps.DevCaps2 & D3DDEVCAPS2_CAN_STRETCHRECT_FROM_TEXTURES))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "Device does not support stretching from textures.");
		return false;
	}

	if(!dispqt_d3d9_device_ctx_init(evr_data, video_surface_infos))
		return false;

	// load shader compiler DLL
	for (int version = 43; version > 23; version--)
	{
		char filename[16];
		snprintf(filename, sizeof(filename), "D3dx9_%d.dll", version);
		d3d_api->d3dcompiler_dll_hnd = LoadLibrary(filename);
		if (d3d_api->d3dcompiler_dll_hnd)
			break;
	}
	if(d3d_api->d3dcompiler_dll_hnd)
	{
		// get shader compiler function
		d3d_api->d3d_shader_compiler_func = (PFN_D3D9_COMPILE_SHADER)GetProcAddress(d3d_api->d3dcompiler_dll_hnd, "D3DXCompileShader");
		if(!d3d_api->d3d_shader_compiler_func)
		{
			mpxplay_debugf(DISPQT_DEBUG_WARNING, "D3DXCompileShader not found in D3dx9_.dll");
		}
	}
	else
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "D3dx9_NN.dll shader DLL not found");
	}

	return true;
}

static void dispqt_d3d9_reopen_gui(void)
{
	d3d9_static_datas.reopen_devicemanager = TRUE;
	funcbit_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD);
	mpxplay_dispqt_main_close();
}

static bool dispqt_d3d9_swapchain_init(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	D3DPRESENT_PARAMETERS *d3dpp = &evr_data->d3d_present_params;

	if((d3dpp->BackBufferWidth == video_surface_infos->fullscreen_size_x) && (d3dpp->BackBufferHeight == video_surface_infos->fullscreen_size_y) && evr_data->d3d_screen_target_surface)
		return true; // output resolution has not changed

	mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN CHANGE w:%d h:%d", video_surface_infos->fullscreen_size_x, video_surface_infos->fullscreen_size_y);

	if(d3d9_static_datas.d3d_device_manager || funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_TEXBUFFORSWDECODED))
	{   // if hw decoding is initiated or pool buffer is used, we cannot reset the d3d9 device, we close and reopen the whole gui
		mpxplay_dispqt_mainthread_callback_init((void *)dispqt_d3d9_reopen_gui, NULL, 0);
		return false;
	}

	return dispqt_d3d9_device_ctx_init(evr_data, video_surface_infos);
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_d3d9_sampler_state_create(struct dispqt_render_d3d9_data_s *evr_data)
{
	evr_data->d3d_device_context->SetSamplerState(0, D3DSAMP_ADDRESSU, D3DTADDRESS_CLAMP);
	evr_data->d3d_device_context->SetSamplerState(0, D3DSAMP_ADDRESSV, D3DTADDRESS_CLAMP);

	if (evr_data->d3d_device_caps.TextureFilterCaps & D3DPTFILTERCAPS_MINFLINEAR)
 		evr_data->d3d_device_context->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR);
	else
 		evr_data->d3d_device_context->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_POINT);
	if (evr_data->d3d_device_caps.TextureFilterCaps & D3DPTFILTERCAPS_MAGFLINEAR)
 		evr_data->d3d_device_context->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);
	else
		evr_data->d3d_device_context->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_POINT);

	evr_data->d3d_device_context->SetRenderState(D3DRS_AMBIENT, D3DCOLOR_XRGB(255,255,255));
	evr_data->d3d_device_context->SetRenderState(D3DRS_CULLMODE, D3DCULL_NONE);
	evr_data->d3d_device_context->SetRenderState(D3DRS_ZENABLE, D3DZB_FALSE);
	evr_data->d3d_device_context->SetRenderState(D3DRS_LIGHTING, FALSE);
	evr_data->d3d_device_context->SetRenderState(D3DRS_DITHERENABLE, FALSE);
	evr_data->d3d_device_context->SetRenderState(D3DRS_STENCILENABLE, FALSE);
}

static void dispqt_d3d9_alpha_blend_create(struct dispqt_render_d3d9_data_s *evr_data)
{
	evr_data->d3d_device_context->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
	evr_data->d3d_device_context->SetRenderState(D3DRS_SRCBLEND, D3DBLEND_SRCALPHA);
	evr_data->d3d_device_context->SetRenderState(D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA);

	if (evr_data->d3d_device_caps.AlphaCmpCaps & D3DPCMPCAPS_GREATER) {
		evr_data->d3d_device_context->SetRenderState(D3DRS_ALPHATESTENABLE, TRUE);
		evr_data->d3d_device_context->SetRenderState(D3DRS_ALPHAREF, 0x00);
		evr_data->d3d_device_context->SetRenderState(D3DRS_ALPHAFUNC, D3DCMP_GREATER);
	}

	evr_data->d3d_device_context->SetTextureStageState(0, D3DTSS_COLOROP, D3DTOP_SELECTARG1);
	evr_data->d3d_device_context->SetTextureStageState(0, D3DTSS_COLORARG1, D3DTA_TEXTURE);
	evr_data->d3d_device_context->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_MODULATE);
	evr_data->d3d_device_context->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);
	evr_data->d3d_device_context->SetTextureStageState(0, D3DTSS_ALPHAARG2, D3DTA_DIFFUSE);
}

//------------------------------------------------------------------------------------------------------------

static bool dispqt_d3d9_native_format_support_check(struct dispqt_render_d3d9_data_s *evr_data)
{
	struct dispqt_evr_format_desc_s *format_decs = &dispqt_render_d3d9_supported_pixel_formats[0];
	bool supported_native_format_found = false;

	do {
#ifdef MPXPLAY_USE_DEBUGF
		const AVPixFmtDescriptor *desc = av_pix_fmt_desc_get((enum AVPixelFormat)format_decs->av_pixfmt);
#endif
		D3DFORMAT tex_format = (D3DFORMAT)format_decs->d3d_texture_format;
		HRESULT hr = evr_data->d3d_device->CheckDeviceFormatConversion(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, tex_format, evr_data->d3d_present_params.BackBufferFormat);
		if(SUCCEEDED(hr))
		{
			funcbit_enable(format_decs->is_native_support, DISPQT_EVR_SUPPORTFLAG_INPUT);
			supported_native_format_found = true;
			mpxplay_debugf(DISPQT_DEBUGOUT_TEXTURE, "CheckDeviceFormat OK texfm:%8.8X avfmt:%s", tex_format, ((desc)? desc->name : ""));
		}
		else // D3DERR_NOTAVAILABLE
		{
			mpxplay_debugf(DISPQT_DEBUG_WARNING, "CheckDeviceFormat NO hr:%d tex: %d -> %d avfmt:%s",
					(hr & 0xffff), tex_format, evr_data->d3d_present_params.BackBufferFormat, ((desc)? desc->name : ""));
		}
		format_decs++;
	}while(format_decs->av_pixfmt != AV_PIX_FMT_NONE);

	if(!supported_native_format_found)
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "supported_native_format_found NOT FOUND");
	}

	return supported_native_format_found;
}

static bool dispqt_d3d9_create_render_target(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_render_d3d9_input_texture_data_s *texture_datas)
{
	HRESULT hr;

	SAFE_RELEASE(evr_data->d3d_render_target_texture);
	SAFE_RELEASE(evr_data->d3d_render_target_surface);

	if(!dispqt_d3d9_texture_create(evr_data, texture_datas->texture_width, texture_datas->texture_height, evr_data->d3d_present_params.BackBufferFormat, D3DUSAGE_RENDERTARGET, &evr_data->d3d_render_target_texture))
		return false;

	hr = evr_data->d3d_render_target_texture->GetSurfaceLevel(0, &evr_data->d3d_render_target_surface);
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "GetSurfaceLevel FAILED hr:%X", hr);
		return false;
	}

	return true;
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_d3d9_clear_output_pic(struct dispqt_render_d3d9_data_s *evr_data)
{
	D3DCOLOR color = D3DCOLOR_XRGB(0, 0, 0);
	HRESULT hr;
	hr = evr_data->d3d_device_context->Clear(0, NULL, D3DCLEAR_TARGET, color, 1.0f, 0);
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "Clear FAILED hr:%X", hr);
	}
	hr = evr_data->d3d_device_context->ColorFill(evr_data->d3d_screen_target_surface, NULL, color);
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "ColorFill FAILED hr:%X", hr);
	}
}

//============================================================================================================

static void dispqt_d3d9_deinterlace_reconfig(struct dispqt_render_d3d9_data_s *evr_data, AVFrame *video_avframe,
		struct dispqt_video_surface_info_s *video_surface_infos, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_d3d9_deinterlace_data_s *deint_datas = &evr_data->deinterlace_datas;

	if(!mpxplay_dispqt_render_common_deinterlace_condition(video_avframe, gcfg))
	{
		if(deint_datas->deinterlace_switch_status && video_surface_infos->videoout_input_change)
		{
			deint_datas->deinterlace_switchoff_counter = 1;
		}
		if(deint_datas->deinterlace_switchoff_counter)
		{
			deint_datas->deinterlace_switchoff_counter--;
			if(!deint_datas->deinterlace_switchoff_counter)
			{
				deint_datas->deinterlace_switch_status = FALSE;
				dispqt_d3d9_shader_pixel_videoinfos_configure(evr_data);
			}
		}
	}
	else
	{
		if(!deint_datas->deinterlace_switch_status)
		{
			deint_datas->deinterlace_switch_status = TRUE;
			dispqt_d3d9_shader_pixel_videoinfos_configure(evr_data);
		}
		deint_datas->deinterlace_switchoff_counter = DISPQT_D3D9_DEINTERLACE_SWITCHOFF_COUNT;
	}
}

//============================================================================================================
// vertex shader

static void dispqt_d3d9_vertex_close(struct dispqt_render_d3d9_input_texture_data_s *texture_datas)
{
	SAFE_RELEASE(texture_datas->d3d_vertex_buffer);
#if DISPQT_D3D9_USE_SHADER_VS
	SAFE_RELEASE(texture_datas->d3d_vertex_shader_hnd);
#endif
}

static IDirect3DVertexBuffer9 *dispqt_d3d9_vertex_buffer_init(struct dispqt_render_d3d9_data_s *evr_data)
{
	IDirect3DVertexBuffer9 *d3dvtb = NULL;
	ID3DXBuffer *vertex_shader_blob;
	void *vertexbuf_destptr;
	HRESULT hr;

	hr = evr_data->d3d_device_context->CreateVertexBuffer(
		 sizeof(struct d3d_vertex_t) * DISPQT_EVR_D3D_NUMVERTICES_DEFAULT,
		(D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY),
		DISPQT_D3D9_FVF_FORMAT, D3DPOOL_DEFAULT, &d3dvtb, NULL);

	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"CreateVertexBuffer FAILED hr:0x%X", hr);
		goto err_out_build;
	}

	if (FAILED(d3dvtb->Lock(0, sizeof(struct d3d_vertex_t) * DISPQT_EVR_D3D_NUMVERTICES_DEFAULT, &vertexbuf_destptr, D3DLOCK_NOOVERWRITE)))
 	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "d3d_vertex_buffer lock error");
		goto err_out_build;
	}

	pds_memcpy(vertexbuf_destptr, (void *)&vertices_full_texture_default[0], sizeof(struct d3d_vertex_t) * DISPQT_EVR_D3D_NUMVERTICES_DEFAULT);

	if (FAILED(d3dvtb->Unlock())) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "d3d_vertex_buffer unlock error");
		goto err_out_build;
	}

err_out_build:
	return d3dvtb;
}

static bool dispqt_d3d9_vertex_build(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_render_d3d9_input_texture_data_s *texture_datas)
{
	bool retVal = false;

	texture_datas->d3d_vertex_buffer = dispqt_d3d9_vertex_buffer_init(evr_data);
	if(!texture_datas->d3d_vertex_buffer)
		goto err_out_build;

#if DISPQT_D3D9_USE_SHADER_VS
	vertex_shader_blob = dispqt_render_d3d9_shader_compile(evr_data, (char *)d3dVertexShaderFlat, "main", "vs");
	if(vertex_shader_blob)
	{
		HRESULT hr = evr_data->d3d_device_context->CreateVertexShader((const DWORD*)vertex_shader_blob->GetBufferPointer(), &texture_datas->d3d_vertex_shader_hnd);
		if (FAILED(hr))
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR,"CreateVertexShader FAILED hr:0x%X", hr);
		}
	}
#endif

	retVal = true;

err_out_build:
	return retVal;
}

//------------------------------------------------------------------------------------------------------------
void dispqt_d3d9_shader_pixel_assemble(struct dispqt_render_d3d9_input_texture_data_s *texture_datas, char pscode_str_body[8192])
{
	struct dispqt_evr_format_desc_s *pixform_infos = texture_datas->pixel_format_infos;
	struct dispqt_evr_color_transform_matrixes_s color_transform_data;
	char pscode_str_mainfunc[1024];
	HRESULT hr;

	pds_strcpy(pscode_str_body, (char *)dispqt_d3d9_pscode_COMMON_datas);
	if(texture_datas->texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
		pds_strcat(pscode_str_body, (char *)dispqt_d3d9_pscode_CONSTANTDATA0_color_mixer);

	pds_strcpy(pscode_str_mainfunc, (char *)dispqt_d3d9_pscode_mainfunc_BEGIN);

	if(texture_datas->texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
	{
		pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d9_pscode_GET_RGB_packed_deinterlace_blend);
		pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_color_mixer);
		pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_CALL_color_mixer);
	}
	else
	{
		pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d9_pscode_mainfunc_GET_RGB_packed);
	}

	pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_END);
	pds_strcat(pscode_str_body, pscode_str_mainfunc);

	mpxplay_debugf(DISPQT_DEBUGOUT_PIXSHADER, "PS: lenall:%d mainlen:%d \n%s", pds_strlen(pscode_str_body), pds_strlen(pscode_str_mainfunc), pscode_str_body);
}

static void dispqt_d3d9_shader_pixel_close(struct dispqt_render_d3d9_input_texture_data_s *texture_datas)
{
	SAFE_RELEASE(texture_datas->d3d_pixel_shader_hnd);
}

static bool dispqt_d3d9_shader_pixel_build(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_render_d3d9_input_texture_data_s *texture_datas)
{
	ID3DXBuffer *pixel_shader_blob = NULL;
	char pscode_str_body[8192];
	bool retval = false;
	HRESULT hr;

	dispqt_d3d9_shader_pixel_close(texture_datas);

	if(!evr_data->d3d_api_funcs.d3d_shader_compiler_func)
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "d3d_shader_compiler_func is MISSING");
		return retval;
	}

	dispqt_d3d9_shader_pixel_assemble(texture_datas, pscode_str_body);

	pixel_shader_blob = dispqt_render_d3d9_shader_compile(evr_data, &pscode_str_body[0], "main", "ps");
	if(!pixel_shader_blob)
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"dispqt_render_d3d9_shader_compile FAILED");
		goto err_out_build;
	}

	hr = evr_data->d3d_device_context->CreatePixelShader((const DWORD *)pixel_shader_blob->GetBufferPointer(), &texture_datas->d3d_pixel_shader_hnd);
	if (FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"CreatePixelShader FAILED hr:%8.8X", hr);
		goto err_out_build;
	}

	retval = true;

err_out_build:
	SAFE_RELEASE(pixel_shader_blob);
	return retval;
}

//------------------------------------------------------------------------------------------------------------
static void dispqt_d3d9_shader_pixel_videoinfos_configure(struct dispqt_render_d3d9_data_s *evr_data)
{
	struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	struct dispqt_d3d9_video_info_s video_infos;

	if(!texture_datas->d3d_pixel_shader_hnd)
		return;

	video_infos.EnableDeinterlace = (evr_data->deinterlace_datas.deinterlace_switch_status)? 1.0 : 0.0;
	video_infos.VideoHeight = (FLOAT)texture_datas->video_height;

	evr_data->d3d_device_context->SetPixelShaderConstantF(DISPQT_D3D9_PIXSHADERCONST_VIDEOINFOS, (FLOAT *)&video_infos, 1);

	mpxplay_debugf(DISPQT_DEBUGOUT_TEXTURE, "videoinfos_configure deint: %1.0f, height: %.0f", video_infos.EnableDeinterlace, video_infos.VideoHeight);
}

// update color changes for pixel shader
static void dispqt_d3d9_shader_pixel_colormixer_configure(struct dispqt_render_d3d9_data_s *evr_data)
{
	struct dispqt_evr_color_mixer_s color_mixer_data;

	if(!evr_data->update_vmixer_data)
		return;
	evr_data->update_vmixer_data = false;

	if(!evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO].d3d_pixel_shader_hnd)
		return;

	mpxplay_dispqt_render_common_shader_pixel_colormixer_configure(&color_mixer_data, evr_data->vmixer_values);

	evr_data->d3d_device_context->SetPixelShaderConstantF(DISPQT_D3D9_PIXSHADERCONST_COLORMIXER, (FLOAT *)&color_mixer_data.colormixer_values[0], 1);
	evr_data->d3d_device_context->SetPixelShaderConstantF(DISPQT_D3D9_PIXSHADERCONST_COLORMIXHUE, (FLOAT *)&color_mixer_data.colormixer_hue_matrix[0][0], 4);
}

//------------------------------------------------------------------------------------------------------------
static void dispqt_d3d9_shaders_close(struct dispqt_render_d3d9_data_s *evr_data, unsigned int texture_id)
{
	struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[texture_id];
	dispqt_d3d9_vertex_close(texture_datas);
	dispqt_d3d9_shader_pixel_close(texture_datas);
}

static bool dispqt_d3d9_shaders_apply(struct dispqt_render_d3d9_data_s *evr_data,
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas)
{
	HRESULT hr;

	if(texture_datas->texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
	{
#if DISPQT_D3D9_USE_SHADER_VS
		evr_data->d3d_device_context->SetVertexShader(texture_datas->d3d_vertex_shader_hnd);
#endif
		dispqt_d3d9_shader_pixel_colormixer_configure(evr_data);
	}
	evr_data->d3d_device_context->SetPixelShader(texture_datas->d3d_pixel_shader_hnd);

	// Render the vertex buffer contents
	hr = evr_data->d3d_device_context->SetStreamSource(0, texture_datas->d3d_vertex_buffer, 0, sizeof(d3d_vertex_t));
	if (FAILED(hr)){
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"SetStreamSource FAILED");
		return false;
	}

	// we use FVF instead of vertex shader
	hr = evr_data->d3d_device_context->SetFVF(DISPQT_D3D9_FVF_FORMAT);
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"SetFVF FAILED");
		return false;
	}

	hr = evr_data->d3d_device_context->DrawPrimitive(D3DPT_TRIANGLELIST, 0, DISPQT_EVR_D3D_NUMVERTICES_DEFAULT);
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"DrawPrimitive FAILED");
		return false;
	}

	return true;
}

//============================================================================================================

static bool dispqt_d3d9_input_texture_set_crop_and_rotate(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_render_d3d9_input_texture_data_s *texture_datas)
{
	void *vertexbuf_destptr;

	if(FAILED(texture_datas->d3d_vertex_buffer->Lock(0, 0, &vertexbuf_destptr, D3DLOCK_DISCARD)))
	{
 		mpxplay_debugf(DISPQT_DEBUG_ERROR, "d3d_vertex_buffer lock error");
		return false;
	}

	mpxplay_dispqt_render_common_vertex_set_input_texture_crop_and_rotate(vertexbuf_destptr, &texture_datas->texture_crop,
		texture_datas->input_width, texture_datas->input_height, DISPQT_RENDER_D3D9_VERTEX_POSITION_SCALE);

	if(FAILED(texture_datas->d3d_vertex_buffer->Unlock())) {
 		mpxplay_debugf(DISPQT_DEBUG_ERROR, "d3d_vertex_buffer unlock error");
		return false;
	}

	evr_data->clear_counter = DISPQT_D3D9_SWAPCHAN_BUFFERS;

	return true;
}

static void dispqt_d3d9_input_texture_set_viewport(struct dispqt_render_d3d9_data_s *evr_data,
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas, struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_evr_input_viewport_s *texture_viewport = &texture_datas->texture_viewport;
	D3DVIEWPORT9 viewport;

	viewport.X = texture_viewport->pos_x;
	viewport.Y = texture_viewport->pos_y;
	viewport.Width = texture_viewport->width;
	viewport.Height = texture_viewport->height;
	viewport.MinZ = 0.0f;
	viewport.MaxZ = 1.0f;

	if(!video_surface_infos->is_fullscreen && video_surface_infos->windowed_size_x && video_surface_infos->windowed_size_y)
	{   // scale windowed rectangles to the d3d9 backbuffer constant fullscreen resolution
		viewport.X = (viewport.X * evr_data->d3d_present_params.BackBufferWidth / video_surface_infos->windowed_size_x + 1) & ~1;
		viewport.Y = (viewport.Y * evr_data->d3d_present_params.BackBufferHeight / video_surface_infos->windowed_size_y + 1) & ~1;
		viewport.Width = (viewport.Width * evr_data->d3d_present_params.BackBufferWidth / video_surface_infos->windowed_size_x + 1) & ~1;
		viewport.Height = (viewport.Height * evr_data->d3d_present_params.BackBufferHeight / video_surface_infos->windowed_size_y + 1) & ~1;
	}

	evr_data->d3d_device_context->SetViewport(&viewport);
}

//------------------------------------------------------------------------------------------------------------

static bool dispqt_d3d9_input_surface_render(struct dispqt_render_d3d9_data_s *evr_data,
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas, struct dispqt_video_surface_info_s *video_surface_infos)
{
	HRESULT hr;

	if(!evr_data->d3d_render_target_texture)
		return false;

	hr = evr_data->d3d_device_context->SetTexture(0, evr_data->d3d_render_target_texture);
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"SetTexture FAILED");
		return false;
	}

	dispqt_d3d9_input_texture_set_viewport(evr_data, texture_datas, video_surface_infos);

	if(!dispqt_d3d9_shaders_apply(evr_data, texture_datas))
		return false;

	return true;
}

static bool dispqt_d3d9_subtitle_texture_render(struct dispqt_render_d3d9_data_s *evr_data,
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas, struct dispqt_video_surface_info_s *video_surface_infos)
{
	HRESULT hr;

	if(!texture_datas->d3d_input_texture_2d)
		return false;

	hr = evr_data->d3d_device_context->SetRenderState(D3DRS_ALPHABLENDENABLE, TRUE);
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"SetRenderState D3DRS_ALPHABLENDENABLE FAILED");
	}

	hr = evr_data->d3d_device_context->SetTexture(0, texture_datas->d3d_input_texture_2d);
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"SetTexture FAILED");
		return false;
	}

	dispqt_d3d9_input_texture_set_viewport(evr_data, texture_datas, video_surface_infos);

	if(!dispqt_d3d9_shaders_apply(evr_data, texture_datas))
		return false;

	evr_data->d3d_device_context->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);

	return true;
}

//============================================================================================================

// select initial input texture and subtitle format descriptors
static struct dispqt_evr_format_desc_s *dispqt_d3d9_input_select_default_video_frame_format(unsigned int texture_id)
{
	unsigned int select_format = (texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)? 6 : 3; // YV12 / RGBA
	return (struct dispqt_evr_format_desc_s *)&dispqt_render_d3d9_supported_pixel_formats[select_format];
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_d3d9_input_texture_deinit(struct dispqt_render_d3d9_data_s *evr_data, unsigned int texture_id)
{
	struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[texture_id];
	SAFE_RELEASE(texture_datas->d3d_input_surface_2d);
	SAFE_RELEASE(texture_datas->d3d_input_texture_2d);
	dispqt_d3d9_videoframe_download_deinit(evr_data);
	pds_memset(&texture_datas->texture_crop, 0, sizeof(texture_datas->texture_crop));
}

static bool dispqt_d3d9_input_texture_init(struct dispqt_render_d3d9_data_s *evr_data, unsigned int texture_id, int frame_width, int frame_height,
		struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *av_inframe, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[texture_id];
	struct dispqt_evr_format_desc_s *format_datas = texture_datas->pixel_format_infos;
	bool pixfmt_change = (format_datas)? false : true, resolution_change = false;

	if(!frame_width || !frame_height)
		return false;

	if(av_inframe && (texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO))
	{
		if(av_inframe->format != texture_datas->av_pixelformat)
			pixfmt_change = true;
	}

	if(pixfmt_change && (!av_inframe || (av_inframe->format != AV_PIX_FMT_DXVA2_VLD)))
	{
		if(av_inframe)
			format_datas = mpxplay_dispqt_render_common_input_select_native_color_format(&dispqt_render_d3d9_supported_pixel_formats[0], &texture_datas->custom_pixel_format_info, av_inframe->format, TRUE);
		if(!format_datas || (format_datas->d3d_texture_format == D3DFMT_UNKNOWN))
			format_datas = dispqt_d3d9_input_select_default_video_frame_format(texture_id);
		texture_datas->pixel_format_infos = format_datas;
	}

	if((texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO) && evr_data->d3d_hwdec_surface_ptr)
	{
		if((texture_datas->texture_width != texture_datas->input_width) || (texture_datas->texture_height != texture_datas->input_height)
		 || !texture_datas->d3d_input_surface_2d
		){
			texture_datas->texture_width = texture_datas->input_width;
			texture_datas->texture_height = texture_datas->input_height;
			resolution_change = true;
		}
	}
	else if((texture_datas->video_width != frame_width) || (texture_datas->video_height != frame_height)
	 || (!texture_datas->d3d_input_texture_2d && !texture_datas->d3d_input_surface_2d)
	){
		texture_datas->texture_width = frame_width & ~format_datas->align_mask_x;
		texture_datas->texture_height = frame_height & ~format_datas->align_mask_y;
		if(texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
			texture_datas->texture_width = (texture_datas->texture_width + 63) & ~63; // rounding up for SSE image copy
		resolution_change = true;
	}

	if(!resolution_change && !pixfmt_change)
	{
		if(texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
			video_surface_infos->videoout_surface_pix_fmt = (AVPixelFormat)texture_datas->av_pixelformat;
		return true;
	}

	if(av_inframe)
	{
		texture_datas->av_pixelformat = av_inframe->format;
		texture_datas->av_colortrc = av_inframe->color_trc;
		texture_datas->av_colorrange = av_inframe->color_range;
		texture_datas->av_colorspace = av_inframe->colorspace;
		texture_datas->av_colorprimaries = av_inframe->color_primaries;
	}

	if(resolution_change || pixfmt_change)
	{
		dispqt_d3d9_input_texture_deinit(evr_data, texture_id);

		texture_datas->video_width = frame_width;
		texture_datas->video_height = frame_height;

		if(texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
		{
			if(!dispqt_d3d9_create_render_target(evr_data, texture_datas))
				goto err_out_init;
			if(!dispqt_d3d9_surface_create(evr_data->d3d_device_context, texture_datas->texture_width, texture_datas->texture_height, (D3DFORMAT)texture_datas->pixel_format_infos->d3d_texture_format, &texture_datas->d3d_input_surface_2d))
				goto err_out_init;
			dispqt_d3d9_shader_pixel_videoinfos_configure(evr_data);
			evr_data->clear_counter = DISPQT_D3D9_SWAPCHAN_BUFFERS;
			video_surface_infos->videoout_surface_pix_fmt = (AVPixelFormat)texture_datas->av_pixelformat;
		}
		else
		{
			if(!dispqt_d3d9_texture_create(evr_data, texture_datas->texture_width, texture_datas->texture_height, (D3DFORMAT)texture_datas->pixel_format_infos->d3d_texture_format, D3DUSAGE_DYNAMIC, &texture_datas->d3d_input_texture_2d))
				goto err_out_init;
		}

		texture_datas->input_width = texture_datas->texture_width;
		texture_datas->input_height = texture_datas->texture_height;

		mpxplay_debugf(DISPQT_DEBUGOUT_TEXTURE, "input_texture_init OK w:%d h:%d p:%d", texture_datas->texture_width, texture_datas->texture_height, format_datas->av_pixfmt);
	}

	return true;

err_out_init:
	dispqt_d3d9_input_texture_deinit(evr_data, texture_id);
	mpxplay_debugf(DISPQT_DEBUG_ERROR, "input_texture_init FAILED");
	return false;
}

//============================================================================================================

static bool dispqt_d3d9_input_hwdec_texture_process(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_render_d3d9_input_texture_data_s *texture_datas,
		AVFrame *video_avframe, struct dispqt_video_surface_info_s *video_surface_infos, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_render_d3d9_hwdec_data_s *hwdec_datas = &evr_data->hwdec_datas;
	IDirect3DSurface9 *d3d_input_surface = (IDirect3DSurface9 *)video_avframe->data[3];
	D3DSURFACE_DESC desc;

	if(!d3d_input_surface)
		return false;

	pds_memset(&desc, 0, sizeof(desc));
	if(FAILED(d3d_input_surface->GetDesc(&desc)))
		return false;

	if( video_surface_infos->videoout_input_change
	 || (hwdec_datas->hwdec_tex_width != desc.Width) || (hwdec_datas->hwdec_tex_height != desc.Height) || (hwdec_datas->hwdec_tex_format != desc.Format)
	){
		hwdec_datas->hwdec_tex_width = desc.Width;
		hwdec_datas->hwdec_tex_height = desc.Height;
		hwdec_datas->hwdec_tex_format = desc.Format;
		evr_data->clear_counter = DISPQT_D3D9_SWAPCHAN_BUFFERS;
	}
	texture_datas->input_width = hwdec_datas->hwdec_tex_width;
	texture_datas->input_height = hwdec_datas->hwdec_tex_height;
	if(!texture_datas->pixel_format_infos || (texture_datas->pixel_format_infos->d3d_texture_format != hwdec_datas->hwdec_tex_format))
		texture_datas->pixel_format_infos = mpxplay_dispqt_render_common_select_dxgi_format(&dispqt_render_d3d9_supported_pixel_formats[0], hwdec_datas->hwdec_tex_format);

	evr_data->d3d_hwdec_surface_ptr = d3d_input_surface;

	return true;
}

//============================================================================================================
// download and analyze hw decoded or pool buffer video frame (for autocrop)

static void dispqt_d3d9_videoframe_download_deinit(struct dispqt_render_d3d9_data_s *evr_data)
{
	struct dispqt_render_d3d9_textdown_data_s *textdown_datas = &evr_data->textdown_datas;
	SAFE_RELEASE(textdown_datas->d3d_surface_video_download);
	textdown_datas->down_tex_width = textdown_datas->down_tex_height = 0;
	textdown_datas->down_text_format = D3DFMT_UNKNOWN;
}

static bool dispqt_d3d9_videoframe_download_init(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_video_source_info_s *video_source_infos,
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas, struct dispqt_render_d3d9_textdown_data_s *textdown_datas, D3DFORMAT format)
{
	dispqt_d3d9_videoframe_download_deinit(evr_data);

	if(!dispqt_d3d9_surface_create(evr_data->d3d_device_context, texture_datas->input_width, texture_datas->input_height, format, &textdown_datas->d3d_surface_video_download))
		return false;

	textdown_datas->down_tex_width = texture_datas->input_width;
	textdown_datas->down_tex_height = texture_datas->input_height;
	textdown_datas->down_text_format = format;

	return true;
}

static void dispqt_d3d9_videoframe_autocrop_calculate(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos,
		AVFrame *video_avframe, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	struct dispqt_video_source_info_s *video_source_infos = video_surface_infos->video_source_infos;
	struct dispqt_render_d3d9_textdown_data_s *textdown_datas = &evr_data->textdown_datas;
	mpxp_uint64_t curr_time_ms;

	if((gcfg->video_crop_type != DISPQT_VIDEOSCRCROPTYPE_AUTO) || (texture_datas->video_height > gcfg->video_autocrop_limit) || !evr_data->d3d_hwdec_surface_ptr)
		return;

	if(video_surface_infos->videoout_input_change || video_surface_infos->videoout_surface_reset)
		textdown_datas->last_process_time_ms = 0;

	curr_time_ms = pds_gettimem();
	if(curr_time_ms >= (textdown_datas->last_process_time_ms + DISPQT_EVR_AUTOCROP_MIN_PROCESS_INTERVAL_MS))
	{
		struct dispqt_evr_format_desc_s *pixform_datas = texture_datas->pixel_format_infos;
		const D3DFORMAT format = (D3DFORMAT)pixform_datas->d3d_texture_format;
		D3DLOCKED_RECT mapRes;
		HRESULT hr;

		if((textdown_datas->down_tex_width != texture_datas->input_width) || (textdown_datas->down_tex_height != texture_datas->input_height) || (textdown_datas->down_text_format != format))
			if(!dispqt_d3d9_videoframe_download_init(evr_data, video_source_infos, texture_datas, textdown_datas, format))
				return;

		hr = evr_data->d3d_device_context->StretchRect(evr_data->d3d_hwdec_surface_ptr, NULL, textdown_datas->d3d_surface_video_download, NULL, D3DTEXF_NONE);
		if(FAILED(hr))
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "d3d_surface_video_download StretchRect FAILED hr:%8.8X", hr);
			return;
		}

		pds_memset(&mapRes, 0, sizeof(mapRes));
		hr = textdown_datas->d3d_surface_video_download->LockRect(&mapRes, NULL, D3DLOCK_READONLY);
		if(FAILED(hr))
		{
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "d3d_surface_video_download LockRect FAILED hr:%8.8X", hr);
			return;
		}

		mpxplay_dispqt_videotrans_autocrop_process((mpxp_uint8_t *)mapRes.pBits, mapRes.Pitch, pixform_datas->av_pixfmt,
				texture_datas->video_width, texture_datas->video_height, mapRes.Pitch / textdown_datas->down_tex_width,
				video_source_infos, &video_surface_infos->videoout_surface_clearpic);

		video_source_infos->sws_autoxcrop_framecounter--; // PROCESS_INTERVAL timing correction

		textdown_datas->d3d_surface_video_download->UnlockRect();
		textdown_datas->last_process_time_ms = curr_time_ms;
		//mpxplay_debugf(DISPQT_DEBUGOUT_AUTOCROP, "AUTOCROP do");
	}

	mpxplay_dispqt_videotrans_videooutput_surface_window_calculate(video_source_infos, video_surface_infos, gcfg);
}

//============================================================================================================

static void dispqt_d3d9_videoframe_input_output_resize(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	struct dispqt_evr_input_viewport_s *video_input_viewport = &texture_datas->texture_crop;
	struct dispqt_evr_input_viewport_s *video_output_viewport = &texture_datas->texture_viewport;

	if( (video_input_viewport->pos_x != video_surface_infos->sws_crop_x) || (video_input_viewport->pos_y != video_surface_infos->sws_crop_y)
	 || (video_input_viewport->width != video_surface_infos->sws_crop_w) || (video_input_viewport->height != video_surface_infos->sws_crop_h)
	 || (video_input_viewport->text_width != texture_datas->input_width) || (video_input_viewport->text_height != texture_datas->input_height)
	 || (video_input_viewport->rotation_type != video_surface_infos->rotation_type)
	){
		video_input_viewport->pos_x = video_surface_infos->sws_crop_x;
		video_input_viewport->pos_y = video_surface_infos->sws_crop_y;
		video_input_viewport->width = video_surface_infos->sws_crop_w;
		video_input_viewport->height = video_surface_infos->sws_crop_h;
		video_input_viewport->text_width = texture_datas->input_width;
		video_input_viewport->text_height = texture_datas->input_height;
		video_input_viewport->rotation_type = video_surface_infos->rotation_type;

		dispqt_d3d9_input_texture_set_crop_and_rotate(evr_data, texture_datas);
	}

	video_output_viewport->pos_x = video_surface_infos->sws_dst_pos_x;
	video_output_viewport->pos_y = video_surface_infos->sws_dst_pos_y;
	video_output_viewport->width = video_surface_infos->sws_dst_width;
	video_output_viewport->height = video_surface_infos->sws_dst_height;
}

static mpxp_uint32_t dispqt_d3d9_get_frame_delay(void *evr_opaque_data)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)evr_opaque_data;
	mpxp_uint32_t render_delay = 0;

	if(!evr_data || !evr_data->d3d_device)
		return render_delay;

	render_delay = DISPQT_D3D9_SWAPCHAN_BUFFERS;

	return render_delay;
}

static void dispqt_d3d9_do_swapchain_present(struct dispqt_render_d3d9_data_s *evr_data)
{
	HRESULT hr;

	if(!evr_data || !evr_data->d3d_device_context)
		return;

	hr = evr_data->d3d_device_context->Present(NULL, NULL, NULL, NULL);
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "Present FAILED hr:%8.8X", hr);
	}
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_d3d9_deinit_all(struct dispqt_render_d3d9_data_s *evr_data, bool full_close)
{
	int mutex_error;

	if(!evr_data || !evr_data->d3d_device || !evr_data->d3d_device_context)
		return;

	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_close_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);

	mpxplay_dispqt_videorender_common_poolbuf_elems_close(&evr_data->decoder_pool_datas[0]);
	for(int i = 0; i < DISPQT_EVR_INPUTTEXUREID_NUM; i++)
	{
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[i];
		dispqt_d3d9_input_texture_deinit(evr_data, i);
		dispqt_d3d9_shaders_close(evr_data, i);
		pds_memset(texture_datas, 0, sizeof(*texture_datas));
	}
	SAFE_RELEASE(evr_data->d3d_render_target_surface);
	SAFE_RELEASE(evr_data->d3d_render_target_texture);
	SAFE_RELEASE(evr_data->d3d_screen_target_surface);
	if(full_close)
	{
		if(evr_data->d3d_api_funcs.d3dcompiler_dll_hnd)
			FreeLibrary(evr_data->d3d_api_funcs.d3dcompiler_dll_hnd);
		// other handlers are closed in static_close()
		pds_memset(evr_data, 0, sizeof(*evr_data));
	}

	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_close_mutex);
}

//------------------------------------------------------------------------------------------------------------

static bool dispqt_d3d9_check_devicestatus(struct dispqt_render_d3d9_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
    HRESULT hr = evr_data->d3d_device_context->TestCooperativeLevel();
    if(SUCCEEDED(hr))
    	return true;

	if(hr == D3DERR_DEVICENOTRESET)
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "dispqt_d3d9_check_devicestatus RESET");

		if(d3d9_static_datas.d3d_device_manager)
		{   // if hw decoding is initiated, we cannot reset the d3d9 device, we close and reopen the complete gui and playing
			mpxplay_dispqt_mainthread_callback_init((void *)dispqt_d3d9_reopen_gui, NULL, 0);
			return false;
		}

		dispqt_d3d9_deinit_all(evr_data, false);
		dispqt_d3d9_device_ctx_init(evr_data, video_surface_infos);
		dispqt_d3d9_sampler_state_create(evr_data);
		dispqt_d3d9_alpha_blend_create(evr_data);

		for(int i = 0; i < DISPQT_EVR_INPUTTEXUREID_NUM; i++)
		{
			struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[i];
			dispqt_d3d9_vertex_build(evr_data, texture_datas);
			dispqt_d3d9_shader_pixel_build(evr_data, texture_datas);
		}
	}
	else
	{
		dispqt_d3d9_deinit_all(evr_data, false);
		//mpxplay_debugf(DISPQT_DEBUG_ERROR, "dispqt_d3d9_check_devicestatus LOST device err:%d", (hr & 0xffff));
		return false;
	}
	return true;
}

//------------------------------------------------------------------------------------------------------------

bool MMCDispQtVideoRendererD3D9::videorenderer_d3d9_textures_load(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;
	struct dispqt_render_d3d9_input_texture_data_s *video_texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	AVFrame *video_avframe = video_surface_infos->ffmpeg_output_frame;
#ifdef MPXPLAY_USE_DEBUGF
	mpxp_int64_t sendframe_begin = pds_gettimem(), sendframe_lock = 0, sendframe_copy = 0, sendframe_unlock = 0, sendframe_sub;
#endif
	HRESULT hr;

	if(!evr_data->d3d_device_context || !video_avframe || !video_avframe->height)
		return false;

	evr_data->d3d_hwdec_surface_ptr = NULL; // it's set to a d3d9 surface, if received one (hwdec or pool)

	if(video_avframe->format == AV_PIX_FMT_DXVA2_VLD) // got a frame from the GPU
	{
		if(!dispqt_d3d9_input_hwdec_texture_process(evr_data, video_texture_datas, video_avframe, video_surface_infos, this->main_window->gui_config))
			return false; // got a wrong texture
	}
	else if(dispqt_d3d9_poolbufelem_check_and_select(evr_data, video_texture_datas, video_avframe) < 0)
	{	// got something invalid frame (poolbuf elem, but processed or closed already)
		return false;
	}

	if(!dispqt_d3d9_input_texture_init(evr_data, DISPQT_EVR_INPUTTEXUREID_VIDEO, video_avframe->width, video_avframe->height, video_surface_infos, video_avframe, this->main_window->gui_config))
		return false;

	if(!evr_data->d3d_hwdec_surface_ptr) // got a frame in normal memory, copy video frame into d3d texture memory
	{
		struct dispqt_render_mapped_subresource_s mapres;
		D3DLOCKED_RECT d3drect;
		mpxp_bool_t success_copy;
		if(!video_avframe->data[0])
			return false;
#ifdef MPXPLAY_USE_DEBUGF
		sendframe_lock = pds_gettimem();
#endif
		pds_memset(&d3drect, 0, sizeof(d3drect));
		if(FAILED(video_texture_datas->d3d_input_surface_2d->LockRect(&d3drect, NULL, 0)))
			return false;

#ifdef MPXPLAY_USE_DEBUGF
		sendframe_copy = pds_gettimem();
#endif

		mapres.vidmem_beginptr = d3drect.pBits;
		mapres.vidmem_linesize = d3drect.Pitch;
		mapres.vidmem_linenum = video_texture_datas->texture_height;
		success_copy = mpxplay_dispqt_videorender_common_videoframe_copy(&mapres,
				video_texture_datas->pixel_format_infos, video_avframe->data, video_avframe->linesize, video_avframe->height);

#ifdef MPXPLAY_USE_DEBUGF
		sendframe_unlock = pds_gettimem();
#endif
		video_texture_datas->d3d_input_surface_2d->UnlockRect();
		if(!success_copy)
			return false;
	} // else got a hw decoded frame or a frame from the pool

#ifdef MPXPLAY_USE_DEBUGF
	sendframe_sub = pds_gettimem();
#endif

	// calculate autocrop if it's stored in surface (at AV_PIX_FMT_DXVA)
	dispqt_d3d9_videoframe_autocrop_calculate(evr_data, video_surface_infos, video_avframe, this->main_window->gui_config);

	// apply crop and rotate on texture
	dispqt_d3d9_videoframe_input_output_resize(evr_data, video_surface_infos);

	// copy input video frame to render target texture
	IDirect3DSurface9 *src_surface = (evr_data->d3d_hwdec_surface_ptr)? evr_data->d3d_hwdec_surface_ptr : video_texture_datas->d3d_input_surface_2d;
	if(!src_surface || !evr_data->d3d_render_target_surface)
		return false;

	//hr = evr_data->d3d_device_context->StretchRect(src_surface, NULL, evr_data->d3d_render_target_surface, NULL, D3DTEXF_LINEAR);
	hr = evr_data->d3d_device_context->StretchRect(src_surface, NULL, evr_data->d3d_render_target_surface, NULL, D3DTEXF_NONE);
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"StretchRect FAILED hr:%8.8X", hr);
		return false;
	}

	// load subtitle texture (if there's any)
	this->videorenderer_d3d9_texture_subtitle_load(video_surface_infos, subtitle_infos);

	mpxplay_debugf(DISPQT_DEBUGOUT_LOAD, "load END dur:%2d lock:%2d cp:%2d unlock:%2d sub:%2d", (int)(pds_gettimem() - sendframe_begin),
			(int)(sendframe_copy - sendframe_lock), (int)(sendframe_unlock - sendframe_copy),
			(int)(sendframe_sub - sendframe_unlock), (int)(pds_gettimem() - sendframe_sub));

	return true;
}

void MMCDispQtVideoRendererD3D9::videorenderer_d3d9_texture_subtitle_load(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;

	// draw subtitle in memory (if there's any)
	evr_data->subtitle_rendermode = this->renderbase_subtitle_qpaint(video_surface_infos, subtitle_infos, 0);
	if(funcbit_test(evr_data->subtitle_rendermode, (DISPQT_RENDERBASE_SUBPAINT_RETVAL_CLEAR | DISPQT_RENDERBASE_SUBPAINT_RETVAL_NEWRENDER)))
		evr_data->clear_counter = DISPQT_D3D9_SWAPCHAN_BUFFERS; // alpha blending requires full clear at subtitle changes

	if(evr_data->subtitle_rendermode < DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER)
		return;
	if(!dispqt_d3d9_input_texture_init(evr_data, DISPQT_EVR_INPUTTEXUREID_SUBTITLE, this->get_subtitle_pic_width(), this->get_subtitle_pic_height(), NULL, NULL, NULL))
		return;

	struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_SUBTITLE];

	if(!funcbit_test(evr_data->subtitle_rendermode, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER))
	{
		struct dispqt_render_mapped_subresource_s mapres;
		D3DLOCKED_RECT d3drect;

		pds_memset(&d3drect, 0, sizeof(d3drect));
		if(FAILED(texture_datas->d3d_input_texture_2d->LockRect(0, &d3drect, NULL, 0)))
			return;

		mapres.vidmem_beginptr = d3drect.pBits;
		mapres.vidmem_linesize = d3drect.Pitch;
		mapres.vidmem_linenum = texture_datas->texture_height;
		mpxplay_dispqt_videorender_common_videoframecopy_PACKED(&mapres, texture_datas->texture_id,
				this->get_subtitle_image_memory_ptr(), this->get_subtitle_image_memory_linesize(), this->get_subtitle_pic_height());

		texture_datas->d3d_input_texture_2d->UnlockRect(0);
		mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "SUBTITLE RENDER x:%d y:%d w:%d h:%d ls:%d ptr:%8.8X", this->get_subtitle_pic_globalpos_x(), this->get_subtitle_pic_globalpos_y(),
				this->get_subtitle_pic_width(), this->get_subtitle_pic_height(), this->get_subtitle_image_memory_linesize(), (mpxp_ptrsize_t)this->get_subtitle_image_memory_ptr());
	}

	struct dispqt_evr_input_viewport_s *texture_viewport = &texture_datas->texture_viewport;
	texture_viewport->pos_x = this->get_subtitle_pic_globalpos_x();
	texture_viewport->pos_y = this->get_subtitle_pic_globalpos_y();
	texture_viewport->width = this->get_subtitle_pic_width();
	texture_viewport->height = this->get_subtitle_pic_height();
}

bool MMCDispQtVideoRendererD3D9::videorenderer_d3d9_textures_render(struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;
	struct dispqt_render_d3d9_input_texture_data_s *video_texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	AVFrame *video_avframe = video_surface_infos->ffmpeg_output_frame;
	bool texture_render_success = false;
	HRESULT hr;

	hr = evr_data->d3d_device_context->BeginScene();
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "BeginScene FAILED hr:0x%0lx", hr);
	    return texture_render_success;
	}

	// switch on/off de-interlace in ps
	dispqt_d3d9_deinterlace_reconfig(evr_data, video_avframe, video_surface_infos, this->main_window->gui_config);

	// render input video surface to output swapchain (apply shaders)
	texture_render_success = dispqt_d3d9_input_surface_render(evr_data, video_texture_datas, video_surface_infos);

	if(texture_render_success)
	{
		// render subtitle texture (if there's any)
		if(evr_data->subtitle_rendermode >= DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER)
		{
			dispqt_d3d9_subtitle_texture_render(evr_data, &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_SUBTITLE], video_surface_infos);
		}
	}

	hr = evr_data->d3d_device_context->EndScene();
	if (FAILED(hr)) {
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "EndScene FAILED hr:0x%0lx", hr);
		return false;
	}

	return texture_render_success;
}

//------------------------------------------------------------------------------------------------------------

MMCDispQtVideoRendererD3D9::MMCDispQtVideoRendererD3D9(MainWindow *mainwindow, QWidget *parent) : MMCDispQtVideoRendererBase(mainwindow, parent)
{
	struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)pds_calloc(1, sizeof(struct dispqt_render_d3d9_data_s));
	if(!evr_data)
		return;

	if(!vri_datas->d3d_close_mutex)
		pds_threads_mutex_new(&vri_datas->d3d_close_mutex);

	this->private_data = (void *)evr_data;
	evr_data->parent_window_handler = (HWND)parent->winId();
	evr_data->is_transparent_gui_used = mainwindow->mainwin_guibkg_is_transparent();
}

MMCDispQtVideoRendererD3D9::~MMCDispQtVideoRendererD3D9()
{
	struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&vri_datas->d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	vri_datas->d3d_render_handler = NULL;
	if(this->private_data)
	{
		struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;
		this->private_data = NULL;
		dispqt_d3d9_deinit_all(evr_data, true);
		pds_free(evr_data);
	}
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&vri_datas->d3d_device_mutex);
#endif
	if(vri_datas->d3d_close_mutex)
	{
		void *mutex = vri_datas->d3d_close_mutex;
		vri_datas->d3d_close_mutex = NULL;
		pds_threads_mutex_del(&mutex);
	}
}

bool MMCDispQtVideoRendererD3D9::videorenderer_init(struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;
	struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;
	bool is_ps_available = false;
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	int mutex_error;
#endif

	if(!evr_data)
		return false;

#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	mutex_error = PDS_THREADS_MUTEX_LOCK(&vri_datas->d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif

	if(!dispqt_render_d3d9_init(evr_data, video_surface_infos))
		goto err_out_init;
	if(!dispqt_d3d9_native_format_support_check(evr_data))
		goto err_out_init;
	dispqt_d3d9_sampler_state_create(evr_data);
	dispqt_d3d9_alpha_blend_create(evr_data);

	for(int i = 0; i < DISPQT_EVR_INPUTTEXUREID_NUM; i++)
	{
		struct dispqt_render_d3d9_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[i];
		texture_datas->texture_id = i;
		if(i == DISPQT_EVR_INPUTTEXUREID_SUBTITLE)
			texture_datas->pixel_format_infos = mpxplay_dispqt_render_common_select_best_subtitle_format(&dispqt_render_d3d9_supported_pixel_formats[0]);
		if(!texture_datas->pixel_format_infos)
			texture_datas->pixel_format_infos = dispqt_d3d9_input_select_default_video_frame_format(i);
		if(!dispqt_d3d9_vertex_build(evr_data, texture_datas))
			goto err_out_init;
		is_ps_available |= dispqt_d3d9_shader_pixel_build(evr_data, texture_datas); // we have to set a minimal ps to subtitle too for proper alpha blending
	}

	vri_datas->d3d_render_handler = (void *)evr_data;

	vri_datas->hwdevice_avpixfmt = AV_PIX_FMT_DXVA2_VLD;
	vri_datas->render_function_get_frame_delay = dispqt_d3d9_get_frame_delay;
	dispqt_d3d9_poolbuf_callbacks_assign();
	mpxplay_dispqt_videorender_common_poolbuf_functions_assign();
	vri_datas->renderer_private_data = (void *)evr_data;
	d3d9_static_datas.reopen_devicemanager = TRUE;

	vri_datas->renderer_capabilities = DISPQT_VIDEORENDERER_CAPFLAG_DIRECTRENDER | DISPQT_VIDEORENDERER_CAPFLAG_ROTATE;
	if(is_ps_available)
		funcbit_enable(vri_datas->renderer_capabilities, (DISPQT_VIDEORENDERER_CAPFLAG_DEINTERLACE | DISPQT_VIDEORENDERER_CAPFLAG_COLORMIXER));

#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&vri_datas->d3d_device_mutex);
#endif

	return true;

err_out_init:
	dispqt_d3d9_deinit_all(evr_data, true);
	mpxplay_dispqt_videorender_d3d9_static_close();
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&vri_datas->d3d_device_mutex);
#endif
	return false;
}

void MMCDispQtVideoRendererD3D9::videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;
	struct dispqt_evr_format_desc_s *selected_new_format = NULL;
	struct dispqt_evr_format_desc_s format_info_tmp;

	if(decoded_frame)
	{
		pds_memset(&format_info_tmp, 0, sizeof(format_info_tmp));
		selected_new_format = mpxplay_dispqt_render_common_input_select_native_color_format(&dispqt_render_d3d9_supported_pixel_formats[0], &format_info_tmp, decoded_frame->format, TRUE);
		if(!selected_new_format || (selected_new_format->d3d_texture_format == D3DFMT_UNKNOWN))
			selected_new_format = dispqt_d3d9_input_select_default_video_frame_format(DISPQT_EVR_INPUTTEXUREID_VIDEO);
		if(selected_new_format)
			video_surface_infos->videoout_surface_pix_fmt = (AVPixelFormat)selected_new_format->av_pixfmt;
	}

	if(!evr_data || !evr_data->d3d_device)
		return;

	if(!selected_new_format)
		video_surface_infos->videoout_surface_pix_fmt = (AVPixelFormat)evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO].av_pixelformat;
}

void MMCDispQtVideoRendererD3D9::videorenderer_set_mixer(unsigned int u_id, int value)
{
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;
	if(!evr_data || (u_id >= DISPQT_VIDEO_VMIXERTYPE_MAX))
		return;
	if(evr_data->vmixer_values[u_id] != value)
	{
		evr_data->vmixer_values[u_id] = value;
		evr_data->update_vmixer_data = true;
	}
	mpxplay_debugf(DISPQT_DEBUGOUT_MIXER,"videorenderer_set_mixer m:%d v:%d", u_id, value);
}

void MMCDispQtVideoRendererD3D9::videorenderer_set_fullscreen(mpxp_bool_t full)
{
	pds_threads_sleep(0);
}

void MMCDispQtVideoRendererD3D9::videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
#ifdef MPXPLAY_USE_DEBUGF
	mpxp_uint64_t render_begintime_ms = pds_gettimem(), mutex_time_ms, textures_render_ms, present_time_ms;
#endif
	struct dispqt_render_d3d9_data_s *evr_data = (struct dispqt_render_d3d9_data_s *)this->private_data;
	bool clear_pic = video_surface_infos->videoout_surface_clearpic, texture_load_ok;
	AVFrame *video_avframe = video_surface_infos->ffmpeg_output_frame;
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	int mutex_error;
#endif

//	if(video_avframe)
//	{
//		mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "RENDER START sx:%d sy:%d sw:%d sh:%d dx:%3d dy:%3d dw:%3d dh:%3d pix:%d", video_surface_infos->sws_crop_x, video_surface_infos->sws_crop_y,
//			video_surface_infos->sws_crop_w, video_surface_infos->sws_crop_h, video_surface_infos->sws_dst_pos_x, video_surface_infos->sws_dst_pos_y,
//			video_surface_infos->sws_dst_width, video_surface_infos->sws_dst_height, ((video_avframe)? video_avframe->format : -1));
//	}

	if(!evr_data || !evr_data->d3d_device)
		return;

	if(evr_data->clear_counter)
		clear_pic = true;
	if(!clear_pic && !video_avframe) // nothing to present
		return;

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif

	if(!dispqt_d3d9_check_devicestatus(evr_data, video_surface_infos))
		goto err_out_render;

	if(!dispqt_d3d9_swapchain_init(evr_data, video_surface_infos))
		goto err_out_render;

#ifdef MPXPLAY_USE_DEBUGF
	mutex_time_ms = pds_gettimem();
#endif

	texture_load_ok = this->videorenderer_d3d9_textures_load(video_surface_infos, subtitle_infos);

	if(video_surface_infos->videoout_surface_clearpic)
		evr_data->clear_counter = DISPQT_D3D9_SWAPCHAN_BUFFERS;

#ifdef MPXPLAY_USE_DEBUGF
	textures_render_ms = pds_gettimem();
#endif

	if(evr_data->clear_counter)
	{
		evr_data->clear_counter--;
		dispqt_d3d9_clear_output_pic(evr_data);
		clear_pic = true;
	}

	if(texture_load_ok)
		texture_load_ok = this->videorenderer_d3d9_textures_render(video_surface_infos);

#ifdef MPXPLAY_USE_DEBUGF
	present_time_ms = pds_gettimem();
#endif

	if(texture_load_ok || clear_pic)
		dispqt_d3d9_do_swapchain_present(evr_data);

#ifdef MPXPLAY_USE_DEBUGF
	if(video_avframe)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "RENDER END duration:%3d mutex:%3d load:%2d render:%2d present:%d",
			(int)(pds_gettimem() - render_begintime_ms), (int)(mutex_time_ms - render_begintime_ms),
			(int)(textures_render_ms - mutex_time_ms), (int)(present_time_ms - textures_render_ms),
			(int)(pds_gettimem() - present_time_ms));
	}
#endif

err_out_render:
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
	return;
}

#endif // MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
