//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFmpeg video decoding

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_ERROR stderr
#define DISPQT_DEBUGOUT_WARNING stdout
#define DISPQT_DEBUGOUT_VTHREAD NULL // stdout
#define DISPQT_DEBUGOUT_CLEAR stdout
#define DISPQT_DEBUGOUT_VDELAY NULL // stdout
#define DISPQT_DEBUGOUT_DEINT NULL // stdout
#define DISPQT_DEBUGOUT_FRAMEDROP NULL // stdout
#define DISPQT_DEBUGOUT_D3D11 NULL // stdout
#define DISPQT_DEBUGOUT_WV_MERGE NULL // stdout
#define DISPQT_DEBUGOUT_AR NULL // stdout
#define DISPQT_DEBUGOUT_FRAMEDEC NULL // stdout
#define DISPQT_DEBUGOUT_SLEEP NULL // stdout

#include "moc_video_qt.h"

#define INFFMPG_MUTEX_PACKET_TIMEOUT   100
#define INFFMPG_MUTEX_LOWPRIO_TIMEOUT  1000
#define INFFMPG_MUTEX_COMMAND_TIMEOUT  5200

#define INFFMPG_NB_LOST_FRAMES_LIMIT  500 // to switch of KeyframesOnly at videowall

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

static void dispqt_videoworkdec_videodelay_skipkeyframes_process(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt);
static void videoworkdec_videowall_update_window_attribs(struct ffmpegvideo_callback_list_s *cbelem);

struct dispqt_video_filter_format_s mpxplay_videoworkdec_seekpreview_format;
struct dispqt_video_filter_format_s mpxplay_videoworkdec_videowidget_format;

static inline void dispqt_videoworkdec_pktlistelem_clear(struct ffmpegvideo_decoder_s *fvdt)
{
	mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->video_pktlist_elem);
}

static inline mpxp_bool_t dispqt_videoworkdec_is_video_ctx_valid(AVCodecContext *video_ctx)
{
	return mpxplay_ffmpstrm_is_avctx_valid(video_ctx, MPXPLAY_STREAMTYPEINDEX_VIDEO);
}

static inline mpxp_bool_t dispqt_videoworkdec_is_video_ctx_hwaccel(AVCodecContext *video_ctx)
{
	return (video_ctx && video_ctx->hw_device_ctx);
}

static void dispqt_videoworkdec_video_ctx_flush(void *video_ctx, int streamtype_index)
{
	AVCodecContext *avctx = (AVCodecContext *)video_ctx;
	if(dispqt_videoworkdec_is_video_ctx_valid(avctx))
	{
#if 0
		AVFrame *dummy_frame = av_frame_alloc();
		if(dummy_frame)
		{
			int retry = 20;
			do{
				if(avcodec_receive_frame(avctx, dummy_frame) == AVERROR_EOF)
					break;
				pds_threads_sleep(1);
			}while(--retry);
			av_frame_free(&dummy_frame);
		}
#endif
		avcodec_flush_buffers(avctx);
#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
		if(dispqt_videoworkdec_is_video_ctx_hwaccel(avctx))
			mpxplay_dispqt_videowidget_videorender_flush(TRUE);
#endif
	}
}

static void video_worker_videodecoder_context_process(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt, unsigned int *sleep_time, unsigned int *packet_flags)
{
	const unsigned int streamtype_index = fvdt->streamtype_index;
	mpxplay_packetlist_t *videoctx_pktelem = NULL;

	if((cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_CODECCTXINIT_FFMPEG, (mpxp_ptrsize_t)&videoctx_pktelem, streamtype_index, INFFMPG_MUTEX_PACKET_TIMEOUT + 1) == MPXPLAY_ERROR_OK) && videoctx_pktelem)
	{
		if( (fvdt->video_ctx != (AVCodecContext *)videoctx_pktelem->codec_ctx)
		 || (fvdt->videoctx_curr_counter != videoctx_pktelem->codecctx_counter)
		){
			fvdt->video_ctx = (AVCodecContext *)videoctx_pktelem->codec_ctx;
			fvdt->videoctx_curr_counter = videoctx_pktelem->codecctx_counter;
			if(fvdt->video_ctx)
			{
				fvdt->videoctx_sample_aspect_ratio = fvdt->video_ctx->sample_aspect_ratio;
				mpxplay_debugf(DISPQT_DEBUGOUT_AR, "AR CTX vn:%d vd:%d", fvdt->videoctx_sample_aspect_ratio.num, fvdt->videoctx_sample_aspect_ratio.den);
			}
		}
		//mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "video_decoder_worker_thread GET_CODECCTXINIT n:%d pd:%8.8X ctx:%8.8X", fvdt->streamtype_index, (mpxp_ptrsize_t)cbelem->infile_passdata, (mpxp_ptrsize_t)fvdt->video_ctx);
#if 1 // TODO: a better solution for faster seeking
		if(fvdt->resetvideo && !funcbit_test(videoctx_pktelem->flags, (MPXPLAY_PACKETLISTFLAG_STILLPICT | MPXPLAY_PACKETLISTFLAG_LIVESTREAM)) && dispqt_videoworkdec_is_video_ctx_hwaccel(fvdt->video_ctx))
		{
			if(!fvdt->video_pktlist_elem)
				cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_PACKETLIST_ELEM, (mpxp_ptrsize_t)&fvdt->video_pktlist_elem, streamtype_index, INFFMPG_MUTEX_PACKET_TIMEOUT + 2);
			funcbit_enable(fvdt->flags, (DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP | DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME));
			funcbit_copy(*packet_flags, videoctx_pktelem->flags, (MPXPLAY_PACKETLISTFLAG_STILLPICT | MPXPLAY_PACKETLISTFLAG_LIVESTREAM));
			funcbit_disable(videoctx_pktelem->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
		}
		else
#endif
		if(!fvdt->video_pktlist_elem || (videoctx_pktelem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF))
		{
			if(videoctx_pktelem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)
			{
				funcbit_disable(videoctx_pktelem->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
				fvdt->resetvideo = fvdt->resetfilter = fvdt->resetoutqueue = fvdt->resetcodec = true;
				mpxplay_dispqt_videoworkdec_videodelay_reset(&cbelem->videodelays_decoder, TRUE);
				mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "CLEARBUF CTX s:%d", fvdt->streamtype_index);
			}
			funcbit_copy(*packet_flags, videoctx_pktelem->flags, (MPXPLAY_PACKETLISTFLAG_STILLPICT | MPXPLAY_PACKETLISTFLAG_LIVESTREAM));
			if( (cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_SET_PACKETQUEUE_CLEAR, streamtype_index, MPXPLAY_PACKETLISTFLAG_CLEARBUF, INFFMPG_MUTEX_PACKET_TIMEOUT + 3) == 1)
			 || (fvdt->resetvideo && fvdt->video_pktlist_elem && (fvdt->video_pktlist_elem->codec_ctx != fvdt->video_ctx)) // TODO: check (wrong video_pktlist_elem->codec_ctx can crash the program)
			){
				dispqt_videoworkdec_pktlistelem_clear(fvdt);
			}
			dispqt_videoworkdec_videodelay_skipkeyframes_process(cbelem, fvdt);
			if(!fvdt->video_pktlist_elem)
				cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_PACKETLIST_ELEM, (mpxp_ptrsize_t)&fvdt->video_pktlist_elem, streamtype_index, INFFMPG_MUTEX_PACKET_TIMEOUT + 4);
			if(fvdt->video_pktlist_elem)
			{
				if(funcbit_test(fvdt->video_pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF))
				{
					dispqt_videoworkdec_video_ctx_flush(fvdt->video_pktlist_elem->codec_ctx, streamtype_index);
					fvdt->resetvideo = fvdt->resetfilter = fvdt->resetoutqueue = true;
					mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "CLEARBUF PACKET s:%d dts:%lld", fvdt->streamtype_index, (fvdt->video_pktlist_elem->frame_pkt)? ((AVPacket *)fvdt->video_pktlist_elem->frame_pkt)->dts : 0);
				}
				mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "PACKET ELEM s:%d dts:%lld", fvdt->streamtype_index, (fvdt->video_pktlist_elem->frame_pkt)? ((AVPacket *)fvdt->video_pktlist_elem->frame_pkt)->dts : 0);
			}
			if(fvdt->resetvideo && funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO))
				funcbit_enable(fvdt->flags, (DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP | DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME));
			if(*sleep_time > MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT)
				*sleep_time = MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT;
		}
		else if(fvdt->video_ctx)
		{
			if(*sleep_time > MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT)
				*sleep_time = MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT;
		}
	}
	else
	{
		fvdt->video_ctx = NULL;
	}
}

static void videoworkdec_update_metadata_from_vframe(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt, AVFrame *vid_frame_decoded, unsigned int packet_flags)
{
	if(funcbit_test(packet_flags, MPXPLAY_PACKETLISTFLAG_STILLPICT) && vid_frame_decoded && (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO))
	{
		cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_SET_VIDFRAME_METADATA, (mpxp_ptrsize_t)vid_frame_decoded->metadata, 0, INFFMPG_MUTEX_LOWPRIO_TIMEOUT + 1);
	}
}

static void videoworkdec_get_videoframe_timestamp(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt, AVFrame *vid_frame, unsigned int packet_flags)
{
	if(!funcbit_test(packet_flags, MPXPLAY_PACKETLISTFLAG_STILLPICT) && ((fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) || (fvdt->streamtype_index == cbelem->selected_stream_index)))
	{
		cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_CORRECTEDVTIMESTAMP, (mpxp_ptrsize_t)vid_frame, (mpxp_ptrsize_t)&vid_frame->pts, INFFMPG_MUTEX_COMMAND_TIMEOUT + 1);
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------------------

// calculate decoded fps
static void dispqt_videoworkdec_videodelay_calculate_fps(ffmpegvideo_callback_list_s *cbelem, mpxp_int64_t timestamp_vpacket_us)
{
	if(timestamp_vpacket_us < 0)
		return;
	if((timestamp_vpacket_us > cbelem->video_last_del_timestamp) && (timestamp_vpacket_us < (cbelem->video_last_del_timestamp + DISPQT_FFMPGVIDEOCALLBACK_VIDEOFRAME_INTERVAL_1FPS)))
	{
		cbelem->video_average_frame_interval = (cbelem->video_average_frame_interval * 7 + (timestamp_vpacket_us - cbelem->video_last_del_timestamp)) / 8;
		mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "CALC fps:%d a:%d d:%d", ((cbelem->video_average_frame_interval)? (1000000 / cbelem->video_average_frame_interval) : 0),
				cbelem->video_average_frame_interval, (int)(timestamp_vpacket_us - cbelem->video_last_del_timestamp));
	}
	cbelem->video_last_del_timestamp = timestamp_vpacket_us;
}

void mpxplay_dispqt_videoworkdec_videodelay_reset(struct ffmpegvideo_delay_s *video_delays, mpxp_bool_t restart)
{
	if(restart)
	{
		video_delays->video_init_framecounter = DISPQT_FFMPGVIDEOCALLBACK_INITFRAMECOUNTER_START;
	}
	else
	{
		video_delays->video_init_framecounter = DISPQT_FFMPGVIDEOCALLBACK_INITFRAMECOUNTER_DELAY;
	}
	video_delays->video_delay_type = DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_NONE;
	video_delays->video_delay_large_counter = 0;
	video_delays->video_delay_framedrop_value = 0;
	video_delays->video_delay_framedrop_counter = 0;
	video_delays->video_delay_framedrop_restore_retry = 0;
}

static void dispqt_videoworkdec_videodelay_skipkeyframes_init(ffmpegvideo_callback_list_s *cbelem)
{
	struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO];
	fvdt->resetflushcodec = true;
	mpxplay_debugf(DISPQT_DEBUGOUT_FRAMEDROP, "SKIPK a:%d vd:%d", cbelem->video_average_frame_interval, (int)(cbelem->video_delay_us / 1000));
	cbelem->video_average_frame_interval = 0;
	mpxplay_dispqt_videoworkdec_videodelay_reset(&cbelem->videodelays_decoder, TRUE);
	mpxplay_dispqt_videoworkdec_videodelay_reset(&cbelem->videodelays_scheduler, FALSE);
}

static void dispqt_videoworkdec_videodelay_skipkeyframes_process(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt)
{
	if(fvdt->resetflushcodec && (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) && funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO))
	{
		if(cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_SET_PACKETQUEUE_CLEAR, fvdt->streamtype_index, MPXPLAY_PACKETLISTFLAG_KEYFRAME, INFFMPG_MUTEX_PACKET_TIMEOUT + 5) == 1)
		{
			dispqt_videoworkdec_video_ctx_flush(fvdt->video_ctx, fvdt->streamtype_index);
			fvdt->resetvideo = fvdt->resetfilter = true;
			dispqt_videoworkdec_pktlistelem_clear(fvdt);
			mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "resetflushcodec KEYFRAME");
		}
		fvdt->resetflushcodec = false;
		mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "resetflushcodec");
	}
}

// drop output video frames at delay (got from videoworker_video_frame_scheduler / cbelem->video_delay_us)
mpxp_bool_t mpxplay_dispqt_videoworkdec_videodelay_process_output_drop(ffmpegvideo_callback_list_s *cbelem, struct mmc_dispqt_config_s *gui_config,
														struct ffmpegvideo_delay_s *video_delays,
														const mpxp_int32_t fps_limit_interval)
{
	const mpxp_bool_t called_from_decoder = (video_delays == &cbelem->videodelays_decoder)? TRUE : FALSE;
	mpxp_bool_t doFrameDrop = FALSE;
	mpxp_int64_t video_delay;

	if(gui_config->selected_videoplayer_control & MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESKIPFRAME)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_VDELAY, "SKIP1 d:%d a:%d l:%d vd:%d", called_from_decoder, cbelem->video_average_frame_interval, fps_limit_interval, (int)(cbelem->video_delay_us / 1000));
		return doFrameDrop;
	}

	if(video_delays->video_init_framecounter > 0)
	{
		video_delays->video_init_framecounter--;
		return doFrameDrop;
	}

	if(video_delays->video_delay_framedrop_counter > 0)
	{
		video_delays->video_delay_framedrop_counter--;
		doFrameDrop = TRUE;
		mpxplay_debugf(DISPQT_DEBUGOUT_VDELAY, "DROP d:%d r:%d fc:%d fv:%d int:%3d vd:%3d", (int)called_from_decoder, video_delays->video_delay_framedrop_restore_retry,
				video_delays->video_delay_framedrop_counter, video_delays->video_delay_framedrop_value, cbelem->video_average_frame_interval, (int)(cbelem->video_delay_us / 1000));
		return doFrameDrop;
	}

	video_delay = cbelem->video_delay_us;

	if(called_from_decoder)
	{
		if(video_delay > DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYLIMIT_SKIPKEYFRAME) // skip a keyframe
		{
			dispqt_videoworkdec_videodelay_skipkeyframes_init(cbelem);
			return doFrameDrop;
		}
	}

	if((cbelem->video_average_frame_interval <= 0) && (video_delays->video_delay_type == DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_NONE))
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_VDELAY, "SKIP3 d:%d a:%d l:%d vd:%d", called_from_decoder, cbelem->video_average_frame_interval, fps_limit_interval, (int)(cbelem->video_delay_us / 1000));
		return doFrameDrop;
	}

	if(video_delay >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYLIMIT_MIDDLE)
	{
		if(video_delays->video_delay_large_counter >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYCOUNT_MIDDLE) // to avoid peek sync errors
		{
			if((video_delay >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYLIMIT_SIGNIFICANT) && (video_delays->video_delay_large_counter >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYCOUNT_SIGNIFICANT))
			{
				video_delays->video_delay_type = DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_SIGNIFICANT;
				video_delays->video_delay_framedrop_value = fps_limit_interval / cbelem->video_average_frame_interval;
				mpxplay_debugf(DISPQT_DEBUGOUT_VDELAY, "SIG d:%d v:%d a:%d vd:%d", called_from_decoder, video_delays->video_delay_framedrop_value, cbelem->video_average_frame_interval, (int)(video_delay / 1000));
			}
			else if((video_delay >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYLIMIT_LARGE) && (video_delays->video_delay_large_counter >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYCOUNT_LARGE))
			{
				video_delays->video_delay_type = DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_LARGE;
				video_delays->video_delay_framedrop_value = DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYFRAMEDROP_LARGE;
				mpxplay_debugf(DISPQT_DEBUGOUT_VDELAY, "LRG d:%d v:%d a:%d vd:%d", called_from_decoder, video_delays->video_delay_framedrop_value, cbelem->video_average_frame_interval, (int)(video_delay / 1000));
			}
			else
			{
				video_delays->video_delay_type = DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_MIDDLE;
			}
		}
		else
		{
			video_delays->video_delay_type = DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_SMALL;
		}
		if(video_delays->video_delay_large_counter < DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYCOUNT_SIGNIFICANT)
		{
			video_delays->video_delay_large_counter++;
		}
	}
	else
	{
		video_delays->video_delay_type = (video_delay >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYLIMIT_SMALL)? DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_SMALL : DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_NONE;
		if(video_delays->video_delay_large_counter > 0)
		{
			video_delays->video_delay_large_counter--;
		}

		if(video_delay < DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYLIMIT_SMALL)
		{
			if(video_delays->video_delay_framedrop_value)
			{
				const int framedrop_retry = (cbelem->video_average_frame_interval >= DISPQT_FFMPGVIDEOCALLBACK_VIDEOFRAME_INTERVAL_33FPS)? DISPQT_WORKDEC_VIDEODELAY_FRAMEDROP_RESTORE_RETRY : (DISPQT_WORKDEC_VIDEODELAY_FRAMEDROP_RESTORE_RETRY / 2);
				if(video_delays->video_delay_framedrop_restore_retry < framedrop_retry)
				{
					video_delays->video_delay_framedrop_value = 0;
					video_delays->video_delay_framedrop_restore_retry++;
				}
				else // keep dropping every second frame after some retry
				{
					video_delays->video_delay_framedrop_value = 1;
				}
			}
		}
	}

	if(called_from_decoder)
	{   // we skip only keyframes on decoder side (and skip frames in FFmpeg decoder, if supported)
		video_delays->video_delay_framedrop_value = 0;
	}
	else if(video_delays->video_delay_framedrop_value)
	{
		video_delays->video_delay_framedrop_counter = video_delays->video_delay_framedrop_value;
		mpxplay_debugf(DISPQT_DEBUGOUT_VDELAY, "video_delay_framedrop_counter d:%d c:%d vd:%d",
				(int)called_from_decoder, video_delays->video_delay_framedrop_counter, (int)(video_delay / 1000));
	}

	return doFrameDrop;
}

//------------------------------------------------------------------------------------------------------------------------------------------------------

static int video_worker_videodecoder_send_frame_to_scheduler(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt,
		unsigned int packet_flags, AVFrame *src_frame, unsigned int queue_max_elems)
{
	struct mmc_dispqt_config_s *gcfg = mpxplay_dispqt_configini_get();

	if(cbelem)
	{

		if(!funcbit_test(gcfg->selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESKIPFRAME))
		{
			if(mpxplay_dispqt_videoworkdec_videodelay_process_output_drop(cbelem, gcfg, &cbelem->videodelays_decoder, DISPQT_FFMPGVIDEOCALLBACK_VIDEOFRAME_INTERVAL_33FPS))
			{
				mpxplay_debugf(DISPQT_DEBUGOUT_FRAMEDROP, "DROP q:%d/%d=%d qs:%d fps:%d vd:%d",
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
						fvdt->videoframe_output_queue.nb_packets,
#else
						0,
#endif
						fvdt->videoframe_temp_queue.nb_packets,
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
						(fvdt->videoframe_output_queue.nb_packets + fvdt->videoframe_temp_queue.nb_packets),
#else
						0,
#endif
						queue_max_elems, (cbelem->video_average_frame_interval > 0)? (1000000 / cbelem->video_average_frame_interval) : 0,
						(int)(cbelem->video_delay_us / 1000));
				av_frame_free(&src_frame);
				mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_temp_elem);
				return -1;
			}
		}
	}

#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
	if( funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_TEXBUFFORSWDECODED)
		&& (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) && (src_frame->format != mpxplay_video_render_infos.hwdevice_avpixfmt)
		&& mpxplay_video_render_infos.render_function_is_framebuffer_support()
	){
#ifdef MPXPLAY_USE_DEBUGF
		mpxp_int64_t sendframe_begin = pds_gettimem(), sendframe_getbuf, sendframe_conv, sendframe_copy, sendframe_unmap;
#endif
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif

		if(funcbit_test(packet_flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF))
		{
			mpxplay_ffmpgdec_packetqueue_clear(&fvdt->videoframe_temp_queue, 0);
			mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_temp_elem);
		}

		if((fvdt->videoframe_output_queue.nb_packets >= DISPQT_FFMPGVIDEO_MAX_QUEUED_TEXTBUFFRAME) || (fvdt->videoframe_temp_queue.nb_packets > 0))
		{
			mpxplay_ffmpgdec_packetqueue_put(&fvdt->videoframe_temp_queue, MPXPLAY_PACKETTYPE_AVFRAME_VIDEO, 0, NULL, 0, NULL, src_frame);
			src_frame = NULL;
			mpxplay_debugf(DISPQT_DEBUGOUT_D3D11, "temp_queue add %d/%d", fvdt->videoframe_temp_queue.nb_packets, fvdt->videoframe_output_queue.nb_packets);
		}

		do{
			struct mmc_dispqt_config_s *gcfg = mpxplay_dispqt_configini_get();
			AVFrame *dest_frame;
			int retval;
			AVCodecContext avctx;

			if(fvdt->videoframe_output_queue.nb_packets >= DISPQT_FFMPGVIDEO_MAX_QUEUED_TEXTBUFFRAME)
			{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
				if(mutex_error == MPXPLAY_ERROR_OK)
					PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
				return 0;
			}

			if(!fvdt->videoframe_temp_elem)
				mpxplay_ffmpgdec_packetqueue_get(0, &fvdt->videoframe_temp_queue, (mpxplay_packetlist_t **)&fvdt->videoframe_temp_elem, 0);
			if(fvdt->videoframe_temp_elem)
			{
				src_frame = (AVFrame *)fvdt->videoframe_temp_elem->frame_pkt;
				fvdt->videoframe_temp_elem->frame_pkt = NULL;
				mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_temp_elem);
			}
			if(!src_frame)
			{
				//mpxplay_debugf(DISPQT_DEBUGOUT_D3D11, "no src_frame %d/%d", fvdt->videoframe_temp_queue.nb_packets, fvdt->videoframe_output_queue.nb_packets);
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
				if(mutex_error == MPXPLAY_ERROR_OK)
					PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
				return 0;
			}

			dest_frame = av_frame_alloc();
			if(!dest_frame)
			{
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "no dest_frame %d/%d", fvdt->videoframe_temp_queue.nb_packets, fvdt->videoframe_output_queue.nb_packets);
				break;
			}

			pds_memset((void *)&avctx, 0, sizeof(avctx));
			avctx.pix_fmt = (AVPixelFormat)src_frame->format;
			avctx.sw_pix_fmt = (AVPixelFormat)src_frame->format;
			avctx.width = src_frame->width;
			avctx.height = src_frame->height;
			av_frame_copy_props(dest_frame, src_frame);
			dest_frame->width = src_frame->width;
			dest_frame->height = src_frame->height;
			dest_frame->format = src_frame->format;
#ifdef MPXPLAY_USE_DEBUGF
			sendframe_getbuf = pds_gettimem();
#endif
			retval = mpxplay_video_render_infos.render_function_get_frame_buffer((void *)&avctx, (void *)dest_frame, (void *)src_frame, MPXPLAY_VIDEO_RENDERER_FRAME_POOL_ELEMFLAG_WRITEONLY);
			if(retval == 0)
			{
				AVFrame *converted_frame = src_frame;
				mpxp_bool_t d3d_copy_ok;
#ifdef MPXPLAY_USE_DEBUGF
				sendframe_conv = pds_gettimem();
#endif
				if(dest_frame->format != src_frame->format) // if the renderer (d3d11) cannot handle natively the pixelformat of the frame
				{
					struct dispqt_video_filter_format_s output_format = {src_frame->width, src_frame->height, (AVPixelFormat)dest_frame->format};
					if(mpxplay_dispqt_videotrans_ffmpeg_swsctx_conv_frame(&fvdt->frame_conv_infos, src_frame, &output_format))
					{
						converted_frame = fvdt->frame_conv_infos.ffmpeg_sws_frame;
						av_frame_copy_props(converted_frame, src_frame);
					}
					else
					{
						mpxplay_debugf(DISPQT_DEBUG_ERROR, "mpxplay_dispqt_videotrans_ffmpeg_swsctx_conv_frame failed");
					}
				}
#ifdef MPXPLAY_USE_DEBUGF
				sendframe_copy = pds_gettimem();
#endif
				d3d_copy_ok = mpxplay_video_render_infos.render_function_copy_to_pool_buffer((void *)dest_frame, (void *)converted_frame);
#ifdef MPXPLAY_USE_DEBUGF
				sendframe_unmap = pds_gettimem();
#endif
				mpxplay_video_render_infos.render_function_poolbufelem_unmap((void *)dest_frame);
#ifdef MPXPLAY_USE_DEBUGF
				mpxplay_debugf(DISPQT_DEBUGOUT_D3D11, "SENDFRAME END ok:%d q:%d/%d dur:%3d getbuf:%3d conv:%2d copy:%2d unmap:%d ctx:%8.8X",
						d3d_copy_ok, fvdt->videoframe_temp_queue.nb_packets, fvdt->videoframe_output_queue.nb_packets,
						(int)(pds_gettimem() - sendframe_begin), (int)(sendframe_conv - sendframe_getbuf),
						(int)(sendframe_copy - sendframe_conv), (int)(sendframe_unmap - sendframe_copy),
						(int)(pds_gettimem() - sendframe_unmap), fvdt->video_ctx);
#endif
				retval = (d3d_copy_ok)? mpxplay_ffmpgdec_packetqueue_put(&fvdt->videoframe_output_queue, MPXPLAY_PACKETTYPE_AVFRAME_VIDEO, packet_flags, NULL, 0, NULL, dest_frame) : -1;
			}
			else
			{
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "SENDFRAME END render_function_get_frame_buffer failed");
			}
			if(retval < 0)
			{
				av_frame_free(&dest_frame);
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "SENDFRAME END failed");
				break;
			}
			av_frame_free(&src_frame);
		}while(TRUE);
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		if(mutex_error == MPXPLAY_ERROR_OK)
			PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
	}
	return mpxplay_ffmpgdec_packetqueue_put(&fvdt->videoframe_output_queue, MPXPLAY_PACKETTYPE_AVFRAME_VIDEO, packet_flags, NULL, 0, NULL, src_frame);
#else
	mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_pktlist_elem);
	fvdt->videoframe_pktlist_elem = mpxplay_ffmpgdec_queuelistelem_prepare(MPXPLAY_PACKETTYPE_AVFRAME_VIDEO, packet_flags, NULL, 0, NULL, src_frame);
	return 0;
#endif
}

//-----------------------------------------------------------------------------------------------------------------------------------------

static void video_worker_videodecoder_frameprocess_speedup_on_delay(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt, AVCodecContext *video_ctx)
{
	if( (fvdt->streamtype_index >= MPXPLAY_STREAMTYPEINDEX_VIDEOWALL) && (fvdt->streamtype_index != cbelem->selected_stream_index)
	 && funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY) && (fvdt->video_decoder_lost_frames < INFFMPG_NB_LOST_FRAMES_LIMIT)
	 && !funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_NO)
	){
		video_ctx->skip_loop_filter = AVDISCARD_ALL;
		video_ctx->skip_frame = AVDISCARD_NONKEY;
	}
	else if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
	{
		video_ctx->skip_loop_filter = AVDISCARD_ALL;
		video_ctx->skip_frame = AVDISCARD_DEFAULT;
	}
	else
	{
		video_ctx->skip_loop_filter = AVDISCARD_DEFAULT;
		video_ctx->skip_frame = AVDISCARD_DEFAULT;
		switch(cbelem->videodelays_decoder.video_delay_type)
		{
			case DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_SIGNIFICANT:
				video_ctx->skip_loop_filter = AVDISCARD_ALL;
				video_ctx->skip_frame = AVDISCARD_NONREF;
				break;
			case DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_LARGE:
				video_ctx->skip_frame = AVDISCARD_NONREF;//AVDISCARD_NONINTRA;
				break;
			case DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_MIDDLE:
				video_ctx->skip_frame = AVDISCARD_NONREF;//AVDISCARD_BIDIR;
				break;
			case DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_SMALL:
				break;
			default:
				video_ctx->skip_loop_filter = AVDISCARD_DEFAULT;
				break;
		}
		if(funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY) && (fvdt->video_decoder_lost_frames >= INFFMPG_NB_LOST_FRAMES_LIMIT))
		{
			funcbit_enable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_NO);
		}
	}
}

#define MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_NONE    0
#define MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_FILTER  1
#define MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_ALL     2

static unsigned int video_worker_videodecoder_check_skip_frame(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt,
		unsigned int packet_flags, const mpxp_bool_t videowall_is_enabled)
{
	AVFrame *decoded_frame = fvdt->vid_frame_decoded;
	unsigned int skipframe_level = MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_ALL;

	if(!decoded_frame)
	{
		return skipframe_level;
	}

	if( !funcbit_test(packet_flags, MPXPLAY_PACKETLISTFLAG_STILLPICT)
	 || (decoded_frame->width > fvdt->video_last_valid_frame_w)
	 || (decoded_frame->height > fvdt->video_last_valid_frame_h)
	){
		if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
		{
			if(!funcbit_test(mpxplay_dispqt_configini_get()->selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESKIPFRAME) || funcbit_test(mpxplay_video_render_infos.renderer_capabilities, DISPQT_VIDEORENDERER_CAPENABLE_DEINTERDUP))
			{
				dispqt_videoworkdec_videodelay_calculate_fps(cbelem, (mpxp_int64_t)decoded_frame->pts);
			}
		}

		if((fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) && videowall_is_enabled) // TODO: we should use the primary video stream for the selected_stream_index (currently it's a duplicated decoding)
		{
			skipframe_level = MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_FILTER;
		}
		else if((fvdt->streamtype_index != cbelem->selected_stream_index) && !fvdt->video_frame_filter_workerprocess)
		{
			skipframe_level = MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_FILTER;
		}
		else if(cbelem->videodelays_decoder.video_delay_type >= DISPQT_FFMPGVIDEOCALLBACK_VIDEODELAYTYPE_SIGNIFICANT)
		{
			fvdt->video_delay_counter_workdec_filter = ((fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) || (fvdt->streamtype_index == cbelem->selected_stream_index))? MPXPLAY_VIDEO_WORKDEC_FILTER_RESTORE_DELAY : (MPXPLAY_VIDEO_WORKDEC_FILTER_RESTORE_DELAY / 25);
			skipframe_level = MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_FILTER;
		}
		else if(fvdt->video_delay_counter_workdec_filter)
		{
			fvdt->video_delay_counter_workdec_filter--;
			skipframe_level = MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_FILTER;
		}
		else
		{
			skipframe_level = MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_NONE;
		}
		fvdt->video_last_valid_frame_w = decoded_frame->width;
		fvdt->video_last_valid_frame_h = decoded_frame->height;
	}

	if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
	{
		if(funcbit_test(packet_flags, MPXPLAY_PACKETLISTFLAG_STILLPICT))
		{
			funcbit_enable(fvdt->vid_frame_decoded->flags, DISPQT_AVFRAME_FLAG_STILLPIC);
		}
	}

	return skipframe_level;
}

//-----------------------------------------------------------------------------------------------------------------------------------------

// check that videowall is enabled and read primary (currently played) window-id (streamtype_index) from the demultiplexer
static mpxp_bool_t videoworkdec_videowall_check(struct ffmpegvideo_callback_list_s *cbelem, unsigned int *streamindex_max)
{
	struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO];
	mpxp_int32_t new_sel_stream_index = MPXPLAY_STREAMTYPEINDEX_VIDEO;
	mpxp_bool_t videwall_is_enabled = cbelem->is_videowall_enabled;

	*streamindex_max = MPXPLAY_STREAMTYPEINDEX_NUM;
	if(videwall_is_enabled)
	{
		cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLCURRSTI, (mpxp_ptrsize_t)&new_sel_stream_index, (mpxp_ptrsize_t)0, INFFMPG_MUTEX_PACKET_TIMEOUT + 6);
		if(new_sel_stream_index >= MPXPLAY_STREAMTYPEINDEX_VIDEOWALL)
		{
			videoworkdec_videowall_update_window_attribs(cbelem);
			*streamindex_max = MPXPLAY_STREAMTYPEINDEX_MAX;
		}
		else
		{
			videwall_is_enabled = FALSE;
		}
	}

	if(new_sel_stream_index != cbelem->selected_stream_index)
	{
		cbelem->selected_stream_index = new_sel_stream_index;
		fvdt->resetfilter = true;
	}

	fvdt->videowall_window_width = mpxplay_videoworkdec_videowidget_format.width;
	fvdt->videowall_window_height = mpxplay_videoworkdec_videowidget_format.height;

	return videwall_is_enabled;
}

// update videowall stream number and calculate number of sub-windows (2x2, 3x3, 4x4, 5x5, 6x6)
static void videoworkdec_videowall_update_window_attribs(struct ffmpegvideo_callback_list_s *cbelem)
{
	struct dispqt_video_filter_format_s output_format = mpxplay_videoworkdec_videowidget_format;

	if((cbelem->videowall_count_streams) < 1 && (cbelem->videowall_nb_streams < 1))
		return;

	if(output_format.width < 128)
		output_format.width = 128;
	if(output_format.height < 128)
		output_format.height = 128;

	cbelem->videowall_output_format = output_format;
	cbelem->videowall_output_artype = cbelem->video_wall_artype;

	if(cbelem->videowall_count_streams > cbelem->videowall_nb_streams) // number of visible windows (decoded video streams) is increased
		cbelem->videowall_nb_streams = cbelem->videowall_count_streams;
	//else if(cbelem->videowall_count_streams < cbelem->videowall_nb_streams)
	// TODO: timer/counter to reduce number of windows at the case of lost video stream

	cbelem->videowall_nb_windows_x = (int)sqrt((double)cbelem->videowall_nb_streams); // eg: 16 streams -> 4 x 4 wall
	if(cbelem->videowall_nb_windows_x <= 0)
		cbelem->videowall_nb_windows_x = 1;

	if((output_format.width * 9 / output_format.height / 15) > cbelem->videowall_nb_windows_x) // vertical narrow layout (N x 1)
	{
		cbelem->videowall_nb_windows_x = cbelem->videowall_nb_streams;
		cbelem->videowall_nb_windows_y = 1;
	}
	else if((output_format.height * 15 / output_format.width / 10) > cbelem->videowall_nb_windows_x) // horizontal narrow layout (1 x N)
	{
		cbelem->videowall_nb_windows_x = 1;
		cbelem->videowall_nb_windows_y = cbelem->videowall_nb_streams;
	}
	else
	{
		cbelem->videowall_nb_windows_y = cbelem->videowall_nb_windows_x;
		if((cbelem->videowall_nb_windows_x * cbelem->videowall_nb_windows_y) < cbelem->videowall_nb_streams) // nb_streams is not equal with 3x3, 4x4, etc
		{
			if(output_format.width > output_format.height) // horizontal layout (3x2, 4x3)
				cbelem->videowall_nb_windows_x++;
			else                                           // vertical layout (2x3, 3x4)
				cbelem->videowall_nb_windows_y++;

			if((cbelem->videowall_nb_windows_x * cbelem->videowall_nb_windows_y) < cbelem->videowall_nb_streams) // windows is still less than nb_streams
			{
				if(cbelem->videowall_nb_windows_x < cbelem->videowall_nb_windows_y) // extend windows horizontally
					cbelem->videowall_nb_windows_x++;
				else                                       // extend windows vertically
					cbelem->videowall_nb_windows_y++;
			}
		}
	}

	// calculate the window size of a video stream (sub-picture)
	output_format.width = output_format.width / cbelem->videowall_nb_windows_x;
	output_format.height = output_format.height / (((cbelem->videowall_output_artype == DISPQT_VIDEOFULLSCREENARTYPE_FILL) || (cbelem->videowall_nb_windows_y > cbelem->videowall_nb_windows_x))? cbelem->videowall_nb_windows_y : cbelem->videowall_nb_windows_x);
	output_format.width = output_format.width & ~7;   // FIXME: check
	output_format.height = output_format.height & ~3; //

	if((output_format.width != cbelem->videowall_subwindow_format.width) || (output_format.height != cbelem->videowall_subwindow_format.height) || (output_format.pixelformat != cbelem->videowall_subwindow_format.pixelformat))
	{
		struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEOWALL];
		int i;
		for(i = MPXPLAY_STREAMTYPEINDEX_VIDEOWALL; i < MPXPLAY_STREAMTYPEINDEX_MAX; i++, fvdt++)
		{
			if(fvdt->video_frame_decoded_preprocessed)
				av_frame_free(&fvdt->video_frame_decoded_preprocessed);
			funcbit_disable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY);
		}
		cbelem->videowall_subwindow_format = output_format;
	}

	cbelem->videowall_count_streams = 0;
}

// copy decoded and resized sub-video frames into one screen-size video frame   (FIXME: this is a slow software solution)
static void videoworkdec_videowall_merge_windows(struct ffmpegvideo_callback_list_s *cbelem)
{
	struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO];
	const unsigned int packet_flags = (fvdt->resetvideo || fvdt->resetoutqueue)? MPXPLAY_PACKETLISTFLAG_CLEARBUF : 0;
	AVFrame *vid_frame_merged, *vid_frame_primaryvideo;
	const int bytes_per_pixel = 4; // RGB output
	int i, w, linebytes_in, linebytes_out, vertical_shift;
	int videowall_nb_windows_max;

	cbelem->videowall_count_streams = 0;

	vid_frame_primaryvideo = fvdt->video_frame_decoded_preprocessed;
	if(!vid_frame_primaryvideo) // we synchronize to the primary decoded video frame
		return;

#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
	if((fvdt->videoframe_output_queue.nb_packets >= DISPQT_FFMPGVIDEO_MAX_QUEUED_GEN_VIDEOFRAME) && !fvdt->resetvideo)
		return;
#else
	if(fvdt->videoframe_pktlist_elem && !fvdt->resetvideo)
		return;
#endif

	videowall_nb_windows_max = cbelem->videowall_nb_windows_x * cbelem->videowall_nb_windows_y;
	if(videowall_nb_windows_max <= 0)
		return;

	fvdt->resetvideo = fvdt->resetoutqueue = fvdt->resetcodec = false;

	cbelem->videowall_sync_timestamp = vid_frame_primaryvideo->pts;

	vid_frame_merged = av_frame_alloc();
	if(!vid_frame_merged)
		return;

	if( (cbelem->videowall_output_artype != DISPQT_VIDEOFULLSCREENARTYPE_FILL)
	 && (cbelem->videowall_nb_windows_x >= cbelem->videowall_nb_windows_y)
	 && (cbelem->videowall_nb_streams <= (cbelem->videowall_nb_windows_x * (cbelem->videowall_nb_windows_x - 1))) // eg. 3 x 2 windows
	){   // then reduce output frame height to put it vertically centered
		vid_frame_merged->height = cbelem->videowall_output_format.height * (cbelem->videowall_nb_windows_x - 1) / cbelem->videowall_nb_windows_x;
		vertical_shift = (cbelem->videowall_output_format.height - vid_frame_merged->height) / 2; // FIXME: bullshit
	}
	else
	{
		vid_frame_merged->height = cbelem->videowall_output_format.height;
		vertical_shift = 0;
	}
	vid_frame_merged->width = cbelem->videowall_output_format.width;
	vid_frame_merged->format = cbelem->videowall_output_format.pixelformat;
	vid_frame_merged->sample_aspect_ratio.num = vid_frame_merged->sample_aspect_ratio.den = 0;

	if(av_frame_get_buffer(vid_frame_merged, 0) != 0)
	{
		av_frame_free(&vid_frame_merged);
		return;
	}

	linebytes_in = cbelem->videowall_subwindow_format.width * bytes_per_pixel;
	linebytes_out = vid_frame_merged->linesize[0];

	//if((linebytes_out % linebytes_in) || (cbelem->videowall_nb_streams < videowall_nb_windows_max)) // TODO:
	{
		pds_memset(vid_frame_merged->data[0], 0, vid_frame_merged->height * linebytes_out);
	}

	fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEOWALL];
	for(i = MPXPLAY_STREAMTYPEINDEX_VIDEOWALL, w = 0; i < MPXPLAY_STREAMTYPEINDEX_MAX; i++, fvdt++)
	{
		AVFrame *vid_frame_window = fvdt->video_frame_decoded_preprocessed;
		if(vid_frame_window)
		{
			if((w < videowall_nb_windows_max) && (vid_frame_window->linesize[0] >= linebytes_in) && (vid_frame_window->format == cbelem->videowall_subwindow_format.pixelformat))
			{
				const int destpos = (w / cbelem->videowall_nb_windows_x) * linebytes_out * cbelem->videowall_subwindow_format.height + (w % cbelem->videowall_nb_windows_x) * linebytes_in;
				uint8_t *destptr = vid_frame_merged->data[0] + destpos;
				uint8_t *srcptr =  vid_frame_window->data[0];
				int j = vid_frame_window->height;

				do{
					pds_memcpy(destptr, srcptr, linebytes_in);
					srcptr+= linebytes_in;
					destptr += linebytes_out;
				}while(--j);

				fvdt->videowall_window_pos_x = (w % cbelem->videowall_nb_windows_x) * cbelem->videowall_subwindow_format.width;
				fvdt->videowall_window_pos_y = (w / cbelem->videowall_nb_windows_x) * cbelem->videowall_subwindow_format.height + vertical_shift;
				fvdt->videowall_window_width = vid_frame_window->width;
				fvdt->videowall_window_height = vid_frame_window->height;
				if(!w)
				{
					mpxplay_debugf(DISPQT_DEBUGOUT_WV_MERGE, "video_merge w:%d h:%d ow:%d oh:%d", fvdt->videowall_window_width, fvdt->videowall_window_height,
							cbelem->videowall_output_format.width / cbelem->videowall_nb_windows_x, cbelem->videowall_output_format.height / cbelem->videowall_nb_windows_y);
				}
			}
			else
			{
				fvdt->videowall_window_width = 0;
			}
			w++;
		}
		else
		{
			fvdt->videowall_window_width = 0;
		}
	}
	cbelem->videowall_count_streams = w;

	vid_frame_merged->pts = vid_frame_primaryvideo->pts;
	vid_frame_merged->pkt_dts = vid_frame_primaryvideo->pkt_dts;
	vid_frame_merged->best_effort_timestamp = vid_frame_primaryvideo->best_effort_timestamp;
#if !MPXPLAY_USE_FFMPEG_V7X
	vid_frame_merged->pkt_duration = vid_frame_primaryvideo->pkt_duration;
#endif

	fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO]; // TODO: we should use the primary video stream for the selected_stream_index (currently it's a duplicated decoding)
	av_frame_free(&fvdt->video_frame_decoded_preprocessed);
	video_worker_videodecoder_send_frame_to_scheduler(cbelem, fvdt, packet_flags, vid_frame_merged, DISPQT_FFMPGVIDEO_MAX_QUEUED_GEN_VIDEOFRAME);

	mpxplay_debugf(DISPQT_DEBUGOUT_WV_MERGE, "video_merge DONE psi:%d w:%d pts:%lld", cbelem->selected_stream_index, cbelem->videowall_count_streams, vid_frame_merged->pts);
}

//-----------------------------------------------------------------------------------------------------------------------------------------

static int video_worker_videodecoder_filter_process(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt, AVFrame **vid_frame_filtered,
		unsigned int packet_flags, const mpxp_bool_t videowall_is_enabled)
{
	DispQtVideoFrameFilter *video_frame_filter = ((fvdt->streamtype_index == cbelem->selected_stream_index) && !fvdt->video_frame_filter_workerprocess)? cbelem->video_frame_filter_workerprocess : fvdt->video_frame_filter_workerprocess;
	int filter_retVal = DISPQT_VIDEOFILTER_APPLYRETBIT_INPUTFRAME_PROCESSED;
	unsigned int skipframe_level;

	skipframe_level = video_worker_videodecoder_check_skip_frame(cbelem, fvdt, packet_flags, videowall_is_enabled);
	if(skipframe_level == MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_ALL)
	{
		av_frame_free(&fvdt->vid_frame_decoded);
		*vid_frame_filtered = NULL;
	}
	else if((skipframe_level == MPXPLAY_WORKDEC_SKIPFRAME_LEVEL_NONE) && video_frame_filter && (fvdt->vid_frame_decoded->format != mpxplay_video_render_infos.hwdevice_avpixfmt)
		&& ((fvdt->streamtype_index != MPXPLAY_STREAMTYPEINDEX_VIDEO) || !funcbit_test(mpxplay_video_render_infos.renderer_capabilities, DISPQT_VIDEORENDERER_CAPFLAGS_PREPROCESS))
	){
		filter_retVal = video_frame_filter->video_filter_apply(fvdt->vid_frame_decoded, vid_frame_filtered, AV_PIX_FMT_NONE);
		if(fvdt->vid_frame_decoded)
		{
			if(funcbit_test(filter_retVal, DISPQT_VIDEOFILTER_APPLYRETBIT_INPUTFRAME_PROCESSED))
			{
				av_frame_free(&fvdt->vid_frame_decoded);
			}
			else if(!funcbit_test(filter_retVal, DISPQT_VIDEOFILTER_APPLYRETBIT_OUTPUTFRAME_PRODUCED))
			{
				filter_retVal = DISPQT_VIDEOFILTER_APPLYRETBIT_INPUTFRAME_PROCESSED;
				*vid_frame_filtered = fvdt->vid_frame_decoded;
				fvdt->vid_frame_decoded = NULL;
			}
		}
	}
	else
	{
		*vid_frame_filtered = fvdt->vid_frame_decoded;
		fvdt->vid_frame_decoded = NULL;
	}

	// convert video frame to a smaller one and to a target pixelformat (seekpreview and videowall)
	if( (filter_retVal > DISPQT_VIDEOFILTER_APPLYRETBIT_NOFILTERAPPLY) && (fvdt->streamtype_index >= MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
	 && *vid_frame_filtered && ((*vid_frame_filtered)->height > 0)
	){
		AVFrame *vid_frame_input = *vid_frame_filtered;
		struct dispqt_video_filter_format_s output_format;

		*vid_frame_filtered = NULL;

		if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
		{
			output_format = mpxplay_videoworkdec_seekpreview_format;

			if(output_format.width == 0) // then use video aspect ratio
			{
				int ar_num = (vid_frame_input->sample_aspect_ratio.num > 0)? vid_frame_input->sample_aspect_ratio.num : 1;
				int ar_den = (vid_frame_input->sample_aspect_ratio.den > 0)? vid_frame_input->sample_aspect_ratio.den : 1;
				output_format.width = vid_frame_input->width * output_format.height / vid_frame_input->height * ar_num / ar_den;
				output_format.width = ((output_format.width + 4) & ~7);
			}
			else if(output_format.width == 1) // then use pixel aspect ratio
			{
				output_format.width = vid_frame_input->width * output_format.height / vid_frame_input->height;
				output_format.width = ((output_format.width + 4) & ~7);
			}
		}
		else // MPXPLAY_STREAMTYPEINDEX_VIDEOWALL
		{
			output_format = cbelem->videowall_subwindow_format;
		}

		if(mpxplay_dispqt_videotrans_ffmpeg_swsctx_conv_frame(&fvdt->frame_conv_infos, vid_frame_input, &output_format))
		{
			*vid_frame_filtered = av_frame_clone(fvdt->frame_conv_infos.ffmpeg_sws_frame);
			(*vid_frame_filtered)->pts = vid_frame_input->pts;
			(*vid_frame_filtered)->pkt_dts = vid_frame_input->pkt_dts;
			(*vid_frame_filtered)->best_effort_timestamp = vid_frame_input->best_effort_timestamp;
#if !MPXPLAY_USE_FFMPEG_V7X
			(*vid_frame_filtered)->pkt_duration = vid_frame_input->pkt_duration;
#endif
			funcbit_copy((*vid_frame_filtered)->flags, vid_frame_input->flags, DISPQT_AVFRAME_FLAGS_MMC);
		}
		av_frame_free(&vid_frame_input);
	}

	return filter_retVal;
}

static void video_worker_videodecoder_frame_process(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt, AVCodecContext *video_ctx,
		unsigned int *sleep_time, unsigned int packet_flags, const mpxp_bool_t videowall_is_enabled)
{
	int retcode_video_receive = -1, retcode_video_send = -1, max_queued_videoframe = 0;
	AVFrame *vid_frame_filtered = NULL;
	mpxp_bool_t packet_used = FALSE;

	if(fvdt->resetfilter)
	{
		av_frame_free(&fvdt->vid_frame_decoded);
		av_frame_free(&fvdt->video_frame_decoded_preprocessed);
		if(cbelem->video_frame_filter_workerprocess && (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO))
			cbelem->video_frame_filter_workerprocess->video_filter_close();  // TODO: other way to clear?
		if(fvdt->video_frame_filter_workerprocess)
			fvdt->video_frame_filter_workerprocess->video_filter_close();    //
		funcbit_disable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY);
		fvdt->resetfilter = false;
	}

	if(!video_ctx)
		goto err_out_frame_process;

	if(!videowall_is_enabled || (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW))
	{
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
		if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
			max_queued_videoframe = (video_ctx->hw_device_ctx)? MPXPLAY_VIDEO_RENDERER_FRAME_HWDECTEXTURES_QUEUE_SIZE :
				(((video_ctx->codec_id == AV_CODEC_ID_AV1) && (video_ctx->width > 2000))? DISPQT_FFMPGVIDEO_MAX_QUEUED_AV1_VIDEOFRAME : DISPQT_FFMPGVIDEO_MAX_QUEUED_DEC_VIDEOFRAME);
		else if(fvdt->streamtype_index != MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
			max_queued_videoframe = 1;
		if(((fvdt->videoframe_temp_queue.nb_packets + fvdt->videoframe_output_queue.nb_packets) >= max_queued_videoframe) && !fvdt->resetvideo)
			goto err_out_frame_process;
#else
		if(fvdt->videoframe_pktlist_elem && !fvdt->resetvideo)
			goto err_out_frame_process;
#endif
	}
	else if(fvdt->video_frame_decoded_preprocessed && !fvdt->resetvideo
		 && ( (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
		   || ((fvdt->streamtype_index == cbelem->selected_stream_index) && (fvdt->video_frame_decoded_preprocessed->pts > cbelem->videowall_sync_timestamp))) // don't let the main VIDEO stream to run away (syncing with the primary/selected videowall stream)
	){
		goto err_out_frame_process;
	}

	do{
		if(!dispqt_videoworkdec_is_video_ctx_valid(video_ctx))
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "VIDEO CTX_BAD %8.8X", (mpxp_ptrsize_t)video_ctx);
		}
#ifdef DISPQT_VIDEO_DECODER_DETECT_D3D_FAILURE
		else if(video_ctx->hw_device_ctx && mpxplay_dispqt_videowidget_videorender_detect_device_context_failure())
		{   // reopen file (to re-initialize d3d context) if hw decoding is halted by d3d11 device failure // FIXME: this is a workaround only
			if(!funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD))
			{
				funcbit_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD);
				funcbit_enable(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PAUSEDEC);
				mpxplay_timer_addfunc((void *)mpxplay_dispqt_main_close, NULL, MPXPLAY_TIMERTYPE_SIGNAL, MPXPLAY_SIGNALTYPE_GUIREADY);
				mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "avcodec_send_packet ADONE_REOPEN");
			}
			break;
		}
#endif
		else if(!fvdt->vid_frame_decoded)
		{
			fvdt->vid_frame_decoded = av_frame_alloc();
			if(!fvdt->vid_frame_decoded)
				break;

			video_worker_videodecoder_frameprocess_speedup_on_delay(cbelem, fvdt, video_ctx);

			retcode_video_receive = avcodec_receive_frame(video_ctx, fvdt->vid_frame_decoded);

#if 0 // TODO: test this feature's possibilities, but it seems, currently it doesn't work in FFmpeg
			if((retcode_video_receive < 0) || (fvdt->vid_frame_decoded->flags & (AV_FRAME_FLAG_CORRUPT | AV_FRAME_FLAG_DISCARD)) || (fvdt->vid_frame_decoded->decode_error_flags) /* & (FF_DECODE_ERROR_INVALID_BITSTREAM | FF_DECODE_ERROR_MISSING_REFERENCE))*/)
			{
				if(retcode_video_receive != AVERROR(EAGAIN))
				{
					mpxplay_debugf(DISPQT_DEBUGOUT_WARNING, "avcodec_receive_frame ERROR %8.8X r:%d f:%d df:%d d:%8.8X ls:%d ", (mpxp_ptrsize_t)cbelem,
							retcode_video_receive, fvdt->vid_frame_decoded->flags, fvdt->vid_frame_decoded->decode_error_flags, (mpxp_ptrsize_t)fvdt->vid_frame_decoded->data[0], fvdt->vid_frame_decoded->linesize[0]);
				}
			}
#endif
			if(retcode_video_receive == 0)
			{
				if( (fvdt->vid_frame_decoded->width <= 0) || (fvdt->vid_frame_decoded->height <= 0) || (fvdt->vid_frame_decoded->format < 0)
				 || (fvdt->vid_frame_decoded->flags & AV_FRAME_FLAG_DISCARD)
			     || ((fvdt->vid_frame_decoded->linesize[0] <= 0) && (fvdt->vid_frame_decoded->format != mpxplay_video_render_infos.hwdevice_avpixfmt))
				){
					retcode_video_receive = -1000;
				}
				else
				{
					switch(fvdt->vid_frame_decoded->format)
					{
						case AV_PIX_FMT_DXVA2_VLD:
							if(!fvdt->vid_frame_decoded->data[3])
								retcode_video_receive = -1001;
							break;
						default:
							if(!fvdt->vid_frame_decoded->data[0])
								retcode_video_receive = -1002;
							break;
					}
				}
			}

			if(retcode_video_receive == 0)
			{
				fvdt->video_decoder_lost_frames = 0;
				// we switch on this feature only after the first frame received (for a faster initial displaying of videowall)
				funcbit_copy(fvdt->flags, cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY);

				videoworkdec_update_metadata_from_vframe(cbelem, fvdt, fvdt->vid_frame_decoded, packet_flags);

				videoworkdec_get_videoframe_timestamp(cbelem, fvdt, fvdt->vid_frame_decoded, packet_flags); // TODO: remove callback

#ifdef MPXPLAY_USE_DEBUGF
				//if((fvdt->streamtype_index >= MPXPLAY_STREAMTYPEINDEX_VIDEOWALL) && (fvdt->streamtype_index != cbelem->selected_stream_index))
				//if(fvdt->videopkt_init_count < INFFMPEG_VIDEO_INITCOUNT_LIMIT)
				{
					AVFrame *vframe = fvdt->vid_frame_decoded;
					mpxplay_debugf(DISPQT_DEBUGOUT_FRAMEDEC, "VIDEO FRM dts:%lld pts:%lld tb:%d/%d",
							vframe->pkt_dts, vframe->pts, vframe->time_base.den, vframe->time_base.num);
				}
#endif
			}
			else
			{
				av_frame_free(&fvdt->vid_frame_decoded);
				mpxplay_debugf(DISPQT_DEBUGOUT_FRAMEDEC, "VIDEO FRAME_BAD i:%d r:%d", fvdt->streamtype_index, retcode_video_receive);
			}
		}

		if(video_worker_videodecoder_filter_process(cbelem, fvdt, &vid_frame_filtered, packet_flags, videowall_is_enabled) < 0)
		{
			break;
		}

		if(!fvdt->video_pktlist_elem)
		{
			break;
		}

		AVCodecContext *packet_vcodec_ctx = (AVCodecContext *)fvdt->video_pktlist_elem->codec_ctx;
		if(!packet_vcodec_ctx || !fvdt->video_pktlist_elem->frame_pkt || (fvdt->video_pktlist_elem->packet_type != MPXPLAY_PACKETTYPE_AVPACKET)) // invalid packet
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "PACKET invalid");
			packet_used = TRUE;
			break;
		}
		if((packet_vcodec_ctx != video_ctx) && (fvdt->video_pktlist_elem->codecctx_counter < fvdt->videoctx_curr_counter)) // old packet ctx
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "PACKET OLD cp:%d vc:%d", fvdt->video_pktlist_elem->codecctx_counter, fvdt->videoctx_curr_counter);
			packet_used = TRUE;
			break;
		}
		if(!dispqt_videoworkdec_is_video_ctx_valid(packet_vcodec_ctx)) // invalid packet ctx
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "PACKET CTX invalid");
			if(++fvdt->videoctx_init_error_count > INFFMPEG_VIDEO_INITERROR_LIMIT) // we drop the packet (containing the incorrect ctx) after a time
				packet_used = TRUE;
			break;
		}

		retcode_video_send = avcodec_send_packet(packet_vcodec_ctx, (AVPacket *)fvdt->video_pktlist_elem->frame_pkt);
		if(retcode_video_send == 0)
		{
			packet_used = TRUE;
			fvdt->video_decoder_lost_frames++;
#ifdef MPXPLAY_USE_DEBUGF
			fvdt->videopkt_send_error_count = 0;
//			if((fvdt->streamtype_index >= MPXPLAY_STREAMTYPEINDEX_VIDEOWALL) && (fvdt->streamtype_index != cbelem->selected_stream_index))
			//if(fvdt->videopkt_init_count < INFFMPEG_VIDEO_INITCOUNT_LIMIT)
			{
				AVPacket *pkt_video = (AVPacket *)fvdt->video_pktlist_elem->frame_pkt;
				mpxplay_debugf(DISPQT_DEBUGOUT_FRAMEDEC, "VIDEO PKT dts:%lld pts:%lld tb:%d/%d",
						pkt_video->dts, pkt_video->pts, pkt_video->time_base.den, pkt_video->time_base.num);
				fvdt->videopkt_init_count ++;
			}
#endif
		}
		else if(retcode_video_send != AVERROR(EAGAIN))
		{
#ifdef MPXPLAY_USE_DEBUGF
			if(fvdt->videopkt_send_error_count < INFFMPEG_VIDEO_INITCOUNT_LIMIT)
			{
				AVPacket *av_pkt = (AVPacket *)fvdt->video_pktlist_elem->frame_pkt;
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "avcodec_send_packet VIDEO failed ret:%d st:%d si:%d dts:%lld pctx:%8.8X vctx:%8.8X open:%d",
					retcode_video_send, fvdt->streamtype_index, av_pkt->stream_index, av_pkt->dts, packet_vcodec_ctx, video_ctx, avcodec_is_open(packet_vcodec_ctx));
				fvdt->videopkt_send_error_count ++;
			}
#endif
			dispqt_videoworkdec_video_ctx_flush(packet_vcodec_ctx, fvdt->streamtype_index);
			packet_used = TRUE;                       //
#ifdef DISPQT_VIDEO_DECODER_DETECT_D3D_FAILURE
			// reopen file (to re-initialize d3d context) if hw decoding is halted by d3d11 device failure // FIXME: this is a workaround only
//			if(!funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD) && packet_vcodec_ctx->hw_device_ctx && mpxplay_dispqt_videowidget_videorender_detect_device_context_failure())
//			{
//				funcbit_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD);
//				funcbit_enable(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PAUSEDEC);
//				//video_ctx->hwaccel = NULL; // FIXME: else avcodec_free_context() freeze
//				mpxplay_timer_addfunc((void *)mpxplay_dispqt_main_close, NULL, MPXPLAY_TIMERTYPE_SIGNAL, MPXPLAY_SIGNALTYPE_GUIREADY);
//				mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "avcodec_send_packet ADONE_REOPEN");
//			}
#endif
		}
#ifdef MPXPLAY_USE_DEBUGF
		else if(fvdt->videopkt_init_count < INFFMPEG_VIDEO_INITERROR_LIMIT)
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_FRAMEDEC, "VIDEO PKT EAGAIN");
		}
#endif
	}while(FALSE);

	if(packet_used) // clear packet if it's sent
	{
		dispqt_videoworkdec_pktlistelem_clear(fvdt);
		fvdt->videoctx_init_error_count = 0;
		*sleep_time = 0; // if !vid_frame_filtered
	}
	else
	{
		*sleep_time = MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT; // give a short time to the decoder if no picture, but input queue is full
	}

	if(vid_frame_filtered)
	{
		if(vid_frame_filtered->width <= 0)
			vid_frame_filtered->width = video_ctx->width;
		if(vid_frame_filtered->height <= 0)
			vid_frame_filtered->height = video_ctx->height;
		if(vid_frame_filtered->format < 0)
			vid_frame_filtered->format = ((video_ctx->pix_fmt > AV_PIX_FMT_NONE)? video_ctx->pix_fmt : AV_PIX_FMT_YUV420P); // FIXME: default AV_PIX_FMT_YUV420P

		mpxplay_debugf(DISPQT_DEBUGOUT_AR, "AR vn:%d vd:%d fn:%d fd:%d", fvdt->videoctx_sample_aspect_ratio.num, fvdt->videoctx_sample_aspect_ratio.den, vid_frame_filtered->sample_aspect_ratio.num, vid_frame_filtered->sample_aspect_ratio.den);
		if(funcbit_test(packet_flags, MPXPLAY_PACKETLISTFLAG_STILLPICT))
		{   // still pictures always use pixel aspect ratio
			vid_frame_filtered->sample_aspect_ratio.num = vid_frame_filtered->sample_aspect_ratio.den = 1;
		}
		else if((vid_frame_filtered->sample_aspect_ratio.num <= 1) && (vid_frame_filtered->sample_aspect_ratio.den <= 1))
		{   // use container AR, if stream AR seems not valid
			vid_frame_filtered->sample_aspect_ratio = fvdt->videoctx_sample_aspect_ratio;
		}

		if(fvdt->resetcodec)
		{
			funcbit_enable(packet_flags, MPXPLAY_PACKETLISTFLAG_RESET);
		}
		if(fvdt->resetvideo)
		{
			funcbit_enable(packet_flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
		}

		if(fvdt->video_frame_decoded_preprocessed)
		{
			av_frame_free(&fvdt->video_frame_decoded_preprocessed);
		}

		if(!videowall_is_enabled || (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW))
		{
			if((fvdt->streamtype_index == cbelem->selected_stream_index) || (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) )
			{
				mpxplay_debugf(DISPQT_DEBUGOUT_D3D11, "VIDEO DONE i:%d w:%3d h:%3d f:%d pts:%lld nb:%d skip:%d pfl:%8.8X",
						fvdt->streamtype_index, vid_frame_filtered->width, vid_frame_filtered->height, vid_frame_filtered->format,
						vid_frame_filtered->pts,
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
						fvdt->videoframe_output_queue.nb_packets,
#else
						0,
#endif
						video_ctx->skip_frame, packet_flags);
			}
#if 0 //def DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
			if(dispqt_videoworkdec_is_video_ctx_hwaccel(video_ctx) && (fvdt->resetoutqueue || fvdt->resetvideo))
			{
				mpxplay_ffmpgdec_packetqueue_clear(&fvdt->videoframe_output_queue, 0);
#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
				mpxplay_dispqt_videowidget_videorender_flush(TRUE);
#endif
			}
#endif
			video_worker_videodecoder_send_frame_to_scheduler(cbelem, fvdt, packet_flags, vid_frame_filtered, max_queued_videoframe);
		}
		else // save frame for videowall windows merge
		{
			fvdt->video_frame_decoded_preprocessed = vid_frame_filtered;
		}

		if(!videowall_is_enabled || (fvdt->streamtype_index != MPXPLAY_STREAMTYPEINDEX_VIDEO))
		{
			fvdt->resetvideo = fvdt->resetoutqueue = fvdt->resetcodec = false;
		}

		cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_SET_CODECCTX_CONSOLIDATE, (mpxp_ptrsize_t)video_ctx, fvdt->streamtype_index, INFFMPG_MUTEX_PACKET_TIMEOUT + 7);

		//*sleep_time = (video_ctx->hw_device_ctx)? 1 : 0; // for testing only
		*sleep_time = (video_ctx->hw_device_ctx)? MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_HWDEC : 0;
	}

err_out_frame_process:
	return;
}

static void video_worker_audiovisualization_process(struct ffmpegvideo_callback_list_s *cbelem, struct ffmpegvideo_decoder_s *fvdt, unsigned int *sleep_time, unsigned int packet_flags)
{
	int streamtype_index = MPXPLAY_STREAMTYPEINDEX_VIDEO;
	bool audio_input_processed = true;
	if(!fvdt->video_pktlist_elem)
		cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_PACKETLIST_ELEM, (mpxp_ptrsize_t)&fvdt->video_pktlist_elem, streamtype_index, INFFMPG_MUTEX_PACKET_TIMEOUT + 8);
	if(fvdt->video_pktlist_elem)
	{
		AVFrame *audio_frame_decoded = (AVFrame *)fvdt->video_pktlist_elem->frame_pkt;
		if( (fvdt->video_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
		 || (fvdt->videoframe_output_queue.nb_packets > DISPQT_FFMPGVIDEO_MAX_DELAY_VISVIDEOFRAME)
#endif
		){
			funcbit_disable(fvdt->video_pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
			fvdt->resetvideo = true;
		}
		mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "audio visualization pe:%8.8X afd:%8.8X wp:%8.8X pt:%d qn:%d pts:%lld",
				(mpxp_ptrsize_t)fvdt->video_pktlist_elem, (mpxp_ptrsize_t)audio_frame_decoded, cbelem->video_frame_filter_workerprocess,
				fvdt->video_pktlist_elem->packet_type,
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
				fvdt->videoframe_output_queue.nb_packets,
#else
				0,
#endif
				(audio_frame_decoded)? audio_frame_decoded->pts : 0);
		if(audio_frame_decoded && (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) && (fvdt->video_pktlist_elem->packet_type == MPXPLAY_PACKETTYPE_AVFRAME_AUDIO) && cbelem->video_frame_filter_workerprocess
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
		 && (fvdt->videoframe_output_queue.nb_packets < DISPQT_FFMPGVIDEO_MAX_QUEUED_VISVIDEOFRAME)
#else
		 && !fvdt->videoframe_pktlist_elem
#endif
		){
			AVFrame *vid_frame_visual = NULL;
			int retcode = cbelem->video_frame_filter_workerprocess->video_filter_apply(audio_frame_decoded, &vid_frame_visual, DISPQT_VIDEO_FILTERPROCESS_AVFORMAT);
			if(vid_frame_visual)
			{
				if(fvdt->resetvideo)
				{
					videoworkdec_get_videoframe_timestamp(cbelem, fvdt, vid_frame_visual, packet_flags); // FIXME: converts pts by video stream
					cbelem->video_frame_filter_workerprocess->video_filter_close();  // TODO: other way to reset?
					funcbit_enable(packet_flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
					funcbit_enable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP);
					fvdt->resetvideo = false;
					mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "audio visualization RESET n:%d w:%d h:%d pts:%lld",
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
							fvdt->videoframe_output_queue.nb_packets,
#else
							0,
#endif
							vid_frame_visual->width, vid_frame_visual->height, vid_frame_visual->pts);
				}
				// FIXME: this is a workaround for large audio blocks + visual plugins
				else
				{
#if 1
					mpxp_int64_t diff = vid_frame_visual->pts - fvdt->video_dec_last_timestamp;
					mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "vid_frame_visual pts:%lld DIFF:%lld", vid_frame_visual->pts, diff);
					if((diff > -(AV_TIME_BASE / 24)) && (diff < (AV_TIME_BASE / 50))) // visual plugin shall send with 25 fps
					{
						vid_frame_visual->pts = fvdt->video_dec_last_timestamp + (AV_TIME_BASE / 26);
					}
#endif
					if(!funcbit_test(retcode, DISPQT_VIDEOFILTER_APPLYRETBIT_INPUTFRAME_PROCESSED))
					{
						audio_input_processed = false;
					}
				}
				fvdt->video_dec_last_timestamp = vid_frame_visual->pts;
				//funcbit_enable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP); // for testing only
				funcbit_enable(vid_frame_visual->flags, (DISPQT_AVFRAME_FLAG_NOSKIPFRAME | DISPQT_AVFRAME_FLAG_AR_WINDOW));
				if(video_worker_videodecoder_send_frame_to_scheduler(NULL, fvdt, packet_flags, vid_frame_visual, DISPQT_FFMPGVIDEO_MAX_QUEUED_VISVIDEOFRAME) < 0)
				{
					av_frame_free(&vid_frame_visual);
				}
				mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "mpxplay_ffmpgdec_packetqueue_put DONE n:%d w:%d h:%d pts:%lld",
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
						fvdt->videoframe_output_queue.nb_packets,
#else
						0,
#endif
						vid_frame_visual->width, vid_frame_visual->height, vid_frame_visual->pts);
				if(*sleep_time > MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT)
					*sleep_time = MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT;
			}
			else
			{
				*sleep_time = 0;
				mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "vid_frame_visual NONE ret:%d", retcode);
			}
		}

		if(audio_input_processed)
		{
			dispqt_videoworkdec_pktlistelem_clear(fvdt);
		}
	}
}

void mpxplay_dispqt_video_decoder_worker_close(struct ffmpegvideo_callback_list_s *cbelem)
{
	int stream_index;

	for(stream_index = MPXPLAY_STREAMTYPEINDEX_VIDEO; stream_index < MPXPLAY_STREAMTYPEINDEX_MAX; stream_index++)
	{
		struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[stream_index];
		const mpxp_bool_t is_avctx_valid = dispqt_videoworkdec_is_video_ctx_valid(fvdt->video_ctx);
		const mpxp_bool_t is_hwacc_ctx = (is_avctx_valid && dispqt_videoworkdec_is_video_ctx_hwaccel(fvdt->video_ctx));
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		const int mutex_error = (is_hwacc_ctx)? PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT) : -1;
#endif

		//if(is_avctx_valid)
		//	avcodec_flush_buffers(fvdt->video_ctx);

		mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->video_pktlist_elem);
		mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_temp_elem);
		mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_pktlist_elem);
		mpxplay_ffmpgdec_packetqueue_clear(&fvdt->videoframe_temp_queue, 0);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
		mpxplay_ffmpgdec_packetqueue_clear(&fvdt->videoframe_output_queue, 0);
#endif

		if(fvdt->vid_frame_decoded)
			av_frame_free(&fvdt->vid_frame_decoded);
		if(fvdt->video_frame_decoded_preprocessed)
			av_frame_free(&fvdt->video_frame_decoded_preprocessed);

		cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_SET_CODECCTX_CLEAR, fvdt->streamtype_index, 0, INFFMPG_MUTEX_COMMAND_TIMEOUT + 2);

#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
		if(is_hwacc_ctx)
			mpxplay_dispqt_videowidget_videorender_flush(TRUE);
#endif

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		if(mutex_error == MPXPLAY_ERROR_OK)
			PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
	}
}

void mpxplay_dispqt_video_decoder_worker_process(struct ffmpegvideo_callback_list_s *cbelem, unsigned int *sleep_time)
{
	unsigned int stream_index, streamindex_max, packet_flags;
	mpxp_bool_t videwall_is_enabled = FALSE;
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK) && (defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD) || defined(DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD))
	int mutex_error = -1;
#endif

	mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "video_decoder_worker_process BEGIN %8.8X", (mpxp_ptrsize_t)cbelem);

	if(!cbelem || !cbelem->infile_callback || funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_THEXITDONE))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "video_decoder_worker_process ERROR cbelem");
		return;
	}

	if(funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_EXITEVENT))
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "video_decoder_worker_process CLOSE %8.8X", (mpxp_ptrsize_t)cbelem);
		mpxplay_dispqt_video_decoder_worker_close(cbelem);
		funcbit_enable(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_THEXITDONE);
		return;
	}

	videwall_is_enabled = videoworkdec_videowall_check(cbelem, &streamindex_max);

	for(stream_index = MPXPLAY_STREAMTYPEINDEX_VIDEO; stream_index < streamindex_max; stream_index++)
	{
		struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[stream_index];

		if(funcbit_test(cbelem->flags, (DISPQT_FFMPGVIDEOCALLBACKFLAG_EXITEVENT | DISPQT_FFMPGVIDEOCALLBACKFLAG_PAUSEDEC)))
		{
			videwall_is_enabled = FALSE;
			break;
		}
		if(stream_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
			continue;

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK) && (defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD) || defined(DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD))
		if(dispqt_videoworkdec_is_video_ctx_hwaccel(fvdt->video_ctx))
			mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif

		packet_flags = 0;
		video_worker_videodecoder_context_process(cbelem, fvdt, sleep_time, &packet_flags);

		if(funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO))
		{
			if(stream_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
			{
				if(fvdt->video_ctx)
				{
					cbelem->is_videowall_disabled_by_decoder = FALSE;
				}
				else
				{
					cbelem->is_videowall_disabled_by_decoder = TRUE;
					videwall_is_enabled = FALSE;
					streamindex_max = MPXPLAY_STREAMTYPEINDEX_NUM;
				}
			}
			if(fvdt->video_ctx) // TODO: extra switch to show audio visualization for video files too
			{
				video_worker_videodecoder_frame_process(cbelem, fvdt, fvdt->video_ctx, sleep_time, packet_flags, videwall_is_enabled);
				mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "video_decoder_worker_process videodecoder_frame_process END st:%d ctx:%8.8X", stream_index, (mpxp_ptrsize_t)fvdt->video_ctx);
			}
			else if((mpxplay_config_video_audiovisualization_type != DISPQT_AUDIOVISUALIZATIONTYPE_NONE) && (stream_index == MPXPLAY_STREAMTYPEINDEX_VIDEO))
			{
				video_worker_audiovisualization_process(cbelem, fvdt, sleep_time, packet_flags);
				//mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "video_decoder_worker_thread audiovisualization_process END st:%d ctx:%8.8X", stream_index, (mpxp_ptrsize_t)fvdt->video_ctx);
			}
			if(fvdt->resetoutqueue)
			{
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
				if((stream_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) || (stream_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW))
					mpxplay_ffmpgdec_packetqueue_put(&fvdt->videoframe_output_queue, MPXPLAY_PACKETTYPE_AVFRAME_VIDEO, MPXPLAY_PACKETLISTFLAG_CLEARBUF, NULL, 0, NULL, NULL);
#endif
				fvdt->resetoutqueue = false;
				mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "RESETVIDEO END st:%d ctx:%8.8X q:%d sl:%d", stream_index, (mpxp_ptrsize_t)fvdt->video_ctx,
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
						fvdt->videoframe_output_queue.nb_packets,
#else
						0,
#endif
						sleep_time);
			}
		}
		else
		{   // currently we simply drop non primary video packets and primary video packet in videowall mode for MPXPLAY_STREAMTYPEINDEX_VIDEO (later can be video crossfade)
			dispqt_videoworkdec_pktlistelem_clear(fvdt);
			fvdt->resetvideo = fvdt->resetfilter = false;
			if(stream_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
			{
				mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "video_decoder_worker_process DROP END smax:%d", streamindex_max);
			}
		}
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK) && (defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD) || defined(DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD))
		if(mutex_error == MPXPLAY_ERROR_OK)
			PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
	}

	if(videwall_is_enabled)
	{
		videoworkdec_videowall_merge_windows(cbelem);
	}

	mpxplay_debugf(DISPQT_DEBUGOUT_VTHREAD, "video_decoder_worker_process END %8.8X", (mpxp_ptrsize_t)cbelem);
}

#ifdef DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD

#if defined(PDS_THREADS_POSIX_THREAD)
void *mpxplay_dispqt_video_decoder_worker_thread(void *decoder_data)
#else
unsigned int mpxplay_dispqt_video_decoder_worker_thread(void *decoder_data)
#endif
{
	struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos = (struct mpxplay_dispqt_ffmpegvideo_worker_info_s *)decoder_data;

	mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "video_decoder_worker_thread BEGIN %8.8X", (mpxp_ptrsize_t)workinfos);

	do{
		unsigned int sleep_time = MPXPLAY_VIDEO_WORKER_THREAD_SLEEPTIME_DEFAULT;
		struct ffmpegvideo_callback_list_s *cbelem;

		cbelem = workinfos->cblist_first;
		while(cbelem)
		{
			mpxplay_dispqt_video_decoder_worker_process(cbelem, &sleep_time);
			cbelem = cbelem->next;
		}
		if(sleep_time > 0)
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_SLEEP, "video_decoder_worker_thread sleep: %d", sleep_time);
			pds_threads_sleep(sleep_time);
		}
	}while(workinfos->video_decoderthread_control != DISPQT_VIDEOSCHEDULER_CONTROL_REQ_CLOSE);

	workinfos->video_decoderthread_control = DISPQT_VIDEOSCHEDULER_STATUS_CLOSED;

	mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "video_decoder_worker_thread END %8.8X", (mpxp_ptrsize_t)workinfos);
	pds_threads_sleep(DISPQT_VIDEO_MUTEX_TIMEOUT * 2);  // required for pds_threads_thread_close -> TerminateThread

	return 0;
}

#elif defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD)

#if defined(PDS_THREADS_POSIX_THREAD)
void *mpxplay_dispqt_video_decoder_worker_thread(void *decoder_data)
#else
unsigned int mpxplay_dispqt_video_decoder_worker_thread(void *decoder_data)
#endif
{
	struct ffmpegvideo_callback_list_s *cbelem = (struct ffmpegvideo_callback_list_s *)decoder_data;
	int cberror_count = 0;

	mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "video_decoder_worker_thread BEGIN %8.8X", (mpxp_ptrsize_t)cbelem);

	do{
		unsigned int sleep_time = MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_DEFAULT;

		if(!cbelem || !cbelem->infile_callback)
		{
			if(++cberror_count < MPXPLAY_VIDEO_WORKDEC_THREAD_CRITICAL_ERROR_COUNT)
			{
				pds_threads_sleep(sleep_time);
				continue;
			}
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "video_decoder_worker_thread ERROR cbelem");
			break;
		}
		cberror_count = 0;

		mpxplay_dispqt_video_decoder_worker_process(cbelem, &sleep_time);

		if(sleep_time > 0)
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_SLEEP, "sleep: %d", sleep_time);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_COND_SIGNAL
			pds_threads_cond_timedwait(cbelem->videodec_thread_condition_signal_handler, sleep_time);
#else
			pds_threads_sleep(sleep_time);
#endif
		}
	}while(!funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_THEXITDONE));

	mpxplay_debugf(DISPQT_DEBUGOUT_CLEAR, "video_decoder_worker_thread END %8.8X", (mpxp_ptrsize_t)cbelem);

	pds_threads_sleep(DISPQT_VIDEO_MUTEX_TIMEOUT * 2);  // required for pds_threads_thread_close -> TerminateThread

	return 0;
}

#endif // defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD) && !defined(DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD)

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

