/*
	ListPCI - PCI device listing utility
	Copyright (C) 2021 Mercury Thirteen

	This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/



#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <stdbool.h>



// function prototypes
unsigned long		DeviceDataPrint(unsigned long classMatch, unsigned long vendorMatch);
void				DeviceDescriptionStringPrint(unsigned long deviceClass, unsigned long deviceSubclass, unsigned long deviceProgIF);
void				DeviceVendorStringPrint(unsigned long deviceVendor);
bool				PCICheck();
unsigned long		PCIRegisterRead(unsigned short bus, unsigned short device, unsigned short funct, unsigned short reg);
void				PrintAppHeader();
void				PrintHelp();





// global definitions
unsigned char verMajor = 1;
unsigned char verMinor = 1;
unsigned char PCIVerMajor = 0;
unsigned char PCIVerMinor = 0;





int main (int optionCount, char **options)
{
	bool PCIFlag, errorlevelFlag = false, optionFlag = false;
	unsigned char returnCode = 0;
	unsigned int index;
	unsigned long devices, registerValue, optionArgument, filterClass = 0xFFFFFFFF, filterVendor = 0xFFFFFFFF;
	unsigned short bus, device, funct, reg;



	// hello world!
	PrintAppHeader();

	// check for presence of PCI BIOS
	PCIFlag = PCICheck();

	if (PCIFlag)
	{
		// show the PCI BIOS version
		printf("PCI BIOS %x.%x detected.\n\n", PCIVerMajor, PCIVerMinor);

		// if there were no options specified, then we do ALL THE THINGS
		if (optionCount > 1)
		{
			// if we get here, the number of options passed were greater than 1
			// this means we need to process them

			for (index = 1; index < optionCount; index = index + 2)
			{


				if (!strcasecmp(options[index], "/c"))
				{
					if (index + 1 < optionCount)
					{
						filterClass = strtol(options[index + 1], 0, 0);
					} else {
						optionFlag = false;
						break;
					}
					optionFlag = true;
				}


				if (!strcasecmp(options[index], "/n"))
				{
					errorlevelFlag = true;
					optionFlag = true;
				}


				if (!strcasecmp(options[index], "/v"))
				{
				if (index + 1 < optionCount)
					{
						filterVendor = strtol(options[index + 1], 0, 0);
					} else {
						optionFlag = false;
						break;
					}
					optionFlag = true;
				}
			}


			if (!optionFlag)
			{
				PrintHelp();
				return(0);
			}
		}

		// show data for all devices
		devices = DeviceDataPrint(filterClass, filterVendor);

	} else {
		printf("PCI BIOS not detected, exiting.\n\n");
		return (1);
	}

	if (errorlevelFlag)
	{
		return(devices);
	} else {
		return (0);
	}
}





unsigned long DeviceDataPrint(unsigned long classMatch, unsigned long vendorMatch)
{
	unsigned short bus, device, funct;
	unsigned long registerValue;
	unsigned long deviceCount = 0;
	unsigned long deviceVendor, deviceID, deviceClass, deviceSubclass, deviceProgIF, deviceRevision;
	unsigned char headerFlag = false;



	for (bus = 0; bus <= 2; bus++)
	{
		for (device = 0; device <= 31; device++)
		{
			for (funct = 0; funct < 8; funct++)
			{
				registerValue = PCIRegisterRead(bus, device, funct, 0);
				if (registerValue != 0xFFFFFFFF)
				{


					// display this device's data
					deviceVendor = (unsigned long) (registerValue & 0x0000FFFF);
					deviceID = registerValue >> 16;

					registerValue = PCIRegisterRead(bus, device, funct, 8);
					deviceClass = registerValue >> 24;
					deviceSubclass = registerValue >> 16 & 0xFF;
					deviceProgIF = registerValue >> 8 & 0xFF;
					deviceRevision = registerValue & 0xFF;


					// decide if this device matters
					if ( (classMatch > 0xFFFF | classMatch == deviceClass) & (vendorMatch > 0xFFFF | vendorMatch == deviceVendor) )
					{
						// print the header, if we haven't already
						if (!headerFlag)
						{
							printf("Bus\tDev\tFunc\tVendor\tID\tClass\tSub\tPIF\tRev\n");
							headerFlag = true;
						}

						// increment the device counter
						deviceCount++;

						printf("%.3X\t%.2X\t%.1X\t%.4lX\t%.4lX\t%.2lX\t%.2lX\t%.2lX\t%.2lX\n", bus, device, funct, deviceVendor, deviceID, deviceClass, deviceSubclass, deviceProgIF, deviceRevision);

						DeviceVendorStringPrint(deviceVendor);

						DeviceDescriptionStringPrint(deviceClass, deviceSubclass, deviceProgIF);

						printf("\n");
					}
				}
			}
		}
	}

	// if there were any devices listed, print the count summary
	if (deviceCount)
	{
		printf("%d", deviceCount);
	} else {
		printf("No");
	}

	printf(" device");

	if (deviceCount > 1)
	{
		printf("s");
	}

	printf(" found.\n\n");

	return deviceCount;
}





void DeviceDescriptionStringPrint(unsigned long deviceClass, unsigned long deviceSubclass, unsigned long deviceProgIF)
{
	unsigned char far *strPtrA;
	unsigned char far *strPtrB;
	unsigned char far *strPtrC;



	strPtrA = "";
	strPtrB = "";
	strPtrC = "";

	switch (deviceClass)
	{
			case 0x00:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Non-VGA-Compatible";
						break;

					case 0x01:
						strPtrB = "VGA-Compatible";
						break;
				}
				strPtrC = " Unclassified Device";
				break;





			case 0x01:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "SCSI Bus";
						break;

					case 0x01:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "ISA mode only";
								break;

							case 0x05:
								strPtrA = "PCI native mode only";
								break;

							case 0x0A:
								strPtrA = "ISA mode/PCI native mode";
								break;

							case 0x0F:
								strPtrA = "PCI native mode/ISA mode";
								break;

							case 0x80:
								strPtrA = "ISA mode only bus mastering";
								break;

							case 0x85:
								strPtrA = "PCI native mode only bus mastering";
								break;

							case 0x8A:
								strPtrA = "ISA mode/PCI native mode bus mastering";
								break;

							case 0x8F:
								strPtrA = "PCI native mode/supports ISA mode bus mastering";
								break;
						}
						strPtrB = " IDE";
						break;

					case 0x02:
						strPtrB = "Floppy Disk";

					case 0x03:
						strPtrB = "IPI Bus";

					case 0x04:
						strPtrB = "RAID";

					case 0x05:
						switch (deviceProgIF)
						{
							case 0x20:
								strPtrA = "Single DMA";
								break;

							case 0x30:
								strPtrA = "Chained DMA";
								break;
						}
						strPtrB = " ATA";
						break;

					case 0x06:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Vendor-specific-specific interface";
								break;

							case 0x01:
								strPtrA = "AHCI 1.0";
								break;

							case 0x02:
								strPtrA = "Serial Storage Bus";
								break;
						}
						strPtrB = " Serial ATA";
						break;

					case 0x07:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "SAS";
								break;

							case 0x01:
								strPtrA = "Serial Storage Bus";
								break;
						}
						strPtrB = " Serial Attached SCSI";
						break;

					case 0x08:
						switch (deviceProgIF)
						{
							case 0x01:
								strPtrA = "NVMHCI";
								break;

							case 0x02:
								strPtrA = "NVM Express";
								break;
						}
						strPtrB = " Non-Volatile Memory";
						break;

					case 0x80:
						strPtrB = "Other";
				}
				strPtrC = " Mass Storage Controller";
				break;





			case 0x02:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Ethernet";
						break;

					case 0x01:
						strPtrB = "Token Ring";
						break;

					case 0x02:
						strPtrB = "FDDI";
						break;

					case 0x03:
						strPtrB = "ATM";
						break;

					case 0x04:
						strPtrB = "ISDN";
						break;

					case 0x05:
						strPtrB = "WorldFip";
						break;

					case 0x06:
						strPtrB = "PICMG 2.14 Multi Computing";
						break;

					case 0x07:
						strPtrB = "Infiniband";
						break;

					case 0x08:
						strPtrB = "Fabric";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Network Controller";
				break;





			case 0x03:
				switch (deviceSubclass)
				{
					case 0x00:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "VGA";
								break;

							case 0x01:
								strPtrA = "8514 Compatible";
								break;
						}
						break;

					case 0x01:
						strPtrB = "XGA";
						break;

					case 0x02:
						strPtrB = "3D (Not VGA-Compatible)";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Display Controller";
				break;





			case 0x04:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Multimedia Video Controller";
						break;

					case 0x01:
						strPtrB = "Multimedia Audio Controller";
						break;

					case 0x02:
						strPtrB = "Computer Telephony Multimedia Controller";
						break;

					case 0x03:
						strPtrB = "Audio Multimedia Controller";
						break;

					case 0x80:
						strPtrB = "Other Multimedia Controller";
						break;
				}
				break;





			case 0x05:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Random Access";
						break;

					case 0x01:
						strPtrB = "Flash";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Memory Controller";
				break;





			case 0x06:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Host";
						break;

					case 0x01:
						strPtrB = "ISA";
						break;

					case 0x02:
						strPtrB = "EISA";
						break;

					case 0x03:
						strPtrB = "MCA";
						break;

					case 0x04:
						strPtrB = "PCI-to-PCI";
						break;

					case 0x05:
						strPtrB = "PCMCIA";
						break;

					case 0x06:
						strPtrB = "NuBus";
						break;

					case 0x07:
						strPtrB = "CardBus";
						break;

					case 0x08:
						strPtrB = "RACEway";
						break;

					case 0x09:
						strPtrB = "PCI-to-PCI";
						break;

					case 0x0A:
						strPtrB = "InfiniBand-to-PCI Host";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Bridge";
				break;





			case 0x07:
				switch (deviceSubclass)
				{
					case 0x00:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "8250-compatible (Generic XT)";
								break;

							case 0x01:
								strPtrA = "16450-Compatible";
								break;

							case 0x02:
								strPtrA = "16550-Compatible";
								break;

							case 0x03:
								strPtrA = "16650-Compatible";
								break;

							case 0x04:
								strPtrA = "16750-Compatible";
								break;

							case 0x05:
								strPtrA = "16850-Compatible";
								break;

							case 0x06:
								strPtrA = "16950-Compatible";
								break;
						}
						strPtrB = " Serial";
						break;

					case 0x01:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Standard Parallel Port";
								break;

							case 0x01:
								strPtrA = "Bi-Directional Parallel Port";
								break;

							case 0x02:
								strPtrA = "ECP 1.X Compliant Parallel Port";
								break;

							case 0x03:
								strPtrA = "IEEE 1284 Controller";
								break;

							case 0xFE:
								strPtrA = "IEEE 1284 Target Device";
								break;
						}
						strPtrB = " Parallel";
						break;

					case 0x02:
						strPtrB = "Multiport Serial";
						break;

					case 0x03:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Generic";
								break;

							case 0x01:
								strPtrA = "Hayes 16450-Compatible Interface";
								break;

							case 0x02:
								strPtrA = "Hayes 16550-Compatible Interface";
								break;

							case 0x03:
								strPtrA = "Hayes 16650-Compatible Interface";
								break;

							case 0x04:
								strPtrA = "Hayes 16750-Compatible Interface";
								break;
						}
						strPtrB = " Modem";
						break;

					case 0x04:
						strPtrB = "IEEE 488.1/2 (GPIB)";
						break;

					case 0x05:
						strPtrB = "Smart Card";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Simple Communication Controller";
				break;





			case 0x08:
				switch (deviceSubclass)
				{
					case 0x00:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Generic 8259-Compatible";
								break;

							case 0x01:
								strPtrA = "ISA-Compatible";
								break;

							case 0x02:
								strPtrA = "EISA-Compatible";
								break;

							case 0x10:
								strPtrA = "I/O APIC";
								break;

							case 0x20:
								strPtrA = "I/O(x) APIC";
								break;
						}
						strPtrB = " PIC";
						break;

					case 0x01:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Generic 8237-Compatible";
								break;

							case 0x01:
								strPtrA = "ISA-Compatible";
								break;

							case 0x02:
								strPtrA = "EISA-Compatible";
								break;
						}
						strPtrB = " DMA Controller";
						break;

					case 0x02:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Generic 8254-Compatible";
								break;

							case 0x01:
								strPtrA = "ISA-Compatible";
								break;

							case 0x02:
								strPtrA = "EISA-Compatible";
								break;

							case 0x03:
								strPtrA = "HPET";
								break;
						}
						strPtrB = " Timer";
						break;

					case 0x03:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Generic";
								break;

							case 0x01:
								strPtrA = "ISA-Compatible";
								break;
						}
						strPtrB = " RTC Controller";
						break;

					case 0x04:
						strPtrB = "PCI Hot-Plug Controller";
						break;

					case 0x05:
						strPtrB = "SD Host controller";
						break;

					case 0x06:
						strPtrB = "IOMMU";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Base System Peripheral";
				break;





			case 0x09:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Keyboard";
						break;

					case 0x01:
						strPtrB = "Digitizer Pen";
						break;

					case 0x02:
						strPtrB = "Mouse";
						break;

					case 0x03:
						strPtrB = "Scanner";
						break;

					case 0x04:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Generic";
								break;

							case 0x10:
								strPtrA = "Extended";
								break;
						}
						strPtrB = " Gameport";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Input Device Controller";
				break;





			case 0x0A:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Generic";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Docking Station";
				break;





			case 0x0B:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "386";
						break;

					case 0x01:
						strPtrB = "486";
						break;

					case 0x02:
						strPtrB = "Pentium";
						break;

					case 0x03:
						strPtrB = "Pentium Pro";
						break;

					case 0x10:
						strPtrB = "Alpha";
						break;

					case 0x20:
						strPtrB = "PowerPC";
						break;

					case 0x30:
						strPtrB = "MIPS";
						break;

					case 0x40:
						strPtrB = "Co";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Processor";
				break;





			case 0x0C:
				switch (deviceSubclass)
				{
					case 0x00:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "Generic";
								break;

							case 0x10:
								strPtrA = "OHCI";
								break;
						}
						strPtrB = " FireWire (IEEE 1394) Controller";
						break;

					case 0x01:
						strPtrB = "ACCESS Bus";
						break;

					case 0x02:
						strPtrB = "SSA";
						break;

					case 0x03:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "UHCI";
								break;

							case 0x10:
								strPtrA = "OHCI";
								break;

							case 0x20:
								strPtrA = "EHCI (USB2)";
								break;

							case 0x30:
								strPtrA = "XHCI (USB3)";
								break;

							case 0x80:
								strPtrA = "Unspecified";
								break;

							case 0xFE:
								strPtrA = "USB Device attached to";
								break;
						}
						strPtrB = " USB";
						break;

					case 0x04:
						strPtrB = "Fibre Channel";
						break;

					case 0x05:
						strPtrB = "SMBus";
						break;

					case 0x06:
						strPtrB = "InfiniBand";
						break;

					case 0x07:
						switch (deviceProgIF)
						{
							case 0x00:
								strPtrA = "SMIC";
								break;

							case 0x01:
								strPtrA = "Keyboard Controller Style";
								break;

							case 0x02:
								strPtrA = "Block Transfer";
								break;
						}
						strPtrB = " IPMI Interface";
						break;

					case 0x08:
						strPtrB = "SERCOS Interface (IEC 61491)";
						break;

					case 0x09:
						strPtrB = "CANbus";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Serial Bus Controller";
				break;




			case 0x0D:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "iRDA Compatible";
						break;

					case 0x01:
						strPtrB = "Consumer IR";
						break;

					case 0x10:
						strPtrB = "RF";
						break;

					case 0x11:
						strPtrB = "Bluetooth";
						break;

					case 0x12:
						strPtrB = "Broadband";
						break;

					case 0x20:
						strPtrB = "Ethernet (802.1a)";
						break;

					case 0x21:
						strPtrB = "Ethernet (802.1b)";
						break;
				}
				strPtrC = " Wireless Controller";
				break;





			case 0x0E:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "I20";
						break;
				}
				strPtrC = " Intelligent Controller";
				break;





			case 0x0F:
				switch (deviceSubclass)
				{
					case 0x01:
						strPtrB = "TV";
						break;

					case 0x02:
						strPtrB = "Audio";
						break;

					case 0x03:
						strPtrB = "Voice";
						break;

					case 0x04:
						strPtrB = "Data";
						break;
				}
				strPtrC = " Satellite Communication Controller";
				break;





			case 0x10:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "Network and Computing";
						break;

					case 0x10:
						strPtrB = "Entertainment";
						break;

					case 0x80:
						strPtrB = "Other";
						break;
				}
				strPtrC = " Encryption/Decryption Controller";
				break;





			case 0x11:
				switch (deviceSubclass)
				{
					case 0x00:
						strPtrB = "DPIO Module";
						break;

					case 0x01:
						strPtrB = "Performance Counter";
						break;

					case 0x10:
						strPtrB = "Communication Synchronizer";
						break;

					case 0x20:
						strPtrB = "Signal Processing Management Controller";
						break;

					case 0x80:
						strPtrB = "Other Signal Processing Controller";
						break;
				}
				break;





			case 0x12:
				strPtrC = "Processing Accelerator";
				break;





			case 0x13:
				strPtrC = "Non-Essential Instrumentation";
				break;





			case 0x40:
				strPtrC = "Co-Processor";
				break;





			default:
				strPtrC = "Reserved / Unassigned";
				break;
	}
	printf("%Ws%Ws%Ws\n\n", strPtrA, strPtrB, strPtrC);
}





void DeviceVendorStringPrint(unsigned long deviceVendor)
{
	unsigned char *strPtr;
	strPtr = "Unknown";

	switch (deviceVendor)
	{
		case 0x0001:
			strPtr = "SafeNet";
			break;

		case 0x0010:
			strPtr = "Allied Telesis, Inc";
			break;

		case 0x0014:
			strPtr = "Loongson Technology LLC";
			break;

		case 0x001c:
			strPtr = "PEAK-System Technik GmbH";
			break;

		case 0x003d:
			strPtr = "Lockheed Martin-Marietta Corp";
			break;

		case 0x0059:
			strPtr = "Tiger Jet Network Inc.";
			break;

		case 0x0070:
			strPtr = "Hauppauge computer works Inc.";
			break;

		case 0x0071:
			strPtr = "Nebula Electronics Ltd.";
			break;

		case 0x0095:
			strPtr = "Silicon Image, Inc.";
			break;

		case 0x00a7:
			strPtr = "Teles AG";
			break;

		case 0x0100:
			strPtr = "nCipher Security";
			break;

		case 0x0123:
			strPtr = "General Dynamics";
			break;

		case 0x0128:
			strPtr = "Dell";
			break;

		case 0x018a:
			strPtr = "LevelOne";
			break;

		case 0x01de:
			strPtr = "Oxide Computer Company";
			break;

		case 0x0200:
			strPtr = "Dell";
			break;

		case 0x021b:
			strPtr = "Compaq Computer Corporation";
			break;

		case 0x0270:
			strPtr = "Hauppauge computer works Inc.";
			break;

		case 0x0291:
			strPtr = "Davicom Semiconductor, Inc.";
			break;

		case 0x02ac:
			strPtr = "SpeedStream";
			break;

		case 0x02e0:
			strPtr = "XFX Pine Group Inc.";
			break;

		case 0x0303:
			strPtr = "Hewlett-Packard Company";
			break;

		case 0x0308:
			strPtr = "ZyXEL Communications Corporation";
			break;

		case 0x0315:
			strPtr = "SK-Electronics Co., Ltd.";
			break;

		case 0x0357:
			strPtr = "TTTech Computertechnik AG";
			break;

		case 0x0432:
			strPtr = "SCM Microsystems, Inc.";
			break;

		case 0x0497:
			strPtr = "Dell Inc.";
			break;

		case 0x0675:
			strPtr = "Dynalink";
			break;

		case 0x0721:
			strPtr = "Sapphire, Inc.";
			break;

		case 0x0731:
			strPtr = "Jingjia Microelectronics Co Ltd";
			break;

		case 0x0777:
			strPtr = "Ubiquiti Networks, Inc.";
			break;

		case 0x0795:
			strPtr = "Wired Inc.";
			break;

		case 0x07d1:
			strPtr = "D-Link System Inc";
			break;

		case 0x0824:
			strPtr = "T1042 [Freescale]";
			break;

		case 0x0925:
			strPtr = "VIA Technologies, Inc.";
			break;

		case 0x0a89:
			strPtr = "BREA Technologies Inc";
			break;

		case 0x0b0b:
			strPtr = "Rhino Equipment Corp.";
			break;

		case 0x0e11:
			strPtr = "Compaq Computer Corporation";
			break;

		case 0x0e55:
			strPtr = "HaSoTec GmbH";
			break;

		case 0x0eac:
			strPtr = "SHF Communication Technologies AG";
			break;

		case 0x0f62:
			strPtr = "Acrox Technologies Co., Ltd.";
			break;

		case 0x1000:
			strPtr = "Broadcom / LSI";
			break;

		case 0x1001:
			strPtr = "Kolter Electronic";
			break;

		case 0x1002:
			strPtr = "Advanced Micro Devices, Inc. [AMD/ATI]";
			break;

		case 0x1003:
			strPtr = "ULSI Systems";
			break;

		case 0x1004:
			strPtr = "VLSI Technology Inc";
			break;

		case 0x1005:
			strPtr = "Avance Logic Inc. [ALI]";
			break;

		case 0x1006:
			strPtr = "Reply Group";
			break;

		case 0x1007:
			strPtr = "NetFrame Systems Inc";
			break;

		case 0x1008:
			strPtr = "Epson";
			break;

		case 0x100a:
			strPtr = "Phoenix Technologies";
			break;

		case 0x100b:
			strPtr = "National Semiconductor Corporation";
			break;

		case 0x100c:
			strPtr = "Tseng Labs Inc";
			break;

		case 0x100d:
			strPtr = "AST Research Inc";
			break;

		case 0x100e:
			strPtr = "Weitek";
			break;

		case 0x1010:
			strPtr = "Video Logic, Ltd.";
			break;

		case 0x1011:
			strPtr = "Digital Equipment Corporation";
			break;

		case 0x1012:
			strPtr = "Micronics Computers Inc";
			break;

		case 0x1013:
			strPtr = "Cirrus Logic";
			break;

		case 0x1014:
			strPtr = "IBM";
			break;

		case 0x1015:
			strPtr = "LSI Logic Corp of Canada";
			break;

		case 0x1016:
			strPtr = "ICL Personal Systems";
			break;

		case 0x1017:
			strPtr = "SPEA Software AG";
			break;

		case 0x1018:
			strPtr = "Unisys Systems";
			break;

		case 0x1019:
			strPtr = "Elitegroup Computer Systems";
			break;

		case 0x101a:
			strPtr = "AT&T GIS";
			break;

		case 0x101b:
			strPtr = "Vitesse Semiconductor";
			break;

		case 0x101c:
			strPtr = "Western Digital";
			break;

		case 0x101d:
			strPtr = "Maxim Integrated Products";
			break;

		case 0x101e:
			strPtr = "American Megatrends Inc.";
			break;

		case 0x101f:
			strPtr = "PictureTel";
			break;

		case 0x1020:
			strPtr = "Hitachi Computer Products";
			break;

		case 0x1021:
			strPtr = "OKI Electric Industry Co. Ltd.";
			break;

		case 0x1022:
			strPtr = "Advanced Micro Devices, Inc. [AMD]";
			break;

		case 0x1023:
			strPtr = "Trident Microsystems";
			break;

		case 0x1024:
			strPtr = "Zenith Data Systems";
			break;

		case 0x1025:
			strPtr = "Acer Incorporated [ALI]";
			break;

		case 0x1028:
			strPtr = "Dell";
			break;

		case 0x1029:
			strPtr = "Siemens Nixdorf IS";
			break;

		case 0x102a:
			strPtr = "LSI Logic";
			break;

		case 0x102b:
			strPtr = "Matrox Electronics Systems Ltd.";
			break;

		case 0x102c:
			strPtr = "Chips and Technologies";
			break;

		case 0x102d:
			strPtr = "Wyse Technology Inc.";
			break;

		case 0x102e:
			strPtr = "Olivetti Advanced Technology";
			break;

		case 0x102f:
			strPtr = "Toshiba America";
			break;

		case 0x1030:
			strPtr = "TMC Research";
			break;

		case 0x1031:
			strPtr = "Miro Computer Products AG";
			break;

		case 0x1032:
			strPtr = "Compaq";
			break;

		case 0x1033:
			strPtr = "NEC Corporation";
			break;

		case 0x1034:
			strPtr = "Framatome Connectors USA Inc.";
			break;

		case 0x1035:
			strPtr = "Comp. & Comm. Research Lab";
			break;

		case 0x1036:
			strPtr = "Future Domain Corp.";
			break;

		case 0x1037:
			strPtr = "Hitachi Micro Systems";
			break;

		case 0x1038:
			strPtr = "AMP, Inc";
			break;

		case 0x1039:
			strPtr = "Silicon Integrated Systems [SiS]";
			break;

		case 0x103a:
			strPtr = "Seiko Epson Corporation";
			break;

		case 0x103b:
			strPtr = "Tatung Corp. Of America";
			break;

		case 0x103c:
			strPtr = "Hewlett-Packard Company";
			break;

		case 0x103e:
			strPtr = "Solliday Engineering";
			break;

		case 0x103f:
			strPtr = "Synopsys/Logic Modeling Group";
			break;

		case 0x1040:
			strPtr = "Accelgraphics Inc.";
			break;

		case 0x1041:
			strPtr = "Computrend";
			break;

		case 0x1042:
			strPtr = "Micron";
			break;

		case 0x1043:
			strPtr = "ASUSTeK Computer Inc.";
			break;

		case 0x1044:
			strPtr = "Adaptec";
			break;

		case 0x1045:
			strPtr = "OPTi Inc.";
			break;

		case 0x1046:
			strPtr = "IPC Corporation, Ltd.";
			break;

		case 0x1047:
			strPtr = "Genoa Systems Corp";
			break;

		case 0x1048:
			strPtr = "Elsa AG";
			break;

		case 0x1049:
			strPtr = "Fountain Technologies, Inc.";
			break;

		case 0x104a:
			strPtr = "STMicroelectronics";
			break;

		case 0x104b:
			strPtr = "BusLogic";
			break;

		case 0x104c:
			strPtr = "Texas Instruments";
			break;

		case 0x104d:
			strPtr = "Sony Corporation";
			break;

		case 0x104e:
			strPtr = "Oak Technology, Inc";
			break;

		case 0x104f:
			strPtr = "Co-time Computer Ltd";
			break;

		case 0x1050:
			strPtr = "Winbond Electronics Corp";
			break;

		case 0x1051:
			strPtr = "Anigma, Inc.";
			break;

		case 0x1052:
			strPtr = "?Young Micro Systems";
			break;

		case 0x1053:
			strPtr = "Young Micro Systems";
			break;

		case 0x1054:
			strPtr = "Hitachi, Ltd";
			break;

		case 0x1055:
			strPtr = "Microchip Technology / SMSC";
			break;

		case 0x1056:
			strPtr = "ICL";
			break;

		case 0x1057:
			strPtr = "Motorola";
			break;

		case 0x1058:
			strPtr = "Electronics & Telecommunications RSH";
			break;

		case 0x1059:
			strPtr = "Kontron";
			break;

		case 0x105a:
			strPtr = "Promise Technology, Inc.";
			break;

		case 0x105b:
			strPtr = "Foxconn International, Inc.";
			break;

		case 0x105c:
			strPtr = "Wipro Infotech Limited";
			break;

		case 0x105d:
			strPtr = "Number 9 Computer Company";
			break;

		case 0x105e:
			strPtr = "Vtech Computers Ltd";
			break;

		case 0x105f:
			strPtr = "Infotronic America Inc";
			break;

		case 0x1060:
			strPtr = "United Microelectronics [UMC]";
			break;

		case 0x1061:
			strPtr = "I.I.T.";
			break;

		case 0x1062:
			strPtr = "Maspar Computer Corp";
			break;

		case 0x1063:
			strPtr = "Ocean Office Automation";
			break;

		case 0x1064:
			strPtr = "Alcatel";
			break;

		case 0x1065:
			strPtr = "Texas Microsystems";
			break;

		case 0x1066:
			strPtr = "PicoPower Technology";
			break;

		case 0x1067:
			strPtr = "Mitsubishi Electric";
			break;

		case 0x1068:
			strPtr = "Diversified Technology";
			break;

		case 0x1069:
			strPtr = "Mylex Corporation";
			break;

		case 0x106a:
			strPtr = "Aten Research Inc";
			break;

		case 0x106b:
			strPtr = "Apple Inc.";
			break;

		case 0x106c:
			strPtr = "Hynix Semiconductor";
			break;

		case 0x106d:
			strPtr = "Sequent Computer Systems";
			break;

		case 0x106e:
			strPtr = "DFI, Inc";
			break;

		case 0x106f:
			strPtr = "City Gate Development Ltd";
			break;

		case 0x1070:
			strPtr = "Daewoo Telecom Ltd";
			break;

		case 0x1071:
			strPtr = "Mitac";
			break;

		case 0x1072:
			strPtr = "GIT Co Ltd";
			break;

		case 0x1073:
			strPtr = "Yamaha Corporation";
			break;

		case 0x1074:
			strPtr = "NexGen Microsystems";
			break;

		case 0x1075:
			strPtr = "Advanced Integrations Research";
			break;

		case 0x1076:
			strPtr = "Chaintech Computer Co. Ltd";
			break;

		case 0x1077:
			strPtr = "QLogic Corp.";
			break;

		case 0x1078:
			strPtr = "Cyrix Corporation";
			break;

		case 0x1079:
			strPtr = "I-Bus";
			break;

		case 0x107a:
			strPtr = "NetWorth";
			break;

		case 0x107b:
			strPtr = "Gateway, Inc.";
			break;

		case 0x107c:
			strPtr = "LG Electronics [Lucky Goldstar Co. Ltd]";
			break;

		case 0x107d:
			strPtr = "LeadTek Research Inc.";
			break;

		case 0x107e:
			strPtr = "Interphase Corporation";
			break;

		case 0x107f:
			strPtr = "Data Technology Corporation";
			break;

		case 0x1080:
			strPtr = "Contaq Microsystems";
			break;

		case 0x1081:
			strPtr = "Supermac Technology";
			break;

		case 0x1082:
			strPtr = "EFA Corporation of America";
			break;

		case 0x1083:
			strPtr = "Forex Computer Corporation";
			break;

		case 0x1084:
			strPtr = "Parador";
			break;

		case 0x1086:
			strPtr = "J. Bond Computer Systems";
			break;

		case 0x1087:
			strPtr = "Cache Computer";
			break;

		case 0x1088:
			strPtr = "Microcomputer Systems";
			break;

		case 0x1089:
			strPtr = "Data General Corporation";
			break;

		case 0x108a:
			strPtr = "SBS Technologies";
			break;

		case 0x108c:
			strPtr = "Oakleigh Systems Inc.";
			break;

		case 0x108d:
			strPtr = "Olicom";
			break;

		case 0x108e:
			strPtr = "Oracle/SUN";
			break;

		case 0x108f:
			strPtr = "Systemsoft";
			break;

		case 0x1090:
			strPtr = "Compro Computer Services, Inc.";
			break;

		case 0x1091:
			strPtr = "Intergraph Corporation";
			break;

		case 0x1092:
			strPtr = "Diamond Multimedia Systems";
			break;

		case 0x1093:
			strPtr = "National Instruments";
			break;

		case 0x1094:
			strPtr = "First International Computers [FIC]";
			break;

		case 0x1095:
			strPtr = "Silicon Image, Inc.";
			break;

		case 0x1096:
			strPtr = "Alacron";
			break;

		case 0x1097:
			strPtr = "Appian Technology";
			break;

		case 0x1098:
			strPtr = "Quantum Designs";
			break;

		case 0x1099:
			strPtr = "Samsung Electronics Co., Ltd";
			break;

		case 0x109a:
			strPtr = "Packard Bell";
			break;

		case 0x109b:
			strPtr = "Gemlight Computer Ltd.";
			break;

		case 0x109c:
			strPtr = "Megachips Corporation";
			break;

		case 0x109d:
			strPtr = "Zida Technologies Ltd.";
			break;

		case 0x109e:
			strPtr = "Brooktree Corporation";
			break;

		case 0x109f:
			strPtr = "Trigem Computer Inc.";
			break;

		case 0x10a0:
			strPtr = "Meidensha Corporation";
			break;

		case 0x10a1:
			strPtr = "Juko Electronics Ind. Co. Ltd";
			break;

		case 0x10a2:
			strPtr = "Quantum Corporation";
			break;

		case 0x10a3:
			strPtr = "Everex Systems Inc";
			break;

		case 0x10a4:
			strPtr = "Globe Manufacturing Sales";
			break;

		case 0x10a5:
			strPtr = "Smart Link Ltd.";
			break;

		case 0x10a6:
			strPtr = "Informtech Industrial Ltd.";
			break;

		case 0x10a7:
			strPtr = "Benchmarq Microelectronics";
			break;

		case 0x10a8:
			strPtr = "Sierra Semiconductor";
			break;

		case 0x10a9:
			strPtr = "Silicon Graphics Intl. Corp.";
			break;

		case 0x10aa:
			strPtr = "ACC Microelectronics";
			break;

		case 0x10ab:
			strPtr = "Digicom";
			break;

		case 0x10ac:
			strPtr = "Honeywell IAC";
			break;

		case 0x10ad:
			strPtr = "Symphony Labs";
			break;

		case 0x10ae:
			strPtr = "Cornerstone Technology";
			break;

		case 0x10af:
			strPtr = "Micro Computer Systems Inc";
			break;

		case 0x10b0:
			strPtr = "CardExpert Technology";
			break;

		case 0x10b1:
			strPtr = "Cabletron Systems Inc";
			break;

		case 0x10b2:
			strPtr = "Raytheon Company";
			break;

		case 0x10b3:
			strPtr = "Databook Inc";
			break;

		case 0x10b4:
			strPtr = "STB Systems Inc";
			break;

		case 0x10b5:
			strPtr = "PLX Technology, Inc.";
			break;

		case 0x10b6:
			strPtr = "Madge Networks";
			break;

		case 0x10b7:
			strPtr = "3Com Corporation";
			break;

		case 0x10b8:
			strPtr = "Standard Microsystems Corp [SMC]";
			break;

		case 0x10b9:
			strPtr = "ULi Electronics Inc.";
			break;

		case 0x10ba:
			strPtr = "Mitsubishi Electric Corp.";
			break;

		case 0x10bb:
			strPtr = "Dapha Electronics Corporation";
			break;

		case 0x10bc:
			strPtr = "Advanced Logic Research";
			break;

		case 0x10bd:
			strPtr = "Surecom Technology";
			break;

		case 0x10be:
			strPtr = "Tseng Labs International Co.";
			break;

		case 0x10bf:
			strPtr = "Most Inc";
			break;

		case 0x10c0:
			strPtr = "Boca Research Inc.";
			break;

		case 0x10c1:
			strPtr = "ICM Co., Ltd.";
			break;

		case 0x10c2:
			strPtr = "Auspex Systems Inc.";
			break;

		case 0x10c3:
			strPtr = "Samsung Semiconductors, Inc.";
			break;

		case 0x10c4:
			strPtr = "Award Software International Inc.";
			break;

		case 0x10c5:
			strPtr = "Xerox Corporation";
			break;

		case 0x10c6:
			strPtr = "Rambus Inc.";
			break;

		case 0x10c7:
			strPtr = "Media Vision";
			break;

		case 0x10c8:
			strPtr = "Neomagic Corporation";
			break;

		case 0x10c9:
			strPtr = "Dataexpert Corporation";
			break;

		case 0x10ca:
			strPtr = "Fujitsu Microelectr., Inc.";
			break;

		case 0x10cb:
			strPtr = "Omron Corporation";
			break;

		case 0x10cc:
			strPtr = "Mai Logic Incorporated";
			break;

		case 0x10cd:
			strPtr = "Advanced System Products, Inc";
			break;

		case 0x10ce:
			strPtr = "Radius";
			break;

		case 0x10cf:
			strPtr = "Fujitsu Limited.";
			break;

		case 0x10d1:
			strPtr = "FuturePlus Systems Corp.";
			break;

		case 0x10d2:
			strPtr = "Molex Incorporated";
			break;

		case 0x10d3:
			strPtr = "Jabil Circuit Inc";
			break;

		case 0x10d4:
			strPtr = "Hualon Microelectronics";
			break;

		case 0x10d5:
			strPtr = "Autologic Inc.";
			break;

		case 0x10d6:
			strPtr = "Cetia";
			break;

		case 0x10d7:
			strPtr = "BCM Advanced Research";
			break;

		case 0x10d8:
			strPtr = "Advanced Peripherals Labs";
			break;

		case 0x10d9:
			strPtr = "Macronix, Inc. [MXIC]";
			break;

		case 0x10da:
			strPtr = "Compaq IPG-Austin";
			break;

		case 0x10db:
			strPtr = "Rohm LSI Systems, Inc.";
			break;

		case 0x10dc:
			strPtr = "CERN/ECP/EDU";
			break;

		case 0x10dd:
			strPtr = "Evans & Sutherland";
			break;

		case 0x10de:
			strPtr = "NVIDIA Corporation";
			break;

		case 0x10df:
			strPtr = "Emulex Corporation";
			break;

		case 0x10e0:
			strPtr = "Integrated Micro Solutions Inc.";
			break;

		case 0x10e1:
			strPtr = "Tekram Technology Co.,Ltd.";
			break;

		case 0x10e2:
			strPtr = "Aptix Corporation";
			break;

		case 0x10e3:
			strPtr = "Tundra Semiconductor Corp.";
			break;

		case 0x10e4:
			strPtr = "Tandem Computers";
			break;

		case 0x10e5:
			strPtr = "Micro Industries Corporation";
			break;

		case 0x10e6:
			strPtr = "Gainbery Computer Products Inc.";
			break;

		case 0x10e7:
			strPtr = "Vadem";
			break;

		case 0x10e8:
			strPtr = "Applied Micro Circuits Corp.";
			break;

		case 0x10e9:
			strPtr = "Alps Electric Co., Ltd.";
			break;

		case 0x10ea:
			strPtr = "Integraphics";
			break;

		case 0x10eb:
			strPtr = "Artists Graphics";
			break;

		case 0x10ec:
			strPtr = "Realtek Semiconductor Co., Ltd.";
			break;

		case 0x10ed:
			strPtr = "Ascii Corporation";
			break;

		case 0x10ee:
			strPtr = "Xilinx Corporation";
			break;

		case 0x10ef:
			strPtr = "Racore Computer Products, Inc.";
			break;

		case 0x10f0:
			strPtr = "Peritek Corporation";
			break;

		case 0x10f1:
			strPtr = "Tyan Computer";
			break;

		case 0x10f2:
			strPtr = "Achme Computer, Inc.";
			break;

		case 0x10f3:
			strPtr = "Alaris, Inc.";
			break;

		case 0x10f4:
			strPtr = "S-MOS Systems, Inc.";
			break;

		case 0x10f5:
			strPtr = "NKK Corporation";
			break;

		case 0x10f6:
			strPtr = "Creative Electronic Systems SA";
			break;

		case 0x10f7:
			strPtr = "Matsushita Electric Industrial Co., Ltd.";
			break;

		case 0x10f8:
			strPtr = "Altos India Ltd";
			break;

		case 0x10f9:
			strPtr = "PC Direct";
			break;

		case 0x10fa:
			strPtr = "Truevision";
			break;

		case 0x10fb:
			strPtr = "Thesys Gesellschaft fuer Mikroelektronik mbH";
			break;

		case 0x10fc:
			strPtr = "I-O Data Device, Inc.";
			break;

		case 0x10fd:
			strPtr = "Soyo Computer, Inc";
			break;

		case 0x10fe:
			strPtr = "Fast Multimedia AG";
			break;

		case 0x10ff:
			strPtr = "NCube";
			break;

		case 0x1100:
			strPtr = "Jazz Multimedia";
			break;

		case 0x1101:
			strPtr = "Initio Corporation";
			break;

		case 0x1102:
			strPtr = "Creative Labs";
			break;

		case 0x1103:
			strPtr = "HighPoint Technologies, Inc.";
			break;

		case 0x1104:
			strPtr = "RasterOps Corp.";
			break;

		case 0x1105:
			strPtr = "Sigma Designs, Inc.";
			break;

		case 0x1106:
			strPtr = "VIA Technologies, Inc.";
			break;

		case 0x1107:
			strPtr = "Stratus Computers";
			break;

		case 0x1108:
			strPtr = "Proteon, Inc.";
			break;

		case 0x1109:
			strPtr = "Cogent Data Technologies, Inc.";
			break;

		case 0x110a:
			strPtr = "Siemens AG";
			break;

		case 0x110b:
			strPtr = "Chromatic Research Inc.";
			break;

		case 0x110c:
			strPtr = "Mini-Max Technology, Inc.";
			break;

		case 0x110d:
			strPtr = "Znyx Advanced Systems";
			break;

		case 0x110e:
			strPtr = "CPU Technology";
			break;

		case 0x110f:
			strPtr = "Ross Technology";
			break;

		case 0x1110:
			strPtr = "Powerhouse Systems";
			break;

		case 0x1111:
			strPtr = "Santa Cruz Operation";
			break;

		case 0x1112:
			strPtr = "Osicom Technologies Inc";
			break;

		case 0x1113:
			strPtr = "Accton Technology Corporation";
			break;

		case 0x1114:
			strPtr = "Atmel Corporation";
			break;

		case 0x1115:
			strPtr = "3D Labs";
			break;

		case 0x1116:
			strPtr = "Data Translation";
			break;

		case 0x1117:
			strPtr = "Datacube, Inc";
			break;

		case 0x1118:
			strPtr = "Berg Electronics";
			break;

		case 0x1119:
			strPtr = "ICP Vortex Computersysteme GmbH";
			break;

		case 0x111a:
			strPtr = "Efficient Networks, Inc";
			break;

		case 0x111b:
			strPtr = "Teledyne Electronic Systems";
			break;

		case 0x111c:
			strPtr = "Tricord Systems Inc.";
			break;

		case 0x111d:
			strPtr = "Microsemi / PMC / IDT";
			break;

		case 0x111e:
			strPtr = "Eldec";
			break;

		case 0x111f:
			strPtr = "Precision Digital Images";
			break;

		case 0x1120:
			strPtr = "Dell EMC";
			break;

		case 0x1121:
			strPtr = "Zilog";
			break;

		case 0x1122:
			strPtr = "Multi-tech Systems, Inc.";
			break;

		case 0x1123:
			strPtr = "Excellent Design, Inc.";
			break;

		case 0x1124:
			strPtr = "Leutron Vision AG";
			break;

		case 0x1125:
			strPtr = "Eurocore";
			break;

		case 0x1126:
			strPtr = "Vigra";
			break;

		case 0x1127:
			strPtr = "FORE Systems Inc";
			break;

		case 0x1129:
			strPtr = "Firmworks";
			break;

		case 0x112a:
			strPtr = "Hermes Electronics Company, Ltd.";
			break;

		case 0x112b:
			strPtr = "Heidelberger Druckmaschinen AGHeidelberger Druckmaschinen AG";
			break;

		case 0x112c:
			strPtr = "Zenith Data Systems";
			break;

		case 0x112d:
			strPtr = "Ravicad";
			break;

		case 0x112e:
			strPtr = "Infomedia Microelectronics Inc.";
			break;

		case 0x112f:
			strPtr = "Dalsa Inc.";
			break;

		case 0x1130:
			strPtr = "Computervision";
			break;

		case 0x1131:
			strPtr = "Philips Semiconductors";
			break;

		case 0x1132:
			strPtr = "Mitel Corp.";
			break;

		case 0x1133:
			strPtr = "Dialogic Corporation";
			break;

		case 0x1134:
			strPtr = "Mercury Computer Systems";
			break;

		case 0x1135:
			strPtr = "FUJIFILM Business Innovation Corp.";
			break;

		case 0x1136:
			strPtr = "Momentum Data Systems";
			break;

		case 0x1137:
			strPtr = "Cisco Systems Inc";
			break;

		case 0x1138:
			strPtr = "Ziatech Corporation";
			break;

		case 0x1139:
			strPtr = "Dynamic Pictures, Inc";
			break;

		case 0x113a:
			strPtr = "FWB Inc";
			break;

		case 0x113b:
			strPtr = "Network Computing Devices";
			break;

		case 0x113c:
			strPtr = "Cyclone Microsystems, Inc.";
			break;

		case 0x113d:
			strPtr = "Leading Edge Products Inc";
			break;

		case 0x113e:
			strPtr = "Sanyo Electric Co - Computer Engineering Dept";
			break;

		case 0x113f:
			strPtr = "Equinox Systems, Inc.";
			break;

		case 0x1140:
			strPtr = "Intervoice Inc";
			break;

		case 0x1141:
			strPtr = "Crest Microsystem Inc";
			break;

		case 0x1142:
			strPtr = "Alliance Semiconductor Corporation";
			break;

		case 0x1143:
			strPtr = "NetPower, Inc";
			break;

		case 0x1144:
			strPtr = "Cincinnati Milacron";
			break;

		case 0x1145:
			strPtr = "Workbit Corporation";
			break;

		case 0x1146:
			strPtr = "Force Computers";
			break;

		case 0x1147:
			strPtr = "Interface Corp";
			break;

		case 0x1148:
			strPtr = "SysKonnect";
			break;

		case 0x1149:
			strPtr = "Win System Corporation";
			break;

		case 0x114a:
			strPtr = "VMIC";
			break;

		case 0x114b:
			strPtr = "Canopus Co., Ltd";
			break;

		case 0x114c:
			strPtr = "Annabooks";
			break;

		case 0x114d:
			strPtr = "IC Corporation";
			break;

		case 0x114e:
			strPtr = "Nikon Systems Inc";
			break;

		case 0x114f:
			strPtr = "Digi International";
			break;

		case 0x1150:
			strPtr = "Thinking Machines Corp";
			break;

		case 0x1151:
			strPtr = "JAE Electronics Inc.";
			break;

		case 0x1152:
			strPtr = "Megatek";
			break;

		case 0x1153:
			strPtr = "Land Win Electronic Corp";
			break;

		case 0x1154:
			strPtr = "Melco Inc";
			break;

		case 0x1155:
			strPtr = "Pine Technology Ltd";
			break;

		case 0x1156:
			strPtr = "Periscope Engineering";
			break;

		case 0x1157:
			strPtr = "Avsys Corporation";
			break;

		case 0x1158:
			strPtr = "Voarx R & D Inc";
			break;

		case 0x1159:
			strPtr = "MuTech Corporation";
			break;

		case 0x115a:
			strPtr = "Harlequin Ltd";
			break;

		case 0x115b:
			strPtr = "Parallax Graphics";
			break;

		case 0x115c:
			strPtr = "Photron Ltd.";
			break;

		case 0x115d:
			strPtr = "Xircom";
			break;

		case 0x115e:
			strPtr = "Peer Protocols Inc";
			break;

		case 0x115f:
			strPtr = "Maxtor Corporation";
			break;

		case 0x1160:
			strPtr = "Megasoft Inc";
			break;

		case 0x1161:
			strPtr = "PFU Limited";
			break;

		case 0x1162:
			strPtr = "OA Laboratory Co Ltd";
			break;

		case 0x1163:
			strPtr = "Rendition";
			break;

		case 0x1164:
			strPtr = "Advanced Peripherals Technologies";
			break;

		case 0x1165:
			strPtr = "Imagraph Corporation";
			break;

		case 0x1166:
			strPtr = "Broadcom";
			break;

		case 0x1167:
			strPtr = "Mutoh Industries Inc";
			break;

		case 0x1168:
			strPtr = "Thine Electronics Inc";
			break;

		case 0x1169:
			strPtr = "Centre for Development of Advanced Computing";
			break;

		case 0x116a:
			strPtr = "Luminex Software, Inc.";
			break;

		case 0x116b:
			strPtr = "Connectware Inc";
			break;

		case 0x116c:
			strPtr = "Intelligent Resources Integrated Systems";
			break;

		case 0x116d:
			strPtr = "Martin-Marietta";
			break;

		case 0x116e:
			strPtr = "Electronics for Imaging";
			break;

		case 0x116f:
			strPtr = "Workstation Technology";
			break;

		case 0x1170:
			strPtr = "Inventec Corporation";
			break;

		case 0x1171:
			strPtr = "Loughborough Sound Images Plc";
			break;

		case 0x1172:
			strPtr = "Altera Corporation";
			break;

		case 0x1173:
			strPtr = "Adobe Systems, Inc";
			break;

		case 0x1174:
			strPtr = "Bridgeport Machines";
			break;

		case 0x1175:
			strPtr = "Mitron Computer Inc.";
			break;

		case 0x1176:
			strPtr = "SBE Incorporated";
			break;

		case 0x1177:
			strPtr = "Silicon Engineering";
			break;

		case 0x1178:
			strPtr = "Alfa, Inc.";
			break;

		case 0x1179:
			strPtr = "Toshiba Corporation";
			break;

		case 0x117a:
			strPtr = "A-Trend Technology";
			break;

		case 0x117b:
			strPtr = "L G Electronics, Inc.";
			break;

		case 0x117c:
			strPtr = "ATTO Technology, Inc.";
			break;

		case 0x117d:
			strPtr = "Becton & Dickinson";
			break;

		case 0x117e:
			strPtr = "T/R Systems";
			break;

		case 0x117f:
			strPtr = "Integrated Circuit Systems";
			break;

		case 0x1180:
			strPtr = "Ricoh Co Ltd";
			break;

		case 0x1181:
			strPtr = "Telmatics International";
			break;

		case 0x1183:
			strPtr = "Fujikura Ltd";
			break;

		case 0x1184:
			strPtr = "Forks Inc";
			break;

		case 0x1185:
			strPtr = "Dataworld International Ltd";
			break;

		case 0x1186:
			strPtr = "D-Link System Inc";
			break;

		case 0x1187:
			strPtr = "Advanced Technology Laboratories, Inc.";
			break;

		case 0x1188:
			strPtr = "Shima Seiki Manufacturing Ltd.";
			break;

		case 0x1189:
			strPtr = "Matsushita Electronics Co Ltd";
			break;

		case 0x118a:
			strPtr = "Hilevel Technology";
			break;

		case 0x118b:
			strPtr = "Hypertec Pty Limited";
			break;

		case 0x118c:
			strPtr = "Corollary, Inc";
			break;

		case 0x118d:
			strPtr = "BitFlow Inc";
			break;

		case 0x118e:
			strPtr = "Hermstedt GmbH";
			break;

		case 0x118f:
			strPtr = "Green Logic";
			break;

		case 0x1190:
			strPtr = "Tripace";
			break;

		case 0x1191:
			strPtr = "Artop Electronic Corp";
			break;

		case 0x1192:
			strPtr = "Densan Company Ltd";
			break;

		case 0x1193:
			strPtr = "Zeitnet Inc.";
			break;

		case 0x1194:
			strPtr = "Toucan Technology";
			break;

		case 0x1195:
			strPtr = "Ratoc System Inc";
			break;

		case 0x1196:
			strPtr = "Hytec Electronics Ltd";
			break;

		case 0x1197:
			strPtr = "Gage Applied Sciences, Inc.";
			break;

		case 0x1198:
			strPtr = "Lambda Systems Inc";
			break;

		case 0x1199:
			strPtr = "Attachmate Corporation";
			break;

		case 0x119a:
			strPtr = "Mind Share, Inc.";
			break;

		case 0x119b:
			strPtr = "Omega Micro Inc.";
			break;

		case 0x119c:
			strPtr = "Information Technology Inst.";
			break;

		case 0x119d:
			strPtr = "Bug, Inc. Sapporo Japan";
			break;

		case 0x119e:
			strPtr = "Fujitsu Microelectronics Ltd.";
			break;

		case 0x119f:
			strPtr = "Bull HN Information Systems";
			break;

		case 0x11a0:
			strPtr = "Convex Computer Corporation";
			break;

		case 0x11a1:
			strPtr = "Hamamatsu Photonics K.K.";
			break;

		case 0x11a2:
			strPtr = "Sierra Research and Technology";
			break;

		case 0x11a3:
			strPtr = "Deuretzbacher GmbH & Co. Eng. KG";
			break;

		case 0x11a4:
			strPtr = "Barco Graphics NV";
			break;

		case 0x11a5:
			strPtr = "Microunity Systems Eng. Inc";
			break;

		case 0x11a6:
			strPtr = "Pure Data Ltd.";
			break;

		case 0x11a7:
			strPtr = "Power Computing Corp.";
			break;

		case 0x11a8:
			strPtr = "Systech Corp.";
			break;

		case 0x11a9:
			strPtr = "InnoSys Inc.";
			break;

		case 0x11aa:
			strPtr = "Actel";
			break;

		case 0x11ab:
			strPtr = "Marvell Technology Group Ltd.";
			break;

		case 0x11ac:
			strPtr = "Canon Information Systems Research Aust.";
			break;

		case 0x11ad:
			strPtr = "Lite-On Communications Inc";
			break;

		case 0x11ae:
			strPtr = "Aztech System Ltd";
			break;

		case 0x11af:
			strPtr = "Avid Technology Inc.";
			break;

		case 0x11b0:
			strPtr = "V3 Semiconductor Inc.";
			break;

		case 0x11b1:
			strPtr = "Apricot Computers";
			break;

		case 0x11b2:
			strPtr = "Eastman Kodak";
			break;

		case 0x11b3:
			strPtr = "Barr Systems Inc.";
			break;

		case 0x11b4:
			strPtr = "Leitch Technology International";
			break;

		case 0x11b5:
			strPtr = "Radstone Technology Plc";
			break;

		case 0x11b6:
			strPtr = "United Video Corp";
			break;

		case 0x11b7:
			strPtr = "Motorola";
			break;

		case 0x11b8:
			strPtr = "XPoint Technologies, Inc";
			break;

		case 0x11b9:
			strPtr = "Pathlight Technology Inc.";
			break;

		case 0x11ba:
			strPtr = "Videotron Corp";
			break;

		case 0x11bb:
			strPtr = "Pyramid Technology";
			break;

		case 0x11bc:
			strPtr = "Network Peripherals Inc";
			break;

		case 0x11bd:
			strPtr = "Pinnacle Systems Inc.";
			break;

		case 0x11be:
			strPtr = "International Microcircuits Inc";
			break;

		case 0x11bf:
			strPtr = "Astrodesign, Inc.";
			break;

		case 0x11c0:
			strPtr = "Hewlett Packard";
			break;

		case 0x11c1:
			strPtr = "LSI Corporation";
			break;

		case 0x11c2:
			strPtr = "Sand Microelectronics";
			break;

		case 0x11c3:
			strPtr = "NEC Corporation";
			break;

		case 0x11c4:
			strPtr = "Document Technologies, Inc";
			break;

		case 0x11c5:
			strPtr = "Shiva Corporation";
			break;

		case 0x11c6:
			strPtr = "Dainippon Screen Mfg. Co. Ltd";
			break;

		case 0x11c7:
			strPtr = "D.C.M. Data Systems";
			break;

		case 0x11c8:
			strPtr = "Dolphin Interconnect Solutions AS";
			break;

		case 0x11c9:
			strPtr = "Magma";
			break;

		case 0x11ca:
			strPtr = "LSI Systems, Inc";
			break;

		case 0x11cb:
			strPtr = "Specialix Research Ltd.";
			break;

		case 0x11cc:
			strPtr = "Michels & Kleberhoff Computer GmbH";
			break;

		case 0x11cd:
			strPtr = "HAL Computer Systems, Inc.";
			break;

		case 0x11ce:
			strPtr = "Netaccess";
			break;

		case 0x11cf:
			strPtr = "Pioneer Electronic Corporation";
			break;

		case 0x11d0:
			strPtr = "Lockheed Martin Federal Systems-Manassas";
			break;

		case 0x11d1:
			strPtr = "Auravision";
			break;

		case 0x11d2:
			strPtr = "Intercom Inc.";
			break;

		case 0x11d3:
			strPtr = "Trancell Systems Inc";
			break;

		case 0x11d4:
			strPtr = "Analog Devices";
			break;

		case 0x11d5:
			strPtr = "Ikon Corporation";
			break;

		case 0x11d6:
			strPtr = "Tekelec Telecom";
			break;

		case 0x11d7:
			strPtr = "Trenton Technology, Inc.";
			break;

		case 0x11d8:
			strPtr = "Image Technologies Development";
			break;

		case 0x11d9:
			strPtr = "TEC Corporation";
			break;

		case 0x11da:
			strPtr = "Novell";
			break;

		case 0x11db:
			strPtr = "Sega Enterprises Ltd";
			break;

		case 0x11dc:
			strPtr = "Questra Corporation";
			break;

		case 0x11dd:
			strPtr = "Crosfield Electronics Limited";
			break;

		case 0x11de:
			strPtr = "Zoran Corporation";
			break;

		case 0x11df:
			strPtr = "New Wave PDG";
			break;

		case 0x11e0:
			strPtr = "Cray Communications A/S";
			break;

		case 0x11e1:
			strPtr = "GEC Plessey Semi Inc.";
			break;

		case 0x11e2:
			strPtr = "Samsung Information Systems America";
			break;

		case 0x11e3:
			strPtr = "Quicklogic Corporation";
			break;

		case 0x11e4:
			strPtr = "Second Wave Inc";
			break;

		case 0x11e5:
			strPtr = "IIX Consulting";
			break;

		case 0x11e6:
			strPtr = "Mitsui-Zosen System Research";
			break;

		case 0x11e7:
			strPtr = "Toshiba America, Elec. Company";
			break;

		case 0x11e8:
			strPtr = "Digital Processing Systems Inc.";
			break;

		case 0x11e9:
			strPtr = "Highwater Designs Ltd.";
			break;

		case 0x11ea:
			strPtr = "Elsag Bailey";
			break;

		case 0x11eb:
			strPtr = "Formation Inc.";
			break;

		case 0x11ec:
			strPtr = "Coreco Inc";
			break;

		case 0x11ed:
			strPtr = "Mediamatics";
			break;

		case 0x11ee:
			strPtr = "Dome Imaging Systems Inc";
			break;

		case 0x11ef:
			strPtr = "Nicolet Technologies B.V.";
			break;

		case 0x11f0:
			strPtr = "Compu-Shack";
			break;

		case 0x11f1:
			strPtr = "Symbios Logic Inc";
			break;

		case 0x11f2:
			strPtr = "Picture Tel Japan K.K.";
			break;

		case 0x11f3:
			strPtr = "Keithley Metrabyte";
			break;

		case 0x11f4:
			strPtr = "Kinetic Systems Corporation";
			break;

		case 0x11f5:
			strPtr = "Computing Devices International";
			break;

		case 0x11f6:
			strPtr = "Compex";
			break;

		case 0x11f7:
			strPtr = "Scientific Atlanta";
			break;

		case 0x11f8:
			strPtr = "PMC-Sierra Inc.";
			break;

		case 0x11f9:
			strPtr = "I-Cube Inc";
			break;

		case 0x11fa:
			strPtr = "Kasan Electronics Company, Ltd.";
			break;

		case 0x11fb:
			strPtr = "Datel Inc";
			break;

		case 0x11fc:
			strPtr = "Silicon Magic";
			break;

		case 0x11fd:
			strPtr = "High Street Consultants";
			break;

		case 0x11fe:
			strPtr = "Pepperl+Fuchs";
			break;

		case 0x11ff:
			strPtr = "Scion Corporation";
			break;

		case 0x1200:
			strPtr = "CSS Corporation";
			break;

		case 0x1201:
			strPtr = "Vista Controls Corp";
			break;

		case 0x1202:
			strPtr = "Network General Corp.";
			break;

		case 0x1203:
			strPtr = "Bayer Corporation, Agfa Division";
			break;

		case 0x1204:
			strPtr = "Lattice Semiconductor Corporation";
			break;

		case 0x1205:
			strPtr = "Array Corporation";
			break;

		case 0x1206:
			strPtr = "Amdahl Corporation";
			break;

		case 0x1208:
			strPtr = "Parsytec GmbH";
			break;

		case 0x1209:
			strPtr = "SCI Systems Inc";
			break;

		case 0x120a:
			strPtr = "Synaptel";
			break;

		case 0x120b:
			strPtr = "Adaptive Solutions";
			break;

		case 0x120c:
			strPtr = "Technical Corp.";
			break;

		case 0x120d:
			strPtr = "Compression Labs, Inc.";
			break;

		case 0x120e:
			strPtr = "Cyclades Corporation";
			break;

		case 0x120f:
			strPtr = "Essential Communications";
			break;

		case 0x1210:
			strPtr = "Hyperparallel Technologies";
			break;

		case 0x1211:
			strPtr = "Braintech Inc";
			break;

		case 0x1213:
			strPtr = "Applied Intelligent Systems, Inc.";
			break;

		case 0x1214:
			strPtr = "Performance Technologies, Inc.";
			break;

		case 0x1215:
			strPtr = "Interware Co., Ltd";
			break;

		case 0x1216:
			strPtr = "Purup Prepress A/S";
			break;

		case 0x1217:
			strPtr = "O2 Micro, Inc.";
			break;

		case 0x1218:
			strPtr = "Hybricon Corp.";
			break;

		case 0x1219:
			strPtr = "First Virtual Corporation";
			break;

		case 0x121a:
			strPtr = "3Dfx Interactive, Inc.";
			break;

		case 0x121b:
			strPtr = "Advanced Telecommunications Modules";
			break;

		case 0x121c:
			strPtr = "Nippon Texaco., Ltd";
			break;

		case 0x121d:
			strPtr = "LiPPERT ADLINK Technology GmbH";
			break;

		case 0x121e:
			strPtr = "CSPI";
			break;

		case 0x121f:
			strPtr = "Arcus Technology, Inc.";
			break;

		case 0x1220:
			strPtr = "Ariel Corporation";
			break;

		case 0x1221:
			strPtr = "Contec Co., Ltd";
			break;

		case 0x1222:
			strPtr = "Ancor Communications, Inc.";
			break;

		case 0x1223:
			strPtr = "Artesyn Communication Products";
			break;

		case 0x1224:
			strPtr = "Interactive Images";
			break;

		case 0x1225:
			strPtr = "Power I/O, Inc.";
			break;

		case 0x1227:
			strPtr = "EIZO Rugged Solutions";
			break;

		case 0x1228:
			strPtr = "Norsk Elektro Optikk A/S";
			break;

		case 0x1229:
			strPtr = "Data Kinesis Inc.";
			break;

		case 0x122a:
			strPtr = "Integrated Telecom";
			break;

		case 0x122b:
			strPtr = "LG Industrial Systems Co., Ltd";
			break;

		case 0x122c:
			strPtr = "Sican GmbH";
			break;

		case 0x122d:
			strPtr = "Aztech System Ltd";
			break;

		case 0x122e:
			strPtr = "Xyratex";
			break;

		case 0x122f:
			strPtr = "Andrew Corporation";
			break;

		case 0x1230:
			strPtr = "Fishcamp Engineering";
			break;

		case 0x1231:
			strPtr = "Woodward McCoach, Inc.";
			break;

		case 0x1232:
			strPtr = "GPT Limited";
			break;

		case 0x1233:
			strPtr = "Bus-Tech, Inc.";
			break;

		case 0x1235:
			strPtr = "SMART Modular Technologies";
			break;

		case 0x1236:
			strPtr = "Sigma Designs Corporation";
			break;

		case 0x1237:
			strPtr = "Alta Technology Corporation";
			break;

		case 0x1238:
			strPtr = "Adtran";
			break;

		case 0x1239:
			strPtr = "3DO Company";
			break;

		case 0x123a:
			strPtr = "Visicom Laboratories, Inc.";
			break;

		case 0x123b:
			strPtr = "Seeq Technology, Inc.";
			break;

		case 0x123c:
			strPtr = "Century Systems, Inc.";
			break;

		case 0x123d:
			strPtr = "Engineering Design Team, Inc.";
			break;

		case 0x123e:
			strPtr = "Simutech, Inc.";
			break;

		case 0x123f:
			strPtr = "LSI Logic";
			break;

		case 0x1240:
			strPtr = "Marathon Technologies Corp.";
			break;

		case 0x1241:
			strPtr = "DSC Communications";
			break;

		case 0x1242:
			strPtr = "JNI Corporation";
			break;

		case 0x1243:
			strPtr = "Delphax";
			break;

		case 0x1244:
			strPtr = "AVM GmbH";
			break;

		case 0x1245:
			strPtr = "A.P.D., S.A.";
			break;

		case 0x1246:
			strPtr = "Dipix Technologies, Inc.";
			break;

		case 0x1247:
			strPtr = "Xylon Research, Inc.";
			break;

		case 0x1248:
			strPtr = "Central Data Corporation";
			break;

		case 0x1249:
			strPtr = "Samsung Electronics Co., Ltd.";
			break;

		case 0x124a:
			strPtr = "AEG Electrocom GmbH";
			break;

		case 0x124b:
			strPtr = "SBS/Greenspring Modular I/O";
			break;

		case 0x124c:
			strPtr = "Solitron Technologies, Inc.";
			break;

		case 0x124d:
			strPtr = "Stallion Technologies, Inc.";
			break;

		case 0x124e:
			strPtr = "Cylink";
			break;

		case 0x124f:
			strPtr = "Infortrend Technology, Inc.";
			break;

		case 0x1250:
			strPtr = "Hitachi Microcomputer System Ltd";
			break;

		case 0x1251:
			strPtr = "VLSI Solutions Oy";
			break;

		case 0x1253:
			strPtr = "Guzik Technical Enterprises";
			break;

		case 0x1254:
			strPtr = "Linear Systems Ltd.";
			break;

		case 0x1255:
			strPtr = "Optibase Ltd";
			break;

		case 0x1256:
			strPtr = "Perceptive Solutions, Inc.";
			break;

		case 0x1257:
			strPtr = "Vertex Networks, Inc.";
			break;

		case 0x1258:
			strPtr = "Gilbarco, Inc.";
			break;

		case 0x1259:
			strPtr = "Allied Telesis";
			break;

		case 0x125a:
			strPtr = "ABB Power Systems";
			break;

		case 0x125b:
			strPtr = "Asix Electronics Corporation";
			break;

		case 0x125c:
			strPtr = "Aurora Technologies, Inc.";
			break;

		case 0x125d:
			strPtr = "ESS Technology";
			break;

		case 0x125e:
			strPtr = "Specialvideo Engineering SRL";
			break;

		case 0x125f:
			strPtr = "Concurrent Technologies, Inc.";
			break;

		case 0x1260:
			strPtr = "Intersil Corporation";
			break;

		case 0x1261:
			strPtr = "Matsushita-Kotobuki Electronics Industries, Ltd.";
			break;

		case 0x1262:
			strPtr = "ES Computer Company, Ltd.";
			break;

		case 0x1263:
			strPtr = "Sonic Solutions";
			break;

		case 0x1264:
			strPtr = "Aval Nagasaki Corporation";
			break;

		case 0x1265:
			strPtr = "Casio Computer Co., Ltd.";
			break;

		case 0x1266:
			strPtr = "Microdyne Corporation";
			break;

		case 0x1267:
			strPtr = "S. A. Telecommunications";
			break;

		case 0x1268:
			strPtr = "Tektronix";
			break;

		case 0x1269:
			strPtr = "Thomson-CSF/TTM";
			break;

		case 0x126a:
			strPtr = "Lexmark International, Inc.";
			break;

		case 0x126b:
			strPtr = "Adax, Inc.";
			break;

		case 0x126c:
			strPtr = "Northern Telecom";
			break;

		case 0x126d:
			strPtr = "Splash Technology, Inc.";
			break;

		case 0x126e:
			strPtr = "Sumitomo Metal Industries, Ltd.";
			break;

		case 0x126f:
			strPtr = "Silicon Motion, Inc.";
			break;

		case 0x1270:
			strPtr = "Olympus Optical Co., Ltd.";
			break;

		case 0x1271:
			strPtr = "GW Instruments";
			break;

		case 0x1272:
			strPtr = "Telematics International";
			break;

		case 0x1273:
			strPtr = "Hughes Network Systems";
			break;

		case 0x1274:
			strPtr = "Ensoniq";
			break;

		case 0x1275:
			strPtr = "Network Appliance Corporation";
			break;

		case 0x1276:
			strPtr = "Switched Network Technologies, Inc.";
			break;

		case 0x1277:
			strPtr = "Comstream";
			break;

		case 0x1278:
			strPtr = "Transtech Parallel Systems Ltd.";
			break;

		case 0x1279:
			strPtr = "Transmeta Corporation";
			break;

		case 0x127a:
			strPtr = "Rockwell International";
			break;

		case 0x127b:
			strPtr = "Pixera Corporation";
			break;

		case 0x127c:
			strPtr = "Crosspoint Solutions, Inc.";
			break;

		case 0x127d:
			strPtr = "Vela Research";
			break;

		case 0x127e:
			strPtr = "Winnov, L.P.";
			break;

		case 0x127f:
			strPtr = "Fujifilm";
			break;

		case 0x1280:
			strPtr = "Photoscript Group Ltd.";
			break;

		case 0x1281:
			strPtr = "Yokogawa Electric Corporation";
			break;

		case 0x1282:
			strPtr = "Davicom Semiconductor, Inc.";
			break;

		case 0x1283:
			strPtr = "Integrated Technology Express, Inc.";
			break;

		case 0x1284:
			strPtr = "Sahara Networks, Inc.";
			break;

		case 0x1285:
			strPtr = "Platform Technologies, Inc.";
			break;

		case 0x1286:
			strPtr = "Mazet GmbH";
			break;

		case 0x1287:
			strPtr = "M-Pact, Inc.";
			break;

		case 0x1288:
			strPtr = "Timestep Corporation";
			break;

		case 0x1289:
			strPtr = "AVC Technology, Inc.";
			break;

		case 0x128a:
			strPtr = "Asante Technologies, Inc.";
			break;

		case 0x128b:
			strPtr = "Transwitch Corporation";
			break;

		case 0x128c:
			strPtr = "Retix Corporation";
			break;

		case 0x128d:
			strPtr = "G2 Networks, Inc.";
			break;

		case 0x128e:
			strPtr = "Hoontech Corporation/Samho Multi Tech Ltd.";
			break;

		case 0x128f:
			strPtr = "Tateno Dennou, Inc.";
			break;

		case 0x1290:
			strPtr = "Sord Computer Corporation";
			break;

		case 0x1291:
			strPtr = "NCS Computer Italia";
			break;

		case 0x1292:
			strPtr = "Tritech Microelectronics Inc";
			break;

		case 0x1293:
			strPtr = "Media Reality Technology";
			break;

		case 0x1294:
			strPtr = "Rhetorex, Inc.";
			break;

		case 0x1295:
			strPtr = "Imagenation Corporation";
			break;

		case 0x1296:
			strPtr = "Kofax Image Products";
			break;

		case 0x1297:
			strPtr = "Holco Enterprise Co, Ltd/Shuttle Computer";
			break;

		case 0x1298:
			strPtr = "Spellcaster Telecommunications Inc.";
			break;

		case 0x1299:
			strPtr = "Knowledge Technology Lab.";
			break;

		case 0x129a:
			strPtr = "VMetro, inc.";
			break;

		case 0x129b:
			strPtr = "Image Access";
			break;

		case 0x129c:
			strPtr = "Jaycor";
			break;

		case 0x129d:
			strPtr = "Compcore Multimedia, Inc.";
			break;

		case 0x129e:
			strPtr = "Victor Company of Japan, Ltd.";
			break;

		case 0x129f:
			strPtr = "OEC Medical Systems, Inc.";
			break;

		case 0x12a0:
			strPtr = "Allen-Bradley Company";
			break;

		case 0x12a1:
			strPtr = "Simpact Associates, Inc.";
			break;

		case 0x12a2:
			strPtr = "Newgen Systems Corporation";
			break;

		case 0x12a3:
			strPtr = "Lucent Technologies";
			break;

		case 0x12a4:
			strPtr = "NTT Electronics Corporation";
			break;

		case 0x12a5:
			strPtr = "Vision Dynamics Ltd.";
			break;

		case 0x12a6:
			strPtr = "Scalable Networks, Inc.";
			break;

		case 0x12a7:
			strPtr = "AMO GmbH";
			break;

		case 0x12a8:
			strPtr = "News Datacom";
			break;

		case 0x12a9:
			strPtr = "Xiotech Corporation";
			break;

		case 0x12aa:
			strPtr = "SDL Communications, Inc.";
			break;

		case 0x12ab:
			strPtr = "YUAN High-Tech Development Co., Ltd.";
			break;

		case 0x12ac:
			strPtr = "Measurex Corporation";
			break;

		case 0x12ad:
			strPtr = "Multidata GmbH";
			break;

		case 0x12ae:
			strPtr = "Alteon Networks Inc.";
			break;

		case 0x12af:
			strPtr = "TDK USA Corp";
			break;

		case 0x12b0:
			strPtr = "Jorge Scientific Corp";
			break;

		case 0x12b1:
			strPtr = "GammaLink";
			break;

		case 0x12b2:
			strPtr = "General Signal Networks";
			break;

		case 0x12b3:
			strPtr = "Inter-Face Co Ltd";
			break;

		case 0x12b4:
			strPtr = "FutureTel Inc";
			break;

		case 0x12b5:
			strPtr = "Granite Systems Inc.";
			break;

		case 0x12b6:
			strPtr = "Natural Microsystems";
			break;

		case 0x12b7:
			strPtr = "Cognex Corporation";
			break;

		case 0x12b8:
			strPtr = "Korg";
			break;

		case 0x12b9:
			strPtr = "3Com Corp, Modem Division";
			break;

		case 0x12ba:
			strPtr = "BittWare, Inc.";
			break;

		case 0x12bb:
			strPtr = "Nippon Unisoft Corporation";
			break;

		case 0x12bc:
			strPtr = "Array Microsystems";
			break;

		case 0x12bd:
			strPtr = "Computerm Corp.";
			break;

		case 0x12be:
			strPtr = "Anchor Chips Inc.";
			break;

		case 0x12bf:
			strPtr = "Fujifilm Microdevices";
			break;

		case 0x12c0:
			strPtr = "Infimed";
			break;

		case 0x12c1:
			strPtr = "GMM Research Corp";
			break;

		case 0x12c2:
			strPtr = "Mentec Limited";
			break;

		case 0x12c3:
			strPtr = "Holtek Microelectronics Inc";
			break;

		case 0x12c4:
			strPtr = "Connect Tech Inc";
			break;

		case 0x12c5:
			strPtr = "Picture Elements Incorporated";
			break;

		case 0x12c6:
			strPtr = "Mitani Corporation";
			break;

		case 0x12c7:
			strPtr = "Dialogic Corp";
			break;

		case 0x12c8:
			strPtr = "G Force Co, Ltd";
			break;

		case 0x12c9:
			strPtr = "Gigi Operations";
			break;

		case 0x12ca:
			strPtr = "Integrated Computing Engines";
			break;

		case 0x12cb:
			strPtr = "Antex Electronics Corporation";
			break;

		case 0x12cc:
			strPtr = "Pluto Technologies International";
			break;

		case 0x12cd:
			strPtr = "Aims Lab";
			break;

		case 0x12ce:
			strPtr = "Netspeed Inc.";
			break;

		case 0x12cf:
			strPtr = "Prophet Systems, Inc.";
			break;

		case 0x12d0:
			strPtr = "GDE Systems, Inc.";
			break;

		case 0x12d1:
			strPtr = "PSITech";
			break;

		case 0x12d2:
			strPtr = "NVidia / SGS Thomson";
			break;

		case 0x12d3:
			strPtr = "Vingmed Sound A/S";
			break;

		case 0x12d4:
			strPtr = "Ulticom";
			break;

		case 0x12d5:
			strPtr = "Equator Technologies Inc";
			break;

		case 0x12d6:
			strPtr = "Analogic Corp";
			break;

		case 0x12d7:
			strPtr = "Biotronic SRL";
			break;

		case 0x12d8:
			strPtr = "Pericom Semiconductor";
			break;

		case 0x12d9:
			strPtr = "Aculab PLC";
			break;

		case 0x12da:
			strPtr = "True Time Inc.";
			break;

		case 0x12db:
			strPtr = "Annapolis Micro Systems, Inc";
			break;

		case 0x12dc:
			strPtr = "Symicron Computer Communication Ltd.";
			break;

		case 0x12dd:
			strPtr = "Management Graphics";
			break;

		case 0x12de:
			strPtr = "Rainbow Technologies";
			break;

		case 0x12df:
			strPtr = "SBS Technologies Inc";
			break;

		case 0x12e0:
			strPtr = "Chase Research";
			break;

		case 0x12e1:
			strPtr = "Nintendo Co, Ltd";
			break;

		case 0x12e2:
			strPtr = "Datum Inc. Bancomm-Timing Division";
			break;

		case 0x12e3:
			strPtr = "Imation Corp - Medical Imaging Systems";
			break;

		case 0x12e4:
			strPtr = "Brooktrout Technology Inc";
			break;

		case 0x12e5:
			strPtr = "Apex Semiconductor Inc";
			break;

		case 0x12e6:
			strPtr = "Cirel Systems";
			break;

		case 0x12e7:
			strPtr = "Sunsgroup Corporation";
			break;

		case 0x12e8:
			strPtr = "Crisc Corp";
			break;

		case 0x12e9:
			strPtr = "GE Spacenet";
			break;

		case 0x12ea:
			strPtr = "Zuken";
			break;

		case 0x12eb:
			strPtr = "Aureal Semiconductor";
			break;

		case 0x12ec:
			strPtr = "3A International, Inc.";
			break;

		case 0x12ed:
			strPtr = "Optivision Inc.";
			break;

		case 0x12ee:
			strPtr = "Orange Micro";
			break;

		case 0x12ef:
			strPtr = "Vienna Systems";
			break;

		case 0x12f0:
			strPtr = "Pentek";
			break;

		case 0x12f1:
			strPtr = "Sorenson Vision Inc";
			break;

		case 0x12f2:
			strPtr = "Gammagraphx, Inc.";
			break;

		case 0x12f3:
			strPtr = "Radstone Technology";
			break;

		case 0x12f4:
			strPtr = "Megatel";
			break;

		case 0x12f5:
			strPtr = "Forks";
			break;

		case 0x12f6:
			strPtr = "Dawson France";
			break;

		case 0x12f7:
			strPtr = "Cognex";
			break;

		case 0x12f8:
			strPtr = "Electronic Design GmbH";
			break;

		case 0x12f9:
			strPtr = "Four Fold Ltd";
			break;

		case 0x12fb:
			strPtr = "Spectrum Signal Processing";
			break;

		case 0x12fc:
			strPtr = "Capital Equipment Corp";
			break;

		case 0x12fd:
			strPtr = "I2S";
			break;

		case 0x12fe:
			strPtr = "ESD Electronic System Design GmbH";
			break;

		case 0x12ff:
			strPtr = "Lexicon";
			break;

		case 0x1300:
			strPtr = "Harman International Industries Inc";
			break;

		case 0x1302:
			strPtr = "Computer Sciences Corp";
			break;

		case 0x1303:
			strPtr = "Innovative Integration";
			break;

		case 0x1304:
			strPtr = "Juniper Networks";
			break;

		case 0x1305:
			strPtr = "Netphone, Inc";
			break;

		case 0x1306:
			strPtr = "Duet Technologies";
			break;

		case 0x1307:
			strPtr = "Measurement Computing";
			break;

		case 0x1308:
			strPtr = "Jato Technologies Inc.";
			break;

		case 0x1309:
			strPtr = "AB Semiconductor Ltd";
			break;

		case 0x130a:
			strPtr = "Mitsubishi Electric Microcomputer";
			break;

		case 0x130b:
			strPtr = "Colorgraphic Communications Corp";
			break;

		case 0x130c:
			strPtr = "Ambex Technologies, Inc";
			break;

		case 0x130d:
			strPtr = "Accelerix Inc";
			break;

		case 0x130e:
			strPtr = "Yamatake-Honeywell Co. Ltd";
			break;

		case 0x130f:
			strPtr = "Advanet Inc";
			break;

		case 0x1310:
			strPtr = "Gespac";
			break;

		case 0x1311:
			strPtr = "Videoserver, Inc";
			break;

		case 0x1312:
			strPtr = "Acuity Imaging, Inc";
			break;

		case 0x1313:
			strPtr = "Yaskawa Electric Co.";
			break;

		case 0x1315:
			strPtr = "Wavesat";
			break;

		case 0x1316:
			strPtr = "Teradyne Inc";
			break;

		case 0x1317:
			strPtr = "ADMtek";
			break;

		case 0x1318:
			strPtr = "Packet Engines Inc.";
			break;

		case 0x1319:
			strPtr = "Fortemedia, Inc";
			break;

		case 0x131a:
			strPtr = "Finisar Corp.";
			break;

		case 0x131c:
			strPtr = "Nippon Electro-Sensory Devices Corp";
			break;

		case 0x131d:
			strPtr = "Sysmic, Inc.";
			break;

		case 0x131e:
			strPtr = "Xinex Networks Inc";
			break;

		case 0x131f:
			strPtr = "Siig Inc";
			break;

		case 0x1320:
			strPtr = "Crypto AG";
			break;

		case 0x1321:
			strPtr = "Arcobel Graphics BV";
			break;

		case 0x1322:
			strPtr = "MTT Co., Ltd";
			break;

		case 0x1323:
			strPtr = "Dome Inc";
			break;

		case 0x1324:
			strPtr = "Sphere Communications";
			break;

		case 0x1325:
			strPtr = "Salix Technologies, Inc";
			break;

		case 0x1326:
			strPtr = "Seachange international";
			break;

		case 0x1327:
			strPtr = "Voss scientific";
			break;

		case 0x1328:
			strPtr = "quadrant international";
			break;

		case 0x1329:
			strPtr = "Productivity Enhancement";
			break;

		case 0x132a:
			strPtr = "Microcom Inc.";
			break;

		case 0x132b:
			strPtr = "Broadband Technologies";
			break;

		case 0x132c:
			strPtr = "Micrel Inc";
			break;

		case 0x132d:
			strPtr = "Integrated Silicon Solution, Inc.";
			break;

		case 0x1330:
			strPtr = "MMC Networks";
			break;

		case 0x1331:
			strPtr = "RadiSys Corporation";
			break;

		case 0x1332:
			strPtr = "Micro Memory";
			break;

		case 0x1334:
			strPtr = "Redcreek Communications, Inc";
			break;

		case 0x1335:
			strPtr = "Videomail, Inc";
			break;

		case 0x1337:
			strPtr = "Third Planet Publishing";
			break;

		case 0x1338:
			strPtr = "BT Electronics";
			break;

		case 0x133a:
			strPtr = "Vtel Corp";
			break;

		case 0x133b:
			strPtr = "Softcom Microsystems";
			break;

		case 0x133c:
			strPtr = "Holontech Corp";
			break;

		case 0x133d:
			strPtr = "SS Technologies";
			break;

		case 0x133e:
			strPtr = "Virtual Computer Corp";
			break;

		case 0x133f:
			strPtr = "SCM Microsystems";
			break;

		case 0x1340:
			strPtr = "Atalla Corp";
			break;

		case 0x1341:
			strPtr = "Kyoto Microcomputer Co";
			break;

		case 0x1342:
			strPtr = "Promax Systems Inc";
			break;

		case 0x1343:
			strPtr = "Phylon Communications Inc";
			break;

		case 0x1344:
			strPtr = "Micron Technology Inc";
			break;

		case 0x1345:
			strPtr = "Arescom Inc";
			break;

		case 0x1347:
			strPtr = "Odetics";
			break;

		case 0x1349:
			strPtr = "Sumitomo Electric Industries, Ltd.";
			break;

		case 0x134a:
			strPtr = "DTC Technology Corp.";
			break;

		case 0x134b:
			strPtr = "ARK Research Corp.";
			break;

		case 0x134c:
			strPtr = "Chori Joho System Co. Ltd";
			break;

		case 0x134d:
			strPtr = "PCTel Inc";
			break;

		case 0x134e:
			strPtr = "CSTI";
			break;

		case 0x134f:
			strPtr = "Algo System Co Ltd";
			break;

		case 0x1350:
			strPtr = "Systec Co. Ltd";
			break;

		case 0x1351:
			strPtr = "Sonix Inc";
			break;

		case 0x1353:
			strPtr = "dbeeSet Technology";
			break;

		case 0x1354:
			strPtr = "Dwave System Inc";
			break;

		case 0x1355:
			strPtr = "Kratos Analytical Ltd";
			break;

		case 0x1356:
			strPtr = "The Logical Co";
			break;

		case 0x1359:
			strPtr = "Prisa Networks";
			break;

		case 0x135a:
			strPtr = "Brain Boxes";
			break;

		case 0x135b:
			strPtr = "Giganet Inc";
			break;

		case 0x135c:
			strPtr = "Quatech Inc";
			break;

		case 0x135d:
			strPtr = "ABB Network Partner AB";
			break;

		case 0x135e:
			strPtr = "Sealevel Systems Inc";
			break;

		case 0x135f:
			strPtr = "I-Data International A-S";
			break;

		case 0x1360:
			strPtr = "Meinberg Funkuhren";
			break;

		case 0x1361:
			strPtr = "Soliton Systems K.K.";
			break;

		case 0x1362:
			strPtr = "Fujifacom Corporation";
			break;

		case 0x1363:
			strPtr = "Phoenix Technology Ltd";
			break;

		case 0x1364:
			strPtr = "ATM Communications Inc";
			break;

		case 0x1365:
			strPtr = "Hypercope GmbH";
			break;

		case 0x1366:
			strPtr = "Teijin Seiki Co. Ltd";
			break;

		case 0x1367:
			strPtr = "Hitachi Zosen Corporation";
			break;

		case 0x1368:
			strPtr = "Skyware Corporation";
			break;

		case 0x1369:
			strPtr = "Digigram";
			break;

		case 0x136a:
			strPtr = "High Soft Tech";
			break;

		case 0x136b:
			strPtr = "Kawasaki Steel Corporation";
			break;

		case 0x136c:
			strPtr = "Adtek System Science Co Ltd";
			break;

		case 0x136d:
			strPtr = "Gigalabs Inc";
			break;

		case 0x136f:
			strPtr = "Applied Magic Inc";
			break;

		case 0x1370:
			strPtr = "ATL Products";
			break;

		case 0x1371:
			strPtr = "CNet Technology Inc";
			break;

		case 0x1373:
			strPtr = "Silicon Vision Inc";
			break;

		case 0x1374:
			strPtr = "Silicom Ltd.";
			break;

		case 0x1375:
			strPtr = "Argosystems Inc";
			break;

		case 0x1376:
			strPtr = "LMC";
			break;

		case 0x1377:
			strPtr = "Electronic Equipment Production & Distribution GmbH";
			break;

		case 0x1378:
			strPtr = "Telemann Co. Ltd";
			break;

		case 0x1379:
			strPtr = "Asahi Kasei Microsystems Co Ltd";
			break;

		case 0x137a:
			strPtr = "Mark of the Unicorn Inc";
			break;

		case 0x137b:
			strPtr = "PPT Vision";
			break;

		case 0x137c:
			strPtr = "Iwatsu Electric Co Ltd";
			break;

		case 0x137d:
			strPtr = "Dynachip Corporation";
			break;

		case 0x137e:
			strPtr = "Patriot Scientific Corporation";
			break;

		case 0x137f:
			strPtr = "Japan Satellite Systems Inc";
			break;

		case 0x1380:
			strPtr = "Sanritz Automation Co Ltd";
			break;

		case 0x1381:
			strPtr = "Brains Co. Ltd";
			break;

		case 0x1382:
			strPtr = "Marian - Electronic & Software";
			break;

		case 0x1383:
			strPtr = "Controlnet Inc";
			break;

		case 0x1384:
			strPtr = "Reality Simulation Systems Inc";
			break;

		case 0x1385:
			strPtr = "Netgear";
			break;

		case 0x1386:
			strPtr = "Video Domain Technologies";
			break;

		case 0x1387:
			strPtr = "Systran Corp";
			break;

		case 0x1388:
			strPtr = "Hitachi Information Technology Co Ltd";
			break;

		case 0x1389:
			strPtr = "Applicom International";
			break;

		case 0x138a:
			strPtr = "Fusion Micromedia Corp";
			break;

		case 0x138b:
			strPtr = "Tokimec Inc";
			break;

		case 0x138c:
			strPtr = "Silicon Reality";
			break;

		case 0x138d:
			strPtr = "Future Techno Designs pte Ltd";
			break;

		case 0x138e:
			strPtr = "Basler GmbH";
			break;

		case 0x138f:
			strPtr = "Patapsco Designs Inc";
			break;

		case 0x1390:
			strPtr = "Concept Development Inc";
			break;

		case 0x1391:
			strPtr = "Development Concepts Inc";
			break;

		case 0x1392:
			strPtr = "Medialight Inc";
			break;

		case 0x1393:
			strPtr = "Moxa Technologies Co Ltd";
			break;

		case 0x1394:
			strPtr = "Level One Communications";
			break;

		case 0x1395:
			strPtr = "Ambicom Inc";
			break;

		case 0x1396:
			strPtr = "Cipher Systems Inc";
			break;

		case 0x1397:
			strPtr = "Cologne Chip Designs GmbH";
			break;

		case 0x1398:
			strPtr = "Clarion co. Ltd";
			break;

		case 0x1399:
			strPtr = "Rios systems Co Ltd";
			break;

		case 0x139a:
			strPtr = "Alacritech Inc";
			break;

		case 0x139b:
			strPtr = "Mediasonic Multimedia Systems Ltd";
			break;

		case 0x139c:
			strPtr = "Quantum 3d Inc";
			break;

		case 0x139d:
			strPtr = "EPL limited";
			break;

		case 0x139e:
			strPtr = "Media4";
			break;

		case 0x139f:
			strPtr = "Aethra s.r.l.";
			break;

		case 0x13a0:
			strPtr = "Crystal Group Inc";
			break;

		case 0x13a1:
			strPtr = "Kawasaki Heavy Industries Ltd";
			break;

		case 0x13a2:
			strPtr = "Ositech Communications Inc";
			break;

		case 0x13a3:
			strPtr = "Hifn Inc.";
			break;

		case 0x13a4:
			strPtr = "Rascom Inc";
			break;

		case 0x13a5:
			strPtr = "Audio Digital Imaging Inc";
			break;

		case 0x13a6:
			strPtr = "Videonics Inc";
			break;

		case 0x13a7:
			strPtr = "Teles AG";
			break;

		case 0x13a8:
			strPtr = "Exar Corp.";
			break;

		case 0x13a9:
			strPtr = "Siemens Medical Systems, Ultrasound Group";
			break;

		case 0x13aa:
			strPtr = "Broadband Networks Inc";
			break;

		case 0x13ab:
			strPtr = "Arcom Control Systems Ltd";
			break;

		case 0x13ac:
			strPtr = "Motion Media Technology Ltd";
			break;

		case 0x13ad:
			strPtr = "Nexus Inc";
			break;

		case 0x13ae:
			strPtr = "ALD Technology Ltd";
			break;

		case 0x13af:
			strPtr = "T.Sqware";
			break;

		case 0x13b0:
			strPtr = "Maxspeed Corp";
			break;

		case 0x13b1:
			strPtr = "Tamura corporation";
			break;

		case 0x13b2:
			strPtr = "Techno Chips Co. Ltd";
			break;

		case 0x13b3:
			strPtr = "Lanart Corporation";
			break;

		case 0x13b4:
			strPtr = "Wellbean Co Inc";
			break;

		case 0x13b5:
			strPtr = "ARM";
			break;

		case 0x13b6:
			strPtr = "Dlog GmbH";
			break;

		case 0x13b7:
			strPtr = "Logic Devices Inc";
			break;

		case 0x13b8:
			strPtr = "Nokia Telecommunications oy";
			break;

		case 0x13b9:
			strPtr = "Elecom Co Ltd";
			break;

		case 0x13ba:
			strPtr = "Oxford Instruments";
			break;

		case 0x13bb:
			strPtr = "Sanyo Technosound Co Ltd";
			break;

		case 0x13bc:
			strPtr = "Bitran Corporation";
			break;

		case 0x13bd:
			strPtr = "Sharp corporation";
			break;

		case 0x13be:
			strPtr = "Miroku Jyoho Service Co. Ltd";
			break;

		case 0x13bf:
			strPtr = "Sharewave Inc";
			break;

		case 0x13c0:
			strPtr = "Microgate Corporation";
			break;

		case 0x13c1:
			strPtr = "3ware Inc";
			break;

		case 0x13c2:
			strPtr = "Technotrend Systemtechnik GmbH";
			break;

		case 0x13c3:
			strPtr = "Janz Computer AG";
			break;

		case 0x13c4:
			strPtr = "Phase Metrics";
			break;

		case 0x13c5:
			strPtr = "Alphi Technology Corp";
			break;

		case 0x13c6:
			strPtr = "Condor Engineering Inc";
			break;

		case 0x13c7:
			strPtr = "Blue Chip Technology Ltd";
			break;

		case 0x13c8:
			strPtr = "Apptech Inc";
			break;

		case 0x13c9:
			strPtr = "Eaton Corporation";
			break;

		case 0x13ca:
			strPtr = "Iomega Corporation";
			break;

		case 0x13cb:
			strPtr = "Yano Electric Co Ltd";
			break;

		case 0x13cc:
			strPtr = "BARCO";
			break;

		case 0x13cd:
			strPtr = "Compatible Systems Corporation";
			break;

		case 0x13ce:
			strPtr = "Cocom A/S";
			break;

		case 0x13cf:
			strPtr = "Studio Audio & Video Ltd";
			break;

		case 0x13d0:
			strPtr = "Techsan Electronics Co Ltd";
			break;

		case 0x13d1:
			strPtr = "Abocom Systems Inc";
			break;

		case 0x13d2:
			strPtr = "Shark Multimedia Inc";
			break;

		case 0x13d4:
			strPtr = "Graphics Microsystems Inc";
			break;

		case 0x13d5:
			strPtr = "Media 100 Inc";
			break;

		case 0x13d6:
			strPtr = "K.I. Technology Co Ltd";
			break;

		case 0x13d7:
			strPtr = "Toshiba Engineering Corporation";
			break;

		case 0x13d8:
			strPtr = "Phobos corporation";
			break;

		case 0x13d9:
			strPtr = "Apex PC Solutions Inc";
			break;

		case 0x13da:
			strPtr = "Intresource Systems pte Ltd";
			break;

		case 0x13db:
			strPtr = "Janich & Klass Computertechnik GmbH";
			break;

		case 0x13dc:
			strPtr = "Netboost Corporation";
			break;

		case 0x13dd:
			strPtr = "Multimedia Bundle Inc";
			break;

		case 0x13de:
			strPtr = "ABB Robotics Products AB";
			break;

		case 0x13df:
			strPtr = "E-Tech Inc";
			break;

		case 0x13e0:
			strPtr = "GVC Corporation";
			break;

		case 0x13e1:
			strPtr = "Silicom Multimedia Systems Inc";
			break;

		case 0x13e2:
			strPtr = "Dynamics Research Corporation";
			break;

		case 0x13e3:
			strPtr = "Nest Inc";
			break;

		case 0x13e4:
			strPtr = "Calculex Inc";
			break;

		case 0x13e5:
			strPtr = "Telesoft Design Ltd";
			break;

		case 0x13e6:
			strPtr = "Argosy research Inc";
			break;

		case 0x13e7:
			strPtr = "NAC Incorporated";
			break;

		case 0x13e8:
			strPtr = "Chip Express Corporation";
			break;

		case 0x13e9:
			strPtr = "Intraserver Technology Inc";
			break;

		case 0x13ea:
			strPtr = "Dallas Semiconductor";
			break;

		case 0x13eb:
			strPtr = "Hauppauge Computer Works Inc";
			break;

		case 0x13ec:
			strPtr = "Zydacron Inc";
			break;

		case 0x13ed:
			strPtr = "Raytheion E-Systems";
			break;

		case 0x13ee:
			strPtr = "Hayes Microcomputer Products Inc";
			break;

		case 0x13ef:
			strPtr = "Coppercom Inc";
			break;

		case 0x13f0:
			strPtr = "Sundance Technology Inc / IC Plus Corp";
			break;

		case 0x13f1:
			strPtr = "Oce' - Technologies B.V.";
			break;

		case 0x13f2:
			strPtr = "Ford Microelectronics Inc";
			break;

		case 0x13f3:
			strPtr = "Mcdata Corporation";
			break;

		case 0x13f4:
			strPtr = "Troika Networks, Inc.";
			break;

		case 0x13f5:
			strPtr = "Kansai Electric Co. Ltd";
			break;

		case 0x13f6:
			strPtr = "C-Media Electronics Inc";
			break;

		case 0x13f7:
			strPtr = "Wildfire Communications";
			break;

		case 0x13f8:
			strPtr = "Ad Lib Multimedia Inc";
			break;

		case 0x13f9:
			strPtr = "NTT Advanced Technology Corp.";
			break;

		case 0x13fa:
			strPtr = "Pentland Systems Ltd";
			break;

		case 0x13fb:
			strPtr = "Aydin Corp";
			break;

		case 0x13fc:
			strPtr = "Computer Peripherals International";
			break;

		case 0x13fd:
			strPtr = "Micro Science Inc";
			break;

		case 0x13fe:
			strPtr = "Advantech Co. Ltd";
			break;

		case 0x13ff:
			strPtr = "Silicon Spice Inc";
			break;

		case 0x1400:
			strPtr = "Artx Inc";
			break;

		case 0x1401:
			strPtr = "CR-Systems A/S";
			break;

		case 0x1402:
			strPtr = "Meilhaus Electronic GmbH";
			break;

		case 0x1403:
			strPtr = "Ascor Inc";
			break;

		case 0x1404:
			strPtr = "Fundamental Software Inc";
			break;

		case 0x1405:
			strPtr = "Excalibur Systems Inc";
			break;

		case 0x1406:
			strPtr = "Oce' Printing Systems GmbH";
			break;

		case 0x1407:
			strPtr = "Lava Computer mfg Inc";
			break;

		case 0x1408:
			strPtr = "Aloka Co. Ltd";
			break;

		case 0x1409:
			strPtr = "Timedia Technology Co Ltd";
			break;

		case 0x140a:
			strPtr = "DSP Research Inc";
			break;

		case 0x140b:
			strPtr = "Abaco Systems, Inc.";
			break;

		case 0x140c:
			strPtr = "Elmic Systems Inc";
			break;

		case 0x140d:
			strPtr = "Matsushita Electric Works Ltd";
			break;

		case 0x140e:
			strPtr = "Goepel Electronic GmbH";
			break;

		case 0x140f:
			strPtr = "Salient Systems Corp";
			break;

		case 0x1410:
			strPtr = "Midas lab Inc";
			break;

		case 0x1411:
			strPtr = "Ikos Systems Inc";
			break;

		case 0x1412:
			strPtr = "VIA Technologies Inc.";
			break;

		case 0x1413:
			strPtr = "Addonics";
			break;

		case 0x1414:
			strPtr = "Microsoft Corporation";
			break;

		case 0x1415:
			strPtr = "Oxford Semiconductor Ltd";
			break;

		case 0x1416:
			strPtr = "Multiwave Innovation pte Ltd";
			break;

		case 0x1417:
			strPtr = "Convergenet Technologies Inc";
			break;

		case 0x1418:
			strPtr = "Kyushu electronics systems Inc";
			break;

		case 0x1419:
			strPtr = "Excel Switching Corp";
			break;

		case 0x141a:
			strPtr = "Apache Micro Peripherals Inc";
			break;

		case 0x141b:
			strPtr = "Zoom Telephonics Inc";
			break;

		case 0x141d:
			strPtr = "Digitan Systems Inc";
			break;

		case 0x141e:
			strPtr = "Fanuc Ltd";
			break;

		case 0x141f:
			strPtr = "Visiontech Ltd";
			break;

		case 0x1420:
			strPtr = "Psion Dacom plc";
			break;

		case 0x1421:
			strPtr = "Ads Technologies Inc";
			break;

		case 0x1422:
			strPtr = "Ygrec Systems Co Ltd";
			break;

		case 0x1423:
			strPtr = "Custom Technology Corp.";
			break;

		case 0x1424:
			strPtr = "Videoserver Connections";
			break;

		case 0x1425:
			strPtr = "Chelsio Communications Inc";
			break;

		case 0x1426:
			strPtr = "Storage Technology Corp.";
			break;

		case 0x1427:
			strPtr = "Better On-Line Solutions";
			break;

		case 0x1428:
			strPtr = "Edec Co Ltd";
			break;

		case 0x1429:
			strPtr = "Unex Technology Corp.";
			break;

		case 0x142a:
			strPtr = "Kingmax Technology Inc";
			break;

		case 0x142b:
			strPtr = "Radiolan";
			break;

		case 0x142c:
			strPtr = "Minton Optic Industry Co Ltd";
			break;

		case 0x142d:
			strPtr = "Pix stream Inc";
			break;

		case 0x142e:
			strPtr = "Vitec Multimedia";
			break;

		case 0x142f:
			strPtr = "Radicom Research Inc";
			break;

		case 0x1430:
			strPtr = "ITT Aerospace/Communications Division";
			break;

		case 0x1431:
			strPtr = "Gilat Satellite Networks";
			break;

		case 0x1432:
			strPtr = "Edimax Computer Co.";
			break;

		case 0x1433:
			strPtr = "Eltec Elektronik GmbH";
			break;

		case 0x1435:
			strPtr = "RTD Embedded Technologies, Inc.";
			break;

		case 0x1436:
			strPtr = "CIS Technology Inc";
			break;

		case 0x1437:
			strPtr = "Nissin Inc Co";
			break;

		case 0x1438:
			strPtr = "Atmel-dream";
			break;

		case 0x1439:
			strPtr = "Outsource Engineering & Mfg. Inc";
			break;

		case 0x143a:
			strPtr = "Stargate Solutions Inc";
			break;

		case 0x143b:
			strPtr = "Canon Research Center, America";
			break;

		case 0x143c:
			strPtr = "Amlogic Inc";
			break;

		case 0x143d:
			strPtr = "Tamarack Microelectronics Inc";
			break;

		case 0x143e:
			strPtr = "Jones Futurex Inc";
			break;

		case 0x143f:
			strPtr = "Lightwell Co Ltd - Zax Division";
			break;

		case 0x1440:
			strPtr = "ALGOL Corp.";
			break;

		case 0x1441:
			strPtr = "AGIE Ltd";
			break;

		case 0x1442:
			strPtr = "Phoenix Contact GmbH & Co.";
			break;

		case 0x1443:
			strPtr = "Unibrain S.A.";
			break;

		case 0x1444:
			strPtr = "TRW";
			break;

		case 0x1445:
			strPtr = "Logical DO Ltd";
			break;

		case 0x1446:
			strPtr = "Graphin Co Ltd";
			break;

		case 0x1447:
			strPtr = "AIM GmBH";
			break;

		case 0x1448:
			strPtr = "Alesis Studio Electronics";
			break;

		case 0x1449:
			strPtr = "TUT Systems Inc";
			break;

		case 0x144a:
			strPtr = "Adlink Technology";
			break;

		case 0x144b:
			strPtr = "Verint Systems Inc.";
			break;

		case 0x144c:
			strPtr = "Catalina Research Inc";
			break;

		case 0x144d:
			strPtr = "Samsung Electronics Co Ltd";
			break;

		case 0x144e:
			strPtr = "OLITEC";
			break;

		case 0x144f:
			strPtr = "Askey Computer Corp.";
			break;

		case 0x1450:
			strPtr = "Octave Communications Ind.";
			break;

		case 0x1451:
			strPtr = "SP3D Chip Design GmBH";
			break;

		case 0x1453:
			strPtr = "MYCOM Inc";
			break;

		case 0x1454:
			strPtr = "Altiga Networks";
			break;

		case 0x1455:
			strPtr = "Logic Plus Plus Inc";
			break;

		case 0x1456:
			strPtr = "Advanced Hardware Architectures";
			break;

		case 0x1457:
			strPtr = "Nuera Communications Inc";
			break;

		case 0x1458:
			strPtr = "Gigabyte Technology Co., Ltd";
			break;

		case 0x1459:
			strPtr = "DOOIN Electronics";
			break;

		case 0x145a:
			strPtr = "Escalate Networks Inc";
			break;

		case 0x145b:
			strPtr = "PRAIM SRL";
			break;

		case 0x145c:
			strPtr = "Cryptek";
			break;

		case 0x145d:
			strPtr = "Gallant Computer Inc";
			break;

		case 0x145e:
			strPtr = "Aashima Technology B.V.";
			break;

		case 0x145f:
			strPtr = "Baldor Electric Company";
			break;

		case 0x1460:
			strPtr = "DYNARC INC";
			break;

		case 0x1461:
			strPtr = "Avermedia Technologies Inc";
			break;

		case 0x1462:
			strPtr = "Micro-Star International Co., Ltd. [MSI]";
			break;

		case 0x1463:
			strPtr = "Fast Corporation";
			break;

		case 0x1464:
			strPtr = "Interactive Circuits & Systems Ltd";
			break;

		case 0x1465:
			strPtr = "GN NETTEST Telecom DIV.";
			break;

		case 0x1466:
			strPtr = "Designpro Inc.";
			break;

		case 0x1467:
			strPtr = "DIGICOM SPA";
			break;

		case 0x1468:
			strPtr = "AMBIT Microsystem Corp.";
			break;

		case 0x1469:
			strPtr = "Cleveland Motion Controls";
			break;

		case 0x146a:
			strPtr = "Aeroflex";
			break;

		case 0x146b:
			strPtr = "Parascan Technologies Ltd";
			break;

		case 0x146c:
			strPtr = "Ruby Tech Corp.";
			break;

		case 0x146d:
			strPtr = "Tachyon, INC.";
			break;

		case 0x146e:
			strPtr = "Williams Electronics Games, Inc.";
			break;

		case 0x146f:
			strPtr = "Multi Dimensional Consulting Inc";
			break;

		case 0x1470:
			strPtr = "Bay Networks";
			break;

		case 0x1471:
			strPtr = "Integrated Telecom Express Inc";
			break;

		case 0x1472:
			strPtr = "DAIKIN Industries, Ltd";
			break;

		case 0x1473:
			strPtr = "ZAPEX Technologies Inc";
			break;

		case 0x1474:
			strPtr = "Doug Carson & Associates";
			break;

		case 0x1475:
			strPtr = "PICAZO Communications";
			break;

		case 0x1476:
			strPtr = "MORTARA Instrument Inc";
			break;

		case 0x1477:
			strPtr = "Net Insight";
			break;

		case 0x1478:
			strPtr = "DIATREND Corporation";
			break;

		case 0x1479:
			strPtr = "TORAY Industries Inc";
			break;

		case 0x147a:
			strPtr = "FORMOSA Industrial Computing";
			break;

		case 0x147b:
			strPtr = "ABIT Computer Corp.";
			break;

		case 0x147c:
			strPtr = "AWARE, Inc.";
			break;

		case 0x147d:
			strPtr = "Interworks Computer Products";
			break;

		case 0x147e:
			strPtr = "Matsushita Graphic Communication Systems, Inc.";
			break;

		case 0x147f:
			strPtr = "NIHON UNISYS, Ltd.";
			break;

		case 0x1480:
			strPtr = "SCII Telecom";
			break;

		case 0x1481:
			strPtr = "BIOPAC Systems Inc";
			break;

		case 0x1482:
			strPtr = "ISYTEC - Integrierte Systemtechnik GmBH";
			break;

		case 0x1483:
			strPtr = "LABWAY Corporation";
			break;

		case 0x1484:
			strPtr = "Logic Corporation";
			break;

		case 0x1485:
			strPtr = "ERMA - Electronic GmBH";
			break;

		case 0x1486:
			strPtr = "L3 Communications Telemetry & Instrumentation";
			break;

		case 0x1487:
			strPtr = "MARQUETTE Medical Systems";
			break;

		case 0x1489:
			strPtr = "KYE Systems Corporation";
			break;

		case 0x148a:
			strPtr = "OPTO";
			break;

		case 0x148b:
			strPtr = "INNOMEDIALOGIC Inc.";
			break;

		case 0x148c:
			strPtr = "Tul Corporation / PowerColor";
			break;

		case 0x148d:
			strPtr = "DIGICOM Systems, Inc.";
			break;

		case 0x148e:
			strPtr = "OSI Plus Corporation";
			break;

		case 0x148f:
			strPtr = "Plant Equipment, Inc.";
			break;

		case 0x1490:
			strPtr = "Stone Microsystems PTY Ltd.";
			break;

		case 0x1491:
			strPtr = "ZEAL Corporation";
			break;

		case 0x1492:
			strPtr = "Time Logic Corporation";
			break;

		case 0x1493:
			strPtr = "MAKER Communications";
			break;

		case 0x1494:
			strPtr = "WINTOP Technology, Inc.";
			break;

		case 0x1495:
			strPtr = "TOKAI Communications Industry Co. Ltd";
			break;

		case 0x1496:
			strPtr = "JOYTECH Computer Co., Ltd.";
			break;

		case 0x1497:
			strPtr = "SMA Regelsysteme GmBH";
			break;

		case 0x1498:
			strPtr = "TEWS Technologies GmbH";
			break;

		case 0x1499:
			strPtr = "EMTEC CO., Ltd";
			break;

		case 0x149a:
			strPtr = "ANDOR Technology Ltd";
			break;

		case 0x149b:
			strPtr = "SEIKO Instruments Inc";
			break;

		case 0x149c:
			strPtr = "OVISLINK Corp.";
			break;

		case 0x149d:
			strPtr = "NEWTEK Inc";
			break;

		case 0x149e:
			strPtr = "Mapletree Networks Inc.";
			break;

		case 0x149f:
			strPtr = "LECTRON Co Ltd";
			break;

		case 0x14a0:
			strPtr = "SOFTING GmBH";
			break;

		case 0x14a1:
			strPtr = "Systembase Co Ltd";
			break;

		case 0x14a2:
			strPtr = "Millennium Engineering Inc";
			break;

		case 0x14a3:
			strPtr = "Maverick Networks";
			break;

		case 0x14a4:
			strPtr = "Lite-On Technology Corporation";
			break;

		case 0x14a5:
			strPtr = "XIONICS Document Technologies Inc";
			break;

		case 0x14a6:
			strPtr = "INOVA Computers GmBH & Co KG";
			break;

		case 0x14a7:
			strPtr = "MYTHOS Systems Inc";
			break;

		case 0x14a8:
			strPtr = "FEATRON Technologies Corporation";
			break;

		case 0x14a9:
			strPtr = "HIVERTEC Inc";
			break;

		case 0x14aa:
			strPtr = "Advanced MOS Technology Inc";
			break;

		case 0x14ab:
			strPtr = "Mentor Graphics Corp.";
			break;

		case 0x14ac:
			strPtr = "Novaweb Technologies Inc";
			break;

		case 0x14ad:
			strPtr = "Time Space Radio AB";
			break;

		case 0x14ae:
			strPtr = "CTI, Inc";
			break;

		case 0x14af:
			strPtr = "Guillemot Corporation";
			break;

		case 0x14b0:
			strPtr = "BST Communication Technology Ltd";
			break;

		case 0x14b1:
			strPtr = "Nextcom K.K.";
			break;

		case 0x14b2:
			strPtr = "ENNOVATE Networks Inc";
			break;

		case 0x14b3:
			strPtr = "XPEED Inc";
			break;

		case 0x14b4:
			strPtr = "PHILIPS Business Electronics B.V.";
			break;

		case 0x14b5:
			strPtr = "Creamware GmBH";
			break;

		case 0x14b6:
			strPtr = "Quantum Data Corp.";
			break;

		case 0x14b7:
			strPtr = "PROXIM Inc";
			break;

		case 0x14b8:
			strPtr = "Techsoft Technology Co Ltd";
			break;

		case 0x14b9:
			strPtr = "Cisco Aironet Wireless Communications";
			break;

		case 0x14ba:
			strPtr = "INTERNIX Inc.";
			break;

		case 0x14bb:
			strPtr = "SEMTECH Corporation";
			break;

		case 0x14bc:
			strPtr = "Globespan Semiconductor Inc.";
			break;

		case 0x14bd:
			strPtr = "CARDIO Control N.V.";
			break;

		case 0x14be:
			strPtr = "L3 Communications";
			break;

		case 0x14bf:
			strPtr = "SPIDER Communications Inc.";
			break;

		case 0x14c0:
			strPtr = "COMPAL Electronics Inc";
			break;

		case 0x14c1:
			strPtr = "MYRICOM Inc.";
			break;

		case 0x14c2:
			strPtr = "DTK Computer";
			break;

		case 0x14c3:
			strPtr = "MEDIATEK Corp.";
			break;

		case 0x14c4:
			strPtr = "IWASAKI Information Systems Co Ltd";
			break;

		case 0x14c5:
			strPtr = "Automation Products AB";
			break;

		case 0x14c6:
			strPtr = "Data Race Inc";
			break;

		case 0x14c7:
			strPtr = "Modular Technology Holdings Ltd";
			break;

		case 0x14c8:
			strPtr = "Turbocomm Tech. Inc.";
			break;

		case 0x14c9:
			strPtr = "ODIN Telesystems Inc";
			break;

		case 0x14ca:
			strPtr = "PE Logic Corp.";
			break;

		case 0x14cb:
			strPtr = "Billionton Systems Inc";
			break;

		case 0x14cc:
			strPtr = "NAKAYO Telecommunications Inc";
			break;

		case 0x14cd:
			strPtr = "Universal Global Scientific Industrial Co.,Ltd";
			break;

		case 0x14ce:
			strPtr = "Whistle Communications";
			break;

		case 0x14cf:
			strPtr = "TEK Microsystems Inc.";
			break;

		case 0x14d0:
			strPtr = "Ericsson Axe R & D";
			break;

		case 0x14d1:
			strPtr = "Computer Hi-Tech Co Ltd";
			break;

		case 0x14d2:
			strPtr = "Titan Electronics Inc";
			break;

		case 0x14d3:
			strPtr = "CIRTECH";
			break;

		case 0x14d4:
			strPtr = "Panacom Technology Corp";
			break;

		case 0x14d5:
			strPtr = "Nitsuko Corporation";
			break;

		case 0x14d6:
			strPtr = "Accusys Inc";
			break;

		case 0x14d7:
			strPtr = "Hirakawa Hewtech Corp";
			break;

		case 0x14d8:
			strPtr = "HOPF Elektronik GmBH";
			break;

		case 0x14d9:
			strPtr = "Alliance Semiconductor Corporation";
			break;

		case 0x14da:
			strPtr = "National Aerospace Laboratories";
			break;

		case 0x14db:
			strPtr = "AFAVLAB Technology Inc";
			break;

		case 0x14dc:
			strPtr = "Amplicon Liveline Ltd";
			break;

		case 0x14dd:
			strPtr = "Boulder Design Labs Inc";
			break;

		case 0x14de:
			strPtr = "Applied Integration Corporation";
			break;

		case 0x14df:
			strPtr = "ASIC Communications Corp";
			break;

		case 0x14e1:
			strPtr = "INVERTEX";
			break;

		case 0x14e2:
			strPtr = "INFOLIBRIA";
			break;

		case 0x14e3:
			strPtr = "AMTELCO";
			break;

		case 0x14e4:
			strPtr = "Broadcom Inc. and subsidiaries";
			break;

		case 0x14e5:
			strPtr = "Pixelfusion Ltd";
			break;

		case 0x14e6:
			strPtr = "SHINING Technology Inc";
			break;

		case 0x14e7:
			strPtr = "3CX";
			break;

		case 0x14e8:
			strPtr = "RAYCER Inc";
			break;

		case 0x14e9:
			strPtr = "GARNETS System CO Ltd";
			break;

		case 0x14ea:
			strPtr = "Planex Communications, Inc";
			break;

		case 0x14eb:
			strPtr = "SEIKO EPSON Corp";
			break;

		case 0x14ec:
			strPtr = "Agilent Technologies";
			break;

		case 0x14ed:
			strPtr = "DATAKINETICS Ltd";
			break;

		case 0x14ee:
			strPtr = "MASPRO KENKOH Corp";
			break;

		case 0x14ef:
			strPtr = "CARRY Computer ENG. CO Ltd";
			break;

		case 0x14f0:
			strPtr = "CANON RESEACH CENTRE FRANCE";
			break;

		case 0x14f1:
			strPtr = "Conexant Systems, Inc.";
			break;

		case 0x14f2:
			strPtr = "MOBILITY Electronics";
			break;

		case 0x14f3:
			strPtr = "BroadLogic";
			break;

		case 0x14f4:
			strPtr = "TOKYO Electronic Industry CO Ltd";
			break;

		case 0x14f5:
			strPtr = "SOPAC Ltd";
			break;

		case 0x14f6:
			strPtr = "COYOTE Technologies LLC";
			break;

		case 0x14f7:
			strPtr = "WOLF Technology Inc";
			break;

		case 0x14f8:
			strPtr = "AUDIOCODES Inc";
			break;

		case 0x14f9:
			strPtr = "AG COMMUNICATIONS";
			break;

		case 0x14fa:
			strPtr = "WANDEL & GOLTERMANN";
			break;

		case 0x14fb:
			strPtr = "TRANSAS MARINE";
			break;

		case 0x14fc:
			strPtr = "Quadrics Ltd";
			break;

		case 0x14fd:
			strPtr = "JAPAN Computer Industry Inc";
			break;

		case 0x14fe:
			strPtr = "ARCHTEK TELECOM Corp";
			break;

		case 0x14ff:
			strPtr = "TWINHEAD INTERNATIONAL Corp";
			break;

		case 0x1500:
			strPtr = "DELTA Electronics, Inc";
			break;

		case 0x1501:
			strPtr = "BANKSOFT CANADA Ltd";
			break;

		case 0x1502:
			strPtr = "MITSUBISHI ELECTRIC LOGISTICS SUPPORT Co Ltd";
			break;

		case 0x1503:
			strPtr = "KAWASAKI LSI USA Inc";
			break;

		case 0x1504:
			strPtr = "KAISER Electronics";
			break;

		case 0x1505:
			strPtr = "ITA INGENIEURBURO FUR TESTAUFGABEN GmbH";
			break;

		case 0x1506:
			strPtr = "CHAMELEON Systems Inc";
			break;

		case 0x1507:
			strPtr = "Motorola ?? / HTEC";
			break;

		case 0x1508:
			strPtr = "HONDA CONNECTORS/MHOTRONICS Inc";
			break;

		case 0x1509:
			strPtr = "FIRST INTERNATIONAL Computer Inc";
			break;

		case 0x150a:
			strPtr = "FORVUS RESEARCH Inc";
			break;

		case 0x150b:
			strPtr = "YAMASHITA Systems Corp";
			break;

		case 0x150c:
			strPtr = "KYOPAL CO Ltd";
			break;

		case 0x150d:
			strPtr = "WARPSPPED Inc";
			break;

		case 0x150e:
			strPtr = "C-PORT Corp";
			break;

		case 0x150f:
			strPtr = "INTEC GmbH";
			break;

		case 0x1510:
			strPtr = "BEHAVIOR TECH Computer Corp";
			break;

		case 0x1511:
			strPtr = "CENTILLIUM Technology Corp";
			break;

		case 0x1512:
			strPtr = "ROSUN Technologies Inc";
			break;

		case 0x1513:
			strPtr = "Raychem";
			break;

		case 0x1514:
			strPtr = "TFL LAN Inc";
			break;

		case 0x1515:
			strPtr = "Advent design";
			break;

		case 0x1516:
			strPtr = "MYSON Technology Inc";
			break;

		case 0x1517:
			strPtr = "ECHOTEK Corp";
			break;

		case 0x1518:
			strPtr = "Kontron";
			break;

		case 0x1519:
			strPtr = "TELEFON AKTIEBOLAGET LM Ericsson";
			break;

		case 0x151a:
			strPtr = "Globetek";
			break;

		case 0x151b:
			strPtr = "COMBOX Ltd";
			break;

		case 0x151c:
			strPtr = "DIGITAL AUDIO LABS Inc";
			break;

		case 0x151d:
			strPtr = "Fujitsu Computer Products Of America";
			break;

		case 0x151e:
			strPtr = "MATRIX Corp";
			break;

		case 0x151f:
			strPtr = "TOPIC SEMICONDUCTOR Corp";
			break;

		case 0x1520:
			strPtr = "CHAPLET System Inc";
			break;

		case 0x1521:
			strPtr = "BELL Corp";
			break;

		case 0x1522:
			strPtr = "MainPine Ltd";
			break;

		case 0x1523:
			strPtr = "MUSIC Semiconductors";
			break;

		case 0x1524:
			strPtr = "ENE Technology Inc";
			break;

		case 0x1525:
			strPtr = "IMPACT Technologies";
			break;

		case 0x1526:
			strPtr = "ISS, Inc";
			break;

		case 0x1527:
			strPtr = "SOLECTRON";
			break;

		case 0x1528:
			strPtr = "ACKSYS";
			break;

		case 0x1529:
			strPtr = "ON Semiconductor";
			break;

		case 0x152a:
			strPtr = "QUICKTURN DESIGN Systems";
			break;

		case 0x152b:
			strPtr = "FLYTECH Technology CO Ltd";
			break;

		case 0x152c:
			strPtr = "MACRAIGOR Systems LLC";
			break;

		case 0x152d:
			strPtr = "QUANTA Computer Inc";
			break;

		case 0x152e:
			strPtr = "MELEC Inc";
			break;

		case 0x152f:
			strPtr = "PHILIPS - CRYPTO";
			break;

		case 0x1530:
			strPtr = "ACQIS Technology Inc";
			break;

		case 0x1531:
			strPtr = "CHRYON Corp";
			break;

		case 0x1532:
			strPtr = "ECHELON Corp";
			break;

		case 0x1533:
			strPtr = "BALTIMORE";
			break;

		case 0x1534:
			strPtr = "ROAD Corp";
			break;

		case 0x1535:
			strPtr = "EVERGREEN Technologies Inc";
			break;

		case 0x1536:
			strPtr = "ACTIS Computer";
			break;

		case 0x1537:
			strPtr = "DATALEX COMMUNCATIONS";
			break;

		case 0x1538:
			strPtr = "ARALION Inc";
			break;

		case 0x1539:
			strPtr = "ATELIER INFORMATIQUES et ELECTRONIQUE ETUDES S.A.";
			break;

		case 0x153a:
			strPtr = "ONO SOKKI";
			break;

		case 0x153b:
			strPtr = "TERRATEC Electronic GmbH";
			break;

		case 0x153c:
			strPtr = "ANTAL Electronic";
			break;

		case 0x153d:
			strPtr = "FILANET Corp";
			break;

		case 0x153e:
			strPtr = "TECHWELL Inc";
			break;

		case 0x153f:
			strPtr = "MIPS Technologies, Inc.";
			break;

		case 0x1540:
			strPtr = "PROVIDEO MULTIMEDIA Co Ltd";
			break;

		case 0x1541:
			strPtr = "MACHONE Communications";
			break;

		case 0x1542:
			strPtr = "Concurrent Real-Time";
			break;

		case 0x1543:
			strPtr = "SILICON Laboratories";
			break;

		case 0x1544:
			strPtr = "DCM DATA Systems";
			break;

		case 0x1545:
			strPtr = "VISIONTEK";
			break;

		case 0x1546:
			strPtr = "IOI Technology Corp";
			break;

		case 0x1547:
			strPtr = "MITUTOYO Corp";
			break;

		case 0x1548:
			strPtr = "JET PROPULSION Laboratory";
			break;

		case 0x1549:
			strPtr = "INTERCONNECT Systems Solutions";
			break;

		case 0x154a:
			strPtr = "MAX Technologies Inc";
			break;

		case 0x154b:
			strPtr = "COMPUTEX Co Ltd";
			break;

		case 0x154c:
			strPtr = "VISUAL Technology Inc";
			break;

		case 0x154d:
			strPtr = "PAN INTERNATIONAL Industrial Corp";
			break;

		case 0x154e:
			strPtr = "SERVOTEST Ltd";
			break;

		case 0x154f:
			strPtr = "STRATABEAM Technology";
			break;

		case 0x1550:
			strPtr = "OPEN NETWORK Co Ltd";
			break;

		case 0x1551:
			strPtr = "SMART Electronic DEVELOPMENT GmBH";
			break;

		case 0x1552:
			strPtr = "RACAL AIRTECH Ltd";
			break;

		case 0x1553:
			strPtr = "CHICONY Electronics Co Ltd";
			break;

		case 0x1554:
			strPtr = "PROLINK Microsystems Corp";
			break;

		case 0x1555:
			strPtr = "GESYTEC GmBH";
			break;

		case 0x1556:
			strPtr = "PLDA";
			break;

		case 0x1557:
			strPtr = "MEDIASTAR Co Ltd";
			break;

		case 0x1558:
			strPtr = "CLEVO/KAPOK Computer";
			break;

		case 0x1559:
			strPtr = "SI LOGIC Ltd";
			break;

		case 0x155a:
			strPtr = "INNOMEDIA Inc";
			break;

		case 0x155b:
			strPtr = "PROTAC INTERNATIONAL Corp";
			break;

		case 0x155c:
			strPtr = "Cemax-Icon Inc";
			break;

		case 0x155d:
			strPtr = "Mac System Co Ltd";
			break;

		case 0x155e:
			strPtr = "LP Elektronik GmbH";
			break;

		case 0x155f:
			strPtr = "Perle Systems Ltd";
			break;

		case 0x1560:
			strPtr = "Terayon Communications Systems";
			break;

		case 0x1561:
			strPtr = "Viewgraphics Inc";
			break;

		case 0x1562:
			strPtr = "Symbol Technologies";
			break;

		case 0x1563:
			strPtr = "A-Trend Technology Co Ltd";
			break;

		case 0x1564:
			strPtr = "Yamakatsu Electronics Industry Co Ltd";
			break;

		case 0x1565:
			strPtr = "Biostar Microtech Int'l Corp";
			break;

		case 0x1566:
			strPtr = "Ardent Technologies Inc";
			break;

		case 0x1567:
			strPtr = "Jungsoft";
			break;

		case 0x1568:
			strPtr = "DDK Electronics Inc";
			break;

		case 0x1569:
			strPtr = "Palit Microsystems Inc.";
			break;

		case 0x156a:
			strPtr = "Avtec Systems";
			break;

		case 0x156b:
			strPtr = "2wire Inc";
			break;

		case 0x156c:
			strPtr = "Vidac Electronics GmbH";
			break;

		case 0x156d:
			strPtr = "Alpha-Top Corp";
			break;

		case 0x156e:
			strPtr = "Alfa Inc";
			break;

		case 0x156f:
			strPtr = "M-Systems Flash Disk Pioneers Ltd";
			break;

		case 0x1570:
			strPtr = "Lecroy Corp";
			break;

		case 0x1571:
			strPtr = "Contemporary Controls";
			break;

		case 0x1572:
			strPtr = "Otis Elevator Company";
			break;

		case 0x1573:
			strPtr = "Lattice - Vantis";
			break;

		case 0x1574:
			strPtr = "Fairchild Semiconductor";
			break;

		case 0x1575:
			strPtr = "Voltaire Advanced Data Security Ltd";
			break;

		case 0x1576:
			strPtr = "Viewcast COM";
			break;

		case 0x1578:
			strPtr = "HITT";
			break;

		case 0x1579:
			strPtr = "Dual Technology Corp";
			break;

		case 0x157a:
			strPtr = "Japan Elecronics Ind Inc";
			break;

		case 0x157b:
			strPtr = "Star Multimedia Corp";
			break;

		case 0x157c:
			strPtr = "Eurosoft";
			break;

		case 0x157d:
			strPtr = "Gemflex Networks";
			break;

		case 0x157e:
			strPtr = "Transition Networks";
			break;

		case 0x157f:
			strPtr = "PX Instruments Technology Ltd";
			break;

		case 0x1580:
			strPtr = "Primex Aerospace Co";
			break;

		case 0x1581:
			strPtr = "SEH Computertechnik GmbH";
			break;

		case 0x1582:
			strPtr = "Cytec Corp";
			break;

		case 0x1583:
			strPtr = "Inet Technologies Inc";
			break;

		case 0x1584:
			strPtr = "Uniwill Computer Corp";
			break;

		case 0x1585:
			strPtr = "Logitron";
			break;

		case 0x1586:
			strPtr = "Lancast Inc";
			break;

		case 0x1587:
			strPtr = "Konica Corp";
			break;

		case 0x1588:
			strPtr = "Solidum Systems Corp";
			break;

		case 0x1589:
			strPtr = "Atlantek Microsystems Pty Ltd";
			break;

		case 0x158a:
			strPtr = "Digalog Systems Inc";
			break;

		case 0x158b:
			strPtr = "Allied Data Technologies";
			break;

		case 0x158c:
			strPtr = "Hitachi Semiconductor & Devices Sales Co Ltd";
			break;

		case 0x158d:
			strPtr = "Point Multimedia Systems";
			break;

		case 0x158e:
			strPtr = "Lara Technology Inc";
			break;

		case 0x158f:
			strPtr = "Ditect Coop";
			break;

		case 0x1590:
			strPtr = "Hewlett Packard Enterprise";
			break;

		case 0x1591:
			strPtr = "ARN";
			break;

		case 0x1592:
			strPtr = "Syba Tech Ltd";
			break;

		case 0x1593:
			strPtr = "Bops Inc";
			break;

		case 0x1594:
			strPtr = "Netgame Ltd";
			break;

		case 0x1595:
			strPtr = "Diva Systems Corp";
			break;

		case 0x1596:
			strPtr = "Folsom Research Inc";
			break;

		case 0x1597:
			strPtr = "Memec Design Services";
			break;

		case 0x1598:
			strPtr = "Granite Microsystems";
			break;

		case 0x1599:
			strPtr = "Delta Electronics Inc";
			break;

		case 0x159a:
			strPtr = "General Instrument";
			break;

		case 0x159b:
			strPtr = "Faraday Technology Corp";
			break;

		case 0x159c:
			strPtr = "Stratus Computer Systems";
			break;

		case 0x159d:
			strPtr = "Ningbo Harrison Electronics Co Ltd";
			break;

		case 0x159e:
			strPtr = "A-Max Technology Co Ltd";
			break;

		case 0x159f:
			strPtr = "Galea Network Security";
			break;

		case 0x15a0:
			strPtr = "Compumaster SRL";
			break;

		case 0x15a1:
			strPtr = "Geocast Network Systems";
			break;

		case 0x15a2:
			strPtr = "Catalyst Enterprises Inc";
			break;

		case 0x15a3:
			strPtr = "Italtel";
			break;

		case 0x15a4:
			strPtr = "X-Net OY";
			break;

		case 0x15a5:
			strPtr = "Toyota Macs Inc";
			break;

		case 0x15a6:
			strPtr = "Sunlight Ultrasound Technologies Ltd";
			break;

		case 0x15a7:
			strPtr = "SSE Telecom Inc";
			break;

		case 0x15a8:
			strPtr = "Shanghai Communications Technologies Center";
			break;

		case 0x15aa:
			strPtr = "Moreton Bay";
			break;

		case 0x15ab:
			strPtr = "Bluesteel Networks Inc";
			break;

		case 0x15ac:
			strPtr = "North Atlantic Instruments";
			break;

		case 0x15ad:
			strPtr = "VMware";
			break;

		case 0x15ae:
			strPtr = "Amersham Pharmacia Biotech";
			break;

		case 0x15b0:
			strPtr = "Zoltrix International Ltd";
			break;

		case 0x15b1:
			strPtr = "Source Technology Inc";
			break;

		case 0x15b2:
			strPtr = "Mosaid Technologies Inc";
			break;

		case 0x15b3:
			strPtr = "Mellanox Technologies";
			break;

		case 0x15b4:
			strPtr = "CCI/TRIAD";
			break;

		case 0x15b5:
			strPtr = "Cimetrics Inc";
			break;

		case 0x15b6:
			strPtr = "Texas Memory Systems Inc";
			break;

		case 0x15b7:
			strPtr = "Sandisk Corp";
			break;

		case 0x15b8:
			strPtr = "ADDI-DATA GmbH";
			break;

		case 0x15b9:
			strPtr = "Maestro Digital Communications";
			break;

		case 0x15ba:
			strPtr = "Impacct Technology Corp";
			break;

		case 0x15bb:
			strPtr = "Portwell Inc";
			break;

		case 0x15bc:
			strPtr = "Agilent Technologies";
			break;

		case 0x15bd:
			strPtr = "DFI Inc";
			break;

		case 0x15be:
			strPtr = "Sola Electronics";
			break;

		case 0x15bf:
			strPtr = "High Tech Computer Corp";
			break;

		case 0x15c0:
			strPtr = "BVM Ltd";
			break;

		case 0x15c1:
			strPtr = "Quantel";
			break;

		case 0x15c2:
			strPtr = "Newer Technology Inc";
			break;

		case 0x15c3:
			strPtr = "Taiwan Mycomp Co Ltd";
			break;

		case 0x15c4:
			strPtr = "EVSX Inc";
			break;

		case 0x15c5:
			strPtr = "Procomp Informatics Ltd";
			break;

		case 0x15c6:
			strPtr = "Technical University of Budapest";
			break;

		case 0x15c7:
			strPtr = "Tateyama System Laboratory Co Ltd";
			break;

		case 0x15c8:
			strPtr = "Penta Media Co Ltd";
			break;

		case 0x15c9:
			strPtr = "Serome Technology Inc";
			break;

		case 0x15ca:
			strPtr = "Bitboys OY";
			break;

		case 0x15cb:
			strPtr = "AG Electronics Ltd";
			break;

		case 0x15cc:
			strPtr = "Hotrail Inc";
			break;

		case 0x15cd:
			strPtr = "Dreamtech Co Ltd";
			break;

		case 0x15ce:
			strPtr = "Genrad Inc";
			break;

		case 0x15cf:
			strPtr = "Hilscher GmbH";
			break;

		case 0x15d1:
			strPtr = "Infineon Technologies AG";
			break;

		case 0x15d2:
			strPtr = "FIC";
			break;

		case 0x15d3:
			strPtr = "NDS Technologies Israel Ltd";
			break;

		case 0x15d4:
			strPtr = "Iwill Corp";
			break;

		case 0x15d5:
			strPtr = "Tatung Co";
			break;

		case 0x15d6:
			strPtr = "Entridia Corp";
			break;

		case 0x15d7:
			strPtr = "Rockwell-Collins Inc";
			break;

		case 0x15d8:
			strPtr = "Cybernetics Technology Co Ltd";
			break;

		case 0x15d9:
			strPtr = "Super Micro Computer Inc";
			break;

		case 0x15da:
			strPtr = "Cyberfirm Inc";
			break;

		case 0x15db:
			strPtr = "Applied Computing Systems Inc";
			break;

		case 0x15dc:
			strPtr = "Litronic Inc";
			break;

		case 0x15dd:
			strPtr = "Sigmatel Inc";
			break;

		case 0x15de:
			strPtr = "Malleable Technologies Inc";
			break;

		case 0x15df:
			strPtr = "Infinilink Corp";
			break;

		case 0x15e0:
			strPtr = "Cacheflow Inc";
			break;

		case 0x15e1:
			strPtr = "Voice Technologies Group Inc";
			break;

		case 0x15e2:
			strPtr = "Quicknet Technologies Inc";
			break;

		case 0x15e3:
			strPtr = "Networth Technologies Inc";
			break;

		case 0x15e4:
			strPtr = "VSN Systemen BV";
			break;

		case 0x15e5:
			strPtr = "Valley technologies Inc";
			break;

		case 0x15e6:
			strPtr = "Agere Inc";
			break;

		case 0x15e7:
			strPtr = "Get Engineering Corp";
			break;

		case 0x15e8:
			strPtr = "National Datacomm Corp";
			break;

		case 0x15e9:
			strPtr = "Pacific Digital Corp";
			break;

		case 0x15ea:
			strPtr = "Tokyo Denshi Sekei K.K.";
			break;

		case 0x15eb:
			strPtr = "DResearch Digital Media Systems GmbH";
			break;

		case 0x15ec:
			strPtr = "Beckhoff GmbH";
			break;

		case 0x15ed:
			strPtr = "Macrolink Inc";
			break;

		case 0x15ee:
			strPtr = "In Win Development Inc";
			break;

		case 0x15ef:
			strPtr = "Intelligent Paradigm Inc";
			break;

		case 0x15f0:
			strPtr = "B-Tree Systems Inc";
			break;

		case 0x15f1:
			strPtr = "Times N Systems Inc";
			break;

		case 0x15f2:
			strPtr = "Diagnostic Instruments Inc";
			break;

		case 0x15f3:
			strPtr = "Digitmedia Corp";
			break;

		case 0x15f4:
			strPtr = "Valuesoft";
			break;

		case 0x15f5:
			strPtr = "Power Micro Research";
			break;

		case 0x15f6:
			strPtr = "Extreme Packet Device Inc";
			break;

		case 0x15f7:
			strPtr = "Banctec";
			break;

		case 0x15f8:
			strPtr = "Koga Electronics Co";
			break;

		case 0x15f9:
			strPtr = "Zenith Electronics Corp";
			break;

		case 0x15fa:
			strPtr = "J.P. Axzam Corp";
			break;

		case 0x15fb:
			strPtr = "Zilog Inc";
			break;

		case 0x15fc:
			strPtr = "Techsan Electronics Co Ltd";
			break;

		case 0x15fd:
			strPtr = "N-CUBED.NET";
			break;

		case 0x15fe:
			strPtr = "Kinpo Electronics Inc";
			break;

		case 0x15ff:
			strPtr = "Fastpoint Technologies Inc";
			break;

		case 0x1600:
			strPtr = "Northrop Grumman - Canada Ltd";
			break;

		case 0x1601:
			strPtr = "Tenta Technology";
			break;

		case 0x1602:
			strPtr = "Prosys-tec Inc";
			break;

		case 0x1603:
			strPtr = "Nokia Wireless Communications";
			break;

		case 0x1604:
			strPtr = "Central System Research Co Ltd";
			break;

		case 0x1605:
			strPtr = "Pairgain Technologies";
			break;

		case 0x1606:
			strPtr = "Europop AG";
			break;

		case 0x1607:
			strPtr = "Lava Semiconductor Manufacturing Inc";
			break;

		case 0x1608:
			strPtr = "Automated Wagering International";
			break;

		case 0x1609:
			strPtr = "Scimetric Instruments Inc";
			break;

		case 0x1612:
			strPtr = "Telesynergy Research Inc.";
			break;

		case 0x1618:
			strPtr = "Stone Ridge Technology";
			break;

		case 0x1619:
			strPtr = "FarSite Communications Ltd";
			break;

		case 0x161f:
			strPtr = "Rioworks";
			break;

		case 0x1621:
			strPtr = "Lynx Studio Technology, Inc.";
			break;

		case 0x1626:
			strPtr = "TDK Semiconductor Corp.";
			break;

		case 0x1629:
			strPtr = "Kongsberg Spacetec AS";
			break;

		case 0x1631:
			strPtr = "Packard Bell B.V.";
			break;

		case 0x1638:
			strPtr = "Standard Microsystems Corp [SMC]";
			break;

		case 0x163c:
			strPtr = "Smart Link Ltd.";
			break;

		case 0x1641:
			strPtr = "MKNet Corp.";
			break;

		case 0x1642:
			strPtr = "Bitlan";
			break;

		case 0x1657:
			strPtr = "Cavium QLogic";
			break;

		case 0x165a:
			strPtr = "Epix Inc";
			break;

		case 0x165c:
			strPtr = "Gidel Ltd.";
			break;

		case 0x165d:
			strPtr = "Hsing Tech. Enterprise Co., Ltd.";
			break;

		case 0x165f:
			strPtr = "Linux Media Labs, LLC";
			break;

		case 0x1661:
			strPtr = "Worldspace Corp.";
			break;

		case 0x1668:
			strPtr = "Actiontec Electronics Inc";
			break;

		case 0x166d:
			strPtr = "Broadcom Corporation";
			break;

		case 0x1677:
			strPtr = "Bernecker + Rainer";
			break;

		case 0x1678:
			strPtr = "NetEffect";
			break;

		case 0x1679:
			strPtr = "Tokyo Electron Device Ltd.";
			break;

		case 0x167b:
			strPtr = "ZyDAS Technology Corp.";
			break;

		case 0x167d:
			strPtr = "Samsung Electro-Mechanics Co., Ltd.";
			break;

		case 0x167e:
			strPtr = "ONNTO Corp.";
			break;

		case 0x1681:
			strPtr = "Hercules";
			break;

		case 0x1682:
			strPtr = "XFX Pine Group Inc.";
			break;

		case 0x1688:
			strPtr = "CastleNet Technology Inc.";
			break;

		case 0x168a:
			strPtr = "Utimaco IS GmbH";
			break;

		case 0x168c:
			strPtr = "Qualcomm Atheros";
			break;

		case 0x1695:
			strPtr = "EPoX Computer Co., Ltd.";
			break;

		case 0x169c:
			strPtr = "Netcell Corporation";
			break;

		case 0x169d:
			strPtr = "Club-3D VB";
			break;

		case 0x16a5:
			strPtr = "Tekram Technology Co.,Ltd.";
			break;

		case 0x16ab:
			strPtr = "Global Sun Technology Inc";
			break;

		case 0x16ae:
			strPtr = "SafeNet Inc";
			break;

		case 0x16af:
			strPtr = "SparkLAN Communications, Inc.";
			break;

		case 0x16b4:
			strPtr = "Aspex Semiconductor Ltd";
			break;

		case 0x16b8:
			strPtr = "Sonnet Technologies, Inc.";
			break;

		case 0x16be:
			strPtr = "Creatix Polymedia GmbH";
			break;

		case 0x16c3:
			strPtr = "Synopsys, Inc.";
			break;

		case 0x16c6:
			strPtr = "Micrel-Kendin";
			break;

		case 0x16c8:
			strPtr = "Octasic Inc.";
			break;

		case 0x16c9:
			strPtr = "EONIC B.V. The Netherlands";
			break;

		case 0x16ca:
			strPtr = "CENATEK Inc";
			break;

		case 0x16cd:
			strPtr = "Advantech Co. Ltd";
			break;

		case 0x16ce:
			strPtr = "Roland Corp.";
			break;

		case 0x16d5:
			strPtr = "Acromag, Inc.";
			break;

		case 0x16da:
			strPtr = "Advantech Co., Ltd.";
			break;

		case 0x16df:
			strPtr = "PIKA Technologies Inc.";
			break;

		case 0x16e2:
			strPtr = "Geotest-MTS";
			break;

		case 0x16e3:
			strPtr = "European Space Agency";
			break;

		case 0x16e5:
			strPtr = "Intellon Corp.";
			break;

		case 0x16ec:
			strPtr = "U.S. Robotics";
			break;

		case 0x16ed:
			strPtr = "Sycron N. V.";
			break;

		case 0x16f2:
			strPtr = "ETAS GmbH";
			break;

		case 0x16f3:
			strPtr = "Jetway Information Co., Ltd.";
			break;

		case 0x16f4:
			strPtr = "Vweb Corp";
			break;

		case 0x16f6:
			strPtr = "VideoTele.com, Inc.";
			break;

		case 0x1702:
			strPtr = "Internet Machines Corporation";
			break;

		case 0x1705:
			strPtr = "Digital First, Inc.";
			break;

		case 0x170b:
			strPtr = "NetOctave";
			break;

		case 0x170c:
			strPtr = "YottaYotta Inc.";
			break;

		case 0x1719:
			strPtr = "EZChip Technologies";
			break;

		case 0x1725:
			strPtr = "Vitesse Semiconductor";
			break;

		case 0x172a:
			strPtr = "Accelerated Encryption";
			break;

		case 0x1734:
			strPtr = "Fujitsu Technology Solutions";
			break;

		case 0x1735:
			strPtr = "Aten International Co. Ltd.";
			break;

		case 0x1737:
			strPtr = "Linksys";
			break;

		case 0x173b:
			strPtr = "Altima";
			break;

		case 0x1743:
			strPtr = "Peppercon AG";
			break;

		case 0x1745:
			strPtr = "ViXS Systems, Inc.";
			break;

		case 0x1749:
			strPtr = "RLX Technologies";
			break;

		case 0x174b:
			strPtr = "PC Partner Limited / Sapphire Technology";
			break;

		case 0x174d:
			strPtr = "WellX Telecom SA";
			break;

		case 0x175c:
			strPtr = "AudioScience Inc";
			break;

		case 0x175e:
			strPtr = "Sanera Systems, Inc.";
			break;

		case 0x1760:
			strPtr = "TEDIA spol. s r. o.";
			break;

		case 0x1761:
			strPtr = "Pickering Interfaces Ltd";
			break;

		case 0x1771:
			strPtr = "InnoVISION Multimedia Ltd.";
			break;

		case 0x1775:
			strPtr = "General Electric";
			break;

		case 0x177d:
			strPtr = "Cavium, Inc.";
			break;

		case 0x1787:
			strPtr = "Hightech Information System Ltd.";
			break;

		case 0x1789:
			strPtr = "Ennyah Technologies Corp.";
			break;

		case 0x1796:
			strPtr = "Research Centre Juelich";
			break;

		case 0x1797:
			strPtr = "Intersil Techwell";
			break;

		case 0x1799:
			strPtr = "Belkin";
			break;

		case 0x179a:
			strPtr = "id Quantique";
			break;

		case 0x179c:
			strPtr = "Data Patterns";
			break;

		case 0x17a0:
			strPtr = "Genesys Logic, Inc";
			break;

		case 0x17aa:
			strPtr = "Lenovo";
			break;

		case 0x17ab:
			strPtr = "Phillips Components";
			break;

		case 0x17af:
			strPtr = "Hightech Information System Ltd.";
			break;

		case 0x17b3:
			strPtr = "Hawking Technologies";
			break;

		case 0x17b4:
			strPtr = "Indra Networks, Inc.";
			break;

		case 0x17c0:
			strPtr = "Wistron Corp.";
			break;

		case 0x17c2:
			strPtr = "Newisys, Inc.";
			break;

		case 0x17cb:
			strPtr = "Qualcomm";
			break;

		case 0x17cc:
			strPtr = "NetChip Technology, Inc";
			break;

		case 0x17cd:
			strPtr = "Cadence Design Systems, Inc.";
			break;

		case 0x17cf:
			strPtr = "Z-Com, Inc.";
			break;

		case 0x17d3:
			strPtr = "Areca Technology Corp.";
			break;

		case 0x17d5:
			strPtr = "Exar Corp.";
			break;

		case 0x17db:
			strPtr = "Cray Inc";
			break;

		case 0x17de:
			strPtr = "KWorld Computer Co. Ltd.";
			break;

		case 0x17df:
			strPtr = "Dini Group";
			break;

		case 0x17e4:
			strPtr = "Sectra AB";
			break;

		case 0x17e6:
			strPtr = "MaxLinear";
			break;

		case 0x17ee:
			strPtr = "Connect Components Ltd";
			break;

		case 0x17f2:
			strPtr = "Albatron Corp.";
			break;

		case 0x17f3:
			strPtr = "RDC Semiconductor, Inc.";
			break;

		case 0x17f7:
			strPtr = "Topdek Semiconductor Inc.";
			break;

		case 0x17f9:
			strPtr = "Gemtek Technology Co., Ltd";
			break;

		case 0x17fc:
			strPtr = "IOGEAR, Inc.";
			break;

		case 0x17fe:
			strPtr = "InProComm Inc.";
			break;

		case 0x17ff:
			strPtr = "Benq Corporation";
			break;

		case 0x1800:
			strPtr = "Qualcore Logic Inc.";
			break;

		case 0x1803:
			strPtr = "ProdaSafe GmbH";
			break;

		case 0x1804:
			strPtr = "Ralink corp.";
			break;

		case 0x1805:
			strPtr = "Euresys S.A.";
			break;

		case 0x1809:
			strPtr = "Lumanate, Inc.";
			break;

		case 0x180c:
			strPtr = "IEI Integration Corp";
			break;

		case 0x1813:
			strPtr = "Ambient Technologies Inc";
			break;

		case 0x1814:
			strPtr = "Ralink corp.";
			break;

		case 0x1815:
			strPtr = "Devolo AG";
			break;

		case 0x1820:
			strPtr = "InfiniCon Systems Inc.";
			break;

		case 0x1822:
			strPtr = "Twinhan Technology Co. Ltd";
			break;

		case 0x182d:
			strPtr = "SiteCom Europe BV";
			break;

		case 0x182e:
			strPtr = "Raza Microelectronics, Inc.";
			break;

		case 0x182f:
			strPtr = "Broadcom";
			break;

		case 0x1830:
			strPtr = "Credence Systems Corporation";
			break;

		case 0x183b:
			strPtr = "MikroM GmbH";
			break;

		case 0x1846:
			strPtr = "Alcatel-Lucent";
			break;

		case 0x1849:
			strPtr = "ASRock Incorporation";
			break;

		case 0x184a:
			strPtr = "Thales Computers";
			break;

		case 0x1850:
			strPtr = "Advantest Corporation";
			break;

		case 0x1851:
			strPtr = "Microtune, Inc.";
			break;

		case 0x1852:
			strPtr = "Anritsu Corp.";
			break;

		case 0x1853:
			strPtr = "SMSC Automotive Infotainment System Group";
			break;

		case 0x1854:
			strPtr = "LG Electronics, Inc.";
			break;

		case 0x185b:
			strPtr = "Compro Technology, Inc.";
			break;

		case 0x185f:
			strPtr = "Wistron NeWeb Corp.";
			break;

		case 0x1864:
			strPtr = "SilverBack";
			break;

		case 0x1867:
			strPtr = "Topspin Communications";
			break;

		case 0x186c:
			strPtr = "Humusoft, s.r.o.";
			break;

		case 0x186f:
			strPtr = "WiNRADiO Communications";
			break;

		case 0x1876:
			strPtr = "L-3 Communications";
			break;

		case 0x187e:
			strPtr = "ZyXEL Communications Corporation";
			break;

		case 0x1885:
			strPtr = "Avvida Systems Inc.";
			break;

		case 0x1888:
			strPtr = "Varisys Ltd";
			break;

		case 0x188a:
			strPtr = "Ample Communications, Inc";
			break;

		case 0x1890:
			strPtr = "Egenera, Inc.";
			break;

		case 0x1894:
			strPtr = "KNC One";
			break;

		case 0x1896:
			strPtr = "B&B Electronics Manufacturing Company, Inc.";
			break;

		case 0x1897:
			strPtr = "AMtek";
			break;

		case 0x18a1:
			strPtr = "Astute Networks Inc.";
			break;

		case 0x18a2:
			strPtr = "Stretch Inc.";
			break;

		case 0x18a3:
			strPtr = "AT&T";
			break;

		case 0x18ac:
			strPtr = "DViCO Corporation";
			break;

		case 0x18b8:
			strPtr = "Ammasso";
			break;

		case 0x18bc:
			strPtr = "GeCube Technologies, Inc.";
			break;

		case 0x18c3:
			strPtr = "Micronas Semiconductor Holding AG";
			break;

		case 0x18c8:
			strPtr = "Cray Inc";
			break;

		case 0x18c9:
			strPtr = "ARVOO Engineering BV";
			break;

		case 0x18ca:
			strPtr = "XGI Technology Inc.";
			break;

		case 0x18d2:
			strPtr = "Sitecom Europe BV";
			break;

		case 0x18d4:
			strPtr = "Celestica";
			break;

		case 0x18d8:
			strPtr = "Dialogue Technology Corp.";
			break;

		case 0x18dd:
			strPtr = "Artimi Inc";
			break;

		case 0x18df:
			strPtr = "LeWiz Communications";
			break;

		case 0x18e6:
			strPtr = "MPL AG";
			break;

		case 0x18eb:
			strPtr = "Advance Multimedia Internet Technology, Inc.";
			break;

		case 0x18ec:
			strPtr = "Cesnet, z.s.p.o.";
			break;

		case 0x18ee:
			strPtr = "Chenming Mold Ind. Corp.";
			break;

		case 0x18f1:
			strPtr = "Spectrum GmbH";
			break;

		case 0x18f4:
			strPtr = "Napatech A/S";
			break;

		case 0x18f6:
			strPtr = "NextIO";
			break;

		case 0x18f7:
			strPtr = "Commtech, Inc.";
			break;

		case 0x18fb:
			strPtr = "Resilience Corporation";
			break;

		case 0x1904:
			strPtr = "Hangzhou Silan Microelectronics Co., Ltd.";
			break;

		case 0x1905:
			strPtr = "Micronas USA, Inc.";
			break;

		case 0x1912:
			strPtr = "Renesas Technology Corp.";
			break;

		case 0x1919:
			strPtr = "Soltek Computer Inc.";
			break;

		case 0x1923:
			strPtr = "Sangoma Technologies Corp.";
			break;

		case 0x1924:
			strPtr = "Solarflare Communications";
			break;

		case 0x192a:
			strPtr = "BiTMICRO Networks Inc.";
			break;

		case 0x192e:
			strPtr = "TransDimension";
			break;

		case 0x1931:
			strPtr = "Option N.V.";
			break;

		case 0x1932:
			strPtr = "DiBcom";
			break;

		case 0x193c:
			strPtr = "MAXIM Integrated Products";
			break;

		case 0x193d:
			strPtr = "Hangzhou H3C Technologies Co., Ltd.";
			break;

		case 0x193f:
			strPtr = "AHA Products Group";
			break;

		case 0x1942:
			strPtr = "ClearSpeed Technology plc";
			break;

		case 0x1947:
			strPtr = "C-guys, Inc.";
			break;

		case 0x1948:
			strPtr = "Alpha Networks Inc.";
			break;

		case 0x194a:
			strPtr = "DapTechnology B.V.";
			break;

		case 0x1954:
			strPtr = "One Stop Systems, Inc.";
			break;

		case 0x1957:
			strPtr = "Freescale Semiconductor Inc";
			break;

		case 0x1958:
			strPtr = "Faster Technology, LLC.";
			break;

		case 0x1959:
			strPtr = "PA Semi, Inc";
			break;

		case 0x1966:
			strPtr = "Orad Hi-Tec Systems";
			break;

		case 0x1969:
			strPtr = "Qualcomm Atheros";
			break;

		case 0x196a:
			strPtr = "Sensory Networks Inc.";
			break;

		case 0x196d:
			strPtr = "Club-3D BV";
			break;

		case 0x196e:
			strPtr = "PNY";
			break;

		case 0x1971:
			strPtr = "AGEIA Technologies, Inc.";
			break;

		case 0x1974:
			strPtr = "Star Electronics GmbH & Co. KG";
			break;

		case 0x1976:
			strPtr = "TRENDnet";
			break;

		case 0x1977:
			strPtr = "Parsec";
			break;

		case 0x197b:
			strPtr = "JMicron Technology Corp.";
			break;

		case 0x1982:
			strPtr = "Distant Early Warning Communications Inc";
			break;

		case 0x1987:
			strPtr = "Phison Electronics Corporation";
			break;

		case 0x1989:
			strPtr = "Montilio Inc.";
			break;

		case 0x198a:
			strPtr = "Nallatech Ltd.";
			break;

		case 0x1993:
			strPtr = "Innominate Security Technologies AG";
			break;

		case 0x1999:
			strPtr = "A-Logics";
			break;

		case 0x199a:
			strPtr = "Pulse-LINK, Inc.";
			break;

		case 0x199d:
			strPtr = "Xsigo Systems";
			break;

		case 0x199f:
			strPtr = "Auvitek";
			break;

		case 0x19a2:
			strPtr = "Emulex Corporation";
			break;

		case 0x19a8:
			strPtr = "DAQDATA GmbH";
			break;

		case 0x19ac:
			strPtr = "Kasten Chase Applied Research";
			break;

		case 0x19ae:
			strPtr = "Progeny Systems Corporation";
			break;

		case 0x19ba:
			strPtr = "ZyXEL Communications Corp.";
			break;

		case 0x19c1:
			strPtr = "Exegy Inc.";
			break;

		case 0x19d1:
			strPtr = "Motorola Expedience";
			break;

		case 0x19d4:
			strPtr = "Quixant Limited";
			break;

		case 0x19da:
			strPtr = "ZOTAC International";
			break;

		case 0x19de:
			strPtr = "Pico Computing";
			break;

		case 0x19e2:
			strPtr = "Vector Informatik GmbH";
			break;

		case 0x19e3:
			strPtr = "DDRdrive LLC";
			break;

		case 0x19e5:
			strPtr = "Huawei Technologies Co., Ltd.";
			break;

		case 0x19e7:
			strPtr = "NET";
			break;

		case 0x19ee:
			strPtr = "Netronome Systems, Inc.";
			break;

		case 0x19f1:
			strPtr = "BFG Tech";
			break;

		case 0x19ff:
			strPtr = "Eclipse Electronic Systems, Inc.";
			break;

		case 0x1a03:
			strPtr = "ASPEED Technology, Inc.";
			break;

		case 0x1a05:
			strPtr = "deltaww";
			break;

		case 0x1a07:
			strPtr = "Kvaser AB";
			break;

		case 0x1a08:
			strPtr = "Sierra semiconductor";
			break;

		case 0x1a0d:
			strPtr = "SEAKR Engineering";
			break;

		case 0x1a0e:
			strPtr = "DekTec Digital Video B.V.";
			break;

		case 0x1a17:
			strPtr = "Force10 Networks, Inc.";
			break;

		case 0x1a1d:
			strPtr = "GFaI e.V.";
			break;

		case 0x1a1e:
			strPtr = "3Leaf Systems, Inc.";
			break;

		case 0x1a22:
			strPtr = "Ambric Inc.";
			break;

		case 0x1a29:
			strPtr = "Fortinet, Inc.";
			break;

		case 0x1a2b:
			strPtr = "Ascom AG";
			break;

		case 0x1a30:
			strPtr = "Lantiq";
			break;

		case 0x1a32:
			strPtr = "Quanta Microsystems, Inc";
			break;

		case 0x1a3b:
			strPtr = "AzureWave";
			break;

		case 0x1a41:
			strPtr = "Tilera Corp.";
			break;

		case 0x1a4a:
			strPtr = "SLAC National Accelerator Lab TID-AIR";
			break;

		case 0x1a51:
			strPtr = "Hectronic AB";
			break;

		case 0x1a55:
			strPtr = "Rohde & Schwarz DVS GmbH";
			break;

		case 0x1a56:
			strPtr = "Rivet Networks";
			break;

		case 0x1a57:
			strPtr = "Highly Reliable Systems";
			break;

		case 0x1a58:
			strPtr = "Razer USA Ltd.";
			break;

		case 0x1a5d:
			strPtr = "Celoxica";
			break;

		case 0x1a5e:
			strPtr = "Aprius Inc.";
			break;

		case 0x1a5f:
			strPtr = "System TALKS Inc.";
			break;

		case 0x1a68:
			strPtr = "VirtenSys Limited";
			break;

		case 0x1a71:
			strPtr = "XenSource, Inc.";
			break;

		case 0x1a73:
			strPtr = "Violin Memory, Inc";
			break;

		case 0x1a76:
			strPtr = "Wavesat";
			break;

		case 0x1a77:
			strPtr = "Lightfleet Corporation";
			break;

		case 0x1a78:
			strPtr = "Virident Systems Inc.";
			break;

		case 0x1a84:
			strPtr = "Commex Technologies";
			break;

		case 0x1a88:
			strPtr = "MEN Mikro Elektronik";
			break;

		case 0x1a8a:
			strPtr = "StarBridge, Inc.";
			break;

		case 0x1a8c:
			strPtr = "Verigy Pte. Ltd.";
			break;

		case 0x1a8e:
			strPtr = "DRS Technologies";
			break;

		case 0x1aa8:
			strPtr = "Ciprico, Inc.";
			break;

		case 0x1aae:
			strPtr = "Global Velocity, Inc.";
			break;

		case 0x1ab4:
			strPtr = "Distributed Management Task Force, Inc.";
			break;

		case 0x1ab6:
			strPtr = "CalDigit, Inc.";
			break;

		case 0x1ab8:
			strPtr = "Parallels, Inc.";
			break;

		case 0x1ab9:
			strPtr = "Espia Srl";
			break;

		case 0x1ac1:
			strPtr = "Global Unichip Corp.";
			break;

		case 0x1ac8:
			strPtr = "Aeroflex Gaisler";
			break;

		case 0x1acc:
			strPtr = "Point of View BV";
			break;

		case 0x1ad7:
			strPtr = "Spectracom Corporation";
			break;

		case 0x1ade:
			strPtr = "Spin Master Ltd.";
			break;

		case 0x1ae0:
			strPtr = "Google, Inc.";
			break;

		case 0x1ae3:
			strPtr = "SANBlaze Technology, Inc.";
			break;

		case 0x1ae7:
			strPtr = "First Wise Media GmbH";
			break;

		case 0x1ae8:
			strPtr = "Silicon Software GmbH";
			break;

		case 0x1ae9:
			strPtr = "Wilocity Ltd.";
			break;

		case 0x1aea:
			strPtr = "Alcor Micro";
			break;

		case 0x1aec:
			strPtr = "Wolfson Microelectronics";
			break;

		case 0x1aed:
			strPtr = "SanDisk";
			break;

		case 0x1aee:
			strPtr = "Caustic Graphics Inc.";
			break;

		case 0x1af4:
			strPtr = "Red Hat, Inc.";
			break;

		case 0x1af5:
			strPtr = "Netezza Corp.";
			break;

		case 0x1afa:
			strPtr = "J & W Electronics Co., Ltd.";
			break;

		case 0x1b03:
			strPtr = "Magnum Semiconductor, Inc,";
			break;

		case 0x1b08:
			strPtr = "MSC Technologies GmbH";
			break;

		case 0x1b0a:
			strPtr = "Pegatron";
			break;

		case 0x1b13:
			strPtr = "Jaton Corp";
			break;

		case 0x1b1a:
			strPtr = "K&F Computing Research Co.";
			break;

		case 0x1b1c:
			strPtr = "Corsair";
			break;

		case 0x1b21:
			strPtr = "ASMedia Technology Inc.";
			break;

		case 0x1b26:
			strPtr = "Netcope Technologies, a.s.";
			break;

		case 0x1b2c:
			strPtr = "Opal-RT Technologies Inc.";
			break;

		case 0x1b36:
			strPtr = "Red Hat, Inc.";
			break;

		case 0x1b37:
			strPtr = "Signal Processing Devices Sweden AB";
			break;

		case 0x1b39:
			strPtr = "sTec, Inc.";
			break;

		case 0x1b3a:
			strPtr = "Westar Display Technologies";
			break;

		case 0x1b3e:
			strPtr = "Teradata Corp.";
			break;

		case 0x1b40:
			strPtr = "Schooner Information Technology, Inc.";
			break;

		case 0x1b47:
			strPtr = "Numascale AS";
			break;

		case 0x1b4b:
			strPtr = "Marvell Technology Group Ltd.";
			break;

		case 0x1b4c:
			strPtr = "GALAX";
			break;

		case 0x1b55:
			strPtr = "NetUP Inc.";
			break;

		case 0x1b66:
			strPtr = "DELTACAST";
			break;

		case 0x1b6f:
			strPtr = "Etron Technology, Inc.";
			break;

		case 0x1b73:
			strPtr = "Fresco Logic";
			break;

		case 0x1b74:
			strPtr = "OpenVox Communication Co. Ltd.";
			break;

		case 0x1b79:
			strPtr = "Absolute Analysis";
			break;

		case 0x1b85:
			strPtr = "OCZ Technology Group, Inc.";
			break;

		case 0x1b94:
			strPtr = "Signatec / Dynamic Signals Corp";
			break;

		case 0x1b96:
			strPtr = "Western Digital";
			break;

		case 0x1b9a:
			strPtr = "XAVi Technologies Corp.";
			break;

		case 0x1baa:
			strPtr = "QNAP Systems, Inc.";
			break;

		case 0x1bad:
			strPtr = "ReFLEX CES";
			break;

		case 0x1bb0:
			strPtr = "SimpliVity Corporation";
			break;

		case 0x1bb1:
			strPtr = "Seagate Technology PLC";
			break;

		case 0x1bb3:
			strPtr = "Bluecherry";
			break;

		case 0x1bb5:
			strPtr = "Quantenna Communications, Inc.";
			break;

		case 0x1bbf:
			strPtr = "Maxeler Technologies Ltd.";
			break;

		case 0x1bc0:
			strPtr = "Innodisk Corporation";
			break;

		case 0x1bcf:
			strPtr = "NEC Corporation";
			break;

		case 0x1bd0:
			strPtr = "Astronics Corporation";
			break;

		case 0x1bd4:
			strPtr = "Inspur Electronic Information Industry Co., Ltd.";
			break;

		case 0x1bee:
			strPtr = "IXXAT Automation GmbH";
			break;

		case 0x1bef:
			strPtr = "Lantiq";
			break;

		case 0x1bf4:
			strPtr = "VTI Instruments Corporation";
			break;

		case 0x1bfd:
			strPtr = "EeeTOP";
			break;

		case 0x1c09:
			strPtr = "CSP, Inc.";
			break;

		case 0x1c1c:
			strPtr = "Symphony";
			break;

		case 0x1c1f:
			strPtr = "SoftLab-NSK";
			break;

		case 0x1c28:
			strPtr = "Lite-On IT Corp. / Plextor";
			break;

		case 0x1c2c:
			strPtr = "Silicom Denmark";
			break;

		case 0x1c32:
			strPtr = "Highland Technology, Inc.";
			break;

		case 0x1c33:
			strPtr = "Daktronics, Inc";
			break;

		case 0x1c36:
			strPtr = "Annapurna Labs Ltd.";
			break;

		case 0x1c3b:
			strPtr = "Accensus, LLC";
			break;

		case 0x1c44:
			strPtr = "Enmotus Inc";
			break;

		case 0x1c58:
			strPtr = "HGST, Inc.";
			break;

		case 0x1c5c:
			strPtr = "SK hynix";
			break;

		case 0x1c5f:
			strPtr = "Beijing Memblaze Technology Co. Ltd.";
			break;

		case 0x1c63:
			strPtr = "Science and Research Centre of Computer Technology";
			break;

		case 0x1c7e:
			strPtr = "TTTech Computertechnik AG";
			break;

		case 0x1c7f:
			strPtr = "Elektrobit Austria GmbH";
			break;

		case 0x1c8a:
			strPtr = "TSF5 Corporation";
			break;

		case 0x1c8c:
			strPtr = "Mobiveil, Inc.";
			break;

		case 0x1cb0:
			strPtr = "Shannon Systems";
			break;

		case 0x1cb1:
			strPtr = "Collion UG & Co.KG";
			break;

		case 0x1cb5:
			strPtr = "Focusrite Audio Engineering Ltd";
			break;

		case 0x1cb8:
			strPtr = "Dawning Information Industry Co., Ltd.";
			break;

		case 0x1cc1:
			strPtr = "ADATA Technology Co., Ltd.";
			break;

		case 0x1cc4:
			strPtr = "Union Memory";
			break;

		case 0x1cc5:
			strPtr = "Embedded Intelligence, Inc.";
			break;

		case 0x1cc7:
			strPtr = "Radian Memory Systems Inc.";
			break;

		case 0x1ccf:
			strPtr = "Zoom Corporation";
			break;

		case 0x1cd2:
			strPtr = "SesKion GmbH";
			break;

		case 0x1cd7:
			strPtr = "Nanjing Magewell Electronics Co., Ltd.";
			break;

		case 0x1cdd:
			strPtr = "secunet Security Networks AG";
			break;

		case 0x1ce4:
			strPtr = "Exablaze";
			break;

		case 0x1cf0:
			strPtr = "Akitio";
			break;

		case 0x1cf7:
			strPtr = "Subspace Dynamics";
			break;

		case 0x1d00:
			strPtr = "Pure Storage";
			break;

		case 0x1d05:
			strPtr = "Tongfang Hongkong Limited";
			break;

		case 0x1d0f:
			strPtr = "Amazon.com, Inc.";
			break;

		case 0x1d17:
			strPtr = "Zhaoxin";
			break;

		case 0x1d18:
			strPtr = "RME";
			break;

		case 0x1d1d:
			strPtr = "CNEX Labs";
			break;

		case 0x1d21:
			strPtr = "Allo";
			break;

		case 0x1d22:
			strPtr = "Baidu Technology";
			break;

		case 0x1d26:
			strPtr = "Kalray Inc.";
			break;

		case 0x1d37:
			strPtr = "NovaSparks";
			break;

		case 0x1d40:
			strPtr = "Techman Electronics";
			break;

		case 0x1d44:
			strPtr = "DPT";
			break;

		case 0x1d49:
			strPtr = "Lenovo";
			break;

		case 0x1d4c:
			strPtr = "Diamanti, Inc.";
			break;

		case 0x1d5c:
			strPtr = "Fantasia Trading LLC";
			break;

		case 0x1d61:
			strPtr = "Technobox, Inc.";
			break;

		case 0x1d62:
			strPtr = "Nebbiolo Technologies";
			break;

		case 0x1d65:
			strPtr = "Imagine Communications Corp.";
			break;

		case 0x1d6a:
			strPtr = "Aquantia Corp.";
			break;

		case 0x1d6c:
			strPtr = "Atomic Rules LLC";
			break;

		case 0x1d72:
			strPtr = "Xiaomi";
			break;

		case 0x1d78:
			strPtr = "DERA Storage";
			break;

		case 0x1d7c:
			strPtr = "Aerotech, Inc.";
			break;

		case 0x1d82:
			strPtr = "NETINT Technologies Inc.";
			break;

		case 0x1d87:
			strPtr = "Rockchip Electronics Co., Ltd";
			break;

		case 0x1d8f:
			strPtr = "Enyx";
			break;

		case 0x1d93:
			strPtr = "YADRO";
			break;

		case 0x1d94:
			strPtr = "Chengdu Haiguang IC Design Co., Ltd.";
			break;

		case 0x1d95:
			strPtr = "Graphcore Ltd";
			break;

		case 0x1d97:
			strPtr = "Shenzhen Longsys Electronics Co., Ltd.";
			break;

		case 0x1d9b:
			strPtr = "Facebook, Inc.";
			break;

		case 0x1da1:
			strPtr = "Teko Telecom S.r.l.";
			break;

		case 0x1da2:
			strPtr = "Sapphire Technology Limited";
			break;

		case 0x1da3:
			strPtr = "Habana Labs Ltd.";
			break;

		case 0x1db2:
			strPtr = "ATP ELECTRONICS INC";
			break;

		case 0x1dbb:
			strPtr = "NGD Systems, Inc.";
			break;

		case 0x1dbf:
			strPtr = "Guizhou Huaxintong Semiconductor Technology Co., Ltd";
			break;

		case 0x1dc5:
			strPtr = "FADU Inc.";
			break;

		case 0x1dcd:
			strPtr = "Liqid Inc.";
			break;

		case 0x1dd8:
			strPtr = "Pensando Systems";
			break;

		case 0x1de0:
			strPtr = "Groq";
			break;

		case 0x1de1:
			strPtr = "Tekram Technology Co.,Ltd.";
			break;

		case 0x1de5:
			strPtr = "Eideticom, Inc";
			break;

		case 0x1dee:
			strPtr = "Biwin Storage Technology Co., Ltd.";
			break;

		case 0x1def:
			strPtr = "Ampere Computing, LLC";
			break;

		case 0x1df3:
			strPtr = "Ethernity Networks";
			break;

		case 0x1df7:
			strPtr = "opencpi.org";
			break;

		case 0x1dfc:
			strPtr = "JSC NT-COM";
			break;

		case 0x1e0f:
			strPtr = "KIOXIA Corporation";
			break;

		case 0x1e17:
			strPtr = "Arnold & Richter Cine Technik GmbH & Co. Betriebs KG";
			break;

		case 0x1e24:
			strPtr = "Squirrels Research Labs";
			break;

		case 0x1e26:
			strPtr = "Fujitsu Client Computing Limited";
			break;

		case 0x1e36:
			strPtr = "Shanghai Enflame Technology Co. Ltd";
			break;

		case 0x1e38:
			strPtr = "Blaize, Inc";
			break;

		case 0x1e3b:
			strPtr = "Shenzhen DAPU Microelectronics Co., Ltd";
			break;

		case 0x1e3d:
			strPtr = "Burlywood, Inc";
			break;

		case 0x1e49:
			strPtr = "Yangtze Memory Technologies Co.,Ltd";
			break;

		case 0x1e4b:
			strPtr = "MAXIO Technology";
			break;

		case 0x1e4c:
			strPtr = "GSI Technology";
			break;

		case 0x1e57:
			strPtr = "Beijing Panyi Technology Co., Ltd";
			break;

		case 0x1e59:
			strPtr = "Oxford Nanopore Technologies";
			break;

		case 0x1e60:
			strPtr = "Hailo Technologies Ltd.";
			break;

		case 0x1e6b:
			strPtr = "Axiado Corp.";
			break;

		case 0x1e7b:
			strPtr = "Dataland";
			break;

		case 0x1e7c:
			strPtr = "Brainchip Inc";
			break;

		case 0x1e85:
			strPtr = "Heitec AG";
			break;

		case 0x1e89:
			strPtr = "ID Quantique SA";
			break;

		case 0x1e94:
			strPtr = "Calian SED";
			break;

		case 0x1e95:
			strPtr = "Solid State Storage Technology Corporation";
			break;

		case 0x1ea0:
			strPtr = "Tencent Technology";
			break;

		case 0x1ea7:
			strPtr = "Intelliprop, Inc";
			break;

		case 0x1eab:
			strPtr = "Hefei DATANG Storage Technology Co.,LTD.";
			break;

		case 0x1eae:
			strPtr = "XFX Limited";
			break;

		case 0x1eb1:
			strPtr = "VeriSilicon Inc";
			break;

		case 0x1ebd:
			strPtr = "EMERGETECH Company Ltd.";
			break;

		case 0x1ed3:
			strPtr = "Yeston";
			break;

		case 0x1ed8:
			strPtr = "Digiteq Automotive";
			break;

		case 0x1ed9:
			strPtr = "Myrtle.ai";
			break;

		case 0x1ee9:
			strPtr = "SUSE LLC";
			break;

		case 0x1efb:
			strPtr = "Flexxon Pte Ltd";
			break;

		case 0x1fc0:
			strPtr = "Ascom";
			break;

		case 0x1fc1:
			strPtr = "QLogic, Corp.";
			break;

		case 0x1fc9:
			strPtr = "Tehuti Networks Ltd.";
			break;

		case 0x1fcc:
			strPtr = "StreamLabs";
			break;

		case 0x1fce:
			strPtr = "Cognio Inc.";
			break;

		case 0x1fd4:
			strPtr = "SUNIX Co., Ltd.";
			break;

		case 0x2000:
			strPtr = "Smart Link Ltd.";
			break;

		case 0x2001:
			strPtr = "Temporal Research Ltd";
			break;

		case 0x2003:
			strPtr = "Smart Link Ltd.";
			break;

		case 0x2004:
			strPtr = "Smart Link Ltd.";
			break;

		case 0x2048:
			strPtr = "Beijing SpaceControl Technology Co.Ltd";
			break;

		case 0x20f4:
			strPtr = "TRENDnet";
			break;

		case 0x2116:
			strPtr = "ZyDAS Technology Corp.";
			break;

		case 0x21c3:
			strPtr = "21st Century Computer Corp.";
			break;

		case 0x22b8:
			strPtr = "Flex-Logix Technologies";
			break;

		case 0x22db:
			strPtr = "Missing Link Electronics, Inc.";
			break;

		case 0x2304:
			strPtr = "Colorgraphic Communications Corp.";
			break;

		case 0x2348:
			strPtr = "Racore";
			break;

		case 0x2646:
			strPtr = "Kingston Technology Company, Inc.";
			break;

		case 0x270b:
			strPtr = "Xantel Corporation";
			break;

		case 0x270f:
			strPtr = "Chaintech Computer Co. Ltd";
			break;

		case 0x2711:
			strPtr = "AVID Technology Inc.";
			break;

		case 0x2955:
			strPtr = "Connectix Virtual PC";
			break;

		case 0x2a15:
			strPtr = "3D Visio";
			break;

		case 0x2bd8:
			strPtr = "ROPEX Industrie-Elektronik GmbH";
			break;

		case 0x3000:
			strPtr = "Hansol Electronics Inc.";
			break;

		case 0x3112:
			strPtr = "Satelco Ingenieria S.A.";
			break;

		case 0x3130:
			strPtr = "AUDIOTRAK";
			break;

		case 0x3142:
			strPtr = "Post Impression Systems.";
			break;

		case 0x31ab:
			strPtr = "Zonet";
			break;

		case 0x3388:
			strPtr = "Hint Corp";
			break;

		case 0x3411:
			strPtr = "Quantum Designs";
			break;

		case 0x3442:
			strPtr = "Bihl+Wiedemann GmbH";
			break;

		case 0x3475:
			strPtr = "Arista Networks, Inc.";
			break;

		case 0x3513:
			strPtr = "ARCOM Control Systems Ltd";
			break;

		case 0x37d9:
			strPtr = "ITD Firm ltd.";
			break;

		case 0x3842:
			strPtr = "eVga.com. Corp.";
			break;

		case 0x38ef:
			strPtr = "4Links";
			break;

		case 0x3d3d:
			strPtr = "3DLabs";
			break;

		case 0x4005:
			strPtr = "Avance Logic Inc.";
			break;

		case 0x4033:
			strPtr = "Addtron Technology Co, Inc.";
			break;

		case 0x4040:
			strPtr = "NetXen Incorporated";
			break;

		case 0x4143:
			strPtr = "Digital Equipment Corp";
			break;

		case 0x4144:
			strPtr = "Alpha Data";
			break;

		case 0x4150:
			strPtr = "ONA Electroerosion";
			break;

		case 0x415a:
			strPtr = "Auzentech, Inc.";
			break;

		case 0x416c:
			strPtr = "Aladdin Knowledge Systems";
			break;

		case 0x4254:
			strPtr = "DVBSky";
			break;

		case 0x4321:
			strPtr = "Tata Power Strategic Electronics Division";
			break;

		case 0x4348:
			strPtr = "WCH.CN";
			break;

		case 0x434e:
			strPtr = "Cornelis Networks";
			break;

		case 0x43bc:
			strPtr = "Tiger Lake-H PCIe Root Port #5";
			break;

		case 0x4444:
			strPtr = "Internext Compression Inc";
			break;

		case 0x4468:
			strPtr = "Bridgeport machines";
			break;

		case 0x4594:
			strPtr = "Cogetec Informatique Inc";
			break;

		case 0x45fb:
			strPtr = "Baldor Electric Company";
			break;

		case 0x4624:
			strPtr = "Budker Institute of Nuclear Physics";
			break;

		case 0x4651:
			strPtr = "TXIC";
			break;

		case 0x4680:
			strPtr = "Umax Computer Corp";
			break;

		case 0x4843:
			strPtr = "Hercules Computer Technology Inc";
			break;

		case 0x4916:
			strPtr = "RedCreek Communications Inc";
			break;

		case 0x4943:
			strPtr = "Growth Networks";
			break;

		case 0x494f:
			strPtr = "ACCES I/O Products, Inc.";
			break;

		case 0x4978:
			strPtr = "Axil Computer Inc";
			break;

		case 0x4a14:
			strPtr = "NetVin";
			break;

		case 0x4b10:
			strPtr = "Buslogic Inc.";
			break;

		case 0x4c48:
			strPtr = "LUNG HWA Electronics";
			break;

		case 0x4c52:
			strPtr = "LR-Link";
			break;

		case 0x4c53:
			strPtr = "SBS Technologies";
			break;

		case 0x4ca1:
			strPtr = "Seanix Technology Inc";
			break;

		case 0x4d51:
			strPtr = "MediaQ Inc.";
			break;

		case 0x4d54:
			strPtr = "Microtechnica Co Ltd";
			break;

		case 0x4d56:
			strPtr = "MATRIX VISION GmbH";
			break;

		case 0x4ddc:
			strPtr = "ILC Data Device Corp";
			break;

		case 0x5045:
			strPtr = "University of Toronto";
			break;

		case 0x5046:
			strPtr = "GemTek Technology Corporation";
			break;

		case 0x5053:
			strPtr = "Voyetra Technologies";
			break;

		case 0x50b2:
			strPtr = "TerraTec Electronic GmbH";
			break;

		case 0x5136:
			strPtr = "S S Technologies";
			break;

		case 0x5143:
			strPtr = "Qualcomm Inc";
			break;

		case 0x5145:
			strPtr = "Ensoniq";
			break;

		case 0x5168:
			strPtr = "Animation Technologies Inc.";
			break;

		case 0x5301:
			strPtr = "Alliance Semiconductor Corp.";
			break;

		case 0x5333:
			strPtr = "S3 Graphics Ltd.";
			break;

		case 0x5431:
			strPtr = "AuzenTech, Inc.";
			break;

		case 0x544c:
			strPtr = "Teralogic Inc";
			break;

		case 0x544d:
			strPtr = "TBS Technologies";
			break;

		case 0x5452:
			strPtr = "SCANLAB AG";
			break;

		case 0x5455:
			strPtr = "Technische Universitaet Berlin";
			break;

		case 0x5456:
			strPtr = "GoTView";
			break;

		case 0x5519:
			strPtr = "Cnet Technologies, Inc.";
			break;

		case 0x5544:
			strPtr = "Dunord Technologies";
			break;

		case 0x5555:
			strPtr = "Genroco, Inc";
			break;

		case 0x5646:
			strPtr = "Vector Fabrics BV";
			break;

		case 0x5654:
			strPtr = "VoiceTronix Pty Ltd";
			break;

		case 0x5678:
			strPtr = "Dawicontrol Computersysteme GmbH";
			break;

		case 0x5700:
			strPtr = "Netpower";
			break;

		case 0x5845:
			strPtr = "X-ES, Inc.";
			break;

		case 0x584d:
			strPtr = "AuzenTech Co., Ltd.";
			break;

		case 0x5851:
			strPtr = "Exacq Technologies";
			break;

		case 0x5853:
			strPtr = "XenSource, Inc.";
			break;

		case 0x5854:
			strPtr = "GoTView";
			break;

		case 0x5ace:
			strPtr = "Beholder International Ltd.";
			break;

		case 0x6205:
			strPtr = "TBS Technologies";
			break;

		case 0x6209:
			strPtr = "TBS Technologies";
			break;

		case 0x631c:
			strPtr = "SmartInfra Ltd";
			break;

		case 0x6356:
			strPtr = "UltraStor";
			break;

		case 0x6374:
			strPtr = "c't Magazin fuer Computertechnik";
			break;

		case 0x6409:
			strPtr = "Logitec Corp.";
			break;

		case 0x6549:
			strPtr = "Teradici Corp.";
			break;

		case 0x6666:
			strPtr = "Decision Computer International Co.";
			break;

		case 0x6688:
			strPtr = "Zycoo Co., Ltd";
			break;

		case 0x6900:
			strPtr = "Red Hat, Inc.";
			break;

		case 0x7063:
			strPtr = "pcHDTV";
			break;

		case 0x7284:
			strPtr = "HT OMEGA Inc.";
			break;

		case 0x7357:
			strPtr = "IOxOS Technologies SA";
			break;

		case 0x7401:
			strPtr = "EndRun Technologies";
			break;

		case 0x7470:
			strPtr = "TP-LINK Technologies Co., Ltd.";
			break;

		case 0x7604:
			strPtr = "O.N. Electronic Co Ltd.";
			break;

		case 0x7bde:
			strPtr = "MIDAC Corporation";
			break;

		case 0x7fed:
			strPtr = "PowerTV";
			break;

		case 0x8008:
			strPtr = "Quancom Electronic GmbH";
			break;

		case 0x807d:
			strPtr = "Asustek Computer, Inc.";
			break;

		case 0x8086:
			strPtr = "Intel Corporation";
			break;

		case 0x8088:
			strPtr = "Beijing Wangxun Technology Co., Ltd.";
			break;

		case 0x80ee:
			strPtr = "InnoTek Systemberatung GmbH";
			break;

		case 0x8322:
			strPtr = "Sodick America Corp.";
			break;

		case 0x8384:
			strPtr = "SigmaTel";
			break;

		case 0x8401:
			strPtr = "TRENDware International Inc.";
			break;

		case 0x8686:
			strPtr = "ScaleMP";
			break;

		case 0x8800:
			strPtr = "Trigem Computer Inc.";
			break;

		case 0x8820:
			strPtr = "Stryker Corporation";
			break;

		case 0x8866:
			strPtr = "T-Square Design Inc.";
			break;

		case 0x8888:
			strPtr = "Silicon Magic";
			break;

		case 0x8912:
			strPtr = "TRX";
			break;

		case 0x8c4a:
			strPtr = "Winbond";
			break;

		case 0x8e0e:
			strPtr = "Computone Corporation";
			break;

		case 0x8e2e:
			strPtr = "KTI";
			break;

		case 0x9004:
			strPtr = "Adaptec";
			break;

		case 0x9005:
			strPtr = "Adaptec";
			break;

		case 0x907f:
			strPtr = "Atronics";
			break;

		case 0x919a:
			strPtr = "Gigapixel Corp";
			break;

		case 0x9412:
			strPtr = "Holtek";
			break;

		case 0x9413:
			strPtr = "Softlogic Co., Ltd.";
			break;

		case 0x9618:
			strPtr = "JusonTech Corporation";
			break;

		case 0x9699:
			strPtr = "Omni Media Technology Inc";
			break;

		case 0x9710:
			strPtr = "MosChip Semiconductor Technology Ltd.";
			break;

		case 0x9850:
			strPtr = "3Com";
			break;

		case 0x9902:
			strPtr = "Stargen Inc.";
			break;

		case 0x9a11:
			strPtr = "Tiger Lake-H Gaussian & Neural Accelerator";
			break;

		case 0x9d32:
			strPtr = "Beijing Starblaze Technology Co. Ltd.";
			break;

		case 0xa000:
			strPtr = "Asix Electronics Corporation";
			break;

		case 0xa0a0:
			strPtr = "AOPEN Inc.";
			break;

		case 0xa0f1:
			strPtr = "UNISYS Corporation";
			break;

		case 0xa200:
			strPtr = "NEC Corporation";
			break;

		case 0xa259:
			strPtr = "Hewlett Packard";
			break;

		case 0xa25b:
			strPtr = "Hewlett Packard GmbH PL24-MKT";
			break;

		case 0xa304:
			strPtr = "Sony";
			break;

		case 0xa727:
			strPtr = "3Com Corporation";
			break;

		case 0xaa00:
			strPtr = "iTuner";
			break;

		case 0xaa01:
			strPtr = "iTuner";
			break;

		case 0xaa02:
			strPtr = "iTuner";
			break;

		case 0xaa03:
			strPtr = "iTuner";
			break;

		case 0xaa04:
			strPtr = "iTuner";
			break;

		case 0xaa05:
			strPtr = "iTuner";
			break;

		case 0xaa06:
			strPtr = "iTuner";
			break;

		case 0xaa07:
			strPtr = "iTuner";
			break;

		case 0xaa08:
			strPtr = "iTuner";
			break;

		case 0xaa09:
			strPtr = "iTuner";
			break;

		case 0xaa0a:
			strPtr = "iTuner";
			break;

		case 0xaa0b:
			strPtr = "iTuner";
			break;

		case 0xaa0c:
			strPtr = "iTuner";
			break;

		case 0xaa0d:
			strPtr = "iTuner";
			break;

		case 0xaa0e:
			strPtr = "iTuner";
			break;

		case 0xaa0f:
			strPtr = "iTuner";
			break;

		case 0xaa42:
			strPtr = "Scitex Digital Video";
			break;

		case 0xaa55:
			strPtr = "Ncomputing X300 PCI-Engine";
			break;

		case 0xaaaa:
			strPtr = "Adnaco Technology Inc.";
			break;

		case 0xabcd:
			strPtr = "Vadatech Inc.";
			break;

		case 0xac1e:
			strPtr = "Digital Receiver Technology Inc";
			break;

		case 0xac3d:
			strPtr = "Actuality Systems";
			break;

		case 0xad00:
			strPtr = "Alta Data Technologies LLC";
			break;

		case 0xaecb:
			strPtr = "Adrienne Electronics Corporation";
			break;

		case 0xaffe:
			strPtr = "Sirrix AG security technologies";
			break;

		case 0xb100:
			strPtr = "OpenVox Communication Co. Ltd.";
			break;

		case 0xb10b:
			strPtr = "Uakron PCI Project";
			break;

		case 0xb1b3:
			strPtr = "Shiva Europe Limited";
			break;

		case 0xb1d9:
			strPtr = "ATCOM Technology co., LTD.";
			break;

		case 0xbd11:
			strPtr = "Pinnacle Systems, Inc.";
			break;

		case 0xbdbd:
			strPtr = "Blackmagic Design";
			break;

		case 0xc001:
			strPtr = "TSI Telsys";
			break;

		case 0xc0a9:
			strPtr = "Micron/Crucial Technology";
			break;

		case 0xc0de:
			strPtr = "Motorola";
			break;

		case 0xc0fe:
			strPtr = "Motion Engineering, Inc.";
			break;

		case 0xca3b:
			strPtr = "Cambrionix Ltd.";
			break;

		case 0xca50:
			strPtr = "Varian Australia Pty Ltd";
			break;

		case 0xcace:
			strPtr = "CACE Technologies, Inc.";
			break;

		case 0xcaed:
			strPtr = "Canny Edge";
			break;

		case 0xcafe:
			strPtr = "Chrysalis-ITS";
			break;

		case 0xcc53:
			strPtr = "ScaleFlux Inc.";
			break;

		case 0xcccc:
			strPtr = "Catapult Communications";
			break;

		case 0xccec:
			strPtr = "Curtiss-Wright Controls Embedded Computing";
			break;

		case 0xcddd:
			strPtr = "Tyzx, Inc.";
			break;

		case 0xceba:
			strPtr = "KEBA AG";
			break;

		case 0xd161:
			strPtr = "Digium, Inc.";
			break;

		case 0xd209:
			strPtr = "Ultimarc";
			break;

		case 0xd4d4:
			strPtr = "Dy4 Systems Inc";
			break;

		case 0xd531:
			strPtr = "I+ME ACTIA GmbH";
			break;

		case 0xd84d:
			strPtr = "Exsys";
			break;

		case 0xdada:
			strPtr = "Datapath Limited";
			break;

		case 0xdb10:
			strPtr = "Diablo Technologies";
			break;

		case 0xdc93:
			strPtr = "Dawicontrol GmbH";
			break;

		case 0xdcba:
			strPtr = "Dynamic Engineering";
			break;

		case 0xdd01:
			strPtr = "Digital Devices GmbH";
			break;

		case 0xdead:
			strPtr = "Indigita Corporation";
			break;

		case 0xdeaf:
			strPtr = "Middle Digital Inc.";
			break;

		case 0xdeda:
			strPtr = "XIMEA";
			break;

		case 0xe000:
			strPtr = "Winbond";
			break;

		case 0xe159:
			strPtr = "Tiger Jet Network Inc. / ICP DAS";
			break;

		case 0xe1c5:
			strPtr = "Elcus";
			break;

		case 0xe4bf:
			strPtr = "EKF Elektronik GmbH";
			break;

		case 0xe4e4:
			strPtr = "Xorcom";
			break;

		case 0xe55e:
			strPtr = "Essence Technology, Inc.";
			break;

		case 0xea01:
			strPtr = "Eagle Technology";
			break;

		case 0xea50:
			strPtr = "Emerson Automation Solutions";
			break;

		case 0xea60:
			strPtr = "RME";
			break;

		case 0xeabb:
			strPtr = "Aashima Technology B.V.";
			break;

		case 0xeace:
			strPtr = "Endace Measurement Systems, Ltd";
			break;

		case 0xec80:
			strPtr = "Belkin Corporation";
			break;

		case 0xecc0:
			strPtr = "Echo Digital Audio Corporation";
			break;

		case 0xedd8:
			strPtr = "ARK Logic Inc";
			break;

		case 0xf043:
			strPtr = "ASUSTeK Computer Inc.";
			break;

		case 0xf05b:
			strPtr = "Foxconn International, Inc.";
			break;

		case 0xf15e:
			strPtr = "SiFive, Inc.";
			break;

		case 0xf1d0:
			strPtr = "AJA Video";
			break;

		case 0xf5f5:
			strPtr = "F5 Networks, Inc.";
			break;

		case 0xf849:
			strPtr = "ASRock Incorporation";
			break;

		case 0xfa57:
			strPtr = "Interagon AS";
			break;

		case 0xfab7:
			strPtr = "Fabric7 Systems, Inc.";
			break;

		case 0xfebd:
			strPtr = "Ultraview Corp.";
			break;

		case 0xfeda:
			strPtr = "Broadcom Inc";
			break;

		case 0xfede:
			strPtr = "Fedetec Inc.";
			break;

		case 0xfffd:
			strPtr = "XenSource, Inc.";
			break;

		case 0xfffe:
			strPtr = "VMWare Inc";
			break;

		case 0xffff:
			strPtr = "Illegal Vendor ID";
			break;
	}
	printf("%s\n", strPtr);
}





bool PCICheck()
{
	unsigned char PCIFlag;

	_asm
	{		
		mov eax, 0xB101
		mov edi, 0x00000000
		int 0x1A
		mov PCIFlag, ah
		mov PCIVerMajor, bh
		mov PCIVerMinor, bl
	}

	if (!PCIFlag)
	{
		return true;
	} else {
		return false;
	}
}





unsigned long PCIRegisterRead(unsigned short bus, unsigned short device, unsigned short funct, unsigned short reg)
{
	unsigned long outValue, regValue;
	unsigned long busL = bus;
	unsigned long deviceL = device;
	unsigned long functL = funct;

	outValue = (unsigned long) ((busL << 16) | (deviceL << 11) | (functL << 8) | (reg & 0xFC) | (unsigned long) (0x80000000));

	_asm
	{
		mov eax, outValue
		mov dx, 0x0CF8
		out dx, eax
		mov dx, 0x0CFC
		in eax, dx
		mov regValue, eax
	}

	return regValue;
}





void PrintAppHeader()
{
	printf("ListPCI     PCI device listing utility\n");
	printf("v%d.%.2d       2021 by Mercury Thirteen\n\n", verMajor, verMinor);
}





void PrintHelp()
{
	printf("Usage:\n");
	printf(" /C n \tShows only devices of the specified class.\n");
	printf(" /N   \tReturns the number of matching devices in the ERRORLEVEL variable.\n");
	printf(" /V n \tShows only devices by the vendor specified.\n");
	printf("\n");
	printf("Numbers for all options may be entered in decimal, hexadecimal, or octal.\n\n");
	printf("Options may be combined. For example, LISTPCI /C 2 /V 0x8086 would list only\nnetwork controllers made by Intel.\n\n");
}
